/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/*****************************************************************************

         Header file of common functions for NPE Module

    $Id: libnpec.h,v 1.2 2004/04/19 17:07:52 orrisroot Exp $

*****************************************************************************/
#ifndef _LIBNPEC_H_
#define _LIBNPEC_H_

#ifdef __cplusplus
extern "C" {
#endif

/* data type definition */
typedef int npe_bool_t;

/* version string */
#define NPE_VERSION_STRING "NPE Version 3.8.1 Apr 19, 2004"

/* definition of macro */
#define NPE_TRUE       (-1)
#define NPE_FALSE      (0)

/* NPE common area file name */
#define  NPECOMMONAREA  "_npe_common.dat"

/* max line buffer */
#define NPE_MAXLINE     512
/* max element name length */
#define NPE_MAXELEMENTNAME 16
/* max length of parameter name */
#define NPE_MAXPARAMNAME     25

/* limit size */
#define NPE_LIMIT 10000

/* parameter flag type */
#define NPE_PARAM_FLAG_NONE      -1
#define NPE_PARAM_FLAG_FIX        0
#define NPE_PARAM_FLAG_VAR        1
#define NPE_PARAM_FLAG_TYPESIZE   2

/* element type */
#define NPE_ELEMENT_NONE       -1
#define NPE_ELEMENT_METHOD      0
#define NPE_ELEMENT_LSEARCH     1
#define NPE_ELEMENT_MODEL       2
#define NPE_ELEMENT_PENALTY     3
#define NPE_ELEMENT_INIT        4
#define NPE_ELEMENT_SCALE       5
#define NPE_ELEMENT_TERM        6
#define NPE_ELEMENT_NUMBER      7
#define NPE_ELEMENT_POINT       8
#define NPE_ELEMENT_DATA        9
#define NPE_ELEMENT_WEIGHT     10
#define NPE_ELEMENT_RESULT     11
#define NPE_ELEMENT_HISTORY    12
#define NPE_ELEMENT_INTEG      13
#define NPE_ELEMENT_DISPLAY    14
#define NPE_ELEMENT_NORM       15
#define NPE_ELEMENT_ALL        16
#define NPE_ELEMENT_TYPESIZE   17

/* optimization method - %method*/
#define NPE_METHOD_NONE     -1
#define NPE_METHOD_SIMPLEX   0
#define NPE_METHOD_BFGS      1
#define NPE_METHOD_DFP       2
#define NPE_METHOD_SSVM      3
#define NPE_METHOD_CONJFR    4
#define NPE_METHOD_CONJPRP   5
#define NPE_METHOD_TYPESIZE  6

/* linear search method - %lsearch */
#define NPE_LSEARCH_NONE     -1
#define NPE_LSEARCH_GOLDEN    0
#define NPE_LSEARCH_CUBIC     1
#define NPE_LSEARCH_BRENT     2
#define NPE_LSEARCH_TYPESIZE  3
/* model type and name - %model */
#define NPE_MODEL_NONE      -1
#define NPE_MODEL_USR        0
#define NPE_MODEL_NCS        1
#define NPE_MODEL_TYPESIZE   2

/* scaling method - %scale */
#define NPE_SCALE_NONE      -1 
#define NPE_SCALE_OFF        0 /* scaling off */
#define NPE_SCALE_NOFIRST    1 /* exclude scaling at the first time only */
#define NPE_SCALE_ALLWAYS    2 /* allways */
#define NPE_SCALE_RESCALING  3 /* re-scaling */
#define NPE_SCALE_TYPESIZE   4

/* terminate number - %term */
#define NPE_TERM_MAXREPEAT    0 /* number of maximum repeat calculation */
#define NPE_TERM_EVALFUNC     1 /* value of evaluation function */
#define NPE_TERM_STDERROR     9 /* standard error of 'simplex' method */
#define NPE_TERM_CRITESIZE   10 /* size of criterion number */

/* display error type - %display */
#define NPE_DISPLAY_NONE        -1
#define NPE_DISPLAY_NO_WEIGHT    0
#define NPE_DISPLAY_WITH_WEIGHT  1
#define NPE_DISPLAY_TYPESIZE     2

/* norm type - %norm */
#define NPE_NORM_NONE      -1
#define NPE_NORM_INF        0
#define NPE_NORM_ONE        1
#define NPE_NORM_TWO        2
#define NPE_NORM_TYPESIZE   3

/* parameter type */
typedef struct _npe_param_t {
  double  value;   /** Value        of a Parameter    **/
  double  scale;   /** Scale Factor of a Parameter    **/
  double  span;    /** Differential Step Size         **/
  int     flag;    /** Whether a Parameter is "Fixed" **/
  int     dummy;   /**   or "Variable"                **/
} npe_param_t;

/* initial estimetion parameter */
typedef struct _npe_estparam_t {
  double  value;      /* init value */
  int     flag;       /* NPE_PARAM_FIX or NPE_PARAM_VAR */
  char   *flag_name;  /* flag name */
  char   *name;       /* parameter name */
  double  span;       /* span */
} npe_estparam_t;

/* memory of system common area */
typedef struct _npe_common_t {
  /* store flag */
  npe_bool_t store_flag;
  /* optimization method */
  int     method_type;   /* type - NPE_METHOD */
  char   *method_name;   /* name */
  /* linear search  */
  int     lsearch_type;  /* type - NPE_LSEARCH */
  char   *lsearch_name;  /* name */
  double  lsearch_value; /* length */
  /* model */
  int     model_type;    /* type - NPE_MODEL */
  char   *model_name;    /* name */
  char   *model_file;    /* file name */
  /* penalty */
  char   *penalty_file;  /* file name */
  /* number of initial parameters */
  unsigned int init_paramnum;
  /* pointer of estimation parameter */
  npe_estparam_t *estparam;
  /* scale type - NPE_SCALE */
  int     scale_type;
  /* terminate logic */
  char   *term_logic;
  /* terminate criterion - NPE_TERM */
  npe_bool_t term_setbit[NPE_TERM_CRITESIZE];
  double     term_crite[NPE_TERM_CRITESIZE];
  /* number of wave */
  int     number_value;
  /* number of data point */
  int     point_value;
  /* file name of experimental data */
  char   *data_file;
  /* file name of weight data */
  char   *weight_file;
  /* file name of result data */
  char   *result_file;
  /* file name of history data */
  char   *history_file;
  /* store interval of history data*/
  int     history_interval;
  /* display error type - NPE_DISPLAY */
  int     display_type;
  /* norm type - NPE_NORM */
  int     norm_type;
} npe_common_t;

/* lib/common/init.c */
npe_common_t *npe_common_new();
int           npe_common_delete(npe_common_t *nc);

/* lib/common/type.c */
int npe_element_type(const char *name);
int npe_element_length(int type);
int npe_method_type(const char *name);
int npe_lsearch_type(const char *name);
int npe_model_type(const char *name);
int npe_param_flag_type(const char *name);
int npe_strincmp(const char *s1, const char *s2, size_t n);

/* lib/common/set.c */
int npe_method_set(npe_common_t *nc, int type);
int npe_lsearch_set(npe_common_t *nc, int type, double value);
int npe_model_set(npe_common_t *nc, int type, const char *fname);
int npe_penalty_set(npe_common_t *nc, const char *fname);
int npe_init_set(npe_common_t *nc, unsigned int size, unsigned int inum, 
                 double value, int flag, const char *name, double span);
int npe_scale_set(npe_common_t *nc, int type);
int npe_logic_set(npe_common_t *nc, const char *logic);
int npe_term_set(npe_common_t *nc, int num, double value, npe_bool_t flag);
int npe_number_set(npe_common_t *nc, int num);
int npe_point_set(npe_common_t *nc, int num);
int npe_data_set(npe_common_t *nc, const char *fname);
int npe_weight_set(npe_common_t *nc, const char *fname);
int npe_result_set(npe_common_t *nc, const char *fname);
int npe_history_set(npe_common_t *nc, const char *fname, int interval);
int npe_display_set(npe_common_t *nc, int type);
int npe_norm_set(npe_common_t *nc, int type);

/* lib/common/load.c */
int npe_common_load(npe_common_t *nc, const char *fname);

/* lib/common/store.c */
int npe_common_store(npe_common_t *nc, const char *fname);

#ifdef __cplusplus
}
#endif

#endif  /* __global_vars__ */
