/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: load.c,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/**************************************************
**  load.c                                       **
**  load NPE common area file                    **
**************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "libnpec.h"
#include "npecpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

/* #define _DEBUG_LOAD_COMMON_AREA */ /* for debug load common area */


#define DELIMITER   " ,:\t\r\n"
#define IDENT       '%'
#define SKIP(c)  while(c!=NULL&&(*c==' '||*c=='\t'))c++

/***************************
** functions in this file **
****************************/
/* function prototype */
static int npe_method_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_lsearch_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_model_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_penalty_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_init_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_scale_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_term_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_number_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_point_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_data_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_weight_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_result_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_history_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_integ_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_display_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int npe_norm_load(npe_common_t *nc, FILE *fp, char *line, int len);
static int search_ident(FILE *fp, char *line, int len, 
                        int *type, int *err);
static int next_line(FILE *fp, char *line, int len);

int npe_common_load(npe_common_t *nc, const char *filename){
  FILE *fp;
  int  type, code, err;
  char line[NPE_MAXLINE];

  if(nc == NULL)
    return 1; /* NPE common area not initialized. */

  if(filename == NULL)
    return 101; /* Can't open NPE common area file */

#ifdef WIN32
  fp = fopen(filename, "rb");
#else
  fp = fopen(filename, "r");
#endif
  if(fp == NULL)
    return 101; /* Can't open NPE common area file */
  
  while(search_ident(fp, line, NPE_MAXLINE, &type, &err)){
    code = 0;
    switch(type){
    case NPE_ELEMENT_METHOD:
      code = npe_method_load(nc, fp, line, NPE_MAXLINE);  break;
    case NPE_ELEMENT_LSEARCH:
      code = npe_lsearch_load(nc, fp, line, NPE_MAXLINE); break;
    case NPE_ELEMENT_MODEL:
      code = npe_model_load(nc, fp, line, NPE_MAXLINE);   break;
    case NPE_ELEMENT_PENALTY:
      code = npe_penalty_load(nc, fp, line, NPE_MAXLINE); break;
    case NPE_ELEMENT_INIT:
      code = npe_init_load(nc, fp, line, NPE_MAXLINE);    break;
    case NPE_ELEMENT_SCALE:
      code = npe_scale_load(nc, fp, line, NPE_MAXLINE);   break;
    case NPE_ELEMENT_TERM:
      code = npe_term_load(nc, fp, line, NPE_MAXLINE);    break;
    case NPE_ELEMENT_NUMBER:
      code = npe_number_load(nc, fp, line, NPE_MAXLINE);  break;
    case NPE_ELEMENT_POINT:
      code = npe_point_load(nc, fp, line, NPE_MAXLINE);   break;
    case NPE_ELEMENT_DATA:
      code = npe_data_load(nc, fp, line, NPE_MAXLINE);    break;
    case NPE_ELEMENT_WEIGHT:
      code = npe_weight_load(nc, fp, line, NPE_MAXLINE);  break;
    case NPE_ELEMENT_RESULT:
      code = npe_result_load(nc, fp, line, NPE_MAXLINE);  break;
    case NPE_ELEMENT_HISTORY:
      code = npe_history_load(nc, fp, line, NPE_MAXLINE); break;
    case NPE_ELEMENT_INTEG:
      code = npe_integ_load(nc, fp, line, NPE_MAXLINE);   break;
    case NPE_ELEMENT_DISPLAY:
      code = npe_display_load(nc, fp, line, NPE_MAXLINE); break;
    case NPE_ELEMENT_NORM:
      code = npe_norm_load(nc, fp, line, NPE_MAXLINE);    break;
    default:
      code = 102;
    }
    if(code != 0){
      err = code;
      break;
    }
  }

  fclose(fp);

  if(err != 0){
    return err;
  }
/*   /\* levenson-marquart method dose not use the linear search *\/ */
/*   if(ncn->method_type == NPE_METHOD_LEVEN){ */
/*     nc->lsearch_type = NPE_LSEARCH_NONE; */
/*     if(nc->lsearch_name != NULL) free(nc->lsearch_name); */
/*     nc->lsearch_name  = NULL; */
/*     nc->lsearch_value = 0.0; */
/*   } */

  if(nc->method_type == NPE_METHOD_SIMPLEX){
    nc->lsearch_type  = NPE_LSEARCH_NONE;
    if(nc->lsearch_name != NULL) free(nc->lsearch_name);
    nc->lsearch_name  = NULL;
    nc->lsearch_value = 0.0;
    nc->scale_type    = NPE_SCALE_NONE;
  }

  nc->store_flag = NPE_TRUE;
  return 0;
}

/* private functions */
static int npe_method_load(npe_common_t *nc, FILE *fp, char *line, int len){
  int   type;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* optimization method */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  type = npe_method_type(cp);
  if(type == NPE_METHOD_NONE)
    return 104; /* illegal optimization method name */

  /* set to common area */
  code = npe_method_set(nc, type);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 105; /* illegal optimization method type */
  }
  return 0;
}


static int npe_lsearch_load(npe_common_t *nc, FILE *fp, char *line, int len){
  double value;
  int    type;
  int    code;
  char  *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* linear search method */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  type = npe_lsearch_type(cp);
  if(type == NPE_LSEARCH_NONE)
    return 106; /* illegal linear search method name */

  /* initial bracket value */
  cp = npe_strtok_r(NULL, DELIMITER, &tail);
  if(cp == NULL)
    return 108; /* illegal linear search initial bracket value */
  sscanf(cp, "%lf", &value);

  /* set to common area */
  code = npe_lsearch_set(nc, type, value);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 107; /* illegal linear search method type */
  case  2: return 108; /* illegal linear search initial bracket value */
  }
  return 0;
}


static int npe_model_load(npe_common_t *nc, FILE *fp, char *line, int len){
  char  *fname;
  int    type;
  int    code;
  char  *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* model type */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 109; /* illegal model type */
  sscanf(cp, "%d", &type);

  /* model file name */
  cp = npe_strtok_r(NULL, DELIMITER, &tail);
  if(cp == NULL)
    return 110; /* illegal model file name */
  fname = cp;

  /* set to common area */
  code = npe_model_set(nc, type, fname);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 109; /* illegal model type */
  case  2: return 110; /* illegal model file name */
  }
  return 0;
}


static int npe_penalty_load(npe_common_t *nc, FILE *fp, char *line, int len){
  char  *fname;
  int    code;
  char  *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;
  /* file name */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 109; /* illegal model type */
  fname = cp;

  /* set to common area */
  code = npe_penalty_set(nc, fname);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 111; /* illegal penalty file name */
  }
  return 0;
}


static int  npe_init_load(npe_common_t *nc, FILE *fp, char *line, int len){
  int    i;
  int    size, flag;
  char  *name;
  double value, span;
  int    code;
  char  *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* number of parameter */
  cp = npe_strtok_r(charp, DELIMITER, &tail);

  if(cp == NULL)
    return 112; /* illegal number of initial parameters */
  sscanf(cp, "%u", &size);

  /* set the initial value */
  for(i=0; i<size; i++){
    if(next_line(fp, line, len) == -1){
      return 113; /* illegal size of initial parameters */
    }
    charp = line;

    /* parameter value */
    cp = npe_strtok_r(charp, DELIMITER, &tail);
    if(cp == NULL)
      return 114; /* illegal initial parameter value */
    sscanf(cp, "%lf", &value);

    /* parameter flag : fix or variable */
    cp = npe_strtok_r(NULL, DELIMITER, &tail);
    if(cp == NULL)
      return 115; /* illegal initial parameter flag */
    flag = (*cp == 'f' || *cp == 'F') ?
      NPE_PARAM_FLAG_FIX : NPE_PARAM_FLAG_VAR;

    /* parameter name */
    cp = npe_strtok_r(NULL, DELIMITER, &tail);
    if(cp == NULL)
      return 116; /* illegal initial parameter name */
    name = cp;

    /* parameter span */
    cp = npe_strtok_r(NULL, DELIMITER, &tail);
    if(cp == NULL)
      return 117; /* illegal initial parameter name */
    sscanf(cp, "%lf", &span);

    /* set to common area */
    code = npe_init_set(nc, size, i+1, value, flag, name, span);
    switch(code){
    case -1: return 1;   /* can't open common area */
    case -2: return 2;   /* out of memory */
    case  1: return 112; /* illegal number of initial parameters */
    case  2: return 115; /* illegal initial perameter flag */
    case  3: return 116; /* illegal initial parameter name */
    }
  }
  return 0;
}


static int npe_scale_load(npe_common_t *nc, FILE *fp, char *line, int len){
  int   type;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* scale */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 118; /* illegal scale type */
  sscanf(cp, "%d", &type);
  if(type <= NPE_SCALE_NONE)
    return 118; /* illegal scale type */

  /* set to common area */
  code = npe_scale_set(nc, type);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 118; /* illegal scale type */
  }
  return 0;
}


static int npe_term_load(npe_common_t *nc, FILE *fp, char *line, int len){
  char  *logic;
  int    num;
  double value;
  int    code;
  char  *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* logical equation */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  logic = cp;

  /* set to common area */
  code = npe_logic_set(nc, logic);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 119; /* illegal logic string */
  }

  /* get the value of each logic */
  while(next_line(fp, line, len)!=-1){
    charp = line;

    /* terminate criterion number */
    cp = npe_strtok_r(charp, DELIMITER, &tail);
    if(cp == NULL)
      return 120; /* illegal terminate criterion number */
    sscanf(cp, "%d", &num);
    if(num < 0 || num >= NPE_TERM_CRITESIZE) 
      return 120; /* illegal terminate criterion number */

    /* terminate criterion value */
    cp = npe_strtok_r(NULL, DELIMITER, &tail);
    if(cp == NULL)
      return 121; /* illegal terminate criterion value */
    sscanf(cp, "%lf", &value);

    code = npe_term_set(nc, num, value, NPE_TRUE);
    switch(code){
    case -1: return 1;   /* can't open common area */
    case -2: return 2;   /* out of memory */
    case  1: return 119; /* illegal logic string */
    }
  }
  return 0;
}


static int npe_number_load(npe_common_t *nc, FILE *fp, char *line, int len){
  int   num;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* number of wave */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 122; /* illegal number of wave */
  sscanf(cp, "%d", &num);
  if(num <= 0)
    return 122; /* illegal number of wave */

  /* set to common area */
  code = npe_number_set(nc, num);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 122; /* illegal number of wave */
  }
  return 0;
}


static int npe_point_load(npe_common_t *nc, FILE *fp, char *line, int len){
  int   point;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* number of wave */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 123; /* illegal number of data point */
  sscanf(cp, "%d", &point);
  if(point <= 0)
    return 123; /* illegal number of data point */

  /* set to common area */
  code = npe_point_set(nc, point);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 123; /* illegal number of data point */
  }
  return 0;
}


static int npe_data_load(npe_common_t *nc, FILE *fp, char *line, int len){
  char *fname;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* data file name */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 124; /* illegal data file name */
  fname = cp;

  /* set to common area */
  code = npe_data_set(nc, fname);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 124; /* illegal data file name */
  }
  return 0;
}


static int npe_weight_load(npe_common_t *nc, FILE *fp, char *line, int len){
  char *fname;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* data file name */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 125; /* illegal weight data file name */
  fname = cp;

  /* set to common area */
  code = npe_weight_set(nc, fname);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 125; /* illegal weight data file name */
  }
  return 0;
}


static int npe_result_load(npe_common_t *nc, FILE *fp, char *line, int len){
  char *fname;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* data file name */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 126; /* illegal result data file name */
  fname = cp;

  /* set to common area */
  code = npe_result_set(nc, fname);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 126; /* illegal result data file name */
  }
  return 0;
}

static int npe_history_load(npe_common_t *nc, FILE *fp, char *line, int len){
  char *fname;
  int   interval;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* data file name */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 127; /* illegal history data file name */
  fname = cp;

  /* interval */
  cp = npe_strtok_r(NULL, DELIMITER, &tail);
  if(cp == NULL)
    interval = 1;
  else
    sscanf(cp, "%d", &interval);
  if(interval <= 0)
    interval = 1;

  /* set to common area */
  code = npe_history_set(nc, fname, interval);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 127; /* illegal history data file name */
  case  2: return 128; /* illegal history interval */
  }
  return 0;
}

static int npe_integ_load(npe_common_t *nc, FILE *fp, char *line, int len){
  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  /* it is obsolete elements */
  return 0;
}

static int npe_display_load(npe_common_t *nc, FILE *fp, char *line, int len){
  int   type;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* display type */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 129; /* illegal display type */
  sscanf(cp, "%d", &type);
  if(type <= NPE_DISPLAY_NONE)
    return 129; /* illegal display type */

  /* set to common area */
  code = npe_display_set(nc, type);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 129; /* illegal display type */
  }
  return 0;
}

static int npe_norm_load(npe_common_t *nc, FILE *fp, char *line, int len){
  int   type;
  int   code;
  char *cp, *charp, *tail;

  if(next_line(fp, line, len) == -1)
    return 103; /* element has no content */
  charp = line;

  /* norm type */
  cp = npe_strtok_r(charp, DELIMITER, &tail);
  if(cp == NULL)
    return 130; /* illegal norm type */
  sscanf(cp, "%d", &type);
  if(type <= NPE_NORM_NONE)
    return 130; /* illegal norm type */

  /* set to common area */
  code = npe_norm_set(nc, type);
  switch(code){
  case -1: return 1;   /* can't open common area */
  case -2: return 2;   /* out of memory */
  case  1: return 130; /* illegal norm type */
  }
  return 0;
}

static int search_ident(FILE *fp, char *line, int len, 
                        int *type, int *err){
  int   elen;
  char *cp, *charp, *tail;

  while(fgets(line, len, fp) != NULL){
#ifdef _DEBUG_LOAD_COMMON_AREA
    printf("search_ident[%s]\n",line);
#endif
    charp = line;
    SKIP(charp);
    if(*charp++ == IDENT){
      cp = npe_strtok_r(charp, DELIMITER, &tail);
      *type = npe_element_type(cp);
      elen = npe_element_length(*type);
      if(elen == -1){
        *err = 102; /* illegal element type */
        return 0;
      }
      return 1;
    }
  }
  *err = 0;
  return 0; /* reached end of file*/
}

/* set the pointer to next line */
static int  next_line(FILE *fp, char *line, int len){
  long  addr;
  char *charp;
  while(1){
    addr  = ftell(fp);
    charp = fgets(line, len, fp);
    if(charp == NULL)
      return -1;
#ifdef _DEBUG_LOAD_COMMON_AREA
    printf("next_line[%s]\n",line);
#endif
    SKIP(charp);         /* delete tab and spc */
    switch(*charp){
    case ':' : continue; /* case of comment line */
    case '%' :           /* case of element */
    case '#' :           /* stop flag ? */
      fseek(fp, addr, SEEK_SET);
      return -1;
    }
    break;
  }
  return 0;
}

#ifdef __cplusplus
}
#endif

/* end of common1.c */

