/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: store.c,v 1.4 2005/02/23 03:33:22 orrisroot Exp $ */
/**************************************************
**  store.c                                      **
**  store NPE common area file                   **
**************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "libnpec.h"

#ifdef __cplusplus
extern "C" {
#endif

int npe_common_store(npe_common_t *nc, const char *fname){
  FILE       *fp;
  unsigned int i;

  if(nc == NULL)
    return 1; /* NPE common area not initialized. */

  if(fname == NULL)
    return 101; /* Can't open NPE common area file */

  fp = fopen(fname, "w");
  if(fp == NULL)
    return 101; /* Can't open NPE common area file */

  /* method */
  if(nc->method_type != NPE_METHOD_NONE)
    fprintf(fp, "%%method\n%s\n:\n", nc->method_name);

  /* lsearch */
  if(nc->lsearch_type != NPE_LSEARCH_NONE)
    fprintf(fp, "%%lsearch\n%s,%g\n:\n", nc->lsearch_name, nc->lsearch_value);

  /* model */
  if(nc->model_type != NPE_MODEL_NONE)
    fprintf(fp, "%%model\n%d,%s\n:\n", nc->model_type, nc->model_file);

  /* penalty */
  if(nc->penalty_file != NULL)
    fprintf(fp, "%%penalty\n%s\n:\n", nc->penalty_file);

  /* init */
  if(nc->init_paramnum != 0){
    fprintf(fp, "%%init\n%d\n", nc->init_paramnum);
    for(i=0; i<nc->init_paramnum; i++){
      fprintf(fp, "%15.8g,\t%s,  %15s,   %g\n", 
              nc->estparam[i].value, nc->estparam[i].flag_name, 
              nc->estparam[i].name,  nc->estparam[i].span);
    }
    fprintf(fp, "\n:\n");
  }

  /* scale */
  if(nc->scale_type != NPE_SCALE_NONE)
    fprintf(fp, "%%scale\n%d\n:\n", nc->scale_type);

  /* term */
  if(nc->term_logic != NULL){
    fprintf(fp, "%%term\n%s", nc->term_logic);
    for(i=0; i<NPE_TERM_CRITESIZE; i++){
      if(nc->term_setbit[i] != NPE_FALSE)
        fprintf(fp, "\n%d,%g", i, nc->term_crite[i]);
    }
    fprintf(fp, "\n:\n");
  }

  /* number */
  if(nc->number_value != -1)
    fprintf(fp, "%%number\n%d\n:\n", nc->number_value);

  /* point */
  if(nc->point_value != -1)
    fprintf(fp, "%%point\n%d\n:\n", nc->point_value);

  /* data */
  if(nc->data_file != NULL)
    fprintf(fp, "%%data\n%s\n:\n", nc->data_file);

  /* weight */
  if(nc->weight_file != NULL)
    fprintf(fp, "%%weight\n%s\n:\n", nc->weight_file);

  /* result */
  if(nc->result_file != NULL)
    fprintf(fp, "%%result\n%s\n:\n", nc->result_file);

  /* history */
  if(nc->history_file != NULL)
    fprintf(fp, "%%history\n%s,%d\n:\n",
            nc->history_file, nc->history_interval);

  /* display */
  if(nc->display_type != NPE_DISPLAY_NONE)
    fprintf(fp, "%%display\n%d\n:\n", nc->display_type);

  /* norm */
  if(nc->norm_type != NPE_NORM_NONE)
    fprintf(fp, "%%norm\n%d\n:\n", nc->norm_type);

  fclose(fp);

  nc->store_flag = NPE_TRUE;
  return 0;
}


#ifdef __cplusplus
}
#endif

/* end of store.c */

