/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: cubic.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/**************************************************
**  cubic.cpp                                    **
**  parameter estimation method                  **
**************************************************/
#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"
#include "npeepriv.h"

#ifdef __cplusplus
extern "C" {
#endif

/* use in bracketing() */
#define  GOLD    1.618034
#define  GLIMIT  100.0
#define  TINY    1.0e-20

#define  CGOLD    0.3819660
#define  TOL     1.0e-12
#define  ZEPS    1.0e-10
#define  ITMAX  100

#define  SHFT(a,b,c,d)   (a)=(b); (b)=(c); (c)=(d);
#define  SIGN(a,b)  ((b) >= 0.0 ? fabs(a) : -fabs(a))
#define  MOV3(a,b,c, d,e,f)  (a)=(d);(b)=(e);(c)=(f);

/* function prototype */
static void  bracketing(double *ax, double *bx, double *cx, double *Xk);


/* function body */
void  npe_lsearch_cubic()
{
  int  i, iter, ok1, ok2;
/*  double  x0, x1, x2, x3, f1, f2,ftmp; */
  double  a, b, d = 0.0, d1, d2, du, dv, dw, dx;
  double  fu, fv, fw, fx, olde;
  double  tol1, tol2, u, u1, u2, v, w, x, xm;
  double  e = 0.0;
  double  ax, bx, cx;
  double  *Xk;

  Xk = (double*)malloc2Dim(0, NumVarParam, 'D');

/* set the both side position at the initial condition */
  ax = 0.0;
  bx = npe_common->lsearch_value;
  for(i=0;i<(int)NumVarParam;i++) 
    Xk[i] = VarParam[i]->value;

/* cx is decided in bracketing function */
  bracketing(&ax,&bx,&cx,Xk); 

/* naked cubic(Davidon) method */

  a = ( ax < cx ? ax : cx );   /* to be a < b  */
  b = ( ax > cx ? ax : cx );

  x = w = v = bx;              /* initialization */

  for (i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + x*Direction[i];
  fw = fv = fx  = npe_errorFunc( AllParam );
  dw = dv = dx = npe_differentiate(VarParam, nablaF, Direction);

  for(iter=0; iter < ITMAX; iter++){
    printf("."); fflush( stdout );
     xm = 0.5*( a+b );
     tol1 = TOL*fabs(x) + ZEPS;
     tol2 = 2.0*tol1;

     if( fabs(x-xm) <= (tol2 - 0.5*(b - a)) ){
        for (i=0;i<(int)NumVarParam;i++)
          VarParam[i]->value = Xk[i] + x*Direction[i];
        ErrorValue = fx;
        free2Dim(Xk, 0, 'D');
	printf("\n");
        return;
     }

/* interpolate by using the before updated value */
    if( fabs(e) > tol1 ){
      d1 = 2.0*( b - a );
      d2 = d1;

      if( dw != dx ) d1 = (w-x)*dx/(dx-dw);
      if( dv != dx ) d2 = (v-x)*dx/(dx-dv);

      u1 = x + d1;
      u2 = x + d2;

      ok1 = ( (a - u1)*(u1 - b) > 0.0 )&&( dx*d1 <= 0.0 );
      ok2 = ( (a - u2)*(u2 - b) > 0.0 )&&( dx*d2 <= 0.0 );
      olde = e;
      e = d;

      if( ok1 || ok2 ){
	if( ok1 && ok2 )
	  d = (fabs(d1) < fabs(d2) ? d1 : d2);
	else if( ok1 )
	  d = d1;
	else
	  d = d2;
	if( fabs(d) <= fabs(0.5*olde) ){
	  u = x+d;
	  if( ((u-a) < tol2) || ((b-u) < tol2) )
	    d = SIGN(tol1, xm-x);
	} else {
	  d = 0.5*( e = (dx >= 0.0 ? a-x : b-x));
        }
      } else {
	d = 0.5*( e = (dx >= 0.0 ? a-x : b-x));
      }
    } else {
      d = 0.5*( e = (dx >= 0.0 ? a-x : b-x));
    }
    if( fabs(d) >= tol1 ){
      u = x+d;
      for (i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );
    } else {
      u = x+SIGN(tol1,d);
      for (i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );
      if( fu > fx ){
        for (i=0;i<(int)NumVarParam;i++)
          VarParam[i]->value = Xk[i] + x*Direction[i];
        ErrorValue = fx;
        printf("\n");
        free2Dim(Xk, 0, 'D');
        return;
      }
    }

    du = npe_differentiate(VarParam, nablaF, Direction);

    if( fu <= fx ){
      if( u >= x ) a = x; else b = x;
      MOV3(v,fv,dv, w,fw,dw)
      MOV3(w,fw,dw, x,fx,dx)
      MOV3(x,fx,dx, u,fu,du)
    } else {
      if( u < x ) a = u; else b = u;
      if( (fu <= fw) || (w == x) ){
        MOV3(v,fv,dv, w,fw,dw)
        MOV3(w,fw,dw, u,fu,du)
      } else if( (fu < fv) || (v == x) || (v == w) ){
        MOV3(v,fv,dv, u,fu,du)
      }
    }
  }

  printf("\nWarrning: Over Max iteration[%d] ... why ?\n", ITMAX);

  for (i=0;i<(int)NumVarParam;i++)
      VarParam[i]->value = Xk[i] + x*Direction[i];
  ErrorValue = fx;
  free2Dim(Xk, 0, 'D');
} 


static void bracketing(double *ax, double *bx, double *cx, double *Xk)
{
  int  i;
  double  fa, fb, fc;
  double  ulim, u,r,q,fu,dum,tmp;

  for(i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*ax)*Direction[i];
  fa = npe_errorFunc( AllParam );  /* fa = func(ax) */

  for(i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*bx)*Direction[i];
  fb = npe_errorFunc( AllParam ); /* fb = func(bx) */

  if( fb > fa ){
      dum = *ax;  *ax = *bx;
      *bx = dum;
      dum = fb; fb = fa;
      fa = dum;  
  }

  *cx = (*bx) + GOLD*(*bx - *ax); /* initial estimated value of cx */

  for(i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*cx)*Direction[i];
  fc = npe_errorFunc( AllParam );    /* fc = func(cx) */

  while( fb > fc ){    /* iterate until enclose */

/* cal. u from a,b,c */
    r = (*bx - *ax)*(fb - fc);
    q = (*bx - *cx)*(fb - fa);
    if(fabs(q-r) > TINY) tmp = fabs(q-r);
    else tmp = TINY;
    if( q-r > 0.0 ) dum = fabs(tmp);
    else dum = -fabs(tmp);
    dum *= 2.0;
    u = (*bx) - ((*bx - *cx)*q - (*bx - *ax)*r)/dum;
/* avoid to divide by zero */
    ulim = (*bx) + GLIMIT*(*cx - *bx);
    if( (*bx - u)*(u - *cx) > 0.0 ){ /* b < u < c */
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam ); /* fu = func(u) */
      if( fu < fc ){           /* found a minimum value between b and c  */
        *ax = (*bx);
        *bx = u;
        fa = fb;
        fb = fu;
        return;
      } else if( fu > fb ) {   /* found a minimum value between a and u  */
        *cx = u;
        fc = fu;
        return;
      }
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );
    } else if((*cx - u)*(u - ulim) > 0.0){
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );

      if( fu < fc ){
        *bx = *cx; *cx = u;
        u = (*cx + GOLD*(*cx - *bx));

        for(i=0;i<(int)NumVarParam;i++)
          VarParam[i]->value = Xk[i] + u*Direction[i];
        tmp = npe_errorFunc( AllParam );
        fb = fc; fc = fu; fu = tmp;
      }
    } else if((u - ulim)*(ulim - *cx) >= 0.0) {

      u = ulim;
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );

    } else {
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );
    }
    *ax = *bx; *bx = *cx; *cx = u;
    fa = fb; fb = fc; fc = fu;
  }
}

#ifdef __cplusplus
}
#endif
