/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: dfp.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/**************************************************
**  dfp.cpp                                      **
**  parameter estimation method                  **
**************************************************/
#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"
#include "npeepriv.h"

#define MAXITERATION  (10000)
/* #define TINY 1.0e-10 */

#ifdef __cplusplus
extern "C" {
#endif

/* private variables */
/*  for Inverse Hessian */
static double **InvHess, *Dif_x, *Dif_nabla, *H_Dif_nabla;

/* function prototypes */
static void       quasiNewtonInit();
static void       initHessian(double  **Hesse);
static void       calcDirection();
static npe_bool_t checkDescentDir();
static npe_bool_t overRestart(int count);
static void       linearSearch();
static void       renewHessian();
static void       free_quasi_Newton();

/*********************************************************************
**  Main routine of quasi-Newton Method                             **
**********************************************************************/
int npe_method_dfp(){
  int  restartCount;
  double  Old_ErrorVal;

  /* start up message */
  printf("[ NPE ] DFP \n");

  quasiNewtonInit();
  Old_ErrorVal = ErrorValue;
  calcDirection();

  for(Step=0; Step < MAXITERATION; Step++){
    npe_display_parameters(AllParam, DispValue);
    npe_storeParamHist(AllParam, NPE_FALSE);

    /* whether stop condition has been satisfied or not */
    if(npe_term()){
      fprintf(stdout,"Esitimation is finishied.\n");
      break;
    }

    npe_scaleTrans(VarParam);

    if((!checkDescentDir())||(Old_ErrorVal<ErrorValue)){
      for(restartCount=0; ; restartCount++){
        fprintf(stderr,"I seem Hess matrix is not positive define...\n");
        fprintf(stderr,"Initilize Hess matrix\n");
        initHessian(InvHess);
        calcDirection();
        if(checkDescentDir()== NPE_TRUE)
          break;
        if(overRestart(restartCount)== NPE_TRUE){
          npe_storeParamHist(AllParam, NPE_TRUE);
          npe_cleanUp();
          free_quasi_Newton();
          return 5; /* too mary restart */
        }
      }
    }
    Old_ErrorVal = ErrorValue;
    linearSearch();  /* linear search */
    renewHessian();  /* modify the inv. Hess matrix */
    /* decide the direction by using modified Hess matrix */
    calcDirection();
  }
  npe_storeParamHist(AllParam, NPE_TRUE);
  npe_cleanUp();
  free_quasi_Newton();
  return 0; /* success */
}

/*********************************************************************
**  Subroutine of Initializition                                    **
**********************************************************************/
static void  quasiNewtonInit(){

  npe_readData();
  npe_readParam();
  npe_readTermCriterion();
  npe_readStoreFile();

  InvHess     = (double**)malloc2Dim(NumVarParam, NumVarParam, 'D');
  nablaF      = (double*)malloc2Dim(0, NumVarParam, 'D');
  Dif_x       = (double*)malloc2Dim(0, NumVarParam, 'D');
  Dif_nabla   = (double*)malloc2Dim(0, NumVarParam, 'D');
  H_Dif_nabla = (double*)malloc2Dim(0, NumVarParam, 'D');
  Direction   = (double*)malloc2Dim(0, NumVarParam, 'D');

  ErrorValue = npe_errorFunc(AllParam);
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
  npe_differentiate(VarParam, nablaF, NULL);
  initHessian(InvHess);
}


static void free_quasi_Newton(){
  int i;

  for(i=0; i < (int)NumVarParam; i++)
    free(InvHess[i]);

  free(InvHess);
  free(nablaF);
  free(Dif_x);
  free(Dif_nabla);
  free(H_Dif_nabla);
  free(Direction);
}

/*********************************************************************
**  Subroutine of Initializition of Hessian                         **
**********************************************************************/
static void  initHessian(double  **Hesse){
  int  i, j;

  for(i=0; i<(int)NumVarParam; i++){
    for(j=0; j<(int)NumVarParam; j++){
      Hesse[i][j] = 0.0;
    }
  }
  for(i=0; i<(int)NumVarParam; i++){
    Hesse[i][i] = 1.0;
  }
}

/*********************************************************************
**  Subroutine of Calculation of Direction                          **
**********************************************************************/
static void  calcDirection(){
  /* Direction : global var. */
  int  i, j;
  double d = 0.0;

  for(i=0; i<(int)NumVarParam; i++){
    Direction[i] = 0.0;
    for(j=0; j<(int)NumVarParam; j++)
      Direction[i] -= InvHess[i][j] * nablaF[j];

    d += Direction[i]*Direction[i];
  }

  if(d != 0.0){
    d = 1.0/sqrt(d);
    for(i=0; i<(int)NumVarParam; i++)
      Direction[i] *= d;
  }

}

/*********************************************************************
**  Subroutine of Checking Descent Direction                        **
**********************************************************************/
static npe_bool_t checkDescentDir(){
  int     i;
  double  d = 0.0;

  for(i=0; i<(int)NumVarParam; i++)
    d += nablaF[i] * Direction[i];

  if(d < 0.0)
    return NPE_TRUE;
  return NPE_FALSE;
}

/*********************************************************************
**  Subroutine of Checking Number of Restart                        **
**********************************************************************/
static npe_bool_t overRestart(int count){
  if(count >= 3)  return NPE_TRUE;
  return NPE_FALSE;
}


/*********************************************************************
**  Subroutine of LinearSearch                                      **
**********************************************************************/
static void  linearSearch(){
  int  i;
  double norm,norm2;

  for(i=0; i<(int)NumVarParam; i++){
    Dif_x[i] = -VarParam[i]->value;  /* Dif_x = Xk+1-Xk       */
    Dif_nabla[i] = -nablaF[i];       /* Dif_nabla = dF(Xk+1)-dF(Xk)*/
  }

  npe_interpolation();
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
  npe_differentiate(VarParam, nablaF, NULL);

  norm = 0.0;
  norm2 = 0.0;
  for(i=0; i<(int)NumVarParam; i++){
    Dif_x[i] += VarParam[i]->value;  /* Dif_x = Xk+1-Xk     */
    Dif_nabla[i] += nablaF[i];       /* Dif_nabla = dF(Xk+1)-dF(Xk)*/
    norm += Dif_nabla[i]*Dif_nabla[i];
    norm2 += Dif_x[i]*Dif_x[i];
  }
  if(norm*norm2 == 0.0){
    fprintf(stderr,"Too few changes... I can't update Hess Matrix\n");
    fprintf(stderr,"I guess the estimation is done. Now storing results\n"); 
    npe_storeParamHist(AllParam, NPE_TRUE);
    free_quasi_Newton();
    exit(0); /* success */
  }
}

/*********************************************************************
*  Subroutine to renew Inverse Hessian : DFP                         *
**********************************************************************/
static void  renewHessian(){
  int  i, j;
  double  sy ,yHy;

/*-------------------------------------------------------------------
  Hy = Hessian*y  yHy = <y,Hy>
--------------------------------------------------------------------*/

  yHy = 0.0;
  for(i=0; i<(int)NumVarParam; i++){
    H_Dif_nabla[i] = 0.0;
    for(j=0; j<(int)NumVarParam; j++)
      H_Dif_nabla[i] += InvHess[i][j]*Dif_nabla[j];

    yHy += Dif_nabla[i]*H_Dif_nabla[i];
  }

/*-------------------------------------------------------------------
  sy = Dif_x^T * Dif_nabla
--------------------------------------------------------------------*/
  sy = 0.0;
  for(i=0; i<(int)NumVarParam; i++)
    sy += Dif_x[i]*Dif_nabla[i];

/*-------------------------------------------------------------------
  InvHess = InvHess - <Hy, yH>/yHy + <Dif_x, Dif_x>/sy
--------------------------------------------------------------------*/
  for(i=0; i<(int)NumVarParam; i++){
    for(j=0; j<i; j++){
      InvHess[i][j] += - H_Dif_nabla[i]*H_Dif_nabla[j]/yHy
                       + Dif_x[i]*Dif_x[j]/sy;
      InvHess[j][i] = InvHess[i][j];
    }
  }
}

#ifdef __cplusplus
}
#endif
