/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: malloc.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/**************************************************
**  malloc.cpp --- Memory Allocation Function	 **
**************************************************/
#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"

#ifdef __cplusplus
extern "C" {
#endif
/********************************************************************/
/* memory allocate and initialize parameter estimator common memory */
/********************************************************************/
DLLEXPORT int npe_estimator_init(){
  int err;
  size_t len;
  char *commonfile;
  const char *tmpdir;
  /* allocate npe common area */
  npe_common = npe_common_new();
  if(npe_common == NULL)
    return 101; /* out of meomry */

  /* generate npe common area file path */
  tmpdir = get_tmpdir();
  len = strlen(tmpdir) + strlen(NPECOMMONAREA) + 2;
  commonfile = (char*)malloc(len);
  if(commonfile == NULL)
    return 101; /* out of memory */
  sprintf(commonfile, "%s/%s", tmpdir, NPECOMMONAREA);

  /* load npe common area */
  err = npe_common_load(npe_common, commonfile);
  free(commonfile);
  return err;
}

DLLEXPORT int npe_estimator_final(){
  return npe_common_delete(npe_common);
}

/*******************************************************************
**  'malloc2Dim()' allocates 2 dimensional array                  **
********************************************************************/
DLLEXPORT void *malloc2Dim(int m, int n, int types){
  int           i, j;
  void        **dim  = NULL;
  int         **iptr = NULL;
  float       **fptr = NULL;
  double      **dptr = NULL;
  npe_param_t **pptr = NULL;

  if(n <= 0 || m < 0)
    exit(102); /* unknown fatal error */

  j = (m == 0) ? 1 : m;

  switch(types){
  case 'I': 
    iptr = (int**)malloc(j * sizeof(int*));
    if(iptr == NULL) exit(101); /* out of memory */
    dim = (void**)iptr;
    break;
  case 'F':
    fptr = (float**)malloc(j * sizeof(float*));
    if(fptr == NULL) exit(101); /* out of memory */
    dim = (void**)fptr;
    break;
  case 'D':
    dptr = (double**)malloc(j * sizeof(double*));
    if(dptr == NULL) exit(101); /* out of memory */
    dim = (void**)dptr;
    break;
  case 'P':
    pptr = (npe_param_t**)malloc(j * sizeof(npe_param_t*));
    if(pptr == NULL) exit(101); /* out of memory */
    dim = (void**)pptr;
    break;
  default :
    exit(102); /* unknown fatal error */
  }

  for(i = 0; i < j; i++){
    switch(types){
    case 'I':
      iptr[i] = (int*)calloc(n, sizeof(int));
      if(iptr[i] == NULL) exit(101); /* out of memory */
      break;
      break;
    case 'F':
      fptr[i] = (float*)calloc(n, sizeof(float));
      if(fptr[i] == NULL) exit(101); /* out of memory */
      break;
    case 'D':
      dptr[i] = (double*)calloc(n, sizeof(double));
      if(dptr[i] == NULL) exit(101); /* out of memory */
      break;
    case 'P':
      pptr[i] = (npe_param_t*)calloc(n, sizeof(npe_param_t));
      if(pptr[i] == NULL) exit(101); /* out of memory */
      break;
    default:
      exit(102); /* unknown fatal error */
    }
  }

  return (m == 0) ? dim[0] : dim;
}

DLLEXPORT void free2Dim(void *ptr, int n, int types){
  int            i;
  int          **iptr = NULL;
  float        **fptr = NULL;
  double       **dptr = NULL;
  npe_param_t  **pptr = NULL;

  if(n < 0)
    exit(102); /* unknown fatal error */

  if(n == 0){ free(ptr); return; }

  switch(types){
  case 'I': iptr = (int**)ptr;         break;
  case 'F': fptr = (float**)ptr;       break;
  case 'D': dptr = (double**)ptr;      break;
  case 'P': pptr = (npe_param_t**)ptr; break;
  }
  for(i=0; i<n; i++){
    switch(types){
    case 'I': free(iptr[i]); break;
    case 'F': free(fptr[i]); break;
    case 'D': free(dptr[i]); break;
    case 'P': free(pptr[i]); break;
    }
  }
  switch(types){
  case 'I': free(iptr); break;
  case 'F': free(fptr); break;
  case 'D': free(dptr); break;
  case 'P': free(pptr); break;
  }
}


#ifdef __cplusplus
}
#endif
