/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: simplex.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */

#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"
#include "npeepriv.h"

#ifdef __cplusplus
extern "C" {
#endif

#define	MAXITERATION    10000
#define	ALPHA           (1.0)	/* 0 <  ALPHA      */ 
#define	BETA            (0.5)	/* 0 <  BETA  < 1  */
#define	GAMMA           (2.0)	/* 1 <  GAMMA      */
#define	Value(pp)       (pp->value*pp->scale)

/* global vars in this file */
typedef struct _npe_simplex_data_t {
  double        Pn, Qn;    /* Constants for initial Simplex            */
  npe_param_t **Simplex;   /* All      Parameters of apices of Simplex */
  npe_param_t ***vSimplex; /* Variable Parameters of apices of Simplex */

  npe_param_t  *Xo, **vXo; /* Center Point of Simplex                  */
  npe_param_t  *Xr, **vXr; /* Reflection Apex                          */
  npe_param_t  *Xe, **vXe; /* Expansion Apex                           */
  npe_param_t  *Xc, **vXc; /* Contraction Apex                         */
  unsigned int  Xh;        /* Indices of highest                       */
  unsigned int  Xs;        /* Indices of secondary                     */
  unsigned int  Xl;        /* Indices of lowest apex of Simplex        */

  double       *Err;       /* Errors                                   */
} npe_simplex_data_t;

static npe_simplex_data_t npe_simplex_data;

#define DATA(x)  npe_simplex_data.x

/* function prototypes */
static void   simplexInit();
static void   sortSimplex();
static int    oneIteration();
static double calcX(char type);
static void   reduction();
static void   paramCopy(npe_param_t **source, npe_param_t **dest);

/*********************************************************************
**  Main routine of Simplex Method                                  **
*********************************************************************/
int npe_method_simplex(){
  /* start up message */
  printf("[ NPE ] SIMPLEX \n");
  fflush(stdout);

  simplexInit();
  for(Step=0; Step < MAXITERATION; Step++){
    sortSimplex();
    npe_display_parameters(DATA(Simplex[DATA(Xl)]), DATA(Err[DATA(Xl)]));

    npe_storeParamHist(DATA(Simplex[DATA(Xl)]), NPE_FALSE);

    /* whether stop condition has been satisfied or not */
    if(npe_term()){
      fprintf(stdout,"Esitimation is finishied.\n");
      break;
    }

    oneIteration();
  }
  npe_storeParamHist(DATA(Simplex[DATA(Xl)]), NPE_TRUE);
  npe_cleanUp();
  return 0;
}

double *npe_simplex_get_errors(){
  return DATA(Err);
}

/*********************************************************************
**  Subroutine of Initializition                                    **
**********************************************************************/
static void  simplexInit()
{
  unsigned int  i, j, k;

  /*
  **  reading information
  */
  npe_readData();
  npe_readParam();
  npe_readTermCriterion();
  npe_readStoreFile();

  /*
  **  allocating variables
  */
  DATA(Simplex)  = (npe_param_t**)malloc2Dim(NumVarParam+1, 
                                             npe_common->init_paramnum, 'P');
  DATA(vSimplex) = (npe_param_t***)malloc2Dim(NumVarParam+1, NumVarParam, 'I');
  DATA(Xo)  = (npe_param_t  *)malloc2Dim(0, npe_common->init_paramnum, 'P');
  DATA(vXo) = (npe_param_t **)malloc2Dim(0, NumVarParam,   'I');
  DATA(Xr)  = (npe_param_t  *)malloc2Dim(0, npe_common->init_paramnum, 'P');
  DATA(vXr) = (npe_param_t **)malloc2Dim(0, NumVarParam,   'I');
  DATA(Xe)  = (npe_param_t  *)malloc2Dim(0, npe_common->init_paramnum, 'P');
  DATA(vXe) = (npe_param_t **)malloc2Dim(0, NumVarParam,   'I');
  DATA(Xc)  = (npe_param_t  *)malloc2Dim(0, npe_common->init_paramnum, 'P');
  DATA(vXc) = (npe_param_t **)malloc2Dim(0, NumVarParam,   'I');
  DATA(Err) = (double *)malloc2Dim(0, NumVarParam+1, 'D');

  /*
  **  initializing variables
  */
  for(i=k=0; i<npe_common->init_paramnum; i++){
    DATA(Xo[i]).scale = DATA(Xr[i]).scale =
      DATA(Xe[i]).scale = DATA(Xc[i]).scale = 1.0;

    DATA(Xo[i]).flag  = DATA(Xr[i]).flag  = 
      DATA(Xe[i]).flag  = DATA(Xc[i]).flag  = AllParam[i].flag;

    if(AllParam[i].flag == NPE_PARAM_FLAG_VAR){
      DATA(vXo[k])   = &DATA(Xo[i]);
      DATA(vXr[k])   = &DATA(Xr[i]);
      DATA(vXe[k])   = &DATA(Xe[i]);
      DATA(vXc[k++]) = &DATA(Xc[i]);
    } else {
      DATA(Xo[i]).value = DATA(Xr[i]).value = 
        DATA(Xe[i]).value = DATA(Xc[i]).value = AllParam[i].value;
    }
  }

  /*
  **  Constants for initial Simplex
  */
  DATA(Pn)  = (sqrt((double)(NumVarParam + 1)) + (double)(NumVarParam - 1)) /
    (NumVarParam * sqrt(2.0));
  DATA(Qn)  = (sqrt((double)(NumVarParam + 1)) - 1) /
    (NumVarParam * sqrt(2.0));
  
  /*
  **  generating initial Simplex
  */
  for(i=0; i<NumVarParam+1; i++){
    for(j=k=0; j<npe_common->init_paramnum; j++){
      DATA(Simplex[i][j]).value = AllParam[j].value;
      DATA(Simplex[i][j]).scale = AllParam[j].scale;
      DATA(Simplex[i][j]).flag  = AllParam[j].flag;
      if(DATA(Simplex[i][j]).flag == NPE_PARAM_FLAG_VAR){
        DATA(vSimplex[i][k++]) = (npe_param_t*)&DATA(Simplex[i][j]);
      } else {
        continue;
      }
      if(i != 0){
        if(j == i-1)
          DATA(Simplex[i][j]).value += DATA(Pn);
        else
          DATA(Simplex[i][j]).value += DATA(Qn);
      }
    }
    DATA(Err[i]) = npe_errorFunc(DATA(Simplex[i]));
  }
}

/*********************************************************************
**  Subroutine of Sorting Apecies of Simplex                        **
*********************************************************************/
static void  sortSimplex()
{
  int  i, j;
  double  ErrXh, ErrXs, ErrXl;
  
  ErrXh = ErrXl = ErrXs = DATA(Err[0]);
  DATA(Xh) = DATA(Xl) = DATA(Xs) = 0;
  
  for(i=1; i<(int)NumVarParam+1; i++){
    if(DATA(Err[i]) > ErrXh){           /*** finding Xh:Highest    ***/
      ErrXs = ErrXh;
      DATA(Xs) = DATA(Xh);
      ErrXh = DATA(Err[DATA(Xh)=i]);
    } else if(DATA(Err[i]) > ErrXs){    /*** finding Xs:Secondary  ***/
      ErrXs = DATA(Err[DATA(Xs)=i]);
    }
    if(DATA(Err[i]) < ErrXl){           /*** finding Xl:Lowest     ***/
      ErrXl = DATA(Err[DATA(Xl)=i]);
    }
  }

  paramCopy(DATA(vSimplex[DATA(Xl)]), VarParam);
  ErrorValue = DATA(Err[DATA(Xl)]);           /**  Terminate   **/

  for(i=0; i<(int)NumVarParam; i++)     /*** Initializing Xo    ***/
    DATA(vXo[i])->value = 0.0;

  for(i=0; i<= (int)NumVarParam; i++){  /*** Calculating Xo     ***/
    if(i == (int)DATA(Xh))
      continue;
    for(j=0; j<(int)NumVarParam; j++)
      DATA(vXo[j])->value += Value(DATA(vSimplex[i][j]));
  }
  for(i=0; i<(int)NumVarParam; i++)
    DATA(vXo[i])->value /= NumVarParam ;
}

/*********************************************************************
**  Subroutine of Main Algorithm of Simplex Method                  **
*********************************************************************/
static int oneIteration(){
  double  ErrXr, ErrXe, ErrXc;

  ErrXr = calcX('r');

  if(ErrXr <= DATA(Err[DATA(Xs)])){
    if(ErrXr < DATA(Err[DATA(Xl)])){
      /*
      **  Reflection is better than Xl!
      */
      ErrXe = calcX('e');
      if(ErrXe < DATA(Err[DATA(Xl)])){
        /*
        **  Expansion is best!!
        */
        paramCopy(DATA(vXe), DATA(vSimplex[DATA(Xh)]));
        DATA(Err[DATA(Xh)]) = ErrXe;
        return 0;
      }
    }  
    paramCopy(DATA(vXr), DATA(vSimplex[DATA(Xh)]));
    DATA(Err[DATA(Xh)]) = ErrXr;
    return 0;
  }

  if(ErrXr < DATA(Err[DATA(Xh)])){
    /*
    **  Xr is better than Xh.
    */
    paramCopy(DATA(vXr), DATA(vSimplex[DATA(Xh)]));
    DATA(Err[DATA(Xh)]) = ErrXr;
  }

  ErrXc = calcX('c');
  if(ErrXc < DATA(Err[DATA(Xh)])){
    /*
    **  Contraction is better than Xh.
    */
    paramCopy(DATA(vXc), DATA(vSimplex[DATA(Xh)]));
    DATA(Err[DATA(Xh)]) = ErrXc;
  } else {
    reduction();
  }

  return 0;
}

/*********************************************************************
**  Subroutine Calculating 'r'eflection,                            **
**                         'e'xpansion,                             **
**                         'c'ontraction                            **
*********************************************************************/
static double calcX(char type){
  int  i;

  switch(type){
  case 'r':
    for(i=0; i<(int)NumVarParam; i++){
      DATA(vXr[i])->value = (ALPHA + 1) * Value(DATA(vXo[i]))
        - ALPHA * Value(DATA(vSimplex[DATA(Xh)][i]));
    }
    return(npe_errorFunc(DATA(Xr)));
    
  case 'e':
    for(i=0; i<(int)NumVarParam; i++){
      DATA(vXe[i])->value = (1 - GAMMA) * Value(DATA(vXo[i]))
        + GAMMA * Value(DATA(vXr[i]));
    }
    return(npe_errorFunc(DATA(Xe)));
    
  case 'c':
    for(i=0; i<(int)NumVarParam; i++){
      DATA(vXc[i])->value = (1 - BETA) * Value(DATA(vXo[i]))
        + BETA * Value(DATA(vSimplex[DATA(Xh)][i]));
    }
    return(npe_errorFunc(DATA(Xc)));
  }
  /* not rearched */
  return 0.0;
}


/*********************************************************************
**  Subrourine of reduction                                         **
**********************************************************************/
static void reduction(){
  int  i, j;
  for(i=0; i<(int)NumVarParam+1; i++){
    for(j=0; j<(int)NumVarParam; j++){
      DATA(vSimplex[i][j])->value = 
        (DATA(vSimplex[i][j])->value + DATA(vSimplex[DATA(Xl)][j])->value)/2.0;
    }
    DATA(Err[i]) = npe_errorFunc(DATA(Simplex[i]));
  }
}

/********************************************************
**  Subrourine copying parameters : 'source' -> 'dest' **
*********************************************************/
static void paramCopy(npe_param_t **source, npe_param_t **dest)
{
  int     i;

  for(i=0; i<(int)NumVarParam; i++){
    dest[i]->value = source[i]->value;
    dest[i]->scale = source[i]->scale;
  }
}

#ifdef __cplusplus
}
#endif
