/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: ssvm.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */

#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"
#include "npeepriv.h"

#ifdef __cplusplus
extern "C" {
#endif

#define MAXITERATION	(10000)

/* private values */
/*  for Hessian */
static double **Hessian, *v_k, *Dif_x, *Dif_nabla, *H_Dif_nabla;

/* private functions */
static void       quasiNewtonInit();
static void       initHessian(double  **Hesse);
static void       calcDirection();
static npe_bool_t checkDescentDir();
static npe_bool_t overRestart(int count);
static void       linearSearch();
static void       renewHessian();
static void       free_quasi_Newton();

/*********************************************************************
**  Main routine of quasi-Newton Method                             **
*********************************************************************/
int npe_method_ssvm(){
  int  restartCount=0;
  double  Old_ErrorVal;

  /* start up message */
  printf("[ NPE ] SSVM \n");
  fflush(stdout);


  quasiNewtonInit();
  Old_ErrorVal = ErrorValue;

  for(Step=0; Step < MAXITERATION; Step++){

    npe_display_parameters(AllParam, DispValue);

    npe_storeParamHist(AllParam, NPE_FALSE);

    if(npe_term()){
      fprintf(stdout,"Esitimation is finishied.\n");
      break;
    }

    npe_scaleTrans(VarParam);

    calcDirection();
    if((!(checkDescentDir()))||(Old_ErrorVal<ErrorValue)){
      fprintf(stderr,"I seem Hess matrix is not positive define...\n");
      fprintf(stderr,"Initilize Hess matrix\n");
      for(restartCount=0; ; restartCount++){
        initHessian(Hessian);
        calcDirection();
        if(checkDescentDir()) break;

        if(overRestart(restartCount)){
          npe_storeParamHist(AllParam, NPE_TRUE);
          npe_cleanUp();
          free_quasi_Newton();
          return 103; /* too many restart */
        }
      }
    }
    Old_ErrorVal = ErrorValue;
    linearSearch();
    renewHessian();
  }
  npe_storeParamHist(AllParam, NPE_TRUE);
  npe_cleanUp();
  free_quasi_Newton();
  return 0; /* success */
}

/*********************************************************************
**  Subroutine of Initializition                                    **
*********************************************************************/
static void  quasiNewtonInit(){

  npe_readData();
  npe_readParam();
  npe_readTermCriterion();
  npe_readStoreFile();

  Hessian   = (double **)malloc2Dim(NumVarParam, NumVarParam, 'D');
  nablaF    = (double *) malloc2Dim(0, NumVarParam, 'D');
  Dif_x     = (double *) malloc2Dim(0, NumVarParam, 'D');
  Dif_nabla     = (double *) malloc2Dim(0, NumVarParam, 'D');
  v_k     = (double *) malloc2Dim(0, NumVarParam, 'D');
  H_Dif_nabla    = (double *) malloc2Dim(0, NumVarParam, 'D');
  Direction = (double *) malloc2Dim(0, NumVarParam, 'D');

  ErrorValue = npe_errorFunc(AllParam);
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
  npe_differentiate(VarParam, nablaF, NULL);
  initHessian(Hessian);
}


static void free_quasi_Newton(){
  int i;

  for(i=0; i< (int)NumVarParam; i++)
    free(Hessian[i]);
  free(Hessian);
  free(nablaF);
  free(Dif_x);
  free(Dif_nabla);
  free(v_k);
  free(Direction);
  free(H_Dif_nabla);
}

/*********************************************************************
**  Subroutine of Initializition of Hessian                         **
*********************************************************************/
static void  initHessian(double  **Hesse){
  int  i, j;

  for(i=0; i<(int)NumVarParam; i++){
    for(j=0; j<(int)NumVarParam; j++){
      Hesse[i][j] = 0.0;
    }
  }
  for(i=0; i<(int)NumVarParam; i++)
    Hesse[i][i] = 1.0;
}

/*********************************************************************
**  Subroutine of Calculation of Direction                          **
*********************************************************************/
static void  calcDirection(){
  int  i, j;
  double  D;

  D = 0.0;
  for(i=0; i<(int)NumVarParam; i++){
    Direction[i] = 0.0;
    for(j=0; j<(int)NumVarParam; j++)
      Direction[i] -= Hessian[i][j] * nablaF[j];

    D += Direction[i]*Direction[i];
  }

  if(D != 0.0){
    D = 1.0/sqrt(D);
    for(i=0; i<(int)NumVarParam; i++)
      Direction[i] *= D;
  }
}

/*********************************************************************
**  Subroutine of Checking Descent Direction                        **
*********************************************************************/
static npe_bool_t checkDescentDir(){
  int  i;
  double  D_ = 0.0;

  for(i=0; i<(int)NumVarParam; i++)
    D_ += nablaF[i] * Direction[i];

  if(D_ < 0.0)  return(NPE_TRUE );
  return(NPE_FALSE);
}

/*********************************************************************
**  Subroutine of Checking Number of Restart                        **
*********************************************************************/
static npe_bool_t overRestart(int count){
  if(count >= 3) return(NPE_TRUE );
  return(NPE_FALSE);
}

/*********************************************************************
**  Subroutine of LinearSearch                                      **
**********************************************************************/
static void  linearSearch(){
  int  i;
  double norm, norm2;

  for(i=0; i<(int)NumVarParam; i++){
    Dif_x[i] = -VarParam[i]->value;  /* s = Xk+1-Xk       */
    Dif_nabla[i] = -nablaF[i];       /* y=dF(Xk+1)-dF(Xk) */
  }

  npe_interpolation();
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;

/* calculate with new parameter */
  npe_differentiate(VarParam, nablaF, NULL);

  norm = 0.0; norm2 = 0.0;
  for(i=0; i<(int)NumVarParam; i++){
    Dif_x[i] += VarParam[i]->value; /* s = Xk+1-Xk      */
    Dif_nabla[i] += nablaF[i];      /* y=dF(Xk+1)-dF(Xk) */
    norm += Dif_nabla[i]*Dif_nabla[i];
    norm2 += Dif_x[i]*Dif_x[i];
  }
  if(norm*norm2 == 0.0){
    fprintf(stderr,"Too few changes... I can't update Hess Matrix\n");
    fprintf(stderr,"I guess the estimation is done. Now storing results\n"); 
    npe_storeParamHist(AllParam, NPE_TRUE);
    free_quasi_Newton();
    exit(0); /* success */
  }
}

/********************************************************************
*  Subroutine of Calculation of Hessian : DFP(Self-Scaling)      *
*                    *
*                       Hy * yH           >s,s<  *
*  Hess   =  omega{ H - -------- + ita*<y,Hy>* >v,v< }+ -----  *
*                        <y,Hy>                 <s,y>  *
*          <s,y>      <s,g>        *
*  omega  =  (1-fai) * ------ + fai * -------        *
*          <y,Hy>      <g,Hy>        *
*      <s,y>     s                *
*  v = ----- - -----  0<=ita,fai<=1           *
*      <y,Hy>  <s,y>              *
*                    *
********************************************************************/
static void  renewHessian(){
  int  i, j;
  double  ita, omega, fai;
  double  cof1, cof2, yHy, sy, gHy, sg;

  ita = 0.7; fai = 0.3;

/*-------------------------------------------------------------
  yHy = <y,Hy>  ,  sy = <s,y>
  cof1 = 1/yHy  ,  cof2 = 1/sy
--------------------------------------------------------------*/
  yHy = 0.0;
  for(i=0; i<(int)NumVarParam; i++){
    H_Dif_nabla[i] = 0.0;
    for(j=0; j<(int)NumVarParam; j++)
      H_Dif_nabla[i] += Hessian[i][j]*Dif_nabla[j];

    yHy += Dif_nabla[i]*H_Dif_nabla[i];
  }
  sy=0.0;
  for(i=0; i<(int)NumVarParam; i++)
    sy += Dif_x[i]*Dif_nabla[i] ;

  cof1 = 1.0/yHy;
  cof2 = 1.0/sy;

/*-------------------------------------------------------------
  omega  =  (1-fai)*cof*<s,y> + fai*<s,g>/<g.Hy>
-------------------------------------------------------------*/
  gHy = 0.0;
  for(i=0; i<(int)NumVarParam; i++)
    gHy += nablaF[i]*H_Dif_nabla[i];

  sg = 0.0;
  for(i=0; i<(int)NumVarParam; i++){
    sg += Dif_x[i]*nablaF[i];
  }
  omega = (1.0 - fai)*cof1*sy + fai*sg/gHy;


/*-------------------------------------------------------------
  v = cof1*<s,y> - cof2*s  
--------------------------------------------------------------*/
  for(i=0; i<(int)NumVarParam; i++)
    v_k[i] = cof1*H_Dif_nabla[i] - cof2*Dif_x[i] ;

/*-------------------------------------------------------------
  Hess   =  omega{ H - cof1*(Hy * yH) + ita*yHy* >v,v< }
      + cof2 * >s,s<  
-------------------------------------------------------------*/


  for(i=0; i<(int)NumVarParam; i++){
    for(j=0; j<i; j++){
      Hessian[i][j] = omega*(Hessian[i][j]
                             - cof1*H_Dif_nabla[i]*H_Dif_nabla[j]
                             + ita*yHy*v_k[i]*v_k[j])
        + cof2*Dif_x[i]*Dif_x[j]; 
      Hessian[j][i] = Hessian[i][j];
    }
  }

}

#ifdef __cplusplus
}
#endif
