/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: term.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/***************************************************
**  term.cpp --- Terminate Functions & Criterions **
****************************************************/
#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"
#include "npeepriv.h"

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH MAXPATHLEN
# else
#  define _MAX_PATH 1024
# endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

/********************************************************************
** term.h Header File 
********************************************************************/
#define NPE_LOGIC_MAXGRAMMAR   7
#define NPE_LOGIC_MAXSTACK    30

#define NPE_LOGIC_NSTATE  (12)
#define NPE_LOGIC_NTERMS   (6)
#define NPE_LOGIC_NNON_T   (4)

#define  NOVAL  (0)
#define  IVAL   (1)
#define  DVAL   (2)
#define  SVAL   (3)

typedef struct _npe_logic_grammar_t {
  char *non_term;
  char *terms;
  int   len_sym;
} npe_logic_grammar_t;

typedef struct _npe_logic_cell_t {
  char  act;
  int   num;
} npe_logic_cell_t;

typedef struct _npe_logic_stack_t {
  char  node_str;
  char  node_sym;
  int   state;
}npe_logic_stack_t;

/* gloval variables */
/* terminate message */
static const char *npe_terminate_messages[] = {
  "Max Iteration :",             /* 0 */
  "Error <",                     /* 1 */
  ":",                           /* 2 */
  ":",                           /* 3 */
  ":",                           /* 4 */
  ":",                           /* 5 */
  ":",                           /* 6 */
  ":",                           /* 7 */
  ":",                           /* 8 */
  "Std. Deviation of Errors <",  /* 9 */
};


/* grammar rules */
static npe_logic_grammar_t npe_logic_grammar[NPE_LOGIC_MAXGRAMMAR];
/* parsing table */
static npe_logic_cell_t    npe_logic_table[NPE_LOGIC_NSTATE][NPE_LOGIC_NTERMS+NPE_LOGIC_NNON_T];
/* stack for parse */
static npe_logic_stack_t   npe_logic_stack[NPE_LOGIC_MAXSTACK];

static char     Postfix[NPE_LOGIC_MAXSTACK];        /* Postfix            */
static double   criterionValue[NPE_TERM_CRITESIZE]; /* Value of Criterion */

/* private functions */
/* grammar rules */
static int      npe_logic_grammar_init();
static void     npe_logic_grammar_final();
/* grammar table */
static void     npe_logic_table_init();
/* parsing */
static int      npe_logic_parse(const char *input);
static int      npe_logic_parse_where(char ch);

static npe_bool_t  is_token(int c);
static const char *get_token(const char *input, char *str, char *sym);
static npe_bool_t  termCheck(int term);

static npe_bool_t term0();
static npe_bool_t term1();
static npe_bool_t term9();


static void  getTermMessage(int id, int valType, int ival, double dval, 
                            char *sval, char *ret);
/* puts message both stdout & header of file */
static void  putTermMessage(int id, int valType, int ival, double dval,
                            char *sval);

/*******************************************************************
**  'readTermCriterion()' reads terminate criterions              **
*******************************************************************/
int npe_readTermCriterion(){
  char *input;
  int  i, criteNum;

  if(npe_common->term_logic)
    input = strdup(npe_common->term_logic);
  else
    input = strdup("");
  if(input == NULL)
    exit(101); /* out of memory */

  for(i=0; i<NPE_TERM_CRITESIZE; i++){
    criterionValue[i] = 0.0;
  }
  for(i=0; i<NPE_LOGIC_MAXSTACK; i++){
    npe_logic_stack[i].node_str = -1;
    npe_logic_stack[i].node_sym = -1;
    npe_logic_stack[i].state = 0;
  }

  if(npe_logic_grammar_init() != 0){
    npe_logic_grammar_final();
    free(input);
    exit(101); /* memory allocation error */
  }
  npe_logic_table_init();

  criteNum = npe_logic_parse(input);
  npe_logic_grammar_final();
  free(input);

  if(criteNum == -1){
    exit(111); /* illegal logical expression of term */
  }

  for(i=0; i<NPE_TERM_CRITESIZE; i++){
    if(npe_common->term_setbit[i] == NPE_TRUE)
      criterionValue[i] = npe_common->term_crite[i];
  }
  return(0);
}

/* check terminate criterion */
npe_bool_t npe_term(){
  int  ip, sp;
  npe_bool_t stk[2]; /* Stack for trans    */
  for(ip=sp=0; Postfix[ip] != '\0'; ip++){
    switch(Postfix[ip]){
    case '|':
      stk[0] = stk[0] | stk[1];
      sp = 1;
      break;
    case '&':
      stk[0] = stk[0] & stk[1];
      sp = 1;
      break;
    default:
      stk[sp++] = termCheck(Postfix[ip]-'0');
    }
  }
  return stk[0];
}

static int npe_logic_grammar_init(){
  int  i;
  
  /* Initializing array */
  for(i=0; i<NPE_LOGIC_MAXGRAMMAR; i++){
    npe_logic_grammar[i].non_term = NULL;
    npe_logic_grammar[i].terms    = NULL;
    npe_logic_grammar[i].len_sym  = 0;
  }

  /**************************
  ** Setting Grammer Table **
  ***************************/
  npe_logic_grammar[0].non_term = strdup("START");
  npe_logic_grammar[1].non_term = strdup("E");
  npe_logic_grammar[2].non_term = strdup("E");
  npe_logic_grammar[3].non_term = strdup("T");
  npe_logic_grammar[4].non_term = strdup("T");
  npe_logic_grammar[5].non_term = strdup("F");
  npe_logic_grammar[6].non_term = strdup("F");
  npe_logic_grammar[0].terms = strdup("E");      /* START -> E      */
  npe_logic_grammar[1].terms = strdup("E | T");  /* E     -> E | T  */
  npe_logic_grammar[2].terms = strdup("T");      /* E     -> T      */
  npe_logic_grammar[3].terms = strdup("T & F");  /* T     -> T & F  */
  npe_logic_grammar[4].terms = strdup("F");      /* T     -> F      */
  npe_logic_grammar[5].terms = strdup("( E )");  /* F     -> ( E )  */
  npe_logic_grammar[6].terms = strdup("*");      /* F     -> number */
  npe_logic_grammar[0].len_sym = 1;
  npe_logic_grammar[1].len_sym = 3;
  npe_logic_grammar[2].len_sym = 1;
  npe_logic_grammar[3].len_sym = 3;
  npe_logic_grammar[4].len_sym = 1;
  npe_logic_grammar[5].len_sym = 3;
  npe_logic_grammar[6].len_sym = 1;

  for(i=0;i<NPE_LOGIC_MAXGRAMMAR;i++){
    if(npe_logic_grammar[i].non_term == NULL) return -1;
    if(npe_logic_grammar[i].terms    == NULL) return -1;
  }
  return 0;
}

static void npe_logic_grammar_final(){
  int i;
  for(i=0; i < NPE_LOGIC_MAXGRAMMAR; i++){
    if(npe_logic_grammar[i].non_term) free(npe_logic_grammar[i].non_term);
    if(npe_logic_grammar[i].terms) free(npe_logic_grammar[i].terms);
    npe_logic_grammar[i].non_term = NULL;
    npe_logic_grammar[i].terms    = NULL;
    npe_logic_grammar[i].len_sym  = 0;
  }
}

static void  npe_logic_table_init(){
  int  i, j, len;

  len = NPE_LOGIC_NTERMS + NPE_LOGIC_NNON_T;

  for(i=0; i < NPE_LOGIC_NSTATE; i++){
    for(j=0; j < len; j++){
      npe_logic_table[i][j].act = 'n';
      npe_logic_table[i][j].num = -1;
    }
  }
  /* parsing table setting */
  /* +----+-----------------------------------+-----------------------+ */
  /* |    |   |     &     (     )  *term    $ |  START  E     T     F | */
  /* +----+-----------------------------------+-----------------------+ */
  /* | 0: |             s 4         s 5       |         1     2     3 | */
  /* | 1: | s 6                           acc |                       | */
  /* | 2: | r 2   s 7         r 2         r 2 |                       | */
  /* | 3: | r 4   r 4         r 4         r 4 |                       | */
  /* | 4: |             s 4         s 5       |         8     2     3 | */
  /* | 5: | r 6   r 6         r 6         r 6 |                       | */
  /* | 6: |             s 4         s 5       |               9     3 | */
  /* | 7: |             s 4         s 5       |                    10 | */
  /* | 8: | s 6               s11             |                       | */
  /* | 9: | r 1   s 7         r 1         r 1 |                       | */
  /* |10: | r 3   r 3         r 3         r 3 |                       | */
  /* |11: | r 5   r 5         r 5         r 5 |                       | */
  /* +----+-----------------------------------+-----------------------+ */
  npe_logic_table[0][2].act = 's';   npe_logic_table[0][2].num = 4;
  npe_logic_table[0][4].act = 's';   npe_logic_table[0][4].num = 5;
  npe_logic_table[1][0].act = 's';   npe_logic_table[1][0].num = 6;
  npe_logic_table[2][0].act = 'r';   npe_logic_table[2][0].num = 2;
  npe_logic_table[2][1].act = 's';   npe_logic_table[2][1].num = 7;
  npe_logic_table[2][3].act = 'r';   npe_logic_table[2][3].num = 2;
  npe_logic_table[2][5].act = 'r';   npe_logic_table[2][5].num = 2;
  npe_logic_table[3][0].act = 'r';   npe_logic_table[3][0].num = 4;
  npe_logic_table[3][1].act = 'r';   npe_logic_table[3][1].num = 4;
  npe_logic_table[3][3].act = 'r';   npe_logic_table[3][3].num = 4;
  npe_logic_table[3][5].act = 'r';   npe_logic_table[3][5].num = 4;
  npe_logic_table[4][2].act = 's';   npe_logic_table[4][2].num = 4;
  npe_logic_table[4][4].act = 's';   npe_logic_table[4][4].num = 5;
  npe_logic_table[5][0].act = 'r';   npe_logic_table[5][0].num = 6;
  npe_logic_table[5][1].act = 'r';   npe_logic_table[5][1].num = 6;
  npe_logic_table[5][3].act = 'r';   npe_logic_table[5][3].num = 6;
  npe_logic_table[5][5].act = 'r';   npe_logic_table[5][5].num = 6;
  npe_logic_table[6][2].act = 's';   npe_logic_table[6][2].num = 4;
  npe_logic_table[6][4].act = 's';   npe_logic_table[6][4].num = 5;
  npe_logic_table[7][2].act = 's';   npe_logic_table[7][2].num = 4;
  npe_logic_table[7][4].act = 's';   npe_logic_table[7][4].num = 5;
  npe_logic_table[8][0].act = 's';   npe_logic_table[8][0].num = 6;
  npe_logic_table[8][3].act = 's';   npe_logic_table[8][3].num = 11;
  npe_logic_table[9][0].act = 'r';   npe_logic_table[9][0].num = 1;
  npe_logic_table[9][1].act = 's';   npe_logic_table[9][1].num = 7;
  npe_logic_table[9][3].act = 'r';   npe_logic_table[9][3].num = 1;
  npe_logic_table[9][5].act = 'r';   npe_logic_table[9][5].num = 1;
  npe_logic_table[10][0].act = 'r';  npe_logic_table[10][0].num = 3;
  npe_logic_table[10][1].act = 'r';  npe_logic_table[10][1].num = 3;
  npe_logic_table[10][3].act = 'r';  npe_logic_table[10][3].num = 3;
  npe_logic_table[10][5].act = 'r';  npe_logic_table[10][5].num = 3;
  npe_logic_table[11][0].act = 'r';  npe_logic_table[11][0].num = 5;
  npe_logic_table[11][1].act = 'r';  npe_logic_table[11][1].num = 5;
  npe_logic_table[11][3].act = 'r';  npe_logic_table[11][3].num = 5;
  npe_logic_table[11][5].act = 'r';  npe_logic_table[11][5].num = 5;
  npe_logic_table[1][5].act = 'a';
  npe_logic_table[0][7].num = 1;
  npe_logic_table[0][8].num = 2;
  npe_logic_table[0][9].num = 3;
  npe_logic_table[4][7].num = 8;
  npe_logic_table[4][8].num = 2;
  npe_logic_table[4][9].num = 3;
  npe_logic_table[6][8].num = 9;
  npe_logic_table[6][9].num = 3;
  npe_logic_table[7][9].num = 10;
}


/* used in get_token() */
static npe_bool_t is_token(int c){
  if(isdigit(c) || c == '|' || c == '&' || c == '(' || c == ')')
    return(NPE_TRUE);
  else  return(NPE_FALSE);
}


static const char *get_token(const char *input, char *str, char *sym){
  size_t i, l;
  const char *ptr;

  l = strlen(input);
  for(i=0; i<l && is_token(input[i]) == NPE_FALSE; i++);
  *sym = input[i++];
  for(; i<l && is_token(input[i]) == NPE_FALSE; i++);
  ptr = &input[i];

  if(*sym == '\0') *str = '$';
  else if(*sym == '|') *str = '|';
  else if(*sym == '&') *str = '&';
  else if(*sym == '(') *str = '(';
  else if(*sym == ')') *str = ')';
  else *str = '*';

  return ptr;
}


static int npe_logic_parse_where(char ch){
  switch(ch){
  case '|': return 0; case '&': return 1; case '(': return 2;
  case ')': return 3; case '*': return 4; case '$': return 5;
  case 'S': return 6; case 'E': return 7; case 'T': return 8;
  case 'F': return 9;
  }
  return -1;
}


static int npe_logic_parse(const char *input){
  int   sp, i, j, k, criteNum = 0;
  char  str, sym;
  const char *ptr;
  char node_str, node_sym;
  int  outp;
  ptr = get_token(input, &str, &sym);
  j = npe_logic_parse_where(str);
  i=sp=outp=0;
  while(1){
    switch(npe_logic_table[i][j].act){
    case 's':
      node_str = str;
      node_sym = sym;
      i = npe_logic_table[i][j].num;
      /* push stack */
      sp++;
      npe_logic_stack[sp].state = i;
      npe_logic_stack[sp].node_sym = node_sym;
      npe_logic_stack[sp].node_str = node_str;
      /* ********** */
      ptr = get_token(ptr, &str, &sym);
      j = npe_logic_parse_where(str);
      break;
    case 'r':
      switch(npe_logic_table[i][j].num){
      case 1: Postfix[outp++] = '|'; break;
      case 3: Postfix[outp++] = '&'; break;
      case 6: Postfix[outp++] = npe_logic_stack[sp].node_sym; 
        criteNum++; break;
      }
      node_str = *npe_logic_grammar[npe_logic_table[i][j].num].non_term;
      node_sym = -1;
      for(k=0; k<npe_logic_grammar[npe_logic_table[i][j].num].len_sym; k++){
        /* pop stack */
        npe_logic_stack[sp].node_str = -1;
        npe_logic_stack[sp].node_sym = -1;
        npe_logic_stack[sp].state = 'n';
        sp--;
        /* ********* */
      }
      k = npe_logic_parse_where(node_str);
      i = npe_logic_stack[sp].state;
      i = npe_logic_table[i][k].num;
      /* push stack */
      sp++;
      npe_logic_stack[sp].state = i;
      npe_logic_stack[sp].node_sym = node_sym;
      npe_logic_stack[sp].node_str = node_str;
      /* ********** */
      break;
    case 'a':
      Postfix[outp] = '\0';
      return criteNum;
    case 'n':
      return -1;
    }
  }
}

static npe_bool_t termCheck(int term){
  switch(term){
  case 0: return(term0());
  case 1: return(term1());
  case 9: return(term9());
  case 2: case 3: case 4: case 5:
  case 6: case 7:
  default: break;
  }
  return(NPE_FALSE);
}

/***********************************************
**   Number of Iteration >= criterion value   **
************************************************/
static npe_bool_t term0(){
  static npe_bool_t satisfy=NPE_FALSE;

  if(Step >= criterionValue[0]){
    if(satisfy == NPE_FALSE){
      satisfy = NPE_TRUE;
      putTermMessage(0, IVAL, Step, 0.0, NULL);
    }
    return(NPE_TRUE);
  }
  return(NPE_FALSE);
}


/******************************************
**   : Value of Error < criterion value  **
*******************************************/
static npe_bool_t term1(){
  static npe_bool_t satisfy=NPE_FALSE;

  if(ErrorValue < criterionValue[1]){
    if(satisfy == NPE_FALSE){
      satisfy = NPE_TRUE;
      putTermMessage(1, DVAL, 0, criterionValue[1], NULL);
    }
    return(NPE_TRUE);
  }
  return(NPE_FALSE);
}


/********************************************************
** Used by Simplex Method                              **
**   : Standard Deviation of Errors < criterion value  **
*********************************************************/
static npe_bool_t term9(){
  int  i;
  double  x, ave=0.0,tmp, *Err;
  static npe_bool_t satisfy=NPE_FALSE;

  if(npe_common->method_type != NPE_METHOD_SIMPLEX)
    return NPE_FALSE;

  /* get simplex errors */
  Err = npe_simplex_get_errors();

  x = 0.0;
  for(i=0; i<(int)NumVarParam; i++) ave += Err[i];

  x = 0.0;
  for(i=0; i<(int)NumVarParam; i++){
     tmp = Err[i]-ave;
    x += square(tmp);
  }
  x = sqrt(x / (double)(NumVarParam+1));
  if(x < criterionValue[9]){
    if(satisfy == NPE_FALSE){
      satisfy = NPE_TRUE;
      putTermMessage(9, DVAL, 0, criterionValue[9], NULL);
    }
    return NPE_TRUE;
  }
  return NPE_FALSE;
}

/*******************************************************************
*   create to print messages                                      **
**  id       is Item Number of Terminate Criterion.               **
**  valType  indicates which type of value to print,              **
**                   where NOVAL means not printing.              **
**  ival     is integer type value about the termination          **
**  dval     is double  type value about the termination          **
**  sval     is string  type value about the termination          **
*******************************************************************/
static void getTermMessage(int id, int valType, int ival, double dval, 
                           char *sval, char *ret){
  const char *mes;
  mes = npe_terminate_messages[id];
  switch(valType){            /** adds value to message    **/
  case IVAL:  sprintf(ret, "  %s %d", mes, ival);  break;
  case DVAL:  sprintf(ret, "  %s %g", mes, dval);  break;
  case SVAL:  sprintf(ret, "  %s %s", mes, sval);  break;
  case NOVAL:
  default:  break;
  }
}

size_t MessagePointer = 0;  /* is pointer of tail of criterion message */

/* puts message both stdout & header of file */
static void putTermMessage(int id, int valType, int ival, 
                           double dval, char *sval){
  char  line[80];
  Header  head;

  /******* print message to stdout ******/
  getTermMessage(id, valType, ival, dval, sval,line);
  puts(line);
  fflush(stdout);

  /******* print message to Header of file ******/
  if(LoadHeader(npe_common->history_file, &head) < 0){
    printf("Warrning: Can't open < %s >\n",npe_common->history_file);
  }

  if(MessagePointer != 0){
    strcpy(&head.comment[MessagePointer], "  ,  ");
    MessagePointer += 5;
  }
  strcpy(&head.comment[MessagePointer], line);
  StoreHeader(npe_common->history_file, &head);
  MessagePointer += strlen(line);
}

#ifdef __cplusplus
}
#endif
