/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: win32dll.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/********************************************************************
** load dynamic link library win32                                 **
********************************************************************/
#define  MOD_NPE_LIBNPEE_EXPORTS
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef WIN32
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>

#include "libsatellite.h"

#include "libnpec.h"
#include "libnpee.h"

#ifdef __cplusplus
extern "C" {
#endif

/* User defined Model Function */
#define MDLLTEMP     "_npe_model.dll"
#define PDLLTEMP     "_npe_penalty.dll"

#define NPE_FILETYPE_NONE      -1
#define NPE_FILETYPE_OBJECT     0
#define NPE_FILETYPE_CSOURCE    1
#define NPE_FILETYPE_CXXSOURCE  2
#define NPE_FILETYPE_DLL        3

typedef int  (*model_func_t)(int, double*, double*);
typedef int  (*penalty_func_t)(double*);

DLLEXPORT model_func_t   model;
DLLEXPORT penalty_func_t penalty;

static void  *model_library = NULL;
static void  *penalty_library = NULL;
static int   _npe_check_filetype(const char *fname);
static char *_npe_tmpfile_path(const char *tmpdir, const char *fname);

DLLEXPORT void  open_dlls(){
  const char *tmpdir_tmp;
  char *tmpdir, *p;
  char *model_file, *penalty_file;
  int   model_file_type, penalty_file_type;
  /* get tmpdir */
  tmpdir_tmp = get_tmpdir();
  tmpdir = strdup(tmpdir_tmp);
  if(tmpdir == NULL)
    exit(101); /* out of memory */
  for(p = tmpdir; *p!='\0'; p++) if(*p == '/') *p='\\';

  /* model object */
  model_file = NULL;
  if(npe_common->model_type == NPE_MODEL_USR){
    model_file_type = _npe_check_filetype(npe_common->model_file);
    if(model_file_type != NPE_FILETYPE_DLL){
      model_file = _npe_tmpfile_path(tmpdir, MDLLTEMP);
    }else{
      model_file = strdup(npe_common->model_file);
    }
    if(model_file == NULL)
      exit(101); /* out of memory */
    model_library = dlopen(model_file, RTLD_LAZY);
    if(model_library == NULL) exit(114);
    model = (model_func_t)dlsym(model_library, "model");
    if(model == NULL)
      exit(114); /* can't load model library */
  }

  /* penalty object */
  penalty_file = NULL;
  penalty_file_type = _npe_check_filetype(npe_common->penalty_file);
  if(penalty_file_type != NPE_FILETYPE_DLL){
    penalty_file = _npe_tmpfile_path(tmpdir, PDLLTEMP);
  }else{
    penalty_file = strdup(npe_common->penalty_file);
  }
  if(penalty_file == NULL)
    exit(101); /* out of memory */
  penalty_library = dlopen(penalty_file, RTLD_LAZY);
  if(penalty_library == NULL) exit(115);
  penalty = (penalty_func_t)dlsym(penalty_library, "penalty");
  if(penalty == NULL)
    exit(115); /* can't load penalty library */

  if(model_file)   free(model_file);
  if(penalty_file) free(penalty_file);
  free(tmpdir);
}

DLLEXPORT void close_dlls(){
  if(model_library)
    dlclose(model_library);
  if(penalty_library)
    dlclose(penalty_library);
}

static int _npe_check_filetype(const char *fname){
  char *ext;
  int   ret = NPE_FILETYPE_OBJECT;
  ext = strrchr(fname, '.');
  if(ext != NULL){
    if(npe_strincmp(ext, ".c", 3) == 0){
      ret = NPE_FILETYPE_CSOURCE;
    }else if(npe_strincmp(ext, ".cc", 4) == 0 || 
             npe_strincmp(ext, ".cxx", 5) == 0 ||
             npe_strincmp(ext, ".cpp", 5) == 0){
      ret = NPE_FILETYPE_CXXSOURCE;
    }else if(npe_strincmp(ext, ".dll", 5) == 0){
      ret = NPE_FILETYPE_DLL;
    }
  }
  return ret;
}

static char *_npe_tmpfile_path(const char *tmpdir, const char *fname){
  char *ret;
  size_t len;
  len = strlen(tmpdir) + strlen(fname) + 2;
  ret = (char*)malloc(len);
  if(ret == NULL)
    return NULL;
  sprintf(ret, "%s\\%s", tmpdir, fname);
  return ret;
}

#ifdef __cplusplus
}
#endif

#endif


