/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/******************************************************************************

    NEURAL CIRCUIT SIMULATOR SIMULATION PROGRAM                    
                       MAIN ROUTINE for NPE

    $Id: ncswrap.cpp,v 1.6 2005/02/23 04:46:26 orrisroot Exp $

*****************************************************************************/
#define LIBNPEN_NCSMAIN_SOURCE

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "libsatellite.h"

#include "libncsc.h"
#include "libncss.h"
#include "ncsmodel.h"

#include "libnpec.h"
#include "libnpee.h"

#include "ncswrap.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef void (*INTEG_METHOD)(INTEG_FUNC,double,double,int,int,double**,double*,int*);

/* private variable */
static double       **ncsg_buffer;
static INTEG_METHOD   Integrl;
static int            rr_end;
static int            calc_end, str_period;
static double         report_end;

/* private functions */
static void ncsf_strd2(int nstrd, int mem_index, double *dptr[]);
static void ncsf_dataout2(int index, long strd_cnt);
static INTEG_METHOD ncsf_integset(char integ_method);
static void cf(double t, double **ncs_y, double **ncs_dy);


static void ncsf_strd2(int nstrd, int mem_index, double *dptr[]){
  register int buff_num;
  double       f_data=0.0;
  double       data;

  for(buff_num = 0; buff_num < npe_common->number_value; buff_num++){

    data = *dptr[buff_num];

    if(data > 1.e+37){
      f_data = 1.e+37;
    }
    if(data < -1.e+37){
      f_data = -1.e+37;
    }
    if(-1.e+37 <= data && data <= 1.e+37){
      if (fabs(data) < 1.e-36){
        f_data = 0.;
      } else {
        f_data = data;
      }
    }

    ncsg_buffer[buff_num][mem_index] = f_data;
    NcsOutput[buff_num][mem_index] = data;
  }
}

/*****************************************************************************
FUNCTION  NCSF_DATAOUT
******************************************************************************
STORE THE RESULTS OF CALCULATION

MSC Ver.5.1                                      Coded by A.Anzai   06/17/1989
UNIX                                          Modifid by S.Hitomi  06/28/1991
*****************************************************************************/

static void ncsf_dataout2(int index, long strd_cnt){
  ncsf_strd2(ncsg_nstrd, index,  ncsg_strdptr);
  if((strd_cnt % ncsg_bufperiod) == 0){
    ncsf_StoreData(index, ncsg_nstrd);
  }
}

/******************************************************************************
*          Cell counter  Modify function                                      *
*                 : Calling from Main function                                *
*                                            Coded by S.Hitomi  09/05/1990    *
******************************************************************************/

static INTEG_METHOD ncsf_integset(char integ_method){
  int  type;

  for(type = 0; type < ncsg_ntout; type++){
    if(ncsg_ncout[type] > ncsg_cell){
      ncsg_ncout[type] = ncsg_cell;
    }
  }
  
  switch(integ_method){
  case 'E':
    return(EulerIntgrl);
    break;
  case 'F':
    return(AdapIntgrl);
    break;
  case 'R':
    return(RKG4Intgrl);
    break;
  default:
    break;
  }

  return(NULL);
}
  
  
  
/******************************************************************************
*          Calling from AdapIntegrl function                                  *
*                                            Coded by S.Hitomi  09/05/1990    *
******************************************************************************/
static void cf(double t, double **ncs_y, double **ncs_dy){
  NCS_TIME = t;

  /*
   * setting up of external input values 
   */
  ncsf_xinset(ncsg_nxinc, ncsg_xindata, ncsg_xfpdata, ncsg_xinfaddr);
  /*
   * NETWORK() is produced by preprocessor 
   */
  /*fprintf(stdout,"time:%f\n",t);*/
  NETWORK(t, ncs_y, ncs_dy);
  /* update memory contents */
  ncsf_update(ncsg_ntout, ncsg_ncout, ncsm_outptr, ncsm_subptr);
}



/******************************************************************************
*          NCS Simulation Program Main function                               *
*                                            Coded by S.Hitomi  09/05/1990    *
******************************************************************************/

void ncsReadConditionFile(){
#ifdef NCS_MODEL_DYNAMIC_LOADING
  if(ncsd_model_open() != 0){
    exit(114); /* can't load model library */
  }
#endif
  /*
   * setting work in DLL
   */
  ncsf_workset();
  /*
   * setting up of constants (This function is produced by
   * preprocessor.) 
   */
  ncsf_constset();

  /* preparation of exinput buffer information */
  ncsg_buff_info = plist_create(PL_ERROR_DISP);
  if(ncsg_buff_info == NULL)
    exit(101); /* out of memory */

  ncsg_buff_info_p = plist_go_top(ncsg_buff_info);
  exbuff_num = 0;

  /* reading of simulation conditions */
  if(ncsf_scfread() == NCS_ERROR){
    exit(113);
  }

  /* setting up of simulation conditions */
  ncsf_scset(&data_point, &calc_end, &str_period, &report_end);
  /* setting up of integration conditions */

  /* Integration Information Setting    */
  Integrl = ncsf_integset(ncsg_intg_method);

  /* Memory Allocation ... y, yp, buffer */
  ncsf_memalloc(ncsg_nstrd, data_point);
  ncsg_buffer = ncs_buffer;

  /* initializing states of components */
  ncsf_initset();
}

void ncsxstart1(){
  int    index = 0;
  int    iflag;
  double time;
  long   calc_cnt, strd_cnt;

  /*
   * initial value set for Integration (This function is produced by
   * preprocessor) 
   */
  initvalue(ncs_y);
  /* ----------------------------------------------------------------- */
  /* Start of Calculation for Time = 0                 */
  /* ----------------------------------------------------------------- */
  /* Integration taking one step (initialize) */
  iflag = 1;

  ncsg_cdpoint = 0;
  time = 0.0;

  /* Initialize of Module output (time = -1) */
  ncsg_delay_flag = OUTPUT_INIT;
  cf(time, ncs_y, ncs_yp);

  /* Initialize of queue */
  ncsg_delay_flag = DELAY_INIT_SET;
  cf(time, ncs_y, ncs_yp);

  ncsg_delay_flag = CALCULATE;

  (*Integrl)(cf, time, ncsg_calstep, ncsg_cell, ncsg_neqn, ncs_y,
         &ncsg_relerr, &iflag);

  /* storing of results into ISPP buffer area */
  strd_cnt = 0;
  ncsf_dataout2(index++, strd_cnt);

  /* ----------------------------------------------------------------- */
  /*                 End of Calculation for Time = 0                   */
  /* ----------------------------------------------------------------- */

  /* ----------------------------------------------------------------- */
  /* Start of Calculation routine                      */
  /* ----------------------------------------------------------------- */
  for(calc_cnt = 0; calc_cnt < calc_end; calc_cnt++){
    /* for interporation (Buffer Input) */
    ncsg_cdpoint = (int) calc_cnt;

    time = ((double) calc_cnt) * ncsg_calstep;
    strd_cnt = calc_cnt + 1;
    
    /*
     * Integration taking one step NETWORK() is produced by
     * preprocessor in cf(). 
     */
    (*Integrl)(cf, time, ncsg_calstep, ncsg_cell, ncsg_neqn, ncs_y,
          &ncsg_relerr, &iflag);

    /* storing of results into buffer area */
    if(strd_cnt % str_period == 0){
      ncsf_dataout2(index++, strd_cnt);
    }
  }

  /* ----------------------------------------------------------------- */
  /* Saving  data into SATELITE Data Buffer               */
  /* ----------------------------------------------------------------- */
  ncsf_StoreData(data_point, ncsg_nstrd); 

  /* ----------------------------------------------------------------- */
  /* End of Calculation routine                        */
  /* ----------------------------------------------------------------- */
}


void ncsPostProcess(){
  /* Free Memory */
  remove_all_exbuf_info(ncsg_buff_info); 
  ncsf_FreeOutBuffer();
  /* free allocated memory for global variable of simulation program */
  ncsf_workcls();
#ifdef NCS_MODEL_DYNAMIC_LOADING
  ncsd_model_close();
#endif
}

#ifdef __cplusplus
}
#endif
