/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: bm.cpp,v 1.6 2005/06/29 15:23:57 orrisroot Exp $ */

#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <process.h>
#include <malloc.h>
#include <commctrl.h>   // includes the common control header

#include "SL_macro.h"
#include "SL_cmd.h"

#define  __IMPORTSYMBOL__
#include "libsatellite.h"
#undef   __IMPORTSYMBOL__

#include "htmlhelp.h"
#include "bm.h"
#include "res/bm_res.h"

TBBUTTON tbButtons[] = {
  { 0, IDM_BM_SCALE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
  { 1, IDM_BM_POWER, TBSTATE_ENABLED, TBSTYLE_CHECK, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 2, IDM_BM_BACKWARD, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 3, IDM_BM_FORWARD, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
  { 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 4, IDM_BM_SUBMODE, TBSTATE_ENABLED, TBSTYLE_CHECK, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  //{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
  { 5, IDM_BM_ABOUT, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0}
};


#define VERT_MESH    4
#define HORZ_MESH    5

#define AXIS_X          0
#define AXIS_Y          1
#define AXIS_LINEAR     0
#define AXIS_LOG        1
#define AXIS_AUTO       0
#define AXIS_FIXED      1
#define GRID_OFF        0
#define GRID_ON         1

#define TIME_SERIES     0
#define POWER_SPECTRUM  1

#ifdef __cplusplus
extern "C" {
#endif

// data
typedef struct _bm_data_tag {
  int		  update;
  int         disp_type;  // time series or power
  symbol_t   *sym;
  SL_Object  *buf_obj;
  int         buf_tsize;
  int         buf_size;
  double     *buf_data;
  int         buf_dim;
  int         buf_index[MAX_INDEX];
  int         cur_dim;
  double	 *disp_data;
  int		 submode;
  // power
  int         nfft;
  Buffer     *re;
  Buffer     *im;
  Buffer     *power;
} bm_data_tag;

// scale
typedef struct _bm_scale_tag {
  // x-axis
  int    xtype;
  int    xmode;
  char   xmax[32];
  char   xmin[32];
  int    xgrid;
  // y-axis
  int    ytype;
  int    ymode;
  char   ymax[32];
  char   ymin[32];
  int    ygrid;
} bm_scale_tag ;

typedef struct _bm_window_context_t {
  bm_data_tag  bm_data;
  bm_scale_tag bm_scale;
  HANDLE    hThread;
  HINSTANCE hInstance;
  HANDLE    hEvent;
  HWND      hWnd;
  HWND      hWndToolbar;
  HWND      hWndEdit;
  HWND		hWndSub;
  WNDPROC   lpfnDefEdit;
} bm_window_context_t;

// global variables
static ATOM      wndClass      = 0;                // window class atom
static LPCSTR    szTitle       = "Buffer Monitor"; // title bar string
static LPCSTR    szWindowClass = "SATELLITE_BM";   // window class name
static HINSTANCE hInstance;

// graph attributes
COLORREF  bmTextColor = RGB(0,0,0);
COLORREF  bmBackColor = GetSysColor(COLOR_BTNFACE)/*RGB(255,255,255)*/;
COLORREF  bmForeColor = RGB(255,255,255);
COLORREF  bmLineColor = RGB(255,0,0);
COLORREF  bmGridColor = RGB(0,0,255);
int       bmLineStyle = PS_SOLID;
int       bmGridStyle = PS_DOT;
int       bmLineWidth = 1;
int       bmGridWidth = 1;
DWORD	  dwCookie;

// functions
DWORD WINAPI CreateBm(LPVOID lpParameter);
static bm_window_context_t *ReadBm(HINSTANCE hInstance);
static void LineDraw(HDC hdc, int xs, int ys, int xe, int ye,
                     int lineStyle, int lineWidth, COLORREF lineColor);
static void FillBackground(HDC hdc, RECT rc);
static void FillForeground(HDC hdc, RECT rc);
static void GetScale(bm_window_context_t *wcon, double *max, double *min);
static void GetArea(HDC hdc, RECT rwnd, RECT* darea, HWND hWnd);
static void SetScale(HDC hdc, RECT rect, bm_window_context_t *wcon,
                     double max, double min, double *dmax, double *dmin);
static void PaintBm(HDC hdc, RECT rect, bm_window_context_t *wcon);

static void ScaleAxis(HWND hWndScale, bm_window_context_t *wcon, int ini);
static BOOL PowerSpectrum(bm_window_context_t *wcon);
static void fft(double xRe[], double xIm[], int n);
static BOOL GetLogData(bm_window_context_t *wcon);
static BOOL IsCalculate(bm_window_context_t *wcon);

static ATOM InitApplication( HINSTANCE hInstance );

/* threads control */
#define BM_THREAD_MUTEX_NAME  "bm_thread_mutex"
static hash_table_t *bm_threads;
static HANDLE        bm_thread_mutex;
static int  bm_thread_init();
static int  bm_thread_clean();
static int  bm_thread_regist(bm_window_context_t *wcon);
static void bm_thread_delfunc(void *data, void *arg);
static void GetTitle(bm_data_tag *p, char *title);
static int GetVertMeshNum(bm_window_context_t *wcon,
		double max, double min, double *dmax, double *dmin, double *dstep);

DLLEXPORT BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD dwReason, 
                              LPVOID lpvReserved)
{
  if(dwReason == DLL_PROCESS_ATTACH){
    hInstance = hinstDLL;
    /* create bm_threads */
    if(bm_thread_init() != 0)
      return FALSE;
    /* register main window class */
    wndClass = InitApplication( hInstance );
    if(wndClass == 0){
      bm_thread_clean();
      return FALSE;
    }
  }else if(dwReason == DLL_PROCESS_DETACH){
    bm_thread_clean();
    /* unregister main window class */
    UnregisterClass(szWindowClass, hInstance);
  }
  return TRUE;
}

static void GetTitle(bm_data_tag *p, char *title)
{
	char tmp[128];
	Base_Buffer* buffer;
	int i,dim,idx[10];

	buffer = p->sym->object->GetBufferPointer();
	dim = buffer->GetDim();
	for(i = 0; i < dim; i++)
		idx[i] = buffer->GetIndex(i);

	if(dim==1)
		sprintf(tmp,"%s",p->sym->name);
	else {
	  if(p->submode)
		sprintf(tmp,"%s[%d]",p->sym->name,p->cur_dim);
	  else
		sprintf(tmp,"%s:[%d]",p->sym->name,p->cur_dim);
	}
	strcpy(title, tmp);
}

static int bm_thread_init(){
  bm_thread_mutex = CreateMutex(NULL, FALSE, BM_THREAD_MUTEX_NAME);
  if(bm_thread_mutex == NULL)
    return -1;
  bm_threads = hash_table_ptr_new();
  if(bm_threads == NULL)
    return -1;
  return 0;
}

static int bm_thread_clean(){
  HANDLE hMutex;
  hMutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, BM_THREAD_MUTEX_NAME);
  WaitForSingleObject(hMutex, INFINITE);
  hash_table_delete(bm_threads, bm_thread_delfunc, NULL);
  ReleaseMutex(hMutex);
  CloseHandle(hMutex);
  CloseHandle(bm_thread_mutex);
  return 0;
}

static int bm_thread_regist(bm_window_context_t *wcon)
{
  HANDLE hMutex;
  int code;
  hMutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, BM_THREAD_MUTEX_NAME);
  WaitForSingleObject(hMutex, INFINITE);
  code = hash_table_insert(bm_threads, wcon, wcon);
  ReleaseMutex(hMutex);
  CloseHandle(hMutex);
  return code;
}

static void bm_thread_delfunc(void *data, void *arg)
{
  bm_window_context_t *wcon;
  wcon = (bm_window_context_t*)data;
  if(wcon->hThread != INVALID_HANDLE_VALUE){
    if(wcon->hWnd != NULL)
      DestroyWindow(wcon->hWnd);
    WaitForSingleObject(wcon->hThread, INFINITE);
    CloseHandle(wcon->hThread);
  }
  free(wcon);
}

//
//new
//
static bm_window_context_t *ReadBm(HINSTANCE hInstance)
{
  int           i;
  symbol_t     *sym;
  SL_Object    *obj;
  SL_OBJ::TYPE  otype;
  bm_window_context_t *wcon;

  /* check symbol */
  sym = syscom->get_symbol(0);
  if(sym == NULL) return NULL;
  /* check symbol type */
  if(symbol_get_type(sym) != SYMBOL_TYPE_VAR) return NULL;
  /* check object */
  obj = symbol_get_object(sym);
  if(obj == NULL) return NULL;
  /* check object type */
  otype = obj->TypeofOBJ();
  if(otype != SL_OBJ::SERIES_O && otype != SL_OBJ::SNAPSHOT_O)
    return NULL;

  /* memory allocate bm window context */
  wcon = (bm_window_context_t*)malloc(sizeof(bm_window_context_t));
  if(wcon == NULL) return NULL;
  /* set initial value to bm_window_context */
  wcon->hThread           = INVALID_HANDLE_VALUE;
  wcon->hInstance         = hInstance;
  wcon->hWnd              = (HWND)NULL;
  wcon->hWndToolbar       = (HWND)NULL;
  wcon->hWndEdit          = (HWND)NULL;
  wcon->hWndSub          = (HWND)NULL;
  wcon->lpfnDefEdit       = (WNDPROC)NULL;
  wcon->bm_data.disp_type = TIME_SERIES;
  wcon->bm_data.update    = FALSE;
  wcon->bm_data.submode = FALSE;
  wcon->bm_data.sym       = sym;
  wcon->bm_data.buf_obj   = obj;
  wcon->bm_data.buf_tsize = 0;
  wcon->bm_data.buf_size  = 0;
  wcon->bm_data.buf_data  = NULL;
  wcon->bm_data.disp_data  = NULL;
  wcon->bm_data.buf_dim   = 0;
  for(i=0;i<MAX_INDEX;i++)
    wcon->bm_data.buf_index[i] = 0;
  wcon->bm_data.cur_dim   = 0;
  wcon->bm_data.re        = NULL;
  wcon->bm_data.im        = NULL;
  wcon->bm_data.power     = NULL;
  wcon->bm_scale.xtype    = AXIS_LINEAR;
  wcon->bm_scale.xmode    = AXIS_AUTO;
  strcpy(wcon->bm_scale.xmax,"10");
  strcpy(wcon->bm_scale.xmin,"-10");
  wcon->bm_scale.xgrid    = GRID_ON;
  wcon->bm_scale.ytype    = AXIS_LINEAR;
  wcon->bm_scale.ymode    = AXIS_AUTO;
  strcpy(wcon->bm_scale.ymax,"10");
  strcpy(wcon->bm_scale.ymin,"-10");
  wcon->bm_scale.ygrid    = GRID_ON;
  return wcon;
}

static void LineDraw(HDC hdc, int xs, int ys, int xe, int ye,
                     int lineStyle, int lineWidth, COLORREF lineColor)
{
  HPEN  hPen;
  HPEN  hOldPen;

  hPen = CreatePen(lineStyle, lineWidth, lineColor);
  hOldPen = (HPEN)SelectObject( hdc, hPen );
  MoveToEx( hdc, xs, ys, NULL );
  LineTo( hdc, xe, ye );
  SelectObject( hdc, hOldPen );
  DeleteObject( hPen );
}

static void FillBackground(HDC hdc, RECT rc)
{
  HBRUSH hbr;
  hbr = CreateSolidBrush(bmBackColor);
  FillRect(hdc, (const RECT *)&rc, hbr);
  DeleteObject(hbr);
}

static void FillForeground(HDC hdc, RECT rc)
{
  RECT rect;
  HBRUSH hbr;
  hbr = CreateSolidBrush(bmForeColor);
  FillRect(hdc, (const RECT *)&rc, hbr);
  DeleteObject(hbr);
  rect.left = rc.left;
  rect.top = rc.top/*+2*/;
  rect.right = rc.right;
  rect.bottom = rc.bottom/*-2*/;
  DrawEdge( hdc, &rect, EDGE_SUNKEN, BF_RECT );
}

static void GetScale(bm_window_context_t *wcon, double *max, double *min)
{
  int i,size,width,dim;
  bm_data_tag* p;
  double tmpmax=-1e-6, tmpmin=1e+6;

  p = &wcon->bm_data;
  size = p->buf_size;
  width = p->buf_dim;
  dim = p->cur_dim;
  if(p->disp_type == TIME_SERIES) {
    for(i = 0; i < size; i++) {
      if(p->disp_data[size*dim+i]>tmpmax)
        tmpmax = p->disp_data[size*dim+i];
      if(p->disp_data[size*dim+i]<tmpmin)
        tmpmin = p->disp_data[size*dim+i];
    }
  }
  else {
    for(i = 0; i < p->nfft; i++) {
      if(p->power[i]>tmpmax)
        tmpmax = p->power[i];
      if(p->power[i]<tmpmin)
        tmpmin = p->power[i];
    }
  }
  *max = tmpmax;
  *min = tmpmin;
}

static BOOL IsCalculate(bm_window_context_t *wcon)
{
  int i,size,width,dim;
  bm_data_tag* p;

  p = &wcon->bm_data;
  size = p->buf_size;
  width = p->buf_dim;
  dim = p->cur_dim;
  if(p->disp_type == TIME_SERIES) {
	/*Is log calculable ?*/
	for(i = 0; i < size; i++) {
	  if(p->buf_data[size*dim+i] <= 0.0)
		  return FALSE;
	}
  }

  return TRUE;
}

static BOOL GetLogData(bm_window_context_t *wcon)
{
  int i,size,width,dim;
  bm_data_tag* p;
  double pw;

  p = &wcon->bm_data;
  size = p->buf_size;
  width = p->buf_dim;
  dim = p->cur_dim;
  if(p->disp_type == TIME_SERIES) {
	  if(wcon->bm_scale.ytype == AXIS_LINEAR) {
		  if(wcon->bm_data.submode) {
		    // submode
			for(i = 0; i < size; i++)
			  p->disp_data[size*dim+i] = p->buf_data[dim+i*width];
		  }
		  else {
		    // normal
			for(i = 0; i < size; i++)
			  p->disp_data[size*dim+i] = p->buf_data[size*dim+i];
		  }
	  }
	  else {
		/*Is log calculable ?*/
		if(wcon->bm_data.submode) {
		  // submode
		  for(i = 0; i < size; i++) {
			if(p->buf_data[dim+i*width] <= 0.0)
				return FALSE;
		  }
		  for(i = 0; i < size; i++)
			p->disp_data[size*dim+i] = log10(p->buf_data[dim+i*width]);
		}
		else {
		  // normal
		  for(i = 0; i < size; i++) {
			if(p->buf_data[size*dim+i] <= 0.0)
				return FALSE;
		  }
		  for(i = 0; i < size; i++)
			p->disp_data[size*dim+i] = log10(p->buf_data[size*dim+i]);
		}
	  }
  }
  else {
    for(i = 0; i < p->nfft; i++){
	  pw = p->re[i] * p->re[i] + p->im[i] * p->im[i];
	  if(wcon->bm_scale.ytype == AXIS_LINEAR)
		p->power[i] = pw;
	  else {
		if(pw == 0.0)
		  p->power[i] = -120;
		else
		  p->power[i] = 10.0 * log10( pw );
	  }
	}
  }

  return TRUE;
}

static void GetArea(HDC hdc, RECT rwnd, RECT* darea, HWND hWnd)
{
  RECT bar;
  int width,height;
  GetWindowRect(hWnd, &bar);
  darea->top = (bar.bottom - bar.top);
  darea->bottom = rwnd.bottom;
  darea->left = rwnd.left;
  darea->right = rwnd.right;
  width = darea->right - darea->left;
  height = darea->bottom - darea->top;
  /*TODO*/
  darea->top += (height * 0.05);
  darea->bottom -= (height * 0.1);
  darea->left += (width * 0.15);
  darea->right -= (width * 0.05);
}

static int GetVertMeshNum(bm_window_context_t *wcon,
	double max, double min, double *dmax, double *dmin, double *dstep)
{
  int vnum,imax,imin,step;
  if(wcon->bm_scale.ytype == AXIS_LINEAR) {
	  step = (int)((max-min)/VERT_MESH+0.5);
	  *dmax = max;
	  *dmin = min;
	  *dstep = step;
	  return VERT_MESH;
  }
  /* log */
  if(wcon->bm_scale.ymode == AXIS_AUTO) {
	  step = (int)((max-min)/VERT_MESH+0.5);
  }
  else {
	  step = (int)((max-min)/VERT_MESH+0.5);
  }
  *dmax = (double)max;
  *dmin = (double)min;
  *dstep = step;
  return VERT_MESH;
}

static void SetScale(HDC hdc, RECT rect, bm_window_context_t *wcon, 
		double max, double min, double *dmax, double *dmin)
{
  int height,xs,ys,xe,ye,i,width,vmesh,ibeam;
  double dstep,hstep,str,strStep;
  SIZE siz;
  char buf[64];
  HFONT hFont,hFont1,hFontOld;

  // font
  hFont = CreateFont(12, 0,0,0,FW_NORMAL,0,0,0,SHIFTJIS_CHARSET,
                     OUT_STROKE_PRECIS,CLIP_DEFAULT_PRECIS,DRAFT_QUALITY,
                     FIXED_PITCH ,"MS Sans Serif");
  hFont1 = CreateFont(10, 0,0,0,FW_NORMAL,0,0,0,SHIFTJIS_CHARSET,
                     OUT_STROKE_PRECIS,CLIP_DEFAULT_PRECIS,DRAFT_QUALITY,
                     FIXED_PITCH ,"MS Sans Serif");
  hFontOld = (HFONT)SelectObject(hdc, hFont);
  SetBkMode(hdc, TRANSPARENT);
  // vertical
  vmesh = GetVertMeshNum(wcon, max, min, dmax, dmin, &dstep);
  height = rect.bottom - rect.top;
  hstep = (double)height / vmesh/*VERT_MESH*/;
  for(i = 0; i <= vmesh/*VERT_MESH*/; i++){
    xs = rect.left;
    xe = rect.right;
    ys = rect.top + (int)(floor(hstep*i+0.5));
    ye = ys;
    if(wcon->bm_scale.ygrid==GRID_ON){
	  if(i != 0 && i != vmesh/*VERT_MESH*/)
	    LineDraw(hdc, xs, ys, xe, ye, bmGridStyle, bmGridWidth, bmGridColor); 
    }
    //
	if(wcon->bm_scale.ytype == AXIS_LINEAR) {
	  // strStep = max - i*dstep;
	  // sprintf(buf, "%.1f", strStep);
	  // ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
		if(i == 0){
		  sprintf(buf, "%.1f", max);
		  ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
		  TextOut(hdc, xs-siz.cx, ye/*-siz.cy/2*/, buf, (int)strlen(buf));
		}else if(i == vmesh/*VERT_MESH*/){
		  sprintf(buf, "%.1f", min);
		  ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
		  TextOut(hdc, xs-siz.cx, ye-siz.cy, buf, (int)strlen(buf));
		}
	}
	else {
	  /*y-axis logarithm*/
	  // strStep = max - i*dstep;
	  // sprintf(buf, "%.1f", strStep);
	  // ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
		if(i == 0){
		  sprintf(buf, "%.1f", max);
		  ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
		  TextOut(hdc, xs-siz.cx, ye/*-siz.cy/2*/, buf, (int)strlen(buf));
		}else if(i == vmesh/*VERT_MESH*/){
		  sprintf(buf, "%.1f", min);
		  ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
		  TextOut(hdc, xs-siz.cx, ye-siz.cy, buf, (int)strlen(buf));
		}
	//	if((i % dstep) == 0) {
	//  strcpy(buf, "10");
	//  ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
	//  TextOut(hdc, xs-2*siz.cx, ye-siz.cy/2, buf, (int)strlen(buf));
	//  SelectObject(hdc, hFont1);
	//  sprintf(buf,"%d",ibeam-i);
	//  TextOut(hdc, xs-siz.cx, ye-2*siz.cy/3, buf, (int)strlen(buf));
	//  SelectObject(hdc, hFont);
	//	}
	}
  }
  // horz
  width = rect.right - rect.left;
  hstep = (double)width / HORZ_MESH;
  if(wcon->bm_scale.xmode == AXIS_AUTO){
	if(wcon->bm_data.disp_type == TIME_SERIES)
      str = wcon->bm_data.buf_size / HORZ_MESH;
    else
      str = wcon->bm_data.nfft / HORZ_MESH;
	// (HORZ_MESH > str)
	if(str < 1.0)
	  str = 1.0;
    strStep = 0.0;
  }else{
    str = (atof(wcon->bm_scale.xmax) - atof(wcon->bm_scale.xmin)) 
      / HORZ_MESH;
    strStep = atof(wcon->bm_scale.xmin);
  }
  for(i = 0; i <= HORZ_MESH; i++){
    xs = rect.left + (int)(floor(hstep*i+0.5));
    xe = xs;
    ys = rect.top;
    ye = rect.bottom;
    if(wcon->bm_scale.xgrid == GRID_ON) {
	  if(i != 0 && i != HORZ_MESH)
      LineDraw(hdc, xs, ys, xe, ye, bmGridStyle, bmGridWidth, bmGridColor);
    }
	// string width
    sprintf(buf, "%d", (int)(strStep));
    ::GetTextExtentPoint32(hdc, buf, (int)strlen(buf), &siz);
    if(i == 0){
      TextOut(hdc, xe, ye, buf, (int)strlen(buf));
    }else if(i == HORZ_MESH){
      TextOut(hdc, xe-siz.cx, ye, buf, (int)strlen(buf));
    }else{
      TextOut(hdc, xe-siz.cx/2, ye, buf, (int)strlen(buf));
    }
    strStep += str;
  }
  SelectObject(hdc, hFontOld);
  DeleteObject( hFont );
  DeleteObject( hFont1 );
}

static void PaintBm(HDC hdc, RECT rect, bm_window_context_t *wcon)
{
  int xs,ys,xe,ye,width,height,height2,ycenter,i;
  Series_Buffer *buf;
  int tsize,size,dim,sigwidth;
  double max,min,dmax,dmin;
  RECT drc;
  void* vptr;
  int pass=0;
  HRGN hRgn;

  /* check redraw symbol */
  buf=(Series_Buffer*)wcon->bm_data.buf_obj->GetBufferPointer();
  if(buf == NULL){
    DestroyWindow(wcon->hWnd);
    return;
  }
  
  buf->GetIndexInfo(&wcon->bm_data.buf_dim, wcon->bm_data.buf_index);
  /* TODO */
  if(wcon->bm_data.buf_dim != 1) {
	  if(!wcon->bm_data.submode)
        wcon->bm_data.buf_dim = wcon->bm_data.buf_index[0];
	  else
        wcon->bm_data.buf_dim = wcon->bm_data.buf_index[1];
  }

  tsize = buf->IndexSize();
  if(tsize < 1)
    return;
  if(tsize != wcon->bm_data.buf_tsize){
    if(wcon->bm_data.buf_data)
      free(wcon->bm_data.buf_data);
    if(wcon->bm_data.disp_data)
      free(wcon->bm_data.disp_data);
    wcon->bm_data.buf_data = (double *)malloc(tsize*sizeof(double)+1);
    wcon->bm_data.disp_data = (double *)malloc(tsize*sizeof(double)+1);
    wcon->bm_data.buf_tsize = tsize;
    if(wcon->bm_data.buf_dim == 1)
      wcon->bm_data.buf_size = wcon->bm_data.buf_index[0];
	else {
	  if(!wcon->bm_data.submode)
        wcon->bm_data.buf_size = wcon->bm_data.buf_index[1];
	  else
        wcon->bm_data.buf_size = wcon->bm_data.buf_index[0];
	}
    if(buf->GetDataPointer()){
      // TODO:
      vptr = memcpy(wcon->bm_data.buf_data, buf->GetDataPointer(),tsize*sizeof(double));
      vptr = memcpy(wcon->bm_data.disp_data, buf->GetDataPointer(),tsize*sizeof(double));
    }
  }
  else {
	if(wcon->bm_data.buf_dim != 1) {
	  if(!wcon->bm_data.submode)
        wcon->bm_data.buf_size = wcon->bm_data.buf_index[1];
	  else
        wcon->bm_data.buf_size = wcon->bm_data.buf_index[0];
	}
    if(buf->GetDataPointer()){
      // TODO:??????
      vptr = memcpy(wcon->bm_data.buf_data, buf->GetDataPointer(),tsize*sizeof(double));
      vptr = memcpy(wcon->bm_data.disp_data, buf->GetDataPointer(),tsize*sizeof(double));
    }
  }

  // change data
  if(wcon->bm_data.update) {
    if(wcon->bm_data.disp_type == POWER_SPECTRUM) {
        // calulatation of power
        PowerSpectrum(wcon);
	}
	if(!GetLogData(wcon)) {
		/*printf("Can't calculate.\n");*/
	}
	wcon->bm_data.update = FALSE;
  }
  // drawing
  FillBackground(hdc, rect);
  // drawing area
  GetArea(hdc, rect, &drc, wcon->hWndToolbar);
  // clear area
  FillForeground(hdc, drc); 
  // get scale
  if(wcon->bm_scale.ymode == AXIS_AUTO){
    GetScale(wcon, &max, &min);
  }else{
    max = atof(wcon->bm_scale.ymax);
    min = atof(wcon->bm_scale.ymin);
  }
  // drawing axis
  SetScale(hdc, drc, wcon, max, min, &dmax, &dmin);

  // drawing data
  hRgn = CreateRectRgn(drc.left, drc.top, drc.right, drc.bottom);
  SelectClipRgn(hdc, hRgn);
  height = drc.bottom - drc.top;
  width = drc.right - drc.left;
  height = drc.bottom - drc.top;
  height2 = (int)(height/2.0+0.5);
  ycenter = drc.top + height2;
  if(wcon->bm_data.disp_type == TIME_SERIES){
    dim = wcon->bm_data.cur_dim;
    sigwidth = wcon->bm_data.buf_dim;
    if(wcon->bm_data.buf_dim == 1)
      size = wcon->bm_data.buf_index[0];
	else {
	  if(!wcon->bm_data.submode)
        size = wcon->bm_data.buf_index[1];
	  else
        size = wcon->bm_data.buf_index[0];
	}
    for(i = 0; i < size; i++){
      xe = drc.left + (int)((double)width*i/size);
      ye = drc.top + height - (int)((double)height*(wcon->bm_data.disp_data[dim*size+i]-dmin)/(dmax-dmin));
      if(i == 0){
        LineDraw(hdc, xe, ye, xe, ye, bmLineStyle, bmLineWidth, bmLineColor);
      }else{
        LineDraw(hdc, xs, ys, xe, ye, bmLineStyle, bmLineWidth, bmLineColor);
      }
      xs = xe; ys = ye;
    }
  }else{
    size = wcon->bm_data.nfft;
    for(i = 0; i < size; i++){
      xe = drc.left + (int)((double)width*i/size);
      ye = drc.top + height - (int)((double)height*(wcon->bm_data.power[i]-dmin)/(dmax-dmin));
      if(i == 0){
        LineDraw(hdc, xe, ye, xe, ye, bmLineStyle, bmLineWidth, bmLineColor);
      }
      else {
        LineDraw(hdc, xs, ys, xe, ye,bmLineStyle, bmLineWidth, bmLineColor);
      }
      xs = xe; ys = ye;
    }
  }
  SelectClipRgn(hdc, NULL);
  DeleteObject(hRgn);
}

void bm_callback_func(int event, symbol_t *sym, void *data)
{
  RECT rect;
  bm_window_context_t *wcon;
  SL_OBJ::TYPE otype;
  HWND hwnd;
  hwnd = (HWND)data;
  wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(hwnd, 0);
  switch(event){
  case SYMBOL_EVENT_DELETE:
    /* sym was already deleted by the SL-Shell */
    wcon->bm_data.sym     = NULL;
    wcon->bm_data.buf_obj = NULL;
    PostMessage(hwnd, WM_DESTROY, 0, 0L);
    break;
  case SYMBOL_EVENT_UPDATE:
    /* check symbol type */
    if(symbol_get_type(sym) != SYMBOL_TYPE_VAR){
      DestroyWindow(hwnd);
      return;
    }
    /* check object */
    wcon->bm_data.buf_obj = symbol_get_object(sym);
    if(wcon->bm_data.buf_obj == NULL){
      DestroyWindow(hwnd);
      return;
    }
    /* check object type */
    otype = wcon->bm_data.buf_obj->TypeofOBJ();
    if(otype != SL_OBJ::SERIES_O && otype != SL_OBJ::SNAPSHOT_O){
      DestroyWindow(hwnd);
      return;
    }
    /* invalidate client rectangle */
	wcon->bm_data.update = TRUE;
    GetClientRect(hwnd, &rect);
    InvalidateRect(hwnd, &rect, TRUE);
    break;
  }
}

//
// This function relays the mouse messages from the edit box
// in order to get tool tips to work.
//
LRESULT CALLBACK EditWndProc( HWND hWnd, UINT uMessage, WPARAM wParam, 
                              LPARAM lParam)
{
  bm_window_context_t *wcon;
  wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(GetParent(hWnd), 0);
  switch(uMessage){
  case WM_MOUSEMOVE:
    MSG msg;
    HWND hWndTT;
    msg.lParam = lParam;
    msg.wParam = wParam;
    msg.message = uMessage;
    msg.hwnd = hWnd;
    hWndTT = (HWND)SendMessage(wcon->hWndToolbar, TB_GETTOOLTIPS, 0,0);
    SendMessage(hWndTT, TTM_RELAYEVENT, 0, (LPARAM)(LPMSG)&msg);
    break;
  default:
    return DefWindowProc( hWnd, uMessage, wParam, lParam );
  }
  return (CallWindowProc(wcon->lpfnDefEdit, hWnd, uMessage, wParam, lParam));
}

//
// This function relays 
//
static void ScaleAxis(HWND hWndScale, bm_window_context_t *wcon, int init)
{
  HWND hctl;
  char buf[32];
  bm_scale_tag* scl;
  scl = &wcon->bm_scale;

  if(scl->xtype == AXIS_LINEAR){
    hctl = GetDlgItem(hWndScale, IDC_BM_X_LINEAR);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_X_LOG);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
  }else{
    hctl = GetDlgItem(hWndScale, IDC_BM_X_LINEAR);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_X_LOG);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
  }
  if(scl->xmode == AXIS_AUTO){
    hctl = GetDlgItem(hWndScale, IDC_BM_X_AUTO);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_X_FIXED);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
  }else{
    hctl = GetDlgItem(hWndScale, IDC_BM_X_AUTO);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_X_FIXED);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
  }
  //hctl = GetDlgItem(hWndScale, IDC_BM_X_MAX);
  //SendMessage(hctl,WM_SETTEXT, 0, (LPARAM)scl->xmax);
  //hctl = GetDlgItem(hWndScale, IDC_BM_X_MIN);
  //SendMessage(hctl,WM_SETTEXT, 0, (LPARAM)scl->xmin);
  hctl = GetDlgItem(hWndScale, IDC_BM_X_GRID);
  if(scl->xgrid == GRID_ON){
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
  }else{
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
  }
  if(scl->ytype == AXIS_LINEAR){
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_LINEAR);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_LOG);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
  }else{
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_LINEAR);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_LOG);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
  }
  if(scl->ymode == AXIS_AUTO){
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_AUTO);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_FIXED);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
  }else{
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_AUTO);
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
    hctl = GetDlgItem(hWndScale, IDC_BM_Y_FIXED);
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
  }
  //hctl = GetDlgItem(hWndScale, IDC_BM_Y_MAX);
  //SendMessage(hctl,WM_SETTEXT, 0, (LPARAM)scl->ymax);
  //hctl = GetDlgItem(hWndScale, IDC_BM_Y_MIN);
  //SendMessage(hctl,WM_SETTEXT, 0, (LPARAM)scl->ymin);
  hctl = GetDlgItem(hWndScale, IDC_BM_Y_GRID);
  if(scl->ygrid == GRID_ON){
    SendMessage(hctl, BM_SETCHECK, BST_CHECKED, 0L);
  }else{
    SendMessage(hctl, BM_SETCHECK, BST_UNCHECKED, 0L);
  }
  if(!init){
    // xmax
    GetDlgItemText(hWndScale, IDC_BM_X_MAX, buf, 32);
    strcpy(scl->xmax, buf);
    // xmin
    GetDlgItemText(hWndScale, IDC_BM_X_MIN, buf, 32);
    strcpy(scl->xmin, buf);
    // ymax
    GetDlgItemText(hWndScale, IDC_BM_Y_MAX, buf, 32);
    strcpy(scl->ymax, buf);
    // ymin
    GetDlgItemText(hWndScale, IDC_BM_Y_MIN, buf, 32);
    strcpy(scl->ymin, buf);
  }
  // xmax
  SetDlgItemText(hWndScale, IDC_BM_X_MAX, scl->xmax);
  // xmin
  SetDlgItemText(hWndScale, IDC_BM_X_MIN, scl->xmin);
  // ymax
  SetDlgItemText(hWndScale, IDC_BM_Y_MAX, scl->ymax);
  // ymin
  SetDlgItemText(hWndScale, IDC_BM_Y_MIN, scl->ymin);
}

LRESULT CALLBACK ScaleWndProc( HWND hWndScale, UINT uMessage, WPARAM wParam,
                               LPARAM lParam)
{
  int wNotifyCode;
  bm_scale_tag* scl;
  char buf[32];
  // int win;
  bm_window_context_t *wcon;
  wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(GetParent(hWndScale), 0);
  scl = &wcon->bm_scale;
  switch(uMessage){
  case WM_INITDIALOG:
    ScaleAxis(hWndScale, wcon, TRUE);
    return TRUE/*break*/;
  case WM_COMMAND:
    wNotifyCode = HIWORD(wParam);
    switch(LOWORD(wParam)){
    case  IDC_BM_X_LINEAR:  ;
    case  IDC_BM_X_LOG:
      if(wNotifyCode == BN_CLICKED){
        if(scl->xtype == AXIS_LINEAR)
          scl->xtype = AXIS_LOG;
        else
          scl->xtype = AXIS_LINEAR;
        ScaleAxis(hWndScale, wcon, FALSE);
      }
      return TRUE/*break*/;
    case  IDC_BM_X_AUTO:  ;
    case  IDC_BM_X_FIXED:
      if(wNotifyCode == BN_CLICKED){
        if(scl->xmode == AXIS_AUTO)
          scl->xmode = AXIS_FIXED;
        else
          scl->xmode = AXIS_AUTO;
        ScaleAxis(hWndScale, wcon, FALSE);
      }
      return TRUE/*break*/;
    case  IDC_BM_Y_LINEAR:  ;
    case  IDC_BM_Y_LOG:
      if(wNotifyCode == BN_CLICKED){
		if(scl->ytype == AXIS_LINEAR) {
          /*Is log calculable ?*/
		  if(!IsCalculate(wcon)) {
			::MessageBox(hWndScale, "Illegal parameter", "error", MB_OK);
			/*return FALSE;*/
		  }
		  else
			scl->ytype = AXIS_LOG;
		}
        else
          scl->ytype = AXIS_LINEAR;
        ScaleAxis(hWndScale, wcon, FALSE);
      }
      return TRUE/*break*/;
    case  IDC_BM_X_GRID:
      if(wNotifyCode == BN_CLICKED){
        if(scl->xgrid == GRID_ON)
          scl->xgrid = GRID_OFF;
        else
          scl->xgrid = GRID_ON;
        ScaleAxis(hWndScale, wcon, FALSE);
      }
      return TRUE/*break*/;
    case  IDC_BM_Y_AUTO:  ;
    case  IDC_BM_Y_FIXED:
      if(wNotifyCode == BN_CLICKED){
        if(scl->ymode == AXIS_AUTO)
          scl->ymode = AXIS_FIXED;
        else
          scl->ymode = AXIS_AUTO;
        ScaleAxis(hWndScale, wcon, FALSE);
      }
      return TRUE/*break*/;
    case  IDC_BM_Y_GRID:
      if(wNotifyCode == BN_CLICKED){
        if(scl->ygrid == GRID_ON)
          scl->ygrid = GRID_OFF;
        else
          scl->ygrid = GRID_ON;
        ScaleAxis(hWndScale, wcon, FALSE);
      }
      return TRUE/*break*/;
    case  IDOK:
      // xmax
      GetDlgItemText(hWndScale, IDC_BM_X_MAX, buf, 32);
      strcpy(scl->xmax, buf);
      // xmin
      GetDlgItemText(hWndScale, IDC_BM_X_MIN, buf, 32);
      strcpy(scl->xmin, buf);
      // ymax
      GetDlgItemText(hWndScale, IDC_BM_Y_MAX, buf, 32);
      strcpy(scl->ymax, buf);
      // ymin
      GetDlgItemText(hWndScale, IDC_BM_Y_MIN, buf, 32);
      strcpy(scl->ymin, buf);
      // close dialog
      EndDialog(hWndScale, wParam);
      return TRUE;
    case  IDCANCEL:
      EndDialog(hWndScale, wParam);
      return TRUE;
    }
    break;
  }
  return FALSE;
}

//
//  window procedure
//
LRESULT CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam,
                             LPARAM lParam)
{
  bm_window_context_t *wcon;
  HWND hWndTT,hctrl;
  int wmId, wmEvent ;
  TOOLINFO lpToolInfo;
  LPTOOLTIPTEXT lpToolTipText;
  static CHAR szBuf[128];
  char buf[64];
  RECT   rect;
  HICON  hIcon;
  CREATESTRUCT *lpcs;
  char szTitle[128];
  Series_Buffer *dptr;
  int retcode;

  switch(message){
  case WM_CREATE:
    lpcs = (CREATESTRUCT*)lParam;
    wcon = (bm_window_context_t*)lpcs->lpCreateParams;

    // create toolbar control
    wcon->hWndToolbar = CreateToolbarEx(
      hWnd,                   // parent
      WS_CHILD|WS_BORDER|WS_VISIBLE|TBSTYLE_TOOLTIPS|CCS_ADJUSTABLE, // style
      ID_BM_TOOLBAR,          // toolbar id
      6,                      // number of bitmaps
      lpcs->hInstance,        // mod instance
      IDB_BM_TOOLBAR,         // resource id for the bitmap
      (LPCTBBUTTON)&tbButtons,// address of buttons
      16,                     // number of buttons
      16,16,                  // width & height of the buttons
      16,16,                  // width & height of the bitmaps
      sizeof(TBBUTTON));      // structure size
    if(wcon->hWndToolbar == NULL ){
      MessageBox (NULL, "Toolbar Bar not created!", NULL, MB_OK );
      break;
    }

    // Create the edit box and add it to the toolbar.
    wcon->hWndEdit = CreateWindowEx(
       0L,                     // No extended styles.
       "EDIT",                 // Class name.
       "0",                    // Default text.
       WS_CHILD|WS_BORDER|WS_VISIBLE|WS_DLGFRAME|ES_AUTOHSCROLL,
       // Styles and defaults.
       78, 0, 62, 24,          // Size and position.
       wcon->hWndToolbar,            // Parent window.
       (HMENU)IDM_BM_INDEX,    // ID.
       lpcs->hInstance,        // mod instance
       NULL );                 // No class data.
    if(wcon->hWndEdit == NULL){  

      MessageBox (NULL, "Edit Window not created!", NULL, MB_OK );
      break;
    }
    if(wcon->hWndEdit){
      // initial value
    }

    // Create the edit box and add it to the toolbar.
    //wcon->hWndSub = CreateWindowEx(
    //   0L,                     // No extended styles.
    //   "EDIT",                 // Class name.
    //   "0",                    // Default text.
    //   WS_CHILD|WS_BORDER|WS_VISIBLE|WS_DLGFRAME|ES_AUTOHSCROLL,
    //   // Styles and defaults.
    //   196, 0, 62, 24,          // Size and position.
    //   wcon->hWndToolbar,            // Parent window.
    //   (HMENU)IDM_BM_SUBINDEX,    // ID.
    //   lpcs->hInstance,        // mod instance
    //   NULL );                 // No class data.
    //if(wcon->hWndSub == NULL){  
    //MessageBox (NULL, "Subindex Window not created!", NULL, MB_OK );
    //  break;
    //}
    //if(wcon->hWndSub){
    //  // initial value
    //}

    // Set the window procedure for the combo box.
    wcon->lpfnDefEdit = (WNDPROC)GetWindowLong/*Ptr*/(wcon->hWndEdit, GWL_WNDPROC);
    SetWindowLong/*Ptr*/(wcon->hWndEdit, GWL_WNDPROC, /*DWORD_PTR*/(unsigned long)(EditWndProc));

    // Get the handle to the tooltip window.
    hWndTT = (HWND)SendMessage(wcon->hWndToolbar, TB_GETTOOLTIPS, 0, 0);
    
    if(hWndTT){
      // Fill in the TOOLINFO structure.
      lpToolInfo.cbSize = sizeof(lpToolInfo);
      lpToolInfo.uFlags = TTF_IDISHWND | TTF_CENTERTIP;
      lpToolInfo.lpszText = (LPSTR)IDM_BM_INDEX;
      lpToolInfo.hwnd = hWnd;
      lpToolInfo.uId = (UINT)wcon->hWndEdit;
      lpToolInfo.hinst = lpcs->hInstance;
      // Set up tooltips for the combo box.
      SendMessage(hWndTT, TTM_ADDTOOL, 0, (LPARAM)(LPTOOLINFO)&lpToolInfo);
    }else{
      MessageBox(NULL, "Could not get tooltip window handle.",NULL, MB_OK);
    }

    // save window context
    SetWindowLong/*Ptr*/(hWnd, 0, /*(LONG_PTR)*/(long)wcon);

    // set title
	GetTitle(&wcon->bm_data, szTitle);
    SendMessage(hWnd, WM_SETTEXT,0,(LPARAM)szTitle);
    // set icon
    hIcon = LoadIcon(lpcs->hInstance, MAKEINTRESOURCE(IDI_BM));
    SendMessage(hWnd, WM_SETICON, TRUE, (LPARAM)hIcon);

    // set callback event
    symbol_add_callback(wcon->bm_data.sym, bm_callback_func, hWnd);

    // syncronized
    SetEvent(wcon->hEvent);
	// Initilize HTML Help
	::HtmlHelp(NULL,NULL,HH_INITIALIZE,(DWORD)&dwCookie);
	// SUBMODE button 
    dptr=(Series_Buffer*)wcon->bm_data.buf_obj->GetBufferPointer();
    if(dptr != NULL){
      dptr->GetIndexInfo(&wcon->bm_data.buf_dim, wcon->bm_data.buf_index);
      if(wcon->bm_data.buf_dim == 1) {
	    // TODO
		hctrl = GetDlgItem(wcon->hWndToolbar, 4/*IDM_BM_SUBMODE*/);
		retcode = EnableWindow(hctrl, FALSE);
	  }
	}
    break ;

  case WM_SIZE:
    // Tell the toolbar to resize itself to fill the top of the window.
    wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(hWnd, 0);
    SendMessage(wcon->hWndToolbar, TB_AUTOSIZE, 0L, 0L);
    break;
    
  case WM_COMMAND:      // user menu
    wmId    = LOWORD( wParam );
    wmEvent = HIWORD( wParam );
    wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(hWnd, 0);
    switch( wmId ){
    case IDM_BM_SCALE:
      if (DialogBox(wcon->hInstance, MAKEINTRESOURCE(IDD_BM_SCALE),
                    hWnd, (DLGPROC)ScaleWndProc) == IDOK){
        // OnOK command
		wcon->bm_data.update = TRUE;
        GetClientRect(hWnd, &rect);
        InvalidateRect(hWnd, &rect, TRUE);
      }else{
        // Cancel command
      }
      break;

    case IDM_BM_POWER:
      if(wcon->bm_data.disp_type == POWER_SPECTRUM){
        wcon->bm_data.disp_type = TIME_SERIES;
      }else{
        wcon->bm_data.disp_type = POWER_SPECTRUM;
        // calulate
        PowerSpectrum(wcon);
      }
      // update
	  wcon->bm_data.update = TRUE;
      GetClientRect(hWnd, &rect);
      InvalidateRect(hWnd, &rect, TRUE);
      break;
      
    case IDM_BM_BACKWARD:
      // drecrement index
      if(wcon->bm_data.cur_dim > 0){
        wcon->bm_data.cur_dim--;
        sprintf(buf,"%d",wcon->bm_data.cur_dim);
        SendMessage(wcon->hWndEdit, WM_SETTEXT, 0, (LPARAM)buf);
        GetTitle(&wcon->bm_data, szTitle);
        SendMessage(hWnd, WM_SETTEXT,0,(LPARAM)szTitle);
        if(wcon->bm_data.disp_type == POWER_SPECTRUM)
          PowerSpectrum(wcon);
        // update
	    wcon->bm_data.update = TRUE;
        GetClientRect(hWnd, &rect);
        InvalidateRect(hWnd, &rect, TRUE);
      }
      break;
      
    case IDM_BM_FORWARD:
      // increment index
      if(wcon->bm_data.cur_dim < wcon->bm_data.buf_dim-1){
        wcon->bm_data.cur_dim++;
        sprintf(buf,"%d",wcon->bm_data.cur_dim);
        SendMessage(wcon->hWndEdit, WM_SETTEXT, 0, (LPARAM)buf);
        GetTitle(&wcon->bm_data, szTitle);
        SendMessage(hWnd, WM_SETTEXT,0,(LPARAM)szTitle);
        if(wcon->bm_data.disp_type == POWER_SPECTRUM)
          PowerSpectrum(wcon);
        // update
	    wcon->bm_data.update = TRUE;
        GetClientRect(hWnd, &rect);
        InvalidateRect(hWnd, &rect, TRUE);
      }
      break;
      
    case IDM_BM_ABOUT:
      // open help
      char  module_path[_MAX_PATH];
      char  doc_path[_MAX_PATH];
      char  *tmp;
      tmp = GetModuleDirectory(module_path, _MAX_PATH);
      sprintf(doc_path,"%s\\..\\doc\\sl4help.chm",module_path);
      ::HtmlHelp(hWnd,doc_path,HH_HELP_CONTEXT,0x10000+IDM_BM_ABOUT);
      break;
      
    case IDM_BM_SUBMODE:
      // mode
	  if(wcon->bm_data.submode == FALSE)
		wcon->bm_data.submode = TRUE;
	  else
		wcon->bm_data.submode = FALSE;
	  wcon->bm_data.cur_dim = 0;
      sprintf(buf,"%d",wcon->bm_data.cur_dim);
      SendMessage(wcon->hWndEdit, WM_SETTEXT, 0, (LPARAM)buf);
      GetTitle(&wcon->bm_data, szTitle);
      SendMessage(hWnd, WM_SETTEXT,0,(LPARAM)szTitle);
      if(wcon->bm_data.disp_type == POWER_SPECTRUM)
        PowerSpectrum(wcon);
	  // update
	  wcon->bm_data.update = TRUE;
      GetClientRect(hWnd, &rect);
      InvalidateRect(hWnd, &rect, TRUE);
      break;

    case ID_BM_EXIT:
      DestroyWindow(hWnd);
      break;
    default:
      return DefWindowProc( hWnd, message, wParam, lParam );
    }
    break ;
    
  case WM_PAINT:
    PAINTSTRUCT ps;
	RECT rect;
    wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(hWnd, 0);
    BeginPaint( hWnd, &ps );
	GetClientRect(hWnd,&rect);
    PaintBm(ps.hdc, rect, wcon);
    EndPaint( hWnd, &ps );
    break;
    
  case WM_NOTIFY:
    wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(hWnd, 0);
    switch(((LPNMHDR) lParam)->code){
    case TTN_NEEDTEXT:    
      // Display tool tip text.
      lpToolTipText = (LPTOOLTIPTEXT)lParam;
      LoadString (wcon->hInstance, 
                  lpToolTipText->hdr.idFrom,    // string ID == cmd ID
                  szBuf,
                  sizeof(szBuf));
      lpToolTipText->lpszText = szBuf;
      break;
      
    case TBN_QUERYDELETE:
      // Toolbar customization -- can we delete this button?
      return TRUE;
      break;
      
    case TBN_GETBUTTONINFO:
      // The toolbar needs information about a button.
      return FALSE;
      break;
      
    case TBN_QUERYINSERT:
      // Can this button be inserted? Just say yo.
      return TRUE;
      break;
      
    case TBN_CUSTHELP:
      // Need to display custom help.
      MessageBox(hWnd, "This help is custom.",NULL, MB_OK);
      break;
      
    case TBN_TOOLBARCHANGE:
      // Done dragging a bitmap to the toolbar.
      SendMessage(wcon->hWndToolbar, TB_AUTOSIZE, 0L, 0L);
      break;
      
    default:
      return TRUE;
      break;
    }
    return 0L;
    break;

  case WM_DESTROY:
	// HTML Help
	::HtmlHelp(NULL, NULL, HH_UNINITIALIZE, (DWORD)dwCookie);

	wcon = (bm_window_context_t*)GetWindowLong/*Ptr*/(hWnd, 0);
    if(wcon->bm_data.re) FreeBuffer(wcon->bm_data.re);
    if(wcon->bm_data.im) FreeBuffer(wcon->bm_data.im);
    if(wcon->bm_data.power) FreeBuffer(wcon->bm_data.power);
    if(wcon->bm_data.buf_data) free(wcon->bm_data.buf_data);
    if(wcon->bm_data.disp_data) free(wcon->bm_data.disp_data);
    if(wcon->bm_data.sym)
      symbol_remove_callback(wcon->bm_data.sym, bm_callback_func, hWnd);
    wcon->bm_data.re       = NULL;
    wcon->bm_data.im       = NULL;
    wcon->bm_data.power    = NULL;
    wcon->bm_data.buf_data = NULL;
    wcon->bm_data.sym      = NULL;
    wcon->bm_data.buf_obj  = NULL;
    wcon->hWnd = (HWND)NULL;
    PostQuitMessage(0);
    break;
    
  default:
    return DefWindowProc( hWnd, message, wParam, lParam );
  }
  return 0;
}


//
// regist main window calss
//
static ATOM InitApplication( HINSTANCE hInstance )
{
  WNDCLASSEX wcex ;
  
  ZeroMemory( &wcex, sizeof wcex );
  wcex.cbSize        = sizeof( WNDCLASSEX );
  wcex.style         = CS_HREDRAW | CS_VREDRAW;
  wcex.lpfnWndProc   = (WNDPROC)MainWndProc ;
  wcex.cbClsExtra    = 0;
  /* memory for bm window context */
  wcex.cbWndExtra    = 8; /* for SetWindowLongPtr() */
  wcex.hInstance     = hInstance ;
  wcex.hIcon         = LoadIcon( hInstance, "ICON" );
  wcex.hIconSm       = LoadIcon( hInstance, "SICON" );
  wcex.hCursor       = LoadCursor( NULL, IDC_ARROW );
  wcex.hbrBackground = (HBRUSH)GetStockObject( WHITE_BRUSH );
  wcex.lpszMenuName  = MAKEINTRESOURCE(IDR_BM_MAIN);
  wcex.lpszClassName = szWindowClass ;

  return RegisterClassEx( &wcex );
}


//
// careate main window
//
BOOL InitInstance( HINSTANCE hInstance, int nCmdShow,
                   bm_window_context_t *wcon)
{
  HWND hWnd ;
  hWnd = CreateWindow( szWindowClass,
                       szTitle,
                       WS_OVERLAPPEDWINDOW,
                       CW_USEDEFAULT,
                       CW_USEDEFAULT,
                       480,
                       320,
                       NULL,
                       NULL/*(HMENU)IDR_BM_MAIN*/,
                       hInstance,
                       wcon);
  if( !hWnd )
    return FALSE ;
  
  wcon->hWnd = hWnd;
  ShowWindow( hWnd, nCmdShow );
  UpdateWindow( hWnd );
  
  return TRUE ;
}

//create the Buffer Monitor
DLLEXPORT int mod_system_bm()
{
  bm_window_context_t *wcon;
  /* read arguments and initailzie bm window context */
  wcon = ReadBm(hInstance);
  if(wcon == NULL) return 1;

  // create BM window
  HANDLE hThread;
  DWORD threadID;
  
  wcon->hEvent = CreateEvent(0, TRUE, FALSE, NULL);
  hThread = CreateThread(0, 0,(LPTHREAD_START_ROUTINE)CreateBm, 
                         wcon, 0, &threadID);
  if(hThread == NULL){
    /* fail to create thread */
    CloseHandle(wcon->hEvent);
    free(wcon);
    return 1;
  }
#if 0
  /* thread priority control */
  rtval = GetThreadPriority( hThread );
  if(rtval == THREAD_PRIORITY_ERROR_RETURN){
    MessageBox(NULL,"GetThreadPriority error.","SL_Shell",MB_OK);
    return 1;
  }
  if(!SetThreadPriority(hThread, THREAD_PRIORITY_ABOVE_NORMAL)){
    MessageBox(NULL,"SetThreadPriority error.","SL_Shell",MB_OK);
    return 1;
  }
#endif
  /* wait for window creation */
  WaitForSingleObject(wcon->hEvent, INFINITE);
  CloseHandle(wcon->hEvent);

  /* regist threads */
  wcon->hThread = hThread;
  bm_thread_regist(wcon);
  
  return(0);
}

DWORD WINAPI CreateBm(LPVOID lpParameter)
{
  MSG msg;
  bm_window_context_t *wcon;
  wcon = (bm_window_context_t*)lpParameter;

  // Ensure that common control DLL is loaded
  InitCommonControls();
  
  // create main window
  if(!InitInstance(wcon->hInstance, SW_SHOW, wcon)){
    SetEvent(wcon->hEvent);
    return -1;
  }
  
  // dispatch messages
  while( GetMessage(&msg, NULL, 0, 0) ){
    TranslateMessage( &msg );
    DispatchMessage( &msg );
  }

  /* release allocated context memory */
  return 0;
}

static BOOL PowerSpectrum(bm_window_context_t *wcon)
{
  bm_data_tag *p;
  int nfft,i,dim,width;
  p = &wcon->bm_data;
  
  // clear buffer
  if(p->re)
    FreeBuffer(p->re);
  if(p->im)
    FreeBuffer(p->im);
  if(p->power)
    FreeBuffer(p->power);
  // set mode
  p->disp_type = POWER_SPECTRUM;
  // set FFT length
  nfft = p->buf_size;
  for(i = 0; nfft > 0; i++){
    nfft = nfft >> 1;
  }
  p->nfft = (int)pow(2, i);
  // set buffer
  p->re = AllocBuffer(p->nfft);
  if(!p->re)
    return FALSE;
  
  p->im = AllocBuffer(p->nfft);
  if(!p->im){
    FreeBuffer(p->re);
    return FALSE;
  }
  p->power = AllocBuffer(p->nfft);
  if(!p->power){
    FreeBuffer(p->re);
    FreeBuffer(p->im);
    return FALSE;
  }
  // set data
  dim = p->cur_dim;
  width = p->buf_dim;
  if(p->submode) {
    // submode
	for(i = 0; i < p->nfft; i++){
	  if(i < p->buf_size)
		p->re[i] = p->buf_data[dim+i*width];
	  else
		p->re[i] = 0.0;
	  p->im[i]=0.0;
	}
  }
  else {
    // normal
	for(i = 0; i < p->nfft; i++){
	  if(i < p->buf_size)
		p->re[i] = p->buf_data[dim*p->buf_size+i];
	  else
		p->re[i] = 0.0;
	  p->im[i]=0.0;
	}
  }
  //cal FFT
  fft(p->re, p->im, p->nfft);

  return TRUE;
}

static void fft(double xRe[], double xIm[], int n)
{
  int i,ip,j,k,m,me,me1,nv2,p;
  double uRe,uIm,vRe,vIm,wRe,wIm,tRe,tIm,pi;
  
  p = (int)(log10( (double)n )/log10( 2.0 )+0.5);
  nv2 = n/2;
  pi = 3.14159265358979;
  /**/
  j = 0;
  for(i=0;i<n-1;i++){
    if(j>i){
      tRe = xRe[j];  tIm = xIm[j];
      xRe[j] = xRe[i];  xIm[j] = xIm[i];
      xRe[i] = tRe;  xIm[i] = tIm;
    }
    k = nv2;
    while(j>=k){
      j -= k;  k /= 2;
    }
    j += k;
  }
  /**/
  for(m=1;m<=p;m++){
    me=1<<m;    me1=me/2;
    uRe=1.0;    uIm=0.0;
    wRe=cos(pi/me1);  wIm=-sin(pi/me1);
    for(j=0;j<me1;j++){
      for(i=j;i<n;i+=me){
        ip=i+me1;
        /**/
        tRe=xRe[ip]*uRe-xIm[ip]*uIm;
        tIm=xRe[ip]*uIm+xIm[ip]*uRe;
        xRe[ip]=xRe[i]-tRe;
        xIm[ip]=xIm[i]-tIm;
        xRe[i]+=tRe;
        xIm[i]+=tIm;
      }
      vRe=uRe*wRe-uIm*wIm;
      vIm=uRe*wIm+uIm*wRe;
      uRe=vRe;  uIm=vIm;
    }
  }
}

#ifdef __cplusplus
}
#endif
