/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: text2string.cpp,v 1.2 2005/10/27 04:24:39 orrisroot Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "SL_macro.h"
#include "SL_cmd.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _text2string_context {
  sl4_mmapfile_t *mf;
  sl4_list_t     *lst;
  sl4_string_t   *str;
} text2string_context;

static void _text2string_context_init(text2string_context *con) {
  con->mf  = NULL;
  con->lst = NULL;
  con->str = NULL;
}

static void _text2string_context_clean(text2string_context *con) {
  sl4_string_t *str;
  if(con->mf != NULL){
    if(sl4_mmapfile_ismapped(con->mf) != 0)
      sl4_mmapfile_unmap(con->mf);
    sl4_mmapfile_delete(con->mf);
    con->mf = NULL;
  }
  if(con->lst != NULL){
    while( sl4_list_size(con->lst) != 0){
      str = (sl4_string_t*)sl4_list_top(con->lst);
      if(str)
        sl4_string_delete(str);
      sl4_list_pop_front(con->lst);
    }
    sl4_list_delete(con->lst);
    con->lst = NULL;
  }
  if(con->str != NULL){
    sl4_string_delete(con->str);
    con->str = NULL;
  }
}


DLLEXPORT int slutil_text2string()
{
  size_t siz, i;
  const char *fn;  /* file name */
  int fdim, fidx[MAX_INDEX];
  int  dim, idx;

  char *mem, c;

  text2string_context con;
  _text2string_context_init(&con);

  /* get 1st argument */
  con.lst = syscom->sl4_syscom_get_string(0, &fdim, fidx);
  if( con.lst == NULL )
    return 3; /* out of memory */
  /* argument check */
  siz = IndexSize(fdim, fidx);
  if(siz != sl4_list_size(con.lst) || siz != 1){
    _text2string_context_clean(&con);
    return 2; /* file open error */
  }

  con.str = (sl4_string_t*)sl4_list_top(con.lst);
  if(con.str == NULL){
    _text2string_context_clean(&con);
    return 1; /* illegal parameter */
  }
  fn = sl4_string_get_cstr(con.str);
  if(fn == "\0"){
    _text2string_context_clean(&con);
    return 1; /* illegal parameter */
  }

  /* file mapping */
  con.mf = sl4_mmapfile_new();
  if(con.mf == NULL) return 3;
  if(sl4_mmapfile_map(con.mf, fn, SL4_MMAPFILE_RD, 0, 0) == -1){
    con.str = NULL;
    _text2string_context_clean(&con);
    return 2; /* file open error */
  }
  /* free arguments */
  while( sl4_list_size(con.lst) != 0 ){
    con.str = (sl4_string_t*)sl4_list_top(con.lst);
    if(con.str)
      sl4_string_delete(con.str);
    sl4_list_pop_front(con.lst);
  }
  sl4_list_delete(con.lst);
  con.str = NULL;
  con.lst = NULL;

  /* allocate new list */
  con.lst = sl4_list_new();
  if(con.lst == NULL){
    _text2string_context_clean(&con);
    return 3; /* out of memory */
  }
  
  mem = (char*)con.mf->mem;

  for(i=0; i<con.mf->size; i++){
    c = mem[i];
    if(c == '\n' || c == '\t' || c == '\r' || c == ' '){
      if(con.str != NULL){
        if(sl4_list_push_back(con.lst, con.str) != 0){
          _text2string_context_clean(&con);
          return 3; /* out of memory */
        }
        con.str = NULL;
      }
    }else{
      if(con.str == NULL){
        con.str = sl4_string_new();
        if(con.str == NULL){
          _text2string_context_clean(&con);
          return 3; /* out of memory */
        }
      }
      if(sl4_string_append_char(con.str, c) != 0){
        _text2string_context_clean(&con);
        return 3; /* out of memory */
      }
    }
  }

  dim = 1;
  idx = (int)sl4_list_size(con.lst);
  if( idx == 0 ) dim = 0;
  syscom->sl4_syscom_return_string(dim,&idx,con.lst);
  _text2string_context_clean(&con);
  return 0;
}

#ifdef __cplusplus
}
#endif
