/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: ad3163.cpp,v 1.1.1.1 2004/03/31 08:15:06 orrisroot Exp $ */
//
//	I/O{[hW[
//	ad3163.cpp
//
//
//	The calling syntax is:
//
//	y=adcon(interval,data_num,channel,board_num,range,mode,
//		trig_mode,trig_delay,trig_point,trig_edge)
//

#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include "SL_macro.h"
#include "SL_cmd.h"
#define  __IMPORTSYMBOL__
#include "libsatellite.h"
#undef   __IMPORTSYMBOL__
#include "resource.h"
#include "FbiAd.h"
#include <commctrl.h>   // includes the common control header

#ifdef __cplusplus
extern "C" {
#endif

static ATOM		wndClass = NULL;
static LPCSTR		szTitle = "Ad3163";	// ^Cgo[eLXg
static LPCSTR		szWindowClass = "SATELLITE_AD3163";	// EBhENX
HINSTANCE	hInstance;
static HANDLE		hAd3163Event;
static HWND		hWndToolBar;
static TOOLINFO	tbToolInfo;
static BOOL OpMode;
static HWND	hParent;
static HFONT hFont;
static COLORREF back_color = RGB(255,255,255);
static COLORREF line_color = RGB(255,0,0);
static COLORREF grid_color = RGB(0,255,0);
static COLORREF text_color = RGB(0,0,0);


static TBBUTTON tbButtons[] = {
	{ 0, IDM_AD_SCALE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 1, IDM_AD_PARAM, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 2, IDM_AD_START, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 3, IDM_AD_STOP, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
    { 4, IDM_AD_ABOUT, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0}
};

// ײݐF
static COLORREF rgb_color[8] = {
	RGB(255, 0, 0),
	RGB(0, 255, 0),
	RGB(255, 255, 0),
	RGB(0, 0, 255),
	RGB(255, 0, 255),
	RGB(0, 255, 255),
	RGB(255, 255, 255),
	RGB(0, 0, 0) };

struct ad3163_tag {
	double	freq;
	int freq_sel;
	long dtlen;
	int ch;
	int board;
	int range;
	int mode;
	int enable;		//Trigger Enable
	int point;		//Trigger Channel
	int edge;		//Trigger Edge
	double level;	//Trigger Level
	int delay;		//Trigger Delay
	Buffer *yp;
};
static struct ad3163_tag ad3163_prm;

HANDLE hDeviceHandle;

DLLEXPORT BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD dwReason, LPVOID lpvReserved){
  if (dwReason == DLL_PROCESS_ATTACH){
    hInstance = hinstDLL;
  }
  return 1;
}

static int adcon(
double yp[],
double *interval,
int *data_num,
int *channel,
int *board_num,
int *range,
int *mode,
int *trig_mode,
int *trig_delay,
int *trig_point,
int *trig_edge,
double *trig_level
)
{
	ADSMPLREQ SmplConfig;
	ADBOARDSPEC SmplInfo;
	ULONG ulSmplNum,len,addr;
	WORD wChannel,ch;
	char strDeviceName[256];
	WORD wData;
	ULONG ulRange,ulFastMode,ulTrigPoint,ulTrigMode;
	LONG lTrigDelay;
	DWORD	waittime;
	HANDLE  hHeap;
	PVOID	pAd;			// Area to store sampling data
	int	nRet;
	float fTrigLevel;
	double scale;
	WORD BASE,MAX;
	char msg[128];

	/* Parameters */
	ulSmplNum = (ULONG)(*data_num);
	wChannel = (WORD)(*channel);
	waittime = (DWORD)((double)ulSmplNum/(*interval)*10);
	/*Device name*/
	sprintf(strDeviceName, "FBIAD%d", *board_num);
	/*Device open*/
	hDeviceHandle = AdOpen(strDeviceName);
	if(hDeviceHandle == INVALID_HANDLE_VALUE) {
		if(OpMode)
			MessageBox(hParent, "AdOpen() error.", NULL, MB_OK);
		else
			printf("AdOpen() error.\n");
		return TRUE;
	}

#if 0
hEvent = CreateEvent(0, TRUE, FALSE, NULL);
if(AdSetBoardConfig(hDeviceHandle, hEvent, NULL, 0) != AD_ERROR_SUCCESS) {
	printf("AdSetBoardConfig() error.\n");
	return;
}
#endif
	/* Device Info */
	if(AdGetDeviceInfo(hDeviceHandle, &SmplInfo) != AD_ERROR_SUCCESS) {
		if(OpMode)
			MessageBox(hParent, "AdGetDeviceInfo() error.", NULL, MB_OK);
		else
			printf("AdGetDeviceInfo() error.\n");
		return TRUE;
	}
	/*Get Sampling Config*/
	if(AdGetSamplingConfig(hDeviceHandle, &SmplConfig) != AD_ERROR_SUCCESS) {
		if(OpMode)
			MessageBox(hParent, "AdGetSamplingConfig() error.", NULL, MB_OK);
		else
			printf("AdGetSamplingConfig() error.\n");
		return TRUE;
	}

	//ONbN
	SmplConfig.fSmplFreq = 0.0;
	if(AdSetSamplingConfig(hDeviceHandle, &SmplConfig) != AD_ERROR_SUCCESS) {
		if(OpMode)
			MessageBox(hParent, "AdSetSamplingConfig() error.", NULL, MB_OK);
		else
			printf("AdSetSamplingConfig() error.\n");
		return TRUE;
	}

	/*Set Sampling Config*/
	ulRange = AD_5V;	/* default */
	switch(*range) {
		case	5:	ulRange = AD_0_1V;
			scale = 1.0;
			BASE = 0x0;	MAX = 0xfff; break;
		case	4:	ulRange = AD_0_2P5V;
			scale = 2.5;
			BASE = 0x0; MAX = 0xfff;	break;
		case	3:	ulRange = AD_0_5V;
			scale = 5.0;
			BASE = 0x0; MAX = 0xfff;	break;
		case	2:	ulRange = AD_1V;
			scale = 1.0;
			BASE = 0x7ff; MAX = 0x7ff;	break;
		case	1:	ulRange = AD_2P5V;
			scale = 2.5;
			BASE = 0x7ff; MAX = 0x7ff;	break;
		case	0:	ulRange = AD_5V;	
			scale = 5.0;
			BASE = 0x7ff; MAX = 0x7ff;	break;
	}
	SmplConfig.fSmplFreq = (float)(*interval);
	SmplConfig.ulSmplNum = ulSmplNum;
	if(*mode == 0)
		ulFastMode = AD_NORMAL_MODE;
	else {
		ulFastMode = AD_FAST_MODE;
		wChannel = 1;
	}
	SmplConfig.ulFastMode = ulFastMode;
	if(wChannel == 1) {
		/*1Channel*/
		SmplConfig.ulChCount = 1;
		SmplConfig.SmplChReq[0].ulChNo = 1;
		SmplConfig.SmplChReq[0].ulRange = ulRange;
	}
	else {
		/*2Channel*/
		SmplConfig.ulChCount = 2;
		SmplConfig.SmplChReq[0].ulChNo = 1;
		SmplConfig.SmplChReq[0].ulRange = ulRange;
		SmplConfig.SmplChReq[1].ulChNo = 2;
		SmplConfig.SmplChReq[1].ulRange = ulRange;
	}
	//gK[[h
	if(*trig_mode == 0)
		ulTrigMode = AD_FREERUN;
	else {
		if((ULONG)(*trig_edge) == 0)
			ulTrigMode = AD_LEVEL_P;	//xgKvX
		else
			ulTrigMode = AD_LEVEL_M;	//xgK}CiX
	}
	SmplConfig.ulTrigMode = ulTrigMode;
	//gK[`l
	//trig_pointɃ`lݒ
	ulTrigPoint = *trig_point;
	SmplConfig.ulTrigCh = ulTrigPoint;
	//gK[fC
	lTrigDelay = *trig_delay;
	SmplConfig.lTrigDelay = lTrigDelay;
	//gK[F
	fTrigLevel = *trig_level;
	SmplConfig.fTrigLevel1 = fTrigLevel;

	if((nRet = AdSetSamplingConfig(hDeviceHandle, &SmplConfig)) != AD_ERROR_SUCCESS) {
		if(OpMode)
			MessageBox(hParent, "AdSetSamplingConfig() error.", NULL, MB_OK);
		else
			printf("AdSetSamplingConfig() error.\n");
		return TRUE;
	}

	/* Sampling start */
	//
	if(AdStartSampling(hDeviceHandle, FLAG_SYNC) != AD_ERROR_SUCCESS) {
		if(OpMode)
			MessageBox(hParent,"AdStartSampling() error.",NULL,MB_OK);
		else
			printf("AdStartSampling() error.\n");
		return TRUE;
	}
	/* Get Data */
	hHeap = GetProcessHeap();
	pAd = HeapAlloc( hHeap, HEAP_ZERO_MEMORY, ulSmplNum * wChannel * sizeof( WORD ));
	nRet = AdGetSamplingData(hDeviceHandle, pAd, &ulSmplNum);
	if((ULONG)(*data_num) != ulSmplNum) {
		if(OpMode)
			MessageBox(hParent, "AdGetSamplingData() error.", NULL, MB_OK);
		else
			printf("AdGetSamplingData() error.\n");
	}
	/* Set Data */
	addr = 0;
	for(len = 0; len < ulSmplNum; len++) {
		for(ch = 0; ch < wChannel; ch++) {
			wData = *( ( LPWORD )pAd + addr );
			yp[addr/*ch*ulSmplNum+len*/] = scale *(double)(wData - BASE)/MAX;
			addr++;
		}
	}
	HeapFree( hHeap, 0, pAd );
	/*Device close*/
	if(AdClose(hDeviceHandle) != AD_ERROR_SUCCESS) {
		if(OpMode)
			MessageBox(hParent, "AdClose() error.", NULL, MB_OK);
		else
			printf("AdClose() error.\n");
		return TRUE;
	}
	return FALSE;
}

//Display
static void Display(HWND hwnd, HDC hdc, double *databuf, long datalen, int chnum, int range)
{
	HRGN	hRgn;
	HRGN	hOldRgn;
	RECT rect, bar, drawing;
	HBRUSH	hBrush;
	HPEN	hPen;
	HGDIOBJ	hOldBrush,hOldPen;
	int ch,addr,xs,ys,xe,ye,xstart,ystart,width,height,height2,hlength;
	double scale,data;
	char upbuf[32],downbuf[32];
	SIZE siz;
    HFONT hFontOld;
	double step;
	int i;

	//̫Đݒ
	hFontOld = (HFONT)SelectObject(hdc, hFont);
	::SetMapMode(hdc, MM_TEXT);
	::SetBkMode(hdc, TRANSPARENT);
	::SetTextColor(hdc, text_color);
	hBrush = ::CreateSolidBrush( back_color );
	hOldBrush = ::SelectObject(hdc, hBrush);
	//set drawing area
	GetWindowRect(hWndToolBar, &bar);
	GetClientRect(hwnd, &rect);
	drawing.top = (bar.bottom - bar.top);
	drawing.bottom = rect.bottom;
	drawing.left = rect.left;
	drawing.right = rect.right;
	FillRect(hdc, &drawing, hBrush);
	DrawEdge( hdc, &drawing, EDGE_SUNKEN, BF_RECT );
	//size
	width = drawing.right - drawing.left;
	height = drawing.bottom - drawing.top;
	height2 = height / 2;
	xstart = 0;
	ystart = drawing.top + height2;
	hlength = height2;
	//grid
	hPen = ::CreatePen(PS_DOT, 1, grid_color);
	hOldPen = ::SelectObject(hdc, hPen);
	xs = 0; ys = drawing.top; xe = width;
	step = (double)height/4.0;
	for(i = 1; i <= 3; i++) {
		ys += (int)(step+0.5);
		::MoveToEx(hdc,xs,ys,NULL);
		::LineTo(hdc,xe,ys);
	}
	xs = drawing.left; ys = drawing.top; ye = drawing.bottom;
	step = (double)width/5.0;
	for(i = 1; i <= 4; i++) {
		xs += (int)(step+0.5);
		::MoveToEx(hdc,xs,ys,NULL);
		::LineTo(hdc,xs,ye);
	}
	SelectObject(hdc, hOldPen);
	DeleteObject(hPen);
	//
	if(!databuf) {
		SelectObject(hdc, hOldBrush);
		DeleteObject(hBrush);
		SelectObject(hdc, hFontOld);
		return;
	}

	//rangeXP[
	scale = 5.0;
	strcpy(upbuf,"5.0");
	strcpy(downbuf,"-5.0");
	switch(range) {
	case	0:
		scale = 5.0;
		strcpy(upbuf,"5.0");
		strcpy(downbuf,"-5.0");
		ystart = drawing.top+height2;
		hlength = height2;
		break;
	case	1:
		scale = 2.5;
		strcpy(upbuf,"2.5");
		strcpy(downbuf,"-2.5");
		ystart = drawing.top+height2;
		hlength = height2;
		break;
	case	2:
		scale = 1.0;
		strcpy(upbuf,"1.0");
		strcpy(downbuf,"-1.0");
		ystart = drawing.top+height2;
		hlength = height2;
		break;
	case	3:
		scale = 5.0;
		strcpy(upbuf,"5.0");
		strcpy(downbuf,"0.0");
		ystart = drawing.top+height;
		hlength = height;
		break;
	case	4:
		scale = 2.5;
		strcpy(upbuf,"2.5");
		strcpy(downbuf,"0.0");
		ystart = drawing.top+height;
		hlength = height;
		break;
	case	5:
		scale = 1.0;
		strcpy(upbuf,"1.0");
		strcpy(downbuf,"0.0");
		ystart = drawing.top+height;
		hlength = height;
		break;
	}


	//Nbv̈w
	hRgn = CreateRectRgn(drawing.left+1, drawing.top+1,
			drawing.right-1, drawing.bottom-1);
	hOldRgn = (HRGN)SelectClipRgn( hdc, hRgn);
	//g`
	for(ch = 0; ch < chnum; ch++) {
if(chnum == 1)
		hPen = ::CreatePen(PS_SOLID, 1, line_color);
else
		hPen = ::CreatePen(PS_SOLID, 1, rgb_color[ch]);

		hOldPen = ::SelectObject(hdc, hPen);
		xs = xstart;
		ys = ystart;
		for(addr = 0; addr < datalen; addr++) {
			data = databuf[chnum*addr+ch];
			//data = databuf[datalen*ch+addr];
			xe = xstart + width*addr/datalen;
			ye = ystart - hlength*data/scale;
			if(addr != 0) {
				::MoveToEx(hdc,xs,ys,NULL);
				::LineTo(hdc,xe,ye);
			}
			xs = xe;
			ys = ye;
		}
		SelectObject(hdc, hOldPen);
		DeleteObject(hPen);
	}

	//XP[
	int xoffset = 3;
	int yoffset = 3;
	GetTextExtentPoint32(hdc,upbuf,strlen(upbuf),&siz);
	TextOut(hdc,drawing.left+xoffset,drawing.top+yoffset,upbuf,strlen(upbuf));
	GetTextExtentPoint32(hdc,downbuf,strlen(downbuf),&siz);
	TextOut(hdc,drawing.left+xoffset,drawing.bottom-yoffset-siz.cy,downbuf,strlen(downbuf));

	SelectClipRgn( hdc, hOldRgn);
	DeleteObject( hRgn );
	SelectObject(hdc, hOldBrush);
	DeleteObject(hBrush);
    SelectObject(hdc, hFontOld);
}

//**************************************************************************** 
//
// LRESULT CALLBACK OptionsDlgProc(HWND hdlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
//
//****************************************************************************
LRESULT CALLBACK AD3163DlgProc(HWND hdlg, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	HWND hbutton,hcomb,hframe,htext;
	char buf[128];
	int index;
    double interval;	//
    int data_num;		//
    int channel;
    int board_num;
	int range;
	int mode;
	int trig_mode;
	int trig_delay;
	int trig_point;		// Trigger Channel
	int trig_edge;
	double trig_level;
	double scale;
    PAINTSTRUCT ps;

switch (uMsg) 
{ 
	case WM_INITDIALOG: 
	{ 
		//WAVE
		hframe = GetDlgItem(hdlg, IDC_3163_WAVE);
		//̕ύX
		hbutton = GetDlgItem(hdlg, IDOK);
		SetWindowText(hbutton, "Start");
		hbutton = GetDlgItem(hdlg, IDCANCEL);
		SetWindowText(hbutton, "Close");
		//TvOg
		hcomb = GetDlgItem(hdlg,IDC_3163_FREQ);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"10MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"5MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"2.5MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"1.25MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"625KHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"312.5KHz");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.freq, 0L);
		//f[^
		sprintf(buf,"%ld",ad3163_prm.dtlen);
		SetDlgItemText(hdlg, IDC_3163_DATA, buf);
		//`l
		hcomb = GetDlgItem(hdlg,IDC_3163_CHANNEL);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"1");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"2");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.ch, 0L);
		//W
		hcomb = GetDlgItem(hdlg,IDC_3163_RANGE);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"}5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"}2.5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"}1V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"0`5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"0`2.5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"0`1V");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.range, 0L);
		//{[hԍ
		sprintf(buf,"%d",ad3163_prm.board);
		SetDlgItemText(hdlg, IDC_3163_BOARD, buf);
		//[h
		hcomb = GetDlgItem(hdlg,IDC_3163_MODE);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Normal");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Fast");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.mode, 0L);
		//gK[Cl[u
		hbutton = GetDlgItem(hdlg,IDC_3163_ENABLE);
		if(ad3163_prm.enable)
			SendMessage(hbutton, BM_SETCHECK, TRUE, 0L);
		else
			SendMessage(hbutton, BM_SETCHECK, FALSE, 0L);
		::EnableWindow(hbutton, FALSE);
		//gK[|Cg(`l)
		hcomb = GetDlgItem(hdlg,IDC_3163_POINT);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"1");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"2");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.point, 0L);
		::EnableWindow(hcomb, FALSE);
		//gK[GbW
		hcomb = GetDlgItem(hdlg,IDC_3163_EDGE);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"+");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"-");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.edge, 0L);
		::EnableWindow(hcomb, FALSE);
		//gK[x
		sprintf(buf,"%f",ad3163_prm.level);
		SetDlgItemText(hdlg, IDC_3163_LEVEL, buf);
		htext = GetDlgItem(hdlg,IDC_3163_LEVEL);
		::EnableWindow(htext, FALSE);
		//gK[fBC
		sprintf(buf,"%d",ad3163_prm.delay);
		SetDlgItemText(hdlg, IDC_3163_DELAY, buf);
		htext = GetDlgItem(hdlg,IDC_3163_DELAY);
		::EnableWindow(htext, FALSE);

		//return TRUE;
	}
	break;
	case WM_ERASEBKGND:
	{
		//if(ad3163_prm.yp) {
		//	Display(hdlg, ad3163_prm.yp, ad3163_prm.dtlen, ad3163_prm.ch+1, ad3163_prm.range);
		//}
	}
	break;
	case WM_PAINT:
	{
		if(ad3163_prm.yp) {
		    BeginPaint(hdlg, &ps);
			//Display(hdlg, ad3163_prm.yp, ad3163_prm.dtlen, ad3163_prm.ch+1, ad3163_prm.range);
		    EndPaint(hdlg, &ps);
		}
		//return TRUE;
	}
	break;
	case WM_COMMAND: 
	switch (wParam) 
	{ 
		case IDOK: 
		{
			//͏
			//Freq
			hcomb = GetDlgItem(hdlg,IDC_3163_FREQ);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.freq = index;
			interval = 1000000;
			//dtlen
			GetDlgItemText(hdlg, IDC_3163_DATA, buf, 128);
			data_num = atoi( buf );
			ad3163_prm.dtlen = data_num;
			//Channel Size
			hcomb = GetDlgItem(hdlg,IDC_3163_CHANNEL);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.ch = index;
			channel = index+1;
			//board
			GetDlgItemText(hdlg, IDC_3163_BOARD, buf, 128);
			board_num = atoi( buf );
			ad3163_prm.board = board_num;
			//range
			hcomb = GetDlgItem(hdlg,IDC_3163_RANGE);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.range = index;
			range = index;
			switch(range) {
			case	0:
				scale = 5.0;
				break;
			case	1:
				scale = 2.5;
				break;
			case	2:
				scale = 1.0;
				break;
			}
			//mode
			hcomb = GetDlgItem(hdlg,IDC_3163_MODE);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.mode = index;
			mode = ad3163_prm.mode;

			//trigger mode
			hbutton = GetDlgItem(hdlg,IDC_3163_ENABLE);
			index = SendMessage(hbutton,BM_GETCHECK,0, 0L);
			if(index == BST_CHECKED)
				ad3163_prm.enable = TRUE;
			else
				ad3163_prm.enable = FALSE;
			trig_mode = ad3163_prm.enable;
			//trigger channel
			hcomb = GetDlgItem(hdlg,IDC_3163_POINT);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.point = index;
			trig_point = ad3163_prm.point + 1;
			//trigger edge
			hcomb = GetDlgItem(hdlg,IDC_3163_EDGE);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.edge = index;
			trig_edge = ad3163_prm.edge;
			//trigger level
			GetDlgItemText(hdlg, IDC_3163_LEVEL, buf, 128);
			trig_level = atof( buf );
			ad3163_prm.level = trig_level;
			//trigger delay;
			GetDlgItemText(hdlg, IDC_3163_DELAY, buf, 128);
			trig_delay = atof( buf );
			ad3163_prm.delay = trig_delay;

			//̈mۂAf[^W
			if(ad3163_prm.yp)
				free(ad3163_prm.yp);
			/* Assign pointers to the various parameters */
			if (( ad3163_prm.yp = (Buffer *)malloc( data_num * channel * sizeof(double) ) ) == NULL )
				exit(2);
			//A/D Conversion
		adcon(ad3163_prm.yp, &interval, &data_num, &channel, &board_num, &range, &mode,
				&trig_mode, &trig_delay, &trig_point, &trig_edge, &trig_level);
			//Data Dispay
			//Display(hdlg, ad3163_prm.yp, data_num, channel, range);


			//EndDialog(hdlg, 0);
			//return TRUE;
		}
		break;
		case IDCANCEL:
		{
			EndDialog(hdlg, 1);
			//return TRUE;
		}
		break;
		default: 
			//return TRUE;
			break;
	} /* switch (cmd) */
	break;
	default: 
		return (DefWindowProc(hdlg, uMsg, wParam, lParam));
	} /* switch (uMsg) */
 
	return FALSE; 	

}

int ReadAd3163()
{
    int data_num,channel,board,range,mode,trig_mode,trig_point,trig_edge,trig_delay;
	int retcode = NULL;
	double interval,scale,trig_level;

	//̈mۂAf[^W
	if(ad3163_prm.yp)
		free(ad3163_prm.yp);
	/* Assign pointers to the various parameters */
	interval = ad3163_prm.freq;
	data_num = ad3163_prm.dtlen;
	channel = ad3163_prm.ch + 1;
	board = ad3163_prm.board;
	range = ad3163_prm.range;
	if (( ad3163_prm.yp = (Buffer *)malloc( data_num * channel * sizeof(double) ) ) == NULL )
		//exit(2);
		return 2;
	//A/D Conversion
	switch(range) {
	case	0:
		scale = 5.0;
		break;
	case	1:
		scale = 2.5;
		break;
	case	2:
		scale = 1.0;
		break;
	}
	mode = ad3163_prm.mode;
	trig_mode = ad3163_prm.enable;
	trig_delay = ad3163_prm.delay;
	trig_point = ad3163_prm.point + 1;
	trig_edge = ad3163_prm.edge;
	trig_level = ad3163_prm.level;

	retcode = adcon(	ad3163_prm.yp,
			&interval, 
			&data_num,
			&channel,
			&board,
			&range,
			&mode,
			&trig_mode,
			&trig_delay,
			&trig_point,
			&trig_edge,
			&trig_level);
	//Data Dispay
	//Display(hdlg, ad3163_prm.yp, data_num, channel, range);
	return retcode;
}

void WriteAd3163()
{
	Buffer *yp;
	int dim, index[MAX_INDEX];
	long addr;

	if(ad3163_prm.yp) {
		if(ad3163_prm.ch == 0) {
			dim = 1;
			index[0] = ad3163_prm.dtlen;
		}
		else {
			dim = 2;
			index[0] = ad3163_prm.dtlen;
			index[1] = ad3163_prm.ch+1;
		}
		//̈mۂ
		if (( yp = AllocBuffer( dim * ad3163_prm.dtlen ) ) == NULL )
			exit(2);
		for(addr = 0; addr < dim * ad3163_prm.dtlen; addr++)
			yp[addr] = ad3163_prm.yp[addr];
		ReturnSeries(yp, dim, index);
		free(ad3163_prm.yp);
		ad3163_prm.yp = NULL;
	}
}

LRESULT CALLBACK Ad3163ParamWndProc( HWND hdlg, UINT uMessage, WPARAM wParam, LPARAM lParam)
{
  int wNotifyCode;
  HWND hctl;
  char buf[32];
  HWND hbutton,hcomb,hframe,htext;
  double interval;
  int data_num,index,channel,board_num,range,mode;
  int trig_mode,trig_point,trig_edge,trig_delay;
  double scale,trig_level;

  switch (uMessage)
  {
    case WM_INITDIALOG:
		// set title
		//SendMessage(hdlg, WM_SETTEXT,0,(LPARAM)szTitle);
		//TvOg
		hcomb = GetDlgItem(hdlg,IDC_3163_FREQ);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"10MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"5MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"2.5MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"1.25MHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"625KHz");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"312.5KHz");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.freq_sel, 0L);
		//f[^
		sprintf(buf,"%ld",ad3163_prm.dtlen);
		SetDlgItemText(hdlg, IDC_3163_DATA, buf);
		//`l
		hcomb = GetDlgItem(hdlg,IDC_3163_CHANNEL);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"1");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"2");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"3");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"4");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"5");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"6");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"7");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"8");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.ch, 0L);
		//W
		hcomb = GetDlgItem(hdlg,IDC_3163_RANGE);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"}5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"}2.5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"}1V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"0`5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"0`2.5V");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"0`1V");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.range, 0L);
		//{[hԍ
		sprintf(buf,"%d",ad3163_prm.board);
		SetDlgItemText(hdlg, IDC_3163_BOARD, buf);
		//[h
		hcomb = GetDlgItem(hdlg,IDC_3163_MODE);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Normal");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Fast");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.mode, 0L);
		//gK[Cl[u
		hcomb = GetDlgItem(hdlg,IDC_3163_ENABLE);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Free");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Level Trigger");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"External Trigger");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.enable, 0L);
		//::EnableWindow(hbutton, FALSE);
		//gK[|Cg(`l)
		hcomb = GetDlgItem(hdlg,IDC_3163_POINT);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"1");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"2");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"3");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"4");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"5");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"6");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"7");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"8");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.point, 0L);
		//::EnableWindow(hcomb, FALSE);
		//gK[GbW
		hcomb = GetDlgItem(hdlg,IDC_3163_EDGE);
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Up");
		SendMessage(hcomb,CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)"Down");
		SendMessage(hcomb,CB_SETCURSEL,ad3163_prm.edge, 0L);
		//::EnableWindow(hcomb, FALSE);
		//gK[x
		sprintf(buf,"%f",ad3163_prm.level);
		SetDlgItemText(hdlg, IDC_3163_LEVEL, buf);
		htext = GetDlgItem(hdlg,IDC_3163_LEVEL);
		//::EnableWindow(htext, FALSE);
		//gK[fBC
		sprintf(buf,"%d",ad3163_prm.delay);
		SetDlgItemText(hdlg, IDC_3163_DELAY, buf);
		htext = GetDlgItem(hdlg,IDC_3163_DELAY);
		//::EnableWindow(htext, FALSE);
	  break;
    case WM_COMMAND:
	  wNotifyCode = HIWORD(wParam);
	  switch (LOWORD(wParam))
	  {
	  case  IDOK:
			//Freq
			hcomb = GetDlgItem(hdlg,IDC_3163_FREQ);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR) {
				ad3163_prm.freq_sel = index;
				switch(index) {
				case 0:
					interval = 10000000;	break;
				case 1:
					interval = 5000000;		break;
				case 2:
					interval = 2500000;		break;
				case 3:
					interval = 1250000;		break;
				case 4:
					interval = 625000;		break;
				case 5:
					interval = 312500;		break;
				}
				ad3163_prm.freq = interval;
			}
			//dtlen
			GetDlgItemText(hdlg, IDC_3163_DATA, buf, 128);
			data_num = atoi( buf );
			ad3163_prm.dtlen = data_num;
			//Channel Size
			hcomb = GetDlgItem(hdlg,IDC_3163_CHANNEL);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.ch = index;
			channel = index+1;
			//board
			GetDlgItemText(hdlg, IDC_3163_BOARD, buf, 128);
			board_num = atoi( buf );
			ad3163_prm.board = board_num;
			//range
			hcomb = GetDlgItem(hdlg,IDC_3163_RANGE);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.range = index;
			range = index;
			switch(range) {
			case	0:
				scale = 5.0;
				break;
			case	1:
				scale = 2.5;
				break;
			case	2:
				scale = 1.0;
				break;
			}
			//mode
			hcomb = GetDlgItem(hdlg,IDC_3163_MODE);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.mode = index;
			mode = ad3163_prm.mode;

			//trigger mode
			hbutton = GetDlgItem(hdlg,IDC_3163_ENABLE);
			index = SendMessage(hbutton,BM_GETCHECK,0, 0L);
			if(index == BST_CHECKED)
				ad3163_prm.enable = TRUE;
			else
				ad3163_prm.enable = FALSE;
			trig_mode = ad3163_prm.enable;
			//trigger channel
			hcomb = GetDlgItem(hdlg,IDC_3163_POINT);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.point = index;
			trig_point = ad3163_prm.point + 1;
			//trigger edge
			hcomb = GetDlgItem(hdlg,IDC_3163_EDGE);
			index = SendMessage(hcomb,CB_GETCURSEL,0, 0L);
			if(index != LB_ERR)
				ad3163_prm.edge = index;
			trig_edge = ad3163_prm.edge;
			//trigger level
			GetDlgItemText(hdlg, IDC_3163_LEVEL, buf, 128);
			trig_level = atof( buf );
			ad3163_prm.level = trig_level;
			//trigger delay;
			GetDlgItemText(hdlg, IDC_3163_DELAY, buf, 128);
			trig_delay = atof( buf );
			ad3163_prm.delay = trig_delay;
			// close dialog
			EndDialog(hdlg, wParam);
			return TRUE;
	  case  IDCANCEL:
		EndDialog(hdlg, wParam);
		return TRUE;
	  }
	  break;

  }
  return FALSE;
}

//
//  window procedure
//
LRESULT CALLBACK Ad3163MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	//static HWND hWndToolbar;      
	int wmId, wmEvent ;
    static HWND hWndTT;
    TOOLINFO lpToolInfo;
	LPTOOLTIPTEXT lpToolTipText;
	static CHAR szBuf[128];
	HICON hIcon;
	char buf[64];
	int retcode;
	RECT rect;

	switch( message ) {

		case WM_CREATE:
			// create toolbar control
			hWndToolBar = CreateToolbarEx( 
				hWnd,                   // parent
				WS_CHILD | WS_BORDER | WS_VISIBLE | TBSTYLE_TOOLTIPS | CCS_ADJUSTABLE,   // style
				ID_AD_TOOLBAR,             // toolbar id
				5,                      // number of bitmaps
				hInstance,                  // mod instance
				IDB_AD_TOOLBAR,            // resource id for the bitmap
				(LPCTBBUTTON)&tbButtons,// address of buttons
				5,                     // number of buttons
				16,16,                  // width & height of the buttons
				16,16,                  // width & height of the bitmaps
				sizeof(TBBUTTON));      // structure size

			if (hWndToolBar == NULL )
			{
				MessageBox (NULL, "Toolbar Bar not created!", NULL, MB_OK );
				break;
			}

            // Get the handle to the tooltip window.
            hWndTT = (HWND)SendMessage(hWndToolBar, TB_GETTOOLTIPS, 0, 0);

            if (hWndTT)
            {
            }
            else
                MessageBox(NULL, "Could not get tooltip window handle.",NULL, MB_OK);

			// set title
			SendMessage(hWnd, WM_SETTEXT,0,(LPARAM)szTitle);
			// set icon
			hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_AD));
			SendMessage(hWnd, WM_SETICON, TRUE, (LPARAM)hIcon);
			hFont = CreateFont(12,
				0,0,0,FW_NORMAL,0,0,0,SHIFTJIS_CHARSET,
				OUT_STROKE_PRECIS,CLIP_DEFAULT_PRECIS,DRAFT_QUALITY,
				DEFAULT_PITCH,"MS PSVbN");

			// exit thread
			//SetEvent(hAd3163Event);
			break ;

        case WM_SIZE:
            // Tell the toolbar to resize itself to fill the top of the window.
            SendMessage(hWndToolBar, TB_AUTOSIZE, 0L, 0L);
            break;

		case WM_ACTIVATE:
			break;

		case WM_COMMAND:			// user menu

			wmId    = LOWORD( wParam ) ;
			wmEvent = HIWORD( wParam ) ;
			switch( wmId ) {
				case IDM_AD_SCALE:
					break;

				case IDM_AD_PARAM:
					//set parameters
					if (DialogBox(hInstance, MAKEINTRESOURCE(IDD_AD3163),
						hWnd, (DLGPROC)Ad3163ParamWndProc) == IDOK) {
						// OnOK command
					}
					else {
						// Cancel command
					}
					break;

				case IDM_AD_START:
					// read
					if(retcode = ReadAd3163()) {
            MessageBox(hWnd, "A/D conversion failed.",NULL, MB_OK);
					}
					else {
						// update
			GetClientRect(hWnd, &rect);
			InvalidateRect(hWnd, &rect, TRUE);
						//WriteAd3163();
					}
					break;
				case IDM_AD_ABOUT:
					//wvJ
					break;

				case ID_AD_EXIT:
					//send data
					WriteAd3163();
					DestroyWindow(hWnd);
					break;
				default:
					return DefWindowProc( hWnd, message, wParam, lParam ) ;
			}
			break ;

		case WM_PAINT:
			PAINTSTRUCT ps;
			RECT rect;
			BeginPaint( hWnd, &ps );
			Display(hWnd, ps.hdc, ad3163_prm.yp, ad3163_prm.dtlen,
				ad3163_prm.ch+1, ad3163_prm.range);
			EndPaint( hWnd, &ps );
			break;

		case WM_NOTIFY:
            switch (((LPNMHDR) lParam)->code) 
            {
                case TTN_NEEDTEXT:    
                    // Display tool tip text.
			        lpToolTipText = (LPTOOLTIPTEXT)lParam;
    				LoadString (hInstance, 
	    				lpToolTipText->hdr.idFrom,    // string ID == cmd ID
		    			szBuf,
			    		sizeof(szBuf));
				    	lpToolTipText->lpszText = szBuf;
			        break;

                case TBN_QUERYDELETE:
                    // Toolbar customization -- can we delete this button?
                    return TRUE;
                    break;

                case TBN_GETBUTTONINFO:
                    // The toolbar needs information about a button.
                    return FALSE;
                    break;

                case TBN_QUERYINSERT:
                    // Can this button be inserted? Just say yo.
                    return TRUE;
                    break;

                case TBN_CUSTHELP:
                    // Need to display custom help.
                    MessageBox(hWnd, "This help is custom.",NULL, MB_OK);
                    break;

                case TBN_TOOLBARCHANGE:
                    // Done dragging a bitmap to the toolbar.
                    SendMessage(hWndToolBar, TB_AUTOSIZE, 0L, 0L);
                    break;

                default:
                    return TRUE;
                    break;
            }
			return 0L;
			break;

		case WM_DESTROY:
			DeleteObject(hFont);
			PostQuitMessage(0);
			SetEvent(hAd3163Event);
			break;

		default:
			return DefWindowProc( hWnd, message, wParam, lParam ) ;
	}
	return 0 ;
}

//
// register main window calss
//
static BOOL InitApplication( HINSTANCE hInstance )
{
	WNDCLASSEX wcex ;

	ZeroMemory( &wcex, sizeof wcex ) ;
	wcex.cbSize        = sizeof( WNDCLASSEX ) ;
	wcex.style         = CS_HREDRAW | CS_VREDRAW ;
	wcex.lpfnWndProc   = (WNDPROC)Ad3163MainWndProc ;
	wcex.cbClsExtra    = 0 ;
	wcex.cbWndExtra    = 0 ;
	wcex.hInstance     = hInstance ;
	wcex.hIcon         = LoadIcon( hInstance, "ICON" ) ;
	wcex.hIconSm       = LoadIcon( hInstance, "SICON" ) ;
	wcex.hCursor       = LoadCursor( NULL, IDC_ARROW );
	wcex.hbrBackground = (HBRUSH)GetStockObject( WHITE_BRUSH ) ;
	wcex.lpszMenuName  = MAKEINTRESOURCE(IDR_AD_MAIN) ;
	wcex.lpszClassName = szWindowClass ;

	return RegisterClassEx( &wcex ) ;
}



//
// careate main window
//
static BOOL InitInstance( HINSTANCE hInstance, int nCmdShow )
{
	HWND hWnd ;

	hWnd = CreateWindow( szWindowClass,
						 szTitle,
						 WS_OVERLAPPEDWINDOW,
						 CW_USEDEFAULT,
						 0,
						 480,
						 320,
						 NULL,
						 NULL/*(HMENU)IDR_BM_MAIN*/,
						 hInstance,
						 NULL);
	if( !hWnd )
		return FALSE ;

	ShowWindow( hWnd, nCmdShow ) ;
	UpdateWindow( hWnd ) ;

	return TRUE ;
}

void __cdecl CreateAd3163()
{
	MSG msg ;
	RECT rc;

	// Ensure that common control DLL is loaded
	InitCommonControls();

	// register main window calss
	if( !(wndClass = InitApplication( hInstance )) ) {
		return;
	}

	// create main window
	if( !InitInstance( hInstance, SW_SHOW ) ) {
		return;
	}

	// dispatch messages
	while( GetMessage(&msg, NULL, 0, 0) ) {
		TranslateMessage( &msg ) ;
		DispatchMessage( &msg ) ;
	}

	if(wndClass) {
		UnregisterClass(szWindowClass, hInstance);
		wndClass = NULL;
	}
}

DLLEXPORT int ad3163()
{
	Buffer *yp;
    double interval;	//
    int data_num;		//
    int channel;
    int board_num;
	int range;
	int mode;
	int trig_mode;
	int trig_delay;
	int trig_point;
	int trig_edge;
	int dim, index[MAX_INDEX];
	long addr;
	double trig_level;
	HINSTANCE   hinstDll;
	char path[_MAX_PATH],dll[_MAX_PATH];

	//system directory
	::GetSystemDirectory(path, _MAX_PATH);
	//whether Fbiad.dll is
	//Load the DLL.
	sprintf(dll,"%s\\fbiad.dll",path);
	hinstDll = ::LoadLibrary(dll);
	if(!hinstDll) {
		printf("Fbiad.dll not exist. (SLIO module)\n");
		return 5;
	}
	else {
		::FreeLibrary(hinstDll);
	}

	// Check for proper number of arguments */
	// Get parameters 
	interval = GetScalar(0);
	if(interval == -1) {
		//Operation mode
		OpMode = TRUE;
		//initialized parameters
		ad3163_prm.freq_sel = 0;
		ad3163_prm.freq = 10000000;		//10M
		ad3163_prm.dtlen = 1000;
		ad3163_prm.ch = 0;				//1ch
		ad3163_prm.board = 1;			//bord number
		ad3163_prm.range = 0;
		ad3163_prm.mode = 0;
		ad3163_prm.enable = 0;
		ad3163_prm.point = 0;
		ad3163_prm.edge = 0;
		ad3163_prm.level = 0.0;
		ad3163_prm.delay = 0;
		ad3163_prm.yp = NULL;

		//window base
		HANDLE hThread;
		DWORD threadID;

		hAd3163Event = CreateEvent(0, TRUE, FALSE, NULL);
		hThread = CreateThread( 0, 0,
			(LPTHREAD_START_ROUTINE)CreateAd3163, NULL, 0, &threadID);
		//rtval = GetThreadPriority( hThread );
		//if(rtval == THREAD_PRIORITY_ERROR_RETURN) {
		//	MessageBox(NULL,"GetThreadPriority error.","SL_Shell",MB_OK);
		//	return 1;
		//}
		//if(!SetThreadPriority(hThread, THREAD_PRIORITY_ABOVE_NORMAL)) {
		//	MessageBox(NULL,"SetThreadPriority error.","SL_Shell",MB_OK);
		//	return 1;
		//}

		// wait during hAd3163Event is not  set
		WaitForSingleObject(hAd3163Event, INFINITE);
		CloseHandle(hAd3163Event);
	}
	else {
		if(GetArgNum() != 11)
			exit(1);

		// Get parameters 
        interval = GetScalar(0);
		//interval`FbN
		if((interval != 10000000) && (interval != 8000000)
			&& (interval != 5000000) && (interval != 4000000)
			&& (interval != 2500000) && (interval != 2000000)
			&& (interval != 1250000) && (interval != 1000000)
			&& (interval != 625000) && (interval != 500000)
			&& (interval != 312500) && (interval != 250000)
			&& (interval != 156250) && (interval != 125000)
			&& (interval != 78125) && (interval != 62500)) {
			//intervalG[
			printf("No such Sampling Frequency.\n");
			exit(1);
		}

        data_num = (int)GetScalar(1);
		if(data_num < 1 || data_num >= 512000) {
			//data_numG[
			printf("No such Data Length.\n");
			exit(1);
		}
        channel = (int)GetScalar(2);
		if(channel != 1 && channel != 2) {
			//Channel SizeG[
			printf("No such Channel Number.\n");
			exit(1);
		}
        board_num = (int)GetScalar(3);
		if(board_num < 1 || board_num > 16) {
			//board_numG[
			printf("No such Board Number.\n");
			exit(1);
		}
		range = (int)GetScalar(4);
		if((range < 0) || (range > 9) || (range > 2 && range < 7)) {
			//rangeG[
			printf("No such Channel Range.\n");
		}
		mode = (int)GetScalar(5);
		if(mode != 0 && mode != 1) {
			//modeG[
			printf("No such Sample Mode.\n");
		}
		if(mode == 1)
			channel = 1;
		trig_mode = (int)GetScalar(6);
		trig_edge = (int)GetScalar(7);
		trig_point = (int)GetScalar(8);		// gK[`l
		trig_level = (int)GetScalar(9);
		trig_delay = (int)GetScalar(10);

        /* Assign pointers to the various parameters */
		if (( yp = AllocBuffer( data_num * channel ) ) == NULL )
			exit(2);

		// message
        printf("Dsptec(c) 2000 All Rights Reserved\n");
		printf("  interval = %g\n",interval);
		printf("  data_num = %d\n",data_num);
		printf("  channel = %d\n",channel);
		printf("  board_num = %d\n",board_num);
		printf("  range = %d\n",range);
		printf("  mode = %d\n",mode);
		printf("  trig_mode = %d\n",trig_mode);
		printf("  trig_edge = %d\n",trig_edge);
		printf("  trig_channel = %d\n",trig_point);
		printf("  trig_level = %f\n",trig_level);
		printf("  trig_delay = %d\n",trig_delay);
        printf("Please wait %g Sec\a\n",(double)data_num/interval);

		adcon(yp, &interval, &data_num, &channel, &board_num, &range, &mode,
				&trig_mode, &trig_delay, &trig_point, &trig_edge, &trig_level);

		if(channel == 1) {
			dim = 1;
			index[0] = data_num;
		}
		else {
			dim = 2;
			index[0] = data_num;
			index[1] = dim;
		}
		ReturnSeries(yp, dim, index);
        return 0;
	}
	return 0;
}

#ifdef __cplusplus
}
#endif
