/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: da3305.cpp,v 1.1.1.1 2004/03/31 08:15:06 orrisroot Exp $ */
//
//	I/O{[hW[
//	da3305.cpp
//
//	The calling syntax is:
//
//	void DAOutput(interval,data_buf,data_num,channel,board_num,range)
//

#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include "SL_macro.h"
#include "SL_cmd.h"
#define  __IMPORTSYMBOL__
#include "libsatellite.h"
#undef   __IMPORTSYMBOL__
#include "FbiDa.h"

#ifdef __cplusplus
extern "C" {
#endif

static void daoutput(
double yp[],
double *interval,
int *data_num,
int *channel,
int *board_num,
int *range,
int *repeat,
int *trig_mode,
int *trig_delay,
int *trig_point,
int *trig_edge
)
{
        HANDLE hDeviceHandle;
        DASMPLREQ SmplConfig;
		DAMODEREQ ModeConfig;
        ULONG ulSmplNum,i,addr;
        WORD wChannel,j;
        char strDeviceName[256];
		WORD *pData;
		WORD wMax,wMin,wMid;
		ULONG ulRange,ulTrigMode,ulTrigPoint,ulTrigEdge,ulMode;
		HANDLE	hEvent;
		double scale,daout;
		int errcode;

        /* Parameters */
        ulSmplNum = (ULONG)(*data_num);
        wChannel = (WORD)(*channel);

        /*Device name*/
        sprintf(strDeviceName, "FBIDA%d", *board_num);

        /*Device open*/
        hDeviceHandle = DaOpen(strDeviceName);
        if(hDeviceHandle == INVALID_HANDLE_VALUE)
                return;

        /*Set board*/
		/*Set EventHandle*/
		if((ULONG)(*trig_mode) == 0)
	        DaSetBoardConfig(hDeviceHandle, ulSmplNum, NULL, NULL, 0);
		else {
			hEvent = CreateEvent(0, TRUE, FALSE, NULL);
        	DaSetBoardConfig(hDeviceHandle, ulSmplNum, hEvent, NULL, 0);
		}

        /*Get Sampling Config*/
        DaGetSamplingConfig(hDeviceHandle, &SmplConfig);

        /*Set Sampling Config*/
		scale = 1.0;
		switch(*range) {
			case	0:
				ulRange = DA_5V;
				ulMode = DA_RANGE_BIPOLAR;
				scale = 5;
		wMax = 0xfff;	wMid = 0x7ff;	wMin = 0x7ff;
				break;
			case	1:
				ulMode = DA_RANGE_BIPOLAR;
				ulRange = DA_2P5V;
				scale = 2.5;
		wMax = 0xfff;	wMid = 0x7ff;	wMin = 0x7ff;
				break;
			case	2:
				ulRange = DA_1V;
				ulMode = DA_RANGE_BIPOLAR;
				scale = 1.0;
		wMax = 0xfff;	wMid = 0x7ff;	wMin = 0x7ff;
				break;
			case	3:
				ulRange = DA_0_5V;
				ulMode = DA_RANGE_UNIPOLAR;
				scale = 5;
		wMax = 0xfff;	wMid = 0xfff;	wMin = 0x0;
				break;
			case	4:
				ulRange = DA_0_2P5V;
				ulMode = DA_RANGE_UNIPOLAR;
				scale = 2.5;
		wMax = 0xfff;	wMid = 0xfff;	wMin = 0x0;
				break;
			case	5:
				ulRange = DA_0_1V;
				ulMode = DA_RANGE_UNIPOLAR;
				scale = 1.0;
		wMax = 0xfff;	wMid = 0xfff;	wMin = 0x0;
				break;
		}

		SmplConfig.ulSmplRepeat = (ULONG)(*repeat);
		SmplConfig.fSmplFreq = (float)(*interval);
        if(wChannel == 1) {
                /*1Channel*/
                SmplConfig.ulChCount = 1;
                SmplConfig.SmplChReq[0].ulChNo = 1;
				SmplConfig.SmplChReq[0].ulRange = ulRange;
        }
        else {
                /*2Channel*/
                SmplConfig.ulChCount = 2;
                SmplConfig.SmplChReq[0].ulChNo = 1;
				SmplConfig.SmplChReq[0].ulRange = ulRange;
                SmplConfig.SmplChReq[1].ulChNo = 2;
				SmplConfig.SmplChReq[1].ulRange = ulRange;
        }
		if((ULONG)(*trig_mode) == 0)
			ulTrigMode = DA_FREERUN;
		else
			ulTrigMode = DA_EXTTRG;
		SmplConfig.ulTrigMode = ulTrigMode;
		SmplConfig.ulTrigDelay = (ULONG)(*trig_delay);
		ulTrigPoint = DA_TRIG_START;
		switch((ULONG)(*trig_point)) {
			case	0:	ulTrigPoint = DA_TRIG_START;	break;
			case	1:	ulTrigPoint = DA_TRIG_STOP;	break;
			case	2:	ulTrigPoint = DA_TRIG_START_STOP;	break;
		}
		SmplConfig.ulTrigPoint = ulTrigPoint;
		if((ULONG)(*trig_edge) == 0)
			ulTrigEdge = DA_DOWN_EDGE;
		else
			ulTrigEdge = DA_UP_EDGE;
		SmplConfig.ulTrigEdge = ulTrigEdge;
        DaSetSamplingConfig(hDeviceHandle, &SmplConfig);

		/* Set mode */
		errcode = DaGetMode(hDeviceHandle, &ModeConfig);
		if(errcode != DA_ERROR_SUCCESS) {
			printf("DaGetMode error.\n");
			return;
		}
        if(wChannel == 1) {
                /*1Channel*/
				ModeConfig.ModeChReq[0].ulRange = ulMode;
				ModeConfig.ModeChReq[0].fVolt = scale;
        }
        else {
                /*2Channel*/
				ModeConfig.ModeChReq[0].ulRange = ulMode;
				ModeConfig.ModeChReq[0].fVolt = scale;
				ModeConfig.ModeChReq[1].ulRange = ulMode;
				ModeConfig.ModeChReq[1].fVolt = scale;
        }
		errcode = DaSetMode(hDeviceHandle, &ModeConfig);
		if(errcode != DA_ERROR_SUCCESS) {
			printf("DaSetMode error.\n");
			return;
		}


		/*Clear the analog output data*/
		DaClearSamplingData(hDeviceHandle);

        /* Set data */
		wMax = 0xC00;	wMid = 0x800;	wMin = 0x400;
		pData = (WORD *)malloc(ulSmplNum * sizeof( WORD ) * wChannel );
		addr = 0;
		for(i = 0; i < ulSmplNum; i++) {
			for(j=0;j<wChannel;j++) {
				daout = yp[j*ulSmplNum+i];
				if(daout >= scale)
					daout = scale;
				if(daout < -scale)
					daout = -scale;
				pData[addr] = (WORD)(wMid * daout / scale/*yp[j*ulSmplNum+i]*/ + wMid);
				addr++;
			}
		}
        DaSetSamplingData(hDeviceHandle, pData, ulSmplNum);
		free(pData);

        /* Sampling start */
		if((ULONG)(*trig_mode) == 0)
	        DaStartSampling(hDeviceHandle, FLAG_SYNC);
		else {
	        DaStartSampling(hDeviceHandle, FLAG_ASYNC);
			WaitForSingleObject(hEvent, INFINITE);
#if 0
			limit = (ULONG)(*data_num / *interval * 1000 + 1);
			WaitForSingleObject(hEvent, limit);
#endif			
			ResetEvent(hEvent);
			CloseHandle(hEvent);
		}

        /* Device close */
        DaClose(hDeviceHandle);
}

DLLEXPORT int da3305()
{
        Buffer *yp;
        double interval;
        int data_num;
        int channel;
        int board_num;
		int range;
		int repeat;
		int trig_mode;
		int trig_delay;
		int trig_point;
		int trig_edge;
		int dim, index[MAX_INDEX];
		int argnum;
		HINSTANCE   hinstDll;
		char path[_MAX_PATH],dll[_MAX_PATH];

		//whether Fbida.dll is
		//Load the DLL.
		sprintf(dll,"%s\\fbida.dll",path);
		hinstDll = ::LoadLibrary(dll);
		if(!hinstDll) {
			printf("Fbida.dll not exist. (SLIO module)\n");
			return 5;
		}
		else {
			::FreeLibrary(hinstDll);
		}
        /* Check for proper number of arguments */
        if((argnum = GetArgNum()) != 11) {
                printf("This da3305 function is for Interface PCI-3305\n");
                printf("Usage:da3305(interval,data_buf,data_num,channel,board_num)\n\n");
                printf("Example:da3305(1000000.0, y, 1024, 1, 1, 8, 1)\n");
				exit(1);
        }

        /* Parameters */
		yp = GetSeries(0,&dim,index);
        interval = GetScalar(1);
        data_num = (int)GetScalar(2);
        channel = (int)GetScalar(3);
        board_num = (int)GetScalar(4);
		range = (int)GetScalar(5);
		repeat = (int)GetScalar(6);
		trig_mode = (int)GetScalar(7);
		trig_delay = (int)GetScalar(8);
		trig_point = (int)GetScalar(9);
		trig_edge = (int)GetScalar(10);
		//message
        printf("Dsptec(c) 2000 All Rights Reserved\n");
		printf("  interval = %g\n",interval);
		printf("  data_num = %d\n",data_num);
		printf("  channel = %d\n",channel);
		printf("  board_num = %d\n",board_num);
		printf("  range = %d\n",range);
		printf("  repeat = %d\n",repeat);
		printf("  trig_mode = %d\n",trig_mode);
		printf("  trig_delay = %d\n",trig_delay);
		printf("  trig_point = %d\n",trig_point);
		printf("  trig_edge = %d\n",trig_edge);
        printf("  Please wait %g Sec\a\n",(double)data_num/interval*repeat);

        daoutput(yp, &interval, &data_num, &channel, &board_num, &range, &repeat,
			&trig_mode, &trig_delay, &trig_point, &trig_edge);

        return(0);
}

#ifdef __cplusplus
}
#endif
