/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: DriveCtrl.cpp,v 1.2 2004/08/02 07:15:22 yoshihiko Exp $ */

#include "config.h"

#include "stdafx.h"
#include "win_shell.h"
#include "DriveCtrl.h"
#include "shell/satellite4.h"

#include "ConsoleDoc.h"
#include "ConsoleView.h"
#include "MainFrm.h"
#include <io.h>
#include <direct.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// global
// Global pointer to the shell's IMalloc interface. 
LPMALLOC g_pMalloc; 
extern CMultiDocTemplate* g_pEditDocTemplate;
extern	CConsoleView* g_pConsole;
extern char g_charScriptText[_MAX_PATH];

// Image indexes
#define ILI_HARD_DISK       0
#define ILI_FLOPPY          1
#define ILI_CD_ROM          2
#define ILI_NET_DRIVE       3
#define ILI_CLOSED_FOLDER   4
#define ILI_OPEN_FOLDER     5
#define ILI_FILE			6
#define ILI_DESKTOP			7
#define ILI_DOCUMENT		8
#define ILI_COMPUTER		9
#define ILI_SCRIPT			10
#define ILI_TEXT			11
#define	ILI_MODEL			12

struct PATHINFO {
	char path[_MAX_PATH];
	CString attr;
	CString comm;;
};

/////////////////////////////////////////////////////////////////////////////
// CDriveCtrl

CDriveCtrl::CDriveCtrl()
{
	m_FileType = FILE_SL;
	m_pDragImage = NULL;
	m_bSelScript = FALSE;
	m_bPopupMenu = FALSE;
	/*The variable for tooltip */
	m_pToolTip = NULL;
	m_hCurItem = NULL;
}

CDriveCtrl::~CDriveCtrl()
{
	if(m_pToolTip)
		delete m_pToolTip;
}


BEGIN_MESSAGE_MAP(CDriveCtrl, CTreeCtrl)
	//{{AFX_MSG_MAP(CDriveCtrl)
	ON_WM_CREATE()
	ON_NOTIFY_REFLECT(TVN_ITEMEXPANDING, OnItemexpanding)
	ON_NOTIFY_REFLECT(TVN_SELCHANGED, OnSelchanged)
	ON_COMMAND(ID_EDIT_CUT, OnEditCut)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CUT, OnUpdateEditCut)
	ON_COMMAND(ID_SCRIPT_RUN, OnScriptRun)
	ON_COMMAND(ID_FILE_OPEN, OnFileOpen)
	ON_NOTIFY_REFLECT(TVN_BEGINDRAG, OnBegindrag)
	ON_WM_MOUSEMOVE()
	ON_WM_DESTROY()
	ON_NOTIFY_REFLECT(NM_DBLCLK, OnDblclk)
	ON_COMMAND(ID_EDIT_DELETE, OnEditDelete)
	ON_UPDATE_COMMAND_UI(ID_EDIT_DELETE, OnUpdateEditDelete)
	ON_WM_RBUTTONDOWN()
	ON_COMMAND(ID_EDIT_RENAME, OnEditRename)
	ON_UPDATE_COMMAND_UI(ID_EDIT_RENAME, OnUpdateEditRename)
	ON_NOTIFY_REFLECT(TVN_ENDLABELEDIT, OnEndlabeledit)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
	ON_COMMAND(ID_FILE_CHDIR, OnFileChdir)
	ON_UPDATE_COMMAND_UI(ID_FILE_CHDIR, OnUpdateFileChdir)
	ON_WM_LBUTTONUP()
	ON_COMMAND(ID_SCRIPT_STEP, OnScriptStep)
	ON_COMMAND(ID_FILE_ALL, OnFileAll)
	ON_UPDATE_COMMAND_UI(ID_FILE_ALL, OnUpdateFileAll)
	ON_COMMAND(ID_FILE_SL, OnFileSl)
	ON_UPDATE_COMMAND_UI(ID_FILE_SL, OnUpdateFileSl)
	ON_COMMAND(ID_FILE_MDL, OnFileMdl)
	ON_UPDATE_COMMAND_UI(ID_FILE_MDL, OnUpdateFileMdl)
	ON_COMMAND(ID_FILE_TXT, OnFileTxt)
	ON_UPDATE_COMMAND_UI(ID_FILE_TXT, OnUpdateFileTxt)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CDriveCtrl bZ[W nh

int CDriveCtrl::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CTreeCtrl::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	// Initialize the image list.
    m_ilDrives.Create (IDB_DRIVEIMAGES, 16, 1, RGB (255, 0, 255));
    SetImageList (&m_ilDrives, TVSIL_NORMAL);
	
	return 0;
}

BOOL CDriveCtrl::PreCreateWindow(CREATESTRUCT& cs) 
{
	if (!CTreeCtrl::PreCreateWindow (cs))
		return FALSE;

    cs.style |= TVS_HASLINES | TVS_LINESATROOT | TVS_HASBUTTONS |
        TVS_SHOWSELALWAYS | TVS_EDITLABELS;
	return TRUE;
}

int CDriveCtrl::GetSpecialFolder(DWORD csidl, char *path, char *specialName)
{

   LPITEMIDLIST pidlList; 
   LPSHELLFOLDER pFolder;
   int ret = 0;

   // Get the shell's allocator. 
   if (!SUCCEEDED(SHGetMalloc(&g_pMalloc))) 
      return ret; 

   // Get the PIDL for the Programs folder. 
   if (SUCCEEDED(SHGetSpecialFolderLocation(NULL, csidl, &pidlList))) 
      { 
      // Start with the Desktop folder. 
      if (SUCCEEDED(SHGetDesktopFolder(&pFolder))) 
         { 
			// Display the name of the subfolder. 
		SHGetPathFromIDList(pidlList, path);
		ret = 1;
    
         // Process each item identifier in the list. 
         LPITEMIDLIST pidl; 
         for (pidl = pidlList; pidl != NULL; pidl = GetNextItemID(pidl)) 
            { 
            STRRET sName; 
            LPSHELLFOLDER pSubFolder; 
            LPITEMIDLIST pidlCopy; 

            // Copy the item identifier to a list by itself. 
            if ((pidlCopy = CopyItemID(pidl)) == NULL) 
               break; 
    
            // Display the name of the subfolder. 
            if (SUCCEEDED(pFolder->GetDisplayNameOf(	pidlCopy, 
														SHGDN_INFOLDER, 
                                                        &sName))) { 
				PrintStrRet(pidlCopy, &sName, specialName);
			}
    
            // Bind to the subfolder. 
            if (!SUCCEEDED(pFolder->BindToObject(		pidlCopy, 
                                                        NULL, 
                                                        IID_IShellFolder, 
                                                        (LPVOID *)(&pSubFolder)))) 
               { 
               g_pMalloc->Free(pidlCopy);
               break; 
               } 
    
            // Free the copy of the item identifier. 
            g_pMalloc->Free(pidlCopy); 
    
            // Release the parent folder and point to the 
            // subfolder. 
            pFolder->Release(); 
            pFolder = pSubFolder; 
            }
    
         // Release the last folder that was bound to. 
         if (pFolder != NULL) 
            pFolder->Release(); 
         } 
    
      // Free the PIDL for the Programs folder. 
      g_pMalloc->Free(pidlList); 
      }
    
   // Release the shell's allocator. 
   g_pMalloc->Release(); 
   return ret;
}

// GetNextItemID - points to the next element in an item identifier list. 
// Returns a PIDL if successful, or NULL if at the end of the list. 
// pidl - previous element
LPITEMIDLIST CDriveCtrl::GetNextItemID(LPITEMIDLIST pidl) 
   { 
   // Get the size of the specified item identifier. 
   int cb = pidl->mkid.cb; 

   // If the size is zero, it is the end of the list. 
   if (cb == 0) 
      return NULL; 

   // Add cb to pidl (casting to increment by bytes). 
   pidl = (LPITEMIDLIST) (((LPBYTE) pidl) + cb); 

   // Return NULL if it is null-terminating, or a pidl otherwise. 
   return (pidl->mkid.cb == 0) ? NULL : pidl; 
   } 

// CopyItemID - creates an item identifier list containing the first 
//     item identifier in the specified list. 
// Returns a PIDL if successful, or NULL if out of memory. 
LPITEMIDLIST CDriveCtrl::CopyItemID(LPITEMIDLIST pidl) 
   { 
   // Get the size of the specified item identifier. 
   int cb = pidl->mkid.cb; 

   // Allocate a new item identifier list. 
   LPITEMIDLIST pidlNew = (LPITEMIDLIST) 
   g_pMalloc->Alloc(cb + sizeof(USHORT)); 
   if (pidlNew == NULL) 
      return NULL; 

   // Copy the specified item identifier. 
   CopyMemory(pidlNew, pidl, cb); 

   // Append a terminating zero. 
   *((USHORT *) (((LPBYTE) pidlNew) + cb)) = 0; 

   return pidlNew; 
   } 

// PrintStrRet - prints the contents of a STRRET structure. 
// pidl - PIDL containing the display name if lpStr->uType is STRRET_OFFSET. 
// lpStr - address of the STRRET structure. 
void CDriveCtrl::PrintStrRet(LPITEMIDLIST pidl, LPSTRRET lpStr, char* name) 
   { 
   LPSTR lpsz; 
   int cch; 
    
   switch (lpStr->uType) 
      { 
      case STRRET_WSTR: 
         cch = WideCharToMultiByte( CP_ACP, 
                                    0, 
                                    lpStr->pOleStr, 
                                    -1, 
                                    NULL, 
                                    0, 
                                    NULL, 
                                    NULL); 
         lpsz = (LPSTR) g_pMalloc->Alloc(cch); 
         if (lpsz != NULL) 
            { 
            WideCharToMultiByte( CP_ACP, 
                                 0, 
                                 lpStr->pOleStr, 
                                 -1, 
                                 lpsz, 
                                 cch, 
                                 NULL, 
                                 NULL); 
            strcpy(name, lpsz); 
            g_pMalloc->Free(lpsz); 
            } 
         break; 
    
      case STRRET_OFFSET: 
         strcpy(name, ((char *) pidl) + lpStr->uOffset); 
         break; 
    
      case STRRET_CSTR: 
         strcpy(name, lpStr->cStr); 
         break; 
      } 
   } 

void CDriveCtrl::Refresh()
{
	/*A tooltip object is created. */
    //m_pToolTip = new CToolTipCtrl;
    //if(m_pToolTip->Create(this/*, dwStyle*/)) {
	//	m_pToolTip->AddTool(this, LPSTR_TEXTCALLBACK );
	//	m_pToolTip->Activate(TRUE);
	//	int width = m_pToolTip->SetMaxTipWidth(256);
	//}

	// Populate the tree view with desctop.
    HTREEITEM hItem,hSelectItem;
	TCHAR pszPath[MAX_PATH],pszName[MAX_PATH], home[_MAX_PATH];
	CString strPath,itemname;

	if(GetSpecialFolder(CSIDL_DESKTOPDIRECTORY, pszPath, pszName)) {
		strPath  = _T(pszPath);
		hItem = AddItem(pszName, ILI_DESKTOP,ILI_DESKTOP, TVGN_ROOT, (LPCTSTR)strPath);

		SetButtonState (hItem, pszPath);

		if (hItem != NULL) {
			strPath = GetItemText(hItem);
			Expand (hItem, TVE_EXPAND);
			Select (hItem, TVGN_CARET);
		}
	}

	//MyDocumennt
	if(GetSpecialFolder(CSIDL_PERSONAL, pszPath, pszName)) {
		strPath  = _T(pszPath);
		hItem = AddItem(pszName, ILI_DOCUMENT,ILI_DOCUMENT, TVGN_ROOT, (LPCTSTR)strPath);
		//SetButtonState (hItem, pszPath);
		if(GetHomeDirectory(home, _MAX_PATH)) {
			//
			AddDirectories(hItem, pszPath);
			//update
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);

			//@/->\\
			ChangeSlash( home );
			hSelectItem = SelectDirectory(hItem, pszPath, home);
		}
	}

	//MyComputer
	if(GetSpecialFolder(CSIDL_DRIVES, pszPath, pszName)) {
		m_strMyComputer = _T(pszName);
		strPath  = _T(pszName);
		hItem = AddItem(pszName, ILI_COMPUTER,ILI_COMPUTER, TVGN_ROOT, (LPCTSTR)strPath);
		//dummy folder
		AddItem(_T(""), ILI_FLOPPY,ILI_FLOPPY, hItem, (LPCTSTR)strPath);
	}
	//MyNetwork
	if(GetSpecialFolder(CSIDL_NETWORK, pszPath, pszName)) {
		m_strMyNetwork = _T(pszName);
		strPath  = _T(pszName);
		hItem = AddItem(pszName, ILI_NET_DRIVE,ILI_NET_DRIVE, TVGN_ROOT, (LPCTSTR)strPath);
		//dummy folder
		AddItem(_T(""), ILI_FLOPPY,ILI_FLOPPY, hItem, (LPCTSTR)strPath);
	}

}

// /->\\
void CDriveCtrl::ChangeSlash(char *path)
{
	char tmp[_MAX_PATH], buf[_MAX_PATH];
	int i,j;

	strcpy(tmp, path);
	i = 0; j = 0;
	while(tmp[i] != '\0') {
		if(tmp[i] == '/') {
			buf[j++] = '\\';
		}
		else {
			buf[j++] = tmp[i];
		}
		i++;
	}
	buf[j++] = '\0';
	strcpy(path, buf);
}


int CDriveCtrl::AddLocals(HTREEITEM hItem, char path[])
{
	NETRESOURCE parent;
	LPNETRESOURCE pnr;
	HANDLE hEnum;
	DWORD BufSize,ResNum,RetCode;
    HTREEITEM hNewItem;
	TCHAR pszLinkPath[MAX_PATH];
	int nCount;
	BOOL bContext;
	DWORD Flags, ConInfo, AccessNameSize = 1000;
	char AccessName[1000];

	if(strlen(path) < 1) {
		ZeroMemory(&parent, sizeof(parent));
		pnr = (LPNETRESOURCE)GlobalAlloc(GMEM_FIXED, 1000);

		parent.dwScope = RESOURCE_GLOBALNET;
		parent.dwType = RESOURCETYPE_ANY;
		parent.dwDisplayType = RESOURCEDISPLAYTYPE_DOMAIN;
		parent.dwUsage = RESOURCEUSAGE_CONTAINER;
		parent.lpRemoteName = path/*""*/;
		parent.lpProvider = "Microsoft Windows Network";

		WNetOpenEnum(RESOURCE_CONTEXT/*RESOURCE_GLOBALNET*/, 
			RESOURCETYPE_DISK/*RESOURCETYPE_ANY*/, 0, NULL/*&parent*/, &hEnum);
		while(BufSize = 1000, ResNum = 1, WNetEnumResource(hEnum, &ResNum, pnr, &BufSize) == NO_ERROR) {
			sprintf(pszLinkPath, "%s",pnr->lpRemoteName);
		}
		WNetCloseEnum(hEnum);
		GlobalFree(pnr);
		bContext = TRUE;
	}
	else {
		//When a remote name is specified 
		strcpy(pszLinkPath, path);
		bContext = FALSE;
	}

	nCount = 0;
	if(strlen(pszLinkPath) > 0) {
		//Local
		ZeroMemory(&parent, sizeof(parent));
		pnr = (LPNETRESOURCE)GlobalAlloc(GMEM_FIXED, 1000);

retry:
		parent.dwScope = RESOURCE_GLOBALNET;
		parent.dwType = RESOURCETYPE_ANY;
		parent.dwDisplayType = RESOURCEDISPLAYTYPE_DOMAIN;
		parent.dwUsage = RESOURCEUSAGE_CONTAINER;
		parent.lpRemoteName = pszLinkPath;
		parent.lpProvider = "Microsoft Windows Network";

		if(bContext)
			RetCode = WNetOpenEnum(RESOURCE_CONTEXT/*RESOURCE_GLOBALNET*/, 
				RESOURCETYPE_DISK/*RESOURCETYPE_ANY*/, 0, NULL/*&parent*/, &hEnum);
		else
			RetCode = WNetOpenEnum(RESOURCE_GLOBALNET, 
				RESOURCETYPE_DISK/*RESOURCETYPE_ANY*/, 0, &parent, &hEnum);
		if(RetCode == NO_ERROR) {
			while(BufSize = 1000, ResNum = 1, WNetEnumResource(hEnum, &ResNum, pnr, &BufSize) == NO_ERROR) {
				if(pnr->lpRemoteName != NULL) {
					sprintf(pszLinkPath, "%s",pnr->lpRemoteName);
					hNewItem = AddItem (_T(pszLinkPath),
						ILI_NET_DRIVE, ILI_NET_DRIVE, hItem,(LPCTSTR)pszLinkPath);
					// dummy
					AddItem (_T(""),
						ILI_CLOSED_FOLDER, ILI_OPEN_FOLDER, hNewItem,(LPCTSTR)pszLinkPath);
					nCount++;
				}
			}
			WNetCloseEnum(hEnum);
		}
		else {
			parent.dwDisplayType = RESOURCETYPE_ANY;
			parent.lpLocalName = NULL;
			parent.lpRemoteName = pszLinkPath;
			parent.lpProvider = "Microsoft Windows Network";
			Flags = 0 | CONNECT_INTERACTIVE | CONNECT_PROMPT;
			RetCode = WNetUseConnection(
				NULL,
				&parent,
				"",
				"",
				Flags,
				AccessName,
				&AccessNameSize,
				&ConInfo);
			if(RetCode != NO_ERROR) {
				this->MessageBox("WNetUseConnection failed."
					"FileViewer", MB_OK);
			}
			else {
				bContext = FALSE;
				goto retry;
			}
		}
		GlobalFree(pnr);
	}
	return nCount;
}

int CDriveCtrl::AddNetworks(HTREEITEM hItem)
{
	TCHAR pszPath[MAX_PATH],pszName[MAX_PATH];
    HANDLE hFind;
    WIN32_FIND_DATA fd;
    HTREEITEM hNewItem;
	CString strName,strData,strPath;
	BOOL bResult;
	int nCount=0;
	TCHAR pszLinkPath[MAX_PATH],pszLinkDescription[MAX_PATH];

	if(GetSpecialFolder(CSIDL_NETHOOD, pszPath, pszName)) {
		strPath = pszPath;
		if (strPath.Right (1) != _T ("\\"))
			strPath += _T ("\\");
		strPath += _T ("*.*");

		if ((hFind = ::FindFirstFile (strPath, &fd)) == INVALID_HANDLE_VALUE) {
			//́A
			if (GetParentItem (hItem) == NULL) {
				strData = pszPath;
				AddItem (_T (""), ILI_NET_DRIVE,
					ILI_NET_DRIVE, hItem, (LPCTSTR)strData);
			}
			return 0;
		}

		do {
			strData = pszPath;
			strData += "\\";
			strData += (LPCTSTR) &fd.cFileName;
			if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
				CString strComp = (LPCTSTR) &fd.cFileName;
				if ((strComp != _T (".")) && (strComp != _T (".."))) {
					CString strNewPath = pszPath;
					if (strNewPath.Right (1) != _T ("\\"))
						strNewPath += _T ("\\");

					strNewPath += (LPCTSTR) &fd.cFileName;
					bResult = AddNetworkItem (	strNewPath,
										pszLinkPath,
										pszLinkDescription);
					if(bResult) {
						//f[^ݒ
						hNewItem = AddItem (_T(fd.cFileName),
							ILI_NET_DRIVE, ILI_NET_DRIVE, hItem,(LPCTSTR)pszLinkPath);
						//if(strlen(pszLinkPath) >0) {
						// dummy
						AddItem (_T(""),
							ILI_CLOSED_FOLDER, ILI_OPEN_FOLDER, hNewItem,(LPCTSTR)pszLinkPath);
						//}
					}
					nCount++;
				}
			}
		} while (::FindNextFile (hFind, &fd));

		::FindClose (hFind);
	}
	return nCount;
}

BOOL CDriveCtrl::AddNetworkItem(	LPCTSTR	lpszPath,
									LPSTR	lpszLinkPath,
									LPSTR	lpszLinkDescription)
{
    HANDLE hFind;
    WIN32_FIND_DATA fd;
    BOOL bResult = FALSE;
	CString pszName,strNewPath;

    CString strPath = lpszPath;
    if (strPath.Right (1) != _T ("\\"))
        strPath += _T ("\\");
    strPath += _T ("*.*");

    if ((hFind = ::FindFirstFile (strPath, &fd)) == INVALID_HANDLE_VALUE)
        return bResult;

    do {
		// path
		strNewPath = lpszPath;
		strNewPath += _T("\\");
		strNewPath += (LPCTSTR) &fd.cFileName;

        if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
			// nothing
        }
		else {
			// delect lnkfile
            pszName = (LPCTSTR) &fd.cFileName;
			if(pszName.Find(_T(".lnk")) != -1) {
				if(NOERROR == GetLinkInfo(strNewPath, lpszLinkPath, lpszLinkDescription)) {
					bResult = TRUE;
				}
			}
		}
    } while (::FindNextFile (hFind, &fd));

    ::FindClose (hFind);
    return bResult;
}

HRESULT CDriveCtrl::GetLinkInfo(LPCTSTR lpszLinkName,
								LPSTR	lpszPath,
								LPSTR	lpszDescription)
{
	HRESULT hres;
	IShellLink *pShLink;
	WIN32_FIND_DATA wfd;

	*lpszPath = '\0';
	*lpszDescription = '\0';

	hres = CoCreateInstance( CLSID_ShellLink,
							NULL,
							CLSCTX_INPROC_SERVER,
							IID_IShellLink,
							(LPVOID *)&pShLink );
	if (SUCCEEDED(hres))
	{
		IPersistFile *ppf;
		hres = pShLink->QueryInterface(	IID_IPersistFile,
										(LPVOID *)&ppf);
		if (SUCCEEDED(hres))
		{
			WORD wsz[MAX_PATH];
			MultiByteToWideChar(	CP_ACP, 0,
									lpszLinkName,
									-1, (LPWSTR)wsz, MAX_PATH );
			hres = ppf->Load((LPWSTR)wsz, STGM_READ);
			if (SUCCEEDED(hres))
			{
				hres = pShLink->Resolve(	GetParent()->GetSafeHwnd(),
										SLR_ANY_MATCH | SLR_NO_UI );
				if (SUCCEEDED(hres))
				{
					// get path
					hres = pShLink->GetPath(	lpszPath,
												MAX_PATH,
												&wfd,
												SLGP_SHORTPATH );
					if (SUCCEEDED(hres))
					{
						hres = pShLink->GetDescription(	lpszDescription,
														MAX_PATH);

					}
				}
			}
			ppf->Release();
		}

		pShLink->Release();
	}
	return hres;
}

int CDriveCtrl::AddDrives(HTREEITEM hParent)
{
    int nPos = 0;
    int nDrivesAdded = 0;
    CString string = _T ("?:\\");

    DWORD dwDriveList = ::GetLogicalDrives ();

    while (dwDriveList) {
        if (dwDriveList & 1) {
            string.SetAt (0, _T ('A') + nPos);
            if (AddDriveItem (string, hParent)) {
                nDrivesAdded++;
			}
        }
        dwDriveList >>= 1;
        nPos++;
    }
    return nDrivesAdded;
}

BOOL CDriveCtrl::AddDriveItem(LPCTSTR pszDrive, HTREEITEM hParent)
{
    CString string;
    HTREEITEM hItem;

    UINT nType = ::GetDriveType (pszDrive);

    switch (nType) {

    case DRIVE_REMOVABLE:
        hItem = AddItem ((char *)pszDrive, ILI_FLOPPY,
            ILI_FLOPPY, hParent, pszDrive);
        AddItem (_T (""), ILI_CLOSED_FOLDER,
            ILI_CLOSED_FOLDER, hItem, pszDrive);
        break;

    case DRIVE_FIXED:
    case DRIVE_RAMDISK:
        hItem = AddItem ((char *)pszDrive, ILI_HARD_DISK,
            ILI_HARD_DISK, hParent, pszDrive);
        SetButtonState (hItem, pszDrive);
        break;

    case DRIVE_REMOTE:
        hItem = AddItem ((char *)pszDrive, ILI_NET_DRIVE,
            ILI_NET_DRIVE, hParent, pszDrive);
        SetButtonState (hItem, pszDrive);
        break;

    case DRIVE_CDROM:
        hItem = AddItem ((char *)pszDrive, ILI_CD_ROM,
            ILI_CD_ROM, hParent, pszDrive);
        AddItem (_T (""), ILI_CLOSED_FOLDER,
            ILI_CLOSED_FOLDER, hItem, pszDrive);
        break;

    default:
        return FALSE;
    }
    return TRUE;
}

BOOL CDriveCtrl::SetButtonState(HTREEITEM hItem, LPCTSTR pszPath)
{
    HANDLE hFind;
    WIN32_FIND_DATA fd;
    BOOL bResult = FALSE;
	CString pszName,strNewPath;
	int nImage;

    CString strPath = pszPath;
    if (strPath.Right (1) != _T ("\\"))
        strPath += _T ("\\");
    strPath += _T ("*.*");

    if ((hFind = ::FindFirstFile (strPath, &fd)) == INVALID_HANDLE_VALUE)
        return bResult;

    do {
		// path
		strNewPath = pszPath;
		strNewPath += _T("\\");
		strNewPath += (LPCTSTR) &fd.cFileName;

        if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            CString strComp = (LPCTSTR) &fd.cFileName;
            if ((strComp != _T (".")) && (strComp != _T (".."))) {
                AddItem (_T (""), ILI_CLOSED_FOLDER,
                    ILI_CLOSED_FOLDER, hItem, (LPCTSTR)strNewPath);
                bResult = TRUE;
                break;
            }
        }
		else {	// append 2003.5.27
            pszName = (LPCTSTR) &fd.cFileName;
			if(m_FileType == FILE_SL) {
				if(pszName.Find(_T(".sl")) != -1) {
					AddItem (_T (fd.cFileName), ILI_SCRIPT,
						ILI_SCRIPT, hItem, (LPCTSTR)strNewPath);
				}
			}
			else if(m_FileType == FILE_MDL) {
				if(pszName.Find(_T(".mdl")) != -1)
					AddItem (_T (fd.cFileName), ILI_MODEL,
						ILI_MODEL, hItem, (LPCTSTR)strNewPath);
			}
			else if(m_FileType == FILE_TXT) {
				if(pszName.Find(_T(".txt")) != -1)
					AddItem (_T (fd.cFileName), ILI_TEXT,
						ILI_TEXT, hItem, (LPCTSTR)strNewPath);
			}
			else {
				if(pszName.Find(_T(".sl")) != -1)
					nImage = ILI_SCRIPT;
				else if(pszName.Find(_T(".txt")) != -1)
					nImage = ILI_TEXT;
				else if(pszName.Find(_T(".mdl")) != -1)
					nImage = ILI_MODEL;
				else
					nImage = ILI_FILE;
				AddItem (_T (fd.cFileName), nImage,
					nImage, hItem, (LPCTSTR)strNewPath);
			}
		}
    } while (::FindNextFile (hFind, &fd));

    ::FindClose (hFind);
    return bResult;
}


void CDriveCtrl::OnItemexpanding(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NM_TREEVIEW* pNMTreeView = (NM_TREEVIEW*)pNMHDR;
    HTREEITEM hItem = pNMTreeView->itemNew.hItem;
    char msg[_MAX_PATH];
	PATHINFO* pItem;
	pItem = (PATHINFO*)GetItemData(hItem);
	CString string = _T(pItem->path);

    *pResult = FALSE;

    if (pNMTreeView->action == TVE_EXPAND) {
		if(string == m_strMyComputer) {
	        DeleteAllChildren (hItem);
			AddDrives(hItem);
		}
		/* conventional method */
		//else if(string == m_strMyNetwork) {
	    //    DeleteAllChildren (hItem);
		//	AddNetworks(hItem);
		//}
		//else if(string == "") {
		//	//nearby computer
	    //    DeleteAllChildren (hItem);
		//	AddLocals(hItem, pItem->path);
		//}
		/* new method */
		else if(string == m_strMyNetwork) {
	        DeleteAllChildren (hItem);
			AddLocals(hItem, "");
		}
		else {
			DeleteAllChildren (hItem);
			if (AddDirectories (hItem, string) == 0) {
				sprintf(msg,"%s cannot be accessed.\nAccess was refused. ",
					string);
				this->MessageBox(msg, "FileViewer",MB_ICONERROR | MB_OK);
				*pResult = TRUE;
			}
		}
    }
    else { // pNMTreeView->action == TVE_COLLAPSE
#if 0
        DeleteAllChildren (hItem);
        if (GetParentItem (hItem) == NULL)
            AddItem (_T (""), ILI_CLOSED_FOLDER,
                ILI_CLOSED_FOLDER, hItem, (LPCTSTR)string);
        else
            SetButtonState (hItem, string);
#endif
    }
}

CString CDriveCtrl::GetPathFromItem(HTREEITEM hItem)
{
    CString strResult = GetItemText (hItem);

    HTREEITEM hParent;
    while ((hParent = GetParentItem (hItem)) != NULL) {
        CString string = GetItemText (hParent);
        if (string.Right (1) != _T ("\\"))
            string += _T ("\\");
        strResult = string + strResult;
        hItem = hParent;
    }
    return strResult;
}

void CDriveCtrl::DeleteFirstChild(HTREEITEM hItem)
{
    HTREEITEM hChildItem;
    if ((hChildItem = GetChildItem (hItem)) != NULL)
        DeleteItem (hChildItem);
}

void CDriveCtrl::DeleteAllChildren(HTREEITEM hItem)
{
    HTREEITEM hChildItem;
	PATHINFO* pItem;
	if(!hItem)
		return;
    if ((hChildItem = GetChildItem (hItem)) == NULL)
        return;

    do {
        HTREEITEM hNextItem = GetNextSiblingItem (hChildItem);
		DeleteChildItemData(hChildItem);	// fixed leaks
		DeleteAllChildren(hNextItem);
		pItem = (PATHINFO *)GetItemData(hChildItem);
		if(pItem)
			delete pItem;
        DeleteItem (hChildItem);
        hChildItem = hNextItem;
    } while (hChildItem != NULL);
}

//
//JĂꍇւ̑Ή
void CDriveCtrl::DeleteLocalFiles(HTREEITEM hItem)
{
    HTREEITEM hChildItem;
	PATHINFO* pItem;
	CString name;

    if ((hChildItem = GetChildItem (hItem)) == NULL)
        return;

    do {
        HTREEITEM hNextItem = GetNextSiblingItem (hChildItem);
		pItem = (PATHINFO *)GetItemData(hChildItem);
		if(!IsDirectory(pItem->path)) {
			// fBNgłȂΏ
			if(pItem)
				delete pItem;
			DeleteItem (hChildItem);
		}
        hChildItem = hNextItem;
    } while (hChildItem != NULL);
}

int CDriveCtrl::AddDirectories(HTREEITEM hItem, LPCTSTR pszPath)
{
    HANDLE hFind;
    WIN32_FIND_DATA fd;
    HTREEITEM hNewItem;
	CString pszName,strData;
	TCHAR pszLinkPath[_MAX_PATH];

    int nImage,nCount = 0;

    CString strPath = pszPath;
    if (strPath.Right (1) != _T ("\\"))
        strPath += _T ("\\");
    strPath += _T ("*.*");

    if ((hFind = ::FindFirstFile (strPath, &fd)) == INVALID_HANDLE_VALUE) {
		// when failed
		strcpy(pszLinkPath, pszPath);
		nCount = AddLocals(hItem, pszLinkPath);
        return nCount;
    }

    do {
		strData = pszPath;
		if (strData.Right (1) != _T ("\\"))
			strData += "\\";
		strData += (LPCTSTR) &fd.cFileName;
        if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            CString strComp = (LPCTSTR) &fd.cFileName;
            if ((strComp != _T (".")) && (strComp != _T (".."))) {
                hNewItem =
                    AddItem (_T(fd.cFileName),
                    ILI_CLOSED_FOLDER, ILI_OPEN_FOLDER, hItem,(LPCTSTR)strData);

                CString strNewPath = pszPath;
                if (strNewPath.Right (1) != _T ("\\"))
                    strNewPath += _T ("\\");

                strNewPath += (LPCTSTR) &fd.cFileName;
                SetButtonState (hNewItem, strNewPath);
                nCount++;
            }
        }
		else {	// append 2003.5.27
            pszName = (LPCTSTR) &fd.cFileName;
			if(m_FileType == FILE_SL) {
				if(pszName.Find(_T(".sl")) != -1) {
					AddItem (_T (fd.cFileName), ILI_SCRIPT,
						ILI_SCRIPT, hItem,(LPCTSTR)strData);
                nCount++;
				}
			}
			else if(m_FileType == FILE_MDL) {
				if(pszName.Find(_T(".mdl")) != -1) {
					AddItem (_T (fd.cFileName), ILI_MODEL,
						ILI_MODEL, hItem,(LPCTSTR)strData);
                nCount++;
				}
			}
			else if(m_FileType == FILE_TXT) {
				if(pszName.Find(_T(".txt")) != -1) {
					AddItem (_T (fd.cFileName), ILI_TEXT,
						ILI_TEXT, hItem,(LPCTSTR)strData);
                nCount++;
				}
			}
			else {
				if(pszName.Find(_T(".sl")) != -1)
					nImage = ILI_SCRIPT;
				else if(pszName.Find(_T(".txt")) != -1)
					nImage = ILI_TEXT;
				else if(pszName.Find(_T(".mdl")) != -1)
					nImage = ILI_MODEL;
				else
					nImage = ILI_FILE;
				AddItem (_T (fd.cFileName), nImage,
					nImage, hItem,(LPCTSTR)strData);
                nCount++;
			}
		}
    } while (::FindNextFile (hFind, &fd));

    ::FindClose (hFind);

	//The results is sorted. 
	SortChildren(hItem);

    return nCount;
}

HTREEITEM CDriveCtrl::SelectDirectory(HTREEITEM hItem, LPCTSTR pszPath, LPCTSTR pszHome)
{
    HTREEITEM hChild;
	HTREEITEM hSelectItem = NULL;
	CString pszName;
	PATHINFO* pItem;

	hChild = GetChildItem(hItem);
	while (hChild) {
		pItem = (PATHINFO *)GetItemData(hChild);
		pszName = _T(pItem->path);
		if(pszName == _T(pszHome)) {
			//WJ
			if(AddDirectories(hChild, pItem->path)) {
				hSelectItem = hChild;
				Select (hChild, TVGN_CARET);
				Expand (hChild, TVE_EXPAND);
				break;
			}
		}
		hChild = GetNextItem(hChild,TVGN_NEXT);
	}
    return hSelectItem;
}

void CDriveCtrl::OnSelchanged(NMHDR* pNMHDR, LRESULT* pResult) 
{
    NM_TREEVIEW* pNMTreeView = (NM_TREEVIEW*) pNMHDR;
    HTREEITEM hItem = pNMTreeView->itemNew.hItem;
	if(!hItem)
		return;
	//if(hItem == pNMTreeView->itemOld.hItem)
	//	return;
	//It develops, when the selected item is a holder. 
	m_bSelScript = TRUE;
	*pResult = 0;

CString name;
hItem = GetStateSearch(TVIS_DROPHILITED);
if(hItem)
	name = GetItemText(hItem);
/*
	PATHINFO* pItem;
	pItem = (PATHINFO *)GetItemData(hItem);
	if(IsDirectory((LPCTSTR)pItem->path)) {
		DeleteAllChildren (hItem);
		if(AddDirectories(hItem, pItem->path)) {
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);
		}
	}
*/
}

void CDriveCtrl::SetFtype(int ftype) 
{
	// \t@C`
	m_FileType = ftype;

	HTREEITEM hItem;
	CString name, strPath;
	UINT state;
	PATHINFO* pItem;

	// ACeXV
	hItem = GetFirstVisibleItem();
	while(hItem) {
		name = GetItemText(hItem);
		state = GetItemState(hItem, TVIF_STATE);
		if((state & TVIS_EXPANDED)) {
			// test
			fprintf(stdout,"%s\n",(LPCTSTR)name);
			// hItemXV
			pItem = (PATHINFO*)GetItemData(hItem);
			strPath = _T(pItem->path);
			DeleteLocalFiles (hItem);
			AddLocalFiles (hItem, strPath);
			//update
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);
		}
		hItem = GetNextVisibleItem(hItem);
	}
	
}


void CDriveCtrl::OnEditCut() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	//File폜Ăݔ
	BOOL delFlag = FALSE;
	CString strPath;
	PATHINFO* pItem;
	HTREEITEM hItem;
	
	hItem = GetSelectedItem ();
	if(hItem) {
		//strPath = GetPathFromItem (hItem);
		pItem = (PATHINFO*)GetItemData(hItem);
		strPath = _T(pItem->path);

		CFileStatus status;
		CFile::GetStatus(strPath, status);
		if(status.m_attribute != 0x10) {
			//fBNgȊO
#if 0
			//Sȍ폜
			_unlink(strPath);
#else
			//ݔ
			delFlag = TRUE;
#endif
		}

		if(delFlag) {
			strcpy(g_charScriptText, strPath);
			OnEditDelete();
		}
	}	
}

void CDriveCtrl::OnUpdateEditCut(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	//File폜Lɂ	
	HTREEITEM hItem = GetSelectedItem ();
	if(!hItem)
		pCmdUI->Enable(FALSE);
	
}

void CDriveCtrl::OnScriptRun() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	HTREEITEM hItem = GetSelectedItem ();
	//CString strPath = GetPathFromItem (hItem);
	PATHINFO* pItem;
	pItem = (PATHINFO*)GetItemData(hItem);
	CString strPath = _T(pItem->path);
	// ChildFrameŎs
	strcpy(g_charScriptText, strPath);
	g_pConsole->ScriptRun(INLINE_MODE);

	//this->GetParent()->PostMessage(ID_SCRIPT_EXE, INLINE_MODE, 0L);
	// Active
	//g_pConsole->SetFocus();
	
}

void CDriveCtrl::OnFileOpen() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	HTREEITEM hItem = GetSelectedItem ();
	//CString strPath = GetPathFromItem (hItem);
	PATHINFO* pItem;
	pItem = (PATHINFO*)GetItemData(hItem);
	CString strPath = _T(pItem->path);
	char strFile[_MAX_PATH];
	BOOL fg;
	fg = ((CShellApp*)AfxGetApp())->TranslateReturnCode(strPath, strFile);
	if(fg) {
		//successful
		if(::CopyFile(strFile, strPath, FALSE)) {
			//succesful
		}
		else {
			AfxMessageBox("TranslateReturnCode failed.",MB_OK,NULL);
		}
	}

	g_pEditDocTemplate->OpenDocumentFile(strPath);
	
}

//void CDriveCtrl::OnUpdateFileOpen(CCmdUI* pCmdUI) 
//{
//IĂt@CslȂL	
//	HTREEITEM hItem = GetSelectedItem ();
//	if(hItem) {
//		CString strPath = GetPathFromItem (hItem);
//		if(strPath.Find(".sl") != -1 || strPath.Find(".mdl") != -1
//			|| strPath.Find(".txt") != -1)
//			pCmdUI->Enable(TRUE);
//		else
//			pCmdUI->Enable(FALSE);
//	}
//	else
//		pCmdUI->Enable(FALSE);
//}

void CDriveCtrl::OnBegindrag(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NM_TREEVIEW* pNMTreeView = (NM_TREEVIEW*)pNMHDR;
	CPoint ptAction;
	CString strPath;
	HTREEITEM hItem;

	//hbOEC[W쐬
	hItem = (HTREEITEM)pNMTreeView->itemNew.hItem;
	if(!hItem)
		return;

	// path
	PATHINFO* pItem;
	pItem = (PATHINFO*)GetItemData(hItem);
	strPath = _T(pItem->path);

	//If it is a holder drag is cancel. 
	if(IsDirectory(strPath))
		return;
	strcpy(g_charScriptText, strPath);

	// image
	m_pDragImage = CreateDragImage(hItem);

	//ACeEC[W쐬̍W擾
	ptAction = pNMTreeView->ptDrag;

	//hbOJn
	m_pDragImage->BeginDrag(0,CPoint(0,0));
	//ǉ
	ShowCursor(FALSE);
	//NULLw
	m_pDragImage->DragEnter(this, ptAction);
	/*A drug item is set up. */
	m_hDragItem = hItem;

	/**/
#if 0
	//hItem = GetStateSearch(TVIS_DROPHILITED);
	hItem = GetStateSearch(TVIS_SELECTED);
	if(hItem) {
		//SetItemState( hItem, 0x0, TVIS_DROPHILITED );
		SetItemState( hItem, 0x0, TVIS_SELECTED );
	}
#endif

	//J[\Lv`
	::SetCapture(GetSafeHwnd());
	
	*pResult = 0;
}

void CDriveCtrl::OnMouseMove(UINT nFlags, CPoint point) 
{
	// TODO: ̈ʒuɃbZ[W nhp̃R[hǉ邩܂̓ftHg̏ĂяoĂ
	HTREEITEM hItem;
	PATHINFO* pItem;
	CDC* pDC = GetDC();
	CRect rcClient;
	UINT hitFlag;

	if(m_pDragImage) {
		/**/
		m_pDragImage->DragMove(point);
		m_pDragImage->Draw(pDC, 0, point, ILD_NORMAL);

		GetClientRect( rcClient );
		if(rcClient.PtInRect( point ) == FALSE) {
			//}EẌO
			m_pDragImage->EndDrag();

			// set g_pConsole->m_hDrag;
			hItem = GetSelectedItem ();
			pItem = (PATHINFO*)GetItemData(hItem);
			if(_access(pItem->path, 0) == 0) {
				g_pConsole->m_pDragImage = CreateDragImage(hItem);
				g_pConsole->m_strDragText = _T(pItem->path);
			}

			delete m_pDragImage;
			m_pDragImage = NULL;
			::ReleaseCapture();
			ShowCursor(TRUE);
		}
		else {
			// directoryɃqbgꍇ
			hItem = HitTest(point, &hitFlag);
			if(hItem) {
				pItem = (PATHINFO*)GetItemData(hItem);
				if(IsDirectory(pItem->path)) {
					/**/
					TreeView_SelectDropTarget(this->GetSafeHwnd(), hItem);
					Invalidate(FALSE);
				}
			}
		}
	}

	/*The present mouse position is registered. */
	m_curMousePos = point;
	CRect rect;

	hItem = HitTest(m_curMousePos);
	if(hItem) {
		GetItemRect(hItem, rect, TRUE);
		if(!rect.PtInRect(m_curMousePos)) {
			if(m_pToolTip)
				m_pToolTip->Pop();
		}
		else if(hItem != m_hCurItem) {
			if(m_pToolTip)
				m_pToolTip->Pop();
		}
	}
	else {
		if(m_pToolTip)
			m_pToolTip->Pop();
	}
	
	CTreeCtrl::OnMouseMove(nFlags, point);
}

HTREEITEM CDriveCtrl::AddItem(char* name, int image, int nimage,
				HTREEITEM hParent, const char *path)
{
	HTREEITEM hChild;
	CFile fd;
	CFileStatus fs;
	char msg[_MAX_PATH];
	CTime tm;
	FILE *fp;

	//hChild = InsertItem (name, image, nimage, hParent);

    PATHINFO* pItem;
	try {
		pItem = new PATHINFO;
	}
	catch (CMemoryException* e) {
		e->Delete ();
		return NULL/*hChild*/;
	}
	// set item data
	strcpy(pItem->path, path);
	fd.GetStatus(path, fs); 
	tm=fs.m_mtime;
	pItem->attr = fs.m_mtime.Format( "%Y/%m/%d %H:%M" );

	// add item
	if(IsDirectory(path))
		strcpy(msg, name);
	else {
		// set tooltip
		pItem->comm = _T("");
		if(strstr(path, ".sl")) {
			fp = fopen(path, "r");
			if(fp) {
				fgets(msg, _MAX_PATH, fp);
				if(strlen(msg) > 1) {
					msg[strlen(msg)-1] = '\0';
					pItem->comm = _T(msg);
				}
				fclose(fp);
			}
		}
		//sprintf(msg, "%s (%s)", name, pItem->attr);
		sprintf(msg, "%s", name);
	}
	hChild = InsertItem (msg, image, nimage, hParent);
	SetItemData(hChild, (DWORD)pItem);

	return hChild;
}

void CDriveCtrl::DeleteChildItemData(HTREEITEM hParent)
{
	HTREEITEM hItem;
	CString strText;
	PATHINFO* pItem;

	hItem = GetChildItem (hParent);
	while (hItem != NULL) {
		//strText = GetItemText (hItem);
		DeleteChildItemData(hItem);
		pItem = (PATHINFO *)GetItemData(hItem);
		strText = _T(pItem->path);
		if(pItem) {
			delete pItem;
		}
		hItem = GetNextSiblingItem (hItem);
	}
}


void CDriveCtrl::OnDestroy() 
{
	HTREEITEM hItem;
	CString strText;
	UINT nCount;
	PATHINFO* pItem;

	nCount = GetCount();

	hItem = GetRootItem();
	while (hItem != NULL) {
		//strText = GetItemText (hItem);
		DeleteChildItemData(hItem);
		pItem = (PATHINFO *)GetItemData(hItem);
		strText = _T(pItem->path);
		if(pItem) {
			delete pItem;
		}
		hItem = GetNextSiblingItem (hItem);
	}

	CTreeCtrl::OnDestroy();
	
}

void CDriveCtrl::OnDblclk(NMHDR* pNMHDR, LRESULT* pResult) 
{
	HTREEITEM hItem = GetSelectedItem ();
	PATHINFO* pItem;
	pItem = (PATHINFO*)GetItemData(hItem);
	CString strPath = _T(pItem->path);

	*pResult = 0;

	// exist ?
	if(Access((LPCTSTR)strPath,SL_FATTR_FOK) != 0)
		return;

	// directory ?
	if(IsDirectory((LPCTSTR)strPath))
		return;

	// File type ?
	int ret1,ret2,ret3;
	ret1 = strPath.Find(_T(".sl"));
	ret2 = strPath.Find(_T(".mdl"));
	ret3 = strPath.Find(_T(".txt"));
	if(ret1 == -1 && ret2 == -1 && ret3 == -1)
		return;

	// t@CJ
	OnFileOpen();
}

void CDriveCtrl::OnEditDelete() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	/*It checks whether I may delete. */
	char msg[_MAX_PATH];
	sprintf(msg,"Do you wish to delete %s ?",g_charScriptText);
	if(MessageBox(msg, PACKAGE_STRING, MB_YESNO | MB_ICONQUESTION) == IDNO)
		return;
	/*It moves to a garbage can. */
	SHFILEOPSTRUCT stFile;
	char src[_MAX_PATH];
	
	stFile.hwnd = GetSafeHwnd();
	stFile.wFunc = FO_DELETE;
	strcpy(src, g_charScriptText);
	src[strlen(src)+1] = '\0';
	stFile.pFrom = g_charScriptText;
	stFile.pTo = NULL;							//Ȃƃ_
	stFile.fFlags = FOF_ALLOWUNDO | FOF_NOCONFIRMATION;
	if(::SHFileOperation( &stFile ) == 0) {
		//succesful

		CString strPath;
		PATHINFO* pItem;
		HTREEITEM hItem;

		hItem = GetSelectedItem ();
		hItem = GetParentItem(hItem);
		//strPath = GetPathFromItem (hItem);
		pItem = (PATHINFO*)GetItemData(hItem);
		strPath = _T(pItem->path);
		DeleteAllChildren (hItem);
		AddDirectories(hItem, strPath);
	}
	
}

void CDriveCtrl::OnUpdateEditDelete(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}

void CDriveCtrl::OnRButtonDown(UINT nFlags, CPoint point) 
{
	// HitTests
	//j[
	PATHINFO* pItem;
	CMenu menu,*pPopup;
	HTREEITEM hItem;
	TVHITTESTINFO info;
	BOOL bHit = TRUE;
	CString strPath;

	info.pt = point;
	hItem = HitTest( &info );
	if( !hItem || info.flags != TVHT_ONITEMLABEL )
		bHit = FALSE;

	if(bHit) {
		//I
		SelectItem( hItem );

		pItem = (PATHINFO*)GetItemData(hItem);
		strPath = _T(pItem->path);

		// fBNgO
		//if(IsDirectory((LPCTSTR)strPath))
		//	return;
		strcpy(g_charScriptText, strPath);
	}

	//}C lbg[NA}C Rs[^͖
	if(strPath.Compare(_T("}C Rs[^")) == 0)
		return;
	if(strPath.Compare(_T("}C lbg[N")) == 0)
		return;

	// }EX|C^XN[Wɕϊ
	ClientToScreen( &point );
	// gɃtH[JXݒ
	SetForegroundWindow();

	// |bvAbvj[̍쐬
	menu.LoadMenu( IDR_POPUP_DRIVE );
	/*The state of the popup menu is set up. */
	if(!bHit) {
			menu.EnableMenuItem(ID_SCRIPT_RUN, MF_GRAYED);
			menu.EnableMenuItem(ID_SCRIPT_STEP, MF_GRAYED);
			menu.EnableMenuItem(ID_EDIT_COPY, MF_GRAYED);
			menu.EnableMenuItem(ID_EDIT_DELETE, MF_GRAYED);
			menu.EnableMenuItem(ID_EDIT_RENAME, MF_GRAYED);
			menu.EnableMenuItem(ID_FILE_OPEN, MF_GRAYED);
			menu.EnableMenuItem(ID_FILE_CHDIR, MF_GRAYED);
	}
	else {
		if(IsDirectory((LPCTSTR)strPath)) {
			menu.EnableMenuItem(ID_SCRIPT_RUN, MF_GRAYED);
			menu.EnableMenuItem(ID_SCRIPT_STEP, MF_GRAYED);
			menu.EnableMenuItem(ID_EDIT_COPY, MF_GRAYED);
			menu.EnableMenuItem(ID_EDIT_DELETE, MF_GRAYED);
			menu.EnableMenuItem(ID_EDIT_RENAME, MF_GRAYED);
			menu.EnableMenuItem(ID_FILE_OPEN, MF_GRAYED);
		}
		else {
			if(strstr(g_charScriptText, ".sl") == NULL)
				menu.EnableMenuItem(ID_SCRIPT_RUN, MF_GRAYED);
			if((strstr(g_charScriptText, ".sl") == NULL) &&
				(strstr(g_charScriptText, ".mdl") == NULL) &&
				(strstr(g_charScriptText, ".txt") == NULL)) {
				menu.EnableMenuItem(ID_EDIT_COPY, MF_GRAYED);
				menu.EnableMenuItem(ID_EDIT_COPY, MF_GRAYED);
			}
			menu.EnableMenuItem(ID_FILE_CHDIR, MF_GRAYED);
		}
		menu.EnableMenuItem(ID_FILE_ALL, MF_GRAYED);
		menu.EnableMenuItem(ID_FILE_SL, MF_GRAYED);
		menu.EnableMenuItem(ID_FILE_MDL, MF_GRAYED);
		menu.EnableMenuItem(ID_FILE_TXT, MF_GRAYED);

	}
	pPopup = menu.GetSubMenu( 0 );
	// |bvAbvj[\
	pPopup->TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON,
			point.x, point.y, this);
	// LoadMenuJ
	menu.DestroyMenu();

	//I
	//SelectItem( NULL );
	m_bPopupMenu = TRUE;
	
	CTreeCtrl::OnRButtonDown(nFlags, point);
}

void CDriveCtrl::OnEditRename() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	HTREEITEM hItem = GetSelectedItem ();
	if(!hItem)
		return;
	EditLabel(hItem);
	
}

void CDriveCtrl::OnUpdateEditRename(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}

void CDriveCtrl::OnEndlabeledit(NMHDR* pNMHDR, LRESULT* pResult) 
{
	TV_DISPINFO* pTVDispInfo = (TV_DISPINFO*)pNMHDR;
	// TODO: ̈ʒuɃRg[ʒmnhp̃R[hǉĂ

	//ACeeLXgĈ̂ɐݒ肵
	if(pTVDispInfo->item.pszText != NULL) {
		//ȑO̖O
		PATHINFO* pItem;
		char oldpath[_MAX_PATH],newpath[_MAX_PATH],drv[_MAX_PATH],dir[_MAX_PATH];

		pItem = (PATHINFO*)GetItemData(pTVDispInfo->item.hItem);
		strcpy(oldpath, pItem->path);
		_splitpath(pItem->path, drv, dir, NULL, NULL);

		//new
		sprintf(newpath,"%s%s%s",drv,dir,pTVDispInfo->item.pszText);
		strcpy(pItem->path, newpath);
		//pItem->path = _T(newpath);

		//ACeύX
		SetItemText(pTVDispInfo->item.hItem, pTVDispInfo->item.pszText);

		//t@CύX
		rename( oldpath, newpath );
	}

	*pResult = 0;
}

//
//Nbv{[hɃt@CeRs[
//
void CDriveCtrl::OnEditCopy() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	CString fname;
	BOOL flag;

	fname = _T(g_charScriptText);
	if(Access((LPCTSTR)fname,SL_FATTR_FOK) != 0)
		return;

	if(IsDirectory((LPCTSTR)fname))
		return;

	if(fname.Find(_T(".sl")) != -1)
		flag = TRUE;
	else if(fname.Find(_T(".txt")) != -1)
		flag = TRUE;
	else if(fname.Find(_T(".mdl")) != -1)
		flag = TRUE;
	else
		return;

	COleDataSource *pData = new COleDataSource();
	HGLOBAL hData;
	char* ptr,buf[16];
	DWORD size;

	// Set Data
	CFile rdfile;
	CFileException fileException;

	if(!rdfile.Open((LPCTSTR)fname, CFile::modeRead, &fileException))
		return;
	size = (DWORD)rdfile.GetLength();
	hData = ::GlobalAlloc(GMEM_SHARE, size+1);
	ptr = (char *)::GlobalLock(hData);

	while(rdfile.Read(buf, 1)) {
		*ptr++ = buf[0];
	}
	rdfile.Close();
	//*ptr++ = '\r';
	*ptr++ = '\n';	//̒ǉōŌ̍ss

	::GlobalUnlock(hData);
	pData->CacheGlobalData(CF_TEXT, hData);
	pData->SetClipboard();
	pData->FlushClipboard();
	
}

void CDriveCtrl::OnUpdateEditCopy(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}

//
//
//Rs[̃Rs[͑ΏۊO
//
BOOL CDriveCtrl::OnEditDuplicate(HTREEITEM targetItem) 
{
	SHFILEOPSTRUCT stFile;
	char dst[_MAX_PATH], drv[_MAX_PATH],dir[_MAX_PATH],fname[128],ext[16];
	char src[_MAX_PATH];
	CString strPath;
	PATHINFO* pItem;
	HTREEITEM hItem;
	char msg[_MAX_PATH];

	/* split drive and directory, and pick filename */
	_splitpath(g_charScriptText, drv, dir, fname, ext);
	dir[strlen(dir)-1] = '\0';
	if(strstr(fname,"Rs[")) {
		AfxMessageBox("Not copy.",MB_OK,NULL);
		return FALSE;
	}
	if(targetItem) {
		// get directory for copy
		pItem = (PATHINFO*)GetItemData(targetItem);
		strPath = _T(pItem->path);
		strPath += _T("\\");
		_splitpath((LPCTSTR)strPath, drv, dir, NULL, NULL);
		dir[strlen(dir)-1] = '\0';
	}
	/*It checks whether I may copy. */
	sprintf(dst,"%s%s\\%s%s",drv,dir,fname,ext);
	if(Access(dst,SL_FATTR_FOK) == 0) {
		/*The file of the same name exists.*/
		sprintf(msg,"The file of the same name exists.\n \\
Do you wish to copy %s%s to %s%s ?",fname, ext, drv, dir);
		if(MessageBox(msg, PACKAGE_STRING, MB_YESNO | MB_ICONQUESTION) == IDNO) {
			//SetItemState( targetItem, 0x0, TVIS_DROPHILITED );
			//SetItemState( targetItem, 0x0, TVIS_SELECTED );
			return FALSE;
		}
		/*TODO*/
		sprintf(dst,"%s%s\\Rs[`%s%s",drv,dir,fname,ext);
	}
	else {
		/*It checks whether I may copy. */
		sprintf(msg,"Do you wish to copy %s%s to %s%s ?",fname, ext, drv, dir);
		if(MessageBox(msg, PACKAGE_STRING, MB_YESNO | MB_ICONQUESTION) == IDNO) {
			//SetItemState( targetItem, 0x0, 0x0/*TVIS_DROPHILITED*/ );
			//SetItemState( targetItem, 0x0, TVIS_SELECTED );
			//TreeView_SelectItem(this->GetSafeHwnd(), targetItem);
			//Invalidate(FALSE);
			return FALSE;
		}
	}


	dst[strlen(dst)+1] = '\0';
	stFile.hwnd = GetSafeHwnd();
	stFile.wFunc = FO_COPY;
	strcpy(src, g_charScriptText);
	src[strlen(src)+1] = '\0';
	stFile.pFrom = src;
	stFile.pTo = dst;							//Ȃƃ_
	stFile.fFlags = FOF_ALLOWUNDO | FOF_NOCONFIRMATION | FOF_SILENT;
	if(::SHFileOperation( &stFile ) == 0) {
		//succesful
		if(targetItem) {
			pItem = (PATHINFO*)GetItemData(targetItem);
			strPath = _T(pItem->path);
			DeleteAllChildren (targetItem);
			AddDirectories (targetItem, strPath);
			//update
			Select (targetItem, TVGN_CARET);
			Expand (targetItem, TVE_EXPAND);
			//normal
			SetItemState( targetItem, 0x0, TVIS_DROPHILITED );
		}
		else {
			hItem = GetSelectedItem ();
			hItem = GetParentItem(hItem);
			pItem = (PATHINFO*)GetItemData(hItem);
			strPath = _T(pItem->path);
			DeleteAllChildren (hItem);
			AddDirectories (hItem, strPath);
			//update
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);
		}
		return TRUE;
	}
	return FALSE;
}

// Local directoryFile typeɏ]ĕҏW
BOOL CDriveCtrl::AddLocalFiles(HTREEITEM hItem, LPCTSTR pszPath)
{
    HANDLE hFind;
    WIN32_FIND_DATA fd;
    BOOL bResult = FALSE;
	CString pszName,strNewPath;
	int nImage;

    CString strPath = pszPath;
    if (strPath.Right (1) != _T ("\\"))
        strPath += _T ("\\");
    strPath += _T ("*.*");

    if ((hFind = ::FindFirstFile (strPath, &fd)) == INVALID_HANDLE_VALUE)
        return bResult;

    do {
		// path
		strNewPath = pszPath;
		strNewPath += _T("\\");
		strNewPath += (LPCTSTR) &fd.cFileName;

        if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
			// 
        }
		else {	// append 2003.5.27
            pszName = (LPCTSTR) &fd.cFileName;
			if(m_FileType == FILE_SL) {
				if(pszName.Find(_T(".sl")) != -1) {
					AddItem (_T (fd.cFileName), ILI_SCRIPT,
						ILI_SCRIPT, hItem, (LPCTSTR)strNewPath);
				}
			}
			else if(m_FileType == FILE_MDL) {
				if(pszName.Find(_T(".mdl")) != -1)
					AddItem (_T (fd.cFileName), ILI_MODEL,
						ILI_MODEL, hItem, (LPCTSTR)strNewPath);
			}
			else if(m_FileType == FILE_TXT) {
				if(pszName.Find(_T(".txt")) != -1)
					AddItem (_T (fd.cFileName), ILI_TEXT,
						ILI_TEXT, hItem, (LPCTSTR)strNewPath);
			}
			else {
				if(pszName.Find(_T(".sl")) != -1)
					nImage = ILI_SCRIPT;
				else if(pszName.Find(_T(".txt")) != -1)
					nImage = ILI_TEXT;
				else if(pszName.Find(_T(".mdl")) != -1)
					nImage = ILI_MODEL;
				else
					nImage = ILI_FILE;
				AddItem (_T (fd.cFileName), nImage,
					nImage, hItem, (LPCTSTR)strNewPath);
			}
		}
    } while (::FindNextFile (hFind, &fd));

    ::FindClose (hFind);
    return bResult;
}

void CDriveCtrl::OnFileChdir() 
{
	// TODO: ̈ʒuɃR}h nhp̃R[hǉĂ
	char path[_MAX_PATH],shortPath[_MAX_PATH];
	CString str;
	char ch;
	DWORD size;
	int i,uch,offset;

	//hItem = GetSelectedItem ();
	//pItem = (PATHINFO*)GetItemData(hItem);
	//_splitpath(pItem->path, drv, dir, NULL, NULL);
	//dir[strlen(dir)-1] = '\0';

	/*'\' is changed into '/'. */
	//sprintf(path,"%s%s",drv,dir);
	strcpy(path, g_charScriptText);
	GetShortPathName(path, shortPath, _MAX_PATH);
	str = _T(shortPath);
	strcpy(path, "chdir ");
	offset = (int)strlen(path);
	for(i=0; i < str.GetLength(); i++) {
		ch = str.GetAt(i);
		if(ch == '\\')
			ch = '/';
		path[offset + i] = ch;
	}
	path[offset + i] = '\n';
	path[offset + i + 1] = '\0';
	/*The chdir command is executed. */
	for(i=0; i<(int)strlen(path); i++){
		uch = (unsigned char)path[i];
		WriteFile(g_pConsole->m_hStdinW, &uch, sizeof(int), &size, NULL);
	}
	
}

void CDriveCtrl::OnUpdateFileChdir(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}

// ACeT[`
HTREEITEM CDriveCtrl::GetStateSearch(UINT targetState)
{
	HTREEITEM hItem;
	CString name, strPath;
	UINT state;

	// ACeXV
	hItem = GetFirstVisibleItem();
	while(hItem) {
		name = GetItemText(hItem);
		state = GetItemState(hItem, TVIF_STATE);
		if((state & targetState)) {
			return(hItem);
		}
		hItem = GetNextVisibleItem(hItem);
	}
	return NULL;
}

void CDriveCtrl::OnLButtonUp(UINT nFlags, CPoint point) 
{
	// TODO: ̈ʒuɃbZ[W nhp̃R[hǉ邩܂̓ftHg̏ĂяoĂ
	HTREEITEM hItem;
	CString name;

	if(m_pDragImage) {
		/*Release the mouse.*/
		::ReleaseCapture();
		/*Show the cursor.*/
		ShowCursor(TRUE);

		//̎̃JgACéA
		hItem = GetStateSearch(TVIS_DROPHILITED);
		if(!hItem) {
			//fBNgɃRs[
			OnEditDuplicate(hItem);
		}
		else {
			name = GetItemText(hItem);
			fprintf(stdout,"Selected %s\n", (LPCTSTR)name);
			/*move or copy*/
			if(!OnEditDuplicate(hItem)) {
			}
		}
		// Reset the drop target to NULL.
		TreeView_SelectDropTarget(this->GetSafeHwnd(), (HTREEITEM)NULL);

		/*Inform the image list that dragging has stopped.*/
		m_pDragImage->EndDrag();
		delete m_pDragImage;
		m_pDragImage = NULL;
		Invalidate(FALSE);
	}
	
	CTreeCtrl::OnLButtonUp(nFlags, point);
}

BOOL CDriveCtrl::IsScriptFile() 
{
	//IĂt@CslȂL	
	HTREEITEM hItem = GetSelectedItem ();
	if(hItem) {
		CString strPath = GetPathFromItem (hItem);
		if(strPath.Find(".sl") == -1)
			return FALSE;
		else
			return TRUE;
	}
	else
		return FALSE;
}

void CDriveCtrl::OnScriptStep() 
{
	HTREEITEM hItem = GetSelectedItem ();
	//CString strPath = GetPathFromItem (hItem);
	PATHINFO* pItem;
	pItem = (PATHINFO*)GetItemData(hItem);
	CString strPath = _T(pItem->path);
	// ChildFrameŎs
	strcpy(g_charScriptText, strPath);
	g_pConsole->ScriptRun(CONSOLE_MODE);	
}

BOOL CDriveCtrl::SetScriptFile() 
{
	HTREEITEM hItem = GetSelectedItem ();
	if(!hItem)
		return FALSE;
	//CString strPath = GetPathFromItem (hItem);
	PATHINFO* pItem;
	pItem = (PATHINFO*)GetItemData(hItem);
	CString strPath = _T(pItem->path);
	if(Access((LPCTSTR)pItem->path,SL_FATTR_FOK) != 0)
		return FALSE;
	// ChildFrameŎs
	strcpy(g_charScriptText, strPath);
	return TRUE;
}

void CDriveCtrl::OnFileAll() 
{
	// \t@C`
	m_FileType = FILE_ALL;

	HTREEITEM hItem;
	CString name, strPath;
	UINT state;
	PATHINFO* pItem;

	// ACeXV
	hItem = GetFirstVisibleItem();
	while(hItem) {
		name = GetItemText(hItem);
		state = GetItemState(hItem, TVIF_STATE);
		if((state & TVIS_EXPANDED)) {
			// test
			fprintf(stdout,"%s\n",(LPCTSTR)name);
			// hItemXV
			pItem = (PATHINFO*)GetItemData(hItem);
			strPath = _T(pItem->path);
			DeleteLocalFiles (hItem);
			AddLocalFiles (hItem, strPath);
			//update
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);
		}
		hItem = GetNextVisibleItem(hItem);
	}
	
}

void CDriveCtrl::OnUpdateFileAll(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}

void CDriveCtrl::OnFileSl() 
{
	// \t@C`
	m_FileType = FILE_SL;

	HTREEITEM hItem;
	CString name, strPath;
	UINT state;
	PATHINFO* pItem;

	// ACeXV
	hItem = GetFirstVisibleItem();
	while(hItem) {
		name = GetItemText(hItem);
		state = GetItemState(hItem, TVIF_STATE);
		if((state & TVIS_EXPANDED)) {
			// test
			fprintf(stdout,"%s\n",(LPCTSTR)name);
			// hItemXV
			pItem = (PATHINFO*)GetItemData(hItem);
			strPath = _T(pItem->path);
			DeleteLocalFiles (hItem);
			AddLocalFiles (hItem, strPath);
			//update
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);
		}
		hItem = GetNextVisibleItem(hItem);
	}
	
}

void CDriveCtrl::OnUpdateFileSl(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}

void CDriveCtrl::OnFileMdl() 
{
	// \t@C`
	m_FileType = FILE_MDL;

	HTREEITEM hItem;
	CString name, strPath;
	UINT state;
	PATHINFO* pItem;

	// ACeXV
	hItem = GetFirstVisibleItem();
	while(hItem) {
		name = GetItemText(hItem);
		state = GetItemState(hItem, TVIF_STATE);
		if((state & TVIS_EXPANDED)) {
			// test
			fprintf(stdout,"%s\n",(LPCTSTR)name);
			// hItemXV
			pItem = (PATHINFO*)GetItemData(hItem);
			strPath = _T(pItem->path);
			DeleteLocalFiles (hItem);
			AddLocalFiles (hItem, strPath);
			//update
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);
		}
		hItem = GetNextVisibleItem(hItem);
	}	
	
}

void CDriveCtrl::OnUpdateFileMdl(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}

void CDriveCtrl::OnFileTxt() 
{
	// \Ăt@C`
	m_FileType = FILE_TXT;

	HTREEITEM hItem;
	CString name, strPath;
	UINT state;
	PATHINFO* pItem;

	// ACeXV
	hItem = GetFirstVisibleItem();
	while(hItem) {
		name = GetItemText(hItem);
		state = GetItemState(hItem, TVIF_STATE);
		if((state & TVIS_EXPANDED)) {
			// test
			fprintf(stdout,"%s\n",(LPCTSTR)name);
			// hItemXV
			pItem = (PATHINFO*)GetItemData(hItem);
			strPath = _T(pItem->path);
			DeleteLocalFiles (hItem);
			AddLocalFiles (hItem, strPath);
			//update
			Select (hItem, TVGN_CARET);
			Expand (hItem, TVE_EXPAND);
		}
		hItem = GetNextVisibleItem(hItem);
	}
	
}

void CDriveCtrl::OnUpdateFileTxt(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	
}
