/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: ConsoleDoc.cpp,v 1.3 2004/04/21 10:49:30 orrisroot Exp $ */

#include "StdAfx.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "WinShell.h"

#include "ConsoleDoc.h"
#include "ConsoleView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CConsoleDoc

IMPLEMENT_DYNCREATE(CConsoleDoc, CDocument)

BEGIN_MESSAGE_MAP(CConsoleDoc, CDocument)
  //{{AFX_MSG_MAP(CConsoleDoc)
  //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CConsoleDoc NX̍\z/

CConsoleDoc::CConsoleDoc()
{
#ifdef _VC6
  m_RealSize = CSize(0,0);
  m_LogicalSize = CSize(0,0);
  m_TextBuffer   = NULL;
  m_TextAttrib   = NULL;
  m_sCopyArea = CPoint(0,0);
  m_eCopyArea = CPoint(0,0);
#else
  m_RealSize.SetSize(0,0);
  m_LogicalSize.SetSize(0,0);
  m_TextBuffer   = NULL;
  m_TextAttrib   = NULL;
  m_sCopyArea.SetPoint(0,0);
  m_eCopyArea.SetPoint(0,0);
#endif
}

CConsoleDoc::~CConsoleDoc()
{
  FreeTextBuffer();
}

BOOL CConsoleDoc::OnNewDocument()
{
  if (!CDocument::OnNewDocument())
    return FALSE;

  // TODO: ̈ʒuɍďǉĂB
  // (SDI hLg͂̃hLgėp܂B)

  return TRUE;
}

BOOL CConsoleDoc::AllocateTextBuffer(int x, int y)
{
  int i,j;
  /* set current status */
#ifdef _VC6
  m_RealSize = CSize(x,y);
  m_LogicalSize = CSize(x,y);
#else
  m_RealSize.SetSize(x,y);
  m_LogicalSize.SetSize(x,y);
#endif

  /* memory allocation */
  m_TextBuffer = (char**)malloc(sizeof(char*)*y);
  if(m_TextBuffer == NULL) goto fatal;
  memset(m_TextBuffer,0,sizeof(char*)*y);

  m_TextAttrib = (int**)malloc(sizeof(int*)*y);
  if(m_TextAttrib == NULL) goto fatal;
  memset(m_TextAttrib,0,sizeof(int*)*y);

  for(i=0; i<y; i++){
    m_TextBuffer[i] = (char*)malloc(sizeof(char)*x);
    if(m_TextBuffer[i] == NULL) goto fatal;

    m_TextAttrib[i] = (int*)malloc(sizeof(int)*x);
    if(m_TextAttrib[i] == NULL) goto fatal;

    for(j=0; j<x; j++){
      m_TextBuffer[i][j]=' ';
      m_TextAttrib[i][j]=TEXT_ATTRIBUTE_NORMAL;
    }
  }
  return TRUE;

fatal:
  FreeTextBuffer();
  return FALSE;
}

void CConsoleDoc::FreeTextBuffer()
{
  int i;
  if(m_TextBuffer){
    for(i=0; i<m_RealSize.cy; i++)
      if(m_TextBuffer[i]) free(m_TextBuffer[i]);
    free(m_TextBuffer);
  }
  if(m_TextAttrib){
    for(i=0; i<m_RealSize.cy; i++)
      if(m_TextAttrib[i]) free(m_TextAttrib[i]);
    free(m_TextAttrib);
  }
  m_TextAttrib   = NULL;
  m_TextBuffer   = NULL;
#ifdef _VC6
  m_RealSize = CSize(0,0);
  m_LogicalSize = CSize(0,0);
  m_sCopyArea = CPoint(0,0);
  m_eCopyArea = CPoint(0,0);
#else
  m_RealSize.SetSize(0,0);
  m_LogicalSize.SetSize(0,0);
  m_sCopyArea.SetPoint(0,0);
  m_eCopyArea.SetPoint(0,0);
#endif
}

BOOL CConsoleDoc::InitializeTextBuffer(int xinit, int maxline)
{
  /* check arguments */
  if(xinit <= 0 || maxline <= 0) return FALSE;
  return AllocateTextBuffer(xinit,maxline);
}


BOOL CConsoleDoc::ResizeTextBuffer(int x, int y)
{
  int i,j,n;
  CSize pre;
 
  if(y > m_RealSize.cy) return FALSE;

#ifdef _VC6
  pre = CSize(m_LogicalSize.cx, m_LogicalSize.cy);
  m_LogicalSize = CSize(x,y);
#else
  pre.SetSize(m_LogicalSize.cx, m_LogicalSize.cy);
  m_LogicalSize.SetSize(x,y);
#endif
  /* resize buffer length */
  if(x > m_RealSize.cx){
    char *buff;
    int  *attr;
    for(i=0; i<m_RealSize.cy; i++){
      buff = (char*)realloc(m_TextBuffer[i], sizeof(char)*x);
      if(buff == NULL) return FALSE;
      m_TextBuffer[i] = buff;
      attr = (int*)realloc(m_TextAttrib[i], sizeof(int)*x);
      if(attr == NULL) return FALSE;
      m_TextAttrib[i] = attr;
      for(j=m_RealSize.cx; j<x; j++){
        m_TextBuffer[i][j] = ' ';
        m_TextAttrib[i][j] = TEXT_ATTRIBUTE_NORMAL;
      }
    }
    m_RealSize.cx = x;
  }

  /* clear resized memory area */
  n = max(pre.cy, m_LogicalSize.cy);
  for(i=m_LogicalSize.cx; i<pre.cx; i++){
    for(j=0; j<n; j++){
      m_TextBuffer[j][i] = ' ';
      m_TextAttrib[j][i] = TEXT_ATTRIBUTE_NORMAL;
    }
  }
  n = max(pre.cx, m_LogicalSize.cx);
  for(i=m_LogicalSize.cy; i<pre.cy; i++){
    for(j=0; j<n; j++){
      m_TextBuffer[i][j] = ' ';
      m_TextAttrib[i][j] = TEXT_ATTRIBUTE_NORMAL;
    }
  }

  return TRUE;
}

BOOL CConsoleDoc::ScrollTextBuffer(bool dir)
{
  int i;
  char *buff;
  int  *attr;
  if(m_RealSize.cy == 0) return FALSE;
  if(m_RealSize.cy == 1) return TRUE;
  if(dir){
    /* backup first buffer */
    buff = m_TextBuffer[0];
    attr = m_TextAttrib[0];
    /* clear oldest line buffer */
    for(i=0; i<m_RealSize.cx; i++){
      buff[i] = ' ';
      attr[i] = TEXT_ATTRIBUTE_NORMAL;
    }
    for(i=1; i<m_RealSize.cy; i++){
      m_TextBuffer[i-1] = m_TextBuffer[i];
      m_TextAttrib[i-1] = m_TextAttrib[i];
    }
    m_TextBuffer[m_RealSize.cy-1] = buff;
    m_TextAttrib[m_RealSize.cy-1] = attr;
  }else{
    buff = m_TextBuffer[m_RealSize.cy-1];
    attr = m_TextAttrib[m_RealSize.cy-1];
    /* clear oldest line buffer */
    for(i=0; i<m_RealSize.cx; i++){
      buff[i] = ' ';
      attr[i] = TEXT_ATTRIBUTE_NORMAL;
    }
    for(i=m_RealSize.cy-1; i>0; i--){
      m_TextBuffer[i] = m_TextBuffer[i-1];
      m_TextAttrib[i] = m_TextAttrib[i-1];
    }
    m_TextBuffer[0] = buff;
    m_TextAttrib[0] = attr;
  }
  return TRUE;
}

BOOL CConsoleDoc::SaveTextBuffer(int xpos, int ypos, int attr, const char *str, size_t len)
{
  size_t i,tail;
  tail = xpos + len;
  if(tail > (size_t)m_LogicalSize.cx || ypos > m_LogicalSize.cy) return FALSE;
  for(i=xpos; i<tail; i++){
    m_TextBuffer[ypos][i] = str[i-xpos];
    m_TextAttrib[ypos][i] = attr;
  }
  return TRUE;
}

void CConsoleDoc::ClearSelectedBuffer()
{
  int i,j;
  for(i=0; i<m_RealSize.cy; i++){
    for(j=0; j<m_RealSize.cx; j++)
      m_TextAttrib[i][j] &= ~TEXT_ATTRIBUTE_SELECT;
  }
  m_sCopyArea.x = 0;
  m_sCopyArea.y = 0;
  m_eCopyArea.x = 0;
  m_eCopyArea.y = 0;
}


BOOL CConsoleDoc::SelectTextBuffer(int xs, int ys, int xe, int ye)
{
  int i,j,tmp;
  if(xs < 0 ) xs = 0;
  if(xe < 0 ) xe = 0;
  if(ys < 0 ) ys = 0;
  if(ye < 0 ) ye = 0;

  /* swap y pos and x pos */
  if(ys > ye){
    tmp = ye; ye = ys; ys = tmp;
    tmp = xe; xe = xs; xs = tmp;
  }else if(ys == ye && xe > xe){
    tmp = xe; xe = xs; xs = tmp;
  }

  /* check buffer size */
  if(ye >= m_LogicalSize.cy) ye = m_LogicalSize.cy - 1;
  if(xe >  m_LogicalSize.cx) xe = m_LogicalSize.cx;

  if(ys == ye && xs == xe) return TRUE;

  m_sCopyArea.x = xs;
  m_sCopyArea.y = ys;
  m_eCopyArea.x = xe;
  m_eCopyArea.y = ye;

  /* first line */
  if(ys == ye) tmp = xe;
  else         tmp = m_LogicalSize.cx;
  for(i=xs; i<tmp; i++){
    m_TextAttrib[ys][i] |= TEXT_ATTRIBUTE_SELECT;
  }
  /* middle line */
  for(i=ys+1; i<ye; i++){
    for(j=0; j<m_LogicalSize.cx; j++){
      m_TextAttrib[i][j] |= TEXT_ATTRIBUTE_SELECT;
    }
  }
  /* last line */
  if(ys != ye){
    for(i=0; i<xe; i++){
      m_TextAttrib[ye][i] |= TEXT_ATTRIBUTE_SELECT;
    }
  }
  return TRUE;
}

const char *CConsoleDoc::GetTextBuffer(int xp, int yp)
{
  if(xp >= m_LogicalSize.cx || yp >= m_LogicalSize.cy)
    return NULL;
  return &m_TextBuffer[yp][xp];
}

int CConsoleDoc::GetTextAttribute(int xp, int yp)
{
  if(xp  >= m_LogicalSize.cx || yp >= m_LogicalSize.cy)
    return -1;
  return m_TextAttrib[yp][xp];
}

BOOL CConsoleDoc::SetTextBufferToClipboard()
{
  int i,j,tmp;
  SIZE_T len;
  COleDataSource *pData;
  HGLOBAL hData;
  char *ptr;

  /* check has selected area */
  if(m_sCopyArea == m_eCopyArea) return FALSE;

  /* count up coping data */
  len = 0;
  /* first line */
  if(m_sCopyArea.y == m_eCopyArea.y) tmp = m_eCopyArea.x;
  else                               tmp = m_RealSize.cx;
  for(i=m_sCopyArea.x; i<tmp; i++){
    len ++;
    if(m_TextBuffer[m_sCopyArea.y][i] == '\n'){
      break;
    }
  }
  /* middle line */
  for(i=m_sCopyArea.y+1; i<m_eCopyArea.y; i++){
    for(j=0; j<m_LogicalSize.cx; j++){
      len ++;
      if(m_TextBuffer[i][j] == '\n'){
        break;
      }
    }
  }
  /* last line */
  if(m_sCopyArea.y != m_eCopyArea.y){
    for(i=0; i<m_eCopyArea.x; i++){
      len ++;
      if(m_TextBuffer[m_eCopyArea.y][i] == '\n'){
        break;
      }
    }
  }
  if( len == 0 ) return FALSE;
  len ++; /* add terminater */

  pData = new COleDataSource;
  hData = ::GlobalAlloc(GMEM_SHARE, len);
  ptr = (char*)::GlobalLock(hData);
  len = 0;
  for(i=m_sCopyArea.x; i<tmp; i++){
    ptr[len] = m_TextBuffer[m_sCopyArea.y][i];
    len ++;
    if(m_TextBuffer[m_sCopyArea.y][i] == '\n') break;
  }
  /* middle line */
  for(i=m_sCopyArea.y+1; i<m_eCopyArea.y; i++){
    for(j=0; j<m_LogicalSize.cx; j++){
      ptr[len] = m_TextBuffer[i][j];
      len ++;
      if(m_TextBuffer[i][j] == '\n') break;
    }
  }
  /* last line */
  if(m_sCopyArea.y != m_eCopyArea.y){
    for(i=0; i<m_eCopyArea.x; i++){
      ptr[len] = m_TextBuffer[m_eCopyArea.y][i];
      len ++;
      if(m_TextBuffer[m_eCopyArea.y][i] == '\n') break;
    }
  }
  ptr[len] = '\0';
  // Get the currently selected data
  ::GlobalUnlock(hData);

  pData->CacheGlobalData( CF_TEXT, hData );
  pData->SetClipboard();
  pData->FlushClipboard();
  return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CConsoleDoc VAC[[V

void CConsoleDoc::Serialize(CArchive& ar)
{
  if (ar.IsStoring()){
    // TODO: ̈ʒuɕۑp̃R[hǉĂB
  }else{
    // TODO: ̈ʒuɓǂݍݗp̃R[hǉĂB
  }
}

/////////////////////////////////////////////////////////////////////////////
// CConsoleDoc NX̐ff

#ifdef _DEBUG
void CConsoleDoc::AssertValid() const
{
  CDocument::AssertValid();
}

void CConsoleDoc::Dump(CDumpContext& dc) const
{
  CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CConsoleDoc R}h

void CConsoleDoc::DeleteContents()
{
  // TODO : ɓȃR[hǉ邩A͊{NXĂяoĂB

  CDocument::DeleteContents();
}
