/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: win32_console.cpp,v 1.5 2004/04/21 10:49:30 orrisroot Exp $ */

#include "StdAfx.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <process.h>
#include <conio.h>

#include "WinShell.h"
#include "ConsoleDoc.h"
#include "ConsoleView.h"
#include "shell/satellite4.h"

#include "win32_console.h"

#define TTY_PIPE_BUFSIZE 16384

using namespace std;


typedef struct _pipe_args_t {
  HANDLE      hstd;
  CConsoleView *view;
} pipe_args_t;

DWORD __cdecl write_stdout_pipe(void *argv){
  HANDLE         hStdout;
  CConsoleView  *cView;
  pipe_args_t   *pa;
  char           buf[TTY_PIPE_BUFSIZE];
  DWORD          len;

  pa = (pipe_args_t*)argv;
  hStdout = pa->hstd;
  cView   = pa->view;

  while(ReadFile(hStdout,buf,TTY_PIPE_BUFSIZE,&len,NULL)){
    cView->SendTextPrint(buf,(size_t)len);
  }
  return 0;
}

win32_console::win32_console() : oflag(true) {
  HANDLE hCurProcess;
  HANDLE hTemp;
  SECURITY_ATTRIBUTES secAtt;

  /* setting of Security Attributes */
  secAtt.nLength = sizeof(SECURITY_ATTRIBUTES);
  secAtt.lpSecurityDescriptor = NULL;
  secAtt.bInheritHandle = TRUE;

  hCurProcess=GetCurrentProcess();

  /* TTY input */
  CreatePipe(&hStdinR,&hTemp,&secAtt,0);
  DuplicateHandle(hCurProcess, hTemp, hCurProcess, &hStdinW,
                  0, FALSE, DUPLICATE_SAME_ACCESS);
  CloseHandle(hTemp);
  ifd.handle = hStdinR;

  /* TTY output */
  CreatePipe(&hTemp,&hStdoutW,&secAtt,0);
  DuplicateHandle(hCurProcess, hTemp, hCurProcess, &hStdoutR,
                  0, FALSE, DUPLICATE_SAME_ACCESS);
  CloseHandle(hTemp);
  ofd.handle = hStdoutW;

  /* TTY error */
  DuplicateHandle(hCurProcess, hStdoutW, hCurProcess, &hStderrW,
                  0, TRUE, DUPLICATE_SAME_ACCESS);
  efd.handle = hStderrW;

  ConsoleView = NULL;
  htTTYout    = INVALID_HANDLE_VALUE;

}

win32_console::~win32_console(){
  CloseHandle(hStdoutW);
  CloseHandle(hStderrW);
  WaitForSingleObject(htTTYout,INFINITE);
  CloseHandle(htTTYout);
}

/* set console view */
void win32_console::SetConsoleView(CConsoleView *pConsole){
  pipe_args_t arg;

  ConsoleView = pConsole;

  /* regist standard in writing handle */
  ConsoleView->RegistTTYinput(hStdinW);

  /* create standard out poling thread */
  arg.hstd = hStdoutR;
  arg.view = ConsoleView;
  htTTYout = CreateThread(NULL,0,(LPTHREAD_START_ROUTINE)write_stdout_pipe,
                          &arg,0,&dwtStdoutID);

  term_normal_mode();
  term_set_fgcolor(SL_TTY::BLACK);
  term_set_bgcolor(SL_TTY::WHITE);
  term_clear_screen();
}


void win32_console::term_edit_mode(){
  ConsoleView->SendTextEditMode(1);
}

void win32_console::term_normal_mode(){
  ConsoleView->SendTextEditMode(0);
}

void win32_console::term_set_fgcolor(SL_TTY::COLOR color){
  bool do_set(false);
  unsigned int col(0x000000);
  switch(color){
  case SL_TTY::BLACK:   col = 0x000000; do_set = true; break;
  case SL_TTY::RED:     col = 0xff0000; do_set = true; break;
  case SL_TTY::GREEN:   col = 0x00ff00; do_set = true; break;
  case SL_TTY::YELLOW:  col = 0xffff00; do_set = true; break;
  case SL_TTY::BLUE:    col = 0x0000ff; do_set = true; break;
  case SL_TTY::MAGENTA: col = 0xff00ff; do_set = true; break;
  case SL_TTY::CYAN:    col = 0x00ffff; do_set = true; break;
  case SL_TTY::WHITE:   col = 0xffffff; do_set = true; break;
  }
  if(do_set){
    ConsoleView->SendTextColor(0,col);
  }
}

void win32_console::term_set_bgcolor(SL_TTY::COLOR color){
  bool do_set(false);
  unsigned int col(0x000000);
  switch(color){
  case SL_TTY::BLACK:   col = 0x000000; do_set = true; break;
  case SL_TTY::RED:     col = 0xff0000; do_set = true; break;
  case SL_TTY::GREEN:   col = 0x00ff00; do_set = true; break;
  case SL_TTY::YELLOW:  col = 0xffff00; do_set = true; break;
  case SL_TTY::BLUE:    col = 0x0000ff; do_set = true; break;
  case SL_TTY::MAGENTA: col = 0xff00ff; do_set = true; break;
  case SL_TTY::CYAN:    col = 0x00ffff; do_set = true; break;
  case SL_TTY::WHITE:   col = 0xffffff; do_set = true; break;
  }
  if(do_set){
    ConsoleView->SendTextColor(1,col);
  }
}


void win32_console::term_move_down(int y){
  ConsoleView->SendCursorDown(y);
}


void win32_console::term_move_up(int y){
  ConsoleView->SendCursorUp(y);
}

void win32_console::term_move_right(int x){
  ConsoleView->SendCursorRight(x);
}

void win32_console::term_move_left(int x){
  ConsoleView->SendCursorLeft(x);
}

void win32_console::term_move_bol(){
  ConsoleView->SendCursorBOL();
}

void win32_console::term_move_newline(){
  ConsoleView->SendCursorNewLine();
}

void win32_console::term_clear_screen(){
  int mx,my;
  COORD coord;
  coord.X=0; coord.Y=0;

  term_getmaxyx(&my,&mx);

  //FillConsoleOutputCharacter(hStdout, ' ', mx*my, coord, &noaw);

  //term_move(0, 0);
}

void win32_console::term_clear_eol(int x){
  int dis,i,mx,my;
  term_getmaxyx(&my,&mx);
  dis=mx-x;
  for(i=0;i<dis;i++) term_putc(' ');
  term_move_left(dis/*dis-1*/);
}

void win32_console::term_flush(sl4_fd_t fd){
  if(fd.handle != INVALID_HANDLE_VALUE)
    FlushFileBuffers(fd.handle);
}

void win32_console::term_getmaxyx(int *y, int *x){
  ConsoleView->SendCursorGetMax(x,y);
}

void win32_console::term_set_attr_normal(){
  ConsoleView->SendTextAttribute(TEXT_ATTRIBUTE_NORMAL);
}

void win32_console::term_set_attr_reverse(){
  ConsoleView->SendTextAttribute(TEXT_ATTRIBUTE_REVERSE);
}

int win32_console::term_getc(){
  int keycode;
  DWORD size;
  while(1){
    ReadFile(hStdinR, &keycode, sizeof(int), &size, 0);
    if(size == 0) return EOF;
    if(keycode <= 0xff) break;
  }
  return keycode;
}

char *win32_console::term_gets(char *buf, int c){
  int keycode;
  int i;
  DWORD size;
  for(i=0;i<c;i++){
    ReadFile( hStdinR, &keycode, sizeof(int), &size, 0);
    if(size == 0) return NULL;
    if(keycode > 0xff){ i--; continue; } /* retry */
    buf[i] = (char)(keycode & 0xff);
    if(keycode == 0x0d) break;
  }
  buf[i]='\0';
  return buf;
}

void win32_console::term_putc(int c){
  static bool is_kanji=false;
  static char kanji[2];
  char buf;
  buf = c & 0xff;
  if(is_kanji){
    kanji[1] = buf;
    ConsoleView->SendTextPrint(kanji,2);
    is_kanji=false;
  }else{
    if(buf<0){
      kanji[0]=buf;
      is_kanji=true;
    }else{
      ConsoleView->SendTextPrint(&buf,1);
    }
  }
}

void win32_console::term_print(const char *str){
  size_t len;
  len = strlen(str);
  ConsoleView->SendTextPrint(str, len);
}

int win32_console::term_keypad_getc(){
  int keycode;
  DWORD size;

  ReadFile(hStdinR, &keycode, sizeof(int), &size, 0);
  if(size == 0) return EOF;
  return keycode;
}

