/*
**  bifst_a.c
**  bif-c
**
**  Created by Joel Rees on 2009/07/25.
**  Copyright 2009 __Reiisi_Kenkyuu__. All rights reserved.
**
** Translated to C from BIFST/A, as mechanically as possible.
**
*/


#include <stddef.h>
#include <stdlib.h>	/* for exit() */
#include <signal.h>	/* For BREAK key handling. */
#include <string.h> /* For debug prints. */

#include <stdio.h>	/* For handling stdio. Also for debugging purposes, since XCODE gets so confused. */


#include "bifst_a.h"
#include "bif_m.h"
#include "bif_vm.h"

#if defined DEBUGGING && DEBUGGING > 0 && DEBUGGING < 0x10
#include "bif1_a.h"	/* For SEMIS and DOT in debugging. */
#include "bif2_a.h" /* For constants, user constants, user variables, etc. in debugging. */
#include "bif3b_a.h" /* For TYPE */
#include "bif5_a.h" /* For DOT */
#endif /* defined DEBUGGING && DEBUGGING > 0 && DEBUGGING < 0x10 */
#include "bif_ex.h"

#include "bif7b_a.h"


/* Not sure why I wanted to make this static, but it can't be static if I want to FORGET the way I did it in BIF-6809. */
#if defined STATIC_MEMORYIMAGE
byte_t memoryImage[ ILIM ] = { 0 };	/* Initialize it to keep the optimizer from discarding it. */
#else /* !defined STATIC_MEMORYIMAGE */
byte_t * memoryImage = (byte_t *) 0;
#endif /* defined STATIC_MEMORYIMAGE */

/* For bifst_a.h:
00000010 * Start up routines for BIF
00020 * BIF Copyright 1989 Joel Matthew Rees (see BIF/ASM)
00100 *
00105 ILIM	EQU $8000 memory
00110 IBUFF	EQU ILIM-BCT*(BWID+4)
00120 IPAD	EQU IBUFF-TWID max
00130 IHASH	EQU IPAD-34
00140 IWPAD	EQU IHASH-NLMASK-2
00150 ITIB	EQU IWPAD-TWID
00160 IUSER	EQU ITIB-UEND
00170 IRP0	EQU IUSER-2
00180 ISP0	EQU IRP0-258
00190 * dictionary below here
00400 *
*/

/* This is not organized well, at all. I'm just going to use it as it is, for now.
** 
00410 ORIG	EQU * initials for task 0, offsets
00420 	NOP 0
00430 	JMP COLD 1
00440 	NOP 4
00450 	JMP WARM 5
00460 	FDB $6809 cpu 8
00470 	FDB 0 rev $0A
00480 	FDB FORGET-CFAOFF CURRENT for COLD $0C
00490 	FDB 8 backspace $0E
00500 	FDB IUSER task 0 user area $10
00510 	FDB ISP0 $12
00520 	FDB IRP0 $14
00530 	FDB ITIB $16
00540 	FDB NLMASK not used $18
00550 	FDB 1 WARNING $1A
00560 	FDB FOLLOW-1 FENCE for COLD $1C
00570 	FDB FOLLOW DP for COLD $1E
00580 	FDB BIF+2 defs root $20
00590 	FDB IPAD $22
00600 	FDB IWPAD $24
00610 	FDB 32 terminal columns $26
00620 	FDB IBUFF $28
00630 	FDB ILIM $2A
01000 *
*/
/* Yes, ORIG is supposed to be a ROMMABLE table of constants. Not yet. */
boot_constants_s	ORIG;

/* Can't use the virtual machine stacks yet, use C parameter passing. */
static natural_t sizeAdjust( natural_t dfltSz, natural_t memSz ) 
{	/* Adjust the default sizes by actual memory size, if there is enough memory.
	// Ideally: calcSz = ( dfltSz / ILIM ) * memSz
	// Use integer math, assume dfltSz is significantly less than ILIM: 
	*/
	natural_t calcSz = memSz / ( ILIM / dfltSz );
	/* Keep it aligned, expansively. */
	calcSz = ( ( calcSz + BYTESPERCELL - 1 ) / BYTESPERCELL ) * BYTESPERCELL;
	return ( calcSz > dfltSz ) ? calcSz : dfltSz;
}

static void patchORIG( void )
{
/* Here's a thought: use the specified memory size from the command line 
// to calculate the sizes to pass to malloc() as we malloc() each area? 
// Later, as we change the bounds checking and such.
*/
	natural_t memorySize = romORIG.initialLimitOfBufferRAM.integer;
	/* natural_t size, offset; */

	ORIG = romORIG;	/* Get the defaults where they are set. */

	/* Calculate actual values from sizes and offsets in romORIG. */

	ORIG.initialLimitOfBufferRAM.bytep = memoryImage + memorySize;

	/* All the buffers need special attention for adjusting size, skipping that for now.
	// The disk buffers need to be adjusted by number of buffers, not bytes. 
	// And we have to make sure the additional buffers are known to the buffer managers.
	// Right now, BCT, the buffer count, is hard-wired. 
	// Will fix this later, perhaps when I start dynamically allocating buffers, etc.
	*/
	ORIG.initialDiskBuffers.bytep 
		= ORIG.initialLimitOfBufferRAM.bytep - romORIG.initialDiskBuffers.integer;

	/* We need to make sure the other buffers can tell how much extra space they have, too.
	*/

	/* The word buffer is bounded by the numeric conversion buffer, we can just adjust it. 
	// (Need to track length to allocate separately.)  
	*/
	ORIG.initialNumericConversionScratchPad.bytep 
		= ORIG.initialDiskBuffers.bytep 
		  - sizeAdjust( romORIG.initialNumericConversionScratchPad.integer, memorySize );

	/* The word buffer is bounded by the numeric conversion buffer, we can just adjust it. 
	// (Need to track length to allocate separately.)  
	*/
	ORIG.initialWordBufferPointer.bytep 
		= ORIG.initialNumericConversionScratchPad.bytep 
		  - sizeAdjust( romORIG.initialWordBufferPointer.integer, memorySize );

	/* TIB is bounded by the word buffer, we can just adjust it. 
	// (Need to track length to allocate separately.)  
	*/
	ORIG.initialTerminalInputBuffer.bytep 
		= ORIG.initialWordBufferPointer.bytep 
		  - sizeAdjust( romORIG.initialTerminalInputBuffer.integer, memorySize );

	/* Need some way of switching tasks, don't we?
	// and of knowing how many tasks we can have, too.
	// All of these are dependent on the structure of the entire memory map.
	// The extra bytes for tracking limits was, I assume, deemed wasteful back then.
	// And, in fact, it was, back then.
	// Fix one thing at a time.
	*/
	ORIG.initialTaskBase.bytep 
		= ORIG.initialTerminalInputBuffer.bytep 
		  - romORIG.initialTaskBase.integer;

	ORIG.initialReturnStackBase.bytep 
		= ORIG.initialTaskBase.bytep 
		  - ORIG.initialReturnStackBase.integer;	/* This is just the gap. */

	/* Adjust the size of the return stack here, since I remember how the stacks check things.
	// Sloppy calculation includes gap, but that's okay, I think. 
	*/
	ORIG.initialParameterStackBase.bytep 
		= ORIG.initialReturnStackBase.bytep 
		  - sizeAdjust( ORIG.initialParameterStackBase.integer, memorySize );

	/* Allocation pointer and fence 
	// are only set above the dictionary base when you have to seed the dictionary, 
	// such as when starting from ROM 
	// (although BIF can't start from ROM at this point anyway).
	*/
	ORIG.initialForgetFence.bytep = memoryImage + ORIG.initialForgetFence.integer;

	ORIG.initialDictionaryAllocationPointer.bytep 
		= memoryImage + ORIG.initialDictionaryAllocationPointer.integer;
}

static void xWARM(void);	/* xWARM doesn't need extern visibility, but we think we need it here. */

boot_constants_s	romORIG =
{	xCOLD,	/* These need to be fixed, to initialize them correctly, */
	xWARM,	/* to avoid the (unlikely?) possibility of calling without a proper setjmp. */
	{ 0x43 },	/* CPUid is 'C' */
	{ 0 },	/* version */
	{ (natural_t) &hFORGET },	/* initial current, slightly wonky cast to get running. */
	{ '\b' },
	{ (natural_t) SZUSERTBLS },	/* task tables */
	{ (natural_t) CONTROL_STACK_SIZE + STACKGAP },	/* SP0 */
	{ (natural_t) STACKGAP },	/* RP0 */
	{ (natural_t) SZTIB },	/* terminal input buffer */
	{ NLMASK },
	{ MSG_INTERNAL_STRINGS },	/* initial disk on-line flag */
	/* Either I change the fence to see a flag or a link, or I copy the headers into memoryImage. */
	{ (natural_t) 0 /* allocatable area;  was FOLLOW - 1 */ },	/* FENCE for FORGET */
	{ (natural_t) 0 /* was FOLLOW */ },	/* initial allocation pointer. */
	{ (natural_t) &hBIF.parameterLink[ 0 ] },	/* initial context root, both contexts. */
	{ (natural_t) SZPAD },	/* numeric conversion buffer */
	{ (natural_t) SZWPAD + SZHASH },	/* word buffer */
	{ 32 },
	{ (natural_t) SZBUFF },	/* Disk buffers */
	{ (natural_t) ILIM },
	{ (natural_t) &hFORGET /* since the headers move, end of the linked list;  was marked by FENCE. */ },
	{ 0 /* Assume standard library echos. */ }
};


/* Initially, one big (so to speak) array of bytes, following the 6809 code (so to speak). 
*/


/* Need to do BREAK key handling, will do it strictly by the K&R book example.
*/
static sig_atomic_t got_break = 0;

static void catch_break( int siggie /* Shouldn't be much we need to look at here, anyway. Nothing, actually. */ )
{	signal( SIGINT, catch_break );
	got_break = 1;
	return;
}

int break_pressed( void )
{	int breakseen = ( got_break != 0 );
	got_break = 0;
	return breakseen;
}

/* Could declare the functions here, but want to follow the 6809 code order where I can. */
static void DOREGS(void);
static void MARK(void);
static void PRUNE(void);
static void CHECK_ORDER(void);
static void CHECK(void);
static void COUNTCLEAR(void);
static void LOADTREES(void);
static void BALANCETREES(void);


/* Need to handle stdin, stdout, stderr, and "stdlog". 
// This is not really what I want to do, even in the FIG bootstrap.
// But, since stdout is apparently not an lvalue, we need to indirect.
*/
FILE * standardInput = NULL;
FILE * standardOutput = NULL;
FILE * standardError = NULL;
FILE * standardLog = NULL;
/* For the BIF/6809 compatible bootstrap, all I really want is a kind of "stdlog".
// That can be done in the same file as >PRT and >VID.
*/


/*
01010 	FCC 'COLD'
01020 	FCB 4
01030 	FCB MFORE
01040 	FDB 0
01050 	FDB BIF+2
01060 	FDB 0
01070 	FDB 0
*/
#define WARM_JUMP_FLAG	0x5741
static jmp_buf warmBuffer;
#define COLD_JUMP_FLAG	0x435f
jmp_buf coldBuffer;
static character_t sCOLD[] = "\x4" "COLD";
definition_header_s hCOLD =	
{	{ (natural_t) sCOLD },
	{ 0 },
	{ 0 },	/* This is the very first definition in the initial base vocabulary. */
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },	/* Full leaf node. */
	{ 0 },
	{ (natural_t) COLD }
};
/* HEADER( COLD, 0 / * no special modes. * /, WARM, BIF, COLD, 0 ); */
void COLD(void)
{	longjmp( coldBuffer, COLD_JUMP_FLAG );
}

void xCOLD(void)
{
	standardInput = stdin;
	standardOutput = stdout;
	standardError = stderr;
	/* Check that we won't shoot ourselves in the foot with the definition of dblnatural_t . */
	if ( ( ( sizeof (dblnatural_t) / sizeof (cell_u) ) > 4 ) || ( ( sizeof (dblnatural_t) % sizeof (cell_u) ) != 0 ) )
	{	
		fprintf( standardError, "dblnatural_t is size %ld and cell_u is size %ld: exiting.\n", 
				 (long) sizeof (dblnatural_t), (long) sizeof (cell_u) );
		return;
	}
	/*  */
	signal( SIGINT, catch_break );	/* prepare for BREAK key handling. */
/* 01080 COLD	LEAY DOREGS,PCR */
/* 01090 	EXG PC,Y call */ /* Set up the essential registers, called without stack. */
	patchORIG();
#if defined DEBUGGING
fprintf( standardError, "memory image at: %p, ORIG at: %p, USER table at %p\n", 
	memoryImage, &ORIG, ORIG.initialTaskBase.bytep );
#endif /* defined DEBUGGING */
	DOREGS();	/* Points W to the initialization table instead of Y, use UP instead of X. */
/* 01100 	SETDP VDP */
/* 01110 	CLR $71 for BASIC reset */ /* But we aren't taking over the whole machine. */
/* 01120 	LDD -4,Y */
/* 01130 	STD UCURR,X */
	UP.task->lastDefined = W.bootConstantsp->initialCurrent;
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial current's name: %s\n", UP.task->lastDefined.definitionp->nameLink.chString + 1 );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
	/* work from here, fixing the initialization table as I go */
/* 01140 	LDD $0C,Y */
/* 01150 	STD UFENCE,X */	
	UP.task->forgetFence = W.bootConstantsp->initialForgetFence;
	/* But the FENCE won't work like this for built-ins. Not that it needs to. */
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial FENCE: %lx\n", UP.task->forgetFence.integer );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
	UP.task->tailFence = W.bootConstantsp->initialTailFence;
	/* For FENCEing by mark. */
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial TAIL: %lx\n", UP.task->tailFence.integer );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
/* 01160 	LDD $0E,Y */
/* 01170 	STD UDP,X */
	UP.task->dictionaryAllocationPointer = W.bootConstantsp->initialDictionaryAllocationPointer;
#if defined DEBUGGING
fprintf( standardError, "initial allocation pointer: %p\n", UP.task->dictionaryAllocationPointer.cellp );
#endif /* defined DEBUGGING */
/* 01180 	LDD $10,Y */
/* 01190 	STD UROOT,X */
/* 01200 	STD UDROOT,X */
	UP.task->definitionContextRoot = UP.task->searchContextRoot = W.bootConstantsp->initialDefinitionContextRoot;
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial context's name: %s\n", ( UP.task->searchContextRoot - ( PFAOFF / sizeof (cell_u) ).definitionp->nameLink.chString + 1 );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
	/* Currently hand-building the tree. Would it save some object space to explicitly build it? */
/* 01210 	PSHU D     Initial vocabularies */
/* 01220 	BSR PRUNE  must ALL be PRUNEd! */
	COUNTCLEAR();
	LOADTREES();
	BALANCETREES();
	MARK();	/* We want to know whether everything is linked in. */
	( * --SP ).definitionp = &hBIF;
	PRUNE();
	( * --SP ).definitionp = &hBIF;
	CHECK_ORDER();
/* 01230 	LDD #EDITOR+2 */
/* 01240 	PSHU D */
/* 01250 	BSR PRUNE */
	( * --SP ).definitionp = &hEDITOR;
	PRUNE();
	( * --SP ).definitionp = &hEDITOR;
	CHECK_ORDER();
/* 01260 	LDD #ASMBLR+2 */
/* 01270 	PSHU D */
/* 01280 	BSR PRUNE */
	( * --SP ).definitionp = &hASMBLR;
	PRUNE();
	( * --SP ).definitionp = &hASMBLR;
	CHECK_ORDER();
	( * --SP ).definitionp = &hUTIL;	/* bug in BIF6809 */
	PRUNE();
	( * --SP ).definitionp = &hUTIL;
	CHECK_ORDER();
	( * --SP ).definitionp = &hEXTENSIONS;
	PRUNE();
	( * --SP ).definitionp = &hEXTENSIONS;
	CHECK_ORDER();
	CHECK();
	/* 01290 	JMP WARM+5 */
	setjmp( warmBuffer );
	xWARM();
}

#if defined DBG_SHOW_SYMBOL_TABLE
static void dumpNameString( definition_header_s * header )
{	if ( header == ( definition_header_s * ) 0 )
	{	fputs( "{NULL pointer}", standardError );
	}
	else
	{	char * name = (char *) header->nameLink.chString;
		if ( name == (char *) 0 )
		{	fputs( "{NULL name string pointer}", standardError );
		}
		else if ( name[ 0 ] == 0 )
		{	fputs( "{empty name}", standardError );
		}
		else 
		{	fputs( name + 1, standardError );
		}
	}
}
#endif /* defined DBG_SHOW_SYMBOL_TABLE */


#if defined DBG_SHOW_SYMBOL_TABLE
static void dumpNodeStructure( definition_header_s * header )
{	if ( header == ( definition_header_s * ) 0 )
	{	fputs( "{NULL node}", standardError );
	}
	else
	{	dumpNameString( header );
		fputs( ":(left)->", standardError );
		dumpNameString( header->leftLink.definitionp );
		fputs( ":(right)->", standardError );
		dumpNameString( header->rightLink.definitionp );
	}
}
#endif /* defined DBG_SHOW_SYMBOL_TABLE */

static void dumpHeader( definition_header_s * header, int linksFlag )
{	fprintf( standardError, "hd:%p", header );
	if ( header != ( definition_header_s * ) 0 )
	{	character_t * name = header->nameLink.chString;
		if ( linksFlag )
		{	fprintf( standardError, "\tup:%p", header->vocabLink.definitionp );
		}
		fprintf( standardError, "\tnm:%p", name );
		if ( name != (character_t *) 0 )
		{	unsigned long length = (unsigned long) strlen( (char *) name + 1 );
			unsigned lengthByte = (unsigned char) name[ 0 ];
			fprintf( standardError, "\t%s\t%d:%lu%c", name + 1, 
					 lengthByte, length, (lengthByte != length) ? '*' : ' '
				   );
		}
		else
		{	fputs( "\t*-none-*\t0:0!", standardError );
		}
		fprintf( standardError, "\tim:%04lx\tam:%04lx\tal:%p", 
				 header->interpMode.integer, 
				 header->allocMode.integer,
				 header->allocLink.definitionp 
			   );
		if ( linksFlag )
		{	fprintf( standardError, "\tl:%p\tr:%p\tcd:%p\tp[0]:%08lx", 
					 header->leftLink.definitionp, 
					 header->rightLink.definitionp,
					 header->codeLink.icodep, 
					 header->parameterLink[ 0 ].integer 
				   );
		}
	}
	fputc( '\n', standardError );
}


/* Eventually, we want to do this with "native" definitions.
*/
#define CCRECWID	4
#define CC_ARRAY( adr, x )	( (adr)[ ( (x) * CCRECWID ) + 0 ] )
#define CC_VOCAB( adr, x )	( (adr)[ ( (x) * CCRECWID ) + 1 ] )
#define CC_CNT( adr, x )	( (adr)[ ( (x) * CCRECWID ) + 2 ] )
#define CC_XTRA( adr, x )	( (adr)[ ( (x) * CCRECWID ) + 3 ] )
static void COUNTCLEAR(void)
{	cell_u * memory = UP.task->dictionaryAllocationPointer.cellp;
	natural_t vocabCount = 0;
	definition_header_s * tracer;

	for ( tracer = UP.task->tailFence.definitionp;
	      tracer != (definition_header_s *) 0;
	      tracer = tracer->allocLink.definitionp
	     )
	{	definition_header_s * found = (definition_header_s *) 0;
		natural_t i;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		for ( i = 0; i < vocabCount; ++i )
		{	if ( tracer->vocabLink.definitionp == CC_VOCAB( memory, i ).definitionp )
			{	found = tracer->vocabLink.definitionp;
				++( CC_CNT( memory, i ).integer );
				break;
			}
		}
		if ( found == (definition_header_s *) 0 )
		{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( stderr, "Found new vocab #%ld (at %p): \n", 
		vocabCount + 1, memory + vocabCount * CCRECWID );
dumpHeader( tracer->vocabLink.definitionp, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
			CC_ARRAY( memory, vocabCount ).integer 
				= CC_XTRA( memory, vocabCount ).integer = 0;
			CC_VOCAB( memory, i ).definitionp = tracer->vocabLink.definitionp;
			CC_CNT( memory, i ).integer = 1;
			++vocabCount;
			/* Clear the vocabulary root link. */
			tracer->vocabLink.definitionp->parameterLink[ 0 ].bytep = (byte_p) 0;
		}
		tracer->leftLink.bytep = tracer->rightLink.bytep = (byte_p) 0;
	}
	/* push the count. Adjust the dictionary pointer? No. */
	( * --SP ).integer = vocabCount;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE || 1
{
natural_t i;
fprintf( stderr, "Found %ld vocabs\n", vocabCount );
for ( i = 0; i < vocabCount; ++i )
{
	fprintf( stderr, "%12p: %12ld %12p %12ld %12ld\n", 
		memory + i * CCRECWID, 
		memory[ i * CCRECWID ].integer, memory[ i * CCRECWID + 1 ].definitionp,
		memory[ i * CCRECWID + 2 ].integer, memory[ i * CCRECWID + 3 ].integer );
	dumpHeader( memory[ i * CCRECWID + 1 ].definitionp, 0 );
}
}
/* UP.task->dictionaryAllocationPointer.cellp = memory + vocabCount * CCRECWID; */
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
}

static void LOADTREES(void)
{	definition_header_s * tracer;

	for ( tracer = UP.task->tailFence.definitionp;
	      tracer != (definition_header_s *) 0;
	      tracer = tracer->allocLink.definitionp
	     )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "** LOADTREE()ing: ", stderr );
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		( * --SP ).bytep = tracer->nameLink.bytep + 1;	/* Name string. */
		( * --SP ).cellp = &( tracer->vocabLink.definitionp->parameterLink[ 0 ] );
		PREF();
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "Thought we found: ", stderr );
dumpHeader( (struct definition_header_s *) ( SP[ 1 ].bytep - SP[ 0 ].integer ), 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		( * ( SP[ 1 ].cellp ) ).definitionp = tracer;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "Did we link it?   ", stderr );
dumpHeader( (struct definition_header_s *) ( SP[ 1 ].bytep - SP[ 0 ].integer ), 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		SP += 3;
	}
}

/* ( vocab_header array -- vocab_header array ) */
static void FLATTENTREE(void)
{	cell_u * list = SP[ 0 ].cellp;
	definition_header_s * node 
		= SP[ 1 ].definitionp->parameterLink[ 0 ].definitionp;
	cell_u * visitedMark = RP;

#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( stderr, "Starting to flatten down %p\n", list );
dumpHeader( node, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
	if ( ( list == (cell_u *) 0 ) 
	     || ( node == (definition_header_s *) 0 ) )
	{	return;
	}
	for ( ;; )
	{
		while ( node != (definition_header_s *) 0 )
		{	if ( RP <= UP.task->dataStackBase.cellp + STACK_BUMPER_HOLE )
			{	fputs( "Control stack overflow in symbol table balancing.\n", stderr );
				exit( EXIT_FAILURE );
			}
			( * --RP ).definitionp = node;
			node = node->leftLink.definitionp;
		}
		if ( RP >= visitedMark )	/* No left child, any waiting? */
		{	break;
		}
		node = ( * RP++ ).definitionp;	/* Do the one waiting. */
		( * list++ ).definitionp = node;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
dumpHeader( node, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		node = node->rightLink.definitionp;	/* Put the right child on the waiting list. */
	}
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( stderr, "Ending flatten %p\n", list );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
}

/* ( vocab_header array count -- vocab_header array count ) */
static void RELOADTREE(void)
{	
	cell_u * list = SP[ 1 ].cellp;
	cell_u * nodep 
		= & ( SP[ 2 ].definitionp->parameterLink[ 0 ] );
	natural_t base = 0;
	natural_t bound = SP[ 0 ].integer;	/* bound is always one beyond. */
	cell_u * visitedMark = RP;

	/* Need to check both stack sizes, not just RP. 
	// And that means I need to properly allocate the arrays I'm using.
	// Eh, heh heh, ahem.
	// Next time around.
	*/
	( * --SP ).integer = bound;
	ULOG2();
	if ( ( ( * SP++ ).sinteger + 1 ) > ( CONTROL_STACK_SIZE / BYTESPERCELL ) )
	{	fprintf( stderr, "RP stack too small to rebuild startup symbol table.\n" );
		exit( EXIT_FAILURE );
	}
	while ( base < bound )
	{
		natural_t mid = base + ( bound - base ) / 2;
		definition_header_s * node = list[ mid ].definitionp;

#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( stderr, "** RELOADTREE() %ld %ld %ld starting: ", base, mid, bound );
dumpHeader( node, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */

		/* Firewall */
		node->leftLink.definitionp 
			= node->rightLink.definitionp = (definition_header_s *) 0;
		( * nodep ).definitionp = node;	/* Link mid in. */
		if ( base < mid )
		{
			if ( mid + 1 < bound )
			{	/* Save the right child for later. */
				( * --RP ).cellp = &( node->rightLink );
				( * --SP ).integer = mid + 1;	/* For new base. */
				( * --SP ).integer = bound;	/* Keep bound. */
				bound = mid - 1;
			}
			bound = mid;	/* New bound, keep base. */
			nodep = & ( node->leftLink );
		}
		else	/* No left child in this branch. */
		{	if ( RP < visitedMark )	/* any waiting? */
			{	bound = ( * SP++ ).integer;
				base = ( * SP++ ).integer;
				nodep = ( * RP++ ).cellp;
			}
			else
			{	break;		/* Anyway, we're done if we get here. */
			}
		}
	}
}


static void BALANCETREES(void)
{	cell_u * memory = UP.task->dictionaryAllocationPointer.cellp;
	natural_t vocabCount = SP[ 0 ].integer;
	cell_u * free = memory + vocabCount * CCRECWID;
	natural_t i;

	CC_ARRAY( memory, 0 ).cellp = free;
	for ( i = 1; i < vocabCount; ++i )
	{	CC_ARRAY( memory, i ).cellp 
			= CC_ARRAY( memory, i - 1 ).cellp + CC_CNT( memory, i - 1 ).integer;
	}
	for ( i = 0; i < vocabCount; ++i )
	{	( * --SP ).definitionp = CC_VOCAB( memory, i ).definitionp;
		( * --SP ).cellp = CC_ARRAY( memory, i ).cellp;
		FLATTENTREE();
		( * --SP ).integer = CC_CNT( memory, i ).integer;
		RELOADTREE();
		SP += 3;
	}
}


static void MARK(void)
{	definition_header_s * tracer = UP.task->tailFence.definitionp;
	while ( tracer != (definition_header_s *) 0 )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		tracer->allocMode.integer &= ~( VOC_MEMO_MARK | VOC_CHECKED_MARK );
		tracer->allocMode.integer |= BOOT_VOC_MARK;
		tracer = tracer->allocLink.definitionp;
	}
}


/* 01300 	SETDP 0 */
/* 01310 * Watch stack (vocabulary) depth! */
/* 01315 	SETDP VDP */
/* 01320 PRUNE	PSHS D,X,Y */
/* 01330 	LDY <UP */
/* 01340 	LDX ,U++ vocab */
/* 01344 	BEQ PRUNEX+2 */
/* 01348 	LDD ,X root */
/* 01352 	BEQ PRUNEX+2 */
/* 01356 	CMPD UFENCE,Y */
/* 01360 	BLS PRUNSK */
/* 01364 	LDD #0 */
/* 01368 	STD ,X */
/* 01372 	BRA PRUNEX+2 */
/* 01376 PRUNSK	TFR D,X */
/* 01380 	LDD #0 */
/* 01386 	PSHS D mark */
/* 01390 PRUNL	LDD RTOFF,X */
/* 01400 	BEQ PRUNLF */
/* 01410 	CMPD UFENCE,Y */
/* 01420 	BLS PRUNLF-2 */
/* 01430 	LDD #0 make leaf */
/* 01440 	STD RTOFF,X */
/* 01450 	BRA PRUNLF */
/* 01460 	PSHS D for later */
/* 01470 PRUNLF	LDD LFTOFF,X */
/* 01480 	BEQ PRUNEX-2 */
/* 01490 	CMPD UFENCE,Y */
/* 01500 	BHI PRUNL0 */
/* 01510 	TFR D,X go left */
/* 01520 	BRA PRUNL */
/* 01530 PRUNL0	LDD #0 make leaf */
/* 01540 	STD LFTOFF,X */
/* 01550 	LDX ,S++ go right? */
/* 01560 PRUNEX	BNE PRUNL */
/* 01570 	PULS D,X,Y,PC */
/* 01580 	SETDP 0 */
/* 01590 * 
*/

static void PRUNE(void)	/* Static binary tree, not balanced at run-time. */
{	definition_header_s * tracer = ( * SP++ ).definitionp;
	definition_header_s * probe;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "Starting the prune.\n", standardError );
fprintf( standardError, "allocation area limits, bottom:%p, top:%p\n", 
		 UP.task->forgetFence.definitionp, W.bootConstantsp->initialLimitOfBufferRAM.definitionp );
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
	if ( ( tracer == (definition_header_s *) 0 )
		 || ( tracer->codeLink.icode != XVOC ) 
		 || ( tracer->parameterLink[ 0 ].cellp == (cell_u *) 0 )
	   )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( standardError, "Early exit on %p\n", tracer );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		return;
	}
	probe = tracer->parameterLink[ 0 ].definitionp;
	/* If I start balancing the tree, I'll need to go to the leaf nodes before pruning. */
	if ( ( probe > UP.task->forgetFence.definitionp ) 
		 && ( probe < W.bootConstantsp->initialLimitOfBufferRAM.definitionp ) )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( standardError, "Whole tree lopped under %p\n", tracer );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		tracer->parameterLink[ 0 ].integer = 0;	/* Assume the whole vocabulary can be lopped. */
		return;
	}
	( * --SP ).integer = 0;	/* Mark our stopping point with a NULL. (On the parameter stack instead of the return stack.) */
	/* Look right and go left. */
	while ( probe != (definition_header_s *) 0 )
	{
		if ( ( probe->allocMode.integer & VOC_MEMO_MARK ) != 0 )
		{
#if defined DEBUGGING
fputs( "** Retracing our steps! **", standardError );
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING */
			break;
		}

		probe->allocMode.integer |= VOC_MEMO_MARK;
		tracer = probe;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		probe = tracer->rightLink.definitionp;
		if ( probe != (definition_header_s *) 0 )
		{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "looking right, at ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
			if ( ( probe > UP.task->forgetFence.definitionp ) 
				 && ( probe < W.bootConstantsp->initialLimitOfBufferRAM.definitionp ) )
			{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "pruned right \n", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
				tracer->rightLink.definitionp = (definition_header_s *) 0;	/* Assume the whole subtree can be lopped. */
			}
			else
			{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "saving right branch to ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
				( * --SP ).definitionp = probe;	/* Save it to come back to later. */
			}
		}
		probe = tracer->leftLink.definitionp;
		if ( probe != (definition_header_s *) 0 )
		{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "looking left, at ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
			if ( ( probe > UP.task->forgetFence.definitionp ) 
				  && ( probe < W.bootConstantsp->initialLimitOfBufferRAM.definitionp ) )
			{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "pruned left", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
				probe = tracer->leftLink.definitionp = (definition_header_s *) 0;	/* Assume the whole subtree can be lopped. */
				/* Leave probe as a flag for descent or no descent. */
			}
		}
		if ( probe == (definition_header_s *) 0 )	/* Might have been cleared above. */
		{
			probe = ( * SP++ ).definitionp;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "moving right to ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		}
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
		else
		{
fputs( "descending left to", standardError );
dumpHeader( probe, 1 );
		}
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
	}
}


/*  AARRGGHHH!!!!!!! Don't pass this a NULL, anyway. */
static void recurseOrder( definition_header_s *** listp, definition_header_s * probe, int depth )	/* (Gave up trying to mix FORTH and C parameters here.) */
{	if ( ( probe->allocMode.integer & VOC_CHECKED_MARK ) != 0 )
	{	fputs( ">> Vocabulary link cycle found in order check! ** STOPPING ** <<", standardError );
		dumpHeader( probe, 1 );
		sysSIG.integer = ICODE_LIST_INITIALIZATION_ERROR;
		return;
	}
	probe->allocMode.integer |= VOC_CHECKED_MARK;
	if ( !( probe->allocMode.integer & BOOT_VOC_MARK ) )
	{	fputs( ">> Header not linked in allocation list! <<", standardError );
		dumpHeader( probe, 1 );
		sysSIG.integer = ICODE_LIST_INITIALIZATION_ERROR;
	}
	probe->allocMode.integer &= ~BOOT_VOC_MARK;
	if ( probe->leftLink.definitionp != (definition_header_s *) 0 )
	{	recurseOrder( listp, probe->leftLink.definitionp, depth + 1 );
	}
	* ( * listp )++ = probe;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
{	int i;
	for ( i = 0; i < depth; ++i )	fputs( "+\t", standardError );
	fprintf( standardError, "%d: ", depth );
}
dumpNodeStructure( probe );
fputc( '\n', standardError );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
	if ( probe->rightLink.definitionp != (definition_header_s *) 0 )
	{	recurseOrder( listp, probe->rightLink.definitionp, depth + 1 );
	}
}

static void CHECK_ORDER(void)	/* But we need to show/check the static tree. */
{	definition_header_s * probe = ( * SP++ ).definitionp;
	definition_header_s ** list = (definition_header_s **) memoryImage;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "*****\nDumping the tree in order:\n*****\n", standardError );
fprintf( standardError, "allocation area limits, bottom:%p, top:%p\n", 
		 UP.task->forgetFence.definitionp, W.bootConstantsp->initialLimitOfBufferRAM.definitionp );
dumpHeader( probe, 1 );
fputs( "--------showing the tree--------\n", standardError );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
	if ( probe == (definition_header_s *) 0 )
	{	fputs( "<*><*><*>Tree order check attempted on a NULL pointer!<*><*><*>\n", standardError );
		return;
	}
	if ( probe->codeLink.icode != XVOC ) 
	{	fputs( "<*><*><*>Tree order check attempted on a non-vocabulary:<*><*><*>\n", standardError );
		dumpHeader( probe, 1 );
		return;
	}
	if ( probe->parameterLink[ 0 ].cellp == (cell_u *) 0 )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( standardError, "Early exit on empty vocabulary %p.\n", probe );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		return;
	}
	/* Start down the tree: */
	probe = probe->parameterLink[ 0 ].definitionp;
	recurseOrder( &list, probe, 0 );
	* list++ = (definition_header_s *) 0;
fputs( "--------checking the order--------\n", standardError );
	list = (definition_header_s **) memoryImage;
	for ( ;; ) 
	{	definition_header_s * left = list[ 0 ];
		definition_header_s * right = list[ 1 ];
		if ( (left == (definition_header_s *) 0 ) || ( right == (definition_header_s *) 0 ) )
		{	break;
		}
		if ( strcmp( (char *) left->nameLink.chString + 1, (char *) right->nameLink.chString + 1 ) > 0 )
		{	fputs( "*** SYMBOL TABLE OUT OF ORDER ***\n", standardError );
			dumpHeader( left, 1 );
			dumpHeader(	right, 1 );
		}
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
dumpHeader( left, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		++list;
	} 
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
		dumpHeader( * list, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
}


static void CHECK(void)
{	definition_header_s * tracer = UP.task->tailFence.definitionp;
	while ( tracer != (definition_header_s *) 0 )
	{	natural_t flags = tracer->allocMode.integer;
		if ( ( flags & VOC_MEMO_MARK ) == 0 )
		{	fputs( "** !!Broken Symbol Table Links!! **: ", standardError );
			dumpHeader( tracer, 1 );
		}
		tracer->allocMode.integer &= ~VOC_MEMO_MARK;
		tracer = tracer->allocLink.definitionp;
	}
}


/* Point the essential registers to real/available memory.
*/
/* 02000 	SETDP 0 */
static void DOREGS(void)
{
#if defined DEBUGGING
	void * stackAddressDummy = NULL;
#endif /* defined DEBUGGING */
/* 02010 DOREGS	LDS #IRP0 */
	/* These are to make the CPU sane. Need to get these from the initializations table, instead? */
/*	RP = (cell_u *) ( memoryImage + IRP0 );	// * woops! Don't make that 4 times allocation. */
	RP = ORIG.initialReturnStackBase.cellp;
/* 02020 	PSHS CC save */ /* Not real registers, no need to mask interrupts. */
/* 02030 	ORCC #$50 mask ints */
/* 02040 	PSHS Y return adr */ /* Using a standard C call, no need to manipulate the return address. */
/* 02050 	LDD #DPAGE */
/* 02060 	TFR A,DP */
/* 02070 	SETDP VDP */ /* Not messing with short (DP) addressing. */
/* 02080 	LDY #ORIG+$10 */ /* We'll point at with W, instead. */
	W.bytep = (byte_p) &ORIG;
/* 02090 	LDX ,Y */
/* 02100 	STX <UP */
/*	UP = W.bootConstantsp->initialTaskBase; */
	UP = ORIG.initialTaskBase;
/* 02110 	LDU #ISP0 */
/*	SP = (cell_u *) ( memoryImage + ISP0 ); */
	SP = ORIG.initialParameterStackBase.cellp;
/* 02120 	LDD #0 */
/* 02130 	STD >0 trap [NULL]s */
	/* Can't really do this portably, need to conditionally compile for architectures that don't trap NULL refs. */
#if !defined SYSTEM_TRAPS_NULLS
	* ( (natural_t * ) 0 ) = 0;
#endif
	sysSIG.integer = 0;	/* Since the C version uses this flag variable to get out of the interpreter loop. */
/* 02140 	RTS */
#if defined DEBUGGING
fprintf( standardError, "RP: %p, SP: %p, DOREG local variable address: %p\n", RP, SP, &stackAddressDummy );
fprintf( standardError, "UP.bytep: %p, UP.integer: %lx, W.bootConstantsp: %p, W.integer: %lx\n", UP.bytep, UP.integer, W.bootConstantsp, W.integer );
#endif /* defined DEBUGGING */
}
/* 02150 	SETDP 0 */
/* 02160 * 
*/


#define BYE_JUMP_FLAG	0x4259
static jmp_buf byeBuffer;
static character_t sBYE[] = "\x3" "BYE";
definition_header_s hBYE = 
{	{ (natural_t) sBYE },
	{ 0 },
	{ (natural_t) &hCOLD },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) BYE }
};
void BYE( void )
{	longjmp( byeBuffer, BYE_JUMP_FLAG );
}


static cell_u iWARM[];
/*
02230 	FCC 'WARM'
02240 	FCB 4
02250 	FCB MFORE
02260 	FDB COLD-CFAOFF
02270 	FDB BIF+2
02280 	FDB VLIST-CFAOFF
02290 	FDB WARN-CFAOFF
*/
static character_t sWARM[] = "\x4" "WARM";
definition_header_s hWARM =	
{	{ (natural_t) sWARM },
	{ 0 },
	{ (natural_t) &hBYE },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) WARM }
};
void WARM(void)
{	longjmp( warmBuffer, WARM_JUMP_FLAG );
}

void xWARM(void)
{
//HEADER( WARM, 0 /* no special modes. */, COLD, BIF, WARM, 0 );
/* 02300 WARM	LEAY <DOREGS,PCR */
/* 02310 	EXG PC,Y call */
	DOREGS();
/* 02320 	SETDP VDP */
/* 02360 	LDD 2,Y */
/* 02370 	STD US0,X */
	UP.task->dataStackBase = W.bootConstantsp->initialParameterStackBase;
#if defined DEBUGGING
fprintf( standardError, "initial parameter stack base: %p\n", UP.task->dataStackBase.cellp );
#endif /* defined DEBUGGING */
/* 02380 	STD UCSP,X */
	UP.task->compilerStackMarker = W.bootConstantsp->initialParameterStackBase;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial compile stack mark: %p\n", UP.task->compilerStackMarker.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02390 	LDD 4,Y */
/* 02400 	STD UR0,X */
	UP.task->returnStackBase = W.bootConstantsp->initialReturnStackBase;
#if defined DEBUGGING
fprintf( standardError, "initial return stack base: %p\n", UP.task->returnStackBase.cellp );
#endif /* defined DEBUGGING */
/* 02410 	LDD -2,Y */
/* 02420 	STD UBS,X */
	UP.task->backSpaceConstant = W.bootConstantsp->terminalBackSpace;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial backspace: %lx\n", UP.task->backSpaceConstant.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02430 	LDD 6,Y */
/* 02440 	STD UTIB,X */
	UP.task->terminalInputBuffer = W.bootConstantsp->initialTerminalInputBuffer;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial terminal input buffer: %p\n", UP.task->terminalInputBuffer.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02450 	LDD $0A,Y */
/* 02460 	STD UWARN,X */
	UP.task->diskOnLine = W.bootConstantsp->initialDiskOnLine;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial disk on line flag: %lx\n", UP.task->diskOnLine.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02470 	LEAY $10,Y */
/* 02480 	LDD 2,Y */
/* 02490 	STD UPAD,X */
/* 02500 	STD UHLD,X */
	UP.task->numericConversionScratchPad = UP.task->padMarker = W.bootConstantsp->initialNumericConversionScratchPad;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial scratchpad & mark: %p\n", UP.task->numericConversionScratchPad.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02510 	LDD 4,Y */
/* 02520 	STD UWP,X */
	UP.task->wordBufferPointer = W.bootConstantsp->initialWordBufferPointer;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial word buffer pointer: %p\n", UP.task->wordBufferPointer.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02530 	LDD 6,Y */
/* 02540 	STD UCOLUM,X */
	UP.task->terminalColumns = W.bootConstantsp->initialTerminalColumns;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial terminal width: %lx\n", UP.task->terminalColumns.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02550 	LDD 8,Y */
/* 02560 	STD UFIRST,X */
/* 02570 	STD UUSE,X */
/* 02580 	STD UPREV,X */
	UP.task->firstByteOfBuffers = UP.task->leastRecentBuffer = UP.task->mostRecentBuffer = W.bootConstantsp->initialDiskBuffers;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial disk buffers: %p\n", UP.task->firstByteOfBuffers.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02590 	LDD $0A,Y */
/* 02600 	STD ULIMIT,X */
	UP.task->limitOfBufferRAM = W.bootConstantsp->initialLimitOfBufferRAM;
#if defined DEBUGGING
fprintf( standardError, "initial RAM limit: %p\n", UP.task->limitOfBufferRAM.cellp );
#endif /* defined DEBUGGING */
/* 02610 	LDD #16 */
/* 02620 	STD UBASE,X */
	UP.task->numericBase.integer = 16;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial numeric base: %lx\n", UP.task->numericBase.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02630 	LDD #0 */
/* 02640 	STD [UR0,X] hole */
	( * UP.task->returnStackBase.cellp ).integer = 0;
/* 02650 	STD [US0,X] hole */
	( * UP.task->dataStackBase.cellp ).integer = 0;
#if defined DEBUGGING
fprintf( standardError, "hit the holes?: %lx %lx (zeros?)\n", SP[ 0 ].integer, RP[ 0 ].integer );
#endif /* defined DEBUGGING */
	{
/* 02660 	LDA #(UEND-UIN) */
		snatural_t count = UEND - UIN;
/* 02670 	LEAY UIN,X */
		byte_t * ptr = UP.bytep + UIN;
#if defined DEBUGGING && DEBUGGING <= 2
/* Watch this. Some compilers think size_t is unsigned long, some think it is int, etc. */
fprintf( standardError, "UP.task: %p UEND: %ld, UIN: %ld\n", UP.task, (natural_t) UEND, (natural_t) UIN );
fprintf( standardError, "one past end should be %p == %p\n", UP.task->spare + UP_SPARE_COUNT, ptr + count );
fprintf( standardError, "clearing %ld bytes from: %p, UIN should be at: %p\n", count, ptr, &( UP.task->bufferInputOffset ) );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02680 WARMLY	STB ,Y+ */ /* was safe because I knew it was non-zero. */
/* 02690 	DECA */
		while ( --count >= 0 )
		{
#if defined DEBUGGING && DEBUGGING <= 2
/* dbg */ if ( ptr >= UP.bytep + UEND - 1 ) fprintf( standardError, "clear final at %p\n", ptr );
/* dbg */ if ( ptr >= (byte_p) ( UP.task->spare + UP_SPARE_COUNT ) ) fprintf( standardError, "overrun at %p\n", ptr );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
			* ptr++ = 0;	/* Clear out the rest. Bytewise is not as fast, fix it later. */
/* 02700 	BNE WARMLY */
		}
	}
	/* Until I decide to brave curses or whatever. */
	UP.task->terminalEcho = W.bootConstantsp->initialTerminalEcho;
	/* Patching i-code list tail robbing branches, since C requires initializations to be constant. 
	** Not really a good thing to do, refactoring would be better, but I need to get it running first.
	*/
	hDIFIND.parameterLink[ 3 ].integer 
		=  sizeof (cell_u) * ( ( hDDFIND.parameterLink + 2 ) - ( hDIFIND.parameterLink + 4 ) );
	
/* 02740 	PULS CC */	/* Not messing with interrupts. */
/* 02745 	ANDCC #$EF enable IRQ (disc) */
	/* Call high level code. Maybe I want to do this a different way, maybe I don't. 
	*/
	/* This wasn't what I wanted:
	** W.cellp = iWARM + 1;	/ * Since DOCOL doesn't grab it from the stack, here, ... */
	/* DOCOL(); */
	/* ( * --RP ).definitionp = &hWARM; This didn't make a good buffer, anyway. */
#if defined DBG_TRACE_NEXT
	W.definitionp = &hCOLD;	/* As a visual, non-functional marker for debugging. */
#endif
	IP = iWARM;	/* As if we just executed the first definition. Might want to make a macro for shifting gears? */
	/* Gotta be able to do NEXT into the WARM boot code? 
	*/
	if ( setjmp( byeBuffer ) == 0 )	/* May need to eventually do something else so I can clean up. */
	{	XNEXT();
	}
/*	while ( sysSIG.integer == 0 )
	{	W = ( * IP++ );
		( * ( * W.definitionp ).codeLink.icode )();
	}
*/
}
/* 02750 	DOCOL */
#include "bif_eval.h"	/* for testing */
#include "bif_io.h"	/* for testing */
static cell_u iWARM[] =
{
	/* And this is another place I end up stuck. empty-buffers, yes. 
	// ABORT Could be a security hole to (C subroutine) call from here. 
	*/
/* 02760 	FDB EMTBUF */	/* bif4/a -- EMTBUF was assembly language in my 6809 version */
	{ (natural_t) &hEMTBUF },
/* 02770 	FDB ABORT */	/* bif7b/a --> many, of course */
	{ (natural_t) &hPHABORT }	/* This must be the one that actually does the setjmp(); */
};
/* 02780 	SETDP 0 */
/* 02890 * */



