/*
**  bif_vm.c
**  bif-c
**
**  Created by Joel Rees on 2009/07/22.
**  Copyright 2009 __Reiisi_Kenkyuu__. All rights reserved.
**
** Translated to C from BIFDP/A, as mechanically as possible.
**
*/


#include "bif_vm.h"
#include "bif5b_a.h" /* for mERROR() */
#include "bif7b_a.h"


/*
** I am not going to try to emulate what I did with the direct page.
** Wouldn't really make sense.
** It's all absolute addressed with short absolute addresses, anyway.
00020 * Direct Page definitions for BIF
00030 * BIF Copyright 1989 Joel Matthew Rees (see BIF/ASM)
00060 * This file is dependent on BIF/M definitions
00100 DPAGE	EQU *
00105 VDP	EQU DPAGE/256
00110 * ORG needs to leave space for this code
00120 	SETDP VDP
00130 * COLD needs to load VDP into DP register
00500 *
00510 * Direct Page variables
*/


/* Since this is where most of the inner interpreter got defined, 
** I'll define the virtual machine registers here, too.
** Notes from BIFDOC.TXT:
*******************************************************************************
                        The BIF Virtual Machine

fig     6809
UP      [DP]    pointer to the per-USER variable table (USER Pointer)
IP      Y       pointer to the next definition (Instruction Pointer)
RP      S       return/control stack pointer
SP      U       parameter/data stack pointer
W       [S]     pointer to executing definition's parameter field
*/


/* This is a pointer to the taskrecord_s structure for the current task.
** It will be set in COLD, in bifst_a.
** But this is a prime candidate for early refactoring. Or maybe not.
00600 UP	RMB 2
*/
cell_u	UP;	/* [DP]    pointer to the per-USER variable table (USER Pointer) */
cell_u	* IP;	/* Y       Pointer to the next definition to execute (Instruction Pointer). */
cell_u	* RP;	/* S       return/control stack pointer */
cell_u	* SP;	/* U       parameter/data stack pointer */
/* W was originally (ephemeral) on the 6809 S stack because there weren't enough registers. */
cell_u	W;	/* [S]     pointer to executing definition's parameter field */
cell_u	volatile sysSIG;	/* Added as a way to break out of the inner interpreter. */

/* No need for these spares in the C runtime environment? Well, maybe.
00610 	RMB 8 spares
*/
cell_u spares[ SPARE_REG_COUNT ];


#if defined KERNEL_VOCABULARY_JUNK
/* Had this idea late last night, then thought better of it.
**
** The fig FORTH does not provide symbol table entries for these, 
** but, since I'm using C conventions and linking to the bases of the structures,
** I'm going to provide symbol table entries in a separate vocabulary.
** This could be convenient for debugging the symbol table early, as well.
*/
static character_t sxKERNEL[] = "\x7" "xKERNEL";
definition_header_s hxKERNEL =
{	{ (natural_t) sxKERNEL },
	{ MIMM },
	{ 0 },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) XVOC }, 
	{ { (natural_t) &hxKERNEL } }
};


static character_t sXNEXT[] = "\x5" "XNEXT";
definition_header_s hXNEXT =	
{	{ (natural_t) sXNEXT },
	{ MCOMP },
	{ (natural_t) &hxKERNEL },
	{ MFORE },
	{ (natural_t) &hxKERNEL },
	{ 0 },
	{ 0 },
	{ (natural_t) XNEXT }
};
#endif
/* My 6809 implementation had no specific inner interpreter. 
** See the NEXT macro in BIF/M (bif_m.h).
** Since we can't jump outside the function definition in C, we can't do that here.
** I'll provide sysSIG as a way to return to main().
** Note that the fig forth linked directly to fields inside the definition header structure, 
** saving offset calculations, and so did my 6809 implementation.
** It could be done with C, but I'm not going to. I'll use C conventions, instead.
** See XCOL for more on what's happening here.
**
** It's tempting to want to use a direct function call, but then how does one get anything 
** but the function from the i-code? How do you find the entry in the symbol table or even 
** extract the address being called without actually calling it and then trying to look in 
** the CPU's program counter?
*/
/* Do not absorb this into the WARM boot code:
*/
#if defined DBG_TRACE_NEXT
#define DUMP_W( tag )	\
	fprintf( standardError, "{%s} W:{%p:%s} ", tag, \
	W.bytep, W.definitionp->nameLink.chString + 1 )

static void dumpStack(void)
{	cell_u * i;
	fprintf( standardError, "SP:{%p:", SP );
	for ( i = UP.task->dataStackBase.cellp - 1; i >= SP; --i )
	{	fprintf( standardError, "%08lx", (unsigned long) ( * i ).integer );
		if ( i > SP )
		{	fputc( ',', standardError );
		}
	}
	fputc( '}', standardError );
}

void dumpState( char * tag )
{	DUMP_W( tag );
	dumpStack();
	fputc( '\n', standardError );
}

static int tracing = 0;

int isTracing( void )
{	return tracing;
}

static character_t sTRACEON[] = "\x7" "TRACEON";
definition_header_s hTRACEON =	
{	{ (natural_t) sTRACEON },
	{ 0 },
	{ (natural_t) NULL },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) NULL },
	{ 0 },
	{ (natural_t) TRACEON }
};
extern void TRACEON( void )
{	tracing = 1;
}

static character_t sTRACEOFF[] = "\x8" "TRACEOFF";
definition_header_s hTRACEOFF =	
{	{ (natural_t) sTRACEOFF },
	{ 0 },
	{ (natural_t) NULL },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) NULL },
	{ 0 },
	{ (natural_t) TRACEOFF }
};
extern void TRACEOFF( void )
{	tracing = 0;
}


#endif

/*
NEXT    ( --- )         jmp [,y++] (macro in bif.m)
        Causes the next definition to execute.
*/
inline void XNEXT (void)
{
#if defined DBG_TRACE_NEXT
	cell_u localW = W;
#endif
	while ( ( sysSIG.integer == ICODE_LIST_CONTINUE ) && ( ( W = ( * IP++ ) ).icode != (icode_f) 0 ) )
	{	/* W = ( * IP++ ); */
#if defined DBG_TRACE_NEXT
if ( tracing )
{
	DUMP_W( "NEXT" );
	dumpStack();
	fprintf( standardError, " IP:{%p} callee:{%p:%s}\n", 
		  IP, localW.bytep, localW.definitionp->nameLink.chString + 1 );
}
#endif
		( * ( * W.definitionp ).codeLink.icode )();
	}
#if defined DBG_TRACE_NEXT
if ( tracing )
{	fprintf( standardError, "exiting list:{%p:%s} with exception code {%lX}", 
		  localW.bytep, localW.definitionp->nameLink.chString + 1, (unsigned long) sysSIG.integer );
	dumpStack();
	fputc( '\n', standardError );
}
#endif
	if ( sysSIG.integer > ICODE_LIST_CONTINUE )
	{	--sysSIG.integer;
	}
}
/* Another possibility would be to return a continuation flag and pass the object parameter:
void XNEXT (void)
{	natural_t flag;
	do
	{	W = ( * IP++ );
		flag = ( * ( * W.definitionp ).codeLink.icode )( W );
	} while ( ( sysSIG.integer = flag ) == 0 )
	if ( ( flag > 0 ) && ( flag < INTERPRETER_NEST_LIMIT ) )
	{	--flag;	/ * or something like this, negative for errors, positive to terminate nested interp loops n levels.
	}
	return flag;
}
*/

/* 
01000 *
01010 * nest into icode list
01020 * call by JSR
01030 XCOL	LDX ,S	list ptr
01040 	STY ,S	nest old
01050 	TFR X,Y	interpret
01060 	NEXT
** In my 6809 implementation, 
** the first thing in the parameter field was machine code 
** to subroutine jump to the machine-level characteristic code.
** The result was that the top address on RP pointed to the parameter field.
** That won't work in C (and wasn't really a great idea, after all).
** So I must do it another way.
** So, I could put the parameter field pointer a little deeper in the stack
** (and get rid of W) by having the inner interpreter call the characteristic 
** with the address of the parameter field as a C parameter,
** or I could be fig-conventional and store the address of the parameter field in W.
** I will be conventional, to keep from hiding things.
** See XNEXT for more.
*/
/*
DOCOL   ( *** IP )      jsr <XCOL (see bif.m, bifdp.a)
        Characteristic of a colon (:) definition.  Begins execution of a
        high-level definition, i. e., nests the definition and begins
        processing icodes.  Mechanically, it pushes the IP (Y register)
        and loads the Parameter Field Address of the definition which
        called it into the IP.
*/
void XCOL(void)
{	cell_u * IPsave = IP;
	/* ( * --RP ).cellp = IP */
	IP = &( ( * W.definitionp ).parameterLink[ 0 ] );
	XNEXT();
	IP = IPsave;
}
/* 01070 * */

/*
01080 * PUSH FOLLOWING WORD
01090 * call by JSR for CONSTANT
01100 XCON	LDD [,S++]
01110 	PSHU D
01120 	NEXT
*/
/*
DOCON   ( --- n )       jsr <XCON (bif.m, bifdp.a)
        Characteristic of a CONSTANT.  A CONSTANT simply loads its value
        from its parameter field and pushes it on the stack.
*/
void XCON(void)
{	( * --SP ) = ( * W.definitionp ).parameterLink[ 0 ];
}
/* 01130 * */

/*
01140 * push following address
01150 * call by JSR for VARIABLE address
01160 XVAR	PULS D
01170 	PSHU D
01180 	NEXT
01190 * (same code as R>, but in DP)
*/
/*
DOVAR   ( --- vadr )    jsr <XVAR (bif.m, bifdp.a)
        Characteristic of a VARIABLE.  A VARIABLE pushes its PFA address
        on the stack.  The parameter field of a VARIABLE is the actual
        allocation of the variable, so that pushing its address allows
        its contents to be @ed (fetched).  Ordinary arrays and strings
        that do not subscript themselves may be allocated by defining a
        variable and immediately ALLOTting the remaining space.
        VARIABLES are global to all users, and thus should have been
        hidden in resource monitors, but aren't.
*/
void XVAR(void)
{	( * --SP ).cellp = &( ( * W.definitionp ).parameterLink[ 0 ] );
}
/* 01200 * */

/*
01210 * push address of USER variable
01215 * call by JSR
01220 XUSER	LDB [,S++]	one byte unsigned offset
01225 	CLRA
01230 	ADDD <UP
01240 	PSHU D
01250 	NEXT
** The 6809 source was optimized to straight byte-addressing, byte-packed,
** but the C source will be optimized to cell_u boundaries where it makes sense.
*/
/*
DOUSER  ( --- vadr )    jsr <XUSER (bif.m, bifdp.a)
        Characteristic of a per-USER variable. USER variables are
        similiar to VARIABLEs, but are allocated (by hand!) in the
        per-user table.  A USER variable's parameter field contains its
        offset in the per-user table.
*/
void XUSER(void)
{	( * --SP ).bytep = UP.bytep + ( * W.definitionp ).parameterLink[ 0 ].integer;
}
/* 01260 * */

/*
01300 * push address of LINEAR array entry
01305 * call by JSR
01310 X1ARR	PULS X
01320 	LDD ,U index
01330 	SUBD ,X base
01340 	CMPD 2,X width
01350 	BLO X1ARRS
01360 	LDD #$0C
01365 	PSHU D
01370 	JMP ERROR
01380 X1ARRS	PSHS A
01390 	LDA 4,X size
01400 	MUL
01410 	STD ,U
01420 	PULS A ms byte
01430 	LDB 4,X
01440 	MUL
01450 	TFR B,A
01460 	ADDA ,U
01470 	LDB 1,U
01480 	LEAX 5,X past header
01490 	LEAX D,X
01500 	STX ,U
01510 	NEXT
01520 * LINEAR array header format:
01530 *	FDB BASE (lowest index)
01540 *	FDB WIDTH (# of elements)
01550 *	FCB SIZE (< 256 bytes / element)
01560 *	RMB SIZE*WIDTH (data)
*/
/* untested, 2009.09.14 */
/*
DO1ARR  ( index --- eadr )      jsr <X1ARR (bif.m, bifdp.a)
        Characteristic of a linear array. Linear arrays take the top
        word on the stack as an index to the array, and return the
        address of the element indexed.  So this routine subtracts the
        base index of the array, limit checks the result, then
        multiplies by the size of the array elements.  If the index is
        out of bounds, it returns a NULL pointer (0).  At some point I
        intended to implement multi-dimensional arrays in a similar
        manner, but I haven't.  It would be a quick and interesting
        project for anyone interested.
*/
void X1ARR(void)
{	cell_u * base = &( ( * W.definitionp ).parameterLink[ 0 ] );
	snatural_t index = ( * SP ).sinteger;
	index -= ( base[ 0 ] ).sinteger;	/* Adjust by lower limit. */
	if ( index < 0 || index > base[ 1 ].sinteger )	/* Check bounds. */
	{	mERROR( ARRAY_REFERENCE_OUT_OF_BOUNDS );
	}
	else
	{	byte_p arraybase = (byte_p) &( base[ 3 ] );	/* ergo, LINEARRAY_DATAOFFSET, which should not be externally visible, anyway. */
		( * SP ).bytep = &( arraybase[ index * base[ 2 ].integer ] );
	}
}
/* 01700 * */

/*
01710 * push content of USER variable
01715 * call by JSR
01720 XUCON	LDX <UP
01722 	LDB [,S++]	one byte unsigned offset
01725 	CLRA
01730 	LDD D,X
01740 	PSHU D
01750 	NEXT
** The 6809 source was optimized to straight byte-addressing, byte-packed,
** but the C source will be optimized to cell_u boundaries.
*/
/*
DOUCON  ( --- n )       jsr <XUCON (bif.m, bifdp.a)
        Characteristic of a USER variable treated as a CONSTANT, i. e.,
        fetches the value stored at the specified offset in the per-user
        table.
*/
void XUCON(void)
{	( * --SP ) = * ( (cell_u *) ( UP.bytep + ( * W.definitionp ).parameterLink[ 0 ].integer ) );
}
/* 01800 * */

/*
01802 * store VOCABULARY pointer in root
01805 * call by JSR
01810 XVOC	LDX <UP
01820 	PULS D
01830 	STD UROOT,X
01840 	NEXT
*/
/* untested, 2009.09.14 */
/*
DOVOC   ( --- )         jsr <XVOC (bif.m, bifdp.a)
        Characteristic of a VOCABULARY.  A VOCABULARY stores a pointer
        to itself in the current interpretation ROOT per-USER variable.
        It contains a pointer to the definition at the root of its
        symbol table tree.  This allows the symbol table routines to
        treat the root as a leaf node.  This is also not standard FORTH!
*/
void XVOC(void)
{	( * UP.task ).searchContextRoot.cellp = (cell_u *) &( ( * W.definitionp ).parameterLink[ 0 ] );
}
/* 01850 * */

/*
01860 * indirect into icode list following
01870 * DOES> in defining definition
01880 * call by JSR
01890 XDOES	LDX ,S pfa
01900 	LDD ,X++	list ptr
01920 	PSHU X	parameters
01930 	STY ,S	nest old
01940 	TFR D,Y	interpret
01950 	NEXT
*/
/* untested, 2009.09.14 */
/*
*/
/*
        ( --- PFA )     ( *** IP )      jsr <XDOES (routine in bifdp.a)
        Characteristic of a DOES> defined word.  The characteristics of
        DOES> definitions are written in high-level icodes rather than
        machine level code. The first parameter word points to the
        high-level characteristic.  This routine's job is to push the
        IP, load the high level characteristic pointer in IP, and leave
        the address following the characteristic pointer on the stack so
        the parameter field can be accessed.
*/
void XDOES(void)
{
/* Refer to XCOL 
// and rememeber that a <BUILDS ... DOES> definition is not the sames as a colon definition.
*/ 
	cell_u * IPsave = IP;
/* XDOES	LDX ,S pfa ; get the characteristing pointer (definition elsewhere to execute) */
	definition_header_s * definitionp = W.definitionp;
/* 	LDD ,X++	list ptr ; get the definition list (i-code) pointer, point to the parameters */
	cell_u * listp = definitionp->parameterLink[ 0 ].cellp;
/* 	PSHU X	parameters ; push the pointer to the parameters on the parameter stack */
	( * --SP ).cellp = & ( definitionp->parameterLink[ 1 ] );
/* 	STY ,S	nest old ; Replace the characteristic pointer with the address to return to */
	/* IPsave = IP; did this in the initialization. */
/* 	TFR D,Y	interpret */
	IP = listp;
/* 	NEXT */
	XNEXT();
	IP = IPsave;
/* After I test the above, this should also do the trick:
	cell_u * IPsave = IP;
	cell_u * parametersp = &( ( * W.definitionp ).parameterLink[ 0 ] );
	( * --SP ).cellp = parametersp + 1;
	IP = parametersp[ 0 ].cellp;
	XNEXT();
	IP = IPsave;
//
// tested with 
// ( base limit barray name )
//
// : barray <BUILDS OVER - DUP >R SWAP , , 0 , R> ALLOT 0 , DOES> DUP >R @ - R> CELL-SIZE 3 * + + ;
on 2011.01.18:JMR
*/
}
/* 01960 * */

/*
02000 * push double constant
02010 * call by JSR
02020 XDCON	PULS X
02030 	LDD ,X++
02040 	LDX ,X++
02050 	PSHU D,X
02060 	NEXT
*/
/* untested, 2009.09.14 */
/*
        ( --- d )       jsr <XDCON (bifdp.a)
        Characteristic of a double integer constant; the parameter field
        contains two words instead of one, both of which get pushed.
*/
void XDCON(void)
{
/* Handled by the type.
// #if !defined MANUFACTURED_DOUBLE && !defined LOW_C_CELL_FIRST	// * FORTH CPU byte order for doubles.
*/
	register dblnatural_t * stack = (dblnatural_t *) ( (char *) SP );
	* --stack = * (dblnatural_t *) ( (char *) ( * W.definitionp ).parameterLink );
	SP = (cell_u *) ( (char *) stack );
/*
// #else // * defined MANUFACTURED_DOUBLE || defined LOW_C_CELL_FIRST * /
//	SP -= 2;
//	SP[ 1 ] = ( * W.definitionp ).parameterLink[ 1 ];
//	SP[ 0 ] = ( * W.definitionp ).parameterLink[ 0 ];
//#endif // * !defined MANUFACTURED_DOUBLE && !defined LOW_C_CELL_FIRST 
*/
}
/* 02070 * */

/*
40000 	ORG DPAGE+256 if code fits
40010 	SETDP 0 not yet valid
*/

/* Need this for shifting between modes?
** No, use mEXEC(), the macro-wrapped EXEC(), instead.
void callDefinition( definition_header_s * headerp )
{	cell_u saveW = W;
	W.definitionp = headerp;
	( * ( * W.definitionp ).codeLink.icode )();
	W = saveW;
}
*/


/*
01100 	FCC 'LIT'
01110 	FCB MCOMP.OR.3
01120 	FCB MFORE
01130 	FDB STORE-CFAOFF
01140 	FDB BIF+2
01150 	FDB 0 * LIST-CFAOFF
01160 	FDB 0
*/
static character_t sLIT[] = "\x3" "LIT";
definition_header_s hLIT =	
{	{ (natural_t) sLIT },
	{ MCOMP },
	{ (natural_t) &hWARM },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) LIT }
};
/*
01170 LIT	LDD ,Y++ push literal from code
01180 	PSHU D
01190 	NEXT
01200 *
*/
/*
LIT     ( --- n )                                               C
        Push the following word from the instruction stream as a
        literal, or immediate value.
*/
void LIT(void)
{	* --SP = * ( IP++ );
}


/*
01210 	FCC 'DLIT'
01220 	FCB MCOMP.OR.4
01230 	FCB MFORE
01240 	FDB LIT-CFAOFF
01250 	FDB BIF+2
01260 	FDB 0
01270 	FDB 0
*/
static character_t sDLIT[] = "\x4" "DLIT";
definition_header_s hDLIT =	
{	{ (natural_t) sDLIT },
	{ MCOMP },
	{ (natural_t) &hLIT },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) DLIT }
};
/*
01280 * push double literal from code
01290 DLIT	LDD ,Y++
01300 	LDX ,Y++
01310 	PSHU D,X
01320 	NEXT
01330 *
*/
/*
DLIT    ( --- d )                                               C
        Push a double integer literal (see LIT).
*/
void DLIT(void)
{
/* Handled by the type.
// #if !defined MANUFACTURED_DOUBLE && !defined LOW_C_CELL_FIRST	// * FORTH CPU byte order for doubles. 
*/
	dblnatural_t * stack = (dblnatural_t *) ( (char *) SP );
	dblnatural_t * list = (dblnatural_t *) ( (char *) IP );
	* --stack = * list++;	/* This is closer to optimized, anyway. */
	SP = (cell_u *) ( (char *) stack );
	IP = (cell_u *) ( (char *) list );
/* #else // * defined MANUFACTURED_DOUBLE || defined LOW_C_CELL_FIRST * /
//	SP -= 2;
//	SP[ 0 ] = IP[ 0 ];	// * Pay attention! * /
//	SP[ 1 ] = IP[ 1 ];	// * Just pushing twice from the list would invert the half-double-cells. * /
//	IP += 2;
// #endif // * !defined MANUFACTURED_DOUBLE && !defined LOW_C_CELL_FIRST 
*/
}


/*
01340 	FCC 'EXECUTE'
01350 	FCB MCOMP.OR.7
01360 	FCB MFORE
01370 	FDB DLIT-CFAOFF
01380 	FDB BIF+2
01390 	FDB 0
01400 	FDB 0
*/
static character_t sEXEC[] = "\x7" "EXECUTE";
definition_header_s hEXEC =	
{	{ (natural_t) sEXEC },
	{ MCOMP },
	{ (natural_t) &hDLIT },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) EXEC }
};
/*
01410 * EXECUTE cfa on stack
01420 EXEC	LDX ,U++
01430 	BEQ *+4
01440 	JMP ,X
01450 	LDD #9
01460 	PSHU D
01462 	JMP ERROR
01464 *
*/
/*
EXECUTE ( adr --- )                                             C
EXEC    Jump to address on stack.  Used by the "outer" interpreter to
        interactively invoke routines.  (Not compile-only in fig.)
*/
void EXEC(void)
{	cell_u target = * SP++;	/* Yeah, probably not optimal. */
	/* W = target;	// Done in the macro call. */
#if defined DBG_TRACE_NEXT
if ( isTracing() )
{	fprintf( standardError, "executing:{%p:%s}", 
		  target.bytep, target.definitionp->nameLink.chString + 1 );
	dumpState( "before" );
}
#endif
	if ( target.integer != 0 )
	{	/* ( * ( * target.definitionp ).codeLink.icode )(); */
		mCALLcell( target );
	}
	else
		mERROR( CAN_T_EXECUTE_A_NULL );
#if defined DBG_TRACE_NEXT
if ( isTracing() )
{	fprintf( standardError, "executed:{%p:%s}", 
		  target.bytep, target.definitionp->nameLink.chString + 1 );
	dumpState( "after" );
}
#endif
}


/*
01466 	FCC '1BRANCH'
01468 	FCB MCOMP.OR.7
01470 	FCB MFORE
01472 	FDB EXEC-CFAOFF
01474 	FDB BIF+2
01476 	FDB 0
01478 	FDB 0
*/
static character_t sTBR[] = "\x7" "1BRANCH";
definition_header_s hTBR =	
{	{ (natural_t) sTBR },
	{ MCOMP },
	{ (natural_t) &hEXEC },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) TBR }
};
/*
01480 TBR	LDD ,U++
01482 	BNE BRANCH
01484 	LEAY 2,Y
01486 	NEXT
01488 *
*/
/*
1BRANCH ( f --- )                                               C
TBR     BRANCH if not zero.  Not as useful as it might appear.
*/
void TBR(void)
{	if ( ( * SP++ ).integer != FALSE )
	{	BRANCH();
	}
	else
	{	IP++;
	}
}


/* tested with 
: thing0 [ ' 0BRANCH , 12 , ' HERE , ' BRANCH , 8 , ' LIT , 3567 , ] ;
: thing1 [ ' 1BRANCH , 12 , ' HERE , ' BRANCH , 8 , ' LIT , 4013 , ] ;
*/
/*
01490 	FCC 'BRANCH'
01500 	FCB MCOMP.OR.6
01510 	FCB MFORE
01520 	FDB TBR-CFAOFF
01530 	FDB BIF+2
01540 	FDB 0
01550 	FDB 0
*/
static character_t sBRANCH[] = "\x6" "BRANCH";
definition_header_s hBRANCH =	
{	{ (natural_t) sBRANCH },
	{ MCOMP },
	{ (natural_t) &hTBR },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) BRANCH }
};
/*
01560 BRANCH	LDD ,Y++
01570 	LEAY D,Y
01580 	NEXT
01590 *
*/
/*
BRANCH  ( --- )                                                 C
        Add the following word from the instruction stream to the
        instruction pointer (Y++).  Causes a program branch.
*/
void BRANCH(void)
{	snatural_t offset = ( * IP++ ).sinteger;
	byte_p nextIcode = (byte_p) IP;
	nextIcode += offset;
	IP = (cell_u *) nextIcode;	/* Let the CPU worry about word boundaries. */
}


/*
01600 	FCC '0BRANCH'
01610 	FCB MCOMP.OR.7
01620 	FCB MFORE
01630 	FDB BRANCH-CFAOFF
01640 	FDB BIF+2
01650 	FDB 0
01660 	FDB 0
*/
static character_t sZBR[] = "\x7" "0BRANCH";
definition_header_s hZBR =	
{	{ (natural_t) sZBR },
	{ MCOMP },
	{ (natural_t) &hBRANCH },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) ZBR }
};
/*
01670 ZBR	LDD ,U++
01680 	BEQ BRANCH
01690 	LEAY 2,Y
01700 	NEXT
01710 *
*/
/*
0BRANCH ( f --- )                                               C
ZBR     BRANCH if flag is zero.
*/
void ZBR(void)	/* 0BRANCH is the more commonly used conditional! */
{	if ( ( * SP++ ).integer == FALSE )
	{	BRANCH();
	}
	else
	{	IP++;
	}
}


/*
01720 	FCC '(LOOP)'
01730 	FCB MCOMP.OR.6
01740 	FCB MFORE
01750 	FDB ZBR-CFAOFF
01760 	FDB BIF+2
01770 	FDB 0
01780 	FDB 0
*/
static character_t sXLOOP[] = "\x6" "(LOOP)";
definition_header_s hXLOOP =	
{	{ (natural_t) sXLOOP },
	{ MCOMP },
	{ (natural_t) &hZBR },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) XLOOP }
};
/*
01790 XLOOP	LDD #1
01800 	ADDD ,S
01810 	STD ,S
01820 	SUBD 2,S
01830 	BLT BRANCH
01840 XLOOPN	LEAY 2,Y
01850 	LEAS 4,S
01860 	NEXT
01870 *
*/
/*
(LOOP)  ( --- )         ( limit index *** limit index+1)        C
XLOOP                   ( limit index *** )
        Counting loop primitive.  The counter and limit are the top two
        words on the return stack.  If the updated index/counter does
        not exceed the limit, a branch occurs.  If it does, the branch
        does not occur, and the index and limit are dropped from the
        return stack.
*/
void XLOOP(void)
{	RP[ 0 ].integer += 1;
	/* The original, not smart loop.
	// Should be good for 32 bit signed.
	// But I want to avoid trouble with the sign of pointers:
	*/
	{	snatural_t difference = RP[ 1 ].integer - RP[ 0 ].integer;
		if ( difference > 0 )
		{	BRANCH();
		}
		else
		{	++IP;
			RP += 2;
		}
	}
}


/*
01880 	FCC '(+LOOP)'
01890 	FCB MCOMP.OR.7
01900 	FCB MFORE
01910 	FDB XLOOP-CFAOFF
01920 	FDB BIF+2
01930 	FDB 0
01940 	FDB 0
*/
static character_t sXPLOOP[] = "\x7" "(+LOOP)";
definition_header_s hXPLOOP =	
{	{ (natural_t) sXPLOOP },
	{ MCOMP },
	{ (natural_t) &hXLOOP },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) XPLOOP }
};
/*
01950 XPLOOP	LDD ,U++ inc val
01960 	BPL XLOOP+3
01970 	ADDD ,S
01980 	STD ,S
01990 	SUBD 2,S
02000 	BGT BRANCH
02010 	BRA XLOOPN
02020 *
*/
/*
(+LOOP) ( n --- )       ( limit index *** limit index+n )       C
XPLOOP                  ( limit index *** )
        Loop with a variable increment.  Terminates when the index
        crosses the boundary from one below the limit to the limit.  A
        positive n will cause termination if the result index equals the
        limit.  A negative n must cause the index to become less than
        the limit to cause loop termination.
*/
void XPLOOP(void)
{	snatural_t increment = ( * SP++ ).sinteger;
	RP[ 0 ].sinteger += increment;	/* I want these to be unsigned, but C will never see unsigned < 0. */
	/* The original, not smart +loop. 
	// Should be good for 32 bit signed.
	// But I want to avoid trouble with the sign of pointers:
	*/
	{	snatural_t difference = RP[ 1 ].integer - RP[ 0 ].integer;
		if ( ( ( increment > 0 ) && ( difference > 0 ) ) /* How did I get these backwards the first time? */
			 || ( ( increment < 0 ) && ( difference <= 0 ) ) )	/* Don't forget that the boundary is below. */
		{	BRANCH();	/* Conflate these with a macro, or just not be slavish? */
		}
		else
		{	++IP;
			RP += 2;
		}
	}
}

/*
02030 	FCC '(DO)'
02040 	FCB 4
02050 	FCB MFORE
02060 	FDB XPLOOP-CFAOFF
02070 	FDB BIF+2
02080 	FDB 0
02090 	FDB 0
*/
static character_t sXDO[] = "\x4" "(DO)";
definition_header_s hXDO =	
{	{ (natural_t) sXDO },
	{ 0 },
	{ (natural_t) &hXPLOOP },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) XDO }
};
/*
02100 XDO	PULU D,X
02110 	PSHS D,X
02120 	NEXT
02130 *
*/
/*
(DO)    ( limit index --- )     ( *** limit index )
XDO     Move the loop parameters to the return stack.  Synonym for D>R.
*/
void XDO(void)
{	RP -= 2;	/* Take care not to invert. */
	RP[ 0 ] = SP[ 0 ];
	RP[ 1 ] = SP[ 1 ];
	SP += 2;
}


/*
04250 	FCC '(;CODE)'
04260 	FCB MCOMP.OR.7
04270 	FCB MFORE
04280 	FDB CR-CFAOFF
04290 	FDB BIF+2
04300 	FDB 0
04310 	FDB 0
*/
static character_t sXSCODE[] = "\x7" "(;CODE)";
definition_header_s hXSCODE =	
{	{ (natural_t) sXSCODE },
	{ MCOMP },
	{ (natural_t) &hXDO },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) XSCODE }
};
/*
04320 XSCODE	LDX <UP
04330 	LDX UCURR,X being defined
04340 	LEAX CFAOFF,X
04350 	LDA AJSR,PCR
04360 	STA ,X+
04370 	STY ,X++ code address
04380 	PULS Y un-nest
04390 	NEXT
04400 * parameters are 3 bytes beyond CFA
04410 AJSR	JSR >0 to be compiled
04420 *
*/
/* Nothing we can do about AJSR at this point, or, at least, nothing we want to do about it in C. 
// *** Probably ought to flag this as unavailable.
*/
/*
(;CODE) ( --- )         ( IP *** )                              C
XSCODE  Compile the latest symbol as a reference to a ;CODE definition;
        overwrite the first three (3!) bytes of the code field of the
        symbol found by LATEST with a jump to the low-level
        characteristic code provided in the defining definition, and pop
        IP.  The machine-level code which follows (;CODE) in the
        instruction stream is not executed by the defining symbol, but
        becomes the characteristic of the defined symbol.  This is the
        usual way to generate the characteristics of VARIABLEs,
        CONSTANTs, etc., when FORTH compiles itself.  BIF, however, was
        hand-optimized to take advantage of direct-page jumps.  So its
        pre-compiled defining symbols with low-level characteristics
        look different from those compiled by BIF, having two bytes in
        their code fields instead of three.
*/
void XSCODE(void)
{	/* Get the parameter[ 1 ] from executing definition and store it in the cfa -- parameter[ 0 ] of the latest. 
	** Hmm. Or should it store the address of said parameter in the code field?
	*/
	definition_header_s * latest = UP.task->lastDefined.definitionp;
	/* latest->parameterLink[ 0 ] = ( W.definitionp )->parameterLink[ 1 ];	/ * Must already be allocated. */
	latest->codeLink.cellp = ( W.definitionp )->parameterLink;
	SEMIS();	/* Don't want it to try to execute the machine code as a list of i-codes. */
	/* I still don't think this is quite correct. */
}


/*
00010 * Include file for BIF
00020 * BIF Copyright 1989 Joel Matthew Rees (see BIF/ASM)
00030 *
01000 	FCC 'MOVE'
01010 	FCB 4
01020 	FCB MFORE
01030 	FDB PREF-CFAOFF
01040 	FDB BIF+2
01050 	FDB MINUS-CFAOFF
01060 	FDB NFA-CFAOFF
*/
static character_t sMOVE[] = "\x4" "MOVE";
definition_header_s hMOVE =
{	{ (natural_t) sMOVE },
	{ 0 },
	{ (natural_t) &hXSCODE },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hMINUS },
	{ (natural_t) &hNFA },
	{ (natural_t) MOVE }
};
/*
01070 MOVE	LDD ,U++ count=0?
01080 	BEQ MOVEX
01090 	PSHS Y,U
01100 	LDY ,U
01110 	LDU 2,U
01120 MOVELP	PULU X
01130 	STX ,Y++
01140 	SUBD #1
01150 	BNE MOVELP
01160 	PULS Y,U
01170 MOVEX	LEAU 4,U
01180 	NEXT
01190 *
*/
/*
MOVE    ( source target count --- )
        Copy/move count words from source to target.  Moves ascending
        addresses, so that overlapping only works if the source is
        above the destination.
*/
void MOVE(void)	/* Ascending, overlapped works only when source is above destination, see below. */
{	cell_u * source = SP[ 2 ].cellp;
	cell_u * target = SP[ 1 ].cellp;
	natural_t count = SP[ 0 ].integer;
	SP += 3;
	for ( ; count > 0; --count )
	{	* target++ = * source++;
	}
}
/* About overlap, the full story is that overlopping works with ascending order 
// only when the destination start point is outside the source start point.
// That description more fully covers issues of address wrap-around.
// (Not usually a problem, since most computers don't have their address space entirely filled with RAM
//  and, since data at address 0 is preferably avoided when NULL (nil) is of the ( (void *) 0 ) variety.)
// So, the test would be somehting like this:
//	if ( ( target > source ) && ( target < source + count ) )
//	{	source += count;	// exceptional case, descending
//		target += count;
//		for ( ; count > 0; --count )
//		{	* --target = * --source;
//		}
//	}
//	else
//	{	for ( ; count > 0; --count )	// ascending
//		{	* target++ = * source++;
//		}
//	}
// This still doesn't properly cover the cases, however.
//
// Since it doesn't hurt to copy descending when there is no overlap,
// only the first test is really necessary when neither source nore destination wraps through address zero.
// It also works when the destination wraps through zero.
// It doesn't work when the source wraps through zero and the destination starts in the wrapped portion. 
//
// But I don't really want to test that now, and overlapping is going to be rare enough, 
// I don't want to tell myself I've solved the problem when I don't know that I have.
// Predecrement on the descending copy should be the way to handle the pointers pointing one beyond
// after the ends are calculated, but, as I say, it's not standard idiom, so I shouldn't trust my memory.
//
// If the source end is greater than the source start address (no wrap over zero),
// copy direction is almost safely determined by comparing the target start with the source start: 
//      below -> copy ascending, otherwise descending.
// If the source wraps (end is less than start), 
// copy direction is determined by comparing the target start with the source end:
//      below or equal to actual end -> copy descending, otherwise ascending.
//
// Almost.
//
// If the size of the object being copied exceeds half of memory space, 
// there is a possibility of overlap on both ends. 
// If there is overlap on both ends, we have multiple problems.
// If the object code we are running is in the same memory space (the usual case), 
// we have to move the code we are running.
// If not, we have to play strategy games to avoid overwriting things before they get copied.
//
// This exposes a hidden issue. 
// We are treating copies as if they should never fail, but they might.
// Part of the issue here is that it's impossible to test all the error input conditions in a timely manner.
// Another part of the issue is that the generalized copies are a feature of Unix systems, 
// where 0 is usually NULL. So strings should never wrap through address 0.
//
// Anyway, strings as big as half the memory space should never occur in ordinary programs, 
// and should be assumed to require their own special routines.
//
// But we don't have any way to fail gracefully on such input, 
// so we don't try to implement a universal move.
*/


/*
01200 	FCC 'CMOVE'
01210 	FCB 5
01220 	FCB MFORE
01225 	FDB MOVE-CFAOFF
01230 	FDB BIF+2
01240 	FDB CFA-CFAOFF
01250 	FDB COMP-CFAOFF
*/
static character_t sCMOVE[] = "\x5" "CMOVE";
definition_header_s hCMOVE =
{	{ (natural_t) sCMOVE },
	{ 0 },
	{ (natural_t) &hMOVE },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hCFA },
	{ (natural_t) &hCOMP },
	{ (natural_t) CMOVE }
};
/*
01260 CMOVE	LDD #0
01270 	SUBD ,U++
01280 	PSHS A,Y
01290 	PULU X,Y
01300 	BEQ CMOVEX
01310 CMOVEL	LDA ,Y+
01320 	STA ,X+
01330 	INCB
01340 	BNE CMOVEL
01350 	INC ,S
01360 	BNE CMOVEL
01370 CMOVEX	PULS A,Y
01380 	NEXT
01390 *
*/
/*
CMOVE   ( source target count --- )
        Copy/move count bytes from source to target.  Moves ascending
        addresses, so that overlapping only works if the source is
        above the destination.
*/
void CMOVE(void)	/* Ascending, see MOVE for comments on overlap. */
{	byte_t * source = SP[ 2 ].bytep;
	byte_t * target = SP[ 1 ].bytep;
	natural_t count = SP[ 0 ].integer;
	SP += 3;
	for ( ; count > 0; --count )
	{	* target++ = * source++;
	}
}



