SOAPMethod.Parameter:
Sub Parameter(name as string, Assigns value as variant)
  inputParams.value(name) = value
End Sub

SOAPMethod.Constructor:
Sub Constructor()
  initialize
End Sub

SOAPMethod.ExecuteRPC:
Private Function ExecuteRPC() As SOAPResult
  dim response as string
  
  // intialize connection
  if connection = NIL then
    connection = new httpSocket
  end
  
  // set up the post data
  connection.setRequestHeader "SOAPAction",action
  connection.setPostContent formatEnvelope(),"text/xml"
  response = connection.post(url,timeout)
  
  // handle error
  if connection.errorCode() <> 0 then
    return NIL
  end
  
  // parse the response
  return parseSOAPResult(response)
End Function

SOAPMethod.ParseSOAPResult:
Private Function ParseSOAPResult(resultData as string) As SOAPResult
  dim sr as soapResult
  dim i,j as integer
  dim xnode as xmlNode
  
  // parse soap result
  sr = new soapResult
  sr.document = new xmlDocument
  sr.document.loadXML resultData
  
  // extract envelope
  for i = 0 to sr.document.childCount-1
    xnode = sr.document.child(i)
    if xnode.localName = "Envelope" then
      sr.envelope = xnode
      exit
    end
  next
  
  // provide access to the body tag without having to go through envelope
  for i = 0 to sr.envelope.childCount-1
    xnode = sr.envelope.child(i)
    if xnode.localName = "Body" then
      sr.body = xmlElement(xnode)
      exit
    end
  next
  
  // check for faults
  for i = 0 to sr.body.childCount-1
    xnode = sr.body.child(i)
    if xnode.localName = "Fault" then// fault was found
      sr.error = true
      for j = 0  to xnode.childCount-1// extract fault code and message
        if xnode.child(j).localName = "FaultCode" then
          sr.errorCode = val(xnode.child(j).firstChild.value)
        end
        if xnode.child(j).localName = "FaultString" then
          sr.errorMessage = xnode.child(j).firstChild.value
        end
      next
      exit
    end
  next
  
  
  // return the result
  return sr
  
  
exception err as XMLException
  sr.error = true
  sr.errorCode = -1
  sr.errorMessage = err.message
  return sr
End Function

SOAPMethod.Constructor:
Sub Constructor(WSDLurl as string)
  initialize
  loadWSDLFromURL WSDLurl
End Sub

SOAPMethod.LoadWSDLFromURL:
Sub LoadWSDLFromURL(url as string)
  dim pageGrab as _soapSocket
  dim response as string
  
  // use the user defined socket if one exists
  if connection = NIL then
    pageGrab = new httpSocket
  else
    pageGrab = me.connection
  end
  
  // get the wsdl
  response = pageGrab.get(url,timeout)
  if pageGrab.errorCode <> 0 then
    raise getSOAPException("WSDL could not be loaded from URL")
  end
  
  // parse the wsdl
  wsdl = new xmlDocument
  wsdl.loadXml response
End Sub

SOAPMethod.InvokeMethod:
Private Function InvokeMethod(name as string, paramIsArray as boolean) As string
  dim sr as soapResult
  dim resultNode as xmlNode
  
  // parse wsdl if found
  if wsdl <> NIL then
    parseWSDL name,paramIsArray
  else// call function with no wsdl
    methodName = name
  end
  
  // execute the function
  sr = executeRPC()
  if sr = NIL then
    return ""
  end
  
  // find the result node
  resultNode = sr.body.firstChild.firstChild.firstChild
  if resultNode = NIL then
    resultNode = sr.body.firstChild.firstChild
    return resultNode.value
  end
  
  // return the result
  if resultNode.type = 3 then
    return resultNode.value
  else
    return resultNode.toString()
  end if
End Function

SOAPMethod.Operator_Lookup:
Function Operator_Lookup(funcName as string, ParamArray params as variant) As string
  dim i as integer
  
  if ubound(params) = -1 then// there are no parameters
    if wsdl <> NIL then// clear parameters if there is a wsdl defined
      inputParams.clear
    end
  else// parameters exist
    inputParams.clear// clear stale parameters
    for i = 0 to ubound(params)// assign new parameters
      inputParams.value(str(i)) = params(i)
    next
  end
  
  return invokeMethod(funcName,true)
End Function

SOAPMethod.wsdlFindParamNodesParent:
Private Function wsdlFindParamNodesParent(funcName as string) As xmlNode
  dim rlist as xmlNodeList
  dim inputMessage as string
  dim inputmessageNode, sequenceNode as xmlNode
  
  // first find the port input name
  rlist = wsdlQuery("definitions/portType/operation[@name='" + funcName + "']/input")
  inputMessage = xmlElement(rlist.Item(0)).GetAttribute("message")
  inputMessage = nthField(inputMessage, ":", 2)
  
  rlist = wsdlQuery("definitions/message[@name='" + inputMessage + "']")
  inputMessageNode = rlist.Item(0)
  
  if inputMessageNode.childCount = 0 then
    return NIL
  end
  
  // some wsdl services will go ahead and have our param details here
  if inputMessageNode.childCount >= 1 AND xmlElement(inputMessageNode.firstChild).getAttribute("type") <> "" then
    return inputMessageNode
  end
  
  // if there is only one <part> tag and it has an "element" attribute, then we have to find our params
  // in the <types> node
  if inputMessageNode.childCount = 1 AND xmlElement(inputMessageNode.firstChild).getAttribute("element") <> "" then
    rlist = wsdlQuery("definitions/types/schema/element[@name='" + funcName + "']/complexType/sequence")
    sequenceNode = rlist.Item(0)
    return sequenceNode
  end
  
  return NIL
End Function

SOAPMethod.wsdlFunctionExists:
Private Function wsdlFunctionExists(funcName as string) As string
  dim rlist as xmlNodeList
  dim i as integer
  
  rlist = wsdlQuery("definitions/binding/operation[@name]")
  for i = 0 to rlist.length-1
    if xmlElement(rlist.item(i)).getAttribute("name") = funcName then
      return xmlElement(rlist.item(i)).getAttribute("name")
    end
  next
  
  return ""
End Function

SOAPMethod.wsdlGetAction:
Private Function wsdlGetAction(funcName as string) As string
  dim rlist as xmlNodeList
  dim action as string
  
  rlist = wsdlQuery("definitions/binding/operation[@name='" + funcName + "']/operation")
  action = xmlElement(rlist.item(0)).getAttribute("soapAction")
  
  return action
End Function

SOAPMethod.wsdlGetAddress:
Private Function wsdlGetAddress() As string
  dim rlist as xmlNodeList
  dim tmpNode as xmlNode
  dim address as string
  
  // lets get the real url of the service
  rlist = wsdlQuery("definitions/service/port/address")
  address = xmlElement(rlist.Item(0)).getAttribute("location")
  
  return address
End Function

SOAPMethod.wsdlGetMethodNS:
Private Function wsdlGetMethodNS(funcName as string) As string
  dim rlist as xmlNodeList
  dim ns as string
  
  rlist = wsdlQuery("definitions/binding/operation[@name='" + funcName + "']/input/body")
  ns = xmlElement(rlist.item(0)).getAttribute("namespace")
  
  if ns = "" then
    ns = wsdl.documentElement.getAttribute("targetNamespace")
  end
  
  return ns
End Function

SOAPMethod.Initialize:
Private Sub Initialize()
  inputParams = new dictionary
  
  namespaces = new dictionary
  namespaces.value("SOAP") = "http://schemas.xmlsoap.org/soap/envelope/"
  namespaces.value("xsd") = "http://www.w3.org/2001/XMLSchema"
  namespaces.value("xsi") = "http://www.w3.org/2001/XMLSchema-instance"
  namespaces.value("ENC") = "http://schemas.xmlsoap.org/soap/encoding/"
  namespaces.value("si") = "http://soapinterop.org/xsd"
  
  paramTypes = new dictionary
  paramTypes.value("0") = ""// nil
  paramTypes.value("2") = "int"// integer
  paramTypes.value("5") = "double"// double
  paramTypes.value("7") = "string"// date
  paramTypes.value("8") = "string"// string
  paramTypes.value("9") = ""// object
  paramTypes.value("11") = "boolean"// boolean
  paramTypes.value("16") = "string"// color
End Sub

SOAPMethod.Invoke:
Function Invoke(name as string) As SOAPResult
  // record the method name
  methodName = name
  
  // if a wsdl exists then use its definition of the method
  if wsdl <> NIL then
    parseWSDL name,false
  end
  
  // return result
  return executeRPC()
End Function

SOAPMethod.ParseWSDL:
Private Sub ParseWSDL(name as string, paramIsArray as boolean)
  dim paramParent as xmlNode
  dim i as integer
  
  // set method name
  methodName = wsdlFunctionExists(name)
  
  // does the function exist
  if methodName = "" then
    raise getSOAPException("Method name does not exist in WSDL")
  end
  
  // extract method properties from wsdl
  url = wsdlGetAddress()
  action = wsdlGetAction(methodName)
  methodNamespace = wsdlGetMethodNS(methodName)
  paramParent = wsdlFindParamNodesParent(methodName)
  
  // validate parameter count
  if paramParent = NIL then// no parameters found
    if inputParams.count > 0 then// parameters were passed in
      raise getSOAPException("Incorrect parameters")
    end
    return// no parameters so we are done
  end
  if paramParent.childCount <> inputParams.count then// compare input params with needed params
    raise getSOAPException("Incorrect parameters")
  end
  
  // set up parameters
  for i = 0 to paramParent.childCount-1
    if paramIsArray = true then// change array index to name
      inputParams.value(xmlElement(paramParent.child(i)).getAttribute("name")) = inputParams.value(str(i))
      inputParams.remove str(i)
    else// verify parameter names
      if inputParams.hasKey(xmlElement(paramParent.child(i)).getAttribute("name")) = false then
        raise getSOAPException("Incorrect parameters")
      end
    end
  next
End Sub

SOAPMethod.GetSOAPException:
Private Function GetSOAPException(message as string) As SOAPException
  dim err as SOAPException
  
  err = new SOAPException
  err.message = message
  
  return err
End Function

SOAPMethod.FormatEnvelope:
Private Function FormatEnvelope() As string
  dim env as xmlDocument
  dim xbody,xdef as xmlNode
  dim i as integer
  
  // create document
  env = new xmlDocument
  
  // setup envelope and namespaces
  env.appendChild env.createElement(namespaces.value("SOAP"),"SOAP:Envelope")
  for i = 0 to namespaces.count-1
    env.documentElement.setAttribute "xmlns:"+namespaces.key(i),namespaces.value(namespaces.key(i))
  next
  
  // create body
  xbody = env.createElement(namespaces.value("SOAP"),"SOAP:Body")
  env.documentElement.appendChild xbody
  
  // create method
  xdef = env.createElement(methodNameSpace,methodName)
  xbody.appendChild xdef
  
  // set parameters
  for i = 0 to inputParams.count-1
    xdef.appendChild getParameterNode(env,i)
  next
  
  // return envelope string
  return env.toString()
End Function

SOAPMethod.GetParameterNode:
Private Function GetParameterNode(env as xmlDocument, idx as integer) As xmlNode
  dim xnode as xmlElement
  dim pname,ptype,pval as string
  
  // extract parameter name and type
  pname = inputParams.key(idx)
  ptype = "xsd:"+ paramTypes.value(str(inputParams.value(pname).type))
  
  // format parameter types
  select case inputParams.value(pname).type
  case 11// boolean
    if inputParams.value(pname).booleanValue = true then
      pval = "1"
    else
      pval = "0"
    end
    
  case 9// object
    if inputParams.value(pname).objectValue isa XMLNode then
      xnode = xmlElement(inputParams.value(pname).objectValue)
    end
    
  else
    pval = inputParams.value(pname).stringValue
  end
  
  // create node from parameter data if parameter value wasn't already a node
  if xnode = NIL then
    xnode = env.createElement(pname)
    xnode.setAttribute namespaces.value("xsi"),"xsi:type",ptype
    xnode.appendChild env.createTextNode(pval)
  end
  
  // return parameter node
  return xnode
End Function

SOAPMethod.UseSocket:
Sub UseSocket(SOAPsocket as _SOAPSocket)
  me.connection = soapSocket
End Sub

SOAPMethod.ClearParameters:
Sub ClearParameters()
  inputParams.clear
End Sub

SOAPMethod.wsdlQuery:
Private Function wsdlQuery(query as string) As xmlNodeList
  dim i,x as integer
  dim output,qstr,tmp,attr as string
  
  // Format the XQL query to support namespaces.  We these queries after the fact
  // in order to keep them more human readable in code for debugging.
  for i = 1 to countFields(query,"/")
    tmp = nthField(query,"/",i)
    
    x = instr(tmp,"[")
    if x > 0 then
      attr = mid(tmp,x)
      tmp = nthField(tmp,"[",1)
    end
    
    qstr = "*[local-name()='"+ tmp +"']"
    if x > 0 then
      qstr = qstr + attr
    end
    
    output = output +"/"+ qstr
  next
  output = mid(output,2)
  
  // run the query
  return wsdl.xql(output)
End Function

App.quitting:
Sub quitting()
  isQuitting=true
  connectMenu.checked=false
  if ListenerSocket.isListening then
    ListenerSocket.stopListening()
  end
  TransferTimer.mode=0
  NetworkTimer.mode=0
  QuitTimer=new OriginalQuitTimer
  QuitTimer.period=250
  QuitTimer.mode=2
End Sub

App.Close:
Sub Close()
  writeConfig()
  writeHistory()
  OC_Write()
End Sub

App.Open:
Sub Open()
  '2get - Peer to peer file sharing
  '
  'This program is free software; you can redistribute it and/or modify
  'it under the terms of the GNU General Public License as published by
  'the Free Software Foundation; either version 2 of the License,or
  '(at your option) any later version.
  '
  'This program is distributed in the hope that it will be useful,
  'but WITHOUT ANY WARRANTY; without even the implied warranty of
  'MERCHANTABILITY or FITNESS for A PARTICULAR PURPOSE. See the
  'GNU General Public License for more details.
  '
  'You should have received a copy of the GNU General Public License
  'along with this program; if not,write to the Free Software
  'Foundation,Inc.,59 Temple Place,Suite 330,Boston,MA 02111-1307 USA
  
  initProperties()
  initPicture()
  initSound()
  initCursor()
  getConfig()
  getHistory()
  OL_Get()
  OL_SharedBuffer()
  OC_Get()
  
  MainWindow.show()
  #if targetCarbon
    DrawerWindow.showWithin(MainWindow,iFacing)
  #else
    DrawerWindow.show()
  #endif
  
  ListenerSocket=new OriginalListenerSocket
  ListenerSocket.minimumSocketsAvailable=10
  ListenerSocket.maximumSocketsConnected=256
  ListenerSocket.port=iPort
  if iPort<>0 then
    ListenerSocket.listen()
  end
  TransferThread=new OriginalTransferThread
  TransferTimer=new OriginalTransferTimer
  TransferTimer.period=iCPUCycleTime
  TransferTimer.mode=2
  NetworkTimer=new OriginalNetworkTimer
  NetworkTimer.period=250
  NetworkTimer.mode=0
  
  availableMenu.checked=true
  userMenu.text=s_user_name_+chrb(32)+iNick
  sharesMenu.text=s_shared_+chrb(32)+str(ubound(Library))
  speedMenu.text=s_speed_+chrb(32)+getShortSpeed(iSpeed)
  clientMenu.text=s_client_+chrb(32)+iClient
  portMenu.text=s_port_+chrb(32)+str(iPort)
  
  if ubound(Library)=0 then
    getSound(rs_message)
    msgbox about_sharing
  end
  if iUpdateLibrary then
    OL_Update()
  end
  if iAskNapList then
    NS_Prepare()
  end
  if iAutoConnect then
    NetworkTimer.mode=2
    connectMenu.checked=true
  end
End Sub

App.EnableMenuItems:
Sub EnableMenuItems()
  dim i,j as integer
  if isQuitting=false then
    aboutMenu.enabled=true
    updateMenu.enabled=true
    mainMenu.enabled=true
    connectMenu.enabled=(MainWindow.NetworkListBox.listCount<>0)
    libraryMenu.enabled=not isIndexing
    napListMenu.enabled=(ubound(NapListSocket)=0)
    quitMenu.enabled=true
    preferencesMenu.enabled=true
    availableMenu.enabled=true
    awayMenu.enabled=true
    muteMenu.enabled=true
    previousMenu.enabled=true
    nextMenu.enabled=true
    bringMenu.enabled=true
    j=windowCount-1
    for i=0 to j
      if window(i) isa OriginalWindow and OriginalWindow(window(i)).menu<>nil then
        OriginalWindow(window(i)).menu.enabled=true
      end
    next
  else
    aboutMenu.enabled=false
    updateMenu.enabled=false
    mainMenu.enabled=false
    connectMenu.enabled=false
    libraryMenu.enabled=false
    napListMenu.enabled=false
    quitMenu.enabled=false
    preferencesMenu.enabled=false
    availableMenu.enabled=false
    awayMenu.enabled=false
    muteMenu.enabled=false
    previousMenu.enabled=false
    nextMenu.enabled=false
    bringMenu.enabled=false
    j=windowCount-1
    for i=0 to j
      if window(i) isa OriginalWindow and OriginalWindow(window(i)).menu<>nil then
        OriginalWindow(window(i)).menu.enabled=false
      end
    next
  end
exception
End Sub

App.Deactivate:
Sub Deactivate()
  me.active=false
End Sub

App.Activate:
Sub Activate()
  if me.isOpened=false then
    me.isOpened=true
    #if targetCarbon
      if getToolbar(c_network_window) then
        MainWindow.MA_ToggleToolbar()
      end
      getWindow(MainWindow,c_network_window)
    #else
      DrawerWindow.MenuListBox.listIndex=0
    #endif
  end
  
  me.active=true
  notifyStop()
End Sub

OriginalCriticalSection.enter:
Function enter() As Boolean
  if me.isLocked=false then
    me.locks=me.locks+1
    return true
  else
    return false
  end
End Function

OriginalCriticalSection.leave:
Sub leave()
  me.locks=me.locks-1
End Sub

OriginalCriticalSection.signal:
Function signal() As Boolean
  if me.isLocked=false and me.locks=0 then
    me.isLocked=true
    return true
  else
    return false
  end
End Function

OriginalCriticalSection.release:
Sub release()
  me.isLocked=false
End Sub

OriginalEditField.appendText:
Sub appendText(inWhat as string,inColor as color,inBold as boolean)
  #if targetCarbon
    me.selstart=len(me.text)
  #else
    me.selstart=lenb(me.text)
  #endif
  me.selTextColor=inColor
  me.selBold=false
  me.seltextSize=iTextSize
  me.seltextFont=iTextFont
  me.seltext=getDefine(inWhat)
End Sub

OriginalEditField.replaceText:
Sub replaceText(inWhat as string)
  me.text=getDefine(inWhat)
  me.selstart=0
  #if targetCarbon
    me.sellength=len(inWhat)
  #else
    me.sellength=lenb(inWhat)
  #endif
End Sub

OriginalEditField.appendText:
Sub appendText(inWhat as string)
  #if targetCarbon
    me.selstart=len(me.text)
  #else
    me.selstart=lenb(me.text)
  #endif
  me.seltextFont=me.textFont
  me.seltext=getDefine(inWhat)
End Sub

OriginalEditField.appendText:
Sub appendText(inWhat as string,inColor as color)
  #if targetCarbon
    me.selstart=len(me.text)
  #else
    me.selstart=lenb(me.text)
  #endif
  me.selTextColor=inColor
  me.selBold=false
  me.seltextSize=iTextSize
  me.seltextFont=iTextFont
  me.seltext=getDefine(inWhat)
End Sub

OriginalEditField.appendMessage:
Sub appendMessage(inNick as string,inNickColor as integer,inMessage as string,inMessageColor as integer)
  me.appendText(getTimeStamp(),iColors(co_timestamp))
  me.appendText(chrb(91)+inNick+chrb(93)+chrb(32),iColors(inNickColor))
  me.appendText(inMessage+chrb(13),iColors(inMessageColor))
End Sub

OriginalEditField.appendMessage:
Sub appendMessage(inMessage as string,inMessageColor as integer)
  me.appendText(getTimeStamp(),iColors(co_timestamp))
  me.appendText(inMessage+chrb(13),iColors(inMessageColor))
End Sub

OriginalEditField.appendMessage:
Sub appendMessage(inNick as string,inNickColor as integer,inMessage as string,inMessageColor as integer,inBold as boolean)
  me.appendText(getTimeStamp(),iColors(co_timestamp))
  me.appendText(chrb(91)+inNick+chrb(93)+chrb(32),iColors(inNickColor))
  me.appendText(inMessage+chrb(13),iColors(inMessageColor),inBold)
End Sub

OriginalEditField.scrollBottom:
Sub scrollBottom()
  #if targetCarbon
    me.selStart=len(me.text)
  #else
    me.selStart=lenb(me.text)
  #endif
  me.selText=""
End Sub

OriginalEditField.insertText:
Sub insertText(inWhat as string)
  me.seltextFont=me.textFont
  me.seltext=getDefine(inWhat)
End Sub

OriginalEditField.MouseDown:
Function MouseDown(X As Integer, Y As Integer) As Boolean
  if isCMMClick then
    CMMClicked()
    return true
  end
End Function

OriginalEditField.KeyDown:
Function KeyDown(Key As String) As Boolean
  if ascb(key)=27 then
    me.text=""
  end
  return KeyDown(Key)
End Function

OriginalEditField.EnableMenuItems:
Sub EnableMenuItems()
  editClear.enabled=(lenb(me.text)<>0 and me.readOnly=false)
  editSelectAll.enabled=(lenb(me.text)<>0)
End Sub

OriginalLibraryThread.getItem:
Protected Sub getItem(inFi as FolderItem)
  dim i,j,aArray,oldArray,Size as integer
  dim ShortName,FilePath,FullPath,Extension as string
  if isQuitting or inFi=nil or inFi.isReadable=false or inFi.visible=false then
    return
    
  elseif inFi.Directory then
    if strcomp(inFi.name,s_incomplete,0)=0 then
      return
    end
    j=inFi.Count
    for i=1 to j
      me.getItem(inFi.Item(i))
    next
    
  elseif inFi.length<>0 then
    ShortName=getString(inFi.name)
    if ascb(leftb(ShortName,1))=46 then
      return
    end
    if isWinMX then
      ShortName=getWinMXString(ShortName)
    end
    FullPath=getFolderItem2Path(inFi)
    if instrb(1,FullPath,chrb(34))<>0 then
      return
    end
    FilePath=getDefine(replaceallb(FullPath,chrb(58),chrb(92)))
    FilePath=midb(FilePath,me.rootPathLength,lenb(FilePath)-me.rootPathLength-lenb(ShortName)+1)
    if isWinMX then
      FilePath=getWinMXString(FilePath)
    end
    Size=inFi.Length
    Extension=rightb(ShortName,4)
    
    oldArray=OL_SearchOldFileName2Array(FilePath+ShortName)
    if oldArray=c_error then
      if (strcomp(Extension,".mp3",0)=0 and me.getMP3(inFi)=0) or (strcomp(Extension,".m4a",0)=0 and me.getAAC(inFi)=0) then
      else
        me.Bitrate=24
        me.SampleRate=16000
        me.Time=600
      end
      Library.append new OriginalLibrary
      aArray=ubound(Library)
      if Library(aArray)<>nil then
        Library(aArray).fullPath=FullPath
        Library(aArray).filePath=FilePath
        Library(aArray).shortName=ShortName
        Library(aArray).size=getDefine(format(Size,"#"))
        Library(aArray).shortSize=getShortSize(Size)
        Library(aArray).bitrate=getDefine(str(me.Bitrate))
        Library(aArray).sampleRate=getDefine(str(me.SampleRate))
        Library(aArray).time=getDefine(format(me.Time,"#"))
        Library(aArray).md5=getRandomString(true)+getDefine(format(Size,"\-#"))
        Library(aArray).byte=lenb(Library(aArray).shortName+Library(aArray).md5+Library(aArray).size+Library(aArray).bitrate+Library(aArray).sampleRate+Library(aArray).time)+8
        sharesMenu.text=s_shared_+chrb(32)+str(aArray)
        OS_BroadcastShareFile(Library(aArray))
      end
      
    elseif Size<>val(OldLibrary(oldArray).size) then
      if (strcomp(Extension,".mp3",0)=0 and me.getMP3(inFi)=0) or (strcomp(Extension,".m4a",0)=0 and me.getAAC(inFi)=0) then
      else
        me.Bitrate=24
        me.SampleRate=16000
        me.Time=600
      end
      Library.append new OriginalLibrary
      aArray=ubound(Library)
      if Library(aArray)<>nil then
        Library(aArray).fullPath=FullPath
        Library(aArray).filePath=FilePath
        Library(aArray).shortName=ShortName
        Library(aArray).size=getDefine(format(Size,"#"))
        Library(aArray).shortSize=getShortSize(Size)
        Library(aArray).bitrate=getDefine(str(me.Bitrate))
        Library(aArray).sampleRate=getDefine(str(me.SampleRate))
        Library(aArray).time=getDefine(format(me.Time,"#"))
        Library(aArray).md5=getRandomString(true)+getDefine(format(Size,"\-#"))
        Library(aArray).byte=lenb(Library(aArray).shortName+Library(aArray).md5+Library(aArray).size+Library(aArray).bitrate+Library(aArray).sampleRate+Library(aArray).time)+8
        sharesMenu.text=s_shared_+chrb(32)+str(aArray)
        OS_BroadcastUnShareFile(OldLibrary(oldArray))
        OS_BroadcastShareFile(Library(aArray))
      end
      OldLibrary.remove oldArray
      
    else
      Library.append new OriginalLibrary
      aArray=ubound(Library)
      if Library(aArray)<>nil then
        Library(aArray)=OldLibrary(oldArray)
        sharesMenu.text=s_shared_+chrb(32)+str(aArray)
      end
      OldLibrary.remove oldArray
    end
    
  end
End Sub

OriginalLibraryThread.getMP3:
Protected Function getMP3(inFi as FolderItem) As integer
  dim aByte,aResult,aPosition as integer
  me.source=inFi.OpenAsBinaryFile(false)
  me.source.littleEndian=false
  
  if strcomp(me.source.Read(3),"ID3",0)=0 then
    select case me.source.readByte()
    case 2'id3v2.2 found
      me.source.position=5
      aResult=me.read_ID3v22()
    case 3'id3v2.3 found
      me.source.position=5
      aResult=me.read_ID3v23()
    case 4'id3v2.4 found
      me.source.position=5
      aResult=me.read_ID3v24()
    else
      aResult=4'error
    end
    
  else
    aPosition=me.read_ZeroPadding()
    if me.source.readByte()=255 then
      aByte=me.source.readByte()
      if aByte=250 or aByte=251 then
        if me.is_ID3v11() then
          me.source.position=aPosition
          aResult=me.read_ID3v11()
        else
          me.source.position=aPosition
          aResult=me.read_RowMP3()
        end
      else
        aResult=1'error
      end
    else
      aResult=1'error
    end
  end
  
  me.source.close()
  return aResult
exception
  if me.source<>nil then
    me.source.close()
  end
  aResult=1'error
  return aResult
End Function

OriginalLibraryThread.read_MP3Flame:
Protected Function read_MP3Flame() As integer
  dim i,tmp,tmpbr,tmpsr,frameLength,frameBitrate,frameSampleRate,framePadding as integer
  dim tmpBitrate,tmpSampleRate as Double
  for i=0 to 500
    frameLength=0
    if me.source.eof then
      exit
    end
    
    tmp=me.source.readByte()
    if tmp<>255 then
      return 1'error
    end
    
    tmp=me.source.readByte()
    if tmp<>251 and tmp<>250 then
      return 1'error
    end
    
    tmp=me.source.readByte()
    tmpbr=8*getBit(tmp,7)+4*getBit(tmp,6)+2*getBit(tmp,5)+1*getBit(tmp,4)
    select case tmpbr
    case 1
      frameBitrate=32000
    case 2
      frameBitrate=40000
    case 3
      frameBitrate=48000
    case 4
      frameBitrate=56000
    case 5
      frameBitrate=64000
    case 6
      frameBitrate=80000
    case 7
      frameBitrate=96000
    case 8
      frameBitrate=112000
    case 9
      frameBitrate=128000
    case 10
      frameBitrate=160000
    case 11
      frameBitrate=192000
    case 12
      frameBitrate=224000
    case 13
      frameBitrate=256000
    case 14
      frameBitrate=320000
    end
    
    tmpsr=2*getBit(tmp,3)+1*getBit(tmp,2)
    select case tmpsr
    case 0
      frameSampleRate=44100
    case 1
      frameSampleRate=48000
    case 2
      frameSampleRate=32000
    end
    
    framePadding=getBit(tmp,1)
    
    frameLength=144*frameBitrate\frameSampleRate+framePadding
    tmpBitrate=tmpBitrate+frameBitrate
    tmpSampleRate=tmpSampleRate+frameSampleRate
    me.source.position=me.source.position+frameLength-3
  next
  
  me.Bitrate=tmpBitrate\i\1000
  me.SampleRate=tmpSampleRate\i
End Function

OriginalLibraryThread.read_RowMP3:
Protected Function read_RowMP3() As integer
  dim aResult as integer
  aResult=me.read_MP3Flame()
  if aResult=0 then
    me.time=8/1000*(me.source.length)/me.Bitrate
  end
  return aResult
End Function

OriginalLibraryThread.read_AACHeader:
Protected Function read_AACHeader() As integer
  dim tagSize,tagSize2,aPosition,aPosition2,timeScale,timeLength as integer
  dim fileDate as Date
  
  fileDate=new Date
  tagSize=me.source.readLong()
  me.source.position=tagSize
  
  aPosition2=me.source.position
  tagSize=me.source.readLong()
  tagSize2=tagSize
  if strcomp(me.source.Read(4),"moov",0)<>0 then
    return 2
  end
  
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"mvhd",0)<>0 then
    return 2
  end
  
  me.source.position=me.source.position+4
  fileDate.TotalSeconds=me.source.readLong()
  fileDate.TotalSeconds=me.source.readLong()
  timeScale=me.source.readLong()
  timeLength=me.source.readLong()
  me.time=timeLength*1000\timeScale\1000
  
  me.source.position=aPosition+tagSize
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"trak",0)<>0 then
    return 2
  end
  
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"tkhd",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"mdia",0)<>0 then
    return 2
  end
  
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"mdhd",0)<>0 then
    return 2
  end
  
  me.source.position=me.source.position+12
  me.SampleRate=me.source.readLong()
  
  me.source.position=aPosition+tagSize
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"hdlr",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"minf",0)<>0 then
    return 2
  end
  
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"smhd",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"dinf",0)<>0 then
    return 2
  end
  
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"dref",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"stbl",0)<>0 then
    return 2
  end
  
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"stsd",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"stts",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"stsc",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"stsz",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"stco",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"udta",0)<>0 then
    return 2
  end
  
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"meta",0)<>0 then
    return 2
  end
  
  me.source.position=me.source.position+4
  aPosition=me.source.position
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"hdlr",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition+tagSize
  tagSize=me.source.readLong()
  if strcomp(me.source.Read(4),"ilst",0)<>0 then
    return 2
  end
  
  me.source.position=aPosition2+tagSize2
  tagSize=me.source.readLong()
  while me.source.eof=false and strcomp(me.source.Read(4),"mdat",0)<>0
    me.source.position=me.source.position+tagSize-8
    tagSize=me.source.readLong()
  wend
  
  if me.source.eof then
    return 2
  end
  
  me.Bitrate=round(tagSize*8/timeLength*timeScale/1000)
End Function

OriginalLibraryThread.read_ID3v11:
Protected Function read_ID3v11() As integer
  dim aResult as integer
  aResult=me.read_MP3Flame()
  me.time=8/1000*(me.source.length-128)/me.Bitrate
  return aResult
End Function

OriginalLibraryThread.read_ID3v22:
Protected Function read_ID3v22() As integer
  dim aPosition,id3v2Size,id3v2End,aResult,aByte as integer
  if getBit(me.source.readByte(),7)=1 then
    return 7'sync bit flag
  end
  'now tag size
  id3v2Size=getTagSize(me.source.readLong())
  'check if size is correct
  aPosition=me.source.position
  me.source.position=me.source.position+id3v2Size
  id3v2End=me.source.position
  aByte=me.source.readByte()
  if aByte=0 then
    do
      aByte=me.source.readByte()
    loop until aByte<>0
    id3v2End=me.source.position-1
  end
  if aByte<>255 then
    return 2
  end
  'now read mp3 tags (get bitrate,file len,SampleRate,...)
  me.source.position=id3v2End
  aResult=me.read_MP3Flame()
  if aResult<>0 then
    return aResult
  end
  if me.is_ID3v11() then
    me.time=8/1000*(me.source.length-id3v2End-128)/me.Bitrate
  else
    me.time=8/1000*(me.source.length-id3v2End)/me.Bitrate
  end
End Function

OriginalLibraryThread.read_ID3v23:
Protected Function read_ID3v23() As integer
  dim aPosition,id3v2Size,id3v2End,aResult,aByte as integer
  if getBit(me.source.readByte(),7)=1 then
    return 7'sync bit flag
  end
  'now tag size
  id3v2Size=getTagSize(me.source.readLong())
  'check if size is correct
  aPosition=me.source.position
  me.source.position=me.source.position+id3v2Size
  id3v2End=me.source.position
  aByte=me.source.readByte()
  if aByte=0 then
    do
      aByte=me.source.readByte()
    loop until aByte<>0
    id3v2End=me.source.position-1
  end
  if aByte<>255 then
    return 2
  end
  'now read mp3 tags (get bitrate,file len,SampleRate,...)
  me.source.position=id3v2End
  aResult=me.read_MP3Flame()
  if aResult<>0 then
    return aResult
  end
  if me.is_ID3v11() then
    me.time=8/1000*(me.source.length-id3v2End-128)/me.Bitrate
  else
    me.time=8/1000*(me.source.length-id3v2End)/me.Bitrate
  end
End Function

OriginalLibraryThread.read_ID3v24:
Protected Function read_ID3v24() As integer
  dim aPosition,id3v2Size,id3v2End,aResult,aByte as integer
  aByte=me.source.readByte()
  if getBit(aByte,7)=1 then
    return 7'sync bit aFlag
  end
  if getBit(aByte,4)=1 then
    return 6'footer bit aFlag
  end
  'now tag size
  id3v2Size=getTagSize(me.source.readLong())
  'check if size is correct
  aPosition=me.source.position
  me.source.position=me.source.position+id3v2Size
  id3v2End=me.source.position
  aByte=me.source.readByte()
  if aByte=0 then
    do
      aByte=me.source.readByte()
    loop until aByte<>0
    id3v2End=me.source.position-1
  end
  if aByte<>255 then
    return 2
  end
  'now read mp3 tags (get bitrate,file len,SampleRate,...)
  me.source.position=id3v2End
  aResult=me.read_MP3Flame()
  if aResult<>0 then
    return aResult
  end
  if me.is_ID3v11() then
    me.time=8/1000*(me.source.length-id3v2End-128)/me.Bitrate
  else
    me.time=8/1000*(me.source.length-id3v2End)/me.Bitrate
  end
End Function

OriginalLibraryThread.read_ZeroPadding:
Protected Function read_ZeroPadding() As integer
  me.source.position=0
  while me.source.readByte()=0 and me.source.eof=false
  wend
  if me.source.eof then
    me.source.position=me.source.position-2
  else
    me.source.position=me.source.position-1
  end
  return me.source.position
End Function

OriginalLibraryThread.is_ID3v11:
Protected Function is_ID3v11() As boolean
  me.source.position=me.source.length-128
  return strcomp(me.source.Read(3),"TAG",0)=0
End Function

OriginalLibraryThread.getAAC:
Protected Function getAAC(inFi as FolderItem) As integer
  dim aResult as integer
  me.source=inFi.OpenAsBinaryFile(false)
  me.source.littleEndian=false
  
  if me.is_AAC() then
    me.source.position=0
    aResult=me.read_AACHeader()
  else
    aResult=1'error
  end
  
  me.source.close()
  return aResult
exception
  if me.source<>nil then
    me.source.close()
  end
  aResult=1'error
  return aResult
End Function

OriginalLibraryThread.is_AAC:
Protected Function is_AAC() As boolean
  me.source.position=4
  return strcomp(me.source.Read(7),"ftypM4A",0)=0
End Function

OriginalLibraryThread.Run:
Sub Run()
  dim i,j,aArray as integer
  isIndexing=true
  
  if LibraryLock.enter() then
    j=ubound(Library)
    for i=1 to j
      OldLibrary.append new OriginalLibrary
      aArray=ubound(OldLibrary)
      if OldLibrary(aArray)<>nil then
        OldLibrary(aArray)=Library(i)
      end
    next
    LibraryLock.leave()
  end
  
  if LibraryLock.signal() then
    redim Library(0)
    sharesMenu.text=s_shared_+chrb(32)+"0"
    LibraryLock.release()
  end
  
  if LibraryLock.enter() then
    j=ubound(iUploadFolder)
    for i=1 to j
      if iUploadFolder(i).parent=nil then
        me.rootPathLength=0
      else
        me.rootPathLength=lenb(getFolderItem2Path(iUploadFolder(i).parent))+1
      end
      me.getItem(iUploadFolder(i))
    next
    if isQuitting=false then
      OL_Write()
      OL_SharedBuffer()
      OS_BroadcastShareAllFiles()
      j=ubound(OldLibrary)
      for i=1 to j
        OS_BroadcastUnShareFile(OldLibrary(i))
      next
      redim OldLibrary(0)
    end
    LibraryLock.leave()
  end
  
  isIndexing=false
End Sub

OriginalListBox.compareNumber:
Protected Sub compareNumber(inFirst as integer,inLast as integer)
  dim x,i,f,l,s,e as integer
  if me.d=-1 then
    l=inFirst
    f=inLast
  else
    f=inFirst
    l=inLast
  end
  s=f
  e=l
  x=val(me.cell(ceil((f+l)/2),me.c))
  
  do until me.d*f>me.d*l
    while val(me.cell(f,me.c))<x
      f=f+me.d
    wend
    while val(me.cell(l,me.c))>x
      l=l-me.d
    wend
    if me.d*f<=me.d*l then
      if f<>l then
        me.swapCell(f,l)
      end
      f=f+me.d
      l=l-me.d
    end
  loop
  
  if me.d*s<me.d*l then
    me.compareNumber(min(s,l),max(s,l))
  end
  if me.d*f<me.d*e then
    me.compareNumber(min(f,e),max(f,e))
  end
exception
End Sub

OriginalListBox.swapCell:
Protected Sub swapCell(inFirst as integer,inSecond as integer)
  dim temp as string
  dim i,j as integer
  dim p as picture
  dim b as boolean
  j=me.columnCount-1
  for i=0 to j
    temp=""
    temp=me.cell(inSecond,i)
    me.cell(inSecond,i)=me.cell(inFirst,i)
    me.cell(inFirst,i)=temp
    b=me.cellCheck(inSecond,i)
    me.cellCheck(inSecond,i)=me.cellCheck(inFirst,i)
    me.cellCheck(inFirst,i)=b
  next
  p=me.rowPicture(inSecond)
  me.rowPicture(inSecond)=me.rowPicture(inFirst)
  me.rowPicture(inFirst)=p
End Sub

OriginalListBox.sortNumber:
Sub sortNumber(inColumn as integer)
  me.c=inColumn
  me.d=me.columnSortDirection(inColumn)
  me.visible=false
  me.listIndex=-1
  me.compareNumber(0,me.listCount-1)
  me.visible=true
End Sub

OriginalListBox.cellRefresh:
Sub cellRefresh(row as integer,column as integer)
  me.cell(row,column)=me.cell(row,column)
End Sub

OriginalListBox.DoubleClick:
Sub DoubleClick()
  me.refresh()
  CMMClicked()
End Sub

OriginalListBox.CellAction:
Sub CellAction(row As Integer, column As Integer)
  me.cell(row,column)=getString(me.cell(row,column))
End Sub

OriginalListBox.Open:
Sub Open()
  me.defaultY=me.defaultRowHeight-7
  me.defaultColumnWidth=me.columnWidths
  Open()
End Sub

OriginalListBox.LostFocus:
Sub LostFocus()
  dim i as integer
  for i=ubound(me.menus) downto 1
    me.menus(i).close()
    me.menus.remove i
  next
  me.helpTag=""
  me.balloonHelp=""
exception
End Sub

OriginalListBox.MouseDown:
Function MouseDown(x As Integer, y As Integer) As Boolean
  if isCMMClick then
    CMMClicked()
    return true
   end
End Function

OriginalListBox.HeaderPressed:
Function HeaderPressed(column as Integer) As Boolean
  if me.columnSortDirection(column)=0 then
    me.columnSortDirection(column)=1
  elseif me.headingIndex=column and me.columnSortDirection(column)=1 then
    me.columnSortDirection(column)=0
    me.headingIndex=-1
    return true
  end
End Function

OriginalListBox.EnableMenuItems:
Sub EnableMenuItems()
  dim i,j as integer
  columnsMenu.enabled=true
  columnMenus(0).enabled=true
  j=ubound(me.menus)
  for i=1 to j
    me.menus(i).enabled=true
    me.menus(i).checked=(me.column(val(me.menus(i).Tag)).WidthExpression<>"0")
  next
  if me.listCount<>0 then
    editSelectAll.enabled=(me.selectionType=1)
    editCopy.enabled=(me.listIndex<>-1 and lenb(me.helpTag)<>0)
  else
    editSelectAll.enabled=false
    editCopy.enabled=false
  end
End Sub

OriginalListBox.CellBackgroundPaint:
Function CellBackgroundPaint(g As Graphics, row As Integer, column As Integer) As Boolean
  #if targetPPC
    if me.hierarchical then
      return false
    end
  #endif
  
  if me.selected(row) then
    if me.active then
      g.foreColor=rgb(56,116,209)
    else
      g.foreColor=rgb(216,216,216)
    end
    g.fillRect 0,0,g.width,g.height
  elseif me.isColored and (row mod 2)=0 then
    g.foreColor=rgb(235,245,255)
    g.fillRect 0,0,g.width,g.height
  end
  
  if CellBackgroundPaint(g,row,column)=false then
    if me.selected(row) then
      if me.active then
        g.foreColor=rgb(&h92,&hB3,&hEA)
      else
        g.foreColor=rgb(&hE9,&hE9,&hE9)
      end
      g.drawLine 0,g.height-1,g.width,g.height-1
    end
    if me.isColored then
      if me.selected(row) then
        if me.active then
          g.foreColor=rgb(&h30,&h64,&hB7)
        else
          g.foreColor=rgb(&hB4,&hB4,&hB4)
        end
      else
        if me.active then
          g.foreColor=rgb(&hCC,&hCC,&hCC)
        else
          g.foreColor=rgb(&hD9,&hD9,&hD9)
        end
      end
      g.drawLine g.width-1,0,g.width-1,g.height
    end
  end
  return true
End Function

OriginalListBox.SortColumn:
Function SortColumn(column As Integer) As Boolean
  if me.visible then
    if SortColumn(column)=false then
      me.visible=false
      me.sort()
      me.visible=true
    end
    return true
  end
End Function

OriginalListBox.GotFocus:
Sub GotFocus()
  dim i,j,aArray as integer
  j=me.columnCount-1
  for i=0 to j
    if me.column(i).userResizable then
      me.menus.append new columnMenus
      aArray=ubound(me.menus)
      if me.menus(aArray)<>nil then
        me.menus(aArray).text=me.heading(i)
        me.menus(aArray).tag=str(i)
      end
    end
  next
exception
End Sub

OriginalListBox.CellTextPaint:
Function CellTextPaint(g As Graphics, row As Integer, column As Integer, x as Integer, y as Integer) As Boolean
  #if targetPPC
    if me.hierarchical then
      return false
    end
  #endif
  if me.selected(row) and me.active then
    g.foreColor=rgb(&hFF,&hFF,&hFF)
  end
  if CellTextPaint(g,row,column,x+1,me.defaultY)=false then
    g.drawString me.cell(row,column),x+1,me.defaultY
  end
  return true
End Function

OriginalListenerSocket.AddSocket:
Function AddSocket() As TCPSocket
  dim aArray as integer
  dim t as OriginalTransferSocket
  if TransferSocketLock.enter() then
    aArray=TS_Append()
    if aArray<>c_error then
      t=TransferSocket(aArray)
    end
    TransferSocketLock.leave()
  end
  if t=nil then
    t=new OriginalTransferSocket
  end
  return t
End Function

OriginalNapListSocket.getList:
Sub getList()
  select case me.ID
  case 1
    me.host="www.naplist.com"
    me.path="/servers.txt"
  case 2
    me.host="www.gotnap.com"
    me.path="/servers.txt"
  end
  
  if lenb(iHTTPProxy)=0 then
    me.address=me.host
    me.port=80
  else
    me.address=nthfield(iHTTPProxy,chrb(58),1)
    me.port=val(nthfield(iHTTPProxy,chrb(58),2))
  end
  me.connect()
End Sub

OriginalNapListSocket.onError:
Sub onError()
  if me.isError then
    return
  end
  me.isError=true
  
  if me.isConnected then
    me.disconnect()
  else
    me.close()
    me.isFinalized=true
    removingObjects=removingObjects+1
  end
End Sub

OriginalNapListSocket.recieved:
Protected Sub recieved()
  dim buffer,row as string
  dim i,j,aIndex as integer
  buffer=me.readall()
  buffer=replaceallb(buffer,chrb(13),"")
  if NetworkWindowLock.enter() then
    MainWindow.NetworkListBox.visible=false
    j=countfields(buffer,chrb(10))-2
    for i=12 to j
      row=""
      row=nthfield(buffer,chrb(10),i)
      aIndex=MainWindow.NE_Append(getDefine(nthfield(row,chrb(32),4)),getDefine(nthfield(row,chrb(32),9)),getDefine(nthfield(row,chrb(32),3)),false,getDefine(nthfield(row,chrb(32),14)),chrb(42),chrb(42),true)
    next
    MainWindow.NetworkListBox.visible=true
    NetworkWindowLock.leave()
  end
End Sub

OriginalNapListSocket.Connected:
Sub Connected()
  dim aCrlf as string
  if me.isError then
    return
  end
  
  aCrlf=chrb(13)+chrb(10) 
  if lenb(iHTTPProxy)=0 then
    me.write("GET "+me.path+" HTTP/1.0"+aCrlf)
  else
    me.write("GET http://"+me.host+me.path+" HTTP/1.0"+aCrlf)
  end
  me.write("Host: "+me.host+aCrlf)
  me.write("Connection: close"+aCrlf)
  me.write(aCrlf)
End Sub

OriginalNapListSocket.Error:
Sub Error()
  if me.isError then
    me.isFinalized=true
    removingObjects=removingObjects+1
    return
  end
  
  if me.lastErrorCode=102 then
    me.recieved()
  end
  me.onError()
End Sub

OriginalNetworkTimer.Action:
Sub Action()
  me.period=250
  if ubound(OpenNapSocket)>=iConnectionLimits then
    me.isWantDisconnect=me.isWantDisconnect+1
    if me.isWantDisconnect=120 then
      me.isWantDisconnect=0
      OS_DisconnectConnectingServers()
    end
  else
    me.isWantDisconnect=0
  end
  MainWindow.NE_AutoConnect()
End Sub

OriginalOpenNapSocket.onError:
Sub onError()
  if me.isError then
    return
  end
  me.isError=true
  
  if me.isLoggedIn then
    me.isLoggedIn=false
    gOnline=gOnline-1
  end
  if me.isConnected then
    me.disconnect()
  else
    me.close()
    me.isFinalized=true
    removingObjects=removingObjects+1
  end
End Sub

OriginalOpenNapSocket.command:
Sub command(inType as integer,inWhat as string)
  if me.isLoggedIn or (me.isLoggedIn=false and inType=2) then
    me.write(getHeader(lenb(inWhat),inType))
    me.write(inWhat)
  end
End Sub

OriginalOpenNapSocket.checkAhead:
Protected Function checkAhead() As Boolean
  dim aLength,headLength as integer
  dim temp as string
  temp=me.lookahead()
  headLength=lenb(temp)
  aLength=ascb(leftb(temp,1))+ascb(midb(temp,2,1))*256
  return headLength>=aLength+4
End Function

OriginalOpenNapSocket.remove:
Sub remove()
  DrawerWindow.MenuListBox.cellRefresh(0,DR_Count)
  DrawerWindow.MenuListBox.cellRefresh(4,DR_Count)
  MainWindow.NE_Offline(me.ID,me.errorCode,me.isRedirectAvailable,getDefine(me.address)+chrb(58)+getDefine(str(me.port)))
  MainWindow.CO_Offline(me.ID)
  MainWindow.CH_Offline(me.ID)
  MainWindow.WH_Offline(me.ID)
  MainWindow.US_Offline(me.ID)
  MainWindow.TR_Offline(me.ID,0)
  MainWindow.TR_Offline(me.ID,1)
  MainWindow.SE_Offline(me.ID)
  BR_Offline(me.ID)
  ME_Offline(me.ID)
  RO_Offline(me.ID)
End Sub

OriginalOpenNapSocket.recieved:
Protected Sub recieved()
  dim what,temp,partnerNick,targetFileName as string
  dim i,j,aLength,aType,aArray,aIndex,aCount as integer
  aLength=ascb(me.read(1))+ascb(me.read(1))*256
  aType=ascb(me.read(1))+ascb(me.read(1))*256
  what=me.read(aLength,gTextEncoding)
  
  select case aType
  case 0'error message
    if NetworkWindowLock.enter() then
      aIndex=MainWindow.NE_SearchID2ListIndex(me.ID)
      if aIndex<>c_error then
        MainWindow.NetworkListBox.cell(aIndex,NE_Comment)=what
      end
      NetworkWindowLock.leave()
    end
    
  case 3'login ack
    OS_DisconnectExcludeIDAndNetwork(me.ID,me.network)
    me.isLoggedIn=true
    gOnline=gOnline+1
    DrawerWindow.MenuListBox.cellRefresh(0,DR_Count)
    MainWindow.NE_Online(me.ID)
    MainWindow.CO_Online(me.ID,me.network)
    MainWindow.CH_Online(me.ID,me.network)
    MainWindow.WH_Online(me.ID,me.network)
    MainWindow.SE_Online(me.ID,me.network)
    BR_Online(me.ID,me.network)
    ME_Online(me.ID,me.network)
    RO_Online(me.ID,me.network)
    getSound(rs_login)
    
    if isIndexing=false then
      me.write(sharedBuffer)
      me.isShared=true
    end
    if UserListWindowLock.enter() then
      j=MainWindow.UserListListBox.listCount-1
      for i=0 to j
        if strcomp(MainWindow.UserListListBox.cell(i,US_Network),me.network,0)=0 then
          MainWindow.UserListListBox.cell(i,US_ServerID)=str(me.ID)
          if lenb(MainWindow.UserListListBox.cell(i,US_Icon))=1 then
            'hotlist
            me.command(208,MainWindow.UserListListBox.cell(i,US_Nick))
          end
        end
      next
      if me.isWinMX then
        'clear ignore list
        me.command(326,"")
      end
      for i=0 to j
        if val(MainWindow.UserListListBox.cell(i,US_ServerID))=me.ID and lenb(MainWindow.UserListListBox.cell(i,US_Icon))=0 then
          'add user to ignore list
          me.command(322,MainWindow.UserListListBox.cell(i,US_Nick))
        end
      next
      UserListWindowLock.leave()
    end
    if me.isWinMX then
      'change link speed
      me.command(700,getDefine(str(iSpeed)))
    end
    j=gDownloading
    for i=1 to j
      'start download
      me.command(218,"")
    next
    j=gUploading
    for i=1 to j
      'start upload
      me.command(220,"")
    next
    if ChannelLock.enter() then
      j=ubound(Channel)
      for i=1 to j
        if strcomp(Channel(i).network,me.network,0)=0 then
          'join channel
          me.command(400,Channel(i).channel)
        end
      next
      ChannelLock.leave()
    end
    
  case 4'version check
    
  case 8'nickname not registered
    
  case 9'nickname already registered
    
  case 10'invalid nickname
    
  case 12'password OK
    
  case 110'unshare all files
    
  case 201'search response
    if SearchWindowLock.enter() then
      aArray=SE_SearchID2Array(me.searchWindowID) 
      if aArray<>c_error then
        SearchWindow(aArray).SearchList.append new OriginalSearchList
        i=ubound(SearchWindow(aArray).SearchList)
        if SearchWindow(aArray).SearchList(i)<>nil then
          targetFileName=getDefine(nthfield(what,chrb(34),2))
          temp=getDefine(nthfield(targetFileName,chrb(92),countfields(targetFileName,chrb(92))))
          SearchWindow(aArray).SearchList(i).fileName=targetFileName
          SearchWindow(aArray).SearchList(i).shortName=temp
          SearchWindow(aArray).SearchList(i).mediaType=getMediaType(rightb(temp,inbackStrb(1,temp,".")))
          what=nthfield(what,chrb(34),3)
          temp=""
          temp=getDefine(nthfield(what,chrb(32),3))
          SearchWindow(aArray).SearchList(i).size=temp
          SearchWindow(aArray).SearchList(i).shortSize=getShortSize(val(temp))
          SearchWindow(aArray).SearchList(i).nick=getDefine(nthfield(what,chrb(32),7))
          temp=""
          temp=nthfield(what,chrb(32),9)
          SearchWindow(aArray).SearchList(i).speed=temp
          SearchWindow(aArray).SearchList(i).shortSpeed=getShortSpeed(val(temp))
          SearchWindow(aArray).SearchList(i).bitrate=nthfield(what,chrb(32),4)+s__kbps
          SearchWindow(aArray).SearchList(i).serverID=str(me.ID)
          SearchWindow(aArray).SearchList(i).network=me.network
          
          if me.searchWindowID=gSearchWindowID then
            MainWindow.SearchListBox.addrow("")
            aIndex=MainWindow.SearchListBox.lastIndex
            MainWindow.SearchListBox.rowPicture(aIndex)=gMediaPicture(val(SearchWindow(aArray).SearchList(i).mediaType))
            MainWindow.SearchListBox.cell(aIndex,SE_Icon)=SearchWindow(aArray).SearchList(i).mediaType
            MainWindow.SearchListBox.cell(aIndex,SE_FileName)=targetFileName
            MainWindow.SearchListBox.cell(aIndex,SE_ShortName)=SearchWindow(aArray).SearchList(i).shortName
            MainWindow.SearchListBox.cell(aIndex,SE_Size)=SearchWindow(aArray).SearchList(i).size
            MainWindow.SearchListBox.cell(aIndex,SE_ShortSize)=SearchWindow(aArray).SearchList(i).shortSize
            MainWindow.SearchListBox.cell(aIndex,SE_Nick)=SearchWindow(aArray).SearchList(i).nick
            MainWindow.SearchListBox.cell(aIndex,SE_Speed)=SearchWindow(aArray).SearchList(i).shortSpeed
            MainWindow.SearchListBox.cell(aIndex,SE_Bitrate)=SearchWindow(aArray).SearchList(i).bitrate
            MainWindow.SearchListBox.cell(aIndex,SE_ServerID)=SearchWindow(aArray).SearchList(i).serverID
            MainWindow.SearchListBox.cell(aIndex,SE_Network)=me.network
          end
        end
      end
      SearchWindowLock.leave()
    end
    
  case 202'end of search response from server
    if SearchWindowLock.enter() then
      aArray=SE_SearchID2Array(me.searchWindowID) 
      if aArray<>c_error then
        if gSearchWindowID=me.searchWindowID then
          DrawerWindow.DR_UpdateResults(me.searchWindowID,SearchWindow(aArray).title,MainWindow.SearchListBox.listCount,ubound(SearchWindow(aArray).SearchList))
        else
          DrawerWindow.DR_Update(me.searchWindowID,SearchWindow(aArray).title,str(ubound(SearchWindow(aArray).SearchList)))
        end
      end
      SearchWindowLock.leave()
    end
    
  case 204'download ack
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    targetFileName=getDefine(nthfield(what,chrb(34),2))
    temp=getLongIP2ShortIP(nthfield(what,chrb(32),2))
    j=val(nthfield(what,chrb(32),3))
    if TransferWindowLock.enter() then
      aIndex=MainWindow.TR_SearchUDAndNickAndFileNameAndServerID2ListIndex(0,partnerNick,targetFileName,me.ID)
      if aIndex<>c_error and val(MainWindow.TransferListBoxes(0).cell(aIndex,TR_ID))=0 then
        if j<>0 then
          TS_PrepareTransfer(me.ID,me.nick,me.isWinMX,aIndex,temp,j,0,false)
        elseif iPort<>0 then
          TS_PrepareTransfer(me.ID,me.nick,me.isWinMX,aIndex,"",0,0,true)
          'alternate download request
          me.command(500,partnerNick+chrb(32)+chrb(34)+targetFileName+chrb(34))
        else
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=s_both_of_users_are_firewalled
          MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
        end
      end
      TransferWindowLock.leave()
    end
    
  case 205'private message from another user
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    temp=getDefine(replaceallb(midb(what,lenb(partnerNick)+2),chrb(10),""))
    if strcomp(temp,"//WantQueue",0)=0 then
      if me.isWinMX then
        me.wantQueuedNick.append partnerNick
      end
    else
      if MessageWindowLock.enter() then
        aArray=ME_SearchNickAndServerID2Array(partnerNick,me.ID)
        if aArray=c_error then
          aArray=ME_Append(partnerNick,me.ID,me.nick,me.network)
          if aArray<>c_error then
            DrawerWindow.DR_Append(partnerNick,MessageWindow(aArray).ID,c_message_window)
            getMessage(MessageWindow(aArray).textAndStyle,partnerNick,co_other_users,temp,co_default)
            if gMessageWindowID=MessageWindow(aArray).ID then
              MainWindow.MessageEditField.AppendMessage(partnerNick,co_other_users,temp,co_default)
            end
            DrawerWindow.DR_MessageArrived(MessageWindow(aArray).ID)
            if gAutoReply and lenb(iAutoReply)<>0 then
              getMessage(MessageWindow(aArray).textAndStyle,me.nick,co_own_user,iAutoReply,co_default)
              if gMessageWindowID=MessageWindow(aArray).ID then
                MainWindow.MessageEditField.AppendMessage(me.nick,co_own_user,iAutoReply,co_default)
              end
              'private message
              me.command(205,partnerNick+chrb(32)+iAutoReply)
            end
          end
        else
          getMessage(MessageWindow(aArray).textAndStyle,partnerNick,co_other_users,temp,co_default)
          if gMessageWindowID=MessageWindow(aArray).ID then
            MainWindow.MessageEditField.AppendMessage(partnerNick,co_other_users,temp,co_default)
          end
          DrawerWindow.DR_MessageArrived(MessageWindow(aArray).ID)
        end
        notifyStart()
        getSound(rs_message)
        MessageWindowLock.leave()
      end
    end
    
  case 206'get error
    partnerNick=nthfield(what,chrb(32),1)
    temp=midb(what,lenb(partnerNick)+2)
    if ascb(leftb(temp,1))=34 then
      targetFileName=nthfield(temp,chrb(34),2)
    else
      targetFileName=temp
    end
    if TransferWindowLock.enter() then
      aIndex=MainWindow.TR_SearchUDAndNickAndFileNameAndServerID2ListIndex(0,partnerNick,targetFileName,me.ID)
      if aIndex<>c_error and val(MainWindow.TransferListBoxes(0).cell(aIndex,TR_ID))=0 then
        MainWindow.TransferListBoxes(0).cell(aIndex,TR_ServerID)=""
        MainWindow.TransferListBoxes(0).cell(aIndex,TR_Network)=s_offline
        MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_Nick)
        if MainWindow.TransferListBoxes(0).cellCheck(aIndex,TR_Retry) then
          gDownloadRetrying=gDownloadRetrying+1
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Timer)="300"
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=str(TF_RetryingIn)
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(300))
          MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
        else
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=s_accept_failed
          MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
        end
      end
      TransferWindowLock.leave()
    end
    
  case 209'user signon
    partnerNick=nthfield(what,chrb(32),1)
    if UserListWindowLock.enter() then
      aIndex=MainWindow.US_SearchNickAndServerID2ListIndex(partnerNick,me.ID)
      if aIndex<>c_error and lenb(MainWindow.UserListListBox.cell(aIndex,US_Icon))=1 then
        MainWindow.UserListListBox.cell(aIndex,US_Icon)="  "
        MainWindow.UserListListBox.RowPicture(aIndex)=gPicture(r_favorite)
        gHotlist=gHotlist+1
        DrawerWindow.MenuListBox.cellRefresh(4,DR_Count)
        DrawerWindow.MenuListBox.rowPicture(4)=gPicture(r_arrived)
      end
      UserListWindowLock.leave()
    end
    
  case 210'user signoff
    partnerNick=nthfield(what,chrb(32),1)
    if UserListWindowLock.enter() then
      aIndex=MainWindow.US_SearchNickAndServerID2ListIndex(partnerNick,me.ID)
      if aIndex<>c_error and lenb(MainWindow.UserListListBox.cell(aIndex,US_Icon))=2 then
        MainWindow.UserListListBox.cell(aIndex,US_Icon)=" "
        MainWindow.UserListListBox.RowPicture(aIndex)=nil
        gHotlist=gHotlist-1
        DrawerWindow.MenuListBox.cellRefresh(4,DR_Count)
      end
      UserListWindowLock.leave()
    end
    
  case 212'browse response
    if BrowseWindowLock.enter() then
      aArray=BR_SearchNickAndServerID2Array(nthfield(what,chrb(32),1),me.ID)
      if aArray<>c_error then
        BrowseWindow(aArray).BrowseList.append new OriginalBrowseList
        i=ubound(BrowseWindow(aArray).BrowseList)
        if BrowseWindow(aArray).BrowseList(i)<>nil then
          targetFileName=getDefine(nthfield(what,chrb(34),2))
          temp=getDefine(nthfield(targetFileName,chrb(92),countfields(targetFileName,chrb(92))))
          BrowseWindow(aArray).BrowseList(i).fileName=targetFileName
          BrowseWindow(aArray).BrowseList(i).shortName=temp
          BrowseWindow(aArray).BrowseList(i).mediaType=getMediaType(rightb(temp,inbackStrb(1,temp,".")))
          BrowseWindow(aArray).BrowseList(i).folderPath=leftb(targetFileName,lenb(targetFileName)-lenb(temp))
          what=nthfield(what,chrb(34),3)
          temp=""
          temp=getDefine(nthfield(what,chrb(32),3))
          BrowseWindow(aArray).totalSize=BrowseWindow(aArray).totalSize+val(temp)
          BrowseWindow(aArray).BrowseList(i).size=temp
          BrowseWindow(aArray).BrowseList(i).shortSize=getShortSize(val(temp))
          BrowseWindow(aArray).BrowseList(i).bitRate=nthfield(what,chrb(32),4)+s__kbps
          if gBrowseWindowID=BrowseWindow(aArray).ID then
            MainWindow.BrowseListBox.addrow("")
            aIndex=MainWindow.BrowseListBox.lastIndex
            MainWindow.BrowseListBox.rowPicture(aIndex)=gMediaPicture(val(BrowseWindow(aArray).BrowseList(i).mediaType))
            MainWindow.BrowseListBox.cell(aIndex,BR_Icon)=BrowseWindow(aArray).BrowseList(i).mediaType
            MainWindow.BrowseListBox.cell(aIndex,BR_FileName)=targetFileName
            MainWindow.BrowseListBox.cell(aIndex,BR_ShortName)=BrowseWindow(aArray).BrowseList(i).shortName
            MainWindow.BrowseListBox.cell(aIndex,BR_Size)=temp
            MainWindow.BrowseListBox.cell(aIndex,BR_ShortSize)=BrowseWindow(aArray).BrowseList(i).shortSize
            MainWindow.BrowseListBox.cell(aIndex,BR_BitRate)=BrowseWindow(aArray).BrowseList(i).bitRate
          end
        end
      end
      BrowseWindowLock.leave()
    end
    
  case 213'end of browse list
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    if BrowseWindowLock.enter() then
      aArray=BR_SearchNickAndServerID2Array(partnerNick,me.ID)
      if aArray<>c_error then
        if gBrowseWindowID=BrowseWindow(aArray).ID then
          MainWindow.FolderListBox.cell(0,BR_Icon)=getShortSize(BrowseWindow(aArray).totalSize)
          DrawerWindow.DR_UpdateResults(BrowseWindow(aArray).ID,partnerNick,MainWindow.BrowseListBox.listCount,ubound(BrowseWindow(aArray).BrowseList))
        else
          DrawerWindow.DR_Update(BrowseWindow(aArray).ID,partnerNick,str(ubound(BrowseWindow(aArray).BrowseList)))
        end
      end
      BrowseWindowLock.leave()
    end
    
  case 214'server stats
    if NetworkWindowLock.enter() then
      aIndex=MainWindow.NE_SearchID2ListIndex(me.ID)
      if aIndex<>c_error then
        MainWindow.NetworkListBox.cell(aIndex,NE_Comment)=replace(s__users,"%",format(val(nthfield(what,chrb(32),1)),"###,###,###"))
      end
      NetworkWindowLock.leave()
    end
    
  case 216'resume search response
    
  case 217'end of resume search list
    
  case 301'hotlist ack
    
  case 302'hotlist error
    
  case 320'user ignore list
    
  case 321'user ignore list entry
    
  case 322'add user to ignore list
    
  case 323'remove user from ignore list
    
  case 324'user is not ignored
    
  case 325'user is already ignored
    
  case 326'clear ignore list
    
  case 401'part channel
    temp=nthfield(what,chrb(32),1)
    if ChannelLock.signal() then
      aArray=OC_SearchNetworkAndChannel2Array(me.network,temp)
      if aArray<>c_error then
        Channel.remove aArray
      end
      ChannelLock.release()
    end
    if RoomWindowLock.signal() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        redim RoomWindow(aArray).RoomList(0)
        temp=""
        temp=getDefine(replaceb(getString(s__has_leaved),"%",me.nick))
        getMessage(RoomWindow(aArray).textAndStyle,temp,co_notifications)
        RoomWindow(aArray).topic=""
        if gRoomWindowID=RoomWindow(aArray).ID then
          MainWindow.RoomListBox.deleteAllRows()
          MainWindow.RoomEditField.AppendMessage(temp,co_notifications)
          MainWindow.RoomStaticText.text=""
          MainWindow.RoomStaticText.helpTag=""
          MainWindow.RoomStaticText.balloonHelp=""
        end
        DrawerWindow.DR_Update(RoomWindow(aArray).ID,RoomWindow(aArray).channel,"")
      end
      RoomWindowLock.release()
    end
    
  case 403'public message
    temp=getDefine(nthfield(what,chrb(32),1))
    partnerNick=getDefine(nthfield(what,chrb(32),2))
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        temp=getDefine(replaceallb(midb(what,lenb(temp)+lenb(partnerNick)+3),chrb(10),""))
        if strcomp(partnerNick,me.nick,0)=0 then
          i=co_own_user
        else
          i=co_other_users
        end
        getMessage(RoomWindow(aArray).textAndStyle,partnerNick,i,temp,co_default)
        if gRoomWindowID=RoomWindow(aArray).ID then
          MainWindow.RoomEditField.AppendMessage(partnerNick,i,temp,co_default)
        end
        DrawerWindow.DR_MessageArrived(RoomWindow(aArray).ID)
        getSound(rs_click)
      end
      RoomWindowLock.leave()
    end
    
  case 404'error message
    
  case 405'join acknowledge
    temp=getDefine(nthfield(what,chrb(32),1))
    if ChannelLock.enter() then
      aArray=OC_SearchNetworkAndChannel2Array(me.network,temp)
      if aArray=c_error then
        Channel.append new OriginalChannel
        aArray=ubound(Channel)
        if Channel(aArray)<>nil then
          Channel(aArray).network=me.network
          Channel(aArray).channel=temp
        end
      end
      ChannelLock.leave()
    end
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray=c_error then
        aArray=RO_Append(temp,me.ID,me.network)
        if aArray<>c_error then
          DrawerWindow.DR_Append(temp,RoomWindow(aArray).ID,c_room_window)
        end
      else
        temp=""
        temp=getDefine(replaceb(getString(s__has_joined),"%",me.nick))
        getMessage(RoomWindow(aArray).textAndStyle,temp,co_notifications)
        if gRoomWindowID=RoomWindow(aArray).ID then
          MainWindow.RoomEditField.AppendMessage(temp,co_notifications)
        end
      end
      if aArray<>c_error then
        DrawerWindow.DR_MessageArrived(RoomWindow(aArray).ID)
        getSound(rs_bell)
      end
      RoomWindowLock.leave()
    end
    
  case 406'join message
    temp=nthfield(what,chrb(32),1)
    partnerNick=getDefine(nthfield(what,chrb(32),2))
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        RoomWindow(aArray).RoomList.append new OriginalRoomList
        i=ubound(RoomWindow(aArray).RoomList)
        if RoomWindow(aArray).RoomList(i)<>nil then
          RoomWindow(aArray).RoomList(i).nick=partnerNick
          RoomWindow(aArray).RoomList(i).count=nthfield(what,chrb(32),3)
          RoomWindow(aArray).RoomList(i).speed=getShortSpeed(val(nthfield(what,chrb(32),4)))
          if gRoomWindowID=RoomWindow(aArray).ID then
            MainWindow.RoomListBox.addrow(partnerNick)
            aIndex=MainWindow.RoomListBox.lastIndex
            MainWindow.RoomListBox.cell(aIndex,RO_Count)=RoomWindow(aArray).RoomList(i).count
            MainWindow.RoomListBox.cell(aIndex,RO_Speed)=RoomWindow(aArray).RoomList(i).speed
          end
          if iShowJoinLeave then
            temp=""
            temp=getDefine(replaceb(getString(s__has_joined),"%",partnerNick))
            getMessage(RoomWindow(aArray).textAndStyle,temp,co_notifications)
            if gRoomWindowID=RoomWindow(aArray).ID then
              MainWindow.RoomEditField.appendMessage(temp,co_notifications)
            end
          end
          DrawerWindow.DR_Update(RoomWindow(aArray).ID,RoomWindow(aArray).channel,str(ubound(RoomWindow(aArray).RoomList)))
          getSound(rs_join)
        end
      end
      RoomWindowLock.leave()
    end
    
  case 407'user parted channel
    temp=nthfield(what,chrb(32),1)
    partnerNick=getDefine(nthfield(what,chrb(32),2))
    if RoomWindowLock.signal() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        i=RoomWindow(aArray).RO_SearchNick2Array(partnerNick)
        if i<>c_error then
          RoomWindow(aArray).RoomList.remove i
          if gRoomWindowID=RoomWindow(aArray).ID then
            aIndex=MainWindow.RO_SearchNick2ListIndex(partnerNick)
            if aIndex<>c_error then
              MainWindow.RoomListBox.removeRow(aIndex)
            end
          end
          if iShowJoinLeave then
            temp=""
            temp=getDefine(replaceb(getString(s__has_leaved),"%",partnerNick))
            getMessage(RoomWindow(aArray).textAndStyle,temp,co_notifications)
            if gRoomWindowID=RoomWindow(aArray).ID then
              MainWindow.RoomEditField.AppendMessage(temp,co_notifications)
            end
          end
          DrawerWindow.DR_Update(RoomWindow(aArray).ID,RoomWindow(aArray).channel,str(ubound(RoomWindow(aArray).RoomList)))
          getSound(rs_leave)
        end
      end
      RoomWindowLock.release()
    end
    
  case 408'channel user list entry
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(nthfield(what,chrb(32),1),me.ID)
      if aArray<>c_error then
        partnerNick=getDefine(nthfield(what,chrb(32),2))
        RoomWindow(aArray).RoomList.append new OriginalRoomList
        i=ubound(RoomWindow(aArray).RoomList)
        if RoomWindow(aArray).RoomList(i)<>nil then
          RoomWindow(aArray).RoomList(i).nick=partnerNick
          RoomWindow(aArray).RoomList(i).count=nthfield(what,chrb(32),3)
          RoomWindow(aArray).RoomList(i).speed=getShortSpeed(val(nthfield(what,chrb(32),4)))
          if gRoomWindowID=RoomWindow(aArray).ID then
            MainWindow.RoomListBox.addrow(partnerNick)
            aIndex=MainWindow.RoomListBox.lastIndex
            MainWindow.RoomListBox.cell(aIndex,RO_Count)=RoomWindow(aArray).RoomList(i).count
            MainWindow.RoomListBox.cell(aIndex,RO_Speed)=RoomWindow(aArray).RoomList(i).speed
          end
        end
      end
      RoomWindowLock.leave()
    end
    
  case 409'end of channel user list
    temp=getDefine(nthfield(what,chrb(32),1))
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        DrawerWindow.DR_Update(RoomWindow(aArray).ID,temp,str(ubound(RoomWindow(aArray).RoomList)))
      end
      RoomWindowLock.leave()
    end
    
  case 410'channel topic
    temp=nthfield(what,chrb(32),1)
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        if lenb(RoomWindow(aArray).topic)<>0 then
          DrawerWindow.DR_MessageArrived(RoomWindow(aArray).ID)
          getSound(rs_post)
        end
        temp=getDefine(replaceallb(midb(what,lenb(temp)+2),chrb(10),""))
        RoomWindow(aArray).topic=""
        RoomWindow(aArray).topic=temp
        getMessage(RoomWindow(aArray).textAndStyle,temp,co_notifications)
        if gRoomWindowID=RoomWindow(aArray).ID then
          MainWindow.RoomEditField.AppendMessage(temp,co_notifications)
          MainWindow.RoomStaticText.text=temp
          MainWindow.RoomStaticText.helpTag=temp
          MainWindow.RoomStaticText.balloonHelp=temp
        end
      end
      RoomWindowLock.leave()
    end
    
  case 420'end of channel ban list
    
  case 421'channel ban list entry
    temp=nthfield(what,chrb(32),1)
    partnerNick=getDefine(nthfield(what,chrb(32),2))
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        getMessage(RoomWindow(aArray).textAndStyle,partnerNick,co_notifications)
        if gRoomWindowID=RoomWindow(aArray).ID then
          MainWindow.RoomEditField.AppendMessage(partnerNick,co_notifications)
        end
      end
      RoomWindowLock.leave()
    end
    
  case 430'invite a user
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    temp=getDefine(nthfield(what,chrb(32),2))
    if MessageWindowLock.enter() then
      aArray=ME_SearchNickAndServerID2Array(partnerNick,me.ID)
      if aArray=c_error then
        aArray=ME_Append(partnerNick,me.ID,me.nick,me.network)
        if aArray<>c_error then
          DrawerWindow.DR_Append(partnerNick,MessageWindow(aArray).ID,c_message_window)
        end
      end
      if aArray<>c_error then
        MessageWindow(aArray).channel=temp
        temp=getDefine(replaceb(getString(about_invitation),"%",temp))
        getMessage(MessageWindow(aArray).textAndStyle,temp,co_notifications)
        if gMessageWindowID=MessageWindow(aArray).ID then
          MainWindow.MessageEditField.AppendMessage(temp,co_notifications)
        end
        DrawerWindow.DR_MessageArrived(MessageWindow(aArray).ID)
        notifyStart()
        getSound(rs_message)
      end
      MessageWindowLock.leave()
    end
    
  case 501'alternate download ack
    partnerNick=nthfield(what,chrb(32),1)
    targetFileName=nthfield(what,chrb(34),2)
    temp=getLongIP2ShortIP(nthfield(what,chrb(32),2))
    j=val(nthfield(what,chrb(32),3))
    if TransferWindowLock.enter() then
      aIndex=MainWindow.TR_SearchUDAndNickAndFileNameAndServerID2ListIndex(1,partnerNick,targetFileName,me.ID)
      if aIndex<>c_error and val(MainWindow.TransferListBoxes(1).cell(aIndex,TR_ID))=0 then
        if j<>0 then
          TS_PrepareTransfer(me.ID,me.nick,me.isWinMX,aIndex,temp,j,1,true)
        else
          MainWindow.TransferListBoxes(1).cell(aIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(1).cell(aIndex,TR_Status)=s_both_of_users_are_firewalled
          MainWindow.TransferListBoxes(1).cellRefresh(aIndex,TR_ShortName)
        end
      end
      TransferWindowLock.leave()
    end
    
  case 601'link speed response
    
  case 604'whois response
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    if WhoisWindowLock.enter() then
      aIndex=MainWindow.WH_SearchNickAndServerID2ListIndex(partnerNick,me.ID)
      if aIndex=c_error then
        MainWindow.WhoisListBox.addrow("")
        aIndex=MainWindow.WhoisListBox.lastIndex
      end
      MainWindow.WhoisListBox.selected(aIndex)=true
      MainWindow.WhoisListBox.cell(aIndex,WH_Nick)=partnerNick
      what=midb(what,lenb(partnerNick)+2)
      if ascb(leftb(what,1))=34 then
        temp=getDefine(nthfield(what,chrb(34),2))
        MainWindow.WhoisListBox.cell(aIndex,WH_Level)=temp
        what=midb(what,lenb(temp)+4)
      else
        temp=getDefine(nthfield(what,chrb(32),1))
        MainWindow.WhoisListBox.cell(aIndex,WH_Level)=temp
        what=midb(what,lenb(temp)+2)
      end
      temp=""
      temp=nthfield(what,chrb(32),1)
      MainWindow.WhoisListBox.cell(aIndex,WH_Uptime)=getUptime(val(temp))
      what=midb(what,lenb(temp)+2)
      if ascb(leftb(what,1))=34 then
        temp=""
        temp=getDefine(nthfield(what,chrb(34),2))
        MainWindow.WhoisListBox.cell(aIndex,WH_Channel)=temp
        what=midb(what,lenb(temp)+4)
      else
        temp=""
        temp=getDefine(nthfield(what,chrb(32),1))
        MainWindow.WhoisListBox.cell(aIndex,WH_Channel)=temp
        what=midb(what,lenb(temp)+2)
      end
      if ascb(leftb(what,1))=34 then
        temp=""
        temp=getDefine(nthfield(what,chrb(34),2))
        MainWindow.WhoisListBox.cell(aIndex,WH_Status)=temp
        what=midb(what,lenb(temp)+4)
      else
        temp=""
        temp=getDefine(nthfield(what,chrb(32),1))
        MainWindow.WhoisListBox.cell(aIndex,WH_Status)=temp
        what=midb(what,lenb(temp)+2)
      end
      if strcomp(temp,"Active Friend",0)=0 then
        MainWindow.WhoisListBox.cell(aIndex,WH_Icon)="  "
        MainWindow.WhoisListBox.rowPicture(aIndex)=gPicture(r_favorite)
      else
        MainWindow.WhoisListBox.cell(aIndex,WH_Icon)=" "
        MainWindow.WhoisListBox.rowPicture(aIndex)=gPicture(r_online)
      end
      temp=""
      temp=nthfield(what,chrb(32),1)
      MainWindow.WhoisListBox.cell(aIndex,WH_Shares)=temp
      MainWindow.WhoisListBox.cell(aIndex,WH_Transfer)=nthfield(what,chrb(32),2)+chrb(47)+nthfield(what,chrb(32),3)
      MainWindow.WhoisListBox.cell(aIndex,WH_Speed)=getShortSpeed(val(nthfield(what,chrb(32),4)))
      MainWindow.WhoisListBox.cell(aIndex,WH_Client)=getDefine(nthfield(what,chrb(34),2))
      MainWindow.WhoisListBox.cell(aIndex,WH_Network)=me.network
      MainWindow.WhoisListBox.cell(aIndex,WH_ServerID)=str(me.ID)
      WhoisWindowLock.leave()
    end
    if TransferWindowLock.enter() then
      do
        if gDownloadRetrying<>0 then
          aIndex=MainWindow.TR_SearchUDAndNickAndServerID2ListIndex(0,partnerNick,0)
          if aIndex<>c_error then
            MainWindow.TransferListBoxes(0).cell(aIndex,TR_ServerID)=str(me.ID)
            MainWindow.TransferListBoxes(0).cell(aIndex,TR_Network)=me.network
            MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_Nick)
            if val(MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag))=TF_RetryingIn then
              gDownloadRetrying=gDownloadRetrying-1
              MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=str(TF_LocallyQueued)
              MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=s_locallY_queued
              MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
            end
          end
        end
      loop until gDownloadRetrying=0 or aIndex=c_error
      do
        if iAllowFreeloader=false and gUploadPreparing<>0 then
          aIndex=MainWindow.TR_SearchUDAndNickAndFlagAndServerID2ListIndex(1,partnerNick,TF_Preparing,me.ID)
          if aIndex<>c_error then
            gUploadPreparing=gUploadPreparing-1
            targetFileName=""
            targetFileName=MainWindow.TransferListBoxes(1).cell(aIndex,TR_FileName)
            if val(temp)<iFreeloaderShares then
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Flag)=""
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Status)=s_freeloader_refused
              MainWindow.TransferListBoxes(1).cellRefresh(aIndex,TR_ShortName)
            elseif me.isWinMX=false or hasWantQueue(me.wantQueuedNick,partnerNick) then
              aCount=MainWindow.TR_UploadQueueLimit(aIndex)
              if aCount<>c_error then
                MainWindow.TransferListBoxes(1).cell(aIndex,TR_Timer)="600"
                MainWindow.TransferListBoxes(1).cell(aIndex,TR_Flag)=str(TF_LocallyQueued)
                MainWindow.TransferListBoxes(1).cell(aIndex,TR_Status)=s_locallY_queued
                MainWindow.TransferListBoxes(1).cellRefresh(aIndex,TR_ShortName)
                'pong response
                me.command(752,partnerNick)
                'queue limit
                me.command(619,partnerNick+chrb(32)+chrb(34)+targetFileName+chrb(34)+chrb(32)+getDefine(str(aCount)))
              end
            else
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Timer)="600"
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Flag)=str(TF_LocallyQueued)
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Status)=s_locallY_queued
              MainWindow.TransferListBoxes(1).cellRefresh(aIndex,TR_ShortName)
              'pong response
              me.command(752,partnerNick)
              'queue limit
              me.command(619,partnerNick+chrb(32)+chrb(34)+targetFileName+chrb(34)+chrb(32)+getDefine("0"))
            end
          end
        end
      loop until iAllowFreeloader or gUploadPreparing=0 or aIndex=c_error
      TransferWindowLock.leave()
    end
    
  case 605'whowas response
    partnerNick=nthfield(what,chrb(32),1)
    if WhoisWindowLock.enter() then
      aIndex=MainWindow.WH_SearchNickAndServerID2ListIndex(partnerNick,me.ID)
      if aIndex<>c_error then
        MainWindow.WhoisListBox.selected(aIndex)=true
        temp=midb(what,lenb(partnerNick)+2)
        if ascb(leftb(temp,1))=34 then
          MainWindow.WhoisListBox.cell(aIndex,WH_Level)=getDefine(nthfield(temp,chrb(34),2))
        else
          MainWindow.WhoisListBox.cell(aIndex,WH_Level)=getDefine(nthfield(temp,chrb(32),1))
        end
        MainWindow.WhoisListBox.cell(aIndex,WH_Icon)=""
        MainWindow.WhoisListBox.rowPicture(aIndex)=nil
      end
      WhoisWindowLock.leave()
    end
    
  case 607'upload request
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    targetFileName=getDefine(nthfield(what,chrb(34),2))
    if LibraryLock.enter() then
      if me.isWinMX then
        aArray=OL_SearchFileName2Array(midb(targetFileName,4))
      else
        aArray=OL_SearchFileName2Array(targetFileName)
      end
      if aArray<>c_error then
        if TransferWindowLock.enter() then
          aIndex=MainWindow.TR_AppendUploadQueue(targetFileName,Library(aArray).shortName,Library(aArray).size,Library(aArray).shortSize,partnerNick,str(me.ID),me.network)
          if val(MainWindow.TransferListBoxes(1).cell(aIndex,TR_ID))=0 then
            if iAllowFreeloader=false then
              gUploadPreparing=gUploadPreparing+1
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Timer)="120"
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Flag)=str(TF_Preparing)
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Status)=s_preparing
              MainWindow.TransferListBoxes(1).cellRefresh(aIndex,TR_ShortName)
              'whois request
              me.command(603,partnerNick)
            elseif me.isWinMX=false or hasWantQueue(me.wantQueuedNick,partnerNick) then
              aCount=MainWindow.TR_UploadQueueLimit(aIndex)
              if aCount<>c_error then
                MainWindow.TransferListBoxes(1).cell(aIndex,TR_Timer)="600"
                MainWindow.TransferListBoxes(1).cell(aIndex,TR_Flag)=str(TF_LocallyQueued)
                MainWindow.TransferListBoxes(1).cell(aIndex,TR_Status)=s_locallY_queued
                MainWindow.TransferListBoxes(1).cellRefresh(aIndex,TR_ShortName)
                'pong response
                me.command(752,partnerNick)
                'queue limit
                me.command(619,partnerNick+chrb(32)+chrb(34)+targetFileName+chrb(34)+chrb(32)+getDefine(str(aCount)))
              end
            else
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Timer)="600"
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Flag)=str(TF_Queueble)
              MainWindow.TransferListBoxes(1).cell(aIndex,TR_Status)=s_queueble
              MainWindow.TransferListBoxes(1).cellRefresh(aIndex,TR_ShortName)
              'pong response
              me.command(752,partnerNick)
              'queue limit
              me.command(619,partnerNick+chrb(32)+chrb(34)+targetFileName+chrb(34)+chrb(32)+getDefine("0"))
            end
          end
          TransferWindowLock.leave()
        end
      end
      LibraryLock.leave()
    end
    
  case 609'accept failed
    partnerNick=nthfield(what,chrb(32),1)
    temp=midb(what,lenb(partnerNick)+2)
    if ascb(leftb(temp,1))=34 then
      targetFileName=nthfield(temp,chrb(34),2)
    else
      targetFileName=temp
    end
    if TransferWindowLock.enter() then
      aIndex=MainWindow.TR_SearchUDAndNickAndFileNameAndServerID2ListIndex(0,partnerNick,targetFileName,me.ID)
      if aIndex<>c_error and val(MainWindow.TransferListBoxes(0).cell(aIndex,TR_ID))=0 then
        MainWindow.TransferListBoxes(0).cell(aIndex,TR_ServerID)=""
        MainWindow.TransferListBoxes(0).cell(aIndex,TR_Network)=s_offline
        MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_Nick)
        if MainWindow.TransferListBoxes(0).cellCheck(aIndex,TR_Retry) then
          gDownloadRetrying=gDownloadRetrying+1
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Timer)="300"
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=str(TF_RetryingIn)
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(300))
          MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
        else
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=s_accept_failed
          MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
        end
      end
      TransferWindowLock.leave()
    end
    
  case 613'set data port for user
    if me.isWinMX=false then
      iPort=val(what)
      portMenu.text=s_port_+chrb(32)+str(iPort)
      if ListenerSocket.port<>iPort then
        if ListenerSocket.isListening then
          ListenerSocket.stopListening()
        end
        ListenerSocket.port=iPort
        if iPort<>0 then
          ListenerSocket.listen()
        end
        OS_Update()
      end
    end
    
  case 616'ban list entry
    
  case 617'end of channel list
    
  case 618'channel list entry
    temp=getDefine(nthfield(what,chrb(32),1))
    if gChannelServerID=0 or gChannelServerID=me.ID then
      if ChannelWindowLock.enter() then
        aIndex=MainWindow.CH_SearchChannelAndID2ListIndex(temp,me.ID)
        if aIndex=c_error then
          MainWindow.ChannelListBox.addrow(me.network)
          aIndex=MainWindow.ChannelListBox.lastIndex
        end
        MainWindow.ChannelListBox.cell(aIndex,CH_Channel)=temp
        what=midb(what,lenb(temp)+2)
        temp=""
        temp=nthfield(what,chrb(32),1)
        MainWindow.ChannelListBox.cell(aIndex,CH_Count)=temp
        what=midb(what,lenb(temp)+2)
        if ascb(leftb(what,1))=34 then
          MainWindow.ChannelListBox.cell(aIndex,CH_Comment)=getDefine(nthfield(what,chrb(34),2))
        else
          MainWindow.ChannelListBox.cell(aIndex,CH_Comment)=what
        end
        MainWindow.ChannelListBox.cell(aIndex,CH_ServerID)=str(me.ID)
        ChannelWindowLock.leave()
      end
    end
    
  case 620'queue limit
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    targetFileName=getDefine(nthfield(what,chrb(34),2))
    aCount=val(nthfield(what,chrb(32),countfields(what,chrb(32))))
    if TransferWindowLock.enter() then
      aIndex=MainWindow.TR_SearchUDAndNickAndFileNameAndServerID2ListIndex(0,partnerNick,targetFileName,me.ID)
      if aIndex<>c_error and val(MainWindow.TransferListBoxes(0).cell(aIndex,TR_ID))=0 then
        if aCount=0 or aCount>=1000 then
          if MainWindow.TransferListBoxes(0).cellCheck(aIndex,TR_WinMX) then
            MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=""
            MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=s_failed_to_queue
            MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
          else
            MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=str(TF_Queueble)
            MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=s_queueble
            MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
            MainWindow.TransferListBoxes(0).cellCheck(aIndex,TR_WinMX)=true
          end
        else
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Timer)="300"
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Flag)=str(TF_QueuedAt)
          MainWindow.TransferListBoxes(0).cell(aIndex,TR_Status)=replace(s_queued_at_,"%",str(aCount))
          MainWindow.TransferListBoxes(0).cellRefresh(aIndex,TR_ShortName)
        end
      end
      TransferWindowLock.leave()
    end
    
  case 621'message of the day
    
  case 626'data port error
    if me.isWinMX=false then
      iPort=0
      portMenu.text=s_port_+chrb(32)+"0"
      if ListenerSocket.isListening then
        ListenerSocket.stopListening()
      end
      ListenerSocket.port=iPort
      OS_Update()
    end
    
  case 627'operator message
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    temp=getDefine(replaceallb(midb(what,lenb(partnerNick)+2),chrb(10),""))
    if MessageWindowLock.enter() then
      aArray=ME_SearchNickAndServerID2Array(partnerNick,me.ID)
      if aArray=c_error then
        aArray=ME_Append(partnerNick,me.ID,me.nick,me.network)
        if aArray<>c_error then
          DrawerWindow.DR_Append(partnerNick,MessageWindow(aArray).ID,c_message_window)
        end
      end
      if aArray<>c_error then
        getMessage(MessageWindow(aArray).textAndStyle,partnerNick,co_other_users,temp,co_default)
        if gMessageWindowID=MessageWindow(aArray).ID then
          MainWindow.MessageEditField.AppendMessage(partnerNick,co_other_users,temp,co_default)
        end
        DrawerWindow.DR_MessageArrived(MessageWindow(aArray).ID)
        notifyStart()
        getSound(rs_message)
      end
      MessageWindowLock.leave()
    end
    
  case 628'global message
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    temp=getDefine(replaceallb(midb(what,lenb(partnerNick)+2),chrb(10),""))
    if MessageWindowLock.enter() then
      aArray=ME_SearchNickAndServerID2Array(partnerNick,me.ID)
      if aArray=c_error then
        aArray=ME_Append(partnerNick,me.ID,me.nick,me.network)
        if aArray<>c_error then
          DrawerWindow.DR_Append(partnerNick,MessageWindow(aArray).ID,c_message_window)
        end
      end
      if aArray<>c_error then
        getMessage(MessageWindow(aArray).textAndStyle,partnerNick,co_other_users,temp,co_default)
        if gMessageWindowID=MessageWindow(aArray).ID then
          MainWindow.MessageEditField.AppendMessage(partnerNick,co_other_users,temp,co_default)
        end
        DrawerWindow.DR_MessageArrived(MessageWindow(aArray).ID)
      end
      MessageWindowLock.leave()
    end
    
  case 629'banned users
    
  case 640'direct browse request
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    temp=getLongIP2ShortIP(nthfield(what,chrb(32),2))
    j=val(nthfield(what,chrb(32),3))
    if me.isWinMX=false then
      if iPort<>0 then
        TS_PrepareBrowse(me.ID,me.nick,partnerNick,"",0,false,false)
        'direct browse accept
        me.command(641,partnerNick)
      elseif j<>0 then
        'direct browse accept
        me.command(641,partnerNick)
        TS_PrepareBrowse(me.ID,me.nick,partnerNick,temp,j,false,true)
      else
        'direct browse error
        me.command(642,partnerNick)
      end
    end
    
  case 641'direct browse accept
    partnerNick=getDefine(nthfield(what,chrb(32),1))
    temp=getLongIP2ShortIP(nthfield(what,chrb(32),2))
    j=val(nthfield(what,chrb(32),3))
    if me.isWinMX=false then
      if BrowseWindowLock.enter() then
        aArray=BR_SearchNickAndServerID2Array(partnerNick,me.ID)
        if aArray<>c_error then
          if j<>0 then
            TS_PrepareBrowse(me.ID,me.nick,partnerNick,temp,j,true,false)
          elseif iPort=0 then
            TS_DisconnectServerIDAndNickAndTransferType(me.ID,partnerNick,6)
          end
        end
        BrowseWindowLock.leave()
      end
    end
    
  case 642'direct browse error
    TS_DisconnectServerIDAndNickAndTransferType(me.ID,nthfield(what,chrb(32),1),6)
    
  case 748'login attempt
    
  case 750'server ping
    
  case 751'ping user
    if me.isWinMX=false then
      'pong response
      me.command(752,what)
    end
    
  case 752'pong response
    
  case 821'redirect client to another server
    me.isRedirect=true
    me.isRedirectAvailable=true
    what=lowercase(what)
    what=replaceb(what,chrb(32),chrb(58))
    me.address=nthfield(what,chrb(58),1)
    me.port=val(nthfield(what,chrb(58),2))
    if NetworkWindowLock.enter() then
      aIndex=MainWindow.NE_SearchID2ListIndex(me.ID)
      if aIndex<>c_error then
        MainWindow.NetworkListBox.cell(aIndex,NE_Comment)=what
      end
      NetworkWindowLock.leave()
    end
    
  case 822'cycle client
    
  case 824'emote
    partnerNick=getDefine(nthfield(what,chrb(32),2))
    temp=getDefine(replaceallb(nthfield(what,chrb(34),2),chrb(10),""))
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(nthfield(what,chrb(32),1),me.ID)
      if aArray<>c_error then
        select case partnerNick
        case "Server"
          getMessage(RoomWindow(aArray).textAndStyle,partnerNick,co_other_users,temp,co_default)
          if gRoomWindowID=RoomWindow(aArray).ID then
            MainWindow.RoomEditField.appendMessage(partnerNick,co_other_users,temp,co_default)
          end
        case me.nick
          getMessage(RoomWindow(aArray).textAndStyle,partnerNick,co_own_user,temp,co_default,true)
          if gRoomWindowID=RoomWindow(aArray).ID then
            MainWindow.RoomEditField.appendMessage(partnerNick,co_own_user,temp,co_default,true)
          end
          DrawerWindow.DR_MessageArrived(RoomWindow(aArray).ID)
          getSound(rs_click)
        else
          getMessage(RoomWindow(aArray).textAndStyle,partnerNick,co_other_users,temp,co_default,true)
          if gRoomWindowID=RoomWindow(aArray).ID then
            MainWindow.RoomEditField.appendMessage(partnerNick,co_other_users,temp,co_default,true)
          end
          DrawerWindow.DR_MessageArrived(RoomWindow(aArray).ID)
          getSound(rs_click)
        end
      end
      RoomWindowLock.leave()
    end
    
  case 825'user list entry
    partnerNick=nthfield(what,chrb(32),2)
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(nthfield(what,chrb(32),1),me.ID)
      if aArray<>c_error then
        i=RoomWindow(aArray).RO_SearchNick2Array(partnerNick)
        if i<>c_error then
          RoomWindow(aArray).RoomList(i).count=nthfield(what,chrb(32),3)
          RoomWindow(aArray).RoomList(i).speed=getShortSpeed(val(nthfield(what,chrb(32),4)))
          if gRoomWindowID=RoomWindow(aArray).ID then
            aIndex=MainWindow.RO_SearchNick2ListIndex(partnerNick)
            if aIndex<>c_error then
              MainWindow.RoomListBox.cell(aIndex,RO_Count)=RoomWindow(aArray).RoomList(i).count
              MainWindow.RoomListBox.cell(aIndex,RO_Speed)=RoomWindow(aArray).RoomList(i).speed
            end
          end
        end
      end
      RoomWindowLock.leave()
    end
    
  case 827'end of all channels
    
  case 828'all channel list
    
  case 830'end of list users in channel
    
  case 831'end of global user list
    
  case 832'global user list entry
    
  case 931'blocked file
    
  case 10112'show server links
    
  case 10115'show server stats
    
  case 10118'display client information statistics
    
  case 10208'send message to channel ops
    
  case 10209'change channel mode
    temp=nthfield(what,chrb(32),1)
    if RoomWindowLock.enter() then
      aArray=RO_SearchChannelAndServerID2Array(temp,me.ID)
      if aArray<>c_error then
        temp=getDefine(midb(what,lenb(temp)+2))
        getMessage(RoomWindow(aArray).textAndStyle,temp,co_notifications)
        if gRoomWindowID=RoomWindow(aArray).ID then
          MainWindow.RoomEditField.AppendMessage(temp,co_notifications)
        end
      end
      RoomWindowLock.leave()
    end
    
  end
  
  if (aType=0 or (aType=404 and me.lastType<>404) or aType=621 or aType=821) and lenb(what)<>0 then
    me.console(what)
  end
  me.lastType=aType
End Sub

OriginalOpenNapSocket.redirect:
Protected Sub redirect()
  dim what as string
  dim aIndex as integer
  if instrb(1,me.lookahead(),chrb(58))<>0 then
    me.isRedirectAvailable=true
    what=me.readall()
    what=lowercase(what)
    what=replaceb(what,chrb(13),"")
    what=replaceb(what,chrb(10),"")
    me.address=nthfield(what,chrb(58),1)
    me.port=val(nthfield(what,chrb(58),2))
    
    if NetworkWindowLock.enter() then
      aIndex=MainWindow.NE_SearchID2ListIndex(me.ID)
      if aIndex<>c_error then
        MainWindow.NetworkListBox.cell(aIndex,NE_Comment)=what
      end
      NetworkWindowLock.leave()
    end
    me.console(what)
  end
End Sub

OriginalOpenNapSocket.console:
Protected Sub console(inWhat as string)
  dim aArray as integer
  if ConsoleWindowLock.signal() then
    if ubound(ConsoleList)=500 then
      ConsoleList.remove 1
    end
    ConsoleList.append new OriginalConsoleList
    aArray=ubound(ConsoleList)
    if ConsoleList(aArray)<>nil then
      ConsoleList(aArray).serverID=me.ID
      ConsoleList(aArray).message=chrb(91)+me.network+chrb(93)+chrb(32)+inWhat+chrb(13)
      if gConsoleServerID=0 or gConsoleServerID=me.ID then
        if MainWindow.ConsoleEditField.scrollposition>500 then
          MainWindow.ConsoleEditField.text=""
        end
        MainWindow.ConsoleEditField.appendText(ConsoleList(aArray).message)
      end
    end
    ConsoleWindowLock.release()
  end
End Sub

OriginalOpenNapSocket.Error:
Sub Error()
  if me.isError then
    me.isFinalized=true
    removingObjects=removingObjects+1
    return
  end
  
  if me.isRedirect and me.isRedirectAvailable=false then
    me.redirect()
  end
  if me.isRedirect and me.isRedirectAvailable then
    me.isRedirect=false
    if me.isLoggedIn then
      me.isLoggedIn=false
      gOnline=gOnline-1
    end
    me.errorCode=0
    me.close()
    me.connect()
    return
  end
  
  if me.isLoggedIn then
    me.console(getString(s_server_disconnected))
    getSound(rs_error)
  end
  
  me.errorCode=me.lastErrorCode
  me.onError()
End Sub

OriginalOpenNapSocket.DataAvailable:
Sub DataAvailable()
  if me.isError then
    return
  end
  
  while me.isError=false and me.isConnected and me.checkAhead()
    me.recieved()
  wend
End Sub

OriginalOpenNapSocket.Connected:
Sub Connected()
  if me.isError then
    return
  end
  
  if NetworkWindowLock.enter() then
    if MainWindow.NE_SearchNameAndOnlineStatus2ListIndex(me.network)=c_error then
      if me.isRedirect=false then
        'login
        me.command(2,me.nick+chrb(32)+me.password+chrb(32)+getDefine(str(iPort))+chrb(32)+chrb(34)+iClient+chrb(34)+chrb(32)+getDefine(str(iSpeed)))
      end
    else
      me.onError()
    end
    NetworkWindowLock.leave()
  end
End Sub

OriginalPopupMenu.searchRowTag2ListIndex:
Function searchRowTag2ListIndex(inRowTag as variant) As integer
  dim i,j as integer
  j=me.listCount-1
  for i=0 to j
    if me.rowtag(i)<>nil and me.rowtag(i).equals(inRowTag) then
      return i
    end
  next
  return c_error
End Function

OriginalQuitTimer.Action:
Sub Action()
  if isWantQuit then
    me.mode=0
    quit
    
  elseif ubound(NapListSocket)=0 and ubound(OpenNapSocket)=0 and ubound(TransferSocket)=0 and isIndexing=false then
    isWantQuit=true
    
  else
    NS_DisconnectAll()
    OS_DisconnectAll()
    TS_DisconnectAll()
    NS_Remove()
    OS_Remove()
    TS_Remove()
    
  end
End Sub

OriginalRoomWindow.RO_SearchNick2Array:
Function RO_SearchNick2Array(inNick as string) As integer
  dim i,j as integer
  j=ubound(me.RoomList)
  for i=1 to j
    if strcomp(me.RoomList(i).nick,inNick,0)=0 then
      return i
    end
  next
  return c_error
End Function

OriginalRoundedCanvas.setEditField:
Sub setEditField(inEf as OriginalEditField)
  me.targetEditField=inEf
  me.targetEditField.top=me.top+2
  if me.hasSearchIcon then
    me.targetEditField.Left=me.left+gPicture(r_search_icon).width
    me.targetEditField.Width=me.width-gPicture(r_search_icon).width-gPicture(r_right).width
  elseif me.hasEmoIcon then
    me.targetEditField.Left=me.left+gPicture(r_left).width
    me.targetEditField.Width=me.width-gPicture(r_left).width-gPicture(r_emo_icon).width
  end
  me.targetEditField.height=me.height-4
  
  me.targetEditField.LockBottom=me.lockBottom
  me.targetEditField.LockTop=me.lockTop
  me.targetEditField.LockRight=me.lockRight
  me.targetEditField.LockLeft=me.lockLeft
End Sub

OriginalRoundedCanvas.MouseDown:
Function MouseDown(X As Integer, Y As Integer) As Boolean
  if me.enabled and ((me.hasSearchIcon and x>0 and x<gPicture(r_search_icon).width and y>0 and y<gPicture(r_search_icon).height) or (me.hasEmoIcon and x>me.width-gPicture(r_emo_icon).width and x<me.width and y>0 and y<gPicture(r_emo_icon).height)) then
    IconClicked()
    return true
  end
End Function

OriginalRoundedCanvas.Paint:
Sub Paint(g As Graphics)
  if me.hasSearchIcon then
    g.drawPicture gPicture(r_search_icon),0,0
    g.drawPicture gPicture(r_right),gPicture(r_search_icon).width+me.targetEditField.width,0
    g.drawPicture gPicture(r_middle),gPicture(r_search_icon).width,0,me.targetEditField.width,gPicture(r_middle).height,0,0,gPicture(r_middle).width,gPicture(r_middle).height
  elseif me.hasEmoIcon then
    g.drawPicture gPicture(r_left),0,0
    g.drawPicture gPicture(r_emo_icon),gPicture(r_left).width+me.targetEditField.width,0
    g.drawPicture gPicture(r_middle),gPicture(r_left).width,0,me.targetEditField.width,gPicture(r_middle).height, 0,0,gPicture(r_middle).width,gPicture(r_middle).height
  end
End Sub

OriginalToolbar.disable:
Protected Sub disable()
  dim buffer as picture
  buffer=newPicture(me.width,me.height,screen(0).depth)
  buffer.graphics.foreColor=rgb(0,0,0)
  buffer.graphics.fillRect 0,0,buffer.width,buffer.height
  buffer.mask.graphics.foreColor=rgb(255,255,255)
  buffer.mask.graphics.fillRect 0,0,buffer.width,buffer.height
  buffer.mask.graphics.textFont=me.graphics.textFont
  buffer.mask.graphics.textSize=me.graphics.textSize
  buffer.mask.graphics.foreColor=rgb(96,96,96)
  buffer.mask.graphics.drawstring me.caption,me.getStringX(),48
  buffer.graphics.drawPicture me.iconDisabled,me.getPictX(),5
  buffer.mask.graphics.drawPicture me.mask,me.getPictX(),5
  if me.hasMenu then
    buffer.mask.graphics.drawPicture me.arrowDisabled,me.getPictX()+me.icon.width-5,me.icon.height
  end
  
  me.graphics.drawPicture buffer,0,0
End Sub

OriginalToolbar.down:
Protected Sub down()
  dim buffer as picture
  buffer=newPicture(me.width,me.height,screen(0).depth)
  buffer.graphics.foreColor=rgb(0,0,0)
  buffer.graphics.fillRect 0,0,buffer.width,buffer.height
  buffer.mask.graphics.foreColor=rgb(255,255,255)
  buffer.mask.graphics.fillRect 0,0,buffer.width,buffer.height
  buffer.mask.graphics.textFont=me.graphics.textFont
  buffer.mask.graphics.textSize=me.graphics.textSize
  buffer.mask.graphics.foreColor=rgb(0,0,0)
  buffer.mask.graphics.drawstring me.caption,me.getStringX(),48
  buffer.graphics.drawPicture me.iconDown,me.getPictX(),5
  buffer.mask.graphics.drawPicture me.mask,me.getPictX(),5
  if me.hasMenu then
    buffer.mask.graphics.drawPicture me.arrowDown,me.getPictX()+me.icon.width-5,me.icon.height
  end
  
  me.graphics.drawPicture buffer,0,0
End Sub

OriginalToolbar.normal:
Protected Sub normal()
  dim buffer as picture
  buffer=newPicture(me.width,me.height,screen(0).depth)
  buffer.graphics.foreColor=rgb(0,0,0)
  buffer.graphics.fillRect 0,0,buffer.width,buffer.height
  buffer.mask.graphics.foreColor=rgb(255,255,255)
  buffer.mask.graphics.fillRect 0,0,buffer.width,buffer.height
  buffer.mask.graphics.textFont=me.graphics.textFont
  buffer.mask.graphics.textSize=me.graphics.textSize
  buffer.mask.graphics.foreColor=rgb(0,0,0)
  buffer.mask.graphics.drawstring me.caption,me.getStringX(),48
  buffer.graphics.drawPicture me.icon,me.getPictX(),5
  buffer.mask.graphics.drawPicture me.mask,me.getPictX(),5
  if me.hasMenu then
    buffer.mask.graphics.drawPicture me.arrowNormal,me.getPictX()+me.icon.width-5,me.icon.height
  end
  
  me.graphics.drawPicture buffer,0,0
End Sub

OriginalToolbar.setIcon:
Sub setIcon(inID as integer)
  me.icon=getPicture(inID)
  me.mask=getPicture(inID+1)
End Sub

OriginalToolbar.setPosition:
Sub setPosition(inObject as Object)
  me.width=max(me.graphics.stringWidth(me.caption)+4,48)
  if inObject isa RectControl then
    if me.lockRight then
      me.left=RectControl(inObject).left-me.width
    else
      me.left=RectControl(inObject).left+RectControl(inObject).width
    end
  elseif inObject isa OriginalWindow then
    if me.lockRight then
      me.left=OriginalWindow(inObject).width-me.width-6
    else
      me.left=6
    end
  end
End Sub

OriginalToolbar.getPictX:
Protected Function getPictX() As integer
  return (me.width-me.icon.width)\2
End Function

OriginalToolbar.getStringX:
Protected Function getStringX() As integer
  return (me.width-me.graphics.stringWidth(me.caption))\2
End Function

OriginalToolbar.MouseUp:
Sub MouseUp(X As Integer, Y As Integer)
  if X<me.width and X>0 and Y<me.height and Y>0 and me.hasMenu=false then
    Action()
    me.state=1'up
    me.refresh()
  else
    me.state=0'normal
    me.refresh()
  end
End Sub

OriginalToolbar.MouseDrag:
Sub MouseDrag(X As Integer, Y As Integer)
  if X<me.width and X>0 then
    if Y<me.height and Y>0 then
      if me.state=1 then'up
        me.state=2'down
        me.refresh()
      end
      
    else
      if me.state=2 then'down
        me.state=1'up
        me.refresh()
      end
    end
    
  else
    if me.state=2 then'down
      me.state=1'up
      me.refresh()
    end
  end
End Sub

OriginalToolbar.MouseDown:
Function MouseDown(X As Integer, Y As Integer) As Boolean
  me.state=2'down
  me.refresh()
  
  if me.hasMenu then
    Action()
  end
  return true
End Function

OriginalToolbar.Open:
Sub Open()
  dim buffer as picture
  me.graphics.textFont=c_system
  me.graphics.textSize=10
  Open()
  
  if me.hasMenu then
    me.arrowNormal=newPicture(5,4,screen(0).depth)
    me.arrowNormal.graphics.drawPicture getPicture(rp_down_allow),0,0
    me.arrowNormal.mask.graphics.drawPicture getPicture(rp_down_allow+1),0,0
    
    me.arrowDown=me.arrowNormal
    buffer=newPicture(me.arrowDown.width,me.arrowDown.height,screen(0).depth)
    buffer.graphics.foreColor=rgb(0,0,0)
    buffer.graphics.fillRect(0,0,me.icon.width,me.icon.height)
    buffer.mask.graphics.foreColor=rgb(135,135,135)
    buffer.mask.graphics.fillRect(0,0,me.arrowDown.width,me.arrowDown.height)
    me.arrowDown.graphics.drawPicture buffer,0,0
    
    me.arrowDisabled=me.arrowNormal
    buffer=newPicture(me.arrowDisabled.width,me.arrowDisabled.height,screen(0).depth)
    buffer.graphics.foreColor=rgb(255,255,255)
    buffer.graphics.fillRect(0,0,me.arrowDisabled.width,me.arrowDisabled.height)
    buffer.mask.graphics.foreColor=rgb(150,150,150)
    buffer.mask.graphics.fillRect(0,0,me.arrowDisabled.width,me.arrowDisabled.height)
    me.arrowDisabled.graphics.drawPicture buffer,0,0
  end
  
  me.iconDown=newPicture(me.icon.height,me.icon.width,screen(0).depth)
  me.iconDown.graphics.drawPicture me.icon,0,0
  buffer=newPicture(me.iconDown.width,me.iconDown.height,screen(0).depth)
  buffer.graphics.foreColor=rgb(0,0,0)
  buffer.graphics.fillRect(0,0,me.iconDown.width,me.iconDown.height)
  buffer.mask.graphics.foreColor=rgb(135,135,135)
  buffer.mask.graphics.fillRect(0,0,me.iconDown.width,me.iconDown.height)
  me.iconDown.graphics.drawPicture buffer,0,0
  
  me.iconDisabled=newPicture(me.icon.height,me.icon.width,screen(0).depth)
  me.iconDisabled.graphics.drawPicture me.icon,0,0
  buffer=newPicture(me.iconDisabled.width,me.iconDisabled.height,screen(0).depth)
  buffer.graphics.foreColor=rgb(255,255,255)
  buffer.graphics.fillRect(0,0,me.iconDisabled.width,me.iconDisabled.height)
  buffer.mask.graphics.foreColor=rgb(150,150,150)
  buffer.mask.graphics.fillRect(0,0,me.iconDisabled.width,me.iconDisabled.height)
  me.iconDisabled.graphics.drawPicture buffer,0,0
End Sub

OriginalToolbar.Paint:
Sub Paint(g As Graphics)
  if me.enabled then
    select case me.state
    case 0'normal
      me.normal()
    case 1'up
      me.normal()
    case 2'down
      me.down()
    end
    
  else
    me.disable()
    
  end
End Sub

OriginalTransferSocket.isFileSize:
Protected Function isFileSize() As Boolean
  dim temp as string
  temp=format(me.targetFileSize,"#")
  return strcomp(leftb(me.lookahead(),lenb(temp)),temp,0)=0
End Function

OriginalTransferSocket.isDownloadOK:
Protected Function isDownloadOK() As boolean
  if iDownloadFolder=nil or iDownloadFolder.exists=false then
    iDownloadFolder.CreateAsFolder()
  end
  if iIncompleteFolder=nil or iIncompleteFolder.exists=false then
    iIncompleteFolder.CreateAsFolder()
  end
  
  me.targetFolderItem=iIncompleteFolder.Child(getUTF8String(me.targetShortName))
  if me.targetFolderItem=nil or me.targetFolderItem.exists=false then
    me.targetBs=me.targetFolderItem.CreateBinaryFile("special/any")
    me.targetBs.close()
    writeComment(me.targetFolderItem,me.targetShortName)
  end
  
  me.targetBs=me.targetFolderItem.OpenAsBinaryFile(true)
  me.transferedSize=me.targetBs.length
  if me.transferedSize>iDownloadBuffer then
    me.transferedSize=me.transferedSize-iDownloadBuffer
  end
  me.targetBs.position=me.transferedSize
  me.targetBufferSize=me.transferedSize
  return true
exception
  me.showStatus(s_file_writing_error)
  me.onError()
End Function

OriginalTransferSocket.isUploadOK:
Protected Function isUploadOK() As boolean
  dim aArray as integer
  if LibraryLock.enter() then
    if me.isWinMX then
      aArray=OL_SearchFileName2Array(midb(me.targetFileName,4))
    else
      aArray=OL_SearchFileName2Array(me.targetFileName)
    end
    if aArray<>c_error then
      me.targetFolderItem=getPath2FolderItem(Library(aArray).fullPath)
    end
    LibraryLock.leave()
  end
  
  me.targetBs=me.targetFolderItem.OpenAsBinaryFile(false)
  me.targetBs.position=me.transferedSize
  me.targetBufferSize=me.transferedSize
  return true
exception
  me.showStatus(s_file_not_shared)
  me.write(getDefine("FILE NOT SHARED"))
  me.onError()
End Function

OriginalTransferSocket.onError:
Sub onError()
  if me.isError then
    return
  end
  me.isError=true
  
  if me.way=18 then'downloading
    me.download()
  elseif me.way=38 then'downloading
    me.downloadList()
  end
  
  if me.isConnected then
    me.disconnect()
  else
    me.close()
    me.isFinalized=true
    removingObjects=removingObjects+1
  end
End Sub

OriginalTransferSocket.upload:
Sub upload()
  dim aBufferSize as integer
  aBufferSize=min(iUploadBuffer,me.targetFileSize-me.transferedSize)-me.bytesLeftToSend
  if aBufferSize>0 and (iUploadBandwidth=0 or ((me.transferedSize-me.targetBufferSize)/1024)/((Microseconds-me.transferVelocityTime)/1000000)<=iUploadBandwidth/gUploading) then
    me.write(me.targetBs.read(aBufferSize))
    me.transferedSize=me.transferedSize+aBufferSize
    if me.transferedSize>=me.targetFileSize then
      me.way=29'finished
    end
  end
exception
  me.showStatus(s_file_reading_error)
  me.onError()
End Sub

OriginalTransferSocket.showProgress:
Function showProgress() As double
  dim aRemainTime,aIndex,aTransferedSize as integer
  dim aBandWidth as double
  aTransferedSize=me.transferedSize+me.bytesAvailable-me.bytesLeftToSend
  if aTransferedSize<>me.targetBufferSize then
    aBandWidth=((aTransferedSize-me.targetBufferSize)/1024)/((Microseconds-me.transferVelocityTime)/1000000)
    aRemainTime=((me.targetFileSize-aTransferedSize)/1024)/aBandWidth
    me.targetBufferSize=aTransferedSize
    me.transferVelocityTime=Microseconds
    if TransferWindowLock.enter() then
      aIndex=MainWindow.TR_SearchUDAndID2ListIndex(me.targetArray,me.ID)
      if aIndex<>c_error then
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Flag)=""
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_transfering
        MainWindow.TransferListBoxes(me.targetArray).cellRefresh(aIndex,TR_ShortName)
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Size)=getDefine(format(me.targetFileSize,"#"))
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_TransferedSize)=getDefine(format(aTransferedSize,"#"))
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_ShortSize)=getShortSize(aTransferedSize)+chrb(47)+getshortSize(me.targetFileSize)
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Rate)=format(aBandwidth,"#.0")+s__kb_s
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_RemainTime)=getRemainTime(aRemainTime)
      end
      TransferWindowLock.leave()
    end
  else
    if TransferWindowLock.enter() then
      aIndex=MainWindow.TR_SearchUDAndID2ListIndex(me.targetArray,me.ID)
      if aIndex<>c_error then
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Flag)=""
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_transfering
        MainWindow.TransferListBoxes(me.targetArray).cellRefresh(aIndex,TR_ShortName)
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Size)=getDefine(format(me.targetFileSize,"#"))
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_TransferedSize)=getDefine(format(aTransferedSize,"#"))
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_ShortSize)=getShortSize(aTransferedSize)+chrb(47)+getshortSize(me.targetFileSize)
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Rate)=s_stalled
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_RemainTime)=""
      end
      TransferWindowLock.leave()
    end
  end
  return aBandWidth
End Function

OriginalTransferSocket.downloadList:
Protected Sub downloadList()
  dim aIndex,aArray,aPosition,i as integer
  dim what as string
  if BrowseWindowLock.enter() then
    aArray=BR_SearchNickAndServerID2Array(me.partnerNick,me.serverID)
    if aArray<>c_error then
      do
        aPosition=instrb(1,me.lookahead(),chrb(10))
        if aPosition<>0 then
          what=""
          what=me.read(aPosition)
          if aPosition<>1 then
            BrowseWindow(aArray).BrowseList.append new OriginalBrowseList
            i=ubound(BrowseWindow(aArray).BrowseList)
            if BrowseWindow(aArray).BrowseList(i)<>nil then
              targetFileName=""
              targetFileName=getDefine(nthfield(what,chrb(34),2))
              targetShortName=""
              targetShortName=getDefine(nthfield(targetFileName,chrb(92),countfields(targetFileName,chrb(92))))
              BrowseWindow(aArray).BrowseList(i).fileName=targetFileName
              BrowseWindow(aArray).BrowseList(i).shortName=targetShortName
              BrowseWindow(aArray).BrowseList(i).mediaType=getMediaType(rightb(targetShortName,inbackStrb(1,targetShortName,".")))
              BrowseWindow(aArray).BrowseList(i).folderPath=leftb(targetFileName,lenb(targetFileName)-lenb(targetShortName))
              what=nthfield(what,chrb(34),3)
              BrowseWindow(aArray).BrowseList(i).bitRate=nthfield(what,chrb(32),4)+s__kbps
              what=getDefine(nthfield(what,chrb(32),3))
              BrowseWindow(aArray).totalSize=BrowseWindow(aArray).totalSize+val(what)
              BrowseWindow(aArray).BrowseList(i).size=what
              BrowseWindow(aArray).BrowseList(i).shortSize=getShortSize(val(what))
              
              if gBrowseWindowID=BrowseWindow(aArray).ID then
                MainWindow.BrowseListBox.addrow("")
                aIndex=MainWindow.BrowseListBox.lastIndex
                MainWindow.BrowseListBox.rowPicture(aIndex)=gMediaPicture(val(BrowseWindow(aArray).BrowseList(i).mediaType))
                MainWindow.BrowseListBox.cell(aIndex,BR_Icon)=BrowseWindow(aArray).BrowseList(i).mediaType
                MainWindow.BrowseListBox.cell(aIndex,BR_FileName)=targetFileName
                MainWindow.BrowseListBox.cell(aIndex,BR_ShortName)=targetShortName
                MainWindow.BrowseListBox.cell(aIndex,BR_Size)=what
                MainWindow.BrowseListBox.cell(aIndex,BR_ShortSize)=BrowseWindow(aArray).BrowseList(i).shortSize
                MainWindow.BrowseListBox.cell(aIndex,BR_BitRate)=BrowseWindow(aArray).BrowseList(i).bitRate
              end
            end
            
          else
            me.way=39'completed
            if gBrowseWindowID=BrowseWindow(aArray).ID then
              MainWindow.FolderListBox.cell(0,BR_Icon)=getShortSize(BrowseWindow(aArray).totalSize)
              DrawerWindow.DR_UpdateResults(BrowseWindow(aArray).ID,me.partnerNick,MainWindow.BrowseListBox.listCount,ubound(BrowseWindow(aArray).BrowseList))
            else
              DrawerWindow.DR_Update(BrowseWindow(aArray).ID,me.partnerNick,str(ubound(BrowseWindow(aArray).BrowseList)))
            end
            me.onError()
          end
        end
      loop until aPosition=0 or me.isConnected=false
    else
      me.onError()
    end
    BrowseWindowLock.leave()
  end
End Sub

OriginalTransferSocket.uploadList:
Protected Sub uploadList()
  dim i,j as integer
  if LibraryLock.enter() then
    j=ubound(Library)
    for i=1 to j
      me.write(chrb(34)+Library(i).filePath+Library(i).shortName+chrb(34)+chrb(32)+getDefine("0")+chrb(32)+Library(i).size+chrb(32)+Library(i).bitRate+chrb(32)+Library(i).sampleRate+chrb(32)+Library(i).time+chrb(10))
    next
    LibraryLock.leave()
  end
  me.way=49'finished
  me.write(chrb(10))
End Sub

OriginalTransferSocket.remove:
Sub remove()
  dim aIndex as integer
  if me.targetBs<>nil then
    me.targetBs.close()
  end
  
  if TransferWindowLock.signal() then
    aIndex=MainWindow.TR_SearchUDAndID2ListIndex(me.targetArray,me.ID)
    if aIndex<>c_error then
      select case me.way
      case 19'completed
        if iClearDownload then
          MainWindow.TransferListBoxes(me.targetArray).removeRow(aIndex)
          getSound(rs_done)
        end
      case 29'completed
        if iClearUpload then
          MainWindow.TransferListBoxes(me.targetArray).removeRow(aIndex)
          getSound(rs_done)
        end
      end
    end
    TransferWindowLock.release()
  end
  
  if TransferWindowLock.enter() then
    aIndex=MainWindow.TR_SearchUDAndID2ListIndex(me.targetArray,me.ID)
    if aIndex<>c_error then
      if val(MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Flag))=TF_Paused then
      elseif MainWindow.TransferListBoxes(me.targetArray).cellCheck(aIndex,TR_Retry) and me.way<>19 then
        gDownloadRetrying=gDownloadRetrying+1
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Timer)="120"
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Flag)=str(TF_RetryingIn)
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(120))
      else
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Flag)=""
        select case me.way
        case 10'not connected
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_listener_timeout
        case 12'connecting
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_failed_to_connect
        case 15'negociating
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_disconnected
        case 16'negociating
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_disconnected
        case 18'downloading
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_disconnected
          getSound(rs_error)
        case 19'completed
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_completed
          MainWindow.TransferListBoxes(me.targetArray).cellCheck(aIndex,TR_Retry)=false
          MainWindow.TransferListBoxes(me.targetArray).RowPicture(aIndex)=gPicture(r_done)
          getSound(rs_done)
        case 20'not connected
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_listener_timeout
        case 22'connecting
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_failed_to_connect
        case 25'negociating
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_disconnected
        case 26'negociating
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_disconnected
        case 28'uploading
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_disconnected
          getSound(rs_error)
        case 29'completed
          MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=s_completed
          MainWindow.TransferListBoxes(me.targetArray).RowPicture(aIndex)=gPicture(r_done)
          getSound(rs_done)
        end
      end
      MainWindow.TransferListBoxes(me.targetArray).cellRefresh(aIndex,TR_ShortName)
      MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Size)=getDefine(format(me.targetFileSize,"#"))
      MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_TransferedSize)=getDefine(format(me.transferedSize,"#"))
      if me.transferedSize=me.targetFileSize then
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_ShortSize)=getShortSize(me.transferedSize)
      else
        MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_ShortSize)=getShortSize(me.transferedSize)+chrb(47)+getshortSize(me.targetFileSize)
      end
      MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Rate)=""
      MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_RemainTime)=""
      MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_ID)=""
    elseif iPromptRemoving and me.way=18 then'downlading
      me.targetFolderItem.delete()
    end
    TransferWindowLock.leave()
  end
  
  select case me.way
  case 18'downloading
    gDownloading=gDownloading-1
    'download completed
    OS_Broadcast(219,"")
  case 19'completed
    me.downloadCompleted()
    gDownloading=gDownloading-1
    'download completed
    OS_Broadcast(219,"")
    DrawerWindow.MenuListBox.rowPicture(5)=gPicture(r_done)
  case 28'uploading
    gUploading=gUploading-1
    'upload completed
    OS_Broadcast(221,"")
  case 29'completed
    gUploading=gUploading-1
    'upload completed
    OS_Broadcast(221,"")
    DrawerWindow.MenuListBox.rowPicture(6)=gPicture(r_done)
  end
End Sub

OriginalTransferSocket.showStatus:
Protected Sub showStatus(inWhat as string)
  dim aIndex as integer
  if TransferWindowLock.enter() then
    aIndex=MainWindow.TR_SearchUDAndID2ListIndex(me.targetArray,me.ID)
    if aIndex<>c_error then
      MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Flag)=""
      MainWindow.TransferListBoxes(me.targetArray).cell(aIndex,TR_Status)=inWhat
      MainWindow.TransferListBoxes(me.targetArray).cellRefresh(aIndex,TR_ShortName)
    end
    TransferWindowLock.leave()
  end
End Sub

OriginalTransferSocket.download:
Sub download()
  dim aBuffer as string
  if me.bytesAvailable<>0 then
    aBuffer=me.readall()
    me.targetBs.write(aBuffer)
    me.transferedSize=me.transferedSize+lenb(aBuffer)
    if me.transferedSize>=me.targetFileSize then
      me.way=19'completed
      me.onError()
    end
  end
exception
  me.showStatus(s_file_writing_error)
  me.onError()
End Sub

OriginalTransferSocket.recieved:
Protected Sub recieved()
  dim what as string
  dim aArray as integer
  select case me.transferType
  case 0'UNKNOWN
    what=me.lookahead()
    if instrb(1,what,"SENDLIST ")<>0 then
      what=""
      what=me.read(lenb("SENDLIST "))
      what=""
      what=me.lookahead()
      me.partnerNick=me.read(instrb(1,what,chrb(10)),gTextEncoding)
      me.partnerNick=leftb(me.partnerNick,lenb(me.partnerNick)-1)
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchNickAndTransferTypeAndWay2Array(me.partnerNick,6,30)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          TransferSocket(aArray).onError()
          me.transferType=6'FWDL LIST
          me.way=38'downloading
          me.downloadList()
        else
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    elseif strcomp(what,"SENDLIST",0)=0 then
      what=""
      what=me.read(lenb("SENDLIST"))
      me.transferType=6'FWDL LIST
      me.way=35'negociating
      
    elseif instrb(1,what,"SENDLIST")<>0 then
      what=""
      what=me.read(lenb("SENDLIST"))
      what=""
      what=me.lookahead()
      me.partnerNick=me.read(instrb(1,what,chrb(10)),gTextEncoding)
      me.partnerNick=leftb(me.partnerNick,lenb(me.partnerNick)-1)
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchNickAndTransferTypeAndWay2Array(me.partnerNick,6,30)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          TransferSocket(aArray).onError()
          me.transferType=6'FWDL LIST
          me.way=38'downloading
          me.downloadList()
        else
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    elseif strcomp(what,"GETLIST",0)=0 then
      what=""
      what=me.read(lenb("GETLIST"))
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchTransferTypeAndWay2Array(7,40)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          TransferSocket(aArray).onError()
          me.transferType=7'UL LIST
          me.way=48'uploading
          me.write(me.nick+chrb(10))
          me.uploadList()
        else
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    elseif strcomp(what,"GET",0)=0 then
      what=""
      what=me.read(lenb("GET"))
      me.transferType=3'UL
      me.way=25'negociating
      
    elseif instrb(1,what,"GET")<>0 then
      what=""
      what=me.read(lenb("GET"))
      what=""
      what=me.readall()
      me.partnerNick=getDefine(nthfield(what,chrb(32),1))
      me.targetFileName=getDefine(nthfield(what,chrb(34),2))
      me.transferedSize=val(nthfield(what,chrb(32),countfields(what,chrb(32))))
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchNickAndFileNameAndTransferTypeAndWay2Array(me.partnerNick,me.targetFileName,3,20)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          me.targetShortName=TransferSocket(aArray).targetShortName
          me.targetFileSize=TransferSocket(aArray).targetFileSize
          me.isWinMX=TransferSocket(aArray).isWinMX
          me.ID=TransferSocket(aArray).ID
          me.targetArray=TransferSocket(aArray).targetArray
          TransferSocket(aArray).ID=-1
          TransferSocket(aArray).transferType=0'UNKNOWN
          TransferSocket(aArray).way=0'unknown
          TransferSocket(aArray).onError()
          me.transferType=3'UL
          me.way=26'negociating
          if me.isUploadOK() then
            me.write(getDefine(format(me.targetFileSize,"#")))
            me.startUpload()
          end
        else
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    elseif strcomp(what,"SEND",0)=0 then
      what=""
      what=me.read(lenb("SEND"))
      me.transferType=2'FWDL
      me.way=15'negociating
      
    elseif instrb(1,what,"SEND")<>0 then
      what=""
      what=me.read(lenb("SEND"))
      what=""
      what=me.readall()
      me.partnerNick=getDefine(nthfield(what,chrb(32),1))
      me.targetFileName=getDefine(nthfield(what,chrb(34),2))
      me.targetFileSize=val(nthfield(what,chrb(32),countfields(what,chrb(32))))
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchNickAndFileNameAndFileSizeAndTransferTypeAndWay2Array(me.partnerNick,me.targetFileName,me.targetFileSize,2,10)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          me.targetShortName=TransferSocket(aArray).targetShortName
          me.isWinMX=TransferSocket(aArray).isWinMX
          me.ID=TransferSocket(aArray).ID
          me.targetArray=TransferSocket(aArray).targetArray
          TransferSocket(aArray).ID=-1
          TransferSocket(aArray).transferType=0'UNKNOWN
          TransferSocket(aArray).way=0'unknown
          TransferSocket(aArray).onError()
          me.transferType=2'FWDL
          me.way=16'negociating
          if me.isDownloadOK() then
            me.write(getDefine(format(me.transferedSize,"#")))
            me.startDownload()
          end
        else
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    else
      me.invalidRequest()
    end
    
  case 1'DL
    select case me.way
    case 13'connected
      what=me.readall()
      if strcomp(what,"1",0)=0 then
        if me.isDownloadOK() then
          me.way=16'negociating
          me.write(getDefine("GET"))
          me.write(me.nick+chrb(32)+chrb(34)+me.targetFileName+chrb(34)+chrb(32)+getDefine(format(me.transferedSize,"#")))
        end
      else
        me.invalidRequest()
      end
    case 16'negociating
      if me.isFileSize() then
        what=me.read(lenb(format(me.targetFileSize,"#")))
        me.startDownload()
      else
        me.invalidRequest()
      end
    end
    
  case 2'FWDL
    select case me.way
    case 15'negociating
      what=me.readall()
      me.partnerNick=getDefine(nthfield(what,chrb(32),1))
      me.targetFileName=getDefine(nthfield(what,chrb(34),2))
      me.targetFileSize=val(nthfield(what,chrb(32),countfields(what,chrb(32))))
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchNickAndFileNameAndFileSizeAndTransferTypeAndWay2Array(me.partnerNick,me.targetFileName,me.targetFileSize,2,10)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          me.targetShortName=TransferSocket(aArray).targetShortName
          me.isWinMX=TransferSocket(aArray).isWinMX
          me.ID=TransferSocket(aArray).ID
          me.targetArray=TransferSocket(aArray).targetArray
          TransferSocket(aArray).ID=-1
          TransferSocket(aArray).transferType=0'UNKNOWN
          TransferSocket(aArray).way=0'unknown
          TransferSocket(aArray).onError()
          me.way=16'negociating
          if me.isDownloadOK() then
            me.write(getDefine(format(me.transferedSize,"#")))
            me.startDownload()
          end
        else
          me.transferType=0'UNKNOWN
          me.way=0'unknown
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    end
    
  case 3'UL
    select case me.way
    case 25'negociating
      what=me.readall()
      me.partnerNick=getDefine(nthfield(what,chrb(32),1))
      me.targetFileName=getDefine(nthfield(what,chrb(34),2))
      me.transferedSize=val(nthfield(what,chrb(32),countfields(what,chrb(32))))
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchNickAndFileNameAndTransferTypeAndWay2Array(me.partnerNick,me.targetFileName,3,20)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          me.targetShortName=TransferSocket(aArray).targetShortName
          me.targetFileSize=TransferSocket(aArray).targetFileSize
          me.isWinMX=TransferSocket(aArray).isWinMX
          me.ID=TransferSocket(aArray).ID
          me.targetArray=TransferSocket(aArray).targetArray
          TransferSocket(aArray).ID=-1
          TransferSocket(aArray).transferType=0'UNKNOWN
          TransferSocket(aArray).way=0'unknown
          TransferSocket(aArray).onError()
          me.way=26'negociating
          if me.isUploadOK() then
            me.write(getDefine(format(me.targetFileSize,"#")))
            me.startUpload()
          end
        else
          me.transferType=0'UNKNOWN
          me.way=0'unknown
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    end
    
  case 4'FWUL
    select case me.way
    case 23'connected
      what=me.readall()
      if strcomp(what,"1",0)=0 then
        me.way=26'negociating
        me.write(getDefine("SEND"))
        me.write(me.nick+chrb(32)+chrb(34)+me.targetFileName+chrb(34)+chrb(32)+getDefine(format(me.targetFileSize,"#")))
      else
        me.invalidRequest()
      end
    case 26'negociating
      what=me.readall()
      me.transferedSize=val(what)
      if me.isUploadOK() then
        me.startUpload()
      end
    end
    
  case 5'DL LIST
    select case me.way
    case 33'connected
      what=me.readall()
      if strcomp(what,"1",0)=0 then
        me.way=35'negociating
        me.write(getDefine("GETLIST"))
      else
        me.invalidRequest()
      end
    case 35'negociating
      what=me.lookahead()
      if instrb(1,what,me.partnerNick+chrb(10))<>0 then
        what=""
        what=me.read(lenb(me.partnerNick)+1)
        me.way=38'downloading
        me.downloadList()
      else
        me.invalidRequest()
      end
    end
    
  case 6'FWDL LIST
    select case me.way
    case 35'negociating
      what=me.lookahead()
      me.partnerNick=me.read(instrb(1,what,chrb(10)),gTextEncoding)
      me.partnerNick=leftb(me.partnerNick,lenb(me.partnerNick)-1)
      
      if TransferSocketLock.enter() then
        aArray=TS_SearchNickAndTransferTypeAndWay2Array(me.partnerNick,6,30)
        if aArray<>c_error then
          me.serverID=TransferSocket(aArray).serverID
          me.nick=TransferSocket(aArray).nick
          TransferSocket(aArray).onError()
          me.way=38'downloading
          me.downloadList()
        else
          me.invalidRequest()
        end
        TransferSocketLock.leave()
      end
      
    end
    
  case 7'UL LIST
    
  case 8'FWUL LIST
    select case me.way
    case 43'connected
      what=me.readall()
      if strcomp(what,"1",0)=0 then
        me.write(getDefine("SENDLIST")+chrb(32)+me.nick+chrb(10))
        me.way=48'uploading
        me.uploadList()
      else
        me.invalidRequest()
      end
    end
  end
End Sub

OriginalTransferSocket.downloadCompleted:
Protected Sub downloadCompleted()
  dim fi as folderItem
  dim aType as string
  aType=getMediaType(rightb(me.targetShortName,inbackStrb(1,me.targetShortName,".")))
  if aType=rm_music and iMusicFolder<>nil and iMusicFolder.exists then
    fi=iMusicFolder
  elseif aType=rm_movies and iMoviesFolder<>nil and iMoviesFolder.exists then
    fi=iMoviesFolder
  elseif aType=rm_pictures and iPicturesFolder<>nil and iPicturesFolder.exists then
    fi=iPicturesFolder
  elseif iDownloadFolder<>nil and iDownloadFolder.exists then
    fi=iDownloadFolder
  else
    return
  end
  
  if lenb(me.targetFolderName)<>0 then
    fi=fi.child(getUTF8String(me.targetFolderName))
    if fi=nil or fi.exists=false then
      fi.createAsFolder()
    end
    if fi<>nil and fi.exists then
      moveFolderItem(me.targetFolderItem,fi)
    end
  else
    moveFolderItem(me.targetFolderItem,fi)
  end
  
  if isShareDownloads then
    OL_Update()
  end
exception
End Sub

OriginalTransferSocket.startDownload:
Protected Sub startDownload()
  me.way=18'downloading
  gDownloading=gDownloading+1
  'start download
  OS_Broadcast(218,"")
End Sub

OriginalTransferSocket.startUpload:
Protected Sub startUpload()
  me.way=28'uploading
  gUploading=gUploading+1
  'start upload
  OS_Broadcast(220,"")
End Sub

OriginalTransferSocket.invalidRequest:
Protected Sub invalidRequest()
  me.write(getDefine("INVALID REQUEST"))
  me.onError()
End Sub

OriginalTransferSocket.Connected:
Sub Connected()
  if me.isError then
    return
  end
  
  me.transferVelocityTime=Microseconds
  select case me.transferType
  case 0'UNKNOWN
    me.write(getDefine("1"))
    
  case 1'DL
    me.way=13'connected
    
  case 4'FWUL
    me.way=23'connected
    
  case 5'DL LIST
    me.way=33'connected
    
  case 8'FWUL LIST
    me.way=43'connected
  end
End Sub

OriginalTransferSocket.Error:
Sub Error()
  if me.isError then
    me.isFinalized=true
    removingObjects=removingObjects+1
    return
  end
  
  me.onError()
End Sub

OriginalTransferSocket.SendComplete:
Sub SendComplete(userAborted as Boolean)
  if me.isError then
    return
  end
  
  if me.way=28 then'uploading
    if gDownloading<>0 then
      TS_Download()
    end
    me.upload()
  end
End Sub

OriginalTransferSocket.DataAvailable:
Sub DataAvailable()
  if me.isError then
    return
  end
  
  if me.way=18 then'downloading
    me.download()
  elseif me.way=38 then'downloading
    me.downloadList()
  else
    me.recieved()
  end
End Sub

OriginalTransferThread.Run:
Sub Run()
  dim i,j as integer
  dim downloadBandwidth,uploadBandwidth as double
  if TransferSocketLock.enter() then
    j=ubound(TransferSocket)
    for i=1 to j
      if TransferSocket(i).isFinalized=false then
        select case TransferSocket(i).way
        case 18'downloading
          downloadBandwidth=downloadBandwidth+TransferSocket(i).showProgress()
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 28'uploading
          uploadBandwidth=uploadBandwidth+TransferSocket(i).showProgress()
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 10'not connected
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 20'not connected
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 30'not connected
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 40'not connected
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 12'connecting
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 22'connecting
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 32'connecting
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        case 42'connecting
          if Microseconds-TransferSocket(i).transferVelocityTime>60000000 then
            TransferSocket(i).onError()
          end
        end
      end
    next
    TransferSocketLock.leave()
  end
  
  MainWindow.TR_RefreshDownloadQueue()
  MainWindow.TR_RefreshUploadQueue()
  if iShowBandwidth and gDownloading<>0 then
    DrawerWindow.MenuListBox.cell(5,DR_Name)=format(downloadBandwidth,"#.0")+s__kb_s
  else
    DrawerWindow.MenuListBox.cell(5,DR_Name)=s_downloads
  end
  if iShowBandwidth and gUploading<>0 then
    DrawerWindow.MenuListBox.cell(6,DR_Name)=format(uploadBandwidth,"#.0")+s__kb_s
  else
    DrawerWindow.MenuListBox.cell(6,DR_Name)=s_uploads
  end
End Sub

OriginalTransferTimer.Action:
Sub Action()
  dim i,j as integer
  if removingObjects<>0 then
    SE_Remove()
    BR_Remove()
    ME_Remove()
    RO_Remove()
    NS_Remove()
    OS_Remove()
    TS_Remove()
  end
  
  if TransferSocketLock.enter() then
    j=ubound(TransferSocket)
    if gDownloading<>0 then
      for i=1 to j
        if TransferSocket(i).isFinalized=false and TransferSocket(i).way=18 then'downloading
          TransferSocket(i).download()
        end
      next
    end
    if gUploading<>0 then
      for i=1 to j
        if TransferSocket(i).isFinalized=false and TransferSocket(i).way=28 then'uploading
          TransferSocket(i).upload()
        end
      next
    end
    TransferSocketLock.leave()
  end
  
  if ticks-me.refreshTime>=300 then
    me.refreshTime=ticks
    TransferThread.run()
  end
End Sub

OriginalWindow.setTitle:
Sub setTitle(inTitle as string)
  me.title=inTitle
  me.menu.text=inTitle
End Sub

OriginalWindow.Close:
Sub Close()
  me.menu.close()
  Close()
End Sub

OriginalWindow.Open:
Sub Open()
  me.menu=new windowMenus
  me.menu.text=me.title
  Open()
End Sub

OriginalWindow.Activate:
Sub Activate()
  if me.menu<>nil then
    me.menu.checked=true
  end
  Activate()
End Sub

OriginalWindow.Deactivate:
Sub Deactivate()
  me.menu.checked=false
  Deactivate()
End Sub

BrowseModule.BR_Append:
Function BR_Append(inPartnerNick as String,inServerID as Integer,inNetwork as string) As Integer
  dim aArray as integer
  BrowseWindow.append new OriginalBrowseWindow
  aArray=ubound(BrowseWindow)
  if BrowseWindow(aArray)<>nil then
    gTaskID=gTaskID+1
    BrowseWindow(aArray).ID=gTaskID
    BrowseWindow(aArray).title=inPartnerNick+getDefine(" - ")+inNetwork
    BrowseWindow(aArray).partnerNick=inPartnerNick
    BrowseWindow(aArray).serverID=inServerID
    BrowseWindow(aArray).network=inNetwork
    return aArray
  else
    return c_error
  end
End Function

BrowseModule.BR_SearchNickAndServerID2Array:
Function BR_SearchNickAndServerID2Array(inPartnerNick as String,inServerID as integer) As Integer
  dim i,j as integer
  j=ubound(BrowseWindow)
  for i=1 to j
    if BrowseWindow(i).isFinalized=false and strcomp(BrowseWindow(i).partnerNick,inPartnerNick,0)=0 and BrowseWindow(i).serverID=inServerID then
      return i
    end
  next
  return c_error
End Function

BrowseModule.BR_SearchID2Array:
Function BR_SearchID2Array(inID as integer) As Integer
  dim i,j as integer
  j=ubound(BrowseWindow)
  for i=1 to j
    if BrowseWindow(i).isFinalized=false and BrowseWindow(i).ID=inID then
      return i
    end
  next
  return c_error
End Function

BrowseModule.BR_NewBrowse:
Sub BR_NewBrowse(inServerID as integer,inPartnerNick as string,inIsDirectBrowse as boolean)
  dim aOpenNapArray,aArray,aID as integer
  if OpenNapSocketLock.enter() then
    aOpenNapArray=OS_SearchID2Array(inServerID)
    if aOpenNapArray<>c_error then
      
      if BrowseWindowLock.enter() then
        aArray=BR_SearchNickAndServerID2Array(inPartnerNick,inServerID)
        if aArray=c_error then
          aArray=BR_Append(inPartnerNick,inServerID,OpenNapSocket(aOpenNapArray).network)
          if aArray<>c_error then
            DrawerWindow.DR_Append(inPartnerNick,BrowseWindow(aArray).ID,c_browse_window)
            if inIsDIrectBrowse and OpenNapSocket(aOpenNapArray).isWinMX=false then
              if iPort<>0 then
                TS_PrepareBrowse(inServerID,OpenNapSocket(aOpenNapArray).nick,inPartnerNick,"",0,true,true)
              end
              'direct browse request
              OpenNapSocket(aOpenNapArray).command(640,inPartnerNick)
            else
              'browse request
              OpenNapSocket(aOpenNapArray).command(211,inPartnerNick)
            end
          end
        end
        if aArray<>c_error then
          aID=BrowseWindow(aArray).ID
        end
        BrowseWindowLock.leave()
      end
      
      if aID<>0 then
        DrawerWindow.DR_SelectMenu(aID)
      end
    end
    OpenNapSocketLock.leave()
  end
End Sub

BrowseModule.BR_Remove:
Sub BR_Remove()
  dim i as integer
  if BrowseWindowLock.signal() then
    for i=ubound(BrowseWindow) downto 1
      if BrowseWindow(i).isFinalized then
        BrowseWindow(i)=nil
        BrowseWindow.Remove i
        removingObjects=removingObjects-1
      end
    next
    BrowseWindowLock.release()
  end
End Sub

BrowseModule.BR_Offline:
Sub BR_Offline(inServerID as integer)
  dim i,j as integer
  if BrowseWindowLock.enter() then
    j=ubound(BrowseWindow)
    for i=1 to j
      if BrowseWindow(i).isFinalized=false and BrowseWindow(i).serverID=inServerID then
        BrowseWindow(i).serverID=0
      end
    next
    BrowseWindowLock.leave()
  end
End Sub

BrowseModule.BR_Online:
Sub BR_Online(inServerID as integer,inNetwork as string)
  dim i,j as integer
  if BrowseWindowLock.enter() then
    j=ubound(BrowseWindow)
    for i=1 to j
      if BrowseWindow(i).isFinalized=false and strcomp(BrowseWindow(i).network,inNetwork,0)=0 then
        BrowseWindow(i).serverID=inServerID
      end
    next
    BrowseWindowLock.leave()
  end
End Sub

ChannelModule.OC_Get:
Sub OC_Get()
  dim tis as TextInputStream
  dim aArray as integer
  dim temp as string
  if ChannelLock.enter() then
    tis=PreferencesFolder.child(c_2get).child(c_channels).openAsTextFile()
    do until tis.eof
      temp=""
      temp=tis.readline()
      Channel.append new OriginalChannel
      aArray=ubound(Channel)
      if Channel(aArray)<>nil then
        Channel(aArray).network=getDefine(nthfield(temp,chrb(9),1))
        Channel(aArray).channel=getDefine(nthfield(temp,chrb(9),2))
      end
    loop
    tis.close()
    ChannelLock.leave()
  end
exception
  if tis<>nil then
    tis.close()
  end
  ChannelLock.leave()
End Sub

ChannelModule.OC_Write:
Sub OC_Write()
  dim tos as TextOutputStream
  dim i,j as integer
  if ChannelLock.enter() then
    tos=PreferencesFolder.child(c_2get).Child(c_channels).CreateTextFile()
    j=ubound(Channel)
    for i=1 to j
      tos.writeline(Channel(i).network+chrb(9)+Channel(i).channel)
    next
    tos.close()
    ChannelLock.leave()
  end
exception
  if tos<>nil then
    tos.close()
  end
  ChannelLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

ChannelModule.OC_SearchNetworkAndChannel2Array:
Function OC_SearchNetworkAndChannel2Array(inNetwork as string,inChannel as string) As Integer
  dim i,j as integer
  j=ubound(Channel)
  for i=1 to j
    if strcomp(Channel(i).network,inNetwork,0)=0 and strcomp(Channel(i).channel,inChannel,0)=0 then
      return i
    end
  next
  return c_error
End Function

GeneralModule.getRandomString:
Function getRandomString(inIsMD5 as boolean) As String
  dim i as integer
  dim temp as string
  if inIsMD5 then
    for i=1 to 32
      if Round(Rnd)=0 then
        temp=temp+chrb(97+Rnd*6)
      else
        temp=temp+chrb(48+Rnd*10)
      end
    next
    
  else
    for i=1 to 8
      temp=temp+chrb(97+Rnd*26)
    next
  end
  return getDefine(temp)
End Function

GeneralModule.getShortSize:
Function getShortSize(inNumber as double) As String
  if inNumber<1024 then
    return getDefine(format(inNumber,"#")+s__b)
  elseif inNumber<1048576 then
    return getDefine(format(inNumber/1024,"#.0")+s__kb)
  elseif inNumber<1073741824 then
    return getDefine(format(inNumber/1048576,"#.0")+s__mb)
  elseif inNumber<1099511627776 then
    return getDefine(format(inNumber/1073741824,"#.0")+s__gb)
  elseif inNumber<1125899906842624 then
    return getDefine(format(inNumber/1099511627776,"#.0")+s__tb)
  elseif inNumber<1152921504606846976 then
    return getDefine(format(inNumber/1125899906842624,"#.0")+s__pb)
  end
End Function

GeneralModule.getUptime:
Function getUptime(inTime as Integer) As String
  return format(inTime\60\60,"#\:")+format(inTime\60 mod 60,"00")
End Function

GeneralModule.getShortSpeed:
Function getShortSpeed(inType as Integer) As String
  select case inType
  case 0
    return s_unknown
  case 1
    return s_14_kbps
  case 2
    return s_28_kbps
  case 3
    return s_33_kbps
  case 4
    return s_56_kbps
  case 5
    return s_isdn
  case 6
    return s_dual_isdn
  case 7
    return s_cable
  case 8
    return s_dsl
  case 9
    return s_t1
  case 10
    return s_t3
  end
End Function

GeneralModule.getTimeStamp:
Function getTimeStamp() As String
  dim d as date
  if iTimeStamp then
    d=new Date
    return getDefine(lowercase(d.shortTime))+chrb(32)
  end
End Function

GeneralModule.getRandomNumber:
Function getRandomNumber(inFirst as integer,inLast as integer) As Integer
  return inFirst+Rnd*(inLast-inFirst+1)
End Function

GeneralModule.getString:
Function getString(inWhat as string) As String
  return convertEncoding(inWhat,gTextEncoding)
End Function

GeneralModule.getDefine:
Function getDefine(inWhat as string) As String
  return defineEncoding(inWhat,gTextEncoding)
End Function

GeneralModule.getPortRange:
Sub getPortRange()
  dim aFirst,aLast as integer
  if instrb(1,iPortRange,"-")<>0 then
    aFirst=val(nthfield(iPortRange,"-",1))
    aLast=val(nthfield(iPortRange,"-",2))
    if aFirst<1024 or aFirst>32768 or aLast<1024 or aLast>32768 or aFirst>aLast then
      iPort=0
      iPortRange=str(iPort)
    else
      iPort=getRandomNumber(aFirst,aLast)
      iPortRange=str(aFirst)+"-"+str(aLast)
    end
  else
    aFirst=val(iPortRange)
    if (aFirst>0 and aFirst<1024) or aFirst>32768 then
      aFirst=0
    end
    iPort=aFirst
    iPortRange=str(iPort)
  end
End Sub

GeneralModule.initProperties:
Sub initProperties()
  LibraryLock=new OriginalCriticalSection
  ChannelLock=new OriginalCriticalSection
  NapListSocketLock=new OriginalCriticalSection
  OpenNapSocketLock=new OriginalCriticalSection
  TransferSocketLock=new OriginalCriticalSection
  SearchWindowLock=new OriginalCriticalSection
  BrowseWindowLock=new OriginalCriticalSection
  MessageWindowLock=new OriginalCriticalSection
  RoomWindowLock=new OriginalCriticalSection
  NetworkWindowLock=new OriginalCriticalSection
  ConsoleWindowLock=new OriginalCriticalSection
  ChannelWindowLock=new OriginalCriticalSection
  WhoisWindowLock=new OriginalCriticalSection
  UserListWindowLock=new OriginalCriticalSection
  TransferWindowLock=new OriginalCriticalSection
  DrawerWindowLock=new OriginalCriticalSection
End Sub

GeneralModule.getBoolean:
Function getBoolean(inIsValue as boolean) As String
  if inIsValue then
    return getDefine("true")
  else
    return getDefine("false")
  end
End Function

GeneralModule.getRetryTime:
Function getRetryTime(inTime as integer) As String
  dim minute,second as integer
  minute=inTime\60 mod 60
  second=inTime mod 60
  
  if minute<>0 and second<>0 then
    return str(minute)+s__minute+chrb(32)+str(second)+s__second
  elseif minute<>0 then
    return str(minute)+s__minute
  elseif second<>0 then
    return str(second)+s__second
  end
End Function

GeneralModule.getHeader:
Function getHeader(inLength as integer,inType as integer) As String
  return getDefine(chrb(inLength mod 256)+chrb(inLength\256)+chrb(inType mod 256)+chrb(inType\256))
End Function

GeneralModule.revealInFinder:
Sub revealInFinder(inFi as folderitem)
  dim ae as AppleEvent
  ae=NewAppleEvent("misc","mvis","MACS")
  ae.FolderItemParam("----")=inFi
  if ae.Send() then
    ae=NewAppleEvent("misc","actv","MACS")
    call ae.send()
  end
End Sub

GeneralModule.getRemainTime:
Function getRemainTime(inTime as integer) As String
  dim hour,minute,second as integer
  hour=inTime\60\60 mod 60
  minute=inTime\60 mod 60
  second=inTime mod 60
  
  if hour<>0 and minute<>0 then
    return str(hour)+s__hour+chrb(32)+str(minute)+s__minute
  elseif hour<>0 then
    return str(hour)+s__hour
  elseif minute<>0 then
    return str(minute)+s__minute
  elseif second<>0 then
    return str(second)+s__second
  end
End Function

GeneralModule.getFolderItem2Path:
Function getFolderItem2Path(inFi as folderItem) As String
  dim temp as string
  dim fi as folderItem
  fi=inFi
  while fi<>nil
    if fi.directory then
      temp=getString(fi.name)+chrb(58)+temp
    else
      temp=getString(fi.name)
    end
    fi=fi.parent
  wend
  return temp
End Function

GeneralModule.getPath2FolderItem:
Function getPath2FolderItem(inPath as string) As folderItem
  dim fi as folderItem
  dim i,j,k,l as integer
  dim temp as string
  temp=nthfield(inPath,chrb(58),1)
  j=volumeCount-1
  for i=0 to j
    if volume(i).isReadable and volume(i).visible and strcomp(getString(volume(i).name),temp,0)=0 then
      fi=volume(i)
      exit
    end
  next
  
  if ascb(rightb(inPath,1))=58 then
    j=countfields(inPath,chrb(58))-1
  else
    j=countfields(inPath,chrb(58))
  end
  for i=2 to j
    temp=""
    temp=nthfield(inPath,chrb(58),i)
    l=fi.count
    for k=1 to l
      if fi.isreadable and fi.visible and strcomp(getString(fi.item(k).name),temp,0)=0 then
        fi=fi.item(k)
        exit
      end
    next
  next
  return fi
exception
End Function

GeneralModule.getLongIP2ShortIP:
Function getLongIP2ShortIP(inWhat as string) As String
  dim i,first,last,nibble(3) as integer
  select case lenb(inWhat)
  case 10
    first=val(leftb(inWhat,8))
    last=(first mod 256)*100+val(rightb(inWhat,2))
    first=(first\256)*100+(last\256)
    nibble(0)=(last mod 256)
    for i=1 to 3
      nibble(i)=(first mod 256)
      first=first\256
    next
  case 9
    first=val(leftb(inWhat,8))
    last=(first mod 256)*10+val(rightb(inWhat,1))
    first=(first\256)*10+(last\256)
    nibble(0)=(last mod 256)
    for i=1 to 3
      nibble(i)=(first mod 256)
      first=first\256
    next
  else
    first=val(inWhat)
    for i=0 to 3
      nibble(i)=(first mod 256)
      first=first\256
    next
  end
  return str(nibble(0))+chrb(46)+str(nibble(1))+chrb(46)+str(nibble(2))+chrb(46)+str(nibble(3))
End Function

GeneralModule.writeComment:
Sub writeComment(inFi as folderItem, inComment as string)
  dim ae as AppleEvent
  ae=newAppleEvent("core","setd","MACS")
  ae.objectSpecifierParam("----")=getPropertyObjectDescriptor(getFolderItem2Obj(inFi),"comt")
  ae.stringParam("data")=inComment
  call ae.send()
End Sub

GeneralModule.getFolderItem2Obj:
Function getFolderItem2Obj(inFi as folderItem) As appleEventObjectSpecifier
  dim obj as AppleEventObjectSpecifier
  dim i,j as Integer
  dim temp as String
  temp=inFi.absolutePath
  If lenb(temp)<255 Then
    obj=getNamedObjectDescriptor("alis",nil,temp)
  else
    obj=getNamedObjectDescriptor("cdis",nil,getDefine(nthfield(temp,chrb(58),1)))
    j=countfields(temp,chrb(58))-1
    for i=2 to j
      obj=getNamedObjectDescriptor("cfol",obj,getDefine(nthfield(temp,chrb(58),i)))
    next
    if inFi.Directory=false then
      obj=GetNamedObjectDescriptor("file",obj,getDefine(nthfield(temp,chrb(58),j+1)))
    end
  end
  return obj
End Function

GeneralModule.moveFolderItem:
Sub moveFolderItem(inFi1 as folderitem, inFi2 as folderItem)
  dim ae as appleEvent
  ae=NewAppleEvent("core","move","MACS")
  ae.FolderItemParam("----")=inFi1
  ae.FolderItemParam("insh")=inFi2
  call ae.send()
End Sub

GeneralModule.bringAllWindowsToFront:
Sub bringAllWindowsToFront()
  dim i,j as integer
  j=windowCount-1
  for i=0 to j
    if window(i) isa OriginalWindow and window(i).visible then
      window(i).show()
    end
  next
exception
End Sub

GeneralModule.minimizeWindow:
Sub minimizeWindow(w as window)
  dim err as Integer
  #if targetCarbon
    Declare Function CollapseWindow Lib c_carbonlib (w as WindowPtr, b as Boolean) as Integer
  #else
    Declare Function CollapseWindow Lib "AppearanceLib" (w as WindowPtr, b as Boolean) as Integer
  #endif
  err=CollapseWindow(w,true)
End Sub

GeneralModule.getSource:
Function getSource(inWhat as string) As String
  dim temp as string
  temp=inWhat
  if instrb(1,temp,chrb(62))<>0 then
    do
      temp=midb(temp,instrb(1,temp,chrb(62))+1)
    loop until ascb(leftb(temp,1))<>60
  end
  return trim(leftb(temp,instrb(1,temp,chrb(60))-1))
End Function

GeneralModule.getWinMXString:
Function getWinMXString(inWhat as string) As String
  dim temp as string
  dim i,j as integer
  temp=inWhat
  j=lenb(temp)
  for i=1 to j
    i=instrb(i,temp,chrb(&h7C))
    if i=0 then
      exit
    end
    temp=leftb(temp,i-1)+chrb(&h2E)+midb(temp,i+1)
  next
  return getDefine(temp)
End Function

GeneralModule.getBit:
Function getBit(inSource as integer,inNumber as integer) As integer
  dim aSource,bit(7) as integer
  aSource=inSource
  if aSource\128=1 then
    bit(7)=1
    aSource=aSource-128
  else
    bit(7)=0
  end
  if aSource\64=1 then
    bit(6)=1
    aSource=aSource-64
  else
    bit(6)=0
  end
  if aSource\32=1 then
    bit(5)=1
    aSource=aSource-32
  else
    bit(5)=0
  end
  if aSource\16=1 then
    bit(4)=1
    aSource=aSource-16
  else
    bit(4)=0
  end
  if aSource\8=1 then
    bit(3)=1
    aSource=aSource-8
  else
    bit(3)=0
  end
  if aSource\4=1 then
    bit(2)=1
    aSource=aSource-4
  else
    bit(2)=0
  end
  if aSource\2=1 then
    bit(1)=1
    aSource=aSource-2
  else
    bit(1)=0
  end
  bit(0)=aSource
  return bit(inNumber)
End Function

GeneralModule.getTagSize:
Function getTagSize(inTagSize as integer) As integer
  dim byte,aResult as integer
  dim memb as MemoryBlock
  memb=NewMemoryBlock(4)
  memb.littleEndian=false
  memb.long(0)=inTagSize
  
  byte=memb.byte(0)
  if getBit(byte,7)<>0 then
    return -1
  end
  aResult=getBit(byte,6)*134217728+getBit(byte,5)*67108864+getBit(byte,4)*33554432+getBit(byte,3)*16777216+getBit(byte,2)*8388608+getBit(byte,1)*4194304+getBit(byte,0)*2097152
  byte=memb.byte(1)
  if getBit(byte,7)<>0 then
    return -1
  end
  aResult=aResult+getBit(byte,6)*1048576+getBit(byte,5)*524288+getBit(byte,4)*262144+getBit(byte,3)*131072+getBit(byte,2)*65536+getBit(byte,1)*32768+getBit(byte,0)*16384
  byte=memb.byte(2)
  if getBit(byte,7)<>0 then
    return -1
  end
  aResult=aResult+getBit(byte,6)*8192+getBit(byte,5)*4096+getBit(byte,4)*2048+getBit(byte,3)*1024+getBit(byte,2)*512+getBit(byte,1)*256+getBit(byte,0)*128
  byte=memb.byte(3)
  if getBit(byte,7)<>0 then
    return -1
  end
  aResult=aResult+getBit(byte,6)*64+getBit(byte,5)*32+getBit(byte,4)*16+getBit(byte,3)*8+getBit(byte,2)*4+getBit(byte,1)*2+getBit(byte,0)*1
  return aResult
End Function

GeneralModule.getMediaType:
Function getMediaType(inExtension as string) As string
  dim aPosition as integer
  aPosition=instrb(1,".mp3.m4a.wma.wav.aif.aiff.ra.ogg.au.mod.mid.mpg.mpeg.mp2.vob.mp4.avi.wmv.asf.mov.rm.ram.rmvb.ogm.mkv.swf.jpg.jpeg.bmp.pic.pict.tif.tiff.gif.png.psd.txt.log.htm.html.doc.rtf.rtfd.pdf",inExtension)
  if aPosition=0 then
    return rm_files
  elseif aPosition<43 then
    return rm_music
  elseif aPosition<104 then
    return rm_movies
  elseif aPosition<147 then
    return rm_pictures
  else
    return rm_text
  end
End Function

GeneralModule.getURL:
Function getURL(inWhat as string) As String
  dim aPosition as integer
  aPosition=instrb(1,inWhat,"http://")
  if aPosition<>0 then
    return midb(inWhat,aPosition,instrb(aPosition,inWhat,chrb(34))-aPosition)
  end
End Function

GeneralModule.notifyStart:
Sub notifyStart()
  dim error as integer
  if App.active=false and gAutoReply=false and notify=nil then
    notify=newMemoryBlock(36)
    notify.short(4)=8
    notify.short(14)=1
    #if targetCarbon
      Declare Function NMInstall Lib c_carbonlib (nmReqPtr as Ptr) as Short
    #else
      Declare Function NMInstall Lib "InterfaceLib" (nmReqPtr as Ptr) as Short Inline68K("205FA05E3E80")
    #endif
    error=NMInstall(notify)
  end
End Sub

GeneralModule.notifyStop:
Sub notifyStop()
  dim error as integer
  if notify<>nil then
    #if targetCarbon then
      Declare Function NMRemove Lib c_carbonlib (nmReqPtr as Ptr) as Short
    #else
      Declare Function NMRemove Lib "InterfaceLib" (nmReqPtr as Ptr) as Short Inline68K("205FA05F3E80")
    #endif
    error=NMRemove(notify)
    notify=nil
  end
End Sub

GeneralModule.getUTF8String:
Function getUTF8String(inWhat as string) As String
  return convertEncoding(inWhat,encodings.UTF8)
End Function

GeneralModule.getColor2String:
Function getColor2String(inColor as color) As string
  return chrb(inColor.red)+chrb(inColor.green)+chrb(inColor.blue)
End Function

GeneralModule.getString2Color:
Function getString2Color(inString as string) As color
  return rgb(ascb(leftb(inString,1)),ascb(midb(inString,2,1)),ascb(rightb(inString,1)))
End Function

GeneralModule.hasWantQueue:
Function hasWantQueue(inArray() as string,inNick as string) As Boolean
  dim i,j as integer
  j=ubound(inArray)
  for i=1 to j
    if strcomp(inArray(i),inNick,0)=0 then
      inArray.remove i
      return true
    end
  next
End Function

GeneralModule.hasNickAndServerID:
Function hasNickAndServerID(inNickArray() as string,inServerIDArray() as integer,inNick as string,inServerID as integer) As boolean
  dim i,j as integer
  j=ubound(inNickArray)
  for i=1 to j
    if strcomp(inNickArray(i),inNick,0)=0 and inServerIDArray(i)=inServerID then
      return true
    end
  next
End Function

GeneralModule.inbackstrb:
Function inbackstrb(position as integer, source as string, delimitter as string) As integer
  
  dim i, j as integer
  
  for i = lenb(source) - position downto 0
    
    if strcomp(midb(source, i, 1), delimitter, 0) = 0 then return i
    
  next
  
End Function

LibraryModule.OL_Get:
Sub OL_Get()
  dim tis as TextInputStream
  dim aArray as integer
  dim temp as string
  if LibraryLock.enter() then
    tis=PreferencesFolder.child(c_2get).child(c_shared).openAsTextFile()
    do until tis.eof
      temp=""
      temp=tis.readline()
      Library.append new OriginalLibrary
      aArray=ubound(Library)
      if Library(aArray)<>nil then
        Library(aArray).fullPath=getDefine(nthfield(temp,chrb(9),1))
        Library(aArray).filePath=getDefine(nthfield(temp,chrb(9),2))
        Library(aArray).shortName=getDefine(nthfield(temp,chrb(9),3))
        Library(aArray).size=getDefine(nthfield(temp,chrb(9),4))
        Library(aArray).shortSize=getDefine(nthfield(temp,chrb(9),5))
        Library(aArray).bitrate=getDefine(nthfield(temp,chrb(9),6))
        Library(aArray).sampleRate=getDefine(nthfield(temp,chrb(9),7))
        Library(aArray).time=getDefine(nthfield(temp,chrb(9),8))
        Library(aArray).md5=getDefine(nthfield(temp,chrb(9),9))
        Library(aArray).byte=lenb(Library(aArray).shortName+Library(aArray).md5+Library(aArray).size+Library(aArray).bitrate+Library(aArray).sampleRate+Library(aArray).time)+8
      end
    loop
    tis.close()
    LibraryLock.leave()
  end
exception
  if tis<>nil then
    tis.close()
  end
  LibraryLock.leave()
End Sub

LibraryModule.OL_Write:
Sub OL_Write()
  dim tos as TextOutputStream
  dim i,j as integer
  if LibraryLock.enter() then
    tos=PreferencesFolder.child(c_2get).Child(c_shared).CreateTextFile()
    j=ubound(Library)
    for i=1 to j
      tos.writeline(Library(i).fullPath+chrb(9)+Library(i).filePath+chrb(9)+Library(i).shortName+chrb(9)+Library(i).size+chrb(9)+Library(i).shortSize+chrb(9)+Library(i).bitrate+chrb(9)+Library(i).sampleRate+chrb(9)+Library(i).time+chrb(9)+Library(i).md5)
    next
    tos.close()
    LibraryLock.leave()
  end
exception
  if tos<>nil then
    tos.close()
  end
  LibraryLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

LibraryModule.OL_SearchFileName2Array:
Function OL_SearchFileName2Array(inFileName as string) As Integer
  dim i,j as integer
  j=ubound(Library)
  for i=1 to j
    if strcomp(Library(i).filePath+Library(i).shortName,inFileName,0)=0 then
      return i
    end
  next
  return c_error
End Function

LibraryModule.OL_Update:
Sub OL_Update()
  dim LibraryThread as OriginalLibraryThread
  if isIndexing=false then
    LibraryThread=new OriginalLibraryThread
    LibraryThread.run()
  end
End Sub

LibraryModule.OL_SearchOldFileName2Array:
Function OL_SearchOldFileName2Array(inFileName as string) As Integer
  dim i,j as integer
  j=ubound(OldLibrary)
  for i=1 to j
    if strcomp(OldLibrary(i).filePath+OldLibrary(i).shortName,inFileName,0)=0 then
      return i
    end
  next
  return c_error
End Function

LibraryModule.OL_SharedBuffer:
Sub OL_SharedBuffer()
  dim i,j,byte as integer
  dim temp,path as string
  if LibraryLock.enter() then
    sharedBuffer=""
    j=ubound(Library)
    if isWinMX=false then
      for i=1 to j
        path=""
        path=Library(i).filePath
        temp=""
        temp=chrb(34)+path+chrb(34)
        byte=lenb(temp)
        for i=i to j
          temp=temp+chrb(32)+chrb(34)+Library(i).shortName+chrb(34)+chrb(32)+getDefine("0")+chrb(32)+Library(i).size+chrb(32)+Library(i).bitRate+chrb(32)+Library(i).sampleRate+chrb(32)+Library(i).time
          byte=byte+Library(i).byte
          if i+1>j or strcomp(Library(i+1).filePath,path,0)<>0 or byte+Library(i+1).byte>2044 then
            exit
          end
        next
        sharedBuffer=sharedBuffer+getHeader(lenb(temp),870)+temp
      next
      
    else
      for i=1 to j
        path=""
        path=Library(i).filePath
        temp=""
        temp=chrb(34)+getDefine("D")+chrb(58)+chrb(92)+path+chrb(34)
        byte=lenb(temp)
        for i=i to j
          temp=temp+chrb(32)+chrb(34)+Library(i).shortName+chrb(34)+chrb(32)+Library(i).md5+chrb(32)+Library(i).size+chrb(32)+Library(i).bitRate+chrb(32)+Library(i).sampleRate+chrb(32)+Library(i).time
          byte=byte+Library(i).byte
          if i+1>j or strcomp(Library(i+1).filePath,path,0)<>0 or byte+Library(i+1).byte>2044 then
            exit
          end
        next
        sharedBuffer=sharedBuffer+getHeader(lenb(temp),870)+temp
      next
      
    end
    LibraryLock.leave()
  end
End Sub

LibraryModule.OL_SearchFolderPath2Array:
Function OL_SearchFolderPath2Array(inFolderPath1 as string,inFolderPath2 as string,inFolderPath3 as string,inFolderPath4 as string) As integer
  dim i,j as integer
  dim temp as string
  j=ubound(iUploadFolder)
  for i=1 to j
    temp=""
    temp=getFolderItem2Path(iUploadFolder(i))
    if instrb(1,inFolderPath1,temp)=1 or instrb(1,inFolderPath2,temp)=1 or instrb(1,inFolderPath3,temp)=1 or instrb(1,inFolderPath4,temp)=1 then
      return i
    end
  next
  return c_error
End Function

MessageModule.ME_Append:
Function ME_Append(inPartnerNick as String,inServerID as Integer,inNick as string,inNetwork as string) As Integer
  dim aArray as integer
  MessageWindow.append new OriginalMessageWindow
  aArray=ubound(MessageWindow)
  if MessageWindow(aArray)<>nil then
    gTaskID=gTaskID+1
    MessageWindow(aArray).ID=gTaskID
    MessageWindow(aArray).title=inPartnerNick+getDefine(" - ")+inNetwork
    MessageWindow(aArray).partnerNick=inPartnerNick
    MessageWindow(aArray).serverID=inServerID
    MessageWindow(aArray).network=inNetwork
    MessageWindow(aArray).nick=inNick
    MessageWindow(aArray).textAndStyle=new OriginalTextAndStyle
    return aArray
  else
    return c_error
  end
End Function

MessageModule.ME_SearchID2Array:
Function ME_SearchID2Array(inID as integer) As Integer
  dim i,j as integer
  j=ubound(MessageWindow)
  for i=1 to j
    if MessageWindow(i).isFinalized=false and MessageWindow(i).ID=inID then
      return i
    end
  next
  return c_error
End Function

MessageModule.ME_SearchNickAndServerID2Array:
Function ME_SearchNickAndServerID2Array(inPartnerNick as String,inServerID as integer) As Integer
  dim i,j as integer
  j=ubound(MessageWindow)
  for i=1 to j
    if MessageWindow(i).isFinalized=false and strcomp(MessageWindow(i).partnerNick,inPartnerNick,0)=0 and MessageWindow(i).serverID=inServerID then
      return i
    end
  next
  return c_error
End Function

MessageModule.ME_Offline:
Sub ME_Offline(inServerID as integer)
  dim i,j as integer
  dim temp as string
  temp=getString(s_offline)
  if MessageWindowLock.enter() then
    j=ubound(MessageWindow)
    for i=1 to j
      if MessageWindow(i).isFinalized=false and MessageWindow(i).serverID=inServerID then
        getMessage(MessageWindow(i).TextAndStyle,temp,co_notifications)
        if gMessageWindowID=MessageWindow(i).ID then
          MainWindow.MessageEditField.AppendMessage(temp,co_notifications)
        end
      end
    next
    MessageWindowLock.leave()
  end
End Sub

MessageModule.ME_NewMessage:
Sub ME_NewMessage(inServerID as integer,inPartnerNick as string)
  dim aArray,aOpenNapArray,aID as integer
  if OpenNapSocketLock.enter() then
    aOpenNapArray=OS_SearchID2Array(inServerID)
    if aOpenNapArray<>c_error then
      
      if MessageWindowLock.enter() then
        aArray=ME_SearchNickAndServerID2Array(inPartnerNick,inServerID)
        if aArray=c_error then
          aArray=ME_Append(inPartnerNick,inServerID,OpenNapSocket(aOpenNapArray).nick,OpenNapSocket(aOpenNapArray).network)
          if aArray<>c_error then
            DrawerWindow.DR_Append(inPartnerNick,MessageWindow(aArray).ID,c_message_window)
          end
        end
        if aArray<>c_error then
          aID=MessageWindow(aArray).ID
        end
        MessageWindowLock.leave()
      end
      
      if aID<>0 then
        DrawerWindow.DR_SelectMenu(aID)
      end
    end
    OpenNapSocketLock.leave()
  end
End Sub

MessageModule.ME_Remove:
Sub ME_Remove()
  dim i as integer
  if MessageWindowLock.signal() then
    for i=ubound(MessageWindow) downto 1
      if MessageWindow(i).isFinalized then
        MessageWindow(i)=nil
        MessageWindow.Remove i
        removingObjects=removingObjects-1
      end
    next
    MessageWindowLock.release()
  end
End Sub

MessageModule.ME_Online:
Sub ME_Online(inServerID as integer,inNetwork as string)
  dim i,j as integer
  dim temp as string
  temp=getString(s_online)
  if MessageWindowLock.enter() then
    j=ubound(MessageWindow)
    for i=1 to j
      if MessageWindow(i).isFinalized=false and strcomp(MessageWindow(i).network,inNetwork,0)=0 then
        MessageWindow(i).serverID=inServerID
        getMessage(MessageWindow(i).TextAndStyle,temp,co_notifications)
        if gMessageWindowID=MessageWindow(i).ID then
          MainWindow.MessageEditField.AppendMessage(temp,co_notifications)
        end
      end
    next
    MessageWindowLock.leave()
  end
End Sub

PrefrencesModule.getConfig:
Sub getConfig()
  dim rf as ResourceFork
  dim fi as folderItem
  dim i,j as integer
  rf=PreferencesFolder.Child(c_2get).Child(c_2get_preferences).OpenResourceFork()
  
  i=val(replaceallb(rightb(rf.GetResource("pref",0),5),".",""))
  j=196
  if i<j then
    if rf<>nil then
      rf.close()
    end
    InitConfig()
    return
  end
  
  iTextEncoding=getTextEncoding(val(rf.GetResource("pref",205)))
  if iTextEncoding.base=256 then
    iTextEncoding=encodings.UTF8
  end
  gTextEncoding=iTextEncoding
  
  iClearDownload=(strcomp(rf.GetResource("pref",100),"true",0)=0)
  iClearUpload=(strcomp(rf.GetResource("pref",101),"true",0)=0)
  iAllowFreeloader=(strcomp(rf.GetResource("pref",102),"true",0)=0)
  iFreeloaderShares=val(rf.GetResource("pref",103))
  iAskNapList=(strcomp(rf.GetResource("pref",104),"true",0)=0)
  iAutoConnect=(strcomp(rf.GetResource("pref",105),"true",0)=0)
  iAutoReply=getDefine(rf.GetResource("pref",106))
  iUpdateLibrary=(strcomp(rf.GetResource("pref",107),"true",0)=0)
  iClearServer=(strcomp(rf.GetResource("pref",108),"true",0)=0)
  
  iFacing=val(rf.GetResource("pref",200))
  iTimeStamp=(strcomp(rf.GetResource("pref",201),"true",0)=0)
  iShowJoinLeave=(strcomp(rf.GetResource("pref",202),"true",0)=0)
  iTextFont=rf.GetResource("pref",203)
  gTextFontID=getFontName2ID(iTextFont)
  iTextSize=val(rf.GetResource("pref",204))
  iShowBandwidth=(strcomp(rf.GetResource("pref",206),"true",0)=0)
  iPromptRemoving=(strcomp(rf.GetResource("pref",207),"true",0)=0)
  iPromptQuitting=(strcomp(rf.GetResource("pref",208),"true",0)=0)
  
  iSpeed=val(rf.GetResource("pref",300))
  iConnectionLimits=val(rf.GetResource("pref",301))
  iHTTPProxy=rf.GetResource("pref",302)
  iDownloadBuffer=val(rf.GetResource("pref",303))
  iUploadBuffer=val(rf.GetResource("pref",304))
  iCPUCycleTime=val(rf.GetResource("pref",305))
  iPortRange=rf.GetResource("pref",306)
  getPortRange()
  
  iDownloadFolder=getPath2FolderItem(getDefine(rf.GetResource("pref",400)))
  if iDownloadFolder=nil or iDownloadFolder.exists=false then
    iDownloadFolder=DesktopFolder.child(s_downloads)
    if iDownloadFolder=nil or iDownloadFolder.exists=false then
      iDownloadFolder.CreateAsFolder()
    end
  end
  iDownloadLimits=val(rf.GetResource("pref",401))
  iDownloadPerUser=val(rf.GetResource("pref",402))
  iDownloadPerServer=val(rf.GetResource("pref",403))
  iIncompleteFolder=getPath2FolderItem(getDefine(rf.GetResource("pref",404)))
  if iIncompleteFolder=nil or iIncompleteFolder.exists=false then
    iIncompleteFolder=iDownloadFolder.child(s_incomplete)
    if iIncompleteFolder=nil or iIncompleteFolder.exists=false then
      iIncompleteFolder.createAsFolder()
    end
  end
  iMusicFolder=getPath2FolderItem(getDefine(rf.GetResource("pref",405)))
  if iMusicFolder=nil or iMusicFolder.exists=false then
    iMusicFolder=nil
  end
  iMoviesFolder=getPath2FolderItem(getDefine(rf.GetResource("pref",406)))
  if iMoviesFolder=nil or iMoviesFolder.exists=false then
    iMoviesFolder=nil
  end
  iPicturesFolder=getPath2FolderItem(getDefine(rf.GetResource("pref",407)))
  if iPicturesFolder=nil or iPicturesFolder.exists=false then
    iPicturesFolder=nil
  end
  
  if LibraryLock.enter() then
    j=rf.ResourceCount("shar")
    for i=1 to j
      fi=getPath2FolderItem(getDefine(rf.GetResource("shar",i)))
      if fi<>nil then
        iUploadFolder.Append fi
      end
    next
    isShareDownloads=(OL_SearchFolderPath2Array(getFolderItem2Path(iDownloadFolder),getFolderItem2Path(iMusicFolder),getFolderItem2Path(iMoviesFolder),getFolderItem2Path(iPicturesFolder))<>c_error)
    LibraryLock.leave()
  end
  
  iUploadLimits=val(rf.GetResource("pref",500))
  iUploadPerUser=val(rf.GetResource("pref",501))
  iUploadBandwidth=val(rf.GetResource("pref",502))
  
  iNick=getDefine(rf.GetResource("pref",600))
  iPassword=getDefine(rf.GetResource("pref",601))
  iClient=getDefine(rf.GetResource("pref",602))
  if instrb(1,iClient,"2get")<>0 then
    iClient=getDefine(App.LongVersion)
    isWinMX=false
  elseif instrb(1,iClient,"Win")<>0 then
    iClient=getDefine(c_winmx)
    isWinMX=true
  else
    isWinMX=false
  end
  
  iColors(co_default)=getString2Color(rf.GetResource("pref",700))
  iColors(co_background)=getString2Color(rf.GetResource("pref",701))
  iColors(co_timestamp)=getString2Color(rf.GetResource("pref",702))
  iColors(co_own_user)=getString2Color(rf.GetResource("pref",703))
  iColors(co_other_users)=getString2Color(rf.GetResource("pref",704))
  iColors(co_notifications)=getString2Color(rf.GetResource("pref",705))
  
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
  InitConfig()
End Sub

PrefrencesModule.writeConfig:
Sub writeConfig()
  dim rf as ResourceFork
  dim fi as FolderItem
  dim i,j as integer
  fi=PreferencesFolder.Child(c_2get)
  if fi=nil or fi.exists=false then
    fi.createAsFolder()
  end
  fi=fi.Child(c_2get_preferences)
  if fi<>nil and fi.exists then
    rf=fi.OpenResourceFork()
  else
    rf=fi.CreateResourceFork("2get Preferences")
  end
  
  rf.AddResource(App.longVersion,"pref",0,"")
  
  rf.AddResource(getBoolean(iClearDownload),"pref",100,"")
  rf.AddResource(getBoolean(iClearUpload),"pref",101,"")
  rf.AddResource(getBoolean(iAllowFreeloader),"pref",102,"")
  rf.AddResource(str(iFreeloaderShares),"pref",103,"")
  rf.AddResource(getBoolean(iAskNapList),"pref",104,"")
  rf.AddResource(getBoolean(iAutoConnect),"pref",105,"")
  rf.AddResource(iAutoReply,"pref",106,"")
  rf.AddResource(getBoolean(iUpdateLibrary),"pref",107,"")
  rf.AddResource(getBoolean(iClearServer),"pref",108,"")
  
  rf.AddResource(str(iFacing),"pref",200,"")
  rf.AddResource(getBoolean(iTimeStamp),"pref",201,"")
  rf.AddResource(getBoolean(iShowJoinLeave),"pref",202,"")
  rf.AddResource(iTextFont,"pref",203,"")
  rf.AddResource(str(iTextSize),"pref",204,"")
  rf.AddResource(str(iTextEncoding.base),"pref",205,"")
  rf.AddResource(getBoolean(iShowBandwidth),"pref",206,"")
  rf.AddResource(getBoolean(iPromptRemoving),"pref",207,"")
  rf.AddResource(getBoolean(iPromptQuitting),"pref",208,"")
  
  rf.AddResource(str(iSpeed),"pref",300,"")
  rf.AddResource(str(iConnectionLimits),"pref",301,"")
  rf.AddResource(iHTTPProxy,"pref",302,"")
  rf.AddResource(str(iDownloadBuffer),"pref",303,"")
  rf.AddResource(str(iUploadBuffer),"pref",304,"")
  rf.AddResource(str(iCPUCycleTime),"pref",305,"")
  rf.AddResource(iPortRange,"pref",306,"")
  
  rf.AddResource(getFolderItem2Path(iDownloadFolder),"pref",400,"")
  rf.AddResource(str(iDownloadLimits),"pref",401,"")
  rf.AddResource(str(iDownloadPerUser),"pref",402,"")
  rf.AddResource(str(iDownloadPerServer),"pref",403,"")
  rf.AddResource(getFolderItem2Path(iIncompleteFolder),"pref",404,"")
  rf.AddResource(getFolderItem2Path(iMusicFolder),"pref",405,"")
  rf.AddResource(getFolderItem2Path(iMoviesFolder),"pref",406,"")
  rf.AddResource(getFolderItem2Path(iPicturesFolder),"pref",407,"")
  
  for i=rf.ResourceCount("shar") downto 1
    rf.removeResource("shar",i)
  next
  if LibraryLock.enter() then
    j=ubound(iUploadFolder)
    for i=1 to j
      rf.AddResource(getFolderItem2Path(iUploadFolder(i)),"shar",i,"")
    next
    LibraryLock.leave()
  end
  rf.AddResource(str(iUploadLimits),"pref",500,"")
  rf.AddResource(str(iUploadPerUser),"pref",501,"")
  rf.AddResource(str(iUploadBandwidth),"pref",502,"")
  
  rf.AddResource(iNick,"pref",600,"")
  rf.AddResource(iPassword,"pref",601,"")
  rf.AddResource(iClient,"pref",602,"")
  
  rf.AddResource(getColor2String(iColors(co_default)),"pref",700,"")
  rf.AddResource(getColor2String(iColors(co_background)),"pref",701,"")
  rf.AddResource(getColor2String(iColors(co_timestamp)),"pref",702,"")
  rf.AddResource(getColor2String(iColors(co_own_user)),"pref",703,"")
  rf.AddResource(getColor2String(iColors(co_other_users)),"pref",704,"")
  rf.AddResource(getColor2String(iColors(co_notifications)),"pref",705,"")
  
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

PrefrencesModule.writeWindow:
Sub writeWindow(inWindow as Window,inID as Integer)
  dim rf as ResourceFork
  dim fi as FolderItem
  fi=PreferencesFolder.Child(c_2get)
  if fi=nil or fi.exists=false then
    fi.createAsFolder()
  end
  fi=fi.Child(c_2get_preferences)
  if fi<>nil and fi.exists then
    rf=fi.OpenResourceFork()
  else
    rf=fi.CreateResourceFork("2get Preferences")
  end
  rf.AddResource(chrb(inWindow.left\256)+chrb(inWindow.left mod 256)+chrb(inWindow.top\256)+chrb(inWindow.top mod 256)+chrb(inWindow.width\256)+chrb(inWindow.width mod 256)+chrb(inWindow.height\256)+chrb(inWindow.height mod 256),"wpos",inID,"")
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

PrefrencesModule.writeColumn:
Sub writeColumn(inListBox as ListBox,inID as Integer)
  dim rf as ResourceFork
  dim fi as FolderItem
  fi=PreferencesFolder.Child(c_2get)
  if fi=nil or fi.exists=false then
    fi.createAsFolder()
  end
  fi=fi.Child(c_2get_preferences)
  if fi<>nil and fi.exists then
    rf=fi.OpenResourceFork()
  else 
    rf=fi.CreateResourceFork("2get Preferences")
  end
  rf.AddResource(inListBox.columnWidths,"colm",inID,"")
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

PrefrencesModule.initConfig:
Sub initConfig()
  iTextEncoding=encodings.systemDefault
  gTextEncoding=iTextEncoding
  
  iClearDownload=false
  iClearUpload=false
  iAllowFreeloader=true
  iFreeloaderShares=50
  iAskNapList=true
  iAutoConnect=false
  iUpdateLibrary=true
  iAutoReply=""
  iClearServer=true
  
  iFacing=2
  iTimeStamp=false
  iShowJoinLeave=false
  iShowBandwidth=true
  iTextFont="System"
  gTextFontID=getFontName2ID(iTextFont)
  iTextSize=10
  iPromptRemoving=true
  iPromptQuitting=true
  
  iColors(co_default)=rgb(&h00,&h00,&h00)
  iColors(co_background)=rgb(&hFF,&hFF,&hFF)
  iColors(co_timestamp)=rgb(&h80,&h80,&h80)
  iColors(co_own_user)=rgb(&hAF,&h2F,&h1F)
  iColors(co_other_users)=rgb(&hFF,&h80,&h00)
  iColors(co_notifications)=rgb(&h00,&h80,&h00)
  
  iSpeed=0
  iConnectionLimits=10
  iHTTPProxy=""
  iDownloadBuffer=1024*8
  iUploadBuffer=1024*8
  iCPUCycleTime=250
  iPortRange="6600-6700"
  getPortRange()
  
  iDownloadFolder=DesktopFolder.child(s_downloads)
  if iDownloadFolder=nil or iDownloadFolder.exists=false then
    iDownloadFolder.CreateAsFolder()
  end
  iIncompleteFolder=iDownloadFolder.child(s_incomplete)
  if iIncompleteFolder=nil or iIncompleteFolder.exists=false then
    iIncompleteFolder.createAsFolder()
  end
  iMusicFolder=nil
  iMoviesFolder=nil
  iPicturesFolder=nil
  iDownloadLimits=0
  iDownloadPerUser=1
  iDownloadPerServer=0
  
  if LibraryLock.signal() then
    redim iUploadFolder(0)
    iUploadFolder.append iDownloadFolder
    LibraryLock.release()
  end
  isShareDownloads=true
  iUploadLimits=5
  iUploadPerUser=1
  iUploadBandwidth=0
  
  iNick=getRandomString(false)
  iPassword=getRandomString(false)
  iClient=getDefine(App.LongVersion)
  isWinMX=false
exception
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

PrefrencesModule.getWindow:
Sub getWindow(inWindow as Window,inID as Integer)
  dim rf as ResourceFork
  dim temp as string
  rf=PreferencesFolder.Child(c_2get).Child(c_2get_preferences).OpenResourceFork()
  temp=rf.GetResource("wpos",inID)
  inWindow.left=ascb(midb(temp,1,1))*256+ascb(midb(temp,2,1))
  inWindow.top=ascb(midb(temp,3,1))*256+ascb(midb(temp,4,1))
  inWindow.width=ascb(midb(temp,5,1))*256+ascb(midb(temp,6,1))
  inWindow.height=ascb(midb(temp,7,1))*256+ascb(midb(temp,8,1))
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
End Sub

PrefrencesModule.getColumn:
Sub getColumn(inListBox as ListBox,inID as Integer)
  dim rf as ResourceFork
  rf=PreferencesFolder.Child(c_2get).Child(c_2get_preferences).OpenResourceFork()
  inListBox.columnWidths=rf.GetResource("colm",inID)
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
End Sub

PrefrencesModule.getToolbar:
Function getToolbar(inID as Integer) As Boolean
  dim rf as ResourceFork
  dim b as boolean
  rf=PreferencesFolder.Child(c_2get).Child(c_2get_preferences).OpenResourceFork()
  b=(strcomp(rf.GetResource("tool",inID),"true",0)=0)
  rf.close()
  return b
exception
  if rf<>nil then
    rf.close()
  end
End Function

PrefrencesModule.writeToolbar:
Sub writeToolbar(inID as Integer,inState as boolean)
  dim rf as ResourceFork
  dim fi as FolderItem
  fi=PreferencesFolder.Child(c_2get)
  if fi=nil or fi.exists=false then
    fi.createAsFolder()
  end
  fi=fi.Child(c_2get_preferences)
  if fi<>nil and fi.exists then
    rf=fi.OpenResourceFork()
  else
    rf=fi.CreateResourceFork("2get Preferences")
  end
  rf.AddResource(getBoolean(inState),"tool",inID,"")
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

PrefrencesModule.getHistory:
Sub getHistory()
  dim rf as ResourceFork
  dim i,j as integer
  rf=PreferencesFolder.Child(c_2get).Child(c_2get_preferences).OpenResourceFork()
  
  j=rf.ResourceCount("sear")
  for i=1 to j
    appendHistory(searchHistory,getDefine(rf.GetResource("sear",i)))
  next
  
  j=rf.ResourceCount("whois")
  for i=1 to j
    appendHistory(whoisHistory,getDefine(rf.GetResource("whoi",i)))
  next
  
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
End Sub

PrefrencesModule.writeHistory:
Sub writeHistory()
  dim rf as ResourceFork
  dim fi as FolderItem
  dim i,j as integer
  fi=PreferencesFolder.Child(c_2get)
  if fi=nil or fi.exists=false then
    fi.createAsFolder()
  end
  fi=fi.Child(c_2get_preferences)
  if fi<>nil and fi.exists then
    rf=fi.OpenResourceFork()
  else
    rf=fi.CreateResourceFork("2get Preferences")
  end
  
  for i=rf.ResourceCount("sear") downto 1
    rf.removeResource("sear",i)
  next
  j=ubound(searchHistory)
  for i=1 to j
    rf.AddResource(searchHistory(i),"sear",i,"")
  next
  
  for i=rf.ResourceCount("whoi") downto 1
    rf.removeResource("whoi",i)
  next
  j=ubound(whoisHistory)
  for i=1 to j
    rf.AddResource(whoisHistory(i),"whoi",i,"")
  next
  
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

PrefrencesModule.appendHistory:
Sub appendHistory(inHistory() as string,inWhat as string)
  dim i,j as integer
  if lenb(inWhat)=0 then
    return
  end
  j=ubound(inHistory)
  for i=1 to j
    if strcomp(inHistory(i),inWhat,0)=0 then
      return
    end
  next
  inHistory.append inWhat
  if j>15 then
    inHistory.remove 1
  end
End Sub

PrefrencesModule.getSplitter:
Function getSplitter(inID as Integer) As integer
  dim rf as ResourceFork
  dim aLeft as integer
  rf=PreferencesFolder.Child(c_2get).Child(c_2get_preferences).OpenResourceFork()
  aLeft=val(rf.GetResource("splt",inID))
  rf.close()
  return aLeft
exception
  if rf<>nil then
    rf.close()
  end
End Function

PrefrencesModule.writeSplitter:
Sub writeSplitter(inID as Integer,inLeft as integer)
  dim rf as ResourceFork
  dim fi as FolderItem
  fi=PreferencesFolder.Child(c_2get)
  if fi=nil or fi.exists=false then
    fi.createAsFolder()
  end
  fi=fi.Child(c_2get_preferences)
  if fi<>nil and fi.exists then
    rf=fi.OpenResourceFork()
  else
    rf=fi.CreateResourceFork("2get Preferences")
  end
  rf.AddResource(str(inLeft),"splt",inID,"")
  rf.close()
exception
  if rf<>nil then
    rf.close()
  end
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

ResourceModule.getSound:
Sub getSound(inArray as integer)
  if gMute=false then
    if gSound(inArray)<>nil then
      if gSound(inArray).isPlaying=false then
        gSound(inArray).play()
      end
    elseif inArray=rs_message then
      beep()
    end
  end
End Sub

ResourceModule.initSound:
Sub initSound()
  #if debugBuild
  #else
    dim rf as ResourceFork
    rf=GetFolderItem(c_soundset).OpenResourceFork()
    gSound(rs_post)=rf.GetSound(128)
    gSound(rs_click)=rf.GetSound(129)
    gSound(rs_join)=rf.GetSound(130)
    gSound(rs_leave)=rf.GetSound(131)
    gSound(rs_message)=rf.GetSound(132)
    gSound(rs_login)=rf.GetSound(133)
    gSound(rs_error)=rf.GetSound(134)
    gSound(rs_done)=rf.GetSound(135)
    gSound(rs_bell)=rf.GetSound(136)
    rf.close()
  exception
    if rf<>nil then
      rf.close()
    end
  #endif
End Sub

ResourceModule.initPicture:
Sub initPicture()
  gPicture(r_arrived)=getMaskedPicture(rp_arrived)
  gPicture(r_emo_icon)=getMaskedPicture(rp_emo_icon)
  gPicture(r_connecting)=getMaskedPicture(rp_connecting)
  gPicture(r_done)=getMaskedPicture(rp_done)
  gPicture(r_download)=getMaskedPicture(rp_download)
  gPicture(r_favorite)=getMaskedPicture(rp_favorite)
  gPicture(r_folder)=getMaskedPicture(rp_folder)
  gPicture(r_ignore)=getMaskedPicture(rp_ignore)
  gPicture(r_left)=getMaskedPicture(rp_left)
  gPicture(r_middle)=getMaskedPicture(rp_middle)
  gPicture(r_online)=getMaskedPicture(rp_online)
  gPicture(r_download_badge)=getMaskedPicture(rp_download_badge)
  gPicture(r_right)=getMaskedPicture(rp_right)
  gPicture(r_search_icon)=getMaskedPicture(rp_search_icon)
  gPicture(r_unknown)=getMaskedPicture(rp_unknown)
  gPicture(r_upload)=getMaskedPicture(rp_upload)
  gPicture(r_console)=getMaskedPicture(rp_console)
  gPicture(r_channel)=getMaskedPicture(rp_channel)
  gPicture(r_whois)=getMaskedPicture(rp_whois)
  gPicture(r_user_list)=getMaskedPicture(rp_user_list)
  gPicture(r_search)=getMaskedPicture(rp_search)
  gPicture(r_browse)=getMaskedPicture(rp_browse)
  gPicture(r_message)=getMaskedPicture(rp_message)
  gPicture(r_cell_activated)=getMaskedPicture(rp_cell_activated)
  gPicture(r_cell_deactivated)=getMaskedPicture(rp_cell_deactivated)
  
  gMediaPicture(val(rm_music))=getMaskedPicture(rp_music)
  gMediaPicture(val(rm_pictures))=getMaskedPicture(rp_pictures)
  gMediaPicture(val(rm_movies))=getMaskedPicture(rp_movies)
  gMediaPicture(val(rm_text))=getMaskedPicture(rp_text)
  gMediaPicture(val(rm_files))=getMaskedPicture(rp_files)
  
  gMediaPicture(val(rm_music)+5)=getMaskedPicture(rb_music)
  gMediaPicture(val(rm_pictures)+5)=getMaskedPicture(rb_pictures)
  gMediaPicture(val(rm_movies)+5)=getMaskedPicture(rb_movies)
  gMediaPicture(val(rm_text)+5)=getMaskedPicture(rb_text)
  gMediaPicture(val(rm_files)+5)=getMaskedPicture(rb_files)
End Sub

ResourceModule.getMaskedPicture:
Function getMaskedPicture(inID as integer) As Picture
  dim p as picture
  p=getPicture(inID)
  p.mask.graphics.drawPicture getPicture(inID+1),0,0
  return p
End Function

ResourceModule.getPicture:
Function getPicture(inID as integer) As Picture
  return App.ResourceFork.getPicture(inID)
End Function

ResourceModule.initCursor:
Sub initCursor()
  moveCursor=App.ResourceFork.getCursor(6601)
End Sub

RoomModule.RO_Append:
Function RO_Append(inChannel as String,inServerID as Integer,inNetwork as string) As Integer
  dim aArray as integer
  dim sr as styleRun
  RoomWindow.append new OriginalRoomWindow
  aArray=ubound(RoomWindow)
  if RoomWindow(aArray)<>nil then
    gTaskID=gTaskID+1
    RoomWindow(aArray).ID=gTaskID
    RoomWindow(aArray).title=inChannel+getDefine(" - ")+inNetwork
    RoomWindow(aArray).channel=inChannel
    RoomWindow(aArray).network=inNetwork
    RoomWindow(aArray).serverID=inServerID
    RoomWindow(aArray).textAndStyle=new OriginalTextAndStyle
    return aArray
  else
    return c_error
  end
End Function

RoomModule.RO_SearchID2Array:
Function RO_SearchID2Array(inID as integer) As Integer
  dim i,j as integer
  j=ubound(RoomWindow)
  for i=1 to j
    if RoomWindow(i).isFinalized=false and RoomWindow(i).ID=inID then
      return i
    end
  next
  return c_error
End Function

RoomModule.RO_SearchChannelAndServerID2Array:
Function RO_SearchChannelAndServerID2Array(inChannel as String,inServerID as integer) As Integer
  dim i,j as integer
  j=ubound(RoomWindow)
  for i=1 to j
    if RoomWindow(i).isFinalized=false and strcomp(RoomWindow(i).channel,inChannel,0)=0 and RoomWindow(i).serverID=inServerID then
      return i
    end
  next
  return c_error
End Function

RoomModule.RO_Offline:
Sub RO_Offline(inServerID as integer)
  dim i,j as integer
  dim temp as string
  temp=getString(s_offline)
  if RoomWindowLock.signal() then
    j=ubound(RoomWindow)
    for i=1 to j
      if RoomWindow(i).isFinalized=false and RoomWindow(i).serverID=inServerID then
        RoomWindow(i).topic=""
        redim RoomWindow(i).RoomList(0)
        getMessage(RoomWindow(i).textAndStyle,temp,co_notifications)
        if gRoomWindowID=RoomWindow(i).ID then
          MainWindow.RoomListBox.deleteAllRows()
          MainWindow.RoomEditField.AppendMessage(temp,co_notifications)
          MainWindow.RoomStaticText.text=""
          MainWindow.RoomStaticText.helpTag=""
          MainWindow.RoomStaticText.balloonHelp=""
        end
        DrawerWindow.DR_Update(RoomWindow(i).ID,RoomWindow(i).channel,"")
      end
    next
    RoomWindowLock.release()
  end
End Sub

RoomModule.RO_Remove:
Sub RO_Remove()
  dim i as integer
  if RoomWindowLock.signal() then
    for i=ubound(RoomWindow) downto 1
      if RoomWindow(i).isFinalized then
        RoomWindow(i)=nil
        RoomWindow.Remove i
        removingObjects=removingObjects-1
      end
    next
    RoomWindowLock.release()
  end
End Sub

RoomModule.RO_Online:
Sub RO_Online(inServerID as integer,inNetwork as string)
  dim i,j as integer
  if RoomWindowLock.enter() then
    j=ubound(RoomWindow)
    for i=1 to j
      if RoomWindow(i).isFinalized=false and strcomp(RoomWindow(i).network,inNetwork,0)=0 then
        RoomWindow(i).serverID=inServerID
      end
    next
    RoomWindowLock.leave()
  end
End Sub

SearchModule.SE_Append:
Function SE_Append() As Integer
  dim aArray as integer
  SearchWindow.append new OriginalSearchWindow
  aArray=ubound(SearchWindow)
  if SearchWindow(aArray)<>nil then
    gTaskID=gTaskID+1
    SearchWindow(aArray).ID=gTaskID
    return aArray
  else
    return c_error
  end
End Function

SearchModule.SE_SearchID2Array:
Function SE_SearchID2Array(inID as integer) As Integer
  dim i,j as integer
  j=ubound(SearchWindow)
  for i=1 to j
    if SearchWindow(i).isFinalized=false and SearchWindow(i).ID=inID then
      return i
    end
  next
  return c_error
End Function

SearchModule.SE_NewSearch:
Sub SE_NewSearch(inWhat as string)
  dim aArray,aID as integer
  if SearchWindowLock.enter() then
    aArray=SE_SearchTitle2Array(inWhat)
    if aArray=c_error then
      aArray=SE_Append()
      if aArray<>c_error then
        DrawerWindow.DR_Append(inWhat,SearchWindow(aArray).ID,c_search_window)
        MainWindow.SE_Search(inWhat,SearchWindow(aArray).ID)
      end
    end
    if aArray<>c_error then
      aID=SearchWindow(aArray).ID
    end
    SearchWindowLock.leave()
  end
  
  if aID<>0 then
    DrawerWindow.DR_SelectMenu(aID)
  end
End Sub

SearchModule.SE_Remove:
Sub SE_Remove()
  dim i as integer
  if SearchWindowLock.signal() then
    for i=ubound(SearchWindow) downto 1
      if SearchWindow(i).isFinalized then
        SearchWindow(i)=nil
        SearchWindow.Remove i
        removingObjects=removingObjects-1
      end
    next
    SearchWindowLock.release()
  end
End Sub

SearchModule.SE_SearchTitle2Array:
Function SE_SearchTitle2Array(inTitle as string) As Integer
  dim i,j as integer
  j=ubound(SearchWindow)
  for i=1 to j
    if SearchWindow(i).isFinalized=false and strcomp(SearchWindow(i).title,inTitle,0)=0 then
      return i
    end
  next
  return c_error
End Function

SocketModule.TS_SearchID2Array:
Function TS_SearchID2Array(inID as Integer) As Integer
  dim i,j as integer
  j=ubound(TransferSocket)
  for i=1 to j
    if TransferSocket(i).isFinalized=false and TransferSocket(i).ID=inID then
      return i
    end
  next
  return c_error
End Function

SocketModule.OS_SearchID2Array:
Function OS_SearchID2Array(inID as Integer) As Integer
  dim i,j as integer
  j=ubound(OpenNapSocket)
  for i=1 to j
    if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).ID=inID then
      return i
    end
  next
  return c_error
End Function

SocketModule.TS_Append:
Function TS_Append() As Integer
  dim aArray as integer
  TransferSocket.append new OriginalTransferSocket
  aArray=ubound(TransferSocket)
  if TransferSocket(aArray)<>nil then
    gTransferSocketID=gTransferSocketID+1
    TransferSocket(aArray).ID=gTransferSocketID
    return aArray
  else
    return c_error
  end
End Function

SocketModule.OS_Append:
Function OS_Append() As Integer
  dim aArray as integer
  OpenNapSocket.Append new OriginalOpenNapSocket
  aArray=ubound(OpenNapSocket)
  if OpenNapSocket(aArray)<>nil then
    gOpenNapSocketID=gOpenNapSocketID+1
    OpenNapSocket(aArray).ID=gOpenNapSocketID
    return aArray
  else
    return c_error
  end
End Function

SocketModule.NS_Append:
Function NS_Append() As Integer
  dim aArray as integer
  NapListSocket.Append new OriginalNapListSocket
  aArray=ubound(NapListSocket)
  if NapListSocket(aArray)<>nil then
    return aArray
  else
    return c_error
  end
End Function

SocketModule.OS_Command:
Sub OS_Command(inServerID as integer,inType as integer,inWhat as string)
  dim aArray as integer
  if OpenNapSocketLock.enter() then
    aArray=OS_SearchID2Array(inServerID)
    if aArray<>c_error then
      OpenNapSocket(aArray).command(inType,inWhat)
    end
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.OS_Broadcast:
Sub OS_Broadcast(inType as integer,inWhat as string)
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn then
        OpenNapSocket(i).command(inType,inWhat)
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.TS_SearchNickAndTransferTypeAndServerID2Array:
Function TS_SearchNickAndTransferTypeAndServerID2Array(inPartnerNick as string,inTransferType as integer,inServerID as integer) As Integer
  dim i,j as integer
  j=ubound(TransferSocket)
  for i=1 to j
    if TransferSocket(i).isFinalized=false and strcomp(TransferSocket(i).partnerNick,inPartnerNick,0)=0 and TransferSocket(i).transferType=inTransferType and TransferSocket(i).serverID=inServerID then
      return i
    end
  next
  return c_error
End Function

SocketModule.TS_Disconnect:
Sub TS_Disconnect(inID as integer)
  dim aTransferArray as integer
  if TransferSocketLock.enter() then
    aTransferArray=TS_SearchID2Array(inID)
    if aTransferArray<>c_error then
      TransferSocket(aTransferArray).onError()
    end
    TransferSocketLock.leave()
  end
End Sub

SocketModule.OS_Disconnect:
Sub OS_Disconnect(inID as integer)
  dim aArray as integer
  if OpenNapSocketLock.enter() then
    aArray=OS_SearchID2Array(inID)
    if aArray<>c_error then
      OpenNapSocket(aArray).onError()
    end
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.TS_DisconnectServerIDAndNickAndTransferType:
Sub TS_DisconnectServerIDAndNickAndTransferType(inServerID as integer,inPartnerNick as string,inTransferType as integer)
  dim aTransferArray as integer
  if TransferSocketLock.enter() then
    aTransferArray=TS_SearchNickAndTransferTypeAndServerID2Array(inPartnerNick,inTransferType,inServerID)
    if aTransferArray<>c_error then
      TransferSocket(aTransferArray).onError()
    end
    TransferSocketLock.leave()
  end
End Sub

SocketModule.NS_Prepare:
Sub NS_Prepare()
  dim aArray,i as integer
  if NapListSocketLock.enter() then
    for i=1 to 2
      aArray=NS_Append()
      if aArray<>c_error then
        NapListSocket(aArray).ID=i
        NapListSocket(aArray).getList()
      end
    next
    NapListSocketLock.leave()
  end
End Sub

SocketModule.OS_Update:
Sub OS_Update()
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn then
        if OpenNapSocket(i).isWinMX=false then
          'change port
          OpenNapSocket(i).command(703,getDefine(str(iPort)))
        end
        'change Speed
        OpenNapSocket(i).command(700,getDefine(str(iSpeed)))
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.TS_PrepareTransfer:
Sub TS_PrepareTransfer(inServerID as integer,inNick as string,inIsWinMX as boolean,inIndex as Integer,inPartnerIP as string,inPartnerPort as integer,inArray as integer,inIsFirewall as boolean)
  dim aTransferArray as integer
  if TransferSocketLock.enter() then
    aTransferArray=TS_Append()
    if aTransferArray<>c_error then
      TransferSocket(aTransferArray).targetFileName=MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_FileName)
      TransferSocket(aTransferArray).targetShortName=MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_ShortName)
      TransferSocket(aTransferArray).targetFolderName=MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_FolderName)
      TransferSocket(aTransferArray).targetFileSize=val(MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Size))
      TransferSocket(aTransferArray).partnerNick=MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Nick)
      TransferSocket(aTransferArray).serverID=inServerID
      TransferSocket(aTransferArray).nick=inNick
      TransferSocket(aTransferArray).isWinMX=inIsWinMX
      TransferSocket(aTransferArray).transferVelocityTime=Microseconds
      TransferSocket(aTransferArray).targetArray=inArray
      MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_ID)=str(TransferSocket(aTransferArray).ID)
      if inArray=0 then
        if inIsFirewall then
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Status)=s_listening_for_a_connection
          TransferSocket(aTransferArray).transferType=2
          TransferSocket(aTransferArray).way=10'not connected
        else
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Status)=s_connecting
          TransferSocket(aTransferArray).address=inPartnerIP
          TransferSocket(aTransferArray).port=inPartnerPort
          TransferSocket(aTransferArray).transferType=1
          TransferSocket(aTransferArray).way=12'connecting
          TransferSocket(aTransferArray).connect()
        end
      else
        if inIsFirewall then
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Status)=s_connecting
          TransferSocket(aTransferArray).address=inPartnerIP
          TransferSocket(aTransferArray).port=inPartnerPort
          TransferSocket(aTransferArray).transferType=4
          TransferSocket(aTransferArray).way=22'connecting
          TransferSocket(aTransferArray).connect()
        else
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Flag)=""
          MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_Status)=s_listening_for_a_connection
          TransferSocket(aTransferArray).transferType=3
          TransferSocket(aTransferArray).way=20'not connected
        end
      end
      MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_ShortName)=MainWindow.TransferListBoxes(inArray).cell(inIndex,TR_ShortName)
      if TransferSocket(aTransferArray).targetFileSize=2147483647 then
        TransferSocket(aTransferArray).onError()
      end
    end
    TransferSocketLock.leave()
  end
End Sub

SocketModule.TS_PrepareBrowse:
Sub TS_PrepareBrowse(inServerID as integer,inNick as string,inPartnerNick as string,inPartnerIP as string,inPartnerPort as integer,inIsDownload as boolean,inIsFirewall as boolean)
  dim aTransferArray as integer
  if TransferSocketLock.enter() then
    aTransferArray=TS_Append()
    if aTransferArray<>c_error then
      TransferSocket(aTransferArray).partnerNick=inPartnerNick
      TransferSocket(aTransferArray).serverID=inServerID
      TransferSocket(aTransferArray).nick=inNick
      TransferSocket(aTransferArray).transferVelocityTime=Microseconds
      if inIsDownload then
        if inIsFirewall then
          TransferSocket(aTransferArray).transferType=6
          TransferSocket(aTransferArray).way=30'not connected
        else
          TransferSocket(aTransferArray).address=inPartnerIP
          TransferSocket(aTransferArray).port=inPartnerPort
          TransferSocket(aTransferArray).transferType=5
          TransferSocket(aTransferArray).way=32'connecting
          TransferSocket(aTransferArray).connect()
        end
      else
        if inIsFirewall then
          TransferSocket(aTransferArray).address=inPartnerIP
          TransferSocket(aTransferArray).port=inPartnerPort
          TransferSocket(aTransferArray).transferType=8
          TransferSocket(aTransferArray).way=42'connecting
          TransferSocket(aTransferArray).connect()
        else
          TransferSocket(aTransferArray).transferType=7
          TransferSocket(aTransferArray).way=40'not connected
        end
      end
    end
    TransferSocketLock.leave()
  end
End Sub

SocketModule.OS_DisconnectAll:
Sub OS_DisconnectAll()
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false then
        OpenNapSocket(i).onError()
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.TS_DisconnectAll:
Sub TS_DisconnectAll()
  dim i,j as integer
  if TransferSocketLock.enter() then
    j=ubound(TransferSocket)
    for i=1 to j
      if TransferSocket(i).isFinalized=false then
        TransferSocket(i).onError()
      end
    next
    TransferSocketLock.leave()
  end
End Sub

SocketModule.NS_DisconnectAll:
Sub NS_DisconnectAll()
  dim i,j as integer
  if NapListSocketLock.enter() then
    j=ubound(NapListSocket)
    for i=1 to j
      if NapListSocket(i).isFinalized=false then
        NapListSocket(i).onError()
      end
    next
    NapListSocketLock.leave()
  end
End Sub

SocketModule.OS_BroadcastSearch:
Sub OS_BroadcastSearch(inID as integer,inSearchString as string)
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn then
        OpenNapSocket(i).searchWindowID=inID
        'search query
        OpenNapSocket(i).command(200,inSearchString)
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.OS_BroadcastShareFile:
Sub OS_BroadcastShareFile(inLibrary as OriginalLibrary)
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn and OpenNapSocket(i).isShared then
        if OpenNapSocket(i).isWinMX=false then
          'client notification of shared file
          OpenNapSocket(i).command(100,chrb(34)+inLibrary.filePath+inLibrary.shortName+chrb(34)+chrb(32)+getDefine("0")+chrb(32)+inLibrary.size+chrb(32)+inLibrary.bitrate+chrb(32)+inLibrary.sampleRate+chrb(32)+inLibrary.time)
        else
          'client notification of shared file
          OpenNapSocket(i).command(100,chrb(34)+getDefine("D")+chrb(58)+chrb(92)+inLibrary.filePath+inLibrary.shortName+chrb(34)+chrb(32)+inLibrary.md5+chrb(32)+inLibrary.size+chrb(32)+inLibrary.bitrate+chrb(32)+inLibrary.sampleRate+chrb(32)+inLibrary.time)
        end
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.NS_Remove:
Sub NS_Remove()
  dim i as integer
  if NapListSocketLock.signal() then
    for i=ubound(NapListSocket) downto 1
      if NapListSocket(i).isFinalized then
        NapListSocket(i)=nil
        NapListSocket.Remove i
        removingObjects=removingObjects-1
      end
    next
    NapListSocketLock.release()
  end
End Sub

SocketModule.OS_Remove:
Sub OS_Remove()
  dim i as integer
  if OpenNapSocketLock.signal() then
    for i=ubound(OpenNapSocket) downto 1
      if OpenNapSocket(i).isFinalized then
        OpenNapSocket(i).remove()
        OpenNapSocket(i)=nil
        OpenNapSocket.Remove i
        removingObjects=removingObjects-1
      end
    next
    OpenNapSocketLock.release()
  end
End Sub

SocketModule.TS_Remove:
Sub TS_Remove()
  dim i as integer
  if TransferSocketLock.signal() then
    for i=ubound(TransferSocket) downto 1
      if TransferSocket(i).isFinalized then
        TransferSocket(i).remove()
        TransferSocket(i)=nil
        TransferSocket.remove i
        removingObjects=removingObjects-1
      end
    next
    TransferSocketLock.release()
  end
End Sub

SocketModule.OS_Search:
Sub OS_Search(inID as integer,inWindowID as integer,inSearchString as string)
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn and OpenNapSocket(i).ID=inID then
        OpenNapSocket(i).SearchWindowID=inWindowID
        'search query
        OpenNapSocket(i).command(200,inSearchString)
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.OS_Prepare:
Sub OS_Prepare(inIndex as integer)
  dim aOpenNapArray as integer
  if OpenNapSocketLock.enter() then
    aOpenNapArray=OS_Append()
    if aOpenNapArray<>c_error then
      MainWindow.NetworkListBox.cell(inIndex,NE_ID)=str(OpenNapSocket(aOpenNapArray).ID)
      MainWindow.NetworkListBox.cell(inIndex,NE_Icon)="   "
      MainWindow.NetworkListBox.cell(inIndex,NE_Comment)=""
      MainWindow.NetworkListBox.RowPicture(inIndex)=gPicture(r_connecting)
      
      if strcomp(MainWindow.NetworkListBox.cell(inIndex,NE_Nick),chrb(42),0)=0 or lenb(MainWindow.NetworkListBox.cell(inIndex,NE_Nick))=0 then
        OpenNapSocket(aOpenNapArray).nick=iNick
      else
        OpenNapSocket(aOpenNapArray).nick=MainWindow.NetworkListBox.cell(inIndex,NE_Nick)
      end
      if strcomp(MainWindow.NetworkListBox.cell(inIndex,NE_Password),chrb(42),0)=0 or lenb(MainWindow.NetworkListBox.cell(inIndex,NE_Password))=0 then
        OpenNapSocket(aOpenNapArray).password=iPassword
      else
        OpenNapSocket(aOpenNapArray).password=MainWindow.NetworkListBox.cell(inIndex,NE_Password)
      end
      OpenNapSocket(aOpenNapArray).isWinMX=isWinMX
      if strcomp(MainWindow.NetworkListBox.cell(inIndex,NE_Network),"n/a",0)=0 then
        OpenNapSocket(aOpenNapArray).network=MainWindow.NetworkListBox.cell(inIndex,NE_IP)
      else
        OpenNapSocket(aOpenNapArray).network=MainWindow.NetworkListBox.cell(inIndex,NE_Network)
      end
      OpenNapSocket(aOpenNapArray).address=MainWindow.NetworkListBox.cell(inIndex,NE_IP)
      OpenNapSocket(aOpenNapArray).port=val(MainWindow.NetworkListBox.cell(inIndex,NE_Port))
      OpenNapSocket(aOpenNapArray).isRedirect=MainWindow.NetworkListBox.cellCheck(inIndex,NE_Redirect)
      OpenNapSocket(aOpenNapArray).connect()
    end
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.TS_SearchNickAndTransferTypeAndWay2Array:
Function TS_SearchNickAndTransferTypeAndWay2Array(inPartnerNick as string,inTransferType as integer,inWay as integer) As Integer
  dim i,j as integer
  j=ubound(TransferSocket)
  for i=1 to j
    if TransferSocket(i).isFinalized=false and strcomp(TransferSocket(i).partnerNick,inPartnerNick,0)=0 and TransferSocket(i).transferType=inTransferType and TransferSocket(i).way=inWay then
      return i
    end
  next
  return c_error
End Function

SocketModule.TS_SearchTransferTypeAndWay2Array:
Function TS_SearchTransferTypeAndWay2Array(inTransferType as integer,inWay as integer) As Integer
  dim i,j as integer
  j=ubound(TransferSocket)
  for i=1 to j
    if TransferSocket(i).isFinalized=false and TransferSocket(i).transferType=inTransferType and TransferSocket(i).way=inWay then
      return i
    end
  next
  return c_error
End Function

SocketModule.TS_SearchNickAndFileNameAndTransferTypeAndWay2Array:
Function TS_SearchNickAndFileNameAndTransferTypeAndWay2Array(inPartnerNick as string,inFileName as string,inTransferType as integer,inWay as integer) As Integer
  dim i,j as integer
  j=ubound(TransferSocket)
  for i=1 to j
    if TransferSocket(i).isFinalized=false and strcomp(TransferSocket(i).partnerNick,inPartnerNick,0)=0 and strcomp(TransferSocket(i).targetFileName,inFileName,0)=0 and TransferSocket(i).transferType=inTransferType and TransferSocket(i).way=inWay then
      return i
    end
  next
  return c_error
End Function

SocketModule.TS_SearchNickAndFileNameAndFileSizeAndTransferTypeAndWay2Array:
Function TS_SearchNickAndFileNameAndFileSizeAndTransferTypeAndWay2Array(inPartnerNick as string,inFileName as string,inFileSize as integer,inTransferType as integer,inWay as integer) As Integer
  dim i,j as integer
  j=ubound(TransferSocket)
  for i=1 to j
    if TransferSocket(i).isFinalized=false and strcomp(TransferSocket(i).partnerNick,inPartnerNick,0)=0 and strcomp(TransferSocket(i).targetFileName,inFileName,0)=0 and TransferSocket(i).targetFileSize=inFileSize and TransferSocket(i).transferType=inTransferType and TransferSocket(i).way=inWay then
      return i
    end
  next
  return c_error
End Function

SocketModule.OS_BroadcastUnShareFile:
Sub OS_BroadcastUnShareFile(inLibrary as OriginalLibrary)
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn and OpenNapSocket(i).isShared then
        if OpenNapSocket(i).isWinMX=false then
          'unshare file
          OpenNapSocket(i).command(102,inLibrary.filePath+inLibrary.shortName)
        else
          'unshare file
          OpenNapSocket(i).command(102,getDefine("D")+chrb(58)+chrb(92)+inLibrary.filePath+inLibrary.shortName)
        end
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.OS_DisconnectConnectingServers:
Sub OS_DisconnectConnectingServers()
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn=false then
        OpenNapSocket(i).onError()
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.OS_DisconnectExcludeIDAndNetwork:
Sub OS_DisconnectExcludeIDAndNetwork(inID as integer,inNetwork as string)
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).ID<>inID and strcomp(OpenNapSocket(i).network,inNetwork,0)=0 then
        OpenNapSocket(i).onError()
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.TS_Download:
Sub TS_Download()
  dim i,j as integer
  if TransferSocketLock.enter() then
    j=ubound(TransferSocket)
    for i=1 to j
      if TransferSocket(i).isFinalized=false and TransferSocket(i).way=18 then'downloading
        TransferSocket(i).download()
      end
    next
    TransferSocketLock.leave()
  end
End Sub

SocketModule.OS_BroadcastShareAllFiles:
Sub OS_BroadcastShareAllFiles()
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn and OpenNapSocket(i).isShared=false then
        OpenNapSocket(i).write(sharedBuffer)
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

SocketModule.OS_SearchOnlineAndNetwork2Array:
Function OS_SearchOnlineAndNetwork2Array(inNetwork as string) As Integer
  dim i,j as integer
  j=ubound(OpenNapSocket)
  for i=1 to j
    if OpenNapSocket(i).isFinalized=false and OpenNapSocket(i).isLoggedIn and strcomp(OpenNapSocket(i).network,inNetwork,0)=0 then
      return i
    end
  next
  return c_error
End Function

TextStyleDataModule.appendTextStyleData:
Function appendTextStyleData(inTextStyleData as string) As string
  dim m as MemoryBlock
  
  m=NewMemoryBlock(3)
  if lenb(inTextStyleData)<>0 then
    m.PString(0)=leftb(inTextStyleData,2)
    m.Short(1)=m.Short(1)+1
    return m.PString(0)+midb(inTextStyleData,3)
  else
    m.Byte(0)=2
    m.Short(1)=1
    return m.PString(0)
  end
End Function

TextStyleDataModule.getFontName2ID:
Function getFontName2ID(inFontName As String) As Integer
  dim id As Integer
  dim m As MemoryBlock
  
  #if TargetCarbon then
    Declare Sub GetFNum Lib c_carbonlib (name As PString, ID As Ptr)
  #else
    Declare Sub GetFNum Lib "InterfaceLib" (name As PString, ID As Ptr)
  #endif
  m=NewMemoryBlock(2)
  GetFNum inFontName,m
  id=m.Short(0)
  
  return id
End Function

TextStyleDataModule.getStyleByte:
Function getStyleByte(inBold as boolean) As Integer
  if inBold then
    return 1
  end
End Function

TextStyleDataModule.getTextStyleData:
Function getTextStyleData(inStart as integer,inBold as boolean,inColor as color) As String
  dim m as MemoryBlock
  m=NewMemoryBlock(21)
  m.Byte(0)=20
  m.Long(1)=inStart
  m.Short(9)=gTextFontID
  m.Byte(11)=getStyleByte(inBold)
  m.Short(13)=iTextSize
  m.Byte(15)=inColor.Red
  m.Byte(16)=inColor.Red
  m.Byte(17)=inColor.Green
  m.Byte(18)=inColor.Green
  m.Byte(19)=inColor.Blue
  m.Byte(20)=inColor.Blue
  return m.PString(0)
End Function

TextStyleDataModule.getMessage:
Sub getMessage(inTextAndStyle as OriginalTextAndStyle,inNick as string,inNickColor as integer,inMessage as string,inMessageColor as integer)
  dim aStart as integer
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+getTimeStamp()
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,false,iColors(co_timestamp))
  
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+chrb(91)+inNick+chrb(93)+chrb(32)
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,false,iColors(inNickColor))
  
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+inMessage+chrb(13)
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,false,iColors(inMessageColor))
End Sub

TextStyleDataModule.getMessage:
Sub getMessage(inTextAndStyle as OriginalTextAndStyle,inMessage as string,inMessageColor as integer)
  dim aStart as integer
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+getTimeStamp()
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,false,iColors(co_timestamp))
  
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+inMessage+chrb(13)
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,false,iColors(inMessageColor))
End Sub

TextStyleDataModule.getMessage:
Sub getMessage(inTextAndStyle as OriginalTextAndStyle,inNick as string,inNickColor as integer,inMessage as string,inMessageColor as integer,inBold as boolean)
  dim aStart as integer
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+getTimeStamp()
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,false,iColors(co_timestamp))
  
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+chrb(91)+inNick+chrb(93)+chrb(32)
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,false,iColors(inNickColor))
  
  aStart=lenb(inTextAndStyle.text)
  inTextAndStyle.text=inTextAndStyle.text+inMessage+chrb(13)
  inTextAndStyle.textStyleData=appendTextStyleData(inTextAndStyle.textStyleData)+getTextStyleData(aStart,inBold,iColors(inMessageColor))
End Sub

AboutWindow.EnableMenuItems:
Sub EnableMenuItems()
  closeWindowMenu.enabled=true
  minimizeMenu.enabled=true
End Sub

AboutWindow.StaticTexts.Open:
Sub Open(Index As Integer)
  if Index=1 then
    me.text=s_version+chrb(32)+App.ShortVersion
  end
End Sub

AboutWindow.Canvases.Open:
Sub Open()
  me.backdrop=getMaskedPicture(rp_splash)
End Sub

AddChannelWindow.EditFields.KeyDown:
Function KeyDown(key as string) As Boolean
  return instrb(1,chrb(32)+chrb(34),key)<>0
End Function

AddChannelWindow.PushButtons.Action:
Sub Action(Index As Integer)
  select case Index
  case 0
    if lenb(self.EditFields.text)<>0 and self.PopupMenus.listIndex<>-1 then
      'join channel
      OS_Command(self.PopupMenus.rowtag(self.PopupMenus.listIndex),400,getString(trim(self.EditFields.text)))
      MainWindow.CH_Refresh()
      self.close()
    else
      getSound(rs_error)
      msgBox s_input_every_fields_
    end
  case 1
    self.close()
  end
End Sub

AddChannelWindow.PopupMenus.Open:
Sub Open()
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isLoggedIn then
        me.addrow(OpenNapSocket(i).network)
        me.rowTag(me.listCount-1)=OpenNapSocket(i).ID
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

AddServerWindow.EditFields.Open:
Sub Open(Index As Integer)
  select case Index
  case 0
    me.text="n/a"
  case 2
    me.text="8888"
  case 4
    me.text=chrb(42)
  case 5
    me.text=chrb(42)
  end
End Sub

AddServerWindow.EditFields.KeyDown:
Function KeyDown(Index As Integer, key as string) As Boolean
  select case index
  case 4
    return instrb(1,c_invalid_keys+chrb(32),key)<>0
  case 5
    return instrb(1,c_invalid_keys+chrb(32),key)<>0
  end
End Function

AddServerWindow.PushButtons.Action:
Sub Action(Index As Integer)
  dim aIndex as integer
  select case Index
  case 0
    if lenb(self.EditFields(0).text)<>0 and lenb(self.EditFields(1).text)<>0 and lenb(self.EditFields(2).text)<>0 and lenb(self.EditFields(4).text)<>0 and lenb(self.EditFields(5).text)<>0 then
      if NetworkWindowLock.enter() then
        aIndex=self.targetIndex
        if aIndex=c_error then
          aIndex=MainWindow.NE_Append(getString(trim(self.EditFields(0).text)),getString(trim(self.EditFields(1).text)),getString(trim(self.EditFields(2).text)),self.CheckBoxes.value,getString(trim(self.EditFields(3).text)),getString(trim(self.EditFields(4).text)),getString(trim(self.EditFields(5).text)),false)
        else
          MainWindow.NE_Replace(aIndex,getString(trim(self.EditFields(0).text)),getString(trim(self.EditFields(1).text)),getString(trim(self.EditFields(2).text)),self.CheckBoxes.value,getString(trim(self.EditFields(3).text)),getString(trim(self.EditFields(4).text)),getString(trim(self.EditFields(5).text)))
        end
        MainWindow.NetworkListBox.listIndex=aIndex
        MainWindow.NetworkListBox.scrollposition=aIndex
        NetworkWindowLock.leave()
      end
      self.close()
      
    else
      getSound(rs_error)
      msgBox s_input_every_fields_
      
    end
    
  case 1
    self.close()
    
  end
End Sub

AddUserWindow.EditFields.KeyDown:
Function KeyDown(key as string) As Boolean
  return instrb(1,c_invalid_keys+chrb(32),key)<>0
End Function

AddUserWindow.PushButtons.Action:
Sub Action(Index As Integer)
  select case Index
  case 0
    if lenb(self.EditFields.text)<>0 and self.PopupMenus.listIndex<>-1 then
      MainWindow.US_Append(self.PopupMenus.rowtag(self.PopupMenus.listIndex),getString(trim(self.EditFields.text)),self.RadioButtons(0).value)
      self.close()
    else
      getSound(rs_error)
      msgBox s_input_every_fields_
    end
  case 1
    self.close()
  end
End Sub

AddUserWindow.RadioButtons.Action:
Sub Action(Index As Integer)
  if index=0 then
    self.RadioButtons(1).value=not me.value
  else
    self.RadioButtons(0).value=not me.value
  end
End Sub

AddUserWindow.PopupMenus.Open:
Sub Open()
  dim i,j as integer
  if OpenNapSocketLock.enter() then
    j=ubound(OpenNapSocket)
    for i=1 to j
      if OpenNapSocket(i).isLoggedIn then
        me.addrow(OpenNapSocket(i).network)
        me.rowTag(me.listCount-1)=OpenNapSocket(i).ID
      end
    next
    OpenNapSocketLock.leave()
  end
End Sub

DrawerWindow.DR_Append:
Sub DR_Append(inName as string,inID as integer,inType as integer)
  dim aIndex as integer
  if DrawerWindowLock.enter() then
    aIndex=DR_SearchType2LastIndex(inType)
    if aIndex<>c_error then
      me.TaskListBox.insertRow(aIndex+1,"")
    else
      me.TaskListBox.addrow("")
    end
    aIndex=me.TaskListBox.lastIndex
    me.TaskListBox.cell(aIndex,DR_Name)=inName
    me.TaskListBox.cell(aIndex,DR_ID)=str(inID)
    me.TaskListBox.cell(aIndex,DR_Type)=str(inType)
    select case inType
    case c_search_window
      me.TaskListBox.rowPicture(aIndex)=gPicture(r_search)
    case c_browse_window
      me.TaskListBox.rowPicture(aIndex)=gPicture(r_browse)
    case c_message_window
      me.TaskListBox.rowPicture(aIndex)=gPicture(r_message)
    case c_room_window
      me.TaskListBox.rowPicture(aIndex)=gPicture(r_channel)
    end
    me.TaskListBox.scrollposition=aIndex
    DrawerWindowLock.leave()
  end
End Sub

DrawerWindow.DR_Close:
Sub DR_Close(inID as integer,inType as integer)
  dim aArray,aIndex as integer
  aIndex=me.TaskListBox.listIndex
  select case inType
  case c_search_window
    if SearchWindowLock.enter() then
      aArray=SE_SearchID2Array(inID)
      if aArray<>c_error then
        me.DR_Remove(inID)
        SearchWindow(aArray).isFinalized=true
      end
      SearchWindowLock.leave()
    end
  case c_browse_window
    if BrowseWindowLock.enter() then
      aArray=BR_SearchID2Array(inID)
      if aArray<>c_error then
        me.DR_Remove(inID)
        TS_DisconnectServerIDAndNickAndTransferType(BrowseWindow(aArray).serverID,BrowseWindow(aArray).partnerNick,5)
        TS_DisconnectServerIDAndNickAndTransferType(BrowseWindow(aArray).serverID,BrowseWindow(aArray).partnerNick,6)
        BrowseWindow(aArray).isFinalized=true
      end
      BrowseWindowLock.leave()
    end
  case c_message_window
    if MessageWindowLock.enter() then
      aArray=ME_SearchID2Array(inID)
      if aArray<>c_error then
        me.DR_Remove(inID)
        MessageWindow(aArray).isFinalized=true
      end
      MessageWindowLock.leave()
    end
  case c_room_window
    if RoomWindowLock.enter() then
      aArray=RO_SearchID2Array(inID)
      if aArray<>c_error then
        me.DR_Remove(inID)
        'part channel
        OS_Command(RoomWindow(aArray).serverID,401,RoomWindow(aArray).channel)
        RoomWindow(aArray).isFinalized=true
      end
      RoomWindowLock.leave()
    end
  end
  me.TaskListBox.listIndex=min(aIndex,me.TaskListBox.listCount-1)
End Sub

DrawerWindow.DR_MessageArrived:
Sub DR_MessageArrived(inID as integer)
  dim aIndex as integer
  if DrawerWindowLock.enter() then
    aIndex=me.DR_SearchID2ListIndex(inID)
    if aIndex<>c_error and me.TaskListBox.listIndex<>aIndex then
      me.TaskListBox.RowPicture(aIndex)=gPicture(r_arrived)
    end
    DrawerWindowLock.leave()
  end
End Sub

DrawerWindow.DR_Remove:
Sub DR_Remove(inID as integer)
  dim aIndex as integer
  if DrawerWindowLock.signal() then
    aIndex=me.DR_SearchID2ListIndex(inID)
    if aIndex<>c_error then
      me.TaskListBox.removeRow(aIndex)
    end
    DrawerWindowLock.release()
  end
End Sub

DrawerWindow.DR_SearchID2ListIndex:
Protected Function DR_SearchID2ListIndex(inID as integer) As Integer
  dim i,j as integer
  j=me.TaskListBox.listCount-1
  for i=0 to j
    if val(me.TaskListBox.cell(i,DR_ID))=inID then
      return i
    end
  next
  return c_error
End Function

DrawerWindow.DR_Show:
Protected Sub DR_Show(inID as integer,inType as integer)
  dim aArray as integer
  select case inType
  case c_search_window
    if gSearchWindowID<>inID then
      gSearchWindowID=inID
      MainWindow.isChanged=true
      MainWindow.PagePanels.value=7
    elseif MainWindow.PagePanels.value<>7 then
      MainWindow.PagePanels.value=7
    end
  case c_browse_window
    if gBrowseWindowID<>inID then
      gBrowseWindowID=inID
      MainWindow.isChanged=true
      MainWindow.PagePanels.value=8
    elseif MainWindow.PagePanels.value<>8 then
      MainWindow.PagePanels.value=8
    end
  case c_message_window
    if gMessageWindowID<>inID then
      gMessageWindowID=inID
      MainWindow.isChanged=true
      MainWindow.PagePanels.value=9
    elseif MainWindow.PagePanels.value<>9 then
      MainWindow.PagePanels.value=9
    end
  case c_room_window
    if gRoomWindowID<>inID then
      gRoomWindowID=inID
      MainWindow.isChanged=true
      MainWindow.PagePanels.value=10
    elseif MainWindow.PagePanels.value<>10 then
      MainWindow.PagePanels.value=10
    end
  end
End Sub

DrawerWindow.DR_Update:
Sub DR_Update(inID as integer,inWhat as string,inResults as string)
  dim aIndex as integer
  if DrawerWindowLock.enter() then
    aIndex=me.DR_SearchID2ListIndex(inID)
    if aIndex<>c_error then
      me.TaskListBox.cell(aIndex,DR_Results)=inResults
      me.TaskListBox.cell(aIndex,DR_Name)=inWhat
    end
    DrawerWindowLock.leave()
  end
End Sub

DrawerWindow.DR_UpdateResults:
Sub DR_UpdateResults(inID as integer,inWhat as string,inFilterValue as integer,inTotalvalue as integer)
  dim aIndex as integer
  if DrawerWindowLock.enter() then
    aIndex=me.DR_SearchID2ListIndex(inID)
    if aIndex<>c_error then
      if inFilterValue<>inTotalvalue then
        me.TaskListBox.cell(aIndex,DR_Results)=str(inFIlterValue)+chrb(47)+str(inTotalvalue)
      else
        me.TaskListBox.cell(aIndex,DR_Results)=str(inTotalvalue)
      end
      me.TaskListBox.cell(aIndex,DR_Name)=inWhat
    end
    DrawerWindowLock.leave()
  end
End Sub

DrawerWindow.DR_SearchAgain:
Protected Sub DR_SearchAgain(inID as integer)
  dim aArray as integer
  if SearchWindowLock.signal() then
    aArray=SE_SearchID2Array(inID)
    if aArray<>c_error then
      redim SearchWindow(aArray).SearchList(0)
      MainWindow.SearchListBox.deleteAllRows()
      MainWindow.SearchListBox.scrollPosition=0
      MainWindow.SearchListBox.headingIndex=-1
    end
    SearchWindowLock.release()
  end
  
  if SearchWindowLock.enter() then
    aArray=SE_SearchID2Array(inID)
    if aArray<>c_error then
      MainWindow.SE_Search(SearchWindow(aArray).title,inID)
    end
    SearchWindowLock.leave()
  end
End Sub

DrawerWindow.DR_SearchGoogle:
Protected Sub DR_SearchGoogle(inID as integer)
  dim aArray as integer
  if SearchWindowLock.enter() then
    aArray=SE_SearchID2Array(inID)
    if aArray<>c_error then
      showURL getDefine("http://www.google.com/search?q=")+SearchWindow(aArray).title
    end
    SearchWindowLock.leave()
  end
End Sub

DrawerWindow.DR_BrowseAgain:
Protected Sub DR_BrowseAgain(inID as integer)
  dim aArray as integer
  if BrowseWindowLock.signal() then
    aArray=BR_SearchID2Array(inID)
    if aArray<>c_error then
      TS_DisconnectServerIDAndNickAndTransferType(BrowseWindow(aArray).serverID,BrowseWindow(aArray).partnerNick,5)
      TS_DisconnectServerIDAndNickAndTransferType(BrowseWindow(aArray).serverID,BrowseWindow(aArray).partnerNick,6)
      BrowseWindow(aArray).totalSize=0
      redim BrowseWindow(aArray).BrowseList(0)
      MainWindow.BrowseListBox.deleteAllRows()
      MainWindow.BrowseListBox.scrollPosition=0
      MainWindow.BrowseListBox.headingIndex=-1
      MainWindow.FolderListBox.cell(0,BR_Icon)=""
      MainWindow.FolderListBox.expanded(0)=false
      MainWindow.FolderListBox.scrollPosition=0
    end
    BrowseWindowLock.release()
  end
  
  if BrowseWindowLock.enter() then
    aArray=BR_SearchID2Array(inID)
    if aArray<>c_error then
      me.DR_Update(inID,BrowseWindow(aArray).partnerNick,"")
      'browse request
      OS_Command(BrowseWindow(aArray).serverID,211,BrowseWindow(aArray).partnerNick)
    end
    BrowseWindowLock.leave()
  end
End Sub

DrawerWindow.DR_SearchType2LastIndex:
Protected Function DR_SearchType2LastIndex(inType as integer) As Integer
  dim i as integer
  for i=me.TaskListBox.listCount-1 downto 0
    if val(me.TaskListBox.cell(i,DR_Type))=inType then
      return i
    end
  next
  return c_error
End Function

DrawerWindow.DR_SelectMenu:
Sub DR_SelectMenu(inID as integer)
  dim aIndex as integer
  if DrawerWindowLock.enter() then
    aIndex=me.DR_SearchID2ListIndex(inID)
    if aIndex<>c_error then
      me.TaskListBox.listIndex=aIndex
    end
    DrawerWindowLock.leave()
  end
End Sub

DrawerWindow.Close:
Sub Close()
  #if targetPPC
    writeWindow(me,c_drawer_window)
  #endif
End Sub

DrawerWindow.Resized:
Sub Resized()
  #if targetCarbon
    me.height=me.height
    me.width=me.width
    me.left=me.left
    me.top=me.top
  #endif
End Sub

DrawerWindow.Open:
Sub Open()
  #if targetPPC
    getWindow(me,c_drawer_window)
  #endif
End Sub

DrawerWindow.MenuListBox.Change:
Sub Change()
  dim aIndex as integer
  aIndex=me.listIndex
  if aIndex<>-1 then
    MainWindow.PagePanels.value=aIndex
    select case aIndex
    case 0
      me.rowPicture(0)=gPicture(r_connecting)
    case 4
      me.rowPicture(4)=gPicture(r_user_list)
    case 5
      me.rowPicture(5)=gPicture(r_download)
    case 6
      me.rowPicture(6)=gPicture(r_upload)
    end
    self.TaskListBox.listIndex=-1
  end
End Sub

DrawerWindow.MenuListBox.CellTextPaint:
Function CellTextPaint(g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  dim sw,rw,l as integer
  dim s as string
  if column=DR_Name then
    if me.selected(row) then
      g.foreColor=rgb(255,255,255)
    end
    g.drawstring me.cell(row,column),x,y
    g.bold=true
    l=7
    select case row
    case 0'network
      if gOnline<>0 then
        s=str(gOnline)
        sw=g.stringWidth(s)
        rw=max(sw+8,16)
        g.foreColor=rgb(&h00,&hC2,&h00)
        g.fillRoundRect g.width-rw-l,3,rw,16,16,16
        g.foreColor=rgb(&hFF,&hFF,&hFF)
        g.drawstring s,g.width-rw-l+((rw-sw)/2),y
      end
    case 4'user list
      if gHotlist<>0 then
        s=str(gHotlist)
        sw=g.stringWidth(s)
        rw=max(sw+8,16)
        g.foreColor=rgb(&h00,&hC2,&h00)
        g.fillRoundRect g.width-rw-l,3,rw,16,16,16
        g.foreColor=rgb(&hFF,&hFF,&hFF)
        g.drawstring s,g.width-rw-l+((rw-sw)/2),y
      end
    case 5'downloading
      if gDownloadQueue<>0 then
        s=str(gDownloadQueue)
        sw=g.stringWidth(s)
        rw=max(sw+8,16)
        g.foreColor=rgb(&hCC,&hCC,&hCC)
        g.fillRoundRect g.width-rw-l,4,rw,14,14,14
        g.foreColor=rgb(&h65,&h65,&h65)
        g.drawstring s,g.width-rw-l+((rw-sw)/2),y
        l=l+rw-2
      end
      if gDownloading<>0 then
        s=str(gDownloading)
        sw=g.stringWidth(s)
        rw=max(sw+8,16)
        g.foreColor=rgb(&h33,&h73,&hF1)
        g.fillRoundRect g.width-rw-l,3,rw,16,16,16
        g.foreColor=rgb(&hFF,&hFF,&hFF)
        g.drawstring s,g.width-rw-l+((rw-sw)/2),y
      end
    case 6'uploading
      if gUploadQueue<>0 then
        s=str(gUploadQueue)
        sw=g.stringWidth(s)
        rw=max(sw+8,16)
        g.foreColor=rgb(&hCC,&hCC,&hCC)
        g.fillRoundRect g.width-rw-l,4,rw,14,14,14
        g.foreColor=rgb(&h65,&h65,&h65)
        g.drawstring s,g.width-rw-l+((rw-sw)/2),y
        l=l+rw-2
      end
      if gUploading<>0 then
        s=str(gUploading)
        sw=g.stringWidth(s)
        rw=max(sw+8,16)
        g.foreColor=rgb(&h99,&h99,&h99)
        g.fillRoundRect g.width-rw-l,3,rw,16,16,16
        g.foreColor=rgb(&hFF,&hFF,&hFF)
        g.drawstring s,g.width-rw-l+((rw-sw)/2),y
      end
    end
  end
  return true
End Function

DrawerWindow.MenuListBox.Open:
Sub Open()
  me.addrow("")
  me.cell(0,DR_Name)=s_network
  me.cell(0,DR_Count)="0"
  me.rowPicture(0)=gPicture(r_connecting)
  
  me.addrow("")
  me.cell(1,DR_Name)=s_console
  me.rowPicture(1)=gPicture(r_console)
  
  me.addrow("")
  me.cell(2,DR_Name)=s_channel
  me.rowPicture(2)=gPicture(r_channel)
  
  me.addrow("")
  me.cell(3,DR_Name)=s_whois
  me.rowPicture(3)=gPicture(r_whois)
  
  me.addrow("")
  me.cell(4,DR_Name)=s_user_list
  me.cell(4,DR_Count)="0"
  me.rowPicture(4)=gPicture(r_user_list)
  
  me.addrow("")
  me.cell(5,DR_Name)=s_downloads
  me.cell(5,DR_Count)="0/0"
  me.rowPicture(5)=gPicture(r_download)
  
  me.addrow("")
  me.cell(6,DR_Name)=s_uploads
  me.cell(6,DR_Count)="0/0"
  me.rowPicture(6)=gPicture(r_upload)
  
  #if targetCarbon
    me.listIndex=0
  #else
    me.listIndex=1
  #endif
End Sub

DrawerWindow.MenuListBox.CellBackgroundPaint:
Function CellBackgroundPaint(g as graphics,row as integer,column as integer) As Boolean
  if me.selected(row) then
    if me.active then
      g.drawPicture gPicture(r_cell_activated),0,0,g.width,g.height,0,0,gPicture(r_cell_activated).width,gPicture(r_cell_activated).height
    else
      g.drawPicture gPicture(r_cell_deactivated),0,0,g.width,g.height,0,0,gPicture(r_cell_deactivated).width,gPicture(r_cell_deactivated).height
    end
  end
  return true
End Function

DrawerWindow.TaskContextualMenu.Action:
Sub Action(item As String)
  dim aIndex,i as integer
  aIndex=self.TaskListBox.listindex
  select case item
  case s_remove
    if aIndex<>-1 then
      self.DR_Close(val(self.TaskListBox.cell(aIndex,DR_ID)),val(self.TaskListBox.cell(aIndex,DR_Type)))
    end
  case s_remove_all
    for i=self.TaskListBox.listCount-1 downto 0
      self.DR_Close(val(self.TaskListBox.cell(i,DR_ID)),val(self.TaskListBox.cell(i,DR_Type)))
    next
    self.TaskListBox.listindex=0
  case s_search_again
    if aIndex<>-1 then
      self.DR_SearchAgain(val(self.TaskListBox.cell(aIndex,DR_ID)))
    end
  case s_google_search
    if aIndex<>-1 then
      self.DR_SearchGoogle(val(self.TaskListBox.cell(aIndex,DR_ID)))
    end
  case s_browse_again
    if aIndex<>-1 then
      self.DR_BrowseAgain(val(self.TaskListBox.cell(aIndex,DR_ID)))
    end
  end
End Sub

DrawerWindow.TaskListBox.CellTextPaint:
Function CellTextPaint(g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  dim sw,rw as integer
  dim s as string
  if column=DR_Name then
    if me.selected(row) then
      g.foreColor=rgb(255,255,255)
    end
    g.drawstring me.cell(row,column),x,y
    s=me.cell(row,DR_Results)
    if lenb(s)<>0 then
      sw=g.stringWidth(s)
      rw=max(sw+8,16)
      g.foreColor=rgb(235,245,255)
      g.fillRoundRect g.width-rw-7,3,rw,16,16,16
      g.foreColor=rgb(&h00,&h00,&h00)
      g.drawstring s,g.width-rw-7+((rw-sw)/2),y
    end
  end
  return true
End Function

DrawerWindow.TaskListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  dim aIndex as integer
  aIndex=me.listindex
  if aIndex<>-1 and ascb(key)=8 then
    self.DR_Close(val(me.cell(aIndex,DR_ID)),val(me.cell(aIndex,DR_Type)))
  end
End Function

DrawerWindow.TaskListBox.Change:
Sub Change()
  dim aIndex,aID as integer
  aIndex=me.listIndex
  if aIndex<>-1 then
    aID=val(me.cell(aIndex,DR_Type))
    self.DR_Show(val(me.cell(aIndex,DR_ID)),aID)
    select case aID
    case c_message_window
      me.rowPicture(aIndex)=gPicture(r_message)
    case c_room_window
      me.rowPicture(aIndex)=gPicture(r_channel)
    end
    self.MenuListBox.listIndex=-1
  end
End Sub

DrawerWindow.TaskListBox.CMMClicked:
Sub CMMClicked()
  dim aIndex,aType as integer
  aIndex=me.listindex
  self.TaskContextualMenu.deleteAllRows()
  if aIndex<>-1 and val(me.cell(aIndex,DR_ID))<>0 then
    self.TaskContextualMenu.addrow(s_remove)
  end
  self.TaskContextualMenu.addrow(s_remove_all)
  if aIndex<>-1 and val(me.cell(aIndex,DR_ID))<>0 then
    aType=val(me.cell(aIndex,DR_Type))
    select case aType
    case c_search_window
      self.TaskContextualMenu.addseparator()
      self.TaskContextualMenu.addrow(s_search_again)
      self.TaskContextualMenu.addrow(s_google_search)
    case c_browse_window
      self.TaskContextualMenu.addseparator()
      self.TaskContextualMenu.addrow(s_browse_again)
    end
  end
  self.TaskContextualMenu.open()
End Sub

DrawerWindow.TaskListBox.Open:
Sub Open()
  me.addrow("")
  me.cell(0,DR_Name)=s_new_search
  me.cell(0,DR_Type)=str(c_search_window)
  me.cell(0,DR_ID)="0"
  me.rowPicture(0)=gPicture(r_search)
End Sub

DrawerWindow.TaskListBox.CellBackgroundPaint:
Function CellBackgroundPaint(g as graphics,row as integer,column as integer) As Boolean
  if me.selected(row) then
    if me.active then
      g.drawPicture gPicture(r_cell_activated),0,0,g.width,g.height,0,0,gPicture(r_cell_activated).width,gPicture(r_cell_activated).height
    else
      g.drawPicture gPicture(r_cell_deactivated),0,0,g.width,g.height,0,0,gPicture(r_cell_deactivated).width,gPicture(r_cell_deactivated).height
    end
  end
  return true
End Function

MainWindow.NE_Append:
Function NE_Append(inName as string,inIP as string,inPort as string,inIsRedirect as boolean,inURL as string,inNick as string,inPassword as string,inIsNapList as boolean) As Integer
  dim aIndex as integer
  dim aName,aIP as string
  if strcomp(inName,"N/A",0)=0 then
    aName=lowercase(inName)
  else
    aName=inName
  end
  aIP=lowercase(inIP)
  aIndex=me.NE_SearchIPAndPort2ListIndex(aIP,inPort)
  if aIndex=c_error then
    me.NetworkListBox.addrow("")
    aIndex=me.NetworkListBox.lastIndex
    me.NetworkListBox.cell(aIndex,NE_Network)=aName
    me.NetworkListBox.cell(aIndex,NE_IP)=aIP
    me.NetworkListBox.cell(aIndex,NE_Port)=inPort
    me.NetworkListBox.cellCheck(aIndex,NE_Redirect)=inIsRedirect
    me.NetworkListBox.cell(aIndex,NE_URL)=inURL
    me.NetworkListBox.cell(aIndex,NE_Comment)=""
    me.NetworkListBox.cell(aIndex,NE_Nick)=inNick
    me.NetworkListBox.cell(aIndex,NE_Password)=inPassword
    me.NetworkListBox.cell(aIndex,NE_ID)=""
  elseif inIsNapList=false then
    me.NetworkListBox.cell(aIndex,NE_Network)=aName
    me.NetworkListBox.cellCheck(aIndex,NE_Redirect)=inIsRedirect
    me.NetworkListBox.cell(aIndex,NE_Nick)=inNick
    me.NetworkListBox.cell(aIndex,NE_Password)=inPassword
  end
  return aIndex
End Function

MainWindow.NE_AutoConnect:
Sub NE_AutoConnect()
  if NetworkWindowLock.enter() then
    me.targetListIndex=me.targetListIndex+1
    if me.targetListIndex>me.NetworkListBox.listCount-1 then
      me.targetListIndex=0
      NetworkTimer.period=600000
    elseif val(me.NetworkListBox.cell(me.targetListIndex,NE_ID))=0 and me.NE_SearchNameAndOnlineStatus2ListIndex(me.NetworkListBox.cell(me.targetListIndex,NE_Network))=c_error then
      OS_Prepare(me.targetListIndex)
    end
    NetworkWindowLock.leave()
  end
End Sub

MainWindow.NE_Connect:
Protected Sub NE_Connect()
  dim i,j as integer
  if NetworkWindowLock.enter() then
    if me.NetworkListBox.listindex<>-1 then
      j=me.NetworkListBox.listCount-1
      for i=0 to j
        if me.NetworkListBox.selected(i) and val(me.NetworkListBox.cell(i,NE_ID))=0 then
          OS_Prepare(i)
        end
      next
    end
    NetworkWindowLock.leave()
  end
End Sub

MainWindow.NE_Disconnect:
Protected Sub NE_Disconnect()
  dim i,j,aServerID as integer
  if NetworkWindowLock.enter() then
    if me.NetworkListBox.listindex<>-1 then
      j=me.NetworkListBox.listCount-1
      for i=0 to j
        if me.NetworkListBox.selected(i) then
          aServerID=val(me.NetworkListBox.cell(i,NE_ID))
          if aServerID<>0 then
            OS_Disconnect(aServerID)
          end
        end
      next
    end
    NetworkWindowLock.leave()
  end
End Sub

MainWindow.NE_Export:
Protected Sub NE_Export()
  dim tos as TextOutputStream
  dim i,j as integer
  dim d as saveAsDialog
  dim fi as folderItem
  dim t as date
  t=new date
  d=new saveAsDialog
  d.suggestedFileName=t.shortDate+".wsx"
  fi=d.showModalWithin(me)
  if fi<>nil then
    if NetworkWindowLock.enter() then
      tos=fi.CreateTextFile()
      j=me.NetworkListBox.listCount-1
      for i=0 to j
        tos.write(getDefine("N: ")+me.NetworkListBox.cell(i,NE_Network)+chrb(13)+chrb(10))
        tos.write(getDefine("P: ")+me.NetworkListBox.cell(i,NE_Port)+chrb(13)+chrb(10))
        if me.NetworkListBox.cellCheck(i,NE_Redirect) then
          tos.write(getDefine("T: OPENNAP041 RE")+chrb(13)+chrb(10))
        else
          tos.write(getDefine("T: OPENNAP041")+chrb(13)+chrb(10))
        end
        tos.write(getDefine("A: ")+me.NetworkListBox.cell(i,NE_IP)+chrb(13)+chrb(10))
        tos.write(chrb(13)+chrb(10))
      next
      tos.close()
      NetworkWindowLock.leave()
    end
  end
exception
  if tos<>nil then
    tos.close()
  end
  NetworkWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.NE_Import:
Protected Sub NE_Import()
  dim d as openDialog
  dim fi as folderItem
  d=new openDialog
  d.Filter="special/any"
  fi=d.showModalWithin(me)
  if fi<>nil then
    me.NE_ImportWSX(fi)
  end
End Sub

MainWindow.NE_ImportWSX:
Protected Sub NE_ImportWSX(inFi as folderItem)
  dim tis as textInputStream
  dim aIndex as integer
  dim network,ip,port,temp,header as string
  dim isRedirect as boolean
  if NetworkWindowLock.enter() then
    me.NetworkListBox.visible=false
    tis=inFi.openAsTextFile()
    do until tis.eof
      temp=""
      temp=tis.readline()
      header=""
      header=leftb(temp,3)
      temp=midb(temp,4)
      
      if strcomp(header,"N: ",0)=0 then
        network=""
        ip=""
        port=""
        port="8888"
        isRedirect=false
        if ascb(leftb(temp,1))=42 then
          network=midb(temp,2)
        else
          network=temp
        end
        
      elseif strcomp(header,"T: ",0)=0 then
        isRedirect=(instrb(1,temp,"META")<>0 or instrb(1,temp,"RE")<>0)
        
      elseif strcomp(header,"P: ",0)=0 then
        port=temp
        
      elseif strcomp(header,"A: ",0)=0 then
        if instrb(1,temp,chrb(58))<>0 then
          ip=nthfield(temp,chrb(58),1)
          port=nthfield(temp,chrb(58),2)
        else
          ip=temp
        end
        
        if strcomp(leftb(ip,2),"0x",0)=0 then
          ip=replaceallb(ip,"0x","")
          ip=replaceallb(ip,chrb(46),"")
          ip=str(val("&h"+leftb(ip,2)))+chrb(46)+str(val("&h"+midb(ip,3,2)))+chrb(46)+str(val("&h"+midb(ip,5,2)))+chrb(46)+str(val("&h"+midb(ip,7,2)))
        elseif instrb(1,ip,chrb(46))=0 then
          ip=getLongIP2ShortIP(ip)
        end
        
        if lenb(network)<>0 and lenb(ip)<>0 and val(port)<>0 then
          aIndex=me.NE_Append(getDefine(network),getDefine(ip),getDefine(port),isRedirect,"",chrb(42),chrb(42),true)
        end
        
      end
    loop
    tis.close()
    me.NetworkListBox.visible=true
    NetworkWindowLock.leave()
  end
exception
  if tis<>nil then
    tis.close
  end
  me.NetworkListBox.visible=true
  NetworkWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_reading_error
End Sub

MainWindow.NE_Modify:
Protected Sub NE_Modify(inIsModify as boolean)
  dim aIndex as integer
  if NetworkWindowLock.enter() then
    aIndex=me.NetworkListBox.listindex
    if inIsModify then
      if aIndex<>-1 then
        AddServerWindow.hide()
        AddServerWindow.targetIndex=aIndex
        AddServerWindow.EditFields(0).text=me.NetworkListBox.cell(aIndex,NE_Network)
        AddServerWindow.EditFields(1).text=me.NetworkListBox.cell(aIndex,NE_IP)
        AddServerWindow.EditFields(2).text=me.NetworkListBox.cell(aIndex,NE_Port)
        AddServerWindow.EditFields(3).text=me.NetworkListBox.cell(aIndex,NE_URL)
        AddServerWindow.EditFields(4).text=me.NetworkListBox.cell(aIndex,NE_Nick)
        AddServerWindow.EditFields(5).text=me.NetworkListBox.cell(aIndex,NE_Password)
        AddServerWindow.CheckBoxes.value=me.NetworkListBox.cellcheck(aIndex,NE_Redirect)
        AddServerWindow.showWithin(me)
      end
      
    else
      AddServerWindow.hide()
      AddServerWindow.targetIndex=c_error
      if aIndex<>-1 then
        AddServerWindow.EditFields(0).text=me.NetworkListBox.cell(aIndex,NE_Network)
        AddServerWindow.EditFields(3).text=me.NetworkListBox.cell(aIndex,NE_URL)
        AddServerWindow.EditFields(4).text=me.NetworkListBox.cell(aIndex,NE_Nick)
        AddServerWindow.EditFields(5).text=me.NetworkListBox.cell(aIndex,NE_Password)
      end
      AddServerWindow.showWithin(me)
    end
    
    NetworkWindowLock.leave()
  end
End Sub

MainWindow.NE_Offline:
Sub NE_Offline(inServerID as integer,inErrorCode as integer,inIsRedirectAvailable as boolean,inAddress as string)
  dim aIndex as integer
  if iClearServer and inErrorCode=103 and inIsRedirectAvailable=false then
    if NetworkWindowLock.signal() then
      aIndex=me.NE_SearchID2ListIndex(inServerID)
      if aIndex<>c_error then
        me.NetworkListBox.removeRow aIndex
      end
      NetworkWindowLock.release()
    end
  end
  
  if NetworkWindowLock.enter() then
    aIndex=me.NE_SearchID2ListIndex(inServerID)
    if aIndex<>c_error then
      if inErrorCode=103 then
        if inIsRedirectAvailable then
          me.NetworkListBox.cell(aIndex,NE_Icon)=""
          me.NetworkListBox.cell(aIndex,NE_Comment)=s_server_not_found+chrb(32)+inAddress
        else
          me.NetworkListBox.cell(aIndex,NE_Icon)=" "
          me.NetworkListBox.cell(aIndex,NE_Comment)=s_server_not_found
        end
        me.NetworkListBox.rowPicture(aIndex)=gPicture(r_unknown)
      else
        me.NetworkListBox.cell(aIndex,NE_Icon)=""
        me.NetworkListBox.rowPicture(aIndex)=nil
      end
      me.NetworkListBox.cell(aIndex,NE_ID)=""
    end
    NetworkWindowLock.leave()
  end
End Sub

MainWindow.NE_Online:
Sub NE_Online(inServerID as integer)
  dim aIndex as integer
  if NetworkWindowLock.enter() then
    aIndex=me.NE_SearchID2ListIndex(inServerID)
    if aIndex<>c_error then
      me.NetworkListBox.cell(aIndex,NE_Icon)="    "
      me.NetworkListBox.RowPicture(aIndex)=gPicture(r_done)
    end
    NetworkWindowLock.leave()
  end
End Sub

MainWindow.NE_Remove:
Protected Sub NE_Remove()
  dim i,aServerID as integer
  if NetworkWindowLock.signal() then
    if me.NetworkListBox.listindex<>-1 then
      for i=me.NetworkListBox.listCount-1 downto 0
        if me.NetworkListBox.selected(i) then
          aServerID=val(me.NetworkListBox.cell(i,NE_ID))
          if aServerID<>0 then
            OS_Disconnect(aServerID)
          end
          me.NetworkListBox.removeRow(i)
        end
      next
    end
    NetworkWindowLock.release()
  end
End Sub

MainWindow.NE_Replace:
Sub NE_Replace(inIndex as integer,inName as string,inIP as string,inPort as string,inIsRedirect as boolean,inURL as string,inNick as string,inPassword as string)
  dim aName,aIP as string
  if strcomp(inName,"N/A",0)=0 then
    aName=lowercase(inName)
  else
    aName=inName
  end
  aIP=lowercase(inIP)
  me.NetworkListBox.cell(inIndex,NE_Network)=aName
  me.NetworkListBox.cell(inIndex,NE_IP)=aIP
  me.NetworkListBox.cell(inIndex,NE_Port)=inPort
  me.NetworkListBox.cellCheck(inIndex,NE_Redirect)=inIsRedirect
  me.NetworkListBox.cell(inIndex,NE_URL)=inURL
  me.NetworkListBox.cell(inIndex,NE_Nick)=inNick
  me.NetworkListBox.cell(inIndex,NE_Password)=inPassword
End Sub

MainWindow.NE_SearchID2ListIndex:
Function NE_SearchID2ListIndex(inID as integer) As Integer
  dim i,j as integer
  j=me.NetworkListBox.listCount-1
  for i=0 to j
    if val(me.NetworkListBox.cell(i,NE_ID))=inID then
      return i
    end
  next
  return c_error
End Function

MainWindow.NE_SearchIPAndPort2ListIndex:
Function NE_SearchIPAndPort2ListIndex(inIP as string,inPort as string) As Integer
  dim i,j as integer
  j=me.NetworkListBox.listCount-1
  for i=0 to j
    if strcomp(me.NetworkListBox.cell(i,NE_IP),inIP,0)=0 and strcomp(me.NetworkListBox.cell(i,NE_Port),inPort,0)=0 then
      return i
    end
  next
  return c_error
End Function

MainWindow.NE_SearchNameAndOnlineStatus2ListIndex:
Function NE_SearchNameAndOnlineStatus2ListIndex(inName as string) As Integer
  dim i,j as integer
  if strcomp(inName,"n/a",0)<>0 then
    j=me.NetworkListBox.listCount-1
    for i=0 to j
      if lenb(me.NetworkListBox.cell(i,NE_Icon))=4 and strcomp(me.NetworkListBox.cell(i,NE_Network),inName,1)=0 then
        return i
      end
    next
  end
  return c_error
End Function

MainWindow.MA_ToggleToolbar:
Sub MA_ToggleToolbar()
  if me.NetworkListBox.top=56 then
    me.height=me.height-57
    
    me.NetworkListBox.top=-1
    me.NetworkListBox.height=me.NetworkListBox.height+57
    me.Toolbars(0).visible=false
    me.Toolbars(1).visible=false
    me.Toolbars(2).visible=false
    me.Toolbars(3).visible=false
    me.Toolbars(4).visible=false
    me.StaticTexts(0).visible=false
    me.EditFields(0).visible=false
    me.RoundedCanvases(0).visible=false
    
    me.ConsoleEditField.top=-1
    me.ConsoleEditField.height=me.ConsoleEditField.height+57
    me.ConsolePopupMenu.visible=false
    me.Toolbars(10).visible=false
    me.Toolbars(11).visible=false
    me.StaticTexts(1).visible=false
    me.StaticTexts(2).visible=false
    me.EditFields(1).visible=false
    me.RoundedCanvases(1).visible=false
    
    me.ChannelListBox.top=-1
    me.ChannelListBox.height=me.ChannelListBox.height+57
    me.Toolbars(20).visible=false
    me.Toolbars(21).visible=false
    me.Toolbars(22).visible=false
    me.ChannelPopupMenu.visible=false
    me.StaticTexts(3).visible=false
    me.StaticTexts(4).visible=false
    me.EditFields(2).visible=false
    me.RoundedCanvases(2).visible=false
    
    me.WhoisListBox.top=-1
    me.WhoisListBox.height=me.WhoisListBox.height+57
    me.EditFields(3).visible=false
    me.RoundedCanvases(3).visible=false
    me.Toolbars(30).visible=false
    me.Toolbars(31).visible=false
    me.Toolbars(32).visible=false
    me.Toolbars(33).visible=false
    me.Toolbars(34).visible=false
    me.StaticTexts(5).visible=false
    
    me.UserListListBox.top=-1
    me.UserListListBox.height=me.UserListListBox.height+57
    me.Toolbars(40).visible=false
    me.Toolbars(41).visible=false
    me.Toolbars(42).visible=false
    me.Toolbars(43).visible=false
    me.StaticTexts(6).visible=false
    me.EditFields(4).visible=false
    me.RoundedCanvases(4).visible=false
    
    me.TransferListBoxes(0).top=-1
    me.TransferListBoxes(0).height=me.TransferListBoxes(0).height+57
    me.Toolbars(50).visible=false
    me.Toolbars(51).visible=false
    me.Toolbars(52).visible=false
    me.Toolbars(53).visible=false
    me.Toolbars(54).visible=false
    me.Toolbars(55).visible=false
    me.StaticTexts(7).visible=false
    me.EditFields(5).visible=false
    me.RoundedCanvases(5).visible=false
    
    me.TransferListBoxes(1).top=-1
    me.TransferListBoxes(1).height=me.TransferListBoxes(1).height+57
    me.Toolbars(60).visible=false
    me.Toolbars(61).visible=false
    me.Toolbars(62).visible=false
    me.Toolbars(63).visible=false
    me.Toolbars(64).visible=false
    me.Toolbars(65).visible=false
    me.StaticTexts(8).visible=false
    me.EditFields(6).visible=false
    me.RoundedCanvases(6).visible=false
    
    me.SearchListBox.top=-1
    me.SearchListBox.height=me.SearchListBox.height+57
    me.EditFields(7).visible=false
    me.RoundedCanvases(7).visible=false
    me.SearchPopupMenus(0).visible=false
    select case me.SearchPopupMenus(0).listindex
    case 0'media
      me.SearchPopupMenus(1).visible=false
    case 1'speed
      me.SearchSliders(0).visible=false
    case 2'size
      me.SearchSliders(1).visible=false
    case 3'bitrate
      me.SearchSliders(2).visible=false
    case 4'network
      me.SearchPopupMenus(2).visible=false
    case 5'keyword
      me.SearchEditField.visible=false
    end
    me.Toolbars(70).visible=false
    me.Toolbars(71).visible=false
    me.Toolbars(72).visible=false
    me.StaticTexts(9).visible=false
    me.StaticTexts(10).visible=false
    
    me.FolderListBox.top=-1
    me.FolderListBox.height=me.FolderListBox.height+57
    me.BrowseListBox.top=-1
    me.BrowseListBox.height=me.BrowseListBox.height+57
    me.SplitCanvases(0).top=-1
    me.SplitCanvases(0).height=me.SplitCanvases(0).height+57
    me.EditFields(8).visible=false
    me.RoundedCanvases(8).visible=false
    me.BrowsePopupMenus(0).visible=false
    select case me.BrowsePopupMenus(0).listindex
    case 0
      me.BrowsePopupMenus(1).visible=false
    case 1
      me.BrowseSliders(0).visible=false
    case 2
      me.BrowseSliders(1).visible=false
    case 4
      me.BrowseEditField.visible=false
    end
    me.Toolbars(80).visible=false
    me.Toolbars(81).visible=false
    me.StaticTexts(11).visible=false
    me.StaticTexts(12).visible=false
    
    me.MessageEditField.top=-1
    me.MessageEditField.height=me.MessageEditField.height+57
    me.Toolbars(90).visible=false
    me.Toolbars(91).visible=false
    me.EditFields(9).visible=false
    me.RoundedCanvases(9).visible=false
    me.StaticTexts(13).visible=false
    
    me.RoomEditField.top=-1
    me.RoomEditField.height=me.RoomEditField.height+57
    me.RoomListBox.top=-1
    me.RoomListBox.height=me.RoomListBox.height+57
    me.SplitCanvases(1).top=-1
    me.SplitCanvases(1).height=me.SplitCanvases(1).height+57
    me.Toolbars(100).visible=false
    me.Toolbars(101).visible=false
    me.Toolbars(102).visible=false
    me.Toolbars(103).visible=false
    me.EditFields(10).visible=false
    me.RoundedCanvases(10).visible=false
    me.StaticTexts(14).visible=false
    
  else
    me.height=me.height+57
    
    me.NetworkListBox.top=56
    me.NetworkListBox.height=me.NetworkListBox.height-57
    me.Toolbars(0).visible=true
    me.Toolbars(1).visible=true
    me.Toolbars(2).visible=true
    me.Toolbars(3).visible=true
    me.Toolbars(4).visible=true
    me.StaticTexts(0).visible=true
    me.EditFields(0).visible=true
    me.RoundedCanvases(0).visible=true
    
    me.ConsoleEditField.top=56
    me.ConsoleEditField.height=me.ConsoleEditField.height-57
    me.ConsolePopupMenu.visible=true
    me.Toolbars(10).visible=true
    me.Toolbars(11).visible=true
    me.StaticTexts(1).visible=true
    me.StaticTexts(2).visible=true
    me.EditFields(1).visible=true
    me.RoundedCanvases(1).visible=true
    
    me.ChannelListBox.top=56
    me.ChannelListBox.height=me.ChannelListBox.height-57
    me.Toolbars(20).visible=true
    me.Toolbars(21).visible=true
    me.Toolbars(22).visible=true
    me.ChannelPopupMenu.visible=true
    me.StaticTexts(3).visible=true
    me.StaticTexts(4).visible=true
    me.EditFields(2).visible=true
    me.RoundedCanvases(2).visible=true
    
    me.WhoisListBox.top=56
    me.WhoisListBox.height=me.WhoisListBox.height-57
    me.EditFields(3).visible=true
    me.RoundedCanvases(3).visible=true
    me.Toolbars(30).visible=true
    me.Toolbars(31).visible=true
    me.Toolbars(32).visible=true
    me.Toolbars(33).visible=true
    me.Toolbars(34).visible=true
    me.StaticTexts(5).visible=true
    
    me.UserListListBox.top=56
    me.UserListListBox.height=me.UserListListBox.height-57
    me.Toolbars(40).visible=true
    me.Toolbars(41).visible=true
    me.Toolbars(42).visible=true
    me.Toolbars(43).visible=true
    me.StaticTexts(6).visible=true
    me.EditFields(4).visible=true
    me.RoundedCanvases(4).visible=true
    
    me.TransferListBoxes(0).top=56
    me.TransferListBoxes(0).height=me.TransferListBoxes(0).height-57
    me.Toolbars(50).visible=true
    me.Toolbars(51).visible=true
    me.Toolbars(52).visible=true
    me.Toolbars(53).visible=true
    me.Toolbars(54).visible=true
    me.Toolbars(55).visible=true
    me.StaticTexts(7).visible=true
    me.EditFields(5).visible=true
    me.RoundedCanvases(5).visible=true
    
    me.TransferListBoxes(1).top=56
    me.TransferListBoxes(1).height=me.TransferListBoxes(1).height-57
    me.Toolbars(60).visible=true
    me.Toolbars(61).visible=true
    me.Toolbars(62).visible=true
    me.Toolbars(63).visible=true
    me.Toolbars(64).visible=true
    me.Toolbars(65).visible=true
    me.StaticTexts(8).visible=true
    me.EditFields(6).visible=true
    me.RoundedCanvases(6).visible=true
    
    me.SearchListBox.top=56
    me.SearchListBox.height=me.SearchListBox.height-57
    me.EditFields(7).visible=true
    me.RoundedCanvases(7).visible=true
    me.SearchPopupMenus(0).visible=true
    select case me.SearchPopupMenus(0).listindex
    case 0'media
      me.SearchPopupMenus(1).visible=true
    case 1'speed
      me.SearchSliders(0).visible=true
    case 2'size
      me.SearchSliders(1).visible=true
    case 3'bitrate
      me.SearchSliders(2).visible=true
    case 4'network
      me.SearchPopupMenus(2).visible=true
    case 5'keyword
      me.SearchEditField.visible=true
    end
    me.Toolbars(70).visible=true
    me.Toolbars(71).visible=true
    me.Toolbars(72).visible=true
    me.StaticTexts(9).visible=true
    me.StaticTexts(10).visible=true
    
    me.FolderListBox.top=56
    me.FolderListBox.height=me.FolderListBox.height-57
    me.BrowseListBox.top=56
    me.BrowseListBox.height=me.BrowseListBox.height-57
    me.SplitCanvases(0).top=56
    me.SplitCanvases(0).height=me.SplitCanvases(0).height-57
    me.EditFields(8).visible=true
    me.RoundedCanvases(8).visible=true
    me.BrowsePopupMenus(0).visible=true
    select case me.BrowsePopupMenus(0).listindex
    case 0
      me.BrowsePopupMenus(1).visible=true
    case 1
      me.BrowseSliders(0).visible=true
    case 2
      me.BrowseSliders(1).visible=true
    case 4
      me.BrowseEditField.visible=true
    end
    me.Toolbars(80).visible=true
    me.Toolbars(81).visible=true
    me.StaticTexts(11).visible=true
    me.StaticTexts(12).visible=true
    
    me.MessageEditField.top=56
    me.MessageEditField.height=me.MessageEditField.height-57
    me.Toolbars(90).visible=true
    me.Toolbars(91).visible=true
    me.EditFields(9).visible=true
    me.RoundedCanvases(9).visible=true
    me.StaticTexts(13).visible=true
    
    me.RoomEditField.top=56
    me.RoomEditField.height=me.RoomEditField.height-57
    me.RoomListBox.top=56
    me.RoomListBox.height=me.RoomListBox.height-57
    me.SplitCanvases(1).top=56
    me.SplitCanvases(1).height=me.SplitCanvases(1).height-57
    me.Toolbars(100).visible=true
    me.Toolbars(101).visible=true
    me.Toolbars(102).visible=true
    me.Toolbars(103).visible=true
    me.EditFields(10).visible=true
    me.RoundedCanvases(10).visible=true
    me.StaticTexts(14).visible=true
  end
End Sub

MainWindow.NE_WriteItem:
Protected Sub NE_WriteItem()
  dim tos as TextOutputStream
  dim i,j as integer
  if NetworkWindowLock.enter() then
    tos=PreferencesFolder.Child(c_2get).Child(c_hosts).CreateTextFile()
    j=me.NetworkListBox.listCount-1
    for i=0 to j
      tos.writeline(me.NetworkListBox.cell(i,NE_Network)+chrb(9)+me.NetworkListBox.cell(i,NE_IP)+chrb(9)+me.NetworkListBox.cell(i,NE_Port)+chrb(9)+getBoolean(me.NetworkListBox.cellCheck(i,NE_Redirect))+chrb(9)+me.NetworkListBox.cell(i,NE_Nick)+chrb(9)+me.NetworkListBox.cell(i,NE_Password)+chrb(9)+me.NetworkListBox.cell(i,NE_URL))
    next
    tos.close()
    NetworkWindowLock.leave()
  end
exception
  if tos<>nil then
    tos.close()
  end
  NetworkWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.CO_Offline:
Sub CO_Offline(inServerID as integer)
  dim aIndex as integer
  if ConsoleWindowLock.signal() then
    aIndex=me.ConsolePopupMenu.searchRowTag2ListIndex(inServerID)
    if aIndex<>c_error then
      if aIndex=me.ConsolePopupMenu.listindex then
        me.ConsolePopupMenu.listindex=0
      end
      me.ConsolePopupMenu.removeRow(aIndex)
    end
    ConsoleWindowLock.release()
  end
End Sub

MainWindow.CO_Online:
Sub CO_Online(inServerID as integer,inNetwork as string)
  if ConsoleWindowLock.enter() then
    me.ConsolePopupMenu.addrow(inNetwork)
    me.ConsolePopupMenu.RowTag(me.ConsolePopupMenu.listCount-1)=inServerID
    ConsoleWindowLock.leave()
  end
End Sub

MainWindow.CO_Operator:
Protected Sub CO_Operator(inType as integer)
  if gConsoleServerID<>0 then
    select case inType
    case 0
      ME_NewMessage(gConsoleServerID,getDefine("OperServ"))
    case 1
      ME_NewMessage(gConsoleServerID,getDefine("NickServ"))
    case 2
      ME_NewMessage(gConsoleServerID,getDefine("ChanServ"))
    end
  end
End Sub

MainWindow.CO_Refresh:
Protected Sub CO_Refresh()
  if gConsoleServerID<>0 then
    'message of the day
    OS_Command(gConsoleServerID,621,"")
  end
End Sub

MainWindow.CH_Add:
Protected Sub CH_Add()
  dim aIndex,aServerID as integer
  if ChannelWindowLock.enter() then
    if me.ChannelListBox.listindex<>-1 then
      aServerID=val(me.ChannelListBox.cell(me.ChannelListBox.listindex,CH_ServerID))
    elseif me.ChannelPopupMenu.listIndex<>0 then
      aServerID=me.ChannelPopupMenu.rowTag(me.ChannelPopupMenu.listIndex)
    end
    AddChannelWindow.hide()
    if aServerID<>0 then
      aIndex=AddChannelWindow.PopupMenus.searchRowTag2ListIndex(aServerID)
      if aIndex<>c_error then
        AddChannelWindow.PopupMenus.listindex=aIndex
      end
    end
    AddChannelWindow.showWithin(me)
    ChannelWindowLock.leave()
  end
End Sub

MainWindow.CH_Join:
Protected Sub CH_Join()
  dim aIndex as integer
  if ChannelWindowLock.enter() then
    aIndex=me.ChannelListBox.listindex
    if aIndex<>-1 then
      'join channel
      OS_Command(val(me.ChannelListBox.cell(aIndex,CH_ServerID)),400,me.ChannelListBox.cell(aIndex,CH_Channel))
    end
    ChannelWindowLock.leave()
  end
End Sub

MainWindow.CH_Offline:
Sub CH_Offline(inServerID as integer)
  dim i,aIndex as integer
  if ChannelWindowLock.signal() then
    for i=me.ChannelListBox.listCount-1 downto 0
      if val(me.ChannelListBox.cell(i,CH_ServerID))=inServerID then
        me.ChannelListBox.removeRow(i)
      end
    next
    aIndex=me.ChannelPopupMenu.searchRowTag2ListIndex(inServerID)
    if aIndex<>c_error then
      if me.ChannelPopupMenu.listIndex=aIndex then
        me.ChannelPopupMenu.listIndex=0
      end
      me.ChannelPopupMenu.removeRow(aIndex)
    end
    ChannelWindowLock.release()
  end
End Sub

MainWindow.CH_Online:
Sub CH_Online(inServerID as integer,inNetwork as string)
  if ChannelWindowLock.enter() then
    me.ChannelPopupMenu.addrow(inNetwork)
    me.ChannelPopupMenu.RowTag(me.ChannelPopupMenu.listCount-1)=inServerID
    ChannelWindowLock.leave()
  end
End Sub

MainWindow.CH_Refresh:
Sub CH_Refresh()
  if ChannelWindowLock.signal() then
    me.ChannelListBox.deleteAllRows()
    gChannelServerID=me.ChannelPopupMenu.rowTag(me.ChannelPopupMenu.listIndex)
    if gChannelServerID=0 then
      'list channel
      OS_Broadcast(617,"")
    else
      'list channel
      OS_Command(gChannelServerID,617,"")
    end
    ChannelWindowLock.release()
  end
End Sub

MainWindow.CH_SearchChannelAndID2ListIndex:
Function CH_SearchChannelAndID2ListIndex(inChannel as string,inID as integer) As Integer
  dim i,j as integer
  j=me.ChannelListBox.listCount-1
  for i=0 to j
    if strcomp(me.ChannelListBox.cell(i,CH_Channel),inChannel,0)=0 and val(me.ChannelListBox.cell(i,CH_ServerID))=inID then
      return i
    end
  next
  return c_error
End Function

MainWindow.WH_Browse:
Protected Sub WH_Browse(inIsDirectBrowse as boolean)
  dim i,j,aServerID as integer
  if WhoisWindowLock.enter() then
    if me.WhoisListBox.listindex<>-1 then
      j=me.WhoisListBox.listCount-1
      for i=0 to j
        if me.WhoisListBox.selected(i) then
          aServerID=val(me.WhoisListBox.cell(i,WH_ServerID))
          if aServerID<>0 then
            BR_NewBrowse(aServerID,me.WhoisListBox.cell(i,WH_Nick),inIsDirectBrowse)
          end
        end
      next
    end
    WhoisWindowLock.leave()
  end
End Sub

MainWindow.WH_Message:
Protected Sub WH_Message()
  dim i,j,aServerID as integer
  if WhoisWindowLock.enter() then
    if me.WhoisListBox.listindex<>-1 then
      j=me.WhoisListBox.listCount-1
      for i=0 to j
        if me.WhoisListBox.selected(i) then
          aServerID=val(me.WhoisListBox.cell(i,WH_ServerID))
          if aServerID<>0 then
            ME_NewMessage(aServerID,me.WhoisListBox.cell(i,WH_Nick))
          end
        end
      next
    end
    WhoisWindowLock.leave()
  end
End Sub

MainWindow.WH_Offline:
Sub WH_Offline(inServerID as integer)
  dim i,j as integer
  if WhoisWindowLock.enter() then
    j=me.WhoisListBox.listCount-1
    for i=0 to j
      if val(me.WhoisListBox.cell(i,WH_ServerID))=inServerID then
        me.WhoisListBox.cell(i,WH_ServerID)=""
        me.WhoisListBox.cell(i,WH_Icon)=""
        me.WhoisListBox.rowPicture(i)=nil
      end
    next
    WhoisWindowLock.leave()
  end
End Sub

MainWindow.WH_Online:
Sub WH_Online(inServerID as integer,inNetwork as string)
  dim i,j as integer
  if WhoisWindowLock.enter() then
    j=me.WhoisListBox.listCount-1
    for i=0 to j
      if strcomp(me.WhoisListBox.cell(i,WH_Network),inNetwork,0)=0 then
        me.WhoisListBox.cell(i,WH_ServerID)=str(inServerID)
      end
    next
    WhoisWindowLock.leave()
  end
End Sub

MainWindow.WH_Refresh:
Protected Sub WH_Refresh()
  dim i,j,aServerID as integer
  if WhoisWindowLock.enter() then
    if me.WhoisListBox.listindex<>-1 then
      j=me.WhoisListBox.listCount-1
      for i=0 to j
        if me.WhoisListBox.selected(i) then
          aServerID=val(me.WhoisListBox.cell(i,WH_ServerID))
          if aServerID<>0 then
            'whois request
            OS_Command(aServerID,603,me.WhoisListBox.cell(i,WH_Nick))
          end
        end
      next
      me.WhoisListBox.listIndex=-1
    end
    WhoisWindowLock.leave()
  end
End Sub

MainWindow.WH_CleanUp:
Protected Sub WH_CleanUp()
  if WhoisWindowLock.signal() then
    me.WhoisListBox.deleteAllRows()
    WhoisWindowLock.release()
  end
End Sub

MainWindow.WH_SearchNickAndActiveStatus2ListIndex:
Function WH_SearchNickAndActiveStatus2ListIndex(inPartnerNick as string) As Integer
  dim i,j as integer
  j=me.WhoisListBox.listCount-1
  for i=0 to j
    if strcomp(me.WhoisListBox.cell(i,WH_Nick),inPartnerNick,0)=0 and val(me.WhoisListBox.cell(i,WH_ServerID))<>0 then
      return i
    end
  next
  return c_error
End Function

MainWindow.WH_SearchNickAndServerID2ListIndex:
Function WH_SearchNickAndServerID2ListIndex(inPartnerNick as string,inServerID as integer) As Integer
  dim i,j as integer
  j=me.WhoisListBox.listCount-1
  for i=0 to j
    if strcomp(me.WhoisListBox.cell(i,WH_Nick),inPartnerNick,0)=0 and val(me.WhoisListBox.cell(i,WH_ServerID))=inServerID then
      return i
    end
  next
  return c_error
End Function

MainWindow.WH_UserList:
Protected Sub WH_UserList(inIsFavorite as boolean)
  dim i,j,aServerID as integer
  if WhoisWindowLock.enter() then
    if me.WhoisListBox.listindex<>-1 then
      j=me.WhoisListBox.listCount-1
      for i=0 to j
        if me.WhoisListBox.selected(i) then
          aServerID=val(me.WhoisListBox.cell(i,WH_ServerID))
          if aServerID<>0 then
            me.US_Append(aServerID,me.WhoisListBox.cell(i,WH_Nick),inIsFavorite)
          end
        end
      next
    end
    WhoisWindowLock.leave()
  end
End Sub

MainWindow.WH_Whereis:
Protected Sub WH_Whereis(inWhat as string)
  'whois request
  OS_Broadcast(603,inWhat)
  me.WhoisListBox.listIndex=-1
  DrawerWindow.MenuListBox.listIndex=3
End Sub

MainWindow.US_Append:
Sub US_Append(inServerID as integer,inPartnerNick as string,inIsFavorite as boolean)
  dim aIndex,aArray as integer
  if OpenNapSocketLock.enter() then
    aArray=OS_SearchID2Array(inServerID)
    if aArray<>c_error then
      
      if UserListWindowLock.enter() then
        aIndex=me.US_SearchNickAndServerID2ListIndex(inPartnerNick,inServerID)
        if aIndex=c_error then
          me.UserListListBox.insertrow(0,"")
          aIndex=me.UserListListBox.lastIndex
          me.UserListListBox.cell(aIndex,US_Nick)=inPartnerNick
          me.UserListListBox.cell(aIndex,US_Network)=OpenNapSocket(aArray).network
          me.UserListListBox.cell(aIndex,US_Comment)=""
          me.UserListListBox.cell(aIndex,US_ServerID)=str(inServerID)
          if inIsFavorite then
            me.UserListListBox.cell(aIndex,US_Icon)=" "
            if OpenNapSocket(aArray).isWinMX then
              'hotlist
              OpenNapSocket(aArray).command(208,inPartnerNick)
            else
              'add user to hotlist
              OpenNapSocket(aArray).command(207,inPartnerNick)
            end
          else
            me.UserListListBox.RowPicture(aIndex)=gPicture(r_ignore)
            'add user to ignore list
            OpenNapSocket(aArray).command(322,inPartnerNick)
          end
        end
        UserListWindowLock.leave()
      end
      
    end
    OpenNapSocketLock.leave()
  end
End Sub

MainWindow.US_Browse:
Protected Sub US_Browse(inIsDirectBrowse as boolean)
  dim i,j,aServerID as integer
  if UserListWindowLock.enter() then
    if me.UserListListBox.listindex<>-1 then
      j=me.UserListListBox.listCount-1
      for i=0 to j
        if me.UserListListBox.selected(i) then
          aServerID=val(me.UserListListBox.cell(i,US_ServerID))
          if aServerID<>0 then
            BR_NewBrowse(aServerID,me.UserListListBox.cell(i,US_Nick),inIsDirectBrowse)
          end
        end
      next
    end
    UserListWindowLock.leave()
  end
End Sub

MainWindow.US_Message:
Protected Sub US_Message()
  dim i,j,aServerID as integer
  if UserListWindowLock.enter() then
    if me.UserListListBox.listindex<>-1 then
      j=me.UserListListBox.listCount-1
      for i=0 to j
        if me.UserListListBox.selected(i) then
          aServerID=val(me.UserListListBox.cell(i,US_ServerID))
          if aServerID<>0 then
            ME_NewMessage(aServerID,me.UserListListBox.cell(i,US_Nick))
          end
        end
      next
    end
    UserListWindowLock.leave()
  end
End Sub

MainWindow.US_Offline:
Sub US_Offline(inServerID as integer)
  dim i,j as integer
  if UserListWindowLock.enter() then
    j=me.UserListListBox.listCount-1
    for i=0 to j
      if val(me.UserListListBox.cell(i,US_ServerID))=inServerID then
        me.UserListListBox.cell(i,US_ServerID)=""
        if lenb(me.UserListListBox.cell(i,US_Icon))=2 then
          me.UserListListBox.cell(i,US_Icon)=" "
          me.UserListListBox.rowPicture(i)=nil
          gHotlist=gHotlist-1
        end
      end
    next
    UserListWindowLock.leave()
  end
End Sub

MainWindow.US_Remove:
Protected Sub US_Remove()
  dim i,aStatus as integer
  if UserListWindowLock.signal() then
    if me.UserListListBox.listindex<>-1 then
      for i=me.UserListListBox.listCount-1 downto 0
        if me.UserListListBox.selected(i) then
          aStatus=lenb(me.UserListListBox.cell(i,US_Icon))
          if aStatus=2 then
            gHotlist=gHotlist-1
            'remove user from hotlist
            OS_Command(val(me.UserListListBox.cell(i,US_ServerID)),303,me.UserListListBox.cell(i,US_Nick))
          elseif aStatus=0 then
            'remove user from ignore list
            OS_Command(val(me.UserListListBox.cell(i,US_ServerID)),323,me.UserListListBox.cell(i,US_Nick))
          end
          me.UserListListBox.removeRow(i)
        end
      next
      DrawerWindow.MenuListBox.cellRefresh(4,DR_Count)
    end
    UserListWindowLock.release()
  end
End Sub

MainWindow.US_SearchNickAndServerID2ListIndex:
Function US_SearchNickAndServerID2ListIndex(inPartnerNick as string,inServerID as integer) As Integer
  dim i,j as integer
  j=me.UserListListBox.listCount-1
  for i=0 to j
    if strcomp(me.UserListListBox.cell(i,US_Nick),inPartnerNick,0)=0 and val(me.UserListListBox.cell(i,US_ServerID))=inServerID then
      return i
    end
  next
  return c_error
End Function

MainWindow.US_Whois:
Protected Sub US_Whois()
  dim i,j,aServerID as integer
  if UserListWindowLock.enter() then
    if me.UserListListBox.listindex<>-1 then
      j=me.UserListListBox.listCount-1
      for i=0 to j
        if me.UserListListBox.selected(i) then
          aServerID=val(me.UserListListBox.cell(i,US_ServerID))
          if aServerID<>0 then
            'whois request
            OS_Command(aServerID,603,me.UserListListBox.cell(i,US_Nick))
          else
            'whois request
            OS_Broadcast(603,me.UserListListBox.cell(i,US_Nick))
          end
        end
      next
      me.WhoisListBox.listIndex=-1
      DrawerWindow.MenuListBox.ListIndex=3
    end
    UserListWindowLock.leave()
  end
End Sub

MainWindow.US_WriteItem:
Protected Sub US_WriteItem()
  dim tos as TextOutputStream
  dim i,j as integer
  if UserListWindowLock.enter() then
    tos=PreferencesFolder.child(c_2get).Child(c_users).CreateTextFile()
    j=me.UserListListBox.listCount-1
    for i=0 to j
      tos.writeline(me.UserListListBox.cell(i,US_Nick)+chrb(9)+me.UserListListBox.cell(i,US_Network)+chrb(9)+me.UserListListBox.cell(i,US_Comment)+chrb(9)+getBoolean(lenb(me.UserListListBox.cell(i,US_Icon))<>0))
    next
    tos.close()
    UserListWindowLock.leave()
  end
exception
  if tos<>nil then
    tos.close()
  end
  UserListWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.TR_AppendDownloadQueue:
Function TR_AppendDownloadQueue(inIcon as string,inFileName as String,inShortName as String,inFolderName as string,inSize as String,inShortSize as String,inPartnerNick as String,inServerID as String,inNetwork as string) As Integer
  dim aIndex as integer
  aIndex=me.TR_SearchUDAndFileName2ListIndex(0,inFileName)
  if aIndex=c_error then
    me.TransferListBoxes(0).addrow(inIcon)
    aIndex=me.TransferListBoxes(0).lastIndex
    if val(inServerID)=0 then
      me.TransferListBoxes(0).cell(aIndex,TR_Status)=s_offline
      me.TransferListBoxes(0).cell(aIndex,TR_Network)=s_offline
    else
      me.TransferListBoxes(0).cell(aIndex,TR_Flag)=str(TF_LocallyQueued)
      me.TransferListBoxes(0).cell(aIndex,TR_Status)=s_locally_queued
      me.TransferListBoxes(0).cell(aIndex,TR_Network)=inNetwork
    end
    me.TransferListBoxes(0).cell(aIndex,TR_ShortName)=inShortName
    me.TransferListBoxes(0).cell(aIndex,TR_Nick)=inPartnerNick
    me.TransferListBoxes(0).cell(aIndex,TR_Size)=inSize
    me.TransferListBoxes(0).cell(aIndex,TR_TransferedSize)=getDefine("0")
    me.TransferListBoxes(0).cell(aIndex,TR_ShortSize)=inShortSize
    me.TransferListBoxes(0).cell(aIndex,TR_Rate)=""
    me.TransferListBoxes(0).cell(aIndex,TR_FileName)=inFileName
    me.TransferListBoxes(0).cell(aIndex,TR_FolderName)=inFolderName
    me.TransferListBoxes(0).cell(aIndex,TR_ServerID)=inServerID
    if keyboard.optionKey then
      me.TransferListBoxes(0).cellCheck(aIndex,TR_Retry)=true
      me.TransferListBoxes(0).rowPicture(aIndex)=gPicture(r_download_badge)
    end
  end
  return aIndex
End Function

MainWindow.TR_AppendUploadQueue:
Function TR_AppendUploadQueue(inFileName as String,inShortName as String,inSize as String,inShortSize as String,inPartnerNick as String,inServerID as string,inNetwork as string) As Integer
  dim aIndex as integer
  aIndex=me.TR_SearchUDAndNickAndFileNameAndServerID2ListIndex(1,inPartnerNick,inFIleName,val(inServerID))
  if aIndex=c_error then
    aIndex=me.TR_SearchUDAndNick2LastIndex(1,inPartnerNick)
    if aIndex=c_error then
      me.TransferListBoxes(1).addrow(getMediaType(rightb(inShortName,inbackStrb(1,inShortName,"."))))
    else
      me.TransferListBoxes(1).insertrow(aIndex+1,getMediaType(rightb(inShortName,inbackStrb(1,inShortName,"."))))
    end
    aIndex=me.TransferListBoxes(1).lastIndex
    me.TransferListBoxes(1).cell(aIndex,TR_ShortName)=inShortName
    me.TransferListBoxes(1).cell(aIndex,TR_Network)=inNetwork
    me.TransferListBoxes(1).cell(aIndex,TR_Nick)=inPartnerNick
    me.TransferListBoxes(1).cell(aIndex,TR_Size)=inSize
    me.TransferListBoxes(1).cell(aIndex,TR_TransferedSize)=getDefine("0")
    me.TransferListBoxes(1).cell(aIndex,TR_ShortSize)=inShortSize
    me.TransferListBoxes(1).cell(aIndex,TR_Rate)=""
    me.TransferListBoxes(1).cell(aIndex,TR_FileName)=inFileName
    me.TransferListBoxes(1).cell(aIndex,TR_ServerID)=inServerID
  end
  return aIndex
End Function

MainWindow.TR_AutoRetry:
Protected Sub TR_AutoRetry()
  dim i,j as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(0).listIndex<>-1 then
      j=me.TransferListBoxes(0).listCount-1
      for i=0 to j
        if me.TransferListBoxes(0).selected(i) and val(me.TransferListBoxes(0).cell(i,TR_Size))<>val(me.TransferListBoxes(0).cell(i,TR_TransferedSize)) then
          if me.TransferListBoxes(0).cellCheck(i,TR_Retry) then
            me.TransferListBoxes(0).cellCheck(i,TR_Retry)=false
            me.TransferListBoxes(0).rowPicture(i)=nil
          else
            me.TransferListBoxes(0).cellCheck(i,TR_Retry)=true
            me.TransferListBoxes(0).rowPicture(i)=gPicture(r_download_badge)
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Browse:
Protected Sub TR_Browse(inIsDirectBrowse as boolean,inArray as integer)
  dim i,j,aServerID as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(inArray).listindex<>-1 then
      j=me.TransferListBoxes(inArray).listCount-1
      for i=0 to j
        if me.TransferListBoxes(inArray).selected(i) then
          aServerID=val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))
          if aServerID<>0 then
            BR_NewBrowse(aServerID,me.TransferListBoxes(inArray).cell(i,TR_Nick),inIsDirectBrowse)
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_CountDownloadQueue:
Function TR_CountDownloadQueue() As Integer
  dim i,j,k,aFlag as integer
  j=me.TransferListBoxes(0).listCount-1
  for i=0 to j
    aFlag=val(me.TransferListBoxes(0).cell(i,TR_Flag))
    if val(me.TransferListBoxes(0).cell(i,TR_ID))<>0 or aFlag=TF_NegociatingWithSever or aFlag=TF_Queueble or aFlag=TF_QueuedAt or aFlag=TF_RetryingIn or aFlag=TF_Paused then
      k=k+1
    end
  next
  return k
End Function

MainWindow.TR_CountNick2DownloadQueue:
Function TR_CountNick2DownloadQueue(inPartnerNick as string) As Integer
  dim i,j,k,aFlag as integer
  j=me.TransferListBoxes(0).listCount-1
  for i=0 to j
    if strcomp(me.TransferListBoxes(0).cell(i,TR_Nick),inPartnerNick,0)=0 then
      aFlag=val(me.TransferListBoxes(0).cell(i,TR_Flag))
      if val(me.TransferListBoxes(0).cell(i,TR_ID))<>0 or aFlag=TF_NegociatingWithSever or aFlag=TF_Queueble or aFlag=TF_QueuedAt or aFlag=TF_RetryingIn or aFlag=TF_Paused then
        k=k+1
      end
    end
  next
  return k
End Function

MainWindow.TR_CountNick2UploadQueue:
Function TR_CountNick2UploadQueue(inPartnerNick as string) As Integer
  dim i,j,k as integer
  j=me.TransferListBoxes(1).listCount-1
  for i=0 to j
    if strcomp(me.TransferListBoxes(1).cell(i,TR_Nick),inPartnerNick,0)=0 and (val(me.TransferListBoxes(1).cell(i,TR_ID))<>0 or val(me.TransferListBoxes(1).cell(i,TR_Flag))=TF_NegociatingWithSever) then
      k=k+1
    end
  next
  return k
End Function

MainWindow.TR_CountServerID2DownloadQueue:
Function TR_CountServerID2DownloadQueue(inServerID as integer) As Integer
  dim i,j,k,aFlag as integer
  j=me.TransferListBoxes(0).listCount-1
  for i=0 to j
    if val(me.TransferListBoxes(0).cell(i,TR_ServerID))=inServerID then
      aFlag=val(me.TransferListBoxes(0).cell(i,TR_Flag))
      if val(me.TransferListBoxes(0).cell(i,TR_ID))<>0 or aFlag=TF_NegociatingWithSever or aFlag=TF_Queueble or aFlag=TF_QueuedAt or aFlag=TF_RetryingIn or aFlag=TF_Paused then
        k=k+1
      end
    end
  next
  return k
End Function

MainWindow.TR_CountUploadQueue:
Function TR_CountUploadQueue() As Integer
  dim i,j,k as integer
  j=me.TransferListBoxes(1).listCount-1
  for i=0 to j
    if (val(me.TransferListBoxes(1).cell(i,TR_ID))<>0 or val(me.TransferListBoxes(1).cell(i,TR_Flag))=TF_NegociatingWithSever) then
      k=k+1
    end
  next
  return k
End Function

MainWindow.TR_Message:
Protected Sub TR_Message(inArray as integer)
  dim i,j,aServerID as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(inArray).listindex<>-1 then
      j=me.TransferListBoxes(inArray).listCount-1
      for i=0 to j
        if me.TransferListBoxes(inArray).selected(i) then
          aServerID=val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))
          if aServerID<>0 then
            ME_NewMessage(aServerID,me.TransferListBoxes(inArray).cell(i,TR_Nick))
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Offline:
Sub TR_Offline(inServerID as integer,inArray as integer)
  dim i,j as integer
  if TransferWindowLock.enter() then
    j=me.TransferListBoxes(inArray).listCount-1
    for i=0 to j
      if val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))=inServerID then
        me.TransferListBoxes(inArray).cell(i,TR_ServerID)=""
        me.TransferListBoxes(inArray).cell(i,TR_Network)=s_offline
        me.TransferListBoxes(inArray).cellRefresh(i,TR_Nick)
        if val(me.TransferListBoxes(inArray).cell(i,TR_ID))=0 then
          if me.TransferListBoxes(inArray).cellCheck(i,TR_Retry) then
            if val(me.TransferListBoxes(inArray).cell(i,TR_Flag))<>TF_RetryingIn then
              gDownloadRetrying=gDownloadRetrying+1
              me.TransferListBoxes(inArray).cell(i,TR_Timer)="300"
              me.TransferListBoxes(inArray).cell(i,TR_Flag)=str(TF_RetryingIn)
              me.TransferListBoxes(inArray).cell(i,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(300))
              me.TransferListBoxes(inArray).cellRefresh(i,TR_ShortName)
            end
          else
            me.TransferListBoxes(inArray).cell(i,TR_Flag)=""
            me.TransferListBoxes(inArray).cell(i,TR_Status)=s_offline
            me.TransferListBoxes(inArray).cellRefresh(i,TR_ShortName)
          end
        end
      end
    next
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Open:
Protected Sub TR_Open(inArray as integer)
  dim i,j,aArray as integer
  dim aType as string
  dim fi as FolderItem
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(inArray).listindex<>-1 then
      j=me.TransferListBoxes(inArray).listCount-1
      for i=0 to j
        if me.TransferListBoxes(inArray).selected(i) then
          if inArray=0 then
            if val(me.TransferListBoxes(inArray).cell(i,TR_Size))=val(me.TransferListBoxes(inArray).cell(i,TR_TransferedSize)) then
              aType=getMediaType(rightb(me.TransferListBoxes(inArray).cell(i,TR_ShortName),inbackStrb(1,me.TransferListBoxes(inArray).cell(i,TR_ShortName),".")))
              if aType=rm_music and iMusicFolder<>nil and iMusicFolder.exists then
                fi=iMusicFolder
              elseif aType=rm_movies and iMoviesFolder<>nil and iMoviesFolder.exists then
                fi=iMoviesFolder
              elseif aType=rm_pictures and iPicturesFolder<>nil and iPicturesFolder.exists then
                fi=iPicturesFolder
              elseif iDownloadFolder<>nil and iDownloadFolder.exists then
                fi=iDownloadFolder
              end
              if fi<>nil and fi.exists then
                if lenb(me.TransferListBoxes(inArray).cell(i,TR_FolderName))<>0 then
                  fi=fi.child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_FolderName))).Child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_ShortName)))
                else
                  fi=fi.Child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_ShortName)))
                end
              end
            else
              fi=iIncompleteFolder.Child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_ShortName)))
            end
          else
            if LibraryLock.enter() then
              if isWinMX then
                aArray=OL_SearchFileName2Array(midb(me.TransferListBoxes(inArray).cell(i,TR_FileName),4))
              else
                aArray=OL_SearchFileName2Array(me.TransferListBoxes(inArray).cell(i,TR_FileName))
              end
              if aArray<>c_error then
                fi=getPath2FolderItem(Library(aArray).fullPath)
              end
              LibraryLock.leave()
            end
          end
          if fi<>nil and fi.exists then
            fi.launch()
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_UploadQueueLimit:
Function TR_UploadQueueLimit(inIndex as integer) As Integer
  dim i,j,k as integer
  j=me.TransferListBoxes(1).listCount-1
  for i=0 to j
    if val(me.TransferListBoxes(1).cell(i,TR_ServerID))<>0 then
      if i=inIndex then
        k=k+1
        return k
      elseif val(me.TransferListBoxes(1).cell(i,TR_Flag))=TF_LocallyQueued then
        k=k+1
      end
    end
  next
  return c_error
End Function

MainWindow.TR_RefreshDownloadQueue:
Sub TR_RefreshDownloadQueue()
  dim i,j,aOpenNapArray,aCount,aServerID,aFlag,locallyQueuedServerID(0) as integer
  dim partnerNick,locallyQueuedNick(0) as string
  if TransferWindowLock.enter() then
    gDownloadQueue=0
    j=me.TransferListBoxes(0).listCount-1
    for i=0 to j
      aFlag=val(me.TransferListBoxes(0).cell(i,TR_Flag))
      aCount=val(me.TransferListBoxes(0).cell(i,TR_Timer))-5
      
      select case aFlag
      case TF_NegociatingWithSever
        if aCount<=0 then
          me.TransferListBoxes(0).cell(i,TR_ServerID)=""
          me.TransferListBoxes(0).cell(i,TR_Network)=s_offline
          me.TransferListBoxes(0).cellRefresh(i,TR_Nick)
          if me.TransferListBoxes(0).cellCheck(i,TR_Retry) then
            gDownloadRetrying=gDownloadRetrying+1
            me.TransferListBoxes(0).cell(i,TR_Timer)="300"
            me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_RetryingIn)
            me.TransferListBoxes(0).cell(i,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(300))
            me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
          else
            me.TransferListBoxes(0).cell(i,TR_Flag)=""
            me.TransferListBoxes(0).cell(i,TR_Status)=s_accept_failed
            me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
          end
        else
          me.TransferListBoxes(0).cell(i,TR_Timer)=str(aCount)
        end
        
      case TF_RetryingIn
        if aCount<=0 then
          aServerID=val(me.TransferListBoxes(0).cell(i,TR_ServerID))
          if aServerID=0 then
            if me.TransferListBoxes(0).cellCheck(i,TR_Retry) then
              me.TransferListBoxes(0).cell(i,TR_Timer)="300"
              me.TransferListBoxes(0).cell(i,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(300))
              me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
              'whois request
              OS_Broadcast(603,me.TransferListBoxes(0).cell(i,TR_Nick))
            else
              gDownloadRetrying=gDownloadRetrying-1
              me.TransferListBoxes(0).cell(i,TR_Flag)=""
              me.TransferListBoxes(0).cell(i,TR_Status)=s_accept_failed
              me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
            end
          else
            
            if OpenNapSocketLock.enter() then
              aOpenNapArray=OS_SearchID2Array(aServerID)
              if aOpenNapArray<>c_error then
                gDownloadRetrying=gDownloadRetrying-1
                me.TransferListBoxes(0).cell(i,TR_Timer)="120"
                me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
                me.TransferListBoxes(0).cell(i,TR_Status)=s_negociating_with_server
                me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
                me.TransferListBoxes(0).cellCheck(i,TR_WinMX)=false
                'download request
                OpenNapSocket(aOpenNapArray).command(203,me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(32)+chrb(34)+me.TransferListBoxes(0).cell(i,TR_FileName)+chrb(34))
              end
              OpenNapSocketLock.leave()
            end
            
          end
        else
          me.TransferListBoxes(0).cell(i,TR_Timer)=str(aCount)
          me.TransferListBoxes(0).cell(i,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(aCount))
          me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
        end
        
      case TF_Queueble
        
        if OpenNapSocketLock.enter() then
          aOpenNapArray=OS_SearchID2Array(val(me.TransferListBoxes(0).cell(i,TR_ServerID)))
          if aOpenNapArray<>c_error then
            me.TransferListBoxes(0).cell(i,TR_Timer)="300"
            me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
            me.TransferListBoxes(0).cell(i,TR_Status)=s_negociating_with_server
            me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
            if me.TransferListBoxes(0).cellCheck(i,TR_WinMX) then
              'private message
              OpenNapSocket(aOpenNapArray).command(205,me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(32)+getDefine("//WantQueue"))
            end
            'download request
            OpenNapSocket(aOpenNapArray).command(203,me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(32)+chrb(34)+me.TransferListBoxes(0).cell(i,TR_FileName)+chrb(34))
          end
          OpenNapSocketLock.leave()
        end
        
      case TF_QueuedAt
        if aCount<=0 then
          
          if OpenNapSocketLock.enter() then
            aOpenNapArray=OS_SearchID2Array(val(me.TransferListBoxes(0).cell(i,TR_ServerID)))
            if aOpenNapArray<>c_error then
              me.TransferListBoxes(0).cell(i,TR_Timer)="300"
              me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
              me.TransferListBoxes(0).cell(i,TR_Status)=s_negociating_with_server
              me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
              if me.TransferListBoxes(0).cellCheck(i,TR_WinMX) then
                'private message
                OpenNapSocket(aOpenNapArray).command(205,me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(32)+getDefine("//WantQueue"))
              end
              'download request
              OpenNapSocket(aOpenNapArray).command(203,me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(32)+chrb(34)+me.TransferListBoxes(0).cell(i,TR_FileName)+chrb(34))
            end
            OpenNapSocketLock.leave()
          end
          
        else
          gDownloadQueue=gDownloadQueue+1
          me.TransferListBoxes(0).cell(i,TR_Timer)=str(aCount)
        end
        
      case TF_LocallyQueued
        partnerNick=""
        partnerNick=me.TransferListBoxes(0).cell(i,TR_Nick)
        aServerID=val(me.TransferListBoxes(0).cell(i,TR_ServerID))
        if hasNickAndServerID(locallyQueuedNick,locallyQueuedServerID,partnerNick,aServerID)=false then
          if (iDownloadLimits=0 or iDownloadLimits>me.TR_CountDownloadQueue()) and (iDownloadPerUser=0 or iDownloadPerUser>me.TR_CountNick2DownloadQueue(partnerNick)) and (iDownloadPerServer=0 or iDownloadPerServer>me.TR_CountServerID2DownloadQueue(aServerID)) then
            
            if OpenNapSocketLock.enter() then
              aOpenNapArray=OS_SearchID2Array(aServerID)
              if aOpenNapArray<>c_error then
                me.TransferListBoxes(0).cell(i,TR_Timer)="120"
                me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
                me.TransferListBoxes(0).cell(i,TR_Status)=s_negociating_with_server
                me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
                me.TransferListBoxes(0).cellCheck(i,TR_WinMX)=false
                'download request
                OpenNapSocket(aOpenNapArray).command(203,partnerNick+chrb(32)+chrb(34)+me.TransferListBoxes(0).cell(i,TR_FileName)+chrb(34))
              end
              OpenNapSocketLock.leave()
            end
            
          else
            locallyQueuedNick.append partnerNick
            locallyQueuedServerID.append aServerID
          end
        end
        
      end
    next
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Remove:
Protected Sub TR_Remove(inArray as integer)
  dim fi as folderItem
  dim i,aFlag,transferedSize as integer
  dim removeFile as boolean
  if inArray=0 and iPromptRemoving then
    RemovingWindow.showModalWithin(me)
    removeFile=RemovingWindow.removeFile
    RemovingWindow.close()
    if removeFile=false then
      return
    end
  end
  
  if TransferWindowLock.signal() then
    if me.TransferListBoxes(inArray).listindex<>-1 then
      for i=me.TransferListBoxes(inArray).listCount-1 downto 0
        if me.TransferListBoxes(inArray).selected(i) then
          if val(me.TransferListBoxes(inArray).cell(i,TR_ID))<>0 then
            TS_Disconnect(val(me.TransferListBoxes(inArray).cell(i,TR_ID)))
          else
            aFlag=val(me.TransferListBoxes(inArray).cell(i,TR_Flag))
            select case aFlag
            case TF_RetryingIn
              gDownloadRetrying=gDownloadRetrying-1
            case TF_Preparing
              gUploadPreparing=gUploadPreparing-1
            end
          end
          
          if removeFile then
            transferedSize=val(me.TransferListBoxes(inArray).cell(i,TR_TransferedSize))
            if transferedSize<>0 and transferedSize<>val(me.TransferListBoxes(inArray).cell(i,TR_Size))  then
              fi=iIncompleteFolder.Child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_ShortName)))
              if fi<>nil and fi.exists then
                fi.delete()
              end
            end
          end
          
          me.TransferListBoxes(inArray).removeRow(i)
        end
      next
    end
    TransferWindowLock.release()
  end
End Sub

MainWindow.TR_Resume:
Protected Sub TR_Resume()
  dim d as openDialog
  dim fi as folderItem
  dim i,j as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(0).listindex<>-1 then
      j=me.TransferListBoxes(0).listCount-1
      for i=0 to j
        if me.TransferListBoxes(0).selected(i) and val(me.TransferListBoxes(0).cell(i,TR_ID))=0 and val(me.TransferListBoxes(0).cell(i,TR_Size))<>val(me.TransferListBoxes(0).cell(i,TR_TransferedSize)) then
          d=new openDialog
          d.Filter="special/any"
          d.promptText=me.TransferListBoxes(0).cell(i,TR_ShortName)
          d.initialDirectory=iIncompleteFolder
          if d.initialDirectory<>nil and d.initialDirectory.exists then
            fi=d.showModalWithin(me)
            if fi<>nil and fi.exists then
              if fi.parent<>d.initialDirectory then
                moveFolderItem(fi,d.initialDirectory)
              end
              me.TransferListBoxes(0).cell(i,TR_ShortName)=getString(fi.name)
            end
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Reveal:
Protected Sub TR_Reveal(inArray as integer)
  dim i,j,aArray as integer
  dim aType as string
  dim fi as FolderItem
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(inArray).listindex<>-1 then
      j=me.TransferListBoxes(inArray).listCount-1
      for i=0 to j
        if me.TransferListBoxes(inArray).selected(i) then
          if inArray=0 then
            if val(me.TransferListBoxes(inArray).cell(i,TR_Size))=val(me.TransferListBoxes(inArray).cell(i,TR_TransferedSize)) then
              aType=getMediaType(rightb(me.TransferListBoxes(inArray).cell(i,TR_ShortName),inbackStrb(1,me.TransferListBoxes(inArray).cell(i,TR_ShortName),".")))
              if aType=rm_music and iMusicFolder<>nil and iMusicFolder.exists then
                fi=iMusicFolder
              elseif aType=rm_movies and iMoviesFolder<>nil and iMoviesFolder.exists then
                fi=iMoviesFolder
              elseif aType=rm_pictures and iPicturesFolder<>nil and iPicturesFolder.exists then
                fi=iPicturesFolder
              elseif iDownloadFolder<>nil and iDownloadFolder.exists then
                fi=iDownloadFolder
              end
              if fi<>nil and fi.exists then
                if lenb(me.TransferListBoxes(inArray).cell(i,TR_FolderName))<>0 then
                  fi=fi.child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_FolderName))).Child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_ShortName)))
                else
                  fi=fi.Child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_ShortName)))
                end
              end
            else
              fi=iIncompleteFolder.Child(getUTF8String(me.TransferListBoxes(inArray).cell(i,TR_ShortName)))
            end
          else
            if LibraryLock.enter() then
              if isWinMX then
                aArray=OL_SearchFileName2Array(midb(me.TransferListBoxes(inArray).cell(i,TR_FileName),4))
              else
                aArray=OL_SearchFileName2Array(me.TransferListBoxes(inArray).cell(i,TR_FileName))
              end
              if aArray<>c_error then
                fi=getPath2FolderItem(Library(aArray).fullPath)
              end
              LibraryLock.leave()
            end
          end
          if fi<>nil and fi.exists then
            revealInFinder(fi)
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_SearchUDAndID2ListIndex:
Function TR_SearchUDAndID2ListIndex(inArray as integer,inID as Integer) As Integer
  dim i,j as integer
  j=me.TransferListBoxes(inArray).listCount-1
  for i=0 to j
    if val(me.TransferListBoxes(inArray).cell(i,TR_ID))=inID then
      return i
    end
  next
  return c_error
End Function

MainWindow.TR_SearchUDAndNick2LastIndex:
Function TR_SearchUDAndNick2LastIndex(inArray as integer,inPartnerNick as string) As Integer
  dim i as integer
  for i=me.TransferListBoxes(inArray).listCount-1 downto 0
    if strcomp(me.TransferListBoxes(inArray).cell(i,TR_Nick),inPartnerNick,0)=0 then
      return i
    end
  next
  return c_error
End Function

MainWindow.TR_SearchUDAndFileName2ListIndex:
Function TR_SearchUDAndFileName2ListIndex(inArray as integer,inFileName as string) As Integer
  dim i,j as integer
  j=me.TransferListBoxes(inArray).listCount-1
  for i=0 to j
    if strcomp(me.TransferListBoxes(inArray).cell(i,TR_FileName),inFileName,0)=0 then
      return i
    end
  next
  return c_error
End Function

MainWindow.TR_SearchUDAndNickAndFileNameAndServerID2ListIndex:
Function TR_SearchUDAndNickAndFileNameAndServerID2ListIndex(inArray as integer,inPartnerNick as string,inFileName as string,inServerID as integer) As Integer
  dim i,j as integer
  j=me.TransferListBoxes(inArray).listCount-1
  for i=0 to j
    if strcomp(me.TransferListBoxes(inArray).cell(i,TR_Nick),inPartnerNick,0)=0 and strcomp(me.TransferListBoxes(inArray).cell(i,TR_FileName),inFileName,0)=0 and val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))=inServerID then
      return i
    end
  next
  return c_error
End Function

MainWindow.TR_SearchUDAndNickAndFlagAndServerID2ListIndex:
Function TR_SearchUDAndNickAndFlagAndServerID2ListIndex(inArray as integer,inPartnerNick as string,inFlag as integer,inServerID as integer) As Integer
  dim i,j as integer
  j=me.TransferListBoxes(inArray).listCount-1
  for i=0 to j
    if strcomp(me.TransferListBoxes(inArray).cell(i,TR_Nick),inPartnerNick,0)=0 and val(me.TransferListBoxes(inArray).cell(i,TR_Flag))=inFlag and val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))=inServerID then
      return i
    end
  next
  return c_error
End Function

MainWindow.TR_SearchUDAndNickAndServerID2ListIndex:
Function TR_SearchUDAndNickAndServerID2ListIndex(inArray as integer,inPartnerNick as string,inServerID as integer) As Integer
  dim i,j as integer
  j=me.TransferListBoxes(inArray).listCount-1
  for i=0 to j
    if strcomp(me.TransferListBoxes(inArray).cell(i,TR_Nick),inPartnerNick,0)=0 and val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))=inServerID then
      return i
    end
  next
  return c_error
End Function

MainWindow.TR_RetryDownload:
Protected Sub TR_RetryDownload()
  dim i,j,aIndex,aServerID,aFlag as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(0).listindex<>-1 then
      j=me.TransferListBoxes(0).listCount-1
      for i=0 to j
        if me.TransferListBoxes(0).selected(i) and val(me.TransferListBoxes(0).cell(i,TR_ID))=0 then
          aServerID=val(me.TransferListBoxes(0).cell(i,TR_ServerID))
          aFlag=val(me.TransferListBoxes(0).cell(i,TR_Flag))
          if aServerID<>0 then
            select case aFlag
            case TF_LocallyQueued
              me.TransferListBoxes(0).cell(i,TR_Timer)="120"
              me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
              me.TransferListBoxes(0).cell(i,TR_Status)=s_negociating_with_server
              me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
              me.TransferListBoxes(0).cellCheck(i,TR_WinMX)=false
              'download request
              OS_Command(aServerID,203,me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(32)+chrb(34)+me.TransferListBoxes(0).cell(i,TR_FileName)+chrb(34))
            case TF_RetryingIn
              gDownloadRetrying=gDownloadRetrying-1
              me.TransferListBoxes(0).cell(i,TR_Timer)="120"
              me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
              me.TransferListBoxes(0).cell(i,TR_Status)=s_negociating_with_server
              me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
              me.TransferListBoxes(0).cellCheck(i,TR_WinMX)=false
              'download request
              OS_Command(aServerID,203,me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(32)+chrb(34)+me.TransferListBoxes(0).cell(i,TR_FileName)+chrb(34))
            else
              me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_LocallyQueued)
              me.TransferListBoxes(0).cell(i,TR_Status)=s_locallY_queued
              me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
            end
          else
            if aFlag<>TF_RetryingIn then
              gDownloadRetrying=gDownloadRetrying+1
            end
            me.TransferListBoxes(0).cell(i,TR_Timer)="300"
            me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_RetryingIn)
            me.TransferListBoxes(0).cell(i,TR_Status)=replace(s_retrying_in_,"%",getRetryTime(300))
            me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
            'whois request
            OS_Broadcast(603,me.TransferListBoxes(0).cell(i,TR_Nick))
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_UserList:
Protected Sub TR_UserList(inIsFavorite as boolean,inArray as integer)
  dim i,j,aServerID as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(inArray).listindex<>-1 then
      j=me.TransferListBoxes(inArray).listCount-1
      for i=0 to j
        if me.TransferListBoxes(inArray).selected(i) then
          aServerID=val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))
          if aServerID<>0 then
            me.US_Append(aServerID,me.TransferListBoxes(inArray).cell(i,TR_Nick),inIsFavorite)
          end
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Whois:
Protected Sub TR_Whois(inArray as integer)
  dim i,j,aServerID as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(inArray).listindex<>-1 then
      j=me.TransferListBoxes(inArray).listCount-1
      for i=0 to j
        if me.TransferListBoxes(inArray).selected(i) then
          aServerID=val(me.TransferListBoxes(inArray).cell(i,TR_ServerID))
          if aServerID<>0 then
            'whois request
            OS_Command(aServerID,603,me.TransferListBoxes(inArray).cell(i,TR_Nick))
          end
        end
      next
      me.WhoisListBox.listIndex=-1
      DrawerWindow.MenuListBox.ListIndex=3
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_WriteItem:
Protected Sub TR_WriteItem()
  dim tos as TextOutputStream
  dim i,j as integer
  if TransferWindowLock.enter() then
    tos=PreferencesFolder.Child(c_2get).Child(c_resumes).CreateTextFile()
    j=me.TransferListBoxes(0).listCount-1
    for i=0 to j
      if val(me.TransferListBoxes(0).cell(i,TR_Size))<>val(me.TransferListBoxes(0).cell(i,TR_TransferedSize)) then
        tos.writeline(me.TransferListBoxes(0).cell(i,TR_FileName)+chrb(9)+me.TransferListBoxes(0).cell(i,TR_ShortName)+chrb(9)+me.TransferListBoxes(0).cell(i,TR_Size)+chrb(9)+me.TransferListBoxes(0).cell(i,TR_ShortSize)+chrb(9)+me.TransferListBoxes(0).cell(i,TR_TransferedSize)+chrb(9)+me.TransferListBoxes(0).cell(i,TR_Nick)+chrb(9)+me.TransferListBoxes(0).cell(i,TR_FolderName))
      end
    next
    tos.close()
    TransferWindowLock.leave()
  end
exception
  if tos<>nil then
    tos.close()
  end
  TransferWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.CO_Filter:
Protected Sub CO_Filter(inServerID as integer)
  dim i,j,aIndex as integer
  dim temp as string
  if ConsoleWindowLock.signal() then
    
    gConsoleServerID=inServerID
    if gConsoleServerID<>0 then
      j=ubound(ConsoleList)
      for i=1 to j
        if ConsoleList(i).serverID=inServerID then
          temp=temp+ConsoleList(i).message
        end
      next
      me.ConsoleEditField.text=getDefine(temp)
      me.ConsoleEditField.scrollPosition=0
      me.Toolbars(10).enabled=true
      me.Toolbars(11).enabled=true
    else
      j=ubound(ConsoleList)
      for i=1 to j
        temp=temp+ConsoleList(i).message
      next
      me.ConsoleEditField.text=""
      me.ConsoleEditField.appendText(temp)
      me.Toolbars(10).enabled=false
      me.Toolbars(11).enabled=false
    end
    
    ConsoleWindowLock.release()
  end
End Sub

MainWindow.SE_Browse:
Protected Sub SE_Browse(inIsDirectBrowse as boolean)
  dim i,j as integer
  if SearchWindowLock.enter() then
    if me.SearchListBox.listindex<>-1 then
      j=me.SearchListBox.listCount-1
      for i=0 to j
        if me.SearchListBox.selected(i) then
          BR_NewBrowse(val(me.SearchListBox.cell(i,SE_ServerID)),me.SearchListBox.cell(i,SE_Nick),inIsDirectBrowse)
        end
      next
    end
    SearchWindowLock.leave()
  end
End Sub

MainWindow.SE_Download:
Protected Sub SE_Download()
  dim i,j,aIndex as integer
  if TransferWindowLock.enter() then
    
    if SearchWindowLock.enter() then
      if me.SearchListBox.listindex<>-1 then
        j=me.SearchListBox.listCount-1
        for i=0 to j
          if me.SearchListBox.selected(i) then
            aIndex=me.TR_AppendDownloadQueue(me.SearchListBox.cell(i,SE_Icon),me.SearchListBox.cell(i,SE_FileName),me.SearchListBox.cell(i,SE_ShortName),"",me.SearchListBox.cell(i,SE_Size),me.SearchListBox.cell(i,SE_ShortSize),me.SearchListBox.cell(i,SE_Nick),me.SearchListBox.cell(i,SE_ServerID),me.SearchListBox.cell(i,SE_Network))
            me.SearchListBox.RowPicture(i)=gPicture(r_download_badge)
          end
        next
      end
      SearchWindowLock.leave()
    end
    
    me.TransferListBoxes(0).listindex=-1
    me.TransferListBoxes(0).ScrollPosition=aIndex
    TransferWindowLock.leave()
  end
End Sub

MainWindow.SE_Filter:
Protected Sub SE_Filter()
  dim i,j,k,l,aArray,aIndex,isHit,mediaTypeFilter,speedFilter,sizeFilter,bitrateFilter,serverIDFilter as integer
  dim temp,searchString(0) as string
  if SearchWindowLock.signal() then
    if gSearchWindowID<>0 then
      temp=getString(trim(me.SearchEditField.text))
      if lenb(temp)<>0 then
        l=countfields(temp,chrb(32))
        for i=1 to l
          searchString.append nthfield(temp,chrb(32),i)
        next
      end
      select case me.SearchPopupMenus(1).listindex
      case 0
        mediaTypeFilter=0
      case 1
        mediaTypeFilter=val(rm_music)
      case 2
        mediaTypeFilter=val(rm_pictures)
      case 3
        mediaTypeFilter=val(rm_movies)
      case 4
        mediaTypeFilter=val(rm_text)
      case 5
        mediaTypeFilter=val(rm_files)
      case 6
        mediaTypeFilter=val(rm_music)
      end
      speedFilter=me.SearchSliders(0).value
      sizeFilter=me.SearchSliders(1).value*10485760
      bitrateFilter=me.SearchSliders(2).value
      serverIDFilter=me.SearchPopupMenus(2).rowtag(me.SearchPopupMenus(2).listindex)
      
      aArray=SE_SearchID2Array(gSearchWindowID)
      if aArray<>c_error then
        me.SearchListBox.visible=false
        me.SearchListBox.textSize=1
        me.SearchListBox.deleteAllRows()
        j=ubound(SearchWindow(aArray).SearchList)
        for i=1 to j
          isHit=0
          
          if mediaTypeFilter=0 or mediaTypeFilter=val(SearchWindow(aArray).SearchList(i).mediaType) then
            isHit=isHit+1
          end
          
          if speedFilter=0 or speedFilter<=val(SearchWindow(aArray).SearchList(i).speed) then
            isHit=isHit+1
          end
          
          if sizeFilter=0 or sizeFilter<=val(SearchWindow(aArray).SearchList(i).size) then
            isHit=isHit+1
          end
          
          if bitrateFilter=0 or bitrateFilter<=val(SearchWindow(aArray).SearchList(i).bitrate) then
            isHit=isHit+1
          end
          
          if serverIDFilter=0 or serverIDFilter=val(SearchWindow(aArray).SearchList(i).serverID) then
            isHit=isHit+1
          end
          
          for k=1 to l
            if instrb(1,SearchWindow(aArray).SearchList(i).FileName,searchString(k))<>0 then
              isHit=isHit+1
            end
          next
          
          if isHit=5+l then
            me.SearchListBox.addrow("")
            aIndex=me.SearchListBox.lastIndex
            me.SearchListBox.rowPicture(aIndex)=gMediaPicture(val(SearchWindow(aArray).SearchList(i).mediaType))
            me.SearchListBox.cell(aIndex,SE_Icon)=SearchWindow(aArray).SearchList(i).mediaType
            me.SearchListBox.cell(aIndex,SE_FileName)=SearchWindow(aArray).SearchList(i).fileName
            me.SearchListBox.cell(aIndex,SE_ShortName)=SearchWindow(aArray).SearchList(i).shortName
            me.SearchListBox.cell(aIndex,SE_Size)=SearchWindow(aArray).SearchList(i).size
            me.SearchListBox.cell(aIndex,SE_ShortSize)=SearchWindow(aArray).SearchList(i).shortSize
            me.SearchListBox.cell(aIndex,SE_Nick)=SearchWindow(aArray).SearchList(i).nick
            me.SearchListBox.cell(aIndex,SE_Speed)=SearchWindow(aArray).SearchList(i).shortSpeed
            me.SearchListBox.cell(aIndex,SE_BitRate)=SearchWindow(aArray).SearchList(i).bitRate
            me.SearchListBox.cell(aIndex,SE_ServerID)=SearchWindow(aArray).SearchList(i).serverID
            me.SearchListBox.cell(aIndex,SE_Network)=SearchWindow(aArray).SearchList(i).network
          end
          
        next
        me.SearchListBox.scrollposition=0
        me.SearchListBox.headingIndex=-1
        me.SearchListBox.textSize=10
        me.SearchListBox.visible=true
        DrawerWindow.DR_UpdateResults(gSearchWindowID,SearchWindow(aArray).title,me.SearchListBox.listCount,ubound(SearchWindow(aArray).SearchList))
      end
    else
      me.SearchListBox.deleteAllRows()
    end
    SearchWindowLock.release()
  end
End Sub

MainWindow.SE_Message:
Protected Sub SE_Message()
  dim i,j as integer
  if SearchWindowLock.enter() then
    if me.SearchListBox.listindex<>-1 then
      j=me.SearchListBox.listCount-1
      for i=0 to j
        if me.SearchListBox.selected(i) then
          ME_NewMessage(val(me.SearchListBox.cell(i,SE_ServerID)),me.SearchListBox.cell(i,SE_Nick))
        end
      next
    end
    SearchWindowLock.leave()
  end
End Sub

MainWindow.SE_Resume:
Protected Sub SE_Resume()
  dim d as openDialog
  dim fi as folderItem
  dim i,j,aIndex as integer
  if TransferWindowLock.enter() then
    
    if SearchWindowLock.enter() then
      if me.SearchListBox.listindex<>-1 then
        j=me.SearchListBox.listCount-1
        for i=0 to j
          if me.SearchListBox.selected(i) then
            d=new openDialog
            d.Filter="special/any"
            d.promptText=me.SearchListBox.cell(i,SE_ShortName)
            d.initialDirectory=iIncompleteFolder
            if d.initialDirectory<>nil and d.initialDirectory.exists then
              fi=d.showModalWithin(me)
              if fi<>nil and fi.exists then
                if fi.parent<>d.initialDirectory then
                  moveFolderItem(fi,d.initialDirectory)
                end
                aIndex=me.TR_AppendDownloadQueue(me.SearchListBox.cell(i,SE_Icon),me.SearchListBox.cell(i,SE_FileName),getString(fi.name),"",me.SearchListBox.cell(i,SE_Size),me.SearchListBox.cell(i,SE_ShortSize),me.SearchListBox.cell(i,SE_Nick),me.SearchListBox.cell(i,SE_ServerID),me.SearchListBox.cell(i,SE_Network))
                me.SearchListBox.RowPicture(i)=gPicture(r_download_badge)
              end
            end
          end
        next
      end
      SearchWindowLock.leave()
    end
    
    me.TransferListBoxes(0).listindex=-1
    me.TransferListBoxes(0).ScrollPosition=aIndex
    TransferWindowLock.leave()
  end
End Sub

MainWindow.SE_Search:
Sub SE_Search(inSearchString as string,inSearchWindowID as integer)
  dim temp as string
  dim aArray as integer
  if SearchWindowLock.enter() then
    aArray=SE_SearchID2Array(inSearchWindowID)
    if aArray<>c_error then
      
      SearchWindow(aArray).title=inSearchString
      if gSearchWindowID=inSearchWindowID then
        self.setTitle(inSearchString)
      end
      if isWinMX=false then
        temp=getDefine("FILENAME CONTAINS ")+chrb(34)+inSearchString+chrb(34)
        select case me.SearchPopupMenus(1).listindex
        case 0
          temp=temp+getDefine(" TYPE any")
        case 1
          temp=temp+getDefine(" TYPE audio")
        case 2
          temp=temp+getDefine(" TYPE image")
        case 3
          temp=temp+getDefine(" TYPE video")
        case 4
          temp=temp+getDefine(" TYPE text")
        case 5
          temp=temp+getDefine(" TYPE application")
        end
        if me.SearchSliders(0).value<>0 then
          temp=temp+getDefine(" SPEED "+chrb(34)+"AT LEAST"+chrb(34)+chrb(32)+str(me.SearchSliders(0).value))
        end
        if me.SearchSliders(1).value<>0 then
          temp=temp+getDefine(" SIZE "+chrb(34)+"AT LEAST"+chrb(34)+chrb(32)+format(me.SearchSliders(1).value*10485760,"#"))
        end
        if me.SearchSliders(2).value<>0 then
          temp=temp+getDefine(" BITRATE "+chrb(34)+"AT LEAST"+chrb(34)+chrb(32)+str(me.SearchSliders(2).value))
        end
        
      else
        temp=getDefine("FILENAME CONTAINS ")+chrb(34)+inSearchString+chrb(34)
        if me.SearchPopupMenus(1).listindex<>6 then
          temp=temp+getDefine(" BITRATE "+chrb(34)+"EQUAL TO"+chrb(34)+chrb(32)+chrb(34)+"24"+chrb(34)+" FREQ "+chrb(34)+"EQUAL TO"+chrb(34)+chrb(32)+chrb(34)+"16000"+chrb(34))
        elseif me.SearchSliders(2).value<>0 then
          temp=temp+getDefine(" BITRATE "+chrb(34)+"AT LEAST"+chrb(34)+chrb(32)+str(me.SearchSliders(2).value))
        end
        if me.SearchSliders(0).value<>0 then
          temp=temp+getDefine(" SPEED "+chrb(34)+"AT LEAST"+chrb(34)+chrb(32)+str(me.SearchSliders(0).value))
        end
        
      end
      
      if me.SearchPopupMenus(2).listindex=0 then
        OS_BroadcastSearch(inSearchWindowID,temp)
      else
        OS_Search(me.SearchPopupMenus(2).rowTag(me.SearchPopupMenus(2).listindex),inSearchWindowID,temp)
      end
      
    end
    SearchWindowLock.leave()
  end
End Sub

MainWindow.SE_SwitchView:
Protected Sub SE_SwitchView()
  if me.SearchListBox.column(SE_FileName).widthExpression="0" then
    me.SearchListBox.column(SE_FileName).widthExpression=me.SearchListBox.column(SE_ShortName).widthExpression
    me.SearchListBox.column(SE_ShortName).widthExpression="0"
  else
    me.SearchListBox.column(SE_ShortName).widthExpression=me.SearchListBox.column(SE_FileName).widthExpression
    me.SearchListBox.column(SE_FileName).widthExpression="0"
  end
  me.SearchListBox.columnWidths=me.SearchListBox.columnWidths
End Sub

MainWindow.SE_Whois:
Protected Sub SE_Whois()
  dim i,j as integer
  if SearchWindowLock.enter() then
    if me.SearchListBox.listindex<>-1 then
      j=me.SearchListBox.listCount-1
      for i=0 to j
        if me.SearchListBox.selected(i) then
          'whois request
          OS_Command(val(me.SearchListBox.cell(i,SE_ServerID)),603,me.SearchListBox.cell(i,SE_Nick))
        end
      next
      me.WhoisListBox.listIndex=-1
      DrawerWindow.MenuListBox.listIndex=3
    end
    SearchWindowLock.leave()
  end
End Sub

MainWindow.SE_Offline:
Sub SE_Offline(inServerID as integer)
  dim aIndex as integer
  if SearchWindowLock.signal() then
    aIndex=me.SearchPopupMenus(2).searchRowTag2ListIndex(inServerID)
    if aIndex<>c_error then
      if aIndex=me.SearchPopupMenus(2).listindex then
        me.SearchPopupMenus(2).listindex=0
      end
      me.SearchPopupMenus(2).removeRow(aIndex)
    end
    SearchWindowLock.release()
  end
End Sub

MainWindow.SE_Online:
Sub SE_Online(inServerID as integer,inNetwork as string)
  if SearchWindowLock.enter() then
    me.SearchPopupMenus(2).addrow(inNetwork)
    me.SearchPopupMenus(2).rowtag(me.SearchPopupMenus(2).listCount-1)=inServerID
    SearchWindowLock.leave()
  end
End Sub

MainWindow.BR_Download:
Protected Sub BR_Download()
  dim i,j,aIndex,aArray as integer
  if TransferWindowLock.enter() then
    
    if BrowseWindowLock.enter() then
      aArray=BR_SearchID2Array(gBrowseWindowID)
      if aArray<>c_error then
        if me.BrowseListBox.listindex<>-1 then
          j=me.BrowseListBox.listCount-1
          for i=0 to j
            if me.BrowseListBox.selected(i) then
              aIndex=me.TR_AppendDownloadQueue(me.BrowseListBox.cell(i,BR_Icon),me.BrowseListBox.cell(i,BR_FileName),me.BrowseListBox.cell(i,BR_ShortName),"",me.BrowseListBox.cell(i,BR_Size),me.BrowseListBox.cell(i,BR_ShortSize),BrowseWindow(aArray).partnerNick,str(BrowseWindow(aArray).serverID),BrowseWindow(aArray).network)
              me.BrowseListBox.RowPicture(i)=gPicture(r_download_badge)
            end
          next
        end
      end
      BrowseWindowLock.leave()
    end
    
    me.TransferListBoxes(0).listindex=-1
    me.TransferListBoxes(0).ScrollPosition=aIndex
    TransferWindowLock.leave()
  end
End Sub

MainWindow.BR_DownloadFolder:
Protected Sub BR_DownloadFolder()
  dim i,j,aIndex,aArray as integer
  dim temp as string
  if TransferWindowLock.enter() then
    
    if BrowseWindowLock.enter() then
      aIndex=me.FolderListBox.listIndex
      if aIndex>0 then
        temp=me.FolderListBox.cell(aIndex,BR_FolderPath)
        temp=getDefine(nthfield(temp,chrb(92),countfields(temp,chrb(92))-1))
        aArray=BR_SearchID2Array(gBrowseWindowID)
        if aArray<>c_error then
          j=me.BrowseListBox.listCount-1
          for i=0 to j
            aIndex=me.TR_AppendDownloadQueue(me.BrowseListBox.cell(i,BR_Icon),me.BrowseListBox.cell(i,BR_FileName),me.BrowseListBox.cell(i,BR_ShortName),temp,me.BrowseListBox.cell(i,BR_Size),me.BrowseListBox.cell(i,BR_ShortSize),BrowseWindow(aArray).partnerNick,str(BrowseWindow(aArray).serverID),BrowseWindow(aArray).network)
            me.BrowseListBox.RowPicture(i)=gPicture(r_download_badge)
          next
        end
      end
      BrowseWindowLock.leave()
    end
    
    me.TransferListBoxes(0).listindex=-1
    me.TransferListBoxes(0).ScrollPosition=aIndex
    TransferWindowLock.leave()
  end
End Sub

MainWindow.BR_Filter:
Protected Sub BR_Filter()
  dim i,j,k,l,m,aArray,aIndex,isHit,mediaTypeFilter,sizeFilter,bitrateFilter as integer
  dim temp,searchString(0) as string
  if BrowseWindowLock.signal() then
    temp=getString(trim(me.BrowseEditField.text))
    if lenb(temp)<>0 then
      l=countfields(temp,chrb(32))
      for i=1 to l
        searchString.append nthfield(temp,chrb(32),i)
      next
    end
    aIndex=me.FolderListBox.listIndex
    if aIndex>0 then
      temp=""
      temp=me.FolderListBox.cell(aIndex,BR_FolderPath)
      m=1
    end
    select case me.BrowsePopupMenus(1).listindex
    case 0
      mediaTypeFilter=0
    case 1
      mediaTypeFilter=val(rm_music)
    case 2
      mediaTypeFilter=val(rm_pictures)
    case 3
      mediaTypeFilter=val(rm_movies)
    case 4
      mediaTypeFilter=val(rm_text)
    case 5
      mediaTypeFilter=val(rm_files)
    end
    sizeFilter=me.BrowseSliders(0).value*10485760
    bitrateFilter=me.BrowseSliders(1).value
    
    aArray=BR_SearchID2Array(gBrowseWindowID)
    if aArray<>c_error then
      
      if BrowseWindow(aArray).totalSize=0 then
        me.FolderListBox.cell(0,BR_Icon)=""
      else
        me.FolderListBox.cell(0,BR_Icon)=getShortSize(BrowseWindow(aArray).totalSize)
      end
      me.BrowseListBox.visible=false
      me.BrowseListBox.textSize=1
      me.BrowseListBox.deleteAllRows()
      j=ubound(BrowseWindow(aArray).BrowseList)
      for i=1 to j
        isHit=0
        
        if mediaTypeFilter=0 or mediaTypeFilter=val(BrowseWindow(aArray).BrowseList(i).mediaType) then
          isHit=isHit+1
        end
        
        if sizeFilter=0 or sizeFilter<=val(BrowseWindow(aArray).BrowseList(i).size) then
          isHit=isHit+1
        end
        
        if bitrateFilter=0 or bitrateFilter<=val(BrowseWindow(aArray).BrowseList(i).bitrate) then
          isHit=isHit+1
        end
        
        for k=1 to l
          if instrb(1,BrowseWindow(aArray).BrowseList(i).fileName,searchString(k))<>0 then
            isHit=isHit+1
          end
        next
        
        if m=1 and strcomp(BrowseWindow(aArray).BrowseList(i).folderPath,temp,0)=0 then
          isHit=isHit+1
        end
        
        if isHit=3+l+m then
          me.BrowseListBox.addrow("")
          aIndex=me.BrowseListBox.lastIndex
          me.BrowseListBox.rowPicture(aIndex)=gMediaPicture(val(BrowseWindow(aArray).BrowseList(i).mediaType))
          me.BrowseListBox.cell(aIndex,BR_Icon)=BrowseWindow(aArray).BrowseList(i).mediaType
          me.BrowseListBox.cell(aIndex,BR_FileName)=BrowseWindow(aArray).BrowseList(i).fileName
          me.BrowseListBox.cell(aIndex,BR_ShortName)=BrowseWindow(aArray).BrowseList(i).shortName
          me.BrowseListBox.cell(aIndex,BR_Size)=BrowseWindow(aArray).BrowseList(i).size
          me.BrowseListBox.cell(aIndex,BR_ShortSize)=BrowseWindow(aArray).BrowseList(i).shortSize
          me.BrowseListBox.cell(aIndex,BR_BitRate)=BrowseWindow(aArray).BrowseList(i).bitRate
        end
        
      next
      me.BrowseListBox.scrollPosition=0
      me.BrowseListBox.headingIndex=-1
      me.BrowseListBox.textSize=10
      me.BrowseListBox.visible=true
      DrawerWindow.DR_UpdateResults(BrowseWindow(aArray).ID,BrowseWindow(aArray).partnerNick,me.BrowseListBox.listCount,ubound(BrowseWindow(aArray).BrowseList))
      
    end
    BrowseWindowLock.release()
  end
End Sub

MainWindow.BR_Message:
Protected Sub BR_Message()
  dim aArray as integer
  if BrowseWindowLock.enter() then
    aArray=BR_SearchID2Array(gBrowseWindowID)
    if aArray<>c_error then
      ME_NewMessage(BrowseWindow(aArray).serverID,BrowseWindow(aArray).partnerNick)
    end
    BrowseWindowLock.leave()
  end
End Sub

MainWindow.BR_Resume:
Protected Sub BR_Resume()
  dim d as openDialog
  dim fi as folderItem
  dim i,j,aIndex,aArray as integer
  if TransferWindowLock.enter() then
    
    if BrowseWindowLock.enter() then
      aArray=BR_SearchID2Array(gBrowseWindowID)
      if aArray<>c_error then
        
        if me.BrowseListBox.listindex<>-1 then
          j=me.BrowseListBox.listCount-1
          for i=0 to j
            if me.BrowseListBox.selected(i) then
              d=new openDialog
              d.Filter="special/any"
              d.promptText=me.BrowseListBox.cell(i,BR_ShortName)
              d.initialDirectory=iIncompleteFolder
              if d.initialDirectory<>nil and d.initialDirectory.exists then
                fi=d.showModalWithin(me)
                if fi<>nil and fi.exists then
                  if fi.parent<>d.initialDirectory then
                    moveFolderItem(fi,d.initialDirectory)
                  end
                  aIndex=me.TR_AppendDownloadQueue(me.BrowseListBox.cell(i,BR_Icon),me.BrowseListBox.cell(i,BR_FileName),getString(fi.name),"",me.BrowseListBox.cell(i,BR_Size),me.BrowseListBox.cell(i,BR_ShortSize),BrowseWindow(aArray).partnerNick,str(BrowseWindow(aArray).serverID),BrowseWindow(aArray).network)
                  me.BrowseListBox.RowPicture(i)=gPicture(r_download_badge)
                end
              end
            end
          next
        end
        
      end
      BrowseWindowLock.leave()
    end
    
    me.TransferListBoxes(0).listindex=-1
    me.TransferListBoxes(0).ScrollPosition=aIndex
    TransferWindowLock.leave()
  end
End Sub

MainWindow.BR_SearchFolderPath2ListIndex:
Protected Function BR_SearchFolderPath2ListIndex(inFolderPath as string) As integer
  dim i,j as integer
  j=me.FolderListBox.listCount-1
  for i=0 to j
    if strcomp(me.FolderListBox.cell(i,BR_FolderPath),inFolderPath,0)=0 then
      return i
    end
  next
  return c_error
End Function

MainWindow.BR_SwitchView:
Protected Sub BR_SwitchView()
  if me.BrowseListBox.column(BR_FileName).widthExpression="0" then
    me.BrowseListBox.column(BR_FileName).widthExpression=me.BrowseListBox.column(BR_ShortName).widthExpression
    me.BrowseListBox.column(BR_ShortName).widthExpression="0"
  else
    me.BrowseListBox.column(BR_ShortName).widthExpression=me.BrowseListBox.column(BR_FileName).widthExpression
    me.BrowseListBox.column(BR_FileName).widthExpression="0"
  end
  me.BrowseListBox.columnWidths=me.BrowseListBox.columnWidths
End Sub

MainWindow.BR_UserList:
Protected Sub BR_UserList(inIsFavorite as boolean)
  dim aArray as integer
  if BrowseWindowLock.enter() then
    aArray=BR_SearchID2Array(gBrowseWindowID)
    if aArray<>c_error then
      me.US_Append(BrowseWindow(aArray).serverID,BrowseWindow(aArray).partnerNick,inIsFavorite)
    end
    BrowseWindowLock.leave()
  end
End Sub

MainWindow.BR_Whois:
Protected Sub BR_Whois()
  dim aArray as integer
  if BrowseWindowLock.enter() then
    aArray=BR_SearchID2Array(gBrowseWindowID)
    if aArray<>c_error then
      'whois request
      OS_Command(BrowseWindow(aArray).serverID,603,BrowseWindow(aArray).partnerNick)
      me.WhoisListBox.listIndex=-1
      DrawerWindow.MenuListBox.listIndex=3
    end
    BrowseWindowLock.leave()
  end
End Sub

MainWindow.BR_WriteItem:
Protected Sub BR_WriteItem()
  dim d as SaveAsDialog
  dim i,j as integer
  dim fi as FolderItem
  dim tos as TextOutputStream
  if BrowseWindowLock.enter() then
    d=new SaveAsDialog
    d.suggestedFileName=me.title
    fi=d.showModalWithin(me)
    if fi<>nil then
      tos=fi.CreateTextFile()
      j=me.BrowseListBox.listCount-1
      for i=0 to j
        tos.writeline(me.BrowseListBox.cell(i,BR_FileName)+chrb(9)+me.BrowseListBox.cell(i,BR_ShortSize)+chrb(9)+me.BrowseListBox.cell(i,BR_Bitrate))
      next
      tos.close()
    end
    BrowseWindowLock.leave()
  end
exception
  if tos<>nil then
    tos.close()
  end
  BrowseWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.ME_Browse:
Protected Sub ME_Browse(inIsDirectBrowse as boolean)
  dim aArray as integer
  if MessageWindowLock.enter() then
    aArray=ME_SearchID2Array(gMessageWindowID)
    if aArray<>c_error then
      BR_NewBrowse(MessageWindow(aArray).serverID,MessageWindow(aArray).partnerNick,inIsDirectBrowse)
    end
    MessageWindowLock.leave()
  end
End Sub

MainWindow.ME_Message:
Protected Sub ME_Message(inWhat as string)
  dim aArray as integer
  if MessageWindowLock.enter() then
    aArray=ME_SearchID2Array(gMessageWindowID)
    if aArray<>c_error then
      if lenb(MessageWindow(aArray).channel)=0 then
        getMessage(MessageWindow(aArray).textAndStyle,MessageWindow(aArray).nick,co_own_user,inWhat,co_default)
        me.MessageEditField.AppendMessage(MessageWindow(aArray).nick,co_own_user,inWhat,co_default)
        'private message
        OS_Command(MessageWindow(aArray).serverID,205,MessageWindow(aArray).partnerNick+chrb(32)+inWhat)
      else
        if inWhat="Y" then
          getMessage(MessageWindow(aArray).textAndStyle,getString(s_accepted_),co_notifications)
          me.MessageEditField.AppendMessage(getString(s_accepted_),co_notifications)
          'accept invitation
          OS_Command(MessageWindow(aArray).serverID,431,MessageWindow(aArray).partnerNick+chrb(32)+MessageWindow(aArray).channel)
          'join channel
          OS_Command(MessageWindow(aArray).serverID,400,MessageWindow(aArray).channel)
        else
          getMessage(MessageWindow(aArray).textAndStyle,getString(s_declined_),co_notifications)
          me.MessageEditField.AppendMessage(getString(s_declined_),co_notifications)
          'decline invitation
          OS_Command(MessageWindow(aArray).serverID,432,MessageWindow(aArray).partnerNick+chrb(32)+MessageWindow(aArray).channel)
        end
        MessageWindow(aArray).channel=""
      end
      getSound(rs_click)
    end
    MessageWindowLock.leave()
  end
End Sub

MainWindow.ME_UserList:
Protected Sub ME_UserList(inIsFavorite as boolean)
  dim aArray as integer
  if MessageWindowLock.enter() then
    aArray=ME_SearchID2Array(gMessageWindowID)
    if aArray<>c_error then
      me.US_Append(MessageWindow(aArray).serverID,MessageWindow(aArray).partnerNick,inIsFavorite)
    end
    MessageWindowLock.leave()
  end
End Sub

MainWindow.ME_Whois:
Protected Sub ME_Whois()
  dim aArray as integer
  if MessageWindowLock.enter() then
    aArray=ME_SearchID2Array(gMessageWindowID)
    if aArray<>c_error then
      'whois request
      OS_Command(MessageWindow(aArray).serverID,603,MessageWindow(aArray).partnerNick)
      me.WhoisListBox.listIndex=-1
      DrawerWindow.MenuListBox.listIndex=3
    end
    MessageWindowLock.leave()
  end
End Sub

MainWindow.ME_WriteItem:
Protected Sub ME_WriteItem()
  dim d as saveAsDialog
  dim fi as FolderItem
  dim tos as TextOutputStream
  if MessageWindowLock.enter() then
    d=new saveAsDialog
    d.suggestedFileName=me.title
    fi=d.showModalWithin(me)
    if fi<>nil then
      tos=fi.AppendToTextFile()
      tos.write(getString(me.MessageEditField.text))
      tos.close()
    end
    MessageWindowLock.leave()
  end
  me.MessageEditField.setFocus()
  me.CommandEditFields(0).setFocus()
exception
  if tos<>nil then
    tos.close()
  end
  MessageWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.ME_Filter:
Protected Sub ME_Filter()
  dim aArray as integer
  if MessageWindowLock.enter() then
    aArray=ME_SearchID2Array(gMessageWindowID)
    if aArray<>c_error then
      me.MessageEditField.text=""
      me.MessageEditField.setTextAndStyle(MessageWindow(aArray).textAndStyle.text,MessageWindow(aArray).textAndStyle.textStyleData)
      MessageWindow(aArray).textAndStyle.textStyleData=""
      MessageWindow(aArray).textAndStyle.textStyleData=me.MessageEditField.textStyleData
      me.MessageEditField.scrollBottom()
      
      select case MessageWindow(aArray).partnerNick
      case "OperServ"
        me.Toolbars(91).enabled=true
        me.CommandContextualMenus(0).deleteAllRows()
        me.CommandContextualMenus(0).addrow("add_server ")
        me.CommandContextualMenus(0).addrow("block ")
        me.CommandContextualMenus(0).addrow("blocklist")
        me.CommandContextualMenus(0).addrow("chanlevel ")
        me.CommandContextualMenus(0).addrow("chanlimit ")
        me.CommandContextualMenus(0).addrow("clearchan ")
        me.CommandContextualMenus(0).addrow("cloak")
        me.CommandContextualMenus(0).addrow("config ")
        me.CommandContextualMenus(0).addrow("connect ")
        me.CommandContextualMenus(0).addrow("cycle ")
        me.CommandContextualMenus(0).addrow("deop ")
        me.CommandContextualMenus(0).addrow("disconnect ")
        me.CommandContextualMenus(0).addrow("help")
        me.CommandContextualMenus(0).addrow("kick ")
        me.CommandContextualMenus(0).addrow("killserver ")
        me.CommandContextualMenus(0).addrow("links")
        me.CommandContextualMenus(0).addrow("list_server")
        me.CommandContextualMenus(0).addrow("muzzle ")
        me.CommandContextualMenus(0).addrow("nuke ")
        me.CommandContextualMenus(0).addrow("op ")
        me.CommandContextualMenus(0).addrow("reconfig ")
        me.CommandContextualMenus(0).addrow("redirect ")
        me.CommandContextualMenus(0).addrow("register ")
        me.CommandContextualMenus(0).addrow("rehash")
        me.CommandContextualMenus(0).addrow("removeserver ")
        me.CommandContextualMenus(0).addrow("restart")
        me.CommandContextualMenus(0).addrow("setvar ")
        me.CommandContextualMenus(0).addrow("stats")
        me.CommandContextualMenus(0).addrow("unblock ")
        me.CommandContextualMenus(0).addrow("unmuzzle ")
        me.CommandContextualMenus(0).addrow("userflags ")
        me.CommandContextualMenus(0).addrow("userlist")
        me.CommandContextualMenus(0).addrow("usermode ")
        me.CommandContextualMenus(0).addrow("usermode help")
        me.CommandContextualMenus(0).addrow("whowas ")
      case "ChanServ"
        me.Toolbars(91).enabled=true
        me.CommandContextualMenus(0).deleteAllRows()
        me.CommandContextualMenus(0).addrow("ban ")
        me.CommandContextualMenus(0).addrow("banclear ")
        me.CommandContextualMenus(0).addrow("banlist")
        me.CommandContextualMenus(0).addrow("chanannounce ")
        me.CommandContextualMenus(0).addrow("clear")
        me.CommandContextualMenus(0).addrow("deop ")
        me.CommandContextualMenus(0).addrow("drop ")
        me.CommandContextualMenus(0).addrow("help")
        me.CommandContextualMenus(0).addrow("invite ")
        me.CommandContextualMenus(0).addrow("kick ")
        me.CommandContextualMenus(0).addrow("level ")
        me.CommandContextualMenus(0).addrow("limit ")
        me.CommandContextualMenus(0).addrow("mode ")
        me.CommandContextualMenus(0).addrow("muzzle ")
        me.CommandContextualMenus(0).addrow("op ")
        me.CommandContextualMenus(0).addrow("summon ")
        me.CommandContextualMenus(0).addrow("topic ")
        me.CommandContextualMenus(0).addrow("unban ")
        me.CommandContextualMenus(0).addrow("unmuzzle ")
        me.CommandContextualMenus(0).addrow("unvoice ")
        me.CommandContextualMenus(0).addrow("voice ")
        me.CommandContextualMenus(0).addrow("wallop ")
      case "NickServ"
        me.Toolbars(91).enabled=true
        me.CommandContextualMenus(0).deleteAllRows()
        me.CommandContextualMenus(0).addrow("ghost ")
        me.CommandContextualMenus(0).addrow("help")
        me.CommandContextualMenus(0).addrow("pass ")
        me.CommandContextualMenus(0).addrow("rate ")
        me.CommandContextualMenus(0).addrow("register ")
        me.CommandContextualMenus(0).addrow("server ")
        me.CommandContextualMenus(0).addrow("usermode ")
      else
        me.Toolbars(91).enabled=false
      end
    end
    MessageWindowLock.leave()
  end
End Sub

MainWindow.CO_Update:
Sub CO_Update()
  me.ConsoleEditField.textFont=iTextFont
  me.ConsoleEditField.textSize=iTextSize
  me.ConsoleEditField.textColor=iColors(co_default)
  me.ConsoleEditField.backColor=iColors(co_background)
End Sub

MainWindow.RO_Browse:
Protected Sub RO_Browse(inIsDirectBrowse as boolean)
  dim aArray,i,j as integer
  if RoomWindowLock.enter() then
    aArray=RO_SearchID2Array(gRoomWindowID)
    if aArray<>c_error then
      if me.RoomListBox.listindex<>-1 then
        j=me.RoomListBox.listCount-1
        for i=0 to j
          if me.RoomListBox.selected(i) then
            BR_NewBrowse(RoomWindow(aArray).serverID,me.RoomListBox.cell(i,RO_Nick),inIsDirectBrowse)
          end
        next
      end
    end
    RoomWindowLock.leave()
  end
End Sub

MainWindow.RO_Message:
Protected Sub RO_Message()
  dim aArray,i,j as integer
  if RoomWindowLock.enter() then
    aArray=RO_SearchID2Array(gRoomWindowID)
    if aArray<>c_error then
      if me.RoomListBox.listindex<>-1 then
        j=me.RoomListBox.listCount-1
        for i=0 to j
          if me.RoomListBox.selected(i) then
            ME_NewMessage(RoomWindow(aArray).serverID,me.RoomListBox.cell(i,RO_Nick))
          end
        next
      end
    end
    RoomWindowLock.leave()
  end
End Sub

MainWindow.RO_PublicMessage:
Protected Sub RO_PublicMessage(inWhat as string)
  dim aArray as integer
  dim temp as string
  if RoomWindowLock.enter() then
    aArray=RO_SearchID2Array(gRoomWindowID)
    if aArray<>c_error then
      temp=nthfield(inWhat,chrb(32),1)
      select case temp
      case ">invite"
        'invite user
        OS_Command(RoomWindow(aArray).serverID,430,midb(inWhat,lenb(temp)+2)+chrb(32)+RoomWindow(aArray).channel)
      case ">me"
        'emote in channel
        OS_Command(RoomWindow(aArray).serverID,824,RoomWindow(aArray).channel+chrb(32)+chrb(34)+midb(inWhat,lenb(temp)+2)+chrb(34))
      case ">msg"
        'private message to another user
        OS_Command(RoomWindow(aArray).serverID,205,midb(inWhat,lenb(temp)+2))
      case ">part"
        'part channel
        OS_Command(RoomWindow(aArray).serverID,401,RoomWindow(aArray).channel)
      case ">topic"
        'channel topic
        OS_Command(RoomWindow(aArray).serverID,410,RoomWindow(aArray).channel+chrb(32)+midb(inWhat,lenb(temp)+2))
      case ">whois"
        'whois request
        OS_Command(RoomWindow(aArray).serverID,604,midb(inWhat,lenb(temp)+2))
      else
        'public message
        OS_Command(RoomWindow(aArray).serverID,402,RoomWindow(aArray).channel+chrb(32)+inWhat)
      end
    end
    RoomWindowLock.leave()
  end
End Sub

MainWindow.RO_Refresh:
Protected Sub RO_Refresh()
  dim aArray as integer
  if RoomWindowLock.enter() then
    aArray=RO_SearchID2Array(gRoomWindowID)
    if aArray<>c_error then
      'channel userlist request
      OS_Command(RoomWindow(aArray).serverID,830,RoomWindow(aArray).channel)
    end
    RoomWindowLock.leave()
  end
End Sub

MainWindow.RO_SearchNick2ListIndex:
Function RO_SearchNick2ListIndex(inPartnerNick as string) As Integer
  dim i,j as integer
  j=me.RoomListBox.listCount-1
  for i=0 to j
    if strcomp(me.RoomListBox.cell(i,RO_Nick),inPartnerNick,0)=0 then
      return i
    end
  next
  return c_error
End Function

MainWindow.RO_UserList:
Protected Sub RO_UserList(inIsFavorite as boolean)
  dim aArray,i,j as integer
  if RoomWindowLock.enter() then
    aArray=RO_SearchID2Array(gRoomWindowID)
    if aArray<>c_error then
      if me.RoomListBox.listindex<>-1 then
        j=me.RoomListBox.listCount-1
        for i=0 to j
          if me.RoomListBox.selected(i) then
            me.US_Append(RoomWindow(aArray).serverID,me.RoomListBox.cell(i,RO_Nick),inIsFavorite)
          end
        next
      end
    end
    RoomWindowLock.leave()
  end
End Sub

MainWindow.RO_Whois:
Protected Sub RO_Whois()
  dim aArray,i,j as integer
  if RoomWindowLock.enter() then
    aArray=RO_SearchID2Array(gRoomWindowID)
    if aArray<>c_error then
      if me.RoomListBox.listindex<>-1 then
        j=me.RoomListBox.listCount-1
        for i=0 to j
          if me.RoomListBox.selected(i) then
            'whois request
            OS_Command(RoomWindow(aArray).serverID,603,me.RoomListBox.cell(i,RO_Nick))
          end
        next
        me.WhoisListBox.listIndex=-1
        DrawerWindow.MenuListBox.listIndex=3
      end
    end
    RoomWindowLock.leave()
  end
End Sub

MainWindow.RO_WriteItem:
Protected Sub RO_WriteItem()
  dim d as saveAsDialog
  dim fi as FolderItem
  dim tos as TextOutputStream
  if RoomWindowLock.enter() then
    d=new saveAsDialog
    d.suggestedFileName=me.title
    fi=d.showModalWithin(me)
    if fi<>nil then
      tos=fi.AppendToTextFile()
      tos.write(getString(me.RoomEditField.text))
      tos.close()
    end
    RoomWindowLock.leave()
  end
  me.RoomEditField.setFocus()
  me.CommandEditFields(1).setFocus()
exception
  if tos<>nil then
    tos.close()
  end
  RoomWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.RO_Filter:
Protected Sub RO_Filter()
  dim aArray,i,j,aIndex as integer
  if RoomWindowLock.signal() then
    aArray=RO_SearchID2Array(gRoomWindowID)
    if aArray<>c_error then
      me.RoomEditField.text=""
      me.RoomEditField.setTextAndStyle(RoomWindow(aArray).textAndStyle.text,RoomWindow(aArray).textAndStyle.textStyleData)
      RoomWindow(aArray).textAndStyle.textStyleData=""
      RoomWindow(aArray).textAndStyle.textStyleData=me.RoomEditField.textStyleData
      me.RoomEditField.scrollBottom()
      me.RoomStaticText.text=RoomWindow(aArray).topic
      me.RoomStaticText.helpTag=RoomWindow(aArray).topic
      me.RoomStaticText.balloonHelp=RoomWindow(aArray).topic
      me.RoomListBox.visible=false
      me.RoomListBox.textSize=1
      me.RoomListBox.deleteAllRows()
      j=ubound(RoomWindow(aArray).RoomList)
      for i=1 to j
        me.RoomListBox.addrow(RoomWindow(aArray).RoomList(i).nick)
        aIndex=me.RoomListBox.lastIndex
        me.RoomListBox.cell(aIndex,RO_Count)=RoomWindow(aArray).RoomList(i).count
        me.RoomListBox.cell(aIndex,RO_Speed)=RoomWindow(aArray).RoomList(i).speed
      next
      me.RoomListBox.scrollPosition=0
      me.RoomListBox.headingIndex=-1
      me.RoomListBox.textSize=10
      me.RoomListBox.visible=true
    end
    RoomWindowLock.release()
  end
End Sub

MainWindow.CO_WriteItem:
Protected Sub CO_WriteItem()
  dim d as saveAsDialog
  dim fi as FolderItem
  dim tos as TextOutputStream
  if ConsoleWindowLock.enter() then
    d=new saveAsDialog
    d.suggestedFileName=me.title+" - "+me.ConsolePopupMenu.text
    fi=d.showModalWithin(me)
    if fi<>nil then
      tos=fi.AppendToTextFile()
      tos.write(getString(me.ConsoleEditField.text))
      tos.close()
    end
    ConsoleWindowLock.leave()
  end
  me.ConsoleEditField.setFocus()
  me.EditFields(1).setFocus()
exception
  if tos<>nil then
    tos.close()
  end
  ConsoleWindowLock.leave()
  getSound(rs_error)
  msgbox s_file_writing_error
End Sub

MainWindow.NE_CleanUp:
Protected Sub NE_CleanUp()
  dim i,aServerID as integer
  if NetworkWindowLock.signal() then
    for i=me.NetworkListBox.listCount-1 downto 0
      if lenb(me.NetworkListBox.cell(i,NE_Icon))=1 then
        aServerID=val(me.NetworkListBox.cell(i,NE_ID))
        if aServerID<>0 then
          OS_Disconnect(aServerID)
        end
        me.NetworkListBox.removeRow(i)
      end
    next
    NetworkWindowLock.release()
  end
End Sub

MainWindow.TR_CleanUp:
Protected Sub TR_CleanUp(inArray as integer)
  dim i,aFlag,transferedSize as integer
  if TransferWindowLock.signal() then
    for i=me.TransferListBoxes(inArray).listCount-1 downto 0
      aFlag=val(me.TransferListBoxes(inArray).cell(i,TR_Flag))
      transferedSize=val(me.TransferListBoxes(inArray).cell(i,TR_TransferedSize))
      if val(me.TransferListBoxes(inArray).cell(i,TR_ID))=0 and (inArray=1 or (transferedSize=0 or transferedSize=val(me.TransferListBoxes(inArray).cell(i,TR_Size)))) and aFlag<>TF_LocallyQueued and aFlag<>TF_NegociatingWithSever and aFlag<>TF_Queueble and aFlag<>TF_QueuedAt and aFlag<>TF_RetryingIn then
        if aFlag=TF_RetryingIn then
          gDownloadRetrying=gDownloadRetrying-1
        elseif aFlag=TF_Preparing then
          gUploadPreparing=gUploadPreparing-1
        end
        me.TransferListBoxes(inArray).removeRow(i)
      end
    next
    TransferWindowLock.release()
  end
End Sub

MainWindow.SE_UserList:
Protected Sub SE_UserList(inIsFavorite as boolean)
  dim i,j,aServerID as integer
  if SearchWindowLock.enter() then
    if me.SearchListBox.listindex<>-1 then
      j=me.SearchListBox.listCount-1
      for i=0 to j
        aServerID=val(me.SearchListBox.cell(i,SE_ServerID))
        if me.SearchListBox.selected(i) and aServerID<>0 then
          me.US_Append(aServerID,me.SearchListBox.cell(i,SE_Nick),inIsFavorite)
        end
      next
    end
    SearchWindowLock.leave()
  end
End Sub

MainWindow.WH_Remove:
Protected Sub WH_Remove()
  dim i as integer
  if WhoisWindowLock.signal() then
    if me.WhoisListBox.listIndex<>-1 then
      for i=me.WhoisListBox.listCount-1 downto 0
        if me.WhoisListBox.selected(i) then
          me.WhoisListBox.removeRow i
        end
      next
    end
    WhoisWindowLock.release()
  end
End Sub

MainWindow.TR_RefreshUploadQueue:
Sub TR_RefreshUploadQueue()
  dim i,j,aOpenNapArray,aCount,aServerID,aFlag,locallyQueuedServerID(0) as integer
  dim partnerNick,locallyQueuedNick(0) as string
  if TransferWindowLock.enter() then
    gUploadQueue=0
    j=me.TransferListBoxes(1).listCount-1
    for i=0 to j
      aFlag=val(me.TransferListBoxes(1).cell(i,TR_Flag))
      aCount=val(me.TransferListBoxes(1).cell(i,TR_Timer))-5
      
      select case aFlag
      case TF_NegociatingWithSever
        if aCount<=0 then
          me.TransferListBoxes(1).cell(i,TR_ServerID)=""
          me.TransferListBoxes(1).cell(i,TR_Network)=s_offline
          me.TransferListBoxes(1).cellRefresh(i,TR_Nick)
          me.TransferListBoxes(1).cell(i,TR_Flag)=""
          me.TransferListBoxes(1).cell(i,TR_Status)=s_accept_failed
          me.TransferListBoxes(1).cellRefresh(i,TR_ShortName)
        else
          me.TransferListBoxes(1).cell(i,TR_Timer)=str(aCount)
        end
        
      case TF_Preparing
        if aCount<=0 then
          gUploadPreparing=gUploadPreparing-1
          me.TransferListBoxes(1).cell(i,TR_Flag)=""
          me.TransferListBoxes(1).cell(i,TR_Status)=s_accept_failed
          me.TransferListBoxes(1).cellRefresh(i,TR_ShortName)
        else
          me.TransferListBoxes(1).cell(i,TR_Timer)=str(aCount)
        end
        
      case TF_Queueble
        if aCount<=0 then
          me.TransferListBoxes(1).cell(i,TR_Flag)=""
          me.TransferListBoxes(1).cell(i,TR_Status)=s_accept_failed
          me.TransferListBoxes(1).cellRefresh(i,TR_ShortName)
        else
          me.TransferListBoxes(1).cell(i,TR_Timer)=str(aCount)
        end
        
      case TF_LocallyQueued
        partnerNick=""
        partnerNick=me.TransferListBoxes(1).cell(i,TR_Nick)
        aServerID=val(me.TransferListBoxes(1).cell(i,TR_ServerID))
        if hasNickAndServerID(locallyQueuedNick,locallyQueuedServerID,partnerNick,aServerID)=false then
          if (iUploadLimits=0 or iUploadLimits>me.TR_CountUploadQueue()) and (iUploadPerUser=0 or iUploadPerUser>me.TR_CountNick2UploadQueue(partnerNick)) then
            
            if OpenNapSocketLock.enter() then
              aOpenNapArray=OS_SearchID2Array(aServerID)
              if aOpenNapArray<>c_error then
                if iPort<>0 then
                  TS_PrepareTransfer(aServerID,OpenNapSocket(aOpenNapArray).nick,OpenNapSocket(aOpenNapArray).isWinMX,i,"",0,1,false)
                  'accept upload request
                  OpenNapSocket(aOpenNapArray).command(608,partnerNick+chrb(32)+chrb(34)+me.TransferListBoxes(1).cell(i,TR_FileName)+chrb(34))
                else
                  me.TransferListBoxes(1).cell(i,TR_Timer)="120"
                  me.TransferListBoxes(1).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
                  me.TransferListBoxes(1).cell(i,TR_Status)=s_negociating_with_server
                  me.TransferListBoxes(1).cellRefresh(i,TR_ShortName)
                  'accept upload request
                  OpenNapSocket(aOpenNapArray).command(608,partnerNick+chrb(32)+chrb(34)+me.TransferListBoxes(1).cell(i,TR_FileName)+chrb(34))
                end
              end
              OpenNapSocketLock.leave()
            end
            
          else
            locallyQueuedNick.append partnerNick
            locallyQueuedServerID.append aServerID
            if aCount<=0 then
              me.TransferListBoxes(1).cell(i,TR_Flag)=""
              me.TransferListBoxes(1).cell(i,TR_Status)=s_accept_failed
              me.TransferListBoxes(1).cellRefresh(i,TR_ShortName)
            else
              gUploadQueue=gUploadQueue+1
              me.TransferListBoxes(1).cell(i,TR_Timer)=str(aCount)
            end
          end
        else
          if aCount<=0 then
            me.TransferListBoxes(1).cell(i,TR_Flag)=""
            me.TransferListBoxes(1).cell(i,TR_Status)=s_accept_failed
            me.TransferListBoxes(1).cellRefresh(i,TR_ShortName)
          else
            gUploadQueue=gUploadQueue+1
            me.TransferListBoxes(1).cell(i,TR_Timer)=str(aCount)
          end
        end
        
      end
    next
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Pause:
Protected Sub TR_Pause()
  dim i,j,aFlag as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(0).listindex<>-1 then
      j=me.TransferListBoxes(0).listCount-1
      for i=0 to j
        if me.TransferListBoxes(0).selected(i) then
          if val(me.TransferListBoxes(0).cell(i,TR_ID))<>0 then
            TS_Disconnect(val(me.TransferListBoxes(0).cell(i,TR_ID)))
          else
            aFlag=val(me.TransferListBoxes(0).cell(i,TR_Flag))
            select case aFlag
            case TF_RetryingIn
              gDownloadRetrying=gDownloadRetrying-1
            case TF_Preparing
              gUploadPreparing=gUploadPreparing-1
            end
          end
          me.TransferListBoxes(0).cell(i,TR_Flag)=str(TF_Paused)
          me.TransferListBoxes(0).cell(i,TR_Status)=s_paused
          me.TransferListBoxes(0).cellRefresh(i,TR_ShortName)
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.TR_Upload:
Protected Sub TR_Upload()
  dim i,j,aIndex,aOpenNapArray,aServerID as integer
  if TransferWindowLock.enter() then
    if me.TransferListBoxes(1).listindex<>-1 then
      j=me.TransferListBoxes(1).listCount-1
      for i=0 to j
        if me.TransferListBoxes(1).selected(i) and val(me.TransferListBoxes(1).cell(i,TR_Flag))=TF_LocallyQueued then
          aServerID=val(me.TransferListBoxes(1).cell(i,TR_ServerID))
          
          if OpenNapSocketLock.enter() then
            aOpenNapArray=OS_SearchID2Array(aServerID)
            if aOpenNapArray<>c_error then
              if iPort<>0 then
                TS_PrepareTransfer(aServerID,OpenNapSocket(aOpenNapArray).nick,OpenNapSocket(aOpenNapArray).isWinMX,i,"",0,1,false)
                'accept upload request
                OpenNapSocket(aOpenNapArray).command(608,me.TransferListBoxes(1).cell(i,TR_Nick)+chrb(32)+chrb(34)+me.TransferListBoxes(1).cell(i,TR_FileName)+chrb(34))
              else
                me.TransferListBoxes(1).cell(i,TR_Timer)="120"
                me.TransferListBoxes(1).cell(i,TR_Flag)=str(TF_NegociatingWithSever)
                me.TransferListBoxes(1).cell(i,TR_Status)=s_negociating_with_server
                me.TransferListBoxes(1).cellRefresh(i,TR_ShortName)
                'accept upload request
                OpenNapSocket(aOpenNapArray).command(608,me.TransferListBoxes(1).cell(i,TR_Nick)+chrb(32)+chrb(34)+me.TransferListBoxes(1).cell(i,TR_FileName)+chrb(34))
              end
            end
            OpenNapSocketLock.leave()
          end
          
        end
      next
    end
    TransferWindowLock.leave()
  end
End Sub

MainWindow.EnableMenuItems:
Sub EnableMenuItems()
  closeWindowMenu.enabled=true
  closeTabMenu.enabled=(DrawerWindow.TaskListBox.listIndex>0)
  minimizeMenu.enabled=true
  searchMenu.enabled=true
  toolbarMenu.enabled=true
  drawerMenu.enabled=true
  select case self.PagePanels.value
  case 0
    saveMenu.enabled=true
  case 1
    saveAsMenu.enabled=true
    scrollbackMenu.enabled=true
  case 4
    saveMenu.enabled=true
  case 5
    saveMenu.enabled=true
  case 8
    saveAsMenu.enabled=true
  case 9
    saveAsMenu.enabled=true
    scrollbackMenu.enabled=true
  case 10
    saveAsMenu.enabled=true
    scrollbackMenu.enabled=true
  end
End Sub

MainWindow.Close:
Sub Close()
  writeToolbar(c_network_window,(me.NetworkListBox.top=-1))
  writeWindow(me,c_network_window)
End Sub

MainWindow.CancelClose:
Function CancelClose(appQuitting as Boolean) As Boolean
  if isWantQuit=false then
    #if targetCarbon
      DrawerWindow.hide()
    #endif
    me.hide()
    return true
  end
End Function

MainWindow.Open:
Sub Open()
  #if targetPPC
    if getToolbar(c_network_window) then
      me.MA_ToggleToolbar()
    end
    getWindow(me,c_network_window)
  #endif
End Sub

MainWindow.RoundedCanvases.IconClicked:
Sub IconClicked(Index As Integer)
  dim i as integer
  self.HistoryContextualMenu.deleteAllRows()
  if index=3 or index=4 then
    for i=ubound(whoisHistory) downto 1
      self.HistoryContextualMenu.addrow(whoisHistory(i))
    next
    self.HistoryContextualMenu.addSeparator()
    self.HistoryContextualMenu.addRow(s_clear_entries)
    self.HistoryContextualMenu.open()
  else
    for i=ubound(searchHistory) downto 1
      self.HistoryContextualMenu.addrow(searchHistory(i))
    next
    self.HistoryContextualMenu.addSeparator()
    self.HistoryContextualMenu.addRow(s_clear_entries)
    self.HistoryContextualMenu.open()
  end
End Sub

MainWindow.RoundedCanvases.Open:
Sub Open(Index As Integer)
  me.hasSearchIcon=true
  me.setEditField(self.EditFields(index))
End Sub

MainWindow.EditFields.KeyDown:
Function KeyDown(Index As Integer, key as string) As Boolean
  if index=3 or index=4 then
    if ascb(key)=13 and lenb(me.text)<>0 then
      appendHistory(whoisHistory,getString(trim(me.text)))
      self.WH_Whereis(getString(trim(me.text)))
      me.text=""
    elseif instrb(1,c_invalid_keys+chrb(32),key)<>0 then
      return true
    end
    
  else
    if ascb(key)=13 and lenb(me.text)<>0 then
      appendHistory(searchHistory,getString(trim(me.text)))
      SE_NewSearch(getString(trim(me.text)))
      me.text=""
    elseif ascb(key)=34 then
      return true
    end
  end
End Function

MainWindow.HistoryContextualMenu.Action:
Sub Action(item As String)
  dim aIndex as integer
  aIndex=self.PagePanels.value
  self.EditFields(aIndex).setFocus()
  if item=s_clear_entries then
    if aIndex=3 or aIndex=4 then
      redim whoisHistory(0)
    else
      redim searchHistory(0)
    end
    self.EditFields(aIndex).text=""
  else
    self.EditFields(aIndex).ReplaceText(item)
  end
End Sub

MainWindow.Toolbars.Action:
Sub Action(Index As Integer)
  select case index
  case 0
    self.NE_Remove()
  case 1
    self.NE_CleanUp()
  case 2
    self.ToolbarContextualMenus(0).open()
  case 3
    self.NE_Disconnect()
  case 4
    self.NE_Connect()
    
  case 10
    self.ToolbarContextualMenus(1).open()
  case 11
    self.CO_Refresh()
    
  case 20
    self.CH_Add()
  case 21
    self.CH_Refresh()
  case 22
    self.CH_Join()
    
  case 30
    self.WH_Message()
  case 31
    self.ToolbarContextualMenus(2).open()
  case 32
    self.WH_Remove()
  case 33
    self.WH_CleanUp()
  case 34
    self.WH_Refresh()
    
  case 40
    self.US_Message()
  case 41
    self.ToolbarContextualMenus(3).open()
  case 42
    self.US_Remove()
  case 43
    AddUserWindow.showWithin(self)
    
  case 50
    self.TR_Message(0)
  case 51
    self.ToolbarContextualMenus(4).open()
  case 52
    self.TR_Remove(0)
  case 53
    self.TR_CleanUp(0)
  case 54
    self.TR_Reveal(0)
  case 55
    self.TR_RetryDownload()
    
  case 60
    self.TR_Message(1)
  case 61
    self.ToolbarContextualMenus(5).open()
  case 62
    self.TR_Remove(1)
  case 63
    self.TR_CleanUp(1)
  case 64
    self.TR_Reveal(1)
  case 65
    self.TR_Upload()
    
  case 70
    self.SE_Message()
  case 71
    self.ToolbarContextualMenus(6).open()
  case 72
    self.SE_Download()
    
  case 80
    self.BR_Message()
  case 81
    self.BR_Download()
    
  case 90
    self.ToolbarContextualMenus(7).open()
  case 91
    self.CommandContextualMenus(0).open()
    
  case 100
    self.RO_Message()
  case 101
    self.ToolbarContextualMenus(8).open()
  case 102
    self.RO_Refresh()
  case 103
    self.CommandContextualMenus(1).open()
  end
End Sub

MainWindow.Toolbars.Open:
Sub Open(Index As Integer)
  select case index
  case 0
    me.setIcon(rb_remove)
    me.caption=s_remove
    me.setPosition(self)
  case 1
    me.setIcon(rb_clean_up)
    me.caption=s_clean_up
    me.setPosition(self.Toolbars(index-1))
  case 2
    me.setIcon(rb_modify)
    me.caption=s_modify
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
  case 3
    me.setIcon(rb_disconnect)
    me.caption=s_disconnect
    me.setPosition(self.Toolbars(index-1))
  case 4
    me.setIcon(rb_connect)
    me.caption=s_connect
    me.setPosition(self.Toolbars(index-1))
    self.RoundedCanvases(0).width=me.left-20
    self.StaticTexts(0).width=me.left-20
    
  case 10
    me.setIcon(rb_operator)
    me.caption=s_operator
    me.hasMenu=true
    me.setPosition(self)
  case 11
    me.setIcon(rb_refresh)
    me.caption=s_refresh
    me.setPosition(self.Toolbars(index-1))
    self.ConsolePopupMenu.left=me.left-130
    self.StaticTexts(1).left=me.left-130
    self.RoundedCanvases(1).width=me.left-150
    self.StaticTexts(2).width=me.left-150
    
  case 20
    me.setIcon(rb_modify)
    me.caption=s_add
    me.setPosition(self)
  case 21
    me.setIcon(rb_refresh)
    me.caption=s_refresh
    me.setPosition(self.Toolbars(index-1))
  case 22
    me.setIcon(rb_join)
    me.caption=s_join
    me.setPosition(self.Toolbars(index-1))
    self.ChannelPopupMenu.left=me.left-130
    self.StaticTexts(3).left=me.left-130
    self.RoundedCanvases(2).width=me.left-150
    self.StaticTexts(4).width=me.left-150
    
  case 30
    me.setIcon(rb_message)
    me.caption=s_message
    me.setPosition(self)
  case 31
    me.setIcon(rb_browse)
    me.caption=s_browse
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
  case 32
    me.setIcon(rb_remove)
    me.caption=s_remove
    me.setPosition(self.Toolbars(index-1))
  case 33
    me.setIcon(rb_clean_up)
    me.caption=s_clean_up
    me.setPosition(self.Toolbars(index-1))
  case 34
    me.setIcon(rb_refresh)
    me.caption=s_refresh
    me.setPosition(self.Toolbars(index-1))
    self.RoundedCanvases(3).width=me.left-20
    self.StaticTexts(5).width=me.left-20
    
  case 40
    me.setIcon(rb_message)
    me.caption=s_message
    me.setPosition(self)
  case 41
    me.setIcon(rb_browse)
    me.caption=s_browse
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
  case 42
    me.setIcon(rb_remove)
    me.caption=s_remove
    me.setPosition(self.Toolbars(index-1))
  case 43
    me.setIcon(rb_modify)
    me.caption=s_add
    me.setPosition(self.Toolbars(index-1))
    self.RoundedCanvases(4).width=me.left-20
    self.StaticTexts(6).width=me.left-20
    
  case 50
    me.setIcon(rb_message)
    me.caption=s_message
    me.setPosition(self)
  case 51
    me.setIcon(rb_browse)
    me.caption=s_browse
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
  case 52
    me.setIcon(rb_remove)
    me.caption=s_remove
    me.setPosition(self.Toolbars(index-1))
  case 53
    me.setIcon(rb_clean_up)
    me.caption=s_clean_up
    me.setPosition(self.Toolbars(index-1))
  case 54
    me.setIcon(rb_reveal)
    me.caption=s_reveal
    me.setPosition(self.Toolbars(index-1))
  case 55
    me.setIcon(rb_retry)
    me.caption=s_retry
    me.setPosition(self.Toolbars(index-1))
    self.RoundedCanvases(5).width=me.left-20
    self.StaticTexts(7).width=me.left-20
    
  case 60
    me.setIcon(rb_message)
    me.caption=s_message
    me.setPosition(self)
  case 61
    me.setIcon(rb_browse)
    me.caption=s_browse
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
  case 62
    me.setIcon(rb_remove)
    me.caption=s_remove
    me.setPosition(self.Toolbars(index-1))
  case 63
    me.setIcon(rb_clean_up)
    me.caption=s_clean_up
    me.setPosition(self.Toolbars(index-1))
  case 64
    me.setIcon(rb_reveal)
    me.caption=s_reveal
    me.setPosition(self.Toolbars(index-1))
  case 65
    me.setIcon(rb_retry)
    me.caption=s_upload
    me.setPosition(self.Toolbars(index-1))
    self.RoundedCanvases(6).width=me.left-20
    self.StaticTexts(8).width=me.left-20
    
  case 70
    me.setIcon(rb_message)
    me.caption=s_message
    me.setPosition(self)
  case 71
    me.setIcon(rb_browse)
    me.caption=s_browse
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
  case 72
    me.setIcon(rb_download)
    me.caption=s_download
    me.setPosition(self.Toolbars(index-1))
    self.SearchPopupMenus(1).left=me.left-90
    self.SearchSliders(0).left=me.left-90
    self.SearchSliders(1).left=me.left-90
    self.SearchSliders(2).left=me.left-90
    self.SearchPopupMenus(2).left=me.left-90
    self.SearchEditField.left=me.left-90
    self.SearchPopupMenus(0).left=me.left-180
    self.StaticTexts(10).left=me.left-180
    self.RoundedCanvases(7).width=me.left-200
    self.StaticTexts(9).width=me.left-200
    
  case 80
    me.setIcon(rb_message)
    me.caption=s_message
    me.setPosition(self)
  case 81
    me.setIcon(rb_download)
    me.caption=s_download
    me.setPosition(self.Toolbars(index-1))
    self.BrowsePopupMenus(1).left=me.left-90
    self.BrowseSliders(0).left=me.left-90
    self.BrowseSliders(1).left=me.left-90
    self.BrowseEditField.left=me.left-90
    self.BrowsePopupMenus(0).left=me.left-180
    self.StaticTexts(12).left=me.left-180
    self.RoundedCanvases(8).width=me.left-200
    self.StaticTexts(11).width=me.left-200
    
  case 90
    me.setIcon(rb_browse)
    me.caption=s_browse
    me.hasMenu=true
    me.setPosition(self)
  case 91
    me.setIcon(rb_operator)
    me.caption=s_command
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
    self.RoundedCanvases(9).width=me.left-20
    self.StaticTexts(13).width=me.left-20
    
  case 100
    me.setIcon(rb_message)
    me.caption=s_message
    me.setPosition(self)
  case 101
    me.setIcon(rb_browse)
    me.caption=s_browse
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
  case 102
    me.setIcon(rb_refresh)
    me.caption=s_refresh
    me.setPosition(self.Toolbars(index-1))
  case 103
    me.setIcon(rb_operator)
    me.caption=s_command
    me.hasMenu=true
    me.setPosition(self.Toolbars(index-1))
    self.RoundedCanvases(10).width=me.left-20
    self.StaticTexts(14).width=me.left-20
    
  end
End Sub

MainWindow.ToolbarContextualMenus.Open:
Sub Open(Index As Integer)
  select case index
  case 0
    me.addrow(s_add)
    me.addrow(s_modify)
    me.addseparator()
    me.addrow(s_import)
    me.addrow(s_export)
  case 1
    me.addrow(s_server_operator)
    me.addrow(s_user_operator)
    me.addrow(s_channel_operator)
  else
    me.addrow(s_direct_browse)
    me.addrow(s_browse)
  end
End Sub

MainWindow.ToolbarContextualMenus.Action:
Sub Action(Index As Integer, item As String)
  select case index
  case 0
    select case item
    case s_add
      self.NE_Modify(false)
    case s_modify
      self.NE_Modify(true)
    case s_import
      self.NE_Import()
    case s_export
      self.NE_Export()
    end
  case 1
    select case item
    case s_server_operator
      self.CO_Operator(0)
    case s_user_operator
      self.CO_Operator(1)
    case s_channel_operator
      self.CO_Operator(2)
    end
  case 2
    select case item
    case s_direct_browse
      self.WH_Browse(true)
    case s_browse
      self.WH_Browse(false)
    end
  case 3
    select case item
    case s_direct_browse
      self.US_Browse(true)
    case s_browse
      self.US_Browse(false)
    end
  case 4
    select case item
    case s_direct_browse
      self.TR_Browse(true,0)
    case s_browse
      self.TR_Browse(false,0)
    end
  case 5
    select case item
    case s_direct_browse
      self.TR_Browse(true,1)
    case s_browse
      self.TR_Browse(false,1)
    end
  case 6
    select case item
    case s_direct_browse
      self.SE_Browse(true)
    case s_browse
      self.SE_Browse(false)
    end
  case 7
    select case item
    case s_direct_browse
      self.ME_Browse(true)
    case s_browse
      self.ME_Browse(false)
    end
  case 8
    select case item
    case s_direct_browse
      self.RO_Browse(true)
    case s_browse
      self.RO_Browse(false)
    end
  end
End Sub

MainWindow.NetworkListBox.SortColumn:
Function SortColumn(column as integer) As boolean
  if column=NE_Icon then
    me.columnSortDirection(column)=-1
  end
End Function

MainWindow.NetworkListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  select case ascb(key)
  case 13
    self.NE_Connect()
  case 27
    self.NE_CleanUp()
  case 8
    self.NE_Remove()
  end
End Function

MainWindow.NetworkListBox.DropObject:
Sub DropObject(obj As DragItem)
  do
    if obj.FolderItemAvailable then
      self.NE_ImportWSX(obj.FolderItem)
    end
  loop until obj.nextItem=false
End Sub

MainWindow.NetworkListBox.CMMClicked:
Sub CMMClicked()
  self.NetworkContextualMenu.open()
End Sub

MainWindow.NetworkListBox.Close:
Sub Close()
  writeColumn(me,c_network_window)
  self.NE_WriteItem()
End Sub

MainWindow.NetworkListBox.CellTextPaint:
Function CellTextPaint(g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  if column=NE_IP then
    if me.cellCheck(row,NE_Redirect) then
      g.drawstring me.cell(row,NE_IP)+chrb(58)+me.cell(row,NE_Port)+" Re",x,y
    else
      g.drawstring me.cell(row,NE_IP)+chrb(58)+me.cell(row,NE_Port),x,y
    end
    return true
  elseif column=NE_Network and lenb(me.cell(row,NE_URL))<>0 then
    g.underline=true
  end
End Function

MainWindow.NetworkListBox.Open:
Sub Open()
  dim temp as string
  dim aIndex as integer
  dim tis as TextInputStream
  me.height=me.height+16
  me.acceptFileDrop("special/any")
  me.isColored=true
  getColumn(me,c_network_window)
  me.columnType(NE_Redirect)=2
  me.column(NE_ID).widthExpression="0"
  me.column(NE_Nick).widthExpression="0"
  me.column(NE_Password).widthExpression="0"
  me.column(NE_Port).widthExpression="0"
  me.column(NE_URL).widthExpression="0"
  me.column(NE_Icon).userResizable=false
  me.column(NE_ID).userResizable=false
  me.column(NE_Nick).userResizable=false
  me.column(NE_Password).userResizable=false
  me.column(NE_Port).userResizable=false
  me.column(NE_URL).userResizable=false
  self.targetListIndex=-1
  
  if NetworkWindowLock.enter() then
    tis=PreferencesFolder.Child(c_2get).Child(c_hosts).openAsTextFile()
    do until tis.eof
      temp=""
      temp=tis.readline()
      me.addrow("")
      aIndex=me.lastIndex
      me.cell(aIndex,NE_Network)=getDefine(nthfield(temp,chrb(9),1))
      me.cell(aIndex,NE_IP)=getDefine(nthfield(temp,chrb(9),2))
      me.cell(aIndex,NE_Port)=getDefine(nthfield(temp,chrb(9),3))
      me.cellCheck(aIndex,NE_Redirect)=(strcomp(nthfield(temp,chrb(9),4),"true",0)=0)
      me.cell(aIndex,NE_Nick)=getDefine(nthfield(temp,chrb(9),5))
      me.cell(aIndex,NE_Password)=getDefine(nthfield(temp,chrb(9),6))
      me.cell(aIndex,NE_URL)=getDefine(nthfield(temp,chrb(9),7))
    loop
    tis.close()
    NetworkWindowLock.leave()
  end
exception
  if tis<>nil then
    tis.close()
  end
  NetworkWindowLock.leave()
End Sub

MainWindow.NetworkListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  if column=NE_IP then
    if me.cellCheck(row,NE_Redirect) then
      me.helpTag=me.cell(row,NE_IP)+chrb(58)+me.cell(row,NE_Port)+getDefine(" Re")
    else
      me.helpTag=me.cell(row,NE_IP)+chrb(58)+me.cell(row,NE_Port)
    end
  else
    me.helpTag=me.cell(row,column)
  end
  me.balloonHelp=me.helpTag
  if column=NE_Network and lenb(me.cell(row,NE_URL))<>0 then
    me.listIndex=row
    me.refresh()
    showURL me.cell(row,NE_URL)
    return true
  end
End Function

MainWindow.NetworkContextualMenu.Open:
Sub Open()
  me.addrow(s_connect)
  me.addrow(s_disconnect)
  me.addseparator()
  me.addrow(s_add)
  me.addrow(s_modify)
  me.addrow(s_import)
  me.addrow(s_export)
  me.addseparator()
  me.addrow(s_clean_up)
  me.addseparator()
  me.addrow(s_remove)
End Sub

MainWindow.NetworkContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_connect
    self.NE_Connect()
  case s_disconnect
    self.NE_Disconnect()
  case s_add
    self.NE_Modify(false)
  case s_modify
    self.NE_Modify(true)
  case s_import
    self.NE_Import()
  case s_export
    self.NE_Export()
  case s_clean_up
    self.NE_CleanUp()
  case s_remove
    self.NE_Remove()
  end
End Sub

MainWindow.PagePanels.Change:
Sub Change()
  dim aArray as integer
  self.EditFields(me.value).text=""
  select case me.value
  case 0
    self.setTitle(s_network)
    self.EditFields(me.value).setFocus()
  case 1
    self.setTitle(s_console)
    self.EditFields(me.value).setFocus()
  case 2
    self.setTitle(s_channel)
    self.EditFields(me.value).setFocus()
  case 3
    self.setTitle(s_whois)
    self.EditFields(me.value).setFocus()
  case 4
    self.setTitle(s_user_list)
    self.EditFields(me.value).setFocus()
  case 5
    self.setTitle(s_downloads)
    self.EditFields(me.value).setFocus()
  case 6
    self.setTitle(s_uploads)
    self.EditFields(me.value).setFocus()
  case 7
    if SearchWindowLock.enter() then
      if gSearchWindowID<>0 then
        aArray=SE_SearchID2Array(gSearchWindowID)
        if aArray<>c_error then
          self.setTitle(SearchWindow(aArray).title)
        end
      else
        self.setTitle(s_new_search)
      end
      if self.isChanged then
        self.SearchEditField.text=""
      end
      self.EditFields(me.value).setFocus()
      SearchWindowLock.leave()
    end
    if self.isChanged then
      self.SE_Filter()
    end
  case 8
    if BrowseWindowLock.enter() then
      aArray=BR_SearchID2Array(gBrowseWindowID)
      if aArray<>c_error then
        self.setTitle(BrowseWindow(aArray).title)
      end
      if self.isChanged then
        self.FolderListBox.expanded(0)=false
        self.FolderListBox.listIndex=-1
        self.BrowseEditField.text=""
      end
      self.EditFields(me.value).setFocus()
      BrowseWindowLock.leave()
    end
    if self.isChanged then
      self.BR_Filter()
    end
  case 9
    if MessageWindowLock.enter() then
      aArray=ME_SearchID2Array(gMessageWindowID)
      if aArray<>c_error then
        self.setTitle(MessageWindow(aArray).title)
      end
      self.CommandEditFields(0).text=""
      self.CommandEditFields(0).setFocus()
      MessageWindowLock.leave()
    end
    if self.isChanged then
      self.ME_Filter()
    end
  case 10
    if RoomWindowLock.enter() then
      aArray=RO_SearchID2Array(gRoomWindowID)
      if aArray<>c_error then
        self.setTitle(RoomWindow(aArray).title)
      end
      self.CommandEditFields(1).text=""
      self.CommandEditFields(1).setFocus()
      RoomWindowLock.leave()
    end
    if self.isChanged then
      self.RO_Filter()
    end
  end
  self.isChanged=false
End Sub

MainWindow.ConsoleContextualMenu.Open:
Sub Open()
  me.addrow(s_refresh)
  me.addseparator()
  me.addrow(s_server_operator)
  me.addrow(s_user_operator)
  me.addrow(s_channel_operator)
End Sub

MainWindow.ConsoleContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_refresh
    self.CO_Refresh()
  case s_server_operator
    self.CO_Operator(0)
  case s_user_operator
    self.CO_Operator(1)
  case s_channel_operator
    self.CO_Operator(2)
  end
End Sub

MainWindow.ChannelListBox.SortColumn:
Function SortColumn(column as integer) As boolean
  if column=CH_Count then
    if ChannelWindowLock.signal() then
      me.sortNumber(column)
      ChannelWindowLock.release()
    end
    return true
  end
End Function

MainWindow.ChannelListBox.Open:
Sub Open()
  me.height=me.height+16
  me.isColored=true
  getColumn(me,c_channel_window)
  me.columnAlignment(CH_Count)=3
  me.column(CH_ServerID).widthExpression="0"
  me.column(CH_ServerID).userResizable=false
End Sub

MainWindow.ChannelListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  if ascb(key)=13 then
    self.CH_Join()
  end
End Function

MainWindow.ChannelListBox.CMMClicked:
Sub CMMClicked()
  self.ChannelContextualMenu.open()
End Sub

MainWindow.ChannelListBox.Close:
Sub Close()
  writeColumn(me,c_channel_window)
End Sub

MainWindow.ChannelListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  me.helpTag=me.cell(row,column)
  me.balloonHelp=me.helpTag
End Function

MainWindow.ChannelContextualMenu.Open:
Sub Open()
  me.addrow(s_join)
  me.addseparator()
  me.addrow(s_refresh)
  me.addrow(s_add)
End Sub

MainWindow.ChannelContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_join
    self.CH_Join()
  case s_refresh
    self.CH_Refresh()
  case s_add
    self.CH_Add()
  end
End Sub

MainWindow.WhoisListBox.CellTextPaint:
Function CellTextPaint(g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  if me.selected(row)=false and val(me.cell(row,WH_ServerID))=0 then
    g.foreColor=rgb(&h55,&h55,&h55)
  end
End Function

MainWindow.WhoisListBox.Close:
Sub Close()
  writeColumn(me,c_whois_window)
End Sub

MainWindow.WhoisListBox.SortColumn:
Function SortColumn(column as integer) As boolean
  select case column
  case WH_Icon
    me.columnSortDirection(column)=-1
  case WH_Shares
    if WhoisWindowLock.signal() then
      me.sortNumber(column)
      WhoisWindowLock.release()
    end
    return true
  end
End Function

MainWindow.WhoisListBox.Open:
Sub Open()
  me.height=me.height+16
  me.isColored=true
  getColumn(me,c_whois_window)
  me.columnAlignment(WH_Uptime)=3
  me.columnAlignment(WH_Status)=2
  me.columnAlignment(WH_Shares)=3
  me.columnAlignment(WH_Transfer)=2
  me.columnAlignment(WH_Speed)=2
  me.column(WH_ServerID).widthExpression="0"
  me.column(WH_Icon).userResizable=false
  me.column(WH_ServerID).userResizable=false
End Sub

MainWindow.WhoisListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  select case ascb(key)
  case 13
    self.WH_Refresh()
  case 27
    self.WH_CleanUp()
  case 32
    self.WH_Message()
  end
End Function

MainWindow.WhoisListBox.CMMClicked:
Sub CMMClicked()
  self.WhoisContextualMenu.open()
End Sub

MainWindow.WhoisListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  dim i,j as integer
  dim temp as string
  self.EditFields(3).replaceText(me.cell(row,WH_Nick))
  temp=me.cell(row,column)
  me.helpTag=temp
  me.balloonHelp=me.helpTag
  
  if column=WH_Channel then
    j=countfields(temp,chrb(32))-1
    if j<>0 then
      for i=1 to j
        self.WhoisChannelContextualMenu.addrow(getDefine(nthfield(temp,chrb(32),i)))
      next
      me.listIndex=row
      me.refresh()
      self.WhoisChannelContextualMenu.open()
      self.WhoisChannelContextualMenu.deleteAllRows()
      return true
    end
  end
End Function

MainWindow.WhoisChannelContextualMenu.Action:
Sub Action(item As String)
  'join channel
  OS_Command(val(self.WhoisListBox.cell(self.WhoisListBox.listIndex,WH_ServerID)),400,item)
End Sub

MainWindow.WhoisContextualMenu.Open:
Sub Open()
  me.addrow(s_refresh)
  me.addseparator()
  me.addrow(s_clean_up)
  me.addseparator()
  me.addrow(s_remove)
  me.addseparator()
  me.addrow(s_direct_browse)
  me.addrow(s_browse)
  me.addrow(s_message)
  me.addseparator()
  me.addrow(s_favorite)
  me.addrow(s_ignore)
End Sub

MainWindow.WhoisContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_refresh
    self.WH_Refresh()
  case s_clean_up
    self.WH_CleanUp()
  case s_remove
    self.WH_Remove()
  case s_direct_browse
    self.WH_Browse(true)
  case s_browse
    self.WH_Browse(false)
  case s_message
    self.WH_Message()
  case s_favorite
    self.WH_UserList(true)
  case s_ignore
    self.WH_UserList(false)
  end
End Sub

MainWindow.UserListListBox.CellTextPaint:
Function CellTextPaint(g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  if me.selected(row)=false and val(me.cell(row,US_ServerID))=0 then
    g.foreColor=rgb(&h55,&h55,&h55)
  end
End Function

MainWindow.UserListListBox.SortColumn:
Function SortColumn(column as integer) As boolean
  if column=US_Icon then
    me.columnSortDirection(column)=-1
  end
End Function

MainWindow.UserListListBox.Open:
Sub Open()
  dim aIndex as integer
  dim temp as string
  dim tis as TextInputStream
  me.height=me.height+16
  me.isColored=true
  getColumn(me,c_user_list_window)
  me.columnType(US_Comment)=3
  me.column(US_ServerID).widthExpression="0"
  me.column(US_Icon).userResizable=false
  me.column(US_ServerID).userResizable=false
  
  if UserListWindowLock.enter() then
    tis=PreferencesFolder.child(c_2get).Child(c_users).openAsTextFile()
    do until tis.eof
      temp=""
      temp=tis.readline()
      me.addrow("")
      aIndex=me.lastIndex
      me.cell(aIndex,US_Nick)=getDefine(nthfield(temp,chrb(9),1))
      me.cell(aIndex,US_Network)=getDefine(nthfield(temp,chrb(9),2))
      me.cell(aIndex,US_Comment)=getDefine(nthfield(temp,chrb(9),3))
      if strcomp(nthfield(temp,chrb(9),4),"true",0)=0 then
        me.cell(aIndex,US_Icon)=" "
      else
        me.rowPicture(aIndex)=gPicture(r_ignore)
      end
    loop 
    tis.close()
    UserListWindowLock.leave()
  end
exception
  if tis<>nil then
    tis.close()
  end
  UserListWindowLock.leave()
End Sub

MainWindow.UserListListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  select case ascb(key)
  case 8
    self.US_Remove()
  case 32
    self.US_Message()
  end
End Function

MainWindow.UserListListBox.CMMClicked:
Sub CMMClicked()
  self.UserListContextualMenu.open()
End Sub

MainWindow.UserListListBox.Close:
Sub Close()
  writeColumn(me,c_user_list_window)
  self.US_WriteItem()
End Sub

MainWindow.UserListListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  self.EditFields(4).replaceText(me.cell(row,US_Nick))
  me.helpTag=me.cell(row,column)
  me.balloonHelp=me.helpTag
End Function

MainWindow.UserListContextualMenu.Open:
Sub Open()
  me.addrow(s_add)
  me.addseparator()
  me.addrow(s_remove)
  me.addseparator()
  me.addrow(s_direct_browse)
  me.addrow(s_browse)
  me.addrow(s_message)
  me.addrow(s_whois)
End Sub

MainWindow.UserListContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_add
    AddUserWindow.showWithin(self)
  case s_remove
    self.US_Remove()
  case s_direct_browse
    self.US_Browse(true)
  case s_browse
    self.US_Browse(false)
  case s_message
    self.US_Message()
  case s_whois
    self.US_Whois()
  end
End Sub

MainWindow.TransferListBoxes.SortColumn:
Function SortColumn(Index As Integer, column as integer) As boolean
  if column=TR_Rate then
    if TransferWindowLock.signal() then
      me.columnSortDirection(TR_Size)=me.columnSortDirection(column)
      me.sortNumber(TR_Size)
      TransferWindowLock.release()
    end
    return true
  end
End Function

MainWindow.TransferListBoxes.Open:
Sub Open(Index As Integer)
  dim tis as TextInputStream
  dim aIndex as integer
  dim temp,targetShortName as string
  me.height=me.height+16
  me.isColored=true
  getColumn(me,c_transfer_window)
  me.columnType(TR_Retry)=2
  me.columnType(TR_WinMX)=2
  me.columnAlignment(TR_Rate)=2
  me.columnAlignment(TR_RemainTime)=2
  me.column(TR_Icon).widthExpression="38"
  me.column(TR_FileName).widthExpression="0"
  me.column(TR_FolderName).widthExpression="0"
  me.column(TR_ID).widthExpression="0"
  me.column(TR_Network).widthExpression="0"
  me.column(TR_ShortSize).widthExpression="0"
  me.column(TR_ServerID).widthExpression="0"
  me.column(TR_Status).widthExpression="0"
  me.column(TR_Size).widthExpression="0"
  me.column(TR_Timer).widthExpression="0"
  me.column(TR_TransferedSize).widthExpression="0"
  me.column(TR_Flag).widthExpression="0"
  me.column(TR_Icon).userResizable=false
  me.column(TR_FileName).userResizable=false
  me.column(TR_FolderName).userResizable=false
  me.column(TR_ID).userResizable=false
  me.column(TR_Network).userResizable=false
  me.column(TR_ShortSize).userResizable=false
  me.column(TR_ServerID).userResizable=false
  me.column(TR_Status).userResizable=false
  me.column(TR_Size).userResizable=false
  me.column(TR_Timer).userResizable=false
  me.column(TR_TransferedSize).userResizable=false
  me.column(TR_Flag).userResizable=false
  
  if index=0 then
    if TransferWindowLock.enter() then
      tis=PreferencesFolder.Child(c_2get).Child(c_resumes).openAsTextFile()
      do until tis.eof
        temp=""
        temp=tis.readline()
        targetShortName=""
        targetShortName=getDefine(nthfield(temp,chrb(9),2))
        me.addrow(getMediaType(rightb(targetShortName,inbackStrb(1,targetShortName,"."))))
        aIndex=me.lastIndex
        me.cell(aIndex,TR_Status)=s_offline
        me.cell(aIndex,TR_ShortName)=targetShortName
        me.cell(aIndex,TR_Network)=s_offline
        me.cell(aIndex,TR_Nick)=getDefine(nthfield(temp,chrb(9),6))
        me.cell(aIndex,TR_Size)=getDefine(nthfield(temp,chrb(9),3))
        me.cell(aIndex,TR_TransferedSize)=getDefine(nthfield(temp,chrb(9),5))
        me.cell(aIndex,TR_ShortSize)=getDefine(nthfield(temp,chrb(9),4))
        me.cell(aIndex,TR_Rate)=""
        me.cell(aIndex,TR_FileName)=getDefine(nthfield(temp,chrb(9),1))
        me.cell(aIndex,TR_FolderName)=getDefine(nthfield(temp,chrb(9),7))
      loop
      tis.close()
      TransferWindowLock.leave()
    end
  end
exception
  if tis<>nil then
    tis.close()
  end
  TransferWindowLock.leave()
End Sub

MainWindow.TransferListBoxes.KeyDown:
Function KeyDown(Index As Integer, key As String) As Boolean
  select case ascb(key)
  case 13
    if index=0 then
      self.TR_RetryDownload()
    else
      self.TR_Upload()
    end
  case 27
    self.TR_CleanUp(index)
  case 8
    self.TR_Remove(index)
  case 32
    self.TR_Message(index)
  end
End Function

MainWindow.TransferListBoxes.CMMClicked:
Sub CMMClicked(Index As Integer)
  self.TransferContextualMenus(index).open()
End Sub

MainWindow.TransferListBoxes.Close:
Sub Close(Index As Integer)
  if index=0 then
    writeColumn(me,c_transfer_window)
    self.TR_WriteItem()
  end
End Sub

MainWindow.TransferListBoxes.CellTextPaint:
Function CellTextPaint(Index As Integer, g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  select case column
  case TR_ShortName
    g.drawstring me.cell(row,column),x,y-14
    if me.selected(row)=false then
      g.foreColor=rgb(&h55,&h55,&h55)
    end
    g.drawstring me.cell(row,TR_Status),x,y
  case TR_Nick
    g.drawstring me.cell(row,column),x,y-14
    if me.selected(row)=false then
      g.foreColor=rgb(&h55,&h55,&h55)
    end
    g.drawstring me.cell(row,TR_Network),x,y
  case TR_Rate
    g.drawstring me.cell(row,column),x,y-14
    if me.selected(row)=false then
      g.foreColor=rgb(&h55,&h55,&h55)
    end
    g.drawstring me.cell(row,TR_ShortSize),(g.width-g.stringWidth(me.cell(row,TR_ShortSize)))/2,y
  case TR_RemainTime
    if lenb(me.cell(row,column))<>0 then
      g.drawstring me.cell(row,column),x,y-14
      if me.selected(row)=false then
        g.foreColor=rgb(&h55,&h55,&h55)
      end
      g.drawstring s_remaining,(g.width-g.stringWidth(s_remaining))/2,y
    end
  end
  return true
End Function

MainWindow.TransferListBoxes.CellClick:
Function CellClick(Index As Integer, row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  dim i,j as integer
  dim aNick as string
  self.EditFields(5+index).replaceText(me.cell(row,TR_ShortName))
  if column=TR_Rate then
    me.helpTag=me.cell(row,TR_TransferedSize)+chrb(47)+me.cell(row,TR_Size)
  else
    me.helpTag=me.cell(row,column)
  end
  me.balloonHelp=me.helpTag
  
  if column=TR_Nick then
    aNick=me.cell(row,column)
    self.TransferNetworkContextualMenus(index).addrow(s_lookup)
    self.TransferNetworkContextualMenus(index).addseparator()
    
    if WhoisWindowLock.enter() then
      j=self.WhoisListBox.listCount-1
      for i=0 to j
        if val(self.WhoisListBox.cell(i,WH_ServerID))<>0 and strcomp(self.WhoisListBox.cell(i,WH_Nick),aNick,0)=0 then
          self.TransferNetworkContextualMenus(index).addrow(self.WhoisListBox.cell(i,WH_Network))
        end
      next
      WhoisWindowLock.leave()
    end
    
    me.listIndex=row
    me.refresh()
    self.TransferNetworkContextualMenus(index).open()
    self.TransferNetworkContextualMenus(index).deleteAllRows()
    return true
  end
End Function

MainWindow.TransferListBoxes.CellBackgroundPaint:
Function CellBackgroundPaint(Index As Integer, g as graphics,row as integer,column as integer) As Boolean
  select case column
  case TR_Icon
    g.drawPicture gMediaPicture(val(me.cell(row,column))+5),3,2
  case TR_Rate
    if me.selected(row) then
      if me.active then
        g.foreColor=rgb(&h30,&h64,&hB7)
      else
        g.foreColor=rgb(&hB4,&hB4,&hB4)
      end
    else
      if index=0 then
        g.foreColor=rgb(189,222,255)
      else
        g.foreColor=rgb(255,189,222)
      end
    end
    g.fillRect 0,0,g.width*(val(me.cell(row,TR_TransferedSize))/val(me.cell(row,TR_Size))),g.height
  end
End Function

MainWindow.TransferContextualMenus.Open:
Sub Open(Index As Integer)
  select case index
  case 0
    me.addrow(s_open)
    me.addrow(s_reveal_in_finder)
    me.addSeparator()
    me.addrow(s_pause)
    me.addSeparator()
    me.addrow(s_retry)
    me.addrow(s_retry_automatically)
    me.addrow(s_resume_from_)
    me.addSeparator()
    me.addrow(s_clean_up)
    me.addSeparator()
    me.addrow(s_remove)
    me.addSeparator()
    me.addrow(s_direct_browse)
    me.addrow(s_browse)
    me.addrow(s_message)
    me.addrow(s_whois)
    me.addSeparator()
    me.addrow(s_favorite)
    me.addrow(s_ignore)
  case 1
    me.addrow(s_open)
    me.addrow(s_reveal_in_finder)
    me.addSeparator()
    me.addrow(s_upload)
    me.addSeparator()
    me.addrow(s_clean_up)
    me.addSeparator()
    me.addrow(s_remove)
    me.addSeparator()
    me.addrow(s_direct_browse)
    me.addrow(s_browse)
    me.addrow(s_message)
    me.addrow(s_whois)
    me.addSeparator()
    me.addrow(s_favorite)
    me.addrow(s_ignore)
  end
End Sub

MainWindow.TransferContextualMenus.Action:
Sub Action(Index As Integer, item As String)
  select case item
  case s_open
    self.TR_Open(index)
  case s_reveal_in_finder
    self.TR_Reveal(index)
  case s_pause
    self.TR_Pause()
  case s_retry
    self.TR_RetryDownload()
  case s_retry_automatically
    self.TR_AutoRetry()
  case s_resume_from_
    self.TR_Resume()
  case s_upload
    self.TR_Upload()
  case s_clean_up
    self.TR_CleanUp(index)
  case s_remove
    self.TR_Remove(index)
  case s_direct_browse
    self.TR_Browse(true,index)
  case s_browse
    self.TR_Browse(false,index)
  case s_message
    self.TR_Message(index)
  case s_whois
    self.TR_Whois(index)
  case s_favorite
    self.TR_UserList(true,index)
  case s_ignore
    self.TR_UserList(false,index)
  end
End Sub

MainWindow.ConsoleEditField.CMMClicked:
Sub CMMClicked()
  if gConsoleServerID<>0 then
    self.ConsoleContextualMenu.open()
  end
End Sub

MainWindow.ConsoleEditField.Open:
Sub Open()
  me.height=me.height+16
  me.textFont=iTextFont
  me.textSize=iTextSize
  me.textColor=iColors(co_default)
  me.backColor=iColors(co_background)
End Sub

MainWindow.ScrollBars.Open:
Sub Open()
  me.top=me.top+16
End Sub

MainWindow.SearchContextualMenu.Open:
Sub Open()
  me.addrow(s_download)
  me.addrow(s_resume_from_)
  me.addseparator()
  me.addrow(s_direct_browse)
  me.addrow(s_browse)
  me.addrow(s_message)
  me.addrow(s_whois)
  me.addseparator()
  me.addrow(s_favorite)
  me.addrow(s_ignore)
  me.addseparator()
  me.addrow(s_switch_view)
End Sub

MainWindow.SearchContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_download
    self.SE_Download()
  case s_resume_from_
    self.SE_Resume()
  case s_direct_browse
    self.SE_Browse(true)
  case s_browse
    self.SE_Browse(false)
  case s_message
    self.SE_Message()
  case s_whois
    self.SE_Whois()
  case s_favorite
    self.SE_UserList(true)
  case s_ignore
    self.SE_UserList(false)
  case s_switch_view
    self.SE_SwitchView()
  end
End Sub

MainWindow.SearchPopupMenus.Open:
Sub Open(Index As Integer)
  dim i,j as integer
  me.top=self.SearchPopupMenus(0).top
  select case index
  case 0
    me.addrow(s_media)
    me.addrow(s_speed)
    me.addrow(s_size)
    me.addrow(s_bitrate)
    me.addrow(s_network)
    me.addseparator()
    me.addrow(s_keyword)
    me.listindex=0
    
  case 1
    me.addrow(s_all)
    me.addrow(s_music)
    me.addrow(s_pictures)
    me.addrow(s_movies)
    me.addrow(s_text)
    me.addrow(s_files)
    me.addrow(s_mp3_aac)
    me.listindex=0
    
  case 2
    me.addrow(s_all)
    me.rowTag(0)=0
    
    if OpenNapSocketLock.enter() then
      j=ubound(OpenNapSocket)
      for i=1 to j
        if OpenNapSocket(i).isLoggedIn then
          me.addrow(OpenNapSocket(i).network)
          me.rowTag(me.listCount-1)=OpenNapSocket(i).ID
        end
      next
      OpenNapSocketLock.leave()
    end
    
    me.listindex=0
    
  end
End Sub

MainWindow.SearchPopupMenus.Change:
Sub Change(Index As Integer)
  select case index
  case 0
    self.SearchSliders(0).visible=false
    self.SearchSliders(1).visible=false
    self.SearchSliders(2).visible=false
    self.SearchPopupMenus(1).visible=false
    self.SearchPopupMenus(2).visible=false
    self.SearchEditField.visible=false
    select case me.listindex
    case 0'media
      self.SearchPopupMenus(1).visible=true
    case 1'speed
      self.SearchSliders(0).visible=true
    case 2'size
      self.SearchSliders(1).visible=true
    case 3'bitrate
      self.SearchSliders(2).visible=true
    case 4'network
      self.SearchPopupMenus(2).visible=true
    case 6'keyword
      self.SearchEditField.visible=true
      self.SearchEditField.setFocus()
    end
  case 1
    self.SE_Filter()
  case 2
    self.SE_Filter()
  end
End Sub

MainWindow.SearchEditField.TextChange:
Sub TextChange()
  self.SE_Filter()
End Sub

MainWindow.SearchEditField.Open:
Sub Open()
  me.top=self.SearchPopupMenus(0).top
End Sub

MainWindow.SearchSliders.ValueChanged:
Sub ValueChanged(Index As Integer)
  self.SE_Filter()
End Sub

MainWindow.SearchSliders.Open:
Sub Open(Index As Integer)
  me.top=self.SearchPopupMenus(0).top
End Sub

MainWindow.SearchListBox.SortColumn:
Function SortColumn(column as integer) As boolean
  select case column
  case SE_ShortSize
    if SearchWindowLock.signal() then
      me.columnSortDirection(SE_Size)=me.columnSortDirection(column)
      me.sortNumber(SE_Size)
      SearchWindowLock.release()
    end
    return true
  case SE_Bitrate
    if SearchWindowLock.signal() then
      me.sortNumber(SE_Bitrate)
      SearchWindowLock.release()
    end
    return true
  end
End Function

MainWindow.SearchListBox.Open:
Sub Open()
  me.height=me.height+16
  me.isColored=true
  getColumn(me,c_search_window)
  me.columnAlignment(SE_Speed)=2
  me.columnAlignment(SE_ShortSize)=3
  me.columnAlignment(SE_Bitrate)=3
  me.column(SE_Size).widthExpression="0"
  me.column(SE_ServerID).widthExpression="0"
  me.column(SE_Icon).userResizable=false
  me.column(SE_ServerID).userResizable=false
  me.column(SE_Size).userResizable=false
End Sub

MainWindow.SearchListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  select case ascb(key)
  case 13
    self.SE_Download()
  case 32
    self.SE_Message()
  end
End Function

MainWindow.SearchListBox.CMMClicked:
Sub CMMClicked()
  self.SearchContextualMenu.open()
End Sub

MainWindow.SearchListBox.Close:
Sub Close()
  writeColumn(me,c_search_window)
End Sub

MainWindow.SearchListBox.CellTextPaint:
Function CellTextPaint(g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  return column=SE_Icon or (column=SE_BitRate and val(me.cell(row,column))<=24)
End Function

MainWindow.SearchListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  self.EditFields(7).replaceText(me.cell(row,SE_ShortName))
  select case column
  case SE_ShortSize
    me.helpTag=me.cell(row,SE_Size)
  else
    me.helpTag=me.cell(row,column)
  end
  me.balloonHelp=me.helpTag
End Function

MainWindow.BrowsePopupMenus.Change:
Sub Change(Index As Integer)
  select case index
  case 0
    self.BrowsePopupMenus(1).visible=false
    self.BrowseSliders(0).visible=false
    self.BrowseSliders(1).visible=false
    self.BrowseEditField.visible=false
    select case me.listindex
    case 0
      self.BrowsePopupMenus(1).visible=true
    case 1
      self.BrowseSliders(0).visible=true
    case 2
      self.BrowseSliders(1).visible=true
    case 4
      self.BrowseEditField.visible=true
      self.BrowseEditField.setFocus()
    end
  case 1
    self.BR_Filter()
  end
End Sub

MainWindow.BrowsePopupMenus.Open:
Sub Open(Index As Integer)
  me.top=self.BrowsePopupMenus(0).top
  select case index
  case 0
    me.addrow(s_media)
    me.addrow(s_size)
    me.addrow(s_bitrate)
    me.addSeparator()
    me.addrow(s_keyword)
    me.listindex=0
  case 1
    me.addrow(s_all)
    me.addrow(s_music)
    me.addrow(s_pictures)
    me.addrow(s_movies)
    me.addrow(s_text)
    me.addrow(s_files)
    me.listindex=0
  end
End Sub

MainWindow.BrowseSliders.ValueChanged:
Sub ValueChanged(Index As Integer)
  self.BR_Filter()
End Sub

MainWindow.BrowseSliders.Open:
Sub Open(Index As Integer)
  me.top=self.BrowsePopupMenus(0).top
End Sub

MainWindow.BrowseEditField.TextChange:
Sub TextChange()
  self.BR_Filter()
End Sub

MainWindow.BrowseEditField.Open:
Sub Open()
  me.top=self.BrowsePopupMenus(0).top
End Sub

MainWindow.FolderListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  self.EditFields(8).replaceText(me.cell(row,column))
End Function

MainWindow.FolderListBox.Open:
Sub Open()
  me.height=me.height+16
  me.addfolder("")
  me.rowPicture(0)=gPicture(r_folder)
End Sub

MainWindow.FolderListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  dim aIndex as integer
  aIndex=me.listIndex
  if aIndex<>-1 then
    select case ascb(key)
    case 28
      me.expanded(aIndex)=false
    case 29
      me.expanded(aIndex)=true
    end
  end
End Function

MainWindow.FolderListBox.ExpandRow:
Sub ExpandRow(row As Integer)
  dim i,j,aIndex,aArray,targetFolderCount as integer
  dim targetFolderPath,aFolderPath,aFolderName as string
  
  if BrowseWindowLock.enter() then
    aArray=BR_SearchID2Array(gBrowseWindowID)
    if aArray<>c_error then
      
      targetFolderPath=me.cell(row,BR_FolderPath)
      targetFolderCount=countfields(targetFolderPath,chrb(92))
      j=ubound(BrowseWindow(aArray).BrowseList)
      for i=1 to j
        aFolderPath=""
        aFolderPath=BrowseWindow(aArray).BrowseList(i).folderPath
        
        if targetFolderCount=1 or instrb(1,aFolderPath,targetFolderPath)=1 then
          aFolderName=""
          aFolderName=getDefine(nthfield(aFolderPath,chrb(92),targetFolderCount))
          if lenb(aFolderName)<>0 or countfields(aFolderPath,chrb(92))>targetFolderCount then
            aFolderPath=""
            aFolderPath=targetFolderPath+aFolderName+chrb(92)
            if self.BR_SearchFolderPath2ListIndex(aFolderPath)=c_error then
              me.addFolder(aFolderName)
              aIndex=me.lastIndex
              me.cell(aIndex,BR_FileName)=aFolderPath
              me.rowPicture(aIndex)=gPicture(r_folder)
            end
          end
        end
        
      next
      
    end
    BrowseWindowLock.leave()
  end
End Sub

MainWindow.FolderListBox.CMMClicked:
Sub CMMClicked()
  self.FolderContextualMenu.open()
End Sub

MainWindow.FolderListBox.Change:
Sub Change()
  self.BR_Filter()
End Sub

MainWindow.SplitCanvases.Close:
Sub Close(Index As Integer)
  writeSplitter(index,self.width-me.left)
End Sub

MainWindow.SplitCanvases.MouseUp:
Sub MouseUp(Index As Integer, X As Integer, Y As Integer)
  select case index
  case 0
    self.FolderListBox.width=me.left
    self.BrowseListBox.textSize=1
    self.BrowseListBox.left=me.left+9
    self.BrowseListBox.width=self.width-self.BrowseListBox.left+1
    self.BrowseListBox.textSize=10
  case 1
    self.RoomEditField.width=me.left
    self.RoomListBox.textSize=1
    self.RoomListBox.left=me.left+9
    self.RoomListBox.width=self.width-self.RoomListBox.left+1
    self.RoomListBox.textSize=10
  end
  self.refresh()
End Sub

MainWindow.SplitCanvases.MouseExit:
Sub MouseExit(Index As Integer)
  me.mouseCursor=arrowCursor
End Sub

MainWindow.SplitCanvases.MouseEnter:
Sub MouseEnter(Index As Integer)
  me.mouseCursor=moveCursor
End Sub

MainWindow.SplitCanvases.MouseDrag:
Sub MouseDrag(Index As Integer, X As Integer, Y As Integer)
  select case index
  case 0
    me.left=min(max(me.left-(self.targetX-X),0),self.width-24)
  case 1
    me.left=min(max(me.left-(self.targetX-X),self.width-243),self.width-24)
  end
  self.graphics.drawRect me.left,me.top,me.width,me.height
End Sub

MainWindow.SplitCanvases.MouseDown:
Function MouseDown(Index As Integer, X As Integer, Y As Integer) As Boolean
  self.targetX=X
  self.graphics.foreColor=rgb(100,100,100)
  self.graphics.drawRect me.left,me.top,me.width,me.height
  return true
End Function

MainWindow.SplitCanvases.Open:
Sub Open(Index As Integer)
  select case index
  case 0
    me.left=min(max(self.width-getSplitter(index),0),self.width-24)
    self.FolderListBox.width=me.left
    self.BrowseListBox.left=me.left+9
    self.BrowseListBox.width=self.width-self.BrowseListBox.left+1
  case 1
    me.left=min(max(self.width-getSplitter(index),self.width-243),self.width-24)
    self.RoomEditField.width=me.left
    self.RoomListBox.left=me.left+9
    self.RoomListBox.width=self.width-self.RoomListBox.left+1
  end
  me.height=me.height+16
End Sub

MainWindow.BrowseListBox.SortColumn:
Function SortColumn(column as integer) As boolean
  select case column
  case BR_ShortSize
    if BrowseWindowLock.signal() then
      me.columnSortDirection(BR_Size)=me.columnSortDirection(column)
      me.sortNumber(BR_Size)
      BrowseWindowLock.release()
    end
    return true
  case BR_Bitrate
    if BrowseWindowLock.signal() then
      me.sortNumber(BR_Bitrate)
      BrowseWindowLock.release()
    end
    return true
  end
End Function

MainWindow.BrowseListBox.Open:
Sub Open()
  me.height=me.height+16
  me.isColored=true
  getColumn(me,c_browse_window)
  me.columnAlignment(BR_BitRate)=3
  me.columnAlignment(BR_ShortSize)=3
  me.column(BR_Size).widthExpression="0"
  me.column(BR_Icon).userResizable=false
  me.column(BR_Size).userResizable=false
End Sub

MainWindow.BrowseListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  select case ascb(key)
  case 13
    self.BR_Download()
  case 32
    self.BR_Message()
  end
End Function

MainWindow.BrowseListBox.CMMClicked:
Sub CMMClicked()
  self.BrowseContextualMenu.open()
End Sub

MainWindow.BrowseListBox.Close:
Sub Close()
  writeColumn(me,c_browse_window)
End Sub

MainWindow.BrowseListBox.CellTextPaint:
Function CellTextPaint(g as graphics,row as integer,column as integer,x as integer,y as integer) As Boolean
  return column=BR_Icon or (column=BR_BitRate and val(me.cell(row,column))<=24)
End Function

MainWindow.BrowseListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  self.EditFields(8).replaceText(me.cell(row,BR_ShortName))
  select case column
  case BR_ShortSize
    me.helpTag=me.cell(row,BR_Size)
  else
    me.helpTag=me.cell(row,column)
  end
  me.balloonHelp=me.helpTag
End Function

MainWindow.BrowseContextualMenu.Open:
Sub Open()
  me.addrow(s_download)
  me.addrow(s_resume_from_)
  me.addseparator()
  me.addrow(s_message)
  me.addrow(s_whois)
  me.addseparator()
  me.addrow(s_favorite)
  me.addrow(s_ignore)
  me.addseparator()
  me.addrow(s_switch_view)
End Sub

MainWindow.BrowseContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_download
    self.BR_Download()
  case s_resume_from_
    self.BR_Resume()
  case s_message
    self.BR_Message()
  case s_whois
    self.BR_Whois()
  case s_favorite
    self.BR_UserList(true)
  case s_ignore
    self.BR_UserList(false)
  case s_switch_view
    self.BR_SwitchView()
  end
End Sub

MainWindow.FolderContextualMenu.Open:
Sub Open()
  me.addrow(s_expand_all_folder)
  me.addrow(s_collapse_all_folder)
  me.addseparator()
  me.addrow(s_download_current_folder)
End Sub

MainWindow.FolderContextualMenu.Action:
Sub Action(item As String)
  dim i as integer
  select case item
  case s_expand_all_folder
    for i=0 to self.FolderListBox.listCount-1
      self.FolderListBox.expanded(i)=true
    next
  case s_collapse_all_folder
    self.FolderListBox.expanded(0)=false
  case s_download_current_folder
    if self.FolderListBox.listIndex>0 then
      self.BR_DownloadFolder()
    end
  end
End Sub

MainWindow.MessageContextualMenu.Open:
Sub Open()
  me.addrow(s_direct_browse)
  me.addrow(s_browse)
  me.addrow(s_whois)
  me.addseparator()
  me.addrow(s_favorite)
  me.addrow(s_ignore)
End Sub

MainWindow.MessageContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_direct_browse
    self.ME_Browse(true)
  case s_browse
    self.ME_Browse(false)
  case s_whois
    self.ME_Whois()
  case s_favorite
    self.ME_UserList(true)
  case s_ignore
    self.ME_UserList(false)
  end
End Sub

MainWindow.MessageEditField.Open:
Sub Open()
  me.height=me.height+16
  me.textFont=iTextFont
  me.textSize=iTextSize
  me.textColor=iColors(co_default)
  me.backColor=iColors(co_background)
End Sub

MainWindow.MessageEditField.CMMClicked:
Sub CMMClicked()
  self.MessageContextualMenu.open()
End Sub

MainWindow.CommandContextualMenus.Open:
Sub Open(Index As Integer)
  if index=1 then
    me.addrow(">console")
    me.addrow(">ghost")
    me.addrow(">help")
    me.addrow(">invite ")
    me.addrow(">listlevels")
    me.addrow(">me ")
    me.addrow(">msg ")
    me.addrow(">part")
    me.addrow(">point ")
    me.addrow(">summon ")
    me.addrow(">topic ")
    me.addrow(">usermode ")
    me.addrow(">usermode help")
    me.addrow(">version")
    me.addrow(">whois ")
  end
End Sub

MainWindow.CommandContextualMenus.Action:
Sub Action(Index As Integer, item As String)
  self.CommandEditFields(index).setFocus()
  self.CommandEditFields(index).replaceText(item)
End Sub

MainWindow.CommandRoundedCanvases.Open:
Sub Open(Index As Integer)
  me.top=me.top+16
  me.hasEmoIcon=true
  me.setEditField(self.CommandEditFields(index))
End Sub

MainWindow.CommandRoundedCanvases.IconClicked:
Sub IconClicked(Index As Integer)
  self.EmoIconContextualMenus(index).open()
End Sub

MainWindow.CommandEditFields.KeyDown:
Function KeyDown(Index As Integer, key as string) As Boolean
  select case index
  case 0
    if ascb(key)=13 and lenb(me.text)<>0 then
      self.ME_Message(getString(trim(me.text)))
      me.text=""
    end
  case 1
    if ascb(key)=13 and lenb(me.text)<>0 then
      self.RO_PublicMessage(getString(trim(me.text)))
      me.text=""
    end
  end
End Function

MainWindow.RoomContextualMenu.Open:
Sub Open()
  me.addrow(s_refresh)
  me.addseparator()
  me.addrow(s_direct_browse)
  me.addrow(s_browse)
  me.addrow(s_message)
  me.addrow(s_whois)
  me.addseparator()
  me.addrow(s_favorite)
  me.addrow(s_ignore)
End Sub

MainWindow.RoomContextualMenu.Action:
Sub Action(item As String)
  select case item
  case s_refresh
    self.RO_Refresh()
  case s_direct_browse
    self.RO_Browse(true)
  case s_browse
    self.RO_Browse(false)
  case s_message
    self.RO_Message()
  case s_whois
    self.RO_Whois()
  case s_favorite
    self.RO_UserList(true)
  case s_ignore
    self.RO_UserList(false)
  end
End Sub

MainWindow.RoomEditField.Open:
Sub Open()
  me.height=me.height+16
  me.textFont=iTextFont
  me.textSize=iTextSize
  me.textColor=iColors(co_default)
  me.backColor=iColors(co_background)
End Sub

MainWindow.RoomListBox.SortColumn:
Function SortColumn(column as integer) As boolean
  if column=RO_Count then
    if RoomWindowLock.signal() then
      me.sortNumber(column)
      RoomWindowLock.release()
    end
    return true
  end
End Function

MainWindow.RoomListBox.Open:
Sub Open()
  me.height=me.height+16
  me.isColored=true
  getColumn(me,c_room_window)
  me.columnAlignment(RO_Count)=3
  me.columnAlignment(RO_Speed)=2
End Sub

MainWindow.RoomListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  if ascb(key)=32 then
    self.RO_Message()
  end
End Function

MainWindow.RoomListBox.CMMClicked:
Sub CMMClicked()
  self.RoomContextualMenu.open()
End Sub

MainWindow.RoomListBox.Close:
Sub Close()
  writeColumn(me,c_room_window)
End Sub

MainWindow.RoomListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  me.helpTag=me.cell(row,column)
  me.balloonHelp=me.helpTag
End Function

MainWindow.RoomStaticText.Open:
Sub Open()
  me.top=me.top+16
End Sub

MainWindow.RoomStaticText.MouseDown:
Function MouseDown(X As Integer, Y As Integer) As Boolean
  self.CommandEditFields(1).replaceText(getDefine(">topic ")+me.text)
End Function

MainWindow.TransferNetworkContextualMenus.Action:
Sub Action(Index As Integer, item As String)
  dim aArray,aIndex as integer
  aIndex=self.TransferListBoxes(index).listIndex
  if item=s_lookup then
    'whois request
    OS_Broadcast(603,self.TransferListBoxes(index).cell(aIndex,TR_Nick))
    self.WhoisListBox.listIndex=-1
    DrawerWindow.MenuListBox.ListIndex=3
  elseif index=0 and val(self.TransferListBoxes(index).cell(aIndex,TR_ID))=0 and strcomp(self.TransferListBoxes(index).cell(aIndex,TR_Network),item,0)<>0 then
    
    if OpenNapSocketLock.enter() then
      aArray=OS_SearchOnlineAndNetwork2Array(item)
      if aArray<>c_error then
        if val(self.TransferListBoxes(index).cell(aIndex,TR_Flag))=TF_RetryingIn then
          gDownloadRetrying=gDownloadRetrying-1
        end
        self.TransferListBoxes(index).cell(aIndex,TR_Flag)=str(TF_LocallyQueued)
        self.TransferListBoxes(index).cell(aIndex,TR_Status)=s_locally_queued
        self.TransferListBoxes(index).cellRefresh(aIndex,TR_ShortName)
        self.TransferListBoxes(index).cell(aIndex,TR_Network)=item
        self.TransferListBoxes(index).cellRefresh(aIndex,TR_Nick)
        self.TransferListBoxes(index).cell(aIndex,TR_ServerID)=str(OpenNapSocket(aArray).ID)
      end
      OpenNapSocketLock.leave()
    end
    
  end
End Sub

MainWindow.ConsolePopupMenu.Open:
Sub Open()
  me.addrow(s_all)
  me.rowTag(0)=0
  me.listindex=0
End Sub

MainWindow.ConsolePopupMenu.Change:
Sub Change()
  self.CO_Filter(me.RowTag(me.listindex))
End Sub

MainWindow.ChannelPopupMenu.Open:
Sub Open()
  me.addrow(s_all)
  me.rowTag(0)=0
  me.listindex=0
End Sub

MainWindow.ChannelPopupMenu.Change:
Sub Change()
  self.CH_Refresh()
End Sub

MainWindow.EmoIconContextualMenus.Open:
Sub Open(Index As Integer)
  me.addrow(emo_smile)
  me.addrow(emo_happy)
  me.addrow(emo_wink)
  me.addrow(emo_kiss)
  me.addrow(emo_cool)
  me.addrow(emo_frown)
  me.addrow(emo_stoic)
  me.addrow(emo_annoyed)
  me.addrow(emo_cry)
  me.addrow(emo_surprised)
  me.addrow(emo_razz)
  me.addrow(emo_angry)
End Sub

MainWindow.EmoIconContextualMenus.Action:
Sub Action(Index As Integer, item As String)
  self.CommandEditFields(index).setFocus()
  self.CommandEditFields(index).insertText(nthfield(item,chrb(32),1)+chrb(32))
End Sub

PreferencesWindow.PW_ApplyConfig:
Protected Sub PW_ApplyConfig()
  dim i,j as integer
  dim fi as FolderItem
  iAutoReply=getString(trim(me.EditFields(0).text))
  iHTTPProxy=getString(trim(me.EditFields(1).text))
  iPortRange=trim(me.EditFields(2).text)
  getPortRange()
  iFreeloaderShares=max(val(trim(me.EditFields(3).text)),1)
  
  fi=getPath2FolderItem(me.PopupMenus(13).rowTag(me.PopupMenus(13).listIndex))
  if fi<>nil and fi.exists then
    iDownloadFolder=fi
  end
  fi=getPath2FolderItem(me.PopupMenus(14).rowTag(me.PopupMenus(14).listIndex))
  if fi<>nil and fi.exists then
    iIncompleteFolder=fi
  end
  if me.CheckBoxes(13).value then
    fi=getPath2FolderItem(me.PopupMenus(15).rowTag(me.PopupMenus(15).listIndex))
    if fi<>nil and fi.exists then
      iMusicFolder=fi
    end
  else
    iMusicFolder=nil
  end
  if me.CheckBoxes(14).value then
    fi=getPath2FolderItem(me.PopupMenus(16).rowTag(me.PopupMenus(16).listIndex))
    if fi<>nil and fi.exists then
      iMoviesFolder=fi
    end
  else
    iMoviesFolder=nil
  end
  if me.CheckBoxes(15).value then
    fi=getPath2FolderItem(me.PopupMenus(17).rowTag(me.PopupMenus(17).listIndex))
    if fi<>nil and fi.exists then
      iPicturesFolder=fi
    end
  else
    iPicturesFolder=nil
  end
  
  iUploadBandwidth=val(trim(me.EditFields(4).text))
  iNick=getString(trim(me.EditFields(5).text))
  iPassword=getString(trim(me.EditFields(6).text))
  iClient=getString(trim(me.EditFields(7).text))
  if instrb(1,iCLient,"2get")<>0 then
    iClient=getDefine(App.LongVersion)
    isWinMX=false
  elseif instrb(1,iClient,"Win")<>0 then
    iClient=getDefine(c_winmx)
    isWinMX=true
  else
    isWinMX=false
  end
  
  iAutoConnect=me.CheckBoxes(0).value
  iAskNapList=me.CheckBoxes(1).value
  iUpdateLibrary=me.CheckBoxes(2).value
  iClearDownload=me.CheckBoxes(3).value
  iClearUpload=me.CheckBoxes(4).value
  iTimeStamp=me.CheckBoxes(5).value
  iShowJoinLeave=me.CheckBoxes(6).value
  iPromptRemoving=me.CheckBoxes(7).value
  iPromptQuitting=me.CheckBoxes(8).value
  iShowBandwidth=me.CheckBoxes(9).value
  iClearServer=me.CheckBoxes(12).value
  iAllowFreeloader=me.CheckBoxes(16).value
  
  iTextFont=me.PopupMenus(0).list(me.PopupMenus(0).listIndex)
  gTextFontID=getFontName2ID(iTextFont)
  select case me.PopupMenus(1).listIndex
  case 0
    iTextSize=8
  case 1
    iTextSize=9
  case 2
    iTextSize=10
  case 3
    iTextSize=11
  case 4
    iTextSize=12
  case 5
    iTextSize=13
  case 6
    iTextSize=14
  case 7
    iTextSize=15
  case 8
    iTextSize=16
  case 9
    iTextSize=18
  case 10
    iTextSize=20
  case 11
    iTextSize=24
  case 12
    iTextSize=32
  case 13
    iTextSize=36
  end
  i=me.PopupMenus(2).rowTag(me.PopupMenus(2).listIndex)
  if i<>256 then
    iTextEncoding=getTextEncoding(i)
  else
    iTextEncoding=encodings.UTF8
  end
  iSpeed=me.PopupMenus(3).listindex
  select case me.PopupMenus(4).listindex
  case 0
    iConnectionLimits=5
  case 1
    iConnectionLimits=10
  case 2
    iConnectionLimits=15
  case 3
    iConnectionLimits=20
  case 4
    iConnectionLimits=25
  case 5
    iConnectionLimits=30
  end
  select case me.PopupMenus(5).listindex
  case 0
    iDownloadBuffer=1024*1
  case 1
    iDownloadBuffer=1024*2
  case 2
    iDownloadBuffer=1024*4
  case 3
    iDownloadBuffer=1024*8
  case 4
    iDownloadBuffer=1024*12
  case 5
    iDownloadBuffer=1024*16
  case 6
    iDownloadBuffer=1024*20
  case 7
    iDownloadBuffer=1024*24
  case 8
    iDownloadBuffer=1024*28
  case 9
    iDownloadBuffer=1024*32
  case 10
    iDownloadBuffer=1024*48
  case 11
    iDownloadBuffer=1024*64
  case 12
    iDownloadBuffer=1024*128
  case 13
    iDownloadBuffer=1024*256
  case 14
    iDownloadBuffer=1024*512
  end
  select case me.PopupMenus(6).listindex
  case 0
    iUploadBuffer=1024*1
  case 1
    iUploadBuffer=1024*2
  case 2
    iUploadBuffer=1024*4
  case 3
    iUploadBuffer=1024*8
  case 4
    iUploadBuffer=1024*12
  case 5
    iUploadBuffer=1024*16
  case 6
    iUploadBuffer=1024*20
  case 7
    iUploadBuffer=1024*24
  case 8
    iUploadBuffer=1024*28
  case 9
    iUploadBuffer=1024*32
  case 10
    iUploadBuffer=1024*48
  case 11
    iUploadBuffer=1024*64
  case 12
    iUploadBuffer=1024*128
  case 13
    iUploadBuffer=1024*256
  case 14
    iUploadBuffer=1024*512
  end
  select case me.PopupMenus(7).listindex
  case 0
    iCPUCycleTime=0
  case 1
    iCPUCycleTime=50
  case 2
    iCPUCycleTime=100
  case 3
    iCPUCycleTime=150
  case 4
    iCPUCycleTime=200
  case 5
    iCPUCycleTime=250
  case 6
    iCPUCycleTime=300
  case 7
    iCPUCycleTime=350
  case 8
    iCPUCycleTime=400
  case 9
    iCPUCycleTime=450
  case 10
    iCPUCycleTime=500
  end
  iDownloadLimits=me.PopupMenus(8).listindex
  iDownloadPerUser=me.PopupMenus(9).listindex
  iDownloadPerServer=me.PopupMenus(10).listindex
  iUploadLimits=me.PopupMenus(11).listindex
  iUploadPerUser=me.PopupMenus(12).listindex
  
  iColors(co_default)=me.colors(co_default)
  iColors(co_background)=me.colors(co_background)
  iColors(co_timestamp)=me.colors(co_timestamp)
  iColors(co_own_user)=me.colors(co_own_user)
  iColors(co_other_users)=me.colors(co_other_users)
  iColors(co_notifications)=me.colors(co_notifications)
  
  if LibraryLock.signal() then
    redim iUploadFolder(0)
    j=me.PreferencesListBox.ListCount-1
    for i=0 to j
      fi=getPath2FolderItem(me.PreferencesListBox.list(i))
      if fi<>nil then
        iUploadFolder.append fi
      end 
    next
    isShareDownloads=(OL_SearchFolderPath2Array(getFolderItem2Path(iDownloadFolder),getFolderItem2Path(iMusicFolder),getFolderItem2Path(iMoviesFolder),getFolderItem2Path(iPicturesFolder))<>c_error)
    LibraryLock.release()
  end
End Sub

PreferencesWindow.EnableMenuItems:
Sub EnableMenuItems()
  closeWindowMenu.enabled=true
  minimizeMenu.enabled=true
End Sub

PreferencesWindow.CancelClose:
Function CancelClose(appQuitting as Boolean) As Boolean
  if isWantQuit=false then
    if lenb(me.EditFields(5).text)=0 or lenb(me.EditFields(6).text)=0 or lenb(me.EditFields(7).text)=0 then
      getSound(rs_error)
      msgbox s_input_every_fields_
      return true
    else
      
      me.PW_ApplyConfig()
      writeConfig()
      
      if ListenerSocket.port<>iPort then
        if ListenerSocket.isListening then
          ListenerSocket.stopListening()
        end
        ListenerSocket.port=iPort
        if iPort<>0 then
          ListenerSocket.listen()
        end
      end
      TransferTimer.period=iCPUCycleTime
      
      userMenu.text=s_user_name_+chrb(32)+iNick
      sharesMenu.text=s_shared_+chrb(32)+str(ubound(Library))
      speedMenu.text=s_speed_+chrb(32)+getShortSpeed(iSpeed)
      clientMenu.text=s_client_+chrb(32)+iClient
      portMenu.text=s_port_+chrb(32)+str(iPort)
      MainWindow.MessageEditField.backColor=iColors(co_background)
      MainWindow.RoomEditField.backColor=iColors(co_background)
      MainWindow.CO_Update()
      OS_Update()
      if me.isChanged then
        me.isChanged=false
        OL_Update()
      end
      #if targetPPC
        DrawerWindow.show()
      #endif
    end
    
  end
End Function

PreferencesWindow.Open:
Sub Open()
  me.PagePanels.value=gPageIndex
End Sub

PreferencesWindow.Separators.Open:
Sub Open(Index As Integer)
  if index<>0 then
    me.left=20
  end
End Sub

PreferencesWindow.PagePanels.Close:
Sub Close()
  gPageIndex=me.value
End Sub

PreferencesWindow.PagePanels.Change:
Sub Change()
  select case me.value
  case 0
    self.setTitle(s_general)
    self.height=416
  case 1
    self.setTitle(s_appearance)
    self.height=312
  case 2
    self.setTitle(s_network)
    self.height=416
  case 3
    self.setTitle(s_download)
    self.height=386
  case 4
    self.setTitle(s_sharing)
    self.height=440
  case 5
    self.setTitle(s_identification)
    self.height=296
  end
End Sub

PreferencesWindow.CheckBoxes.Open:
Sub Open(Index As Integer)
  select case Index
  case 0
    me.value=iAutoConnect
  case 1
    me.value=iAskNapList
  case 2
    me.value=iUpdateLibrary
  case 3
    me.value=iClearDownload
  case 4
    me.value=iClearUpload
  case 5
    me.value=iTimeStamp
  case 6
    me.value=iShowJoinLeave
  case 7
    me.value=iPromptRemoving
  case 8
    me.value=iPromptQuitting
  case 9
    me.value=iShowBandwidth
  case 10
    me.value=(lenb(iHTTPProxy)<>0)
  case 11
    me.value=(iPort<>0)
  case 12
    me.value=iClearServer
  case 13
    me.value=(iMusicFolder<>nil and iMusicFolder.exists)
  case 14
    me.value=(iMoviesFolder<>nil and iMoviesFolder.exists)
  case 15
    me.value=(iPicturesFolder<>nil and iPicturesFolder.exists)
  case 16
    me.value=iAllowFreeloader
  end
  
  self.graphics.textFont=me.textFont
  self.graphics.textSize=me.textSize
  if self.graphics.stringWidth(me.caption)>me.width-16 then
    me.textSize=10
  end
End Sub

PreferencesWindow.CheckBoxes.Action:
Sub Action(Index As Integer)
  select case index
  case 10
    if me.value then
      self.EditFields(1).text=iHTTPProxy
    else
      self.EditFields(1).text=""
    end
  case 11
    if me.value then
      self.EditFields(2).text=iPortRange
    else
      self.EditFields(2).text="0"
    end
  case 16
    self.StaticTexts(31).enabled=not me.value
    self.EditFields(3).enabled=not me.value
  end
End Sub

PreferencesWindow.PopupMenus.Open:
Sub Open(Index As Integer)
  dim i,j,aIndex as integer
  dim fi as folderItem
  dim temp as string
  select case Index
  case 0
    me.addrow("System")
    me.listIndex=0
    j=fontCount-1
    for i=0 to j
      me.addrow(font(i))
      if strcomp(font(i),iTextFont,0)=0 then
        me.listIndex=i+1
      end
    next
    
  case 1
    me.addrow("8")
    me.addrow("9")
    me.addrow("10")
    me.addrow("11")
    me.addrow("12")
    me.addrow("13")
    me.addrow("14")
    me.addrow("16")
    me.addrow("18")
    me.addrow("20")
    me.addrow("24")
    me.addrow("32")
    me.addrow("36")
    select case iTextSize
    case 8
      me.listIndex=0
    case 9
      me.listIndex=1
    case 10
      me.listIndex=2
    case 11
      me.listIndex=3
    case 12
      me.listIndex=4
    case 13
      me.listIndex=5
    case 14
      me.listIndex=6
    case 15
      me.listIndex=7
    case 16
      me.listIndex=8
    case 18
      me.listIndex=9
    case 20
      me.listIndex=10
    case 24
      me.listIndex=11
    case 32
      me.listIndex=12
    case 36
      me.listIndex=13
    end
    
  case 2
    me.addrow("Western (Mac OS Roman)")
    me.rowtag(me.listCount-1)=0
    me.addrow("Japanese (Mac OS)")
    me.rowtag(me.listCount-1)=1
    me.addrow("Tradiional Chinese (Mac OS)")
    me.rowtag(me.listCount-1)=2
    me.addrow("Korean (Mac OS)")
    me.rowtag(me.listCount-1)=3
    me.addrow("Arabic (Mac OS)")
    me.rowtag(me.listCount-1)=4
    me.addrow("Hebrew (Mac OS)")
    me.rowtag(me.listCount-1)=5
    me.addrow("Greek (Mac OS)")
    me.rowtag(me.listCount-1)=6
    me.addrow("Cyrillic (Mac OS)")
    me.rowtag(me.listCount-1)=7
    me.addrow("Devanagari (Mac OS)")
    me.rowtag(me.listCount-1)=9
    me.addrow("Gurmukhi (Mac OS)")
    me.rowtag(me.listCount-1)=10
    me.addrow("Gujarati (Mac OS)")
    me.rowtag(me.listCount-1)=11
    me.addrow("Thai (Mac OS)")
    me.rowtag(me.listCount-1)=21
    me.addrow("Simplified Chinese (Mac OS)")
    me.rowtag(me.listCount-1)=25
    me.addrow("Tibetan (Mac OS)")
    me.rowtag(me.listCount-1)=26
    me.addrow("Central European (Mac OS)")
    me.rowtag(me.listCount-1)=29
    me.addrow("Symbol (Mac OS)")
    me.rowtag(me.listCount-1)=33
    me.addrow("Dingbats (Mac OS)")
    me.rowtag(me.listCount-1)=34
    me.addrow("Turkish (Mac OS)")
    me.rowtag(me.listCount-1)=35
    me.addrow("Croatian (Mac OS)")
    me.rowtag(me.listCount-1)=36
    me.addrow("Icelandic (Mac OS)")
    me.rowtag(me.listCount-1)=37
    me.addrow("Romanian (Mac OS)")
    me.rowtag(me.listCount-1)=38
    me.addrow("Keyboard Symbols (Mac OS)")
    me.rowtag(me.listCount-1)=41
    me.addrow("Farsi (Mac OS)")
    me.rowtag(me.listCount-1)=&h8C
    me.addrow("Cyrillic (Mac OS Ukrainian)")
    me.rowtag(me.listCount-1)=&h98
    me.addrow("Western (Mac VT100)")
    me.rowtag(me.listCount-1)=&hFC
    me.addseparator()
    me.addrow("Unicode (UTF-8)")
    me.rowtag(me.listCount-1)=256
    me.addseparator()
    me.addrow("Western (ISO Latin 1)")
    me.rowtag(me.listCount-1)=513
    me.addrow("Central European (ISO Latin 2)")
    me.rowtag(me.listCount-1)=514
    me.addrow("Western (ISO Latin 3)")
    me.rowtag(me.listCount-1)=515
    me.addrow("Central European (ISO Latin 4)")
    me.rowtag(me.listCount-1)=516
    me.addrow("Cyrillic (ISO 8859-5)")
    me.rowtag(me.listCount-1)=517
    me.addrow("Arabic (ISO 8859-6)")
    me.rowtag(me.listCount-1)=518
    me.addrow("Greek (ISO 8859-7)")
    me.rowtag(me.listCount-1)=519
    me.addrow("Hebrew (ISO 8859-8)")
    me.rowtag(me.listCount-1)=520
    me.addrow("Turkish (ISO Latin 5)")
    me.rowtag(me.listCount-1)=521
    me.addrow("Nordic (ISO Latin 6)")
    me.rowtag(me.listCount-1)=522
    me.addrow("Baltic Rim (ISO Latin 7)")
    me.rowtag(me.listCount-1)=525
    me.addrow("Celtic (ISO Latin 8)")
    me.rowtag(me.listCount-1)=526
    me.addrow("Western (ISO Latin 9)")
    me.rowtag(me.listCount-1)=527
    me.addrow("Romanian (ISO Latin 10)")
    me.rowtag(me.listCount-1)=528
    me.addseparator()
    me.addrow("Latin-US (DOS)")
    me.rowtag(me.listCount-1)=1024
    me.addrow("Greek (DOS)")
    me.rowtag(me.listCount-1)=1029
    me.addrow("Baltic Rim (DOS)")
    me.rowtag(me.listCount-1)=1030
    me.addrow("Western (DOS Latin 1)")
    me.rowtag(me.listCount-1)=1040
    me.addrow("Central European (DOS Latin 2)")
    me.rowtag(me.listCount-1)=1042
    me.addrow("Turkish (DOS)")
    me.rowtag(me.listCount-1)=1044
    me.addrow("Icelandic (DOS)")
    me.rowtag(me.listCount-1)=1046
    me.addrow("Arabic (DOS)")
    me.rowtag(me.listCount-1)=1049
    me.addrow("Cyrillic (DOS)")
    me.rowtag(me.listCount-1)=1043
    me.addseparator()
    me.addrow("Thai (Windows, DOS)")
    me.rowtag(me.listCount-1)=1053
    me.addrow("Japanese (Windows, DOS)")
    me.rowtag(me.listCount-1)=1056
    me.addrow("Simplified Chinese (Windows, DOS)")
    me.rowtag(me.listCount-1)=1057
    me.addrow("Korean (Windows, DOS)")
    me.rowtag(me.listCount-1)=1058
    me.addrow("Traditional Chinese (Windows, DOS)")
    me.rowtag(me.listCount-1)=1059
    me.addseparator()
    me.addrow("Western (Windows Latin 1)")
    me.rowtag(me.listCount-1)=1280
    me.addrow("Central European (Windows Latin 2)")
    me.rowtag(me.listCount-1)=1281
    me.addrow("Cyrillic (Windows)")
    me.rowtag(me.listCount-1)=1282
    me.addrow("Greek (Windows)")
    me.rowtag(me.listCount-1)=1283
    me.addrow("Turkish (Windows Latin 5)")
    me.rowtag(me.listCount-1)=1284
    me.addrow("Hebrew (Windows)")
    me.rowtag(me.listCount-1)=1285
    me.addrow("Arabic (Windows)")
    me.rowtag(me.listCount-1)=1286
    me.addrow("Baltic Rim (Windows)")
    me.rowtag(me.listCount-1)=1287
    me.addrow("Vietnamese (Windows)")
    me.rowtag(me.listCount-1)=1288
    me.addseparator()
    me.addrow("Western (ASCII)")
    me.rowtag(me.listCount-1)=1536
    me.addrow("Japanese (Shift JIS X0213)")
    me.rowtag(me.listCount-1)=&h628
    me.addrow("Chinese (GBK)")
    me.rowtag(me.listCount-1)=&h631
    me.addrow("Chinese (GB 18030)")
    me.rowtag(me.listCount-1)=&h632
    me.addseparator()
    me.addrow("Japanese (ISO 2022-JP)")
    me.rowtag(me.listCount-1)=&h820
    me.addrow("Korean (ISO 2022-KR)")
    me.rowtag(me.listCount-1)=&h840
    me.addseparator()
    me.addrow("Japanese (EUC)")
    me.rowtag(me.listCount-1)=&h920
    me.addrow("Simplified Chinese (EUC)")
    me.rowtag(me.listCount-1)=&h930
    me.addrow("Traditional Chinese (EUC)")
    me.rowtag(me.listCount-1)=&h931
    me.addrow("Korean (EUC)")
    me.rowtag(me.listCount-1)=&h940
    me.addseparator()
    me.addrow("Japanese (Shift JIS)")
    me.rowtag(me.listCount-1)=&hA01
    me.addrow("Cyrillic (KOI8-R)")
    me.rowtag(me.listCount-1)=&hA02
    me.addrow("Traditional Chinese (Big 5)")
    me.rowtag(me.listCount-1)=&hA03
    me.addrow("Western (Mac Mail)")
    me.rowtag(me.listCount-1)=&hA04
    me.addrow("Simplified Chinese (HZ GB 2312)")
    me.rowtag(me.listCount-1)=&hA05
    me.addrow("Traditional Chinese (Big 5 HKSCS)")
    me.rowtag(me.listCount-1)=&hA06
    me.addrow("Ukrainian (KOI8-U)")
    me.rowtag(me.listCount-1)=&hA08
    me.addrow("Traditional Chinese (Big 5-E)")
    me.rowtag(me.listCount-1)=&hA09
    me.addrow("Western (NextStep)")
    me.rowtag(me.listCount-1)=&hB01
    me.addrow("Cyrillic (KOI8-R)")
    me.rowtag(me.listCount-1)=&hA02
    me.addrow("Western (EBCDIC Latin 1)")
    me.rowtag(me.listCount-1)=&hC02
    
    aIndex=me.searchRowTag2ListIndex(iTextEncoding.base)
    if aIndex<>c_error then
      me.listIndex=aIndex
    end
    
  case 3
    me.addrow(s_unknown)
    me.addrow(s_14_kbps)
    me.addrow(s_28_kbps)
    me.addrow(s_33_kbps)
    me.addrow(s_56_kbps)
    me.addrow(s_isdn)
    me.addrow(s_dual_isdn)
    me.addrow(s_cable)
    me.addrow(s_dsl)
    me.addrow(s_t1)
    me.addrow(s_t3)
    me.listindex=iSpeed
    
  case 4
    me.addrow("5")
    me.addrow("10")
    me.addrow("15")
    me.addrow("20")
    me.addrow("25")
    me.addrow("30")
    select case iConnectionLimits
    case 5
      me.listindex=0
    case 10
      me.listindex=1
    case 15
      me.listindex=2
    case 20
      me.listindex=3
    case 25
      me.listindex=4
    case 30
      me.listindex=5
    end
    
  case 5
    me.addrow("1"+s__kb)
    me.addrow("2"+s__kb)
    me.addrow("4"+s__kb)
    me.addrow("8"+s__kb)
    me.addrow("12"+s__kb)
    me.addrow("16"+s__kb)
    me.addrow("20"+s__kb)
    me.addrow("24"+s__kb)
    me.addrow("28"+s__kb)
    me.addrow("32"+s__kb)
    me.addrow("48"+s__kb)
    me.addrow("64"+s__kb)
    me.addrow("128"+s__kb)
    me.addrow("256"+s__kb)
    me.addrow("512"+s__kb)
    select case iDownloadBuffer
    case 1024*1
      me.listindex=0
    case 1024*2
      me.listindex=1
    case 1024*4
      me.listindex=2
    case 1024*8
      me.listindex=3
    case 1024*12
      me.listindex=4
    case 1024*16
      me.listindex=5
    case 1024*20
      me.listindex=6
    case 1024*24
      me.listindex=7
    case 1024*28
      me.listindex=8
    case 1024*32
      me.listindex=9
    case 1024*48
      me.listindex=10
    case 1024*64
      me.listindex=11
    case 1024*128
      me.listindex=12
    case 1024*256
      me.listindex=13
    case 1024*512
      me.listindex=14
    end
    
  case 6
    me.addrow("1"+s__kb)
    me.addrow("2"+s__kb)
    me.addrow("4"+s__kb)
    me.addrow("8"+s__kb)
    me.addrow("12"+s__kb)
    me.addrow("16"+s__kb)
    me.addrow("20"+s__kb)
    me.addrow("24"+s__kb)
    me.addrow("28"+s__kb)
    me.addrow("32"+s__kb)
    me.addrow("48"+s__kb)
    me.addrow("64"+s__kb)
    me.addrow("128"+s__kb)
    me.addrow("256"+s__kb)
    me.addrow("512"+s__kb)
    select case iUploadBuffer
    case 1024*1
      me.listindex=0
    case 1024*2
      me.listindex=1
    case 1024*4
      me.listindex=2
    case 1024*8
      me.listindex=3
    case 1024*12
      me.listindex=4
    case 1024*16
      me.listindex=5
    case 1024*20
      me.listindex=6
    case 1024*24
      me.listindex=7
    case 1024*28
      me.listindex=8
    case 1024*32
      me.listindex=9
    case 1024*48
      me.listindex=10
    case 1024*64
      me.listindex=11
    case 1024*128
      me.listindex=12
    case 1024*256
      me.listindex=13
    case 1024*512
      me.listindex=14
    end
    
  case 7
    me.addrow("0"+s__ms)
    me.addrow("50"+s__ms)
    me.addrow("100"+s__ms)
    me.addrow("150"+s__ms)
    me.addrow("200"+s__ms)
    me.addrow("250"+s__ms)
    me.addrow("300"+s__ms)
    me.addrow("350"+s__ms)
    me.addrow("400"+s__ms)
    me.addrow("450"+s__ms)
    me.addrow("500"+s__ms)
    select case iCPUCycleTime
    case 0
      me.listIndex=0
    case 50
      me.listIndex=1
    case 100
      me.listIndex=2
    case 150
      me.listIndex=3
    case 200
      me.listIndex=4
    case 250
      me.listIndex=5
    case 300
      me.listIndex=6
    case 350
      me.listIndex=7
    case 400
      me.listIndex=8
    case 450
      me.listIndex=9
    case 500
      me.listIndex=10
    end
    
  case 8
    me.addrow(s_no_limit)
    me.addrow("1")
    me.addrow("2")
    me.addrow("3")
    me.addrow("4")
    me.addrow("5")
    me.addrow("6")
    me.addrow("7")
    me.addrow("8")
    me.addrow("9")
    me.addrow("10")
    me.addrow("11")
    me.addrow("12")
    me.addrow("13")
    me.addrow("14")
    me.addrow("15")
    me.listindex=iDownloadLimits
    
  case 9
    me.addrow(s_no_limit)
    me.addrow("1")
    me.addrow("2")
    me.addrow("3")
    me.addrow("4")
    me.addrow("5")
    me.listindex=iDownloadPerUser
    
  case 10
    me.addrow(s_no_limit)
    me.addrow("1")
    me.addrow("2")
    me.addrow("3")
    me.addrow("4")
    me.addrow("5")
    me.listindex=iDownloadPerServer
    
  case 11
    me.addrow(s_no_limit)
    me.addrow("1")
    me.addrow("2")
    me.addrow("3")
    me.addrow("4")
    me.addrow("5")
    me.addrow("6")
    me.addrow("7")
    me.addrow("8")
    me.addrow("9")
    me.addrow("10")
    me.addrow("11")
    me.addrow("12")
    me.addrow("13")
    me.addrow("14")
    me.addrow("15")
    me.listindex=iUploadLimits
    
  case 12
    me.addrow(s_no_limit)
    me.addrow("1")
    me.addrow("2")
    me.addrow("3")
    me.addrow("4")
    me.addrow("5")
    me.listindex=iUploadPerUser
    
  case 13
    if DesktopFolder<>nil and DesktopFolder.exists then
      me.addrow(DesktopFolder.displayName)
      me.rowTag(0)=getFolderItem2Path(DesktopFolder)
    end
    if DocumentsFolder<>nil and DocumentsFolder.exists then
      me.addrow(DocumentsFolder.displayName)
      me.rowTag(1)=getFolderItem2Path(DocumentsFolder)
      fi=DocumentsFolder.parent.child("Music")
      if fi<>nil and fi.exists then
        me.addrow(fi.displayName)
        me.rowTag(2)=getFolderItem2Path(fi)
      end
      fi=DocumentsFolder.parent.child("Movies")
      if fi<>nil and fi.exists then
        me.addrow(fi.displayName)
        me.rowTag(3)=getFolderItem2Path(fi)
      end
      fi=DocumentsFolder.parent.child("Pictures")
      if fi<>nil and fi.exists then
        me.addrow(fi.displayName)
        me.rowTag(4)=getFolderItem2Path(fi)
      end
    end
    me.addSeparator()
    me.addrow(s_other_)
    temp=getFolderItem2Path(iDownloadFolder)
    aIndex=me.searchRowTag2ListIndex(temp)
    if aIndex=c_error then
      me.addRow(iDownloadFolder.displayName)
      aIndex=me.listCount-1
      me.rowTag(aIndex)=temp
    end
    me.listIndex=aIndex
    
  case 14
    fi=iIncompleteFolder
    if fi<>nil and fi.exists then
      temp=getFolderItem2Path(fi)
      aIndex=me.searchRowTag2ListIndex(temp)
      if aIndex=c_error then
        me.addRow(fi.displayName)
        aIndex=me.listCount-1
        me.rowTag(aIndex)=temp
      end
    end
    me.addSeparator()
    me.addrow(s_other_)
    me.listIndex=aIndex
    
  case 15
    j=self.PopupMenus(13).listCount-1
    for i=0 to j
      temp=""
      temp=self.PopupMenus(13).list(i)
      if lenb(temp)<>0 then
        me.addrow(temp)
        me.rowTag(i)=self.PopupMenus(13).rowTag(i)
        if strcomp(temp,s_other_,0)=0 then
          exit
        end
      else
        me.addSeparator()
      end
    next
    fi=iMusicFolder
    if fi<>nil and fi.exists then
      temp=""
      temp=getFolderItem2Path(fi)
      aIndex=me.searchRowTag2ListIndex(temp)
      if aIndex=c_error then 
        me.addRow(fi.displayName)
        aIndex=me.listCount-1
        me.rowTag(aIndex)=temp
      end
    end
    me.listIndex=aIndex
    
  case 16
    j=self.PopupMenus(13).listCount-1
    for i=0 to j
      temp=""
      temp=self.PopupMenus(13).list(i)
      if lenb(temp)<>0 then
        me.addrow(temp)
        me.rowTag(i)=self.PopupMenus(13).rowTag(i)
        if strcomp(temp,s_other_,0)=0 then
          exit
        end
      else
        me.addSeparator()
      end
    next
    fi=iMoviesFolder
    if fi<>nil and fi.exists then
      temp=""
      temp=getFolderItem2Path(fi)
      aIndex=me.searchRowTag2ListIndex(temp)
      if aIndex=c_error then 
        me.addRow(fi.displayName)
        aIndex=me.listCount-1
        me.rowTag(aIndex)=temp
      end
    end
    me.listIndex=aIndex
    
  case 17
    j=self.PopupMenus(13).listCount-1
    for i=0 to j
      temp=""
      temp=self.PopupMenus(13).list(i)
      if lenb(temp)<>0 then
        me.addrow(temp)
        me.rowTag(i)=self.PopupMenus(13).rowTag(i)
        if strcomp(temp,s_other_,0)=0 then
          exit
        end
      else
        me.addSeparator()
      end
    next
    fi=iPicturesFolder
    if fi<>nil and fi.exists then
      temp=""
      temp=getFolderItem2Path(fi)
      aIndex=me.searchRowTag2ListIndex(temp)
      if aIndex=c_error then 
        me.addRow(fi.displayName)
        aIndex=me.listCount-1
        me.rowTag(aIndex)=temp
      end
    end
    me.listIndex=aIndex
    
  end
End Sub

PreferencesWindow.PopupMenus.Change:
Sub Change(Index As Integer)
  dim d as selectFolderDialog
  dim fi as folderItem
  dim aIndex as integer
  dim temp as string
  select case index
  case 13
    if me.list(me.listIndex)=s_other_ then
      d=new SelectFolderDialog
      d.initialDirectory=iDownloadFolder
      fi=d.showModalWithin(self)
      if fi<>nil and fi.exists then
        temp=getFolderItem2Path(fi)
        aIndex=me.searchRowTag2ListIndex(temp)
        if aIndex=c_error then
          me.addRow(fi.displayName)
          aIndex=me.listCount-1
          me.rowTag(aIndex)=temp
        end
      end
      if me.listIndex<>aIndex then
        me.listIndex=aIndex
      end
    end
    
  case 14
    if me.list(me.listIndex)=s_other_ then
      d=new SelectFolderDialog
      d.initialDirectory=iIncompleteFolder
      fi=d.showModalWithin(self)
      if fi<>nil and fi.exists then
        temp=getFolderItem2Path(fi)
        aIndex=me.searchRowTag2ListIndex(temp)
        if aIndex=c_error then
          me.addrow(fi.displayName)
          aIndex=me.listCount-1
          me.rowTag(aIndex)=temp
        end
      end
      if me.listIndex<>aIndex then
        me.listIndex=aIndex
      end
    end
    
  case 15
    if me.list(me.listIndex)=s_other_ then
      d=new SelectFolderDialog
      d.initialDirectory=iMusicFolder
      fi=d.showModalWithin(self)
      if fi<>nil and fi.exists then
        temp=getFolderItem2Path(fi)
        aIndex=me.searchRowTag2ListIndex(temp)
        if aIndex=c_error then
          me.addRow(fi.displayName)
          aIndex=me.listCount-1
          me.rowTag(aIndex)=temp
        end
      end
      if me.listIndex<>aIndex then
        me.listIndex=aIndex
      end
    end
    
  case 16
    if me.list(me.listIndex)=s_other_ then
      d=new SelectFolderDialog
      d.initialDirectory=iMoviesFolder
      fi=d.showModalWithin(self)
      if fi<>nil and fi.exists then
        temp=getFolderItem2Path(fi)
        aIndex=me.searchRowTag2ListIndex(temp)
        if aIndex=c_error then
          me.addRow(fi.displayName)
          aIndex=me.listCount-1
          me.rowTag(aIndex)=temp
        end
      end
      if me.listIndex<>aIndex then
        me.listIndex=aIndex
      end
    end
    
  case 17
    if me.list(me.listIndex)=s_other_ then
      d=new SelectFolderDialog
      d.initialDirectory=iPicturesFolder
      fi=d.showModalWithin(self)
      if fi<>nil and fi.exists then
        temp=getFolderItem2Path(fi)
        aIndex=me.searchRowTag2ListIndex(temp)
        if aIndex=c_error then
          me.addRow(fi.displayName)
          aIndex=me.listCount-1
          me.rowTag(aIndex)=temp
        end
      end
      if me.listIndex<>aIndex then
        me.listIndex=aIndex
      end
    end
    
  end
End Sub

PreferencesWindow.EditFields.TextChange:
Sub TextChange(Index As Integer)
  if index=4 then
    if val(me.text)=0 then
      self.staticTexts(23).text=s_no_limit
    else
      self.staticTexts(23).text=s_kb_s
    end
  end
End Sub

PreferencesWindow.EditFields.Open:
Sub Open(Index As Integer)
  select case Index
  case 0
    me.text=iAutoReply
  case 1
    me.text=iHTTPProxy
  case 2
    me.text=iPortRange
  case 3
    me.text=str(iFreeloaderShares)
  case 4
    me.text=str(iUploadBandwidth)
  case 5
    me.text=iNick
  case 6
    me.text=iPassword
  case 7
    me.text=iClient
  end
End Sub

PreferencesWindow.EditFields.KeyDown:
Function KeyDown(Index As Integer, key as string) As Boolean
  select case index
  case 5
    return instrb(1,c_invalid_keys+chrb(32),key)<>0
  case 6
    return instrb(1,c_invalid_keys+chrb(32),key)<>0
  case 7
    return instrb(1,c_invalid_keys,key)<>0
  end
End Function

PreferencesWindow.PushButtons.Action:
Sub Action(Index As Integer)
  dim d as selectFolderDialog
  dim fi as FolderItem
  dim i,aIndex as integer
  select case Index
  case 0
    d=new SelectFolderDialog
    fi=d.showModalWithin(self)
    if fi<>nil then
      self.PreferencesListBox.addrow(getFolderItem2Path(fi))
      aIndex=self.PreferencesListBox.lastIndex
      self.PreferencesListBox.RowPicture(aIndex)=gPicture(r_folder)
      self.isChanged=true
    end
    
  case 1
    if self.PreferencesListBox.listindex<>-1 then
      for i=self.PreferencesListBox.listCount-1 downto 0
        if self.PreferencesListBox.selected(i) then
          self.PreferencesListBox.removeRow(i)
          self.isChanged=true
        end
      next
    end
    
  case 2
    showURL "https://sourceforge.net/forum/?group_id=87029"
    
  end
End Sub

PreferencesWindow.PreferencesListBox.KeyDown:
Function KeyDown(key As String) As Boolean
  if ascb(key)=8 then
    self.PushButtons(1).push()
  end
End Function

PreferencesWindow.PreferencesListBox.CellClick:
Function CellClick(row as Integer, column as Integer, x as Integer, y as Integer) As Boolean
  me.helpTag=me.cell(row,column)
  me.balloonHelp=me.helpTag
End Function

PreferencesWindow.PreferencesListBox.DropObject:
Sub DropObject(obj As DragItem)
  dim aIndex as integer
  do
    if obj.FolderItemAvailable then
      me.addrow(getFolderItem2Path(obj.FolderItem))
      aIndex=me.lastIndex
      me.RowPicture(aIndex)=gPicture(r_folder)
      self.isChanged=true
    end
  loop until obj.nextitem=false
End Sub

PreferencesWindow.PreferencesListBox.Open:
Sub Open()
  dim i,j,aIndex as integer
  me.AcceptFileDrop("special/folder")
  if LibraryLock.enter() then
    j=ubound(iUploadFolder)
    for i=1 to j
      me.addrow(getFolderItem2Path(iUploadFolder(i)))
      aIndex=me.lastIndex
      me.RowPicture(aIndex)=gPicture(r_folder)
    next
    LibraryLock.leave()
  end
End Sub

PreferencesWindow.Toolbars.Action:
Sub Action(Index As Integer)
  self.pagePanels.value=index
End Sub

PreferencesWindow.Toolbars.Open:
Sub Open(Index As Integer)
  select case Index
  case 0
    me.setIcon(rb_general)
    me.caption=s_general
    me.setPosition(self)
  case 1
    me.setIcon(rb_appearance)
    me.caption=s_appearance
    me.setPosition(self.Toolbars(index-1))
  case 2
    me.setIcon(rb_network)
    me.caption=s_network
    me.setPosition(self.Toolbars(index-1))
  case 3
    me.setIcon(rb_download)
    me.caption=s_download
    me.setPosition(self.Toolbars(index-1))
  case 4
    me.setIcon(rb_sharing)
    me.caption=s_sharing
    me.setPosition(self.Toolbars(index-1))
  case 5
    me.setIcon(rb_identification)
    me.caption=s_identification
    me.setPosition(self.Toolbars(index-1))
  end
End Sub

PreferencesWindow.Canvases.Open:
Sub Open(Index As Integer)
  dim buffer as picture
  self.colors(index)=iColors(index)
  buffer=newPicture(me.width,me.height,screen(0).depth)
  buffer.graphics.foreColor=self.colors(index)
  buffer.graphics.fillRect 0,0,buffer.width,buffer.height
  buffer.graphics.foreColor=rgb(0,0,0)
  #if targetCarbon
    buffer.graphics.drawRect 0,0,buffer.width,buffer.height
  #else
    buffer.graphics.drawRect 0,0,buffer.width-1,buffer.height-1
  #endif
  me.backdrop=buffer
End Sub

PreferencesWindow.Canvases.MouseDown:
Function MouseDown(Index As Integer, X As Integer, Y As Integer) As Boolean
  dim buffer as picture
  dim c as color
  c=self.colors(index)
  if selectColor(c,"") then
    self.colors(index)=c
    buffer=newPicture(me.width,me.height,screen(0).depth)
    buffer.graphics.foreColor=self.colors(index)
    buffer.graphics.fillRect 0,0,buffer.width,buffer.height
    buffer.graphics.foreColor=rgb(0,0,0)
    #if targetCarbon
      buffer.graphics.drawRect 0,0,buffer.width,buffer.height
    #else
      buffer.graphics.drawRect 0,0,buffer.width-1,buffer.height-1
    #endif
    me.backdrop=buffer
  end
End Function

QuittingWindow.PushButtons.Action:
Sub Action(Index As Integer)
  if index=0 then
    App.quitting()
  end
  self.close()
End Sub

QuittingWindow.Canvases.Open:
Sub Open()
  me.backdrop=getMaskedPicture(rp_splash)
End Sub

RemovingWindow.PushButtons.Action:
Sub Action(Index As Integer)
  self.removeFile=(index=0)
  self.hide()
End Sub

RemovingWindow.Canvases.Open:
Sub Open()
  me.backdrop=getMaskedPicture(rp_splash)
End Sub

