/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2007 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- SVN Information ---
 *  $Id: HostManager.java 3859 2007-07-01 20:15:19Z gregork $
 */
package phex.host;

import java.util.TimerTask;

import phex.common.AbstractManager;
import phex.common.Environment;
import phex.connection.ConnectionObserver;
import phex.connection.NetworkManager;
import phex.connection.OutgoingConnectionDispatcher;
import phex.connection.PingWorker;
import phex.event.NetworkHostsChangeListener;
import phex.prefs.core.ConnectionPrefs;
import phex.prefs.core.NetworkPrefs;
import phex.utils.NLogger;

/**
 * Responsible for managing caught host and the network neighbourhood.
 */
final public class HostManager extends AbstractManager
{
    private static final int MAX_PARALLEL_CONNECTION_TRIES = 20;
    private NetworkManager networkMgr;

    private CaughtHostsContainer caughtHostsContainer;
    private NetworkHostsContainer networkHostsContainer;
    private FavoritesContainer favoritesContainer;
    private UltrapeerCapabilityChecker upChecker;

    private HostManager()
    {
        favoritesContainer = new FavoritesContainer();
        caughtHostsContainer = new CaughtHostsContainer();
    }

    static private class Holder
    {
       static protected final HostManager manager = new HostManager();
    }

    static public HostManager getInstance()
    {
        return HostManager.Holder.manager;
    }



    /**
     * This method is called in order to initialize the manager. This method
     * includes all tasks that must be done to intialize all the several manager.
     * Like instantiating the singleton instance of the manager. Inside
     * this method you can't rely on the availability of other managers.
     * @return true is initialization was successful, false otherwise.
     */
    public boolean initialize()
    {
        return true;
    }

    /**
     * This method is called in order to perform post initialization of the
     * manager. This method includes all tasks that must be done after initializing
     * all the several managers. Inside this method you can rely on the
     * availability of other managers.
     * @return true is initialization was successful, false otherwise.
     */
    public boolean onPostInitialization()
    {
        networkMgr = NetworkManager.getInstance();

        networkHostsContainer = new NetworkHostsContainer();

        ConnectionObserver observer = new ConnectionObserver();
        observer.start();
        
        PingWorker pingWorker = new PingWorker();
        pingWorker.start();

        // Register the NetworkHostsContainer as a NetworkListener
        networkMgr.addNetworkListener( networkHostsContainer );

        upChecker = new UltrapeerCapabilityChecker();

        return true;
    }
    
    /**
     * This method is called after the complete application including GUI completed
     * its startup process. This notification must be used to activate runtime
     * processes that needs to be performed once the application has successfully
     * completed startup.
     */
    public void startupCompletedNotify()
    {
        Environment.getInstance().scheduleTimerTask( 
            new HostCheckTimer(), HostCheckTimer.TIMER_PERIOD,
            HostCheckTimer.TIMER_PERIOD );
    }

    /**
     * This method is called in order to cleanly shutdown the manager. It
     * should contain all cleanup operations to ensure a nice shutdown of Phex.
     */
    public void shutdown()
    {
        try
        {
            caughtHostsContainer.saveHostsContainer();
            favoritesContainer.saveFavoriteHosts();
        }
        catch ( Exception exp )
        {
            NLogger.error( HostManager.class, exp, exp );
        }
    }

    public FavoritesContainer getFavoritesContainer()
    {
        return favoritesContainer;
    }

    public CaughtHostsContainer getCaughtHostsContainer()
    {
        return caughtHostsContainer;
    }

    /////////////// START NetworkHostsContainer wrapper methods ///////////////////////

    public NetworkHostsContainer getNetworkHostsContainer()
    {
        return networkHostsContainer;
    }

    public boolean isShieldedLeafNode()
    {
        return networkHostsContainer.isShieldedLeafNode();
    }

    /**
     * Returns true if this node is allowed to become a ultrapeer, false otherwise.
     * @return true if this node is allowed to become a ultrapeer, false otherwise.
     */
    public boolean isAbleToBecomeUltrapeer()
    {
        // when we already are a ultrapeer we must be able to become one..
        if ( isUltrapeer() )
        {
            return true;
        }
        return !isShieldedLeafNode() && ( ConnectionPrefs.AllowToBecomeUP.get().booleanValue() &&
            upChecker.isUltrapeerCapable() );
    }

    /**
     * Returns true if this node is currently a ultrapeer, false otherwise.
     * This node is currently a ultrapeer if it is forced to be a ultrapeer or
     * has leaf connections.
     * @return true if the node is currently a ultrapeer, false otherwise.
     */
    public boolean isUltrapeer()
    {
        return ( ConnectionPrefs.AllowToBecomeUP.get().booleanValue() &&
            ConnectionPrefs.ForceToBeUltrapeer.get().booleanValue() ) ||
            // if we have leaf connections we are a ultrapeer.
            networkHostsContainer.hasLeafConnections();
    }

    /**
     * Indicates if the peer advertises through pongs that it has incomming slots
     * available.
     * @return true if it will advertise incomming slots. False otherwise.
     */
    public boolean areIncommingSlotsAdvertised()
    {
        if ( networkHostsContainer.isShieldedLeafNode() )
        {   // when shielded leaf we dont like many incomming request therefore
            // we claim to not have any slots available...
            return false;
        }
        return networkHostsContainer.hasUltrapeerSlotsAvailable() ||
            networkHostsContainer.hasLeafSlotsAvailable();
    }


    /**
     * The method checks if we are able to go into leaf state. This is
     * necessary to react accordingly to the "X-UltrapeerNeeded: false" header.
     *
     * @return true if we are able to switch to Leaf state, false otherwise.
     */
    public boolean isAbleToBecomeLeafNode()
    {
        // we are not able to become a leaf if we are able to become a ultrapeer
        // this includes that we might already are a ultrapeer and we have any
        // leaf or ultrapeer connections.
        if ( isAbleToBecomeUltrapeer() &&
            ( networkHostsContainer.hasLeafConnections() ||
              networkHostsContainer.hasUltrapeerConnections() ) )
        {
            return false;
        }
        return true;
    }

    public void addConnectedHost( Host host )
    {
        networkHostsContainer.addConnectedHost( host );
    }

    public void disconnectHost( Host host )
    {
        networkHostsContainer.disconnectHost( host );
    }

    public void addIncomingHost( Host host )
    {
        networkHostsContainer.addIncomingHost( host );
    }

    public void removeAllNetworkHosts()
    {
        networkHostsContainer.removeAllNetworkHosts();
    }

    public void removeNetworkHosts( Host[] hosts )
    {
        networkHostsContainer.removeNetworkHosts( hosts );
    }

    public void removeNetworkHost( Host host )
    {
        networkHostsContainer.removeNetworkHost( host );
    }


    public void addNetworkHostsChangeListener( NetworkHostsChangeListener listener )
    {
        networkHostsContainer.addNetworkHostsChangeListener( listener );
    }

    public void removeNetworkHostsChangeListener( NetworkHostsChangeListener listener )
    {
        networkHostsContainer.removeNetworkHostsChangeListener( listener );
    }
    /////////////// END NetworkHostsContainer wrapper methods ///////////////////////


    public void doAutoConnectCheck()
    {
        if ( !networkMgr.isNetworkJoined() || !networkMgr.isConnected() )
        {
            return;
        }

        int hostCount;
        int requiredHostCount;

        if ( isAbleToBecomeUltrapeer() )
        {
            // as a ultrapeer I'm primary searching for Ultrapeers only...
            // to make sure I'm well connected...
            hostCount = networkHostsContainer.getUltrapeerConnectionCount();
            requiredHostCount = ConnectionPrefs.Up2UpConnections.get().intValue();
        }
        // we dont support legacy peers anymore ( since 3.0 ) therefore we only
        // handle leaf mode here
        else
        {
            // as a leaf I'm primary searching for Ultrapeers only...
            hostCount = networkHostsContainer.getUltrapeerConnectionCount();
            requiredHostCount = ConnectionPrefs.Leaf2UpConnections.get().intValue();
        }

        // count the number of missing connection tryes this is the required count
        // minus the available count. The result is multiplied by four to raise the
        // connection try count.
        int missingCount = ( requiredHostCount - hostCount ) * 4;
        
        // find out the number of hosts where a connection is currently tried...
        int allHostCount = networkHostsContainer.getNetworkHostCount( );
        int errorHostCount = networkHostsContainer.getNetworkHostCount(
            HostStatus.ERROR);
        // make sure the value is not negative.
        int totalCount = networkHostsContainer.getTotalConnectionCount();
        int currentTryCount = Math.max( 0, allHostCount - totalCount - errorHostCount);
        
        // we will never try more then a reasonable parallel tries..
        int upperLimit = Math.min( MAX_PARALLEL_CONNECTION_TRIES,
            NetworkPrefs.MaxConcurrentConnectAttempts.get().intValue() ) - currentTryCount;
        
        int outConnectCount = Math.min( missingCount-currentTryCount, 
            upperLimit );
        if ( outConnectCount > 0 )
        {
            NLogger.debug( HostManager.class, 
                "Auto-connect to " + outConnectCount + " new hosts.");
            OutgoingConnectionDispatcher.dispatchConnectToNextHosts( outConnectCount );
        }
    }    
    
    private class HostCheckTimer extends TimerTask
    {

        public static final long TIMER_PERIOD = 2000;

        /**
         * @see java.util.TimerTask#run()
         */
        public void run()
        {
            try
            {
                doAutoConnectCheck();
                networkHostsContainer.periodicallyCheckHosts();
            }
            catch ( Throwable th )
            {
                NLogger.error( HostManager.class, th, th );
            }
        }
    }
}