/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2007 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- SVN Information ---
 *  $Id: OIOServer.java 3859 2007-07-01 20:15:19Z gregork $
 */
package phex.net;

import java.io.IOException;
import java.net.BindException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;

import phex.common.ThreadPool;
import phex.common.address.DefaultDestAddress;
import phex.common.address.DestAddress;
import phex.common.address.IpAddress;
import phex.connection.IncomingConnectionDispatcher;
import phex.connection.NetworkManager;
import phex.host.HostManager;
import phex.host.NetworkHostsContainer;
import phex.net.repres.PresentationManager;
import phex.net.repres.SocketFacade;
import phex.net.repres.def.DefaultSocketFacade;
import phex.prefs.core.NetworkPrefs;
import phex.security.AccessType;
import phex.security.PhexSecurityException;
import phex.security.PhexSecurityManager;
import phex.utils.NLogger;

public class OIOServer extends Server
{
    public OIOServer()
    {
        super();
    }

    // The listening thread.
    public void run()
    {
        if (NLogger.isDebugEnabled( OIOServer.class ))
            NLogger.debug( OIOServer.class,
                "Listener started. Listening on: "
                    + serverSocket.getInetAddress().getHostAddress() + ':'
                    + serverSocket.getLocalPort());
        
        try
        {
            while ( serverSocket != null && !serverSocket.isClosed() )
            {  
                try
                {
                    Socket incoming = serverSocket.accept();
                    // create facade...
                    DefaultSocketFacade incomingFacade = new DefaultSocketFacade( 
                        incoming );
                    handleIncomingSocket( incomingFacade );
                }
                catch ( SocketException exp )
                {
                    NLogger.debug( OIOServer.class, exp );
                }
                catch ( PhexSecurityException exp )
                {
                    NLogger.debug( OIOServer.class, exp );
                }
                catch (IOException exp)
                {
                    NLogger.error( OIOServer.class, exp, exp);
                }
            }
        }
        catch ( Exception exp )
        {
            NLogger.error( OIOServer.class, exp, exp );
        }

        isRunning = false;
        NLogger.debug( OIOServer.class, "Listener stopped.");
        DestAddress address = PresentationManager.getInstance().createHostAddress(
            new IpAddress(IpAddress.LOCAL_HOST_IP), DefaultDestAddress.DEFAULT_PORT );
        NetworkManager.getInstance().updateLocalAddress( address );
        synchronized(this)
        {
            notifyAll();
        }
    }

    /**
     * @param socketChannel
     * @throws PhexSecurityException
     * @throws SocketException
     */
    private void handleIncomingSocket(SocketFacade clientSocket )
        throws IOException, PhexSecurityException
    {        
        clientSocket.setSoTimeout( NetworkPrefs.TcpRWTimeout.get().intValue() );

        DestAddress address = clientSocket.getRemoteAddress();
        NetworkHostsContainer netHostsContainer = HostManager.getInstance()
            .getNetworkHostsContainer();

        // if not already connected and connection is not from a private address.
        IpAddress remoteIp = address.getIpAddress();
        assert remoteIp != null;
        if (!netHostsContainer.isConnectedToHost(address)
            && !remoteIp.isSiteLocalIP() )
        {
            hasConnectedIncomming = true;
            lastInConnectionTime = System.currentTimeMillis();
        }
        
        // Set this will defeat the Nagle Algorithm, making short bursts of
        // transmission faster, but will be worse for the overall network.
        // incoming.setTcpNoDelay(true);

        // Create a Host object for the incoming connection
        // and hand it off to a ReadWorker to handle.
        AccessType access = PhexSecurityManager.getInstance()
            .controlHostAddressAccess(address);
        switch (access)
        {
            case ACCESS_DENIED:
            case ACCESS_STRONGLY_DENIED:
                throw new PhexSecurityException("Host access denied: " + address );
        }

        NLogger.debug( OIOServer.class, 
            "Accepted incoming connection from: "
                + address.getFullHostName());

        IncomingConnectionDispatcher dispatcher = new IncomingConnectionDispatcher(
            clientSocket );
        ThreadPool.getInstance().addJob( dispatcher,
            "IncomingConnectionDispatcher-" + Integer.toHexString(hashCode()));
    }

    @Override
    protected synchronized void bind() throws IOException
    {
        assert (serverSocket == null);

        int port = NetworkPrefs.ListeningPort.get().intValue();
        serverSocket = new ServerSocket();

        // Create a listening socket at the port.
        int tries = 0;
        boolean error;
        // try to find new port if port not valid
        do
        {
            error = false;

            try
            {
                NLogger.debug( OIOServer.class, "Binding to port " + port);
                serverSocket.bind(new InetSocketAddress(port));
            }
            catch (BindException exp)
            {
                NLogger.debug( OIOServer.class, "Binding failed to port " + port);
                if (tries > 50)
                {
                    throw exp;
                }
                error = true;
                port++;
                tries++;
            }
        }
        while (error == true);

        IpAddress hostIP = resolveLocalHostIP();
        port = serverSocket.getLocalPort();
        DestAddress address = PresentationManager.getInstance().createHostAddress(
            hostIP, port);
        NetworkManager.getInstance().updateLocalAddress( address );
    }
    
    @Override
    protected synchronized void closeServer()
    {
        if ( serverSocket != null )
        {
            try
            {
                serverSocket.close();
            }
            catch (IOException exp)
            {// ignore
            }
            serverSocket = null;
        }
    }
}