    /**
     * Eosコマンドをエミュレートするクラス 
     * @constructor
     * @returns {object}
     * function execute(command, params) {
     */
function Eos() {

    /**
     * include all Eos command's info.
     * For seaching with O(n), the object key name is command name.
     */
    const commandReference = {
        mrcImageInfo: { },
        dcdFilePrint: { }
    };

    const workspace = ['file1.txt', 'file2.txt'];

    return {
        /**
         * execute
         *
         * @param command
         * @param params
         * @returns {object}
         */
        execute: function(command, options) {
            var result = {
                hasDone: false, // true | false
                comment: ''// string
            };

            /**
             * Check of command name
             */
            if(typeof command !== 'string') {
                result.comment = 'Command parameter need to be string';
                return result;
            }

            const hasCommand = Object.keys(commandReference).indexOf(command) > -1;
            if(!hasCommand) {
                result.comment = 'Command name is invalid';
                return result;
            }
            
            /**
             * Check of options
             */

            if(!(Array.isArray(options))) {
                result.comment = 'Options need to be Array';
                return result;
            }

            
            if(options.length === 0) {
                result.comment = 'At least one option is required.';
                return result;
            }

            // translate options to key-value and check whether options include correct member
            const optionsObj = {};
            var hasCorrectMember = true;
            var isArgumentsArray = true;

            options.forEach(function(o) {
                if(!(o.name) && !(o.arguments)) {
                    hasCorrectMember = false;
                } else {
                    if(Array.isArray(o.arguments)) {
                        optionsObj[o.name] = o.arguments;
                    } else {
                        isArgumentsArray = false;
                    }
                }
            });

            // check each object has proberties "name" and "argumets"
            if(!hasCorrectMember) {
                result.comment = 'Options need to include Object which have properties "name" and "arguments"';
                return result;
            }

            // check each "argumets" properties is Array
            if(!isArgumentsArray) {
                result.comment = 'Each "arguments" properties needs to be Array';
                return result;
            }

            // Read OptionControlFile info of command
            const ocf = require(__dirname + '/../../user-specific-files/OptionControlFile/commands/' + command);
            var ocfObj = {};

            // translate ocf info to key-value
            var notIncludingRequiredOptions = [];
            ocf.forEach(function(o) {
                if(o.optionProperties && Object.keys(optionsObj).indexOf(o.option) < 0) {
                    notIncludingRequiredOptions.push(o.option);
                }
                ocfObj[o.option] = o;
            });

            // check whether all required option exist 
            if(notIncludingRequiredOptions.length > 0) {
                result.comment = 'Option ' + notIncludingRequiredOptions.toString() + ' are required';
                return result;
            }

            var invalidArgumentsNumber= [];
            var invalidArgumentType = [];
            var invalidOutputFileName = [];

            // output file Regexp
            const outRegExp = /out|append/;

            options.forEach(function(o) {
                // option number
                const expectNum = ocfObj[o.name].optionNumber;
                const actualNum = o.arguments.length;
                if(expectNum !== actualNum) {
                    invalidArgumentsNumber.push({name: o.name, expect: expectNum, actual: actualNum});
                }


                // argType and outFile name
                o.arguments.forEach(function(arg,i) {
                    // argType
                    var formType = ocfObj[o.name].arg[i].formType
                    if(formType === 'select') { // This argument is filename
                        var exist = workspace.indexOf(arg) > -1;
                        if(!exist) {
                            invalidArgumentType.push({name: o.name, file: arg});
                        }
                    } else {
                        const expectType = formType === 'text' ? 'string' : 'number';
                        const actualType = typeof arg;
                        if(expectType !== actualType) {
                            invalidArgumentType.push({name: o.name, expect: expectType, actual: actualType});
                        }
                    }

                    // outFile name
                    if(outRegExp.test(ocfObj[o.name].arg[i].argType)) {
                        if(workspace.indexOf(o.arguments[i]) > -1) {
                            invalidOutputFileName.push({name: o.name, file: arg});
                        }
                    }
                });
            });

            // check arguments number value
            if(invalidArgumentsNumber.length > 0) {
                result.comment = '"arguments" properties is invalid number.\n';
                invalidArgumentsNumber.forEach(function(i) {
                    result.comment += '     ' + i.name + ' expect to  ' + i.expect + ', but actual ' + i.actual + '.\n';
                });
                return result;
            }

            // check arguments type
            if(invalidArgumentType.length > 0) {
                result.comment = '"arguments" type is invalid.\n';
                invalidArgumentType.forEach(function(i) {
                    if(i.file) {
                        result.comment += '     ' + i.name + ' ' + i.file + ' does not exist.\n';
                    } else {
                        result.comment += '     ' + i.name + ' expect to ' + i.expect + ', but actual ' + i.actual + '.\n';
                    }
                });
                return result;
            }

            // check outFile name
            if(invalidOutputFileName.length > 0) {
                result.comment = 'output file name is invalid.\n';
                invalidOutputFileName.forEach(function(i) {
                    result.comment += '     ' + i.name + ' ' + i.file + ' has already existed.\n';
                });
                return result;
            }

            // set default parameters
            var finalOptions = {};
            ocf.forEach(function(o) {
                o.arg.forEach(function(arg) {
                    if(!(arg.initialValue === "") && arg.initialValue) {
                        if(!(finalOptions[o.option])) {
                            finalOptions[o.option] = [];
                            finalOptions[o.option].push(arg.initialValue);
                        } else {
                            finalOptions[o.option].push(arg.initialValue);
                        }
                    }
                });
            });
            
            // set user setting parameters
            options.forEach(function(o) {
                finalOptions[o.name] = o.arguments;
            });

            // set execution string
            var execStr = command + ' ';
            Object.keys(finalOptions).forEach(function(key) {
                execStr += key + ' ';
                finalOptions[key].forEach(function(arg) {
                    execStr += arg + ' ';
                });
            });

            // remove last blank
            execStr = execStr.slice(0,execStr.length-1);
            
            /**
             * End in Success
             */
            result.hasDone = true;
            result.comment = command + ' has done.';
            return result;
        }
    }
}

module.exports = Eos;
