/*
 * This is a plug-in for the GIMP.
 *
 * Generates clickable image maps.
 *
 * Copyright (C) 1998-1999 Maurits Rijk  lpeek.mrijk@consunet.nl
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <math.h>

#include "imap_circle.h"
#include "imap_main.h"
#include "imap_misc.h"
#include "imap_object_popup.h"
#include "imap_table.h"

#include "circle.xpm"

static gboolean circle_is_valid(Object_t *obj);
static Object_t *circle_clone(Object_t *obj);
static Object_t *circle_assign(Object_t *obj, Object_t *des);
static void circle_draw(Object_t* obj, GdkWindow *window, GdkGC* gc);
static void circle_draw_sashes(Object_t* obj, GdkWindow *window, GdkGC* gc);
static MoveSashFunc_t circle_near_sash(Object_t *obj, gint x, gint y);
static gboolean circle_point_is_on(Object_t *obj, gint x, gint y);
static void circle_get_dimensions(Object_t *obj, gint *x, gint *y,
				  gint *width, gint *height);
static void circle_resize(Object_t *obj, gint percentage_x, gint percentage_y);
static void circle_move(Object_t *obj, gint dx, gint dy);
static gpointer circle_create_info_tab(GtkWidget *notebook);
static void circle_fill_info_tab(Object_t *obj, gpointer data);
static void circle_set_initial_focus(Object_t *obj, gpointer data);
static void circle_update(Object_t* obj, gpointer data);
static void circle_write_csim(Object_t* obj, gpointer param, 
			      OutputFunc_t output);
static void circle_write_cern(Object_t* obj, gpointer param, 
			      OutputFunc_t output);
static void circle_write_ncsa(Object_t* obj, gpointer param, 
			      OutputFunc_t output);
static char** circle_get_icon_data(void);

static ObjectClass_t circle_class = {
   NULL,			/* info_dialog */
   NULL,			/* icon */
   NULL,			/* mask */

   circle_is_valid,
   NULL,			/* circle_destruct */
   circle_clone,
   circle_assign,
   NULL,			/* circle_normalize */
   circle_draw,
   circle_draw_sashes,
   circle_near_sash,
   circle_point_is_on,
   circle_get_dimensions,
   circle_resize,
   circle_move,
   circle_create_info_tab,
   circle_fill_info_tab,
   circle_set_initial_focus,
   circle_update,
   circle_write_csim,
   circle_write_cern,
   circle_write_ncsa,
   object_do_popup,
   circle_get_icon_data
};

Object_t*
create_circle(gint x, gint y, gint r)
{
   Circle_t *circle = g_new(Circle_t, 1);
   circle->x = x;
   circle->y = y;
   circle->r = r;
   return object_init(&circle->obj, &circle_class);
}

static gboolean
circle_is_valid(Object_t *obj)
{
   return ObjectToCircle(obj)->r > 0;
}

static Object_t*
circle_clone(Object_t *obj)
{
   Circle_t *circle = ObjectToCircle(obj);
   Circle_t *clone = g_new(Circle_t, 1);

   clone->x = circle->x;
   clone->y = circle->y;
   clone->r = circle->r;
   return &clone->obj;
}

static Object_t*
circle_assign(Object_t *obj, Object_t *des)
{
   Circle_t *src_circle = ObjectToCircle(obj);
   Circle_t *des_circle = ObjectToCircle(des);
   des_circle->x = src_circle->x;
   des_circle->y = src_circle->y;
   des_circle->r = src_circle->r;
   return object_copy(obj, des);
}

static void
circle_draw(Object_t *obj, GdkWindow *window, GdkGC *gc)
{
   Circle_t *circle = ObjectToCircle(obj);
   draw_circle(window, gc, FALSE, circle->x, circle->y, circle->r);
}

static void
circle_draw_sashes(Object_t *obj, GdkWindow *window, GdkGC *gc)
{
   Circle_t *circle = ObjectToCircle(obj);
   draw_sash(window, gc, circle->x - circle->r, circle->y - circle->r);
   draw_sash(window, gc, circle->x + circle->r, circle->y - circle->r);
   draw_sash(window, gc, circle->x - circle->r, circle->y + circle->r);
   draw_sash(window, gc, circle->x + circle->r, circle->y + circle->r);
}

static gint sash_x;
static gint sash_y;

static void
move_sash(Object_t *obj, gint dx, gint dy)
{
   Circle_t *circle = ObjectToCircle(obj);
   gint rx, ry;
   sash_x += dx;
   sash_y += dy;

   rx = abs(circle->x - sash_x);
   ry = abs(circle->y - sash_y);
   circle->r = (rx > ry) ? rx : ry;
}

static void 
circle_resize(Object_t *obj, gint percentage_x, gint percentage_y)
{
   Circle_t *circle = ObjectToCircle(obj);
   circle->x = circle->x * percentage_x / 100;
   circle->y = circle->y * percentage_y / 100;
   circle->r = circle->r * ((percentage_x < percentage_y) 
			    ? percentage_x : percentage_y) / 100;
}

static MoveSashFunc_t
circle_near_sash(Object_t *obj, gint x, gint y)
{
   Circle_t *circle = ObjectToCircle(obj);
   sash_x = x;
   sash_y = y;
   if (near_sash(circle->x - circle->r, circle->y - circle->r, x, y) ||
       near_sash(circle->x + circle->r, circle->y - circle->r, x, y) ||
       near_sash(circle->x - circle->r, circle->y + circle->r, x, y) ||
       near_sash(circle->x + circle->r, circle->y + circle->r, x, y))
      return move_sash;
   return NULL;
}

static gboolean
circle_point_is_on(Object_t *obj, gint x, gint y)
{
   Circle_t *circle = ObjectToCircle(obj);
   x -= circle->x;
   y -= circle->y;
   return x * x + y * y <= circle->r * circle->r;
}

static void 
circle_get_dimensions(Object_t *obj, gint *x, gint *y,
		      gint *width, gint *height)
{
   Circle_t *circle = ObjectToCircle(obj);
   *x = circle->x - circle->r;
   *y = circle->y - circle->r;
   *width = *height = 2 * circle->r;
}

static void
circle_move(Object_t *obj, gint dx, gint dy)
{
   Circle_t *circle = ObjectToCircle(obj);
   circle->x += dx;
   circle->y += dy;
}

typedef struct {
   GtkWidget *x;
   GtkWidget *y;
   GtkWidget *r;
} CircleProperties_t;

static gpointer
circle_create_info_tab(GtkWidget *notebook)
{
   CircleProperties_t *props = g_new(CircleProperties_t, 1);
   GtkWidget *vbox, *table, *label;
   gint max_width = get_image_width();
   gint max_height = get_image_height();

   vbox = gtk_vbox_new(FALSE, 1);
   gtk_widget_show(vbox);

   table = gtk_table_new(3, 3, FALSE);
   gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(table), 10);

   gtk_table_set_row_spacings(GTK_TABLE(table), 10);
   gtk_table_set_col_spacings(GTK_TABLE(table), 10);
   gtk_widget_show(table);
   
   create_label_in_table(table, 0, 0, "Center x:");
   props->x = create_spin_button_in_table(table, 0, 1, 1, 0, max_width - 1);
   create_label_in_table(table, 0, 2, "pixels");

   create_label_in_table(table, 1, 0, "Center y:");
   props->y = create_spin_button_in_table(table, 1, 1, 1, 0, max_height - 1);
   create_label_in_table(table, 1, 2, "pixels");

   create_label_in_table(table, 2, 0, "Radius:");
   props->r = create_spin_button_in_table(table, 2, 1, 1, 1, G_MAXINT);
   create_label_in_table(table, 2, 2, "pixels");

   label = gtk_label_new("Circle");
   gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
   
   return props;
}

static void 
circle_fill_info_tab(Object_t *obj, gpointer data)
{
   Circle_t *circle = ObjectToCircle(obj);
   CircleProperties_t *props = (CircleProperties_t*) data;

   gtk_spin_button_set_value(GTK_SPIN_BUTTON(props->x), circle->x);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(props->y), circle->y);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(props->r), circle->r);
}

static void 
circle_set_initial_focus(Object_t *obj, gpointer data)
{
   CircleProperties_t *props = (CircleProperties_t*) data;
   gtk_widget_grab_focus(props->x);
}

static void 
circle_update(Object_t* obj, gpointer data)
{
   Circle_t *circle = ObjectToCircle(obj);
   CircleProperties_t *props = (CircleProperties_t*) data;

   circle->x = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(props->x));
   circle->y = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(props->y));
   circle->r = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(props->r));
}

static void
circle_write_csim(Object_t *obj, gpointer param, OutputFunc_t output)
{
   Circle_t *circle = ObjectToCircle(obj);
   output(param, "\"CIRCLE\" COORDS=\"%d,%d,%d\"", circle->x, circle->y,
	  circle->r);
}

static void
circle_write_cern(Object_t *obj, gpointer param, OutputFunc_t output)
{
   Circle_t *circle = ObjectToCircle(obj);
   output(param, "circ (%d,%d) %d", circle->x, circle->y, circle->r);
}

static void
circle_write_ncsa(Object_t *obj, gpointer param, OutputFunc_t output)
{
   Circle_t *circle = ObjectToCircle(obj);
   output(param, "circle %s %d,%d %d,%d", obj->url,
	  circle->x, circle->y, circle->x, circle->y + circle->r);
}

static char** 
circle_get_icon_data(void)
{
   return circle_xpm;
}

static gint _start_x, _start_y;

static Object_t*
circle_factory_create_object1(gint x, gint y)
{
   _start_x = x;
   _start_y = y;
   return create_circle(x, y, 0);
}

static void
circle_factory_set_xy1(Object_t *obj, guint state, gint x, gint y)
{
   Circle_t *circle = ObjectToCircle(obj);

   circle->x = (_start_x + x) / 2;
   circle->y = (_start_y + y) / 2;
   x -= _start_x;
   y -= _start_y;
   circle->r = (gint) sqrt(x * x + y * y) / 2;

   main_set_dimension(circle->r, circle->r);
}

static ObjectFactory_t circle_factory1 = {
   NULL,			/* Object pointer */
   NULL,			/* Finish func */
   NULL,			/* Cancel func */
   circle_factory_create_object1,
   circle_factory_set_xy1
};

static Object_t*
circle_factory_create_object2(gint x, gint y)
{
   return create_circle(x, y, 0);
}

static void
circle_factory_set_xy2(Object_t *obj, guint state, gint x, gint y)
{
   Circle_t *circle = ObjectToCircle(obj);

   x -= circle->x;
   y -= circle->y;
   circle->r = (gint) sqrt(x * x + y * y);

   main_set_dimension(circle->r, circle->r);
}

static ObjectFactory_t circle_factory2 = {
   NULL,			/* Object pointer */
   NULL,			/* Finish func */
   NULL,			/* Cancel func */
   circle_factory_create_object2,
   circle_factory_set_xy2
};

ObjectFactory_t*
get_circle_factory(guint state)
{
   return (state & GDK_SHIFT_MASK) ? &circle_factory1 : &circle_factory2;
}
