//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndTwofish.cpp
 * @brief		twofishÍNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FndTwofish_CPP_

//======================================================================
// include
#include "FndTwofish.h"
#include <string.h>	// for mem***
#include "../../misc/iris_allegrex.h"
#include "../../iris_debug.h"

namespace iris {
namespace fnd
{

//======================================================================
// define
#define FEISTEL		0	//!< != 0 : FEISTEL\ŃEh(slow)

// word -> byte (GfBAlāALSB̃I[_[Ŏ擾)
#define b0(x)			IRIS_DWord2Byte(x, 0)	//!< LSB
#define b1(x)			IRIS_DWord2Byte(x, 1)
#define b2(x)			IRIS_DWord2Byte(x, 2)
#define b3(x)			IRIS_DWord2Byte(x, 3)	//!< MSB

// 
#define ROL(x, n)		iris_allegrex_rotl(x, n)
#define ROR(x, n)		iris_allegrex_rotr(x, n)

// sub key
#define SUBKEY_STEP		0x02020202u
#define SUBKEY_BUMP		0x01010101u
#define SUBKEY_ROTL		9

// read solomon
#define	RS_GF_FDBK		0x14D		/* field generator */
#define	RS_rem(x)		{ \
	u8  b  = (u8) (x >> 24);													\
	u32 g2 = (u32)(((b << 1) ^ ((b & 0x80) ? RS_GF_FDBK : 0 )) & 0xFF);			\
	u32 g3 = (u32)(((b >> 1) & 0x7F) ^ ((b & 1) ? RS_GF_FDBK >> 1 : 0 ) ^ g2);	\
	x = (u32)((x << 8) ^ (g3 << 24) ^ (g2 << 16) ^ (g3 << 8) ^ b);				\
	}

// MDS Matrix
#define MDS_GF_FDBK		0x169	/* primitive polynomial for GF(256)*/
#define LFSR1(x)		( ((x) >> 1) ^ (((x) & 0x01) ?   MDS_GF_FDBK/2 : 0))
#define LFSR2(x)		( ((x) >> 2) ^ (((x) & 0x02) ?   MDS_GF_FDBK/2 : 0)  \
									 ^ (((x) & 0x01) ?   MDS_GF_FDBK/4 : 0))

#define Mtx_1(x)		((u32)(x))
#define Mtx_X(x)		((u32)((x) ^ LFSR1(x)))
#define Mtx_Y(x)		((u32)((x) ^ LFSR1(x) ^ LFSR2(x)))

#define M00		Mtx_1
#define M01		Mtx_Y
#define M02		Mtx_X
#define M03		Mtx_X

#define M10		Mtx_X
#define M11		Mtx_Y
#define M12		Mtx_Y
#define M13		Mtx_1

#define M20		Mtx_Y
#define M21		Mtx_X
#define M22		Mtx_1
#define M23		Mtx_Y

#define M30		Mtx_Y
#define M31		Mtx_1
#define M32		Mtx_Y
#define M33		Mtx_X

// P-Box̎Q
#define P_00	1
#define P_01	0
#define P_02	0
#define P_03	(P_01^1)
#define P_04	1

#define P_10	0
#define P_11	0
#define P_12	1
#define P_13	(P_11^1)
#define P_14	0

#define P_20	1
#define P_21	1
#define P_22	0
#define P_23	(P_21^1)
#define P_24	0

#define P_30	0
#define P_31	1
#define P_32	1
#define P_33	(P_31^1)
#define P_34	1

#define PB(n)	s_PBox[P_##n]

//======================================================================
// variable
static const u8 s_PBox[2][256] = {
	{
		0xA9, 0x67, 0xB3, 0xE8, 0x04, 0xFD, 0xA3, 0x76, 
		0x9A, 0x92, 0x80, 0x78, 0xE4, 0xDD, 0xD1, 0x38, 
		0x0D, 0xC6, 0x35, 0x98, 0x18, 0xF7, 0xEC, 0x6C, 
		0x43, 0x75, 0x37, 0x26, 0xFA, 0x13, 0x94, 0x48, 
		0xF2, 0xD0, 0x8B, 0x30, 0x84, 0x54, 0xDF, 0x23, 
		0x19, 0x5B, 0x3D, 0x59, 0xF3, 0xAE, 0xA2, 0x82, 
		0x63, 0x01, 0x83, 0x2E, 0xD9, 0x51, 0x9B, 0x7C, 
		0xA6, 0xEB, 0xA5, 0xBE, 0x16, 0x0C, 0xE3, 0x61, 
		0xC0, 0x8C, 0x3A, 0xF5, 0x73, 0x2C, 0x25, 0x0B, 
		0xBB, 0x4E, 0x89, 0x6B, 0x53, 0x6A, 0xB4, 0xF1, 
		0xE1, 0xE6, 0xBD, 0x45, 0xE2, 0xF4, 0xB6, 0x66, 
		0xCC, 0x95, 0x03, 0x56, 0xD4, 0x1C, 0x1E, 0xD7, 
		0xFB, 0xC3, 0x8E, 0xB5, 0xE9, 0xCF, 0xBF, 0xBA, 
		0xEA, 0x77, 0x39, 0xAF, 0x33, 0xC9, 0x62, 0x71, 
		0x81, 0x79, 0x09, 0xAD, 0x24, 0xCD, 0xF9, 0xD8, 
		0xE5, 0xC5, 0xB9, 0x4D, 0x44, 0x08, 0x86, 0xE7, 
		0xA1, 0x1D, 0xAA, 0xED, 0x06, 0x70, 0xB2, 0xD2, 
		0x41, 0x7B, 0xA0, 0x11, 0x31, 0xC2, 0x27, 0x90, 
		0x20, 0xF6, 0x60, 0xFF, 0x96, 0x5C, 0xB1, 0xAB, 
		0x9E, 0x9C, 0x52, 0x1B, 0x5F, 0x93, 0x0A, 0xEF, 
		0x91, 0x85, 0x49, 0xEE, 0x2D, 0x4F, 0x8F, 0x3B, 
		0x47, 0x87, 0x6D, 0x46, 0xD6, 0x3E, 0x69, 0x64, 
		0x2A, 0xCE, 0xCB, 0x2F, 0xFC, 0x97, 0x05, 0x7A, 
		0xAC, 0x7F, 0xD5, 0x1A, 0x4B, 0x0E, 0xA7, 0x5A, 
		0x28, 0x14, 0x3F, 0x29, 0x88, 0x3C, 0x4C, 0x02, 
		0xB8, 0xDA, 0xB0, 0x17, 0x55, 0x1F, 0x8A, 0x7D, 
		0x57, 0xC7, 0x8D, 0x74, 0xB7, 0xC4, 0x9F, 0x72, 
		0x7E, 0x15, 0x22, 0x12, 0x58, 0x07, 0x99, 0x34, 
		0x6E, 0x50, 0xDE, 0x68, 0x65, 0xBC, 0xDB, 0xF8, 
		0xC8, 0xA8, 0x2B, 0x40, 0xDC, 0xFE, 0x32, 0xA4, 
		0xCA, 0x10, 0x21, 0xF0, 0xD3, 0x5D, 0x0F, 0x00, 
		0x6F, 0x9D, 0x36, 0x42, 0x4A, 0x5E, 0xC1, 0xE0
	},
	{
		0x75, 0xF3, 0xC6, 0xF4, 0xDB, 0x7B, 0xFB, 0xC8, 
		0x4A, 0xD3, 0xE6, 0x6B, 0x45, 0x7D, 0xE8, 0x4B, 
		0xD6, 0x32, 0xD8, 0xFD, 0x37, 0x71, 0xF1, 0xE1, 
		0x30, 0x0F, 0xF8, 0x1B, 0x87, 0xFA, 0x06, 0x3F, 
		0x5E, 0xBA, 0xAE, 0x5B, 0x8A, 0x00, 0xBC, 0x9D, 
		0x6D, 0xC1, 0xB1, 0x0E, 0x80, 0x5D, 0xD2, 0xD5, 
		0xA0, 0x84, 0x07, 0x14, 0xB5, 0x90, 0x2C, 0xA3, 
		0xB2, 0x73, 0x4C, 0x54, 0x92, 0x74, 0x36, 0x51, 
		0x38, 0xB0, 0xBD, 0x5A, 0xFC, 0x60, 0x62, 0x96, 
		0x6C, 0x42, 0xF7, 0x10, 0x7C, 0x28, 0x27, 0x8C, 
		0x13, 0x95, 0x9C, 0xC7, 0x24, 0x46, 0x3B, 0x70, 
		0xCA, 0xE3, 0x85, 0xCB, 0x11, 0xD0, 0x93, 0xB8, 
		0xA6, 0x83, 0x20, 0xFF, 0x9F, 0x77, 0xC3, 0xCC, 
		0x03, 0x6F, 0x08, 0xBF, 0x40, 0xE7, 0x2B, 0xE2, 
		0x79, 0x0C, 0xAA, 0x82, 0x41, 0x3A, 0xEA, 0xB9, 
		0xE4, 0x9A, 0xA4, 0x97, 0x7E, 0xDA, 0x7A, 0x17, 
		0x66, 0x94, 0xA1, 0x1D, 0x3D, 0xF0, 0xDE, 0xB3, 
		0x0B, 0x72, 0xA7, 0x1C, 0xEF, 0xD1, 0x53, 0x3E, 
		0x8F, 0x33, 0x26, 0x5F, 0xEC, 0x76, 0x2A, 0x49, 
		0x81, 0x88, 0xEE, 0x21, 0xC4, 0x1A, 0xEB, 0xD9, 
		0xC5, 0x39, 0x99, 0xCD, 0xAD, 0x31, 0x8B, 0x01, 
		0x18, 0x23, 0xDD, 0x1F, 0x4E, 0x2D, 0xF9, 0x48, 
		0x4F, 0xF2, 0x65, 0x8E, 0x78, 0x5C, 0x58, 0x19, 
		0x8D, 0xE5, 0x98, 0x57, 0x67, 0x7F, 0x05, 0x64, 
		0xAF, 0x63, 0xB6, 0xFE, 0xF5, 0xB7, 0x3C, 0xA5, 
		0xCE, 0xE9, 0x68, 0x44, 0xE0, 0x4D, 0x43, 0x69, 
		0x29, 0x2E, 0xAC, 0x15, 0x59, 0xA8, 0x0A, 0x9E, 
		0x6E, 0x47, 0xDF, 0x34, 0x35, 0x6A, 0xCF, 0xDC, 
		0x22, 0xC9, 0xC0, 0x9B, 0x89, 0xD4, 0xED, 0xAB, 
		0x12, 0xA2, 0x0D, 0x52, 0xBB, 0x02, 0x2F, 0xA9, 
		0xD7, 0x61, 0x1E, 0xB4, 0x50, 0x04, 0xF6, 0xC2, 
		0x16, 0x25, 0x86, 0x56, 0x55, 0x09, 0xBE, 0x91
	}
};

//======================================================================
// declare
static u32 _ReadSolomon_MDS_Encode(u32 k0, u32 k1);

//======================================================================
// function
u32 _ReadSolomon_MDS_Encode(u32 k0, u32 k1)
{
	u32 ret = 0;
	for( int i=0; i < 2; ++i )
	{
		ret ^= i ? k0 : k1;
		for( int j=4; j > 0; --j )
		{
			RS_rem(ret);
		}
	}
	return ret;
}

//======================================================================
// class

/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
template<u32 _KLB>
CTwofish<_KLB>::CTwofish(void)
{
}

/**********************************************************************//**
 *
 * ̍쐬
 *
 -----------------------------------------------------------------------
 * @param [in]	pKey	= 
 * @param [in]	size	= z
*//***********************************************************************/
template<u32 _KLB>
bool CTwofish<_KLB>::CreateKeys(const u8 *pKey, u32 nSize)
{
	IRIS_ASSERT( pKey != nullptr );
	IRIS_ASSERT( nSize >= KEYSIZE );
	const u32* key = (const u32*)pKey;
	int i;
	int subkey_cnt = KS_LENGTH/2;
	u32 A, B;
	u32 ke[NK64], ko[NK64];
	u32* ks = m_KeySch;

	// S-Box
	for( i=0; i < NK64; ++i )
	{
		ke[i] = *(key++);
		ko[i] = *(key++);
		m_SB[NK64-1-i] = iris::fnd::_ReadSolomon_MDS_Encode(ke[i], ko[i]);
	}

	// input, output, round key
	for( i=0; i < subkey_cnt; ++i )
	{
		A = Substitution(i*SUBKEY_STEP				, ke);
		B = Substitution(i*SUBKEY_STEP+SUBKEY_BUMP	, ko);
		B = ROL(B, 8);
		*(ks++) = A + B;
		*(ks++) = ROL(A+2*B, SUBKEY_ROTL);
	}
	return true;
}

/**********************************************************************//**
 *
 * ÍiubNj
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
*//***********************************************************************/
template<u32 _KLB>
bool CTwofish<_KLB>::EncryptBlock(u8 *dst, const u8 *src)
{
	u32 x[NB];
	u32 t0, t1, tmp;
	u32* pd = (u32*)dst;
	const u32* ps = (const u32*)src;
	u32* ikey = m_KeySch + KS_IN_OFS;
	u32* okey = m_KeySch + KS_OUT_OFS;
	u32* rkey = m_KeySch + KS_ROUND_OFS;
	int i, r;
	for( i=0; i < NB; ++i )
	{
		x[i] = IRIS_ToLittleEndian32(ps[i]) ^ ikey[i];
	}

	for( r=0; r < NR; ++r )
	{
		t0 = Substitution(    x[0]   , m_SB);
		t1 = Substitution(ROL(x[1],8), m_SB);

		x[2]^= t0 +   t1 + rkey[2*r  ]; /* PHT, round keys */
		x[2] = ROR(x[2],1);
		x[3] = ROL(x[3],1);
		x[3]^= t0 + 2*t1 + rkey[2*r+1];

		// swap for next round
		if(r < NR-1)
		{
			tmp = x[0]; x[0]= x[2]; x[2] = tmp;
			tmp = x[1]; x[1]= x[3]; x[3] = tmp;
		}
	}

	for( i=0; i < NB; ++i )
	{
		pd[i] = IRIS_ToLittleEndian32(x[i] ^ okey[i]);
	}
	return true;
}

/**********************************************************************//**
 *
 * iubNj
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
*//***********************************************************************/
template<u32 _KLB>
bool CTwofish<_KLB>::DecryptBlock(u8 *dst, const u8 *src)
{
	u32 x[NB];
	u32 t0, t1, tmp;
	u32* pd = (u32*)dst;
	const u32* ps = (const u32*)src;
	u32* ikey = m_KeySch + KS_IN_OFS;
	u32* okey = m_KeySch + KS_OUT_OFS;
	u32* rkey = m_KeySch + KS_ROUND_OFS;
	int i, r;
	for( i=0; i < NB; ++i )
	{
		x[i] = IRIS_ToLittleEndian32(ps[i]) ^ okey[i];
	}

	for( r=NR-1; r >= 0; --r )
	{
#if	FEISTEL
		t0 = Substitution(ROR(x[0],  ((r+1)>>1)), m_SB);
		t1 = Substitution(ROL(x[1],8+((r+1)>>1)), m_SB);
		x[2] = ROL(t0 +   t1 + rkey[2*r   ], r    >>1);
		x[3] = ROR(t0 + 2*t1 + rkey[2*r+1 ],(r+2) >>1);
#else
		t0 = Substitution(    x[0]   , m_SB);
		t1 = Substitution(ROL(x[1],8), m_SB);

		x[2] = ROL(x[2],1);
		x[2]^= t0 +   t1 + rkey[2*r  ]; /* PHT, round keys */
		x[3]^= t0 + 2*t1 + rkey[2*r+1];
		x[3] = ROR(x[3],1);
#endif
		// swap for next round
		if(r)
		{
			tmp = x[0]; x[0]= x[2]; x[2] = tmp;
			tmp = x[1]; x[1]= x[3]; x[3] = tmp;
		}
	}

#if FEISTEL
	x[0] = ROR(x[0],8);
	x[1] = ROL(x[1],8);
	x[2] = ROR(x[2],8);
	x[3] = ROL(x[3],8);
#endif

	for( i=0; i < NB; ++i )
	{
		pd[i] = IRIS_ToLittleEndian32(x[i] ^ ikey[i]);
	}
	return true;
}

/**********************************************************************//**
 *
 * p
 *
 -----------------------------------------------------------------------
 * @param [in]	x	= pΏ
 * @return p
*//***********************************************************************/
template<>
u32 CTwofish<128>::Substitution(u32 x, const u32* key)
{
	u8 b[4];
	*(u32*)b = IRIS_ToLittleEndian32(x);

	b[0] = (u8)(PB(00)[PB(01)[PB(02)[b[0]] ^ b0(key[1])] ^ b0(key[0])]);
	b[1] = (u8)(PB(10)[PB(11)[PB(12)[b[1]] ^ b1(key[1])] ^ b1(key[0])]);
	b[2] = (u8)(PB(20)[PB(21)[PB(22)[b[2]] ^ b2(key[1])] ^ b2(key[0])]);
	b[3] = (u8)(PB(30)[PB(31)[PB(32)[b[3]] ^ b3(key[1])] ^ b3(key[0])]);

	return	((M00(b[0]) ^ M01(b[1]) ^ M02(b[2]) ^ M03(b[3]))	  ) ^
			((M10(b[0]) ^ M11(b[1]) ^ M12(b[2]) ^ M13(b[3])) <<  8) ^
			((M20(b[0]) ^ M21(b[1]) ^ M22(b[2]) ^ M23(b[3])) << 16) ^
			((M30(b[0]) ^ M31(b[1]) ^ M32(b[2]) ^ M33(b[3])) << 24) ;
}
template<>
u32 CTwofish<192>::Substitution(u32 x, const u32* key)
{
	u8 b[4];
	*(u32*)b = IRIS_ToLittleEndian32(x);

	b[0] = (u8)(PB(03)[b[0]] ^ b0(key[2]));
	b[1] = (u8)(PB(13)[b[1]] ^ b1(key[2]));
	b[2] = (u8)(PB(23)[b[2]] ^ b2(key[2]));
	b[3] = (u8)(PB(33)[b[3]] ^ b3(key[2]));

	return	((M00(b[0]) ^ M01(b[1]) ^ M02(b[2]) ^ M03(b[3]))	  ) ^
			((M10(b[0]) ^ M11(b[1]) ^ M12(b[2]) ^ M13(b[3])) <<  8) ^
			((M20(b[0]) ^ M21(b[1]) ^ M22(b[2]) ^ M23(b[3])) << 16) ^
			((M30(b[0]) ^ M31(b[1]) ^ M32(b[2]) ^ M33(b[3])) << 24) ;
}
template<>
u32 CTwofish<256>::Substitution(u32 x, const u32* key)
{
	u8 b[4];
	*(u32*)b = IRIS_ToLittleEndian32(x);

	b[0] = (u8)(PB(04)[b[0]] ^ b0(key[3]));
	b[1] = (u8)(PB(14)[b[1]] ^ b1(key[3]));
	b[2] = (u8)(PB(24)[b[2]] ^ b2(key[3]));
	b[3] = (u8)(PB(34)[b[3]] ^ b3(key[3]));

	return	((M00(b[0]) ^ M01(b[1]) ^ M02(b[2]) ^ M03(b[3]))	  ) ^
			((M10(b[0]) ^ M11(b[1]) ^ M12(b[2]) ^ M13(b[3])) <<  8) ^
			((M20(b[0]) ^ M21(b[1]) ^ M22(b[2]) ^ M23(b[3])) << 16) ^
			((M30(b[0]) ^ M31(b[1]) ^ M32(b[2]) ^ M33(b[3])) << 24) ;
}

// ^̎̉
#ifdef _IRIS_TWOFISH_KEY
template class CTwofish<_IRIS_TWOFISH_KEY>;
#else
template class CTwofish<128>;
template class CTwofish<192>;
template class CTwofish<256>;
#endif

}	// end of namespace fnd
}	// end of namespace iris

#if	defined(_IRIS_SUPPORT_GOOGLETEST)

//======================================================================
// include
#include "../../unit/gt/gt_inchead.h"
#include "../../iris_using.h"

TEST(CFndTwofishTest, Function)
{
#define TWFH_TEST(cipher) do {							\
	cipher	cp;											\
	char c[512] = "test";								\
	char ck[cipher::KEYSIZE+1] = "0123456789abcdef";	\
	u8 key[cipher::KEYSIZE];							\
	memcpy(key, ck, sizeof(key));						\
	strcpy_s(c, sizeof(c), txt);						\
	printf("%s\n", #cipher);							\
	cp.CreateKeys(key, sizeof(key));					\
	cp.Encrypt((u8*)c, (u8*)c, sizeof(c));				\
	cp.Decrypt((u8*)c, (u8*)c, sizeof(c));				\
	ASSERT_STREQ( txt, c );								\
	} while(0)

	char txt[512] = "FndTowfish";

	TWFH_TEST(CTwofish128);
	TWFH_TEST(CTwofish192);
	TWFH_TEST(CTwofish256);
}

#endif

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "../../iris_iostream.h"
#include <string.h>
#include "../../iris_using.h"

//======================================================================
// test
IRIS_UNITTEST(CFndTwofishUnitTestUnitTest, FndTwofishUnitTest)
{
	while( 1 )
	{
#ifdef _IRIS_TWOFISH_KEY
		typedef CDefTwofish CCipher;
#else
		typedef CTwofish128 CCipher;
#endif
		CCipher twofish;
		char key[CCipher::KEYSIZE] = "anngoukagi";
		char c[CCipher::BLOCKSIZE*16] = "test";

		// ÍL[
		twofish.CreateKeys((u8*)key, (u32)sizeof(key));

#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
		std::cout << "͂ĂB" << std::endl;
		std::cin >> c;
#endif

		twofish.Encrypt((u8*)c, (u8*)c, sizeof(c));
		//char m[CCipher::BLOCKSIZE*16+1];
		//memcpy(m, c, sizeof(c));
		//m[CCipher::BLOCKSIZE*16] = '\0';
		//std::cout << m << std::endl;

		twofish.Decrypt((u8*)c, (u8*)c, sizeof(c));

		std::cout << c << std::endl;
		
#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
		s32 flag=0;
		std::cout << "Iꍇ́A0 ͂ĂB" << std::endl;
		std::safe_cin >> flag;
		if( flag == 0 ) break;
#else
		break;
#endif
	}
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

