//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FBXCamera.cpp
 * @brief		FBX SDK cameraNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FBXCamera_CPP_

//======================================================================
// include
#include "FBXCamera.h"

#ifdef _IRIS_SUPPORT_FBX
#include "../../gl/GXGL.h"
#include "../math/FBXMatrix.h"
#include "../FBXNode.h"
#include "../../../preprocessor/PPUnit.h"
#include "../../../iris_debug.h"

namespace iris {
namespace gx {
namespace fbx
{

//======================================================================
// class
/**********************************************************************//**
 *
 * RXgN^
 * 
*//***********************************************************************/
CFBXCamera::CFBXCamera(void)
{
}

/**********************************************************************//**
 *
 * RXgN^
 * 
 ----------------------------------------------------------------------
 * @param [in]	ptr	= JIuWFNg
*//***********************************************************************/
CFBXCamera::CFBXCamera(kfbx_ptr ptr)
: _Mybase(ptr)
{
}

/**********************************************************************//**
 *
 * RXgN^
 * 
 ----------------------------------------------------------------------
 * @param [in]	obj	= CFBXCamera IuWFNg
*//***********************************************************************/
CFBXCamera::CFBXCamera(CFBXCamera& obj)
: _Mybase(obj)
{
}

/**********************************************************************//**
 *
 * fXgN^
 * 
*//***********************************************************************/
CFBXCamera::~CFBXCamera(void)
{
}

/**********************************************************************//**
 *
 * XV
 * 
 ----------------------------------------------------------------------
 * @param [in]	time	= XVb
*//***********************************************************************/
void CFBXCamera::Update(xf32 time)
{
	if( m_pKfbx == nullptr ) return;
	KTime ktime;
	ktime.SetSecondDouble(static_cast<double>(XF_XF32_TO_F32(time)));
	m_Time += ktime;
}

/**********************************************************************//**
 *
 * `
 * 
*//***********************************************************************/
void CFBXCamera::Draw(void)
{
	if( m_pKfbx == nullptr ) return;
	IrisFMtx44 proj, view;
	if( m_pKfbx->ProjectionType.Get() == KFbxCamera::ePERSPECTIVE )
	{
		GetPerspectiveFov(&proj);
		GetViewMatrix(&view);
		gx::gl::gxglSetMatrix(GXGL_MATRIX_PROJECTION, proj.a);
		gx::gl::gxglSetMatrix(GXGL_MATRIX_VIEW, view.a);
	}
}

/**********************************************************************//**
 *
 * ˉes̎擾
 * 
 ----------------------------------------------------------------------
 * @param [out]	rMatrix	= o
 * @return 
*//***********************************************************************/
IrisFMtx44* CFBXCamera::GetPerspectiveFov(IrisFMtx44* pMatrix)
{
	if( m_pKfbx == nullptr ) return pMatrix;
	f32 ay = static_cast<f32>(m_pKfbx->GetApertureHeight());
	f32 ax = static_cast<f32>(m_pKfbx->GetApertureWidth());
	gl::gxglMatrixPerspectiveFov(pMatrix
		, GetFovy()
		, ax / ay
		, static_cast<f32>(m_pKfbx->GetNearPlane())
		, static_cast<f32>(m_pKfbx->GetFarPlane())
		);
	return pMatrix;
}

/**********************************************************************//**
 *
 * r[s̎擾
 * 
 ----------------------------------------------------------------------
 * @param [out]	rMatrix	= o
 * @return 
*//***********************************************************************/
IrisFMtx44* CFBXCamera::GetViewMatrix(IrisFMtx44* pMatrix)
{
	if( m_pKfbx == nullptr ) return false;
	KFbxNode* node = m_pKfbx->GetNode();
	IRIS_ASSERT( node != nullptr );
	KFbxVector4 center(0.0, 0.0, 0.0);
	KFbxVector4 eye = m_pKfbx->Position.Get();
	KFbxVector4 up = m_pKfbx->UpVector.Get();
	KFbxVector4 dir, right;
	double r = IRIS_Degree2Radian(m_pKfbx->Roll.Get());
	if( node->GetTarget() != nullptr )
	{
		center = fbx::GetGlobalPosition(node->GetTarget(), m_Time);
	}
	else
	{
	}

	dir = center - eye;
	dir.Normalize();
	right = dir.CrossProduct(up);
	right.Normalize();
	up = right.CrossProduct(dir);
	up.Normalize();

	up *= cos(r);
	right *= sin(r);
	up += right;

	IrisTVec4<f32> feye(*(IrisTVec4<f64>*)(f64*)eye);
	IrisTVec4<f32> fcenter(*(IrisTVec4<f64>*)(f64*)center);
	IrisTVec4<f32> fup(*(IrisTVec4<f64>*)(f64*)up);
	return gx::gl::gxglMatrixLookAt(pMatrix, (IrisFVec4*)&feye, (IrisFVec4*)&fcenter, (IrisFVec4*)&fup);
}

/**********************************************************************//**
 *
 * p̎擾
 * 
 ----------------------------------------------------------------------
 * @return p
*//***********************************************************************/
f32 CFBXCamera::GetFovy(void)
{
	if( m_pKfbx == nullptr ) return 0.0f;
	f64 filmH = m_pKfbx->FilmHeight.Get();
	f64 focal = m_pKfbx->FocalLength.Get();
	return static_cast<f32>( atan2( IRIS_Inch2MilliMeter(filmH), focal * 2.0) );
}

}	// end of namespace fbx
}	// end of namespace gx
}	// end of namespace iris

#endif
