//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathMatrix33.inl
 * @brief		3x3}gbNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathMatrix33_inl_
#define INCG_IRIS_MathMatrix33_inl_

namespace iris {
namespace math
{

//======================================================================
// function
/**
 * @brief	Pʃ}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33* FpuMtx33Unit(IrisFMtx33* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	pm0->x.x = pm0->y.y = pm0->z.z = 1.0f;
	pm0->x.y = pm0->x.z = pm0->y.x = pm0->y.z = pm0->z.x = pm0->z.y = 0.0f;
	return pm0;
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Zero(IrisFMtx33* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	FpuVec3Zero(&pm0->x);
	FpuVec3Zero(&pm0->y);
	FpuVec3Zero(&pm0->z);
	return pm0;
}

/**
 * @brief	}gbNX̃Rs[
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Copy(IrisFMtx33* pm0, const IrisFMtx33* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
#if 1
	pm0->_00 = pm1->_00;
	pm0->_01 = pm1->_01;
	pm0->_02 = pm1->_02;
	pm0->_10 = pm1->_10;
	pm0->_11 = pm1->_11;
	pm0->_12 = pm1->_12;
	pm0->_20 = pm1->_20;
	pm0->_21 = pm1->_21;
	pm0->_22 = pm1->_22;
#else
	FpuVec3Copy(&pm0->x, &pm1->x);
	FpuVec3Copy(&pm0->y, &pm1->y);
	FpuVec3Copy(&pm0->z, &pm1->z);
#endif
	return pm0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE IrisFVec3*		FpuMtx33Transform(IrisFVec3* pv0, const IrisFMtx33* pm0, const IrisFVec3* pv1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	IRIS_ASSERT( pv0 != pv1 );	// TODO : AhXɖΉ
	pv0->x = F32_Mul(pm0->x.x, pv1->x) + F32_Mul(pm0->y.x, pv1->y) + F32_Mul(pm0->z.x, pv1->z);
	pv0->y = F32_Mul(pm0->x.y, pv1->x) + F32_Mul(pm0->y.y, pv1->y) + F32_Mul(pm0->z.y, pv1->z);
	pv0->z = F32_Mul(pm0->x.z, pv1->x) + F32_Mul(pm0->y.z, pv1->y) + F32_Mul(pm0->z.z, pv1->z);
	return pv0;
}

/**
 * @brief	Q̃}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pm2	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Mul(IrisFMtx33* pm0, const IrisFMtx33* pm1, const IrisFMtx33* pm2)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	MATH_FPU_NULLASSERT( pm2 );
	IRIS_ASSERT( pm0 != pm1 );
	IRIS_ASSERT( pm0 != pm2 );
	pm0->x.x = F32_Mul(pm1->x.x, pm2->x.x) + F32_Mul(pm1->y.x, pm2->x.y) + F32_Mul(pm1->z.x, pm2->x.z);
	pm0->x.y = F32_Mul(pm1->x.y, pm2->x.x) + F32_Mul(pm1->y.y, pm2->x.y) + F32_Mul(pm1->z.y, pm2->x.z);
	pm0->x.z = F32_Mul(pm1->x.z, pm2->x.x) + F32_Mul(pm1->y.z, pm2->x.y) + F32_Mul(pm1->z.z, pm2->x.z);
	pm0->y.x = F32_Mul(pm1->x.x, pm2->y.x) + F32_Mul(pm1->y.x, pm2->y.y) + F32_Mul(pm1->z.x, pm2->y.z);
	pm0->y.y = F32_Mul(pm1->x.y, pm2->y.x) + F32_Mul(pm1->y.y, pm2->y.y) + F32_Mul(pm1->z.y, pm2->y.z);
	pm0->y.z = F32_Mul(pm1->x.z, pm2->y.x) + F32_Mul(pm1->y.z, pm2->y.y) + F32_Mul(pm1->z.z, pm2->y.z);
	pm0->z.x = F32_Mul(pm1->x.x, pm2->z.x) + F32_Mul(pm1->y.x, pm2->z.y) + F32_Mul(pm1->z.x, pm2->z.z);
	pm0->z.y = F32_Mul(pm1->x.y, pm2->z.x) + F32_Mul(pm1->y.y, pm2->z.y) + F32_Mul(pm1->z.y, pm2->z.z);
	pm0->z.z = F32_Mul(pm1->x.z, pm2->z.x) + F32_Mul(pm1->y.z, pm2->z.y) + F32_Mul(pm1->z.z, pm2->z.z);
	return pm0;
}

/**
 * @brief	}gbNX̃XP[O
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	s	= XJ[l
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Scale(IrisFMtx33* pm0, const IrisFMtx33* pm1, f32 s)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	FpuVec3Scale(&pm0->x, &pm1->x, s);
	FpuVec3Scale(&pm0->y, &pm1->y, s);
	FpuVec3Scale(&pm0->z, &pm1->z, s);
	return pm0;
}

/**
 * @brief	}gbNX̓]us߂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Transpose(IrisFMtx33* pm0, const IrisFMtx33* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	pm0->x.x = pm1->x.x;
	f32 f01 = pm1->x.y;
	f32 f02 = pm1->x.z;
	pm0->x.y = pm1->y.x;
	f32 f11 = pm1->y.y;
	f32 f12 = pm1->y.z;
	pm0->x.z = pm1->z.x;
	pm0->y.x = f01;
	pm0->y.y = f11;
	pm0->y.z = pm1->z.y;
	pm0->z.x = f02;
	pm0->z.y = f12;
	pm0->z.z = pm1->z.z;
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33RotZ(IrisFMtx33* pm0, const IrisFMtx33* pm1, f32 rz)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_Cos(rz);
	f32 s = F32_Sin(rz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = 0.0f;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = 0.0f;
		pm0->z.x = 0.0f;
		pm0->z.y = 0.0f;
		pm0->z.z = 1.0f;
	}
	else
	{
		IrisFMtx33 m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = 0.0f;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = 0.0f;
		m.z.x = 0.0f;
		m.z.y = 0.0f;
		m.z.z = 1.0f;
		FpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33RotIdxZ(IrisFMtx33* pm0, const IrisFMtx33* pm1, u16 idz)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_CosIdx(idz);
	f32 s = F32_SinIdx(idz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = 0.0f;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = 0.0f;
		pm0->z.x = 0.0f;
		pm0->z.y = 0.0f;
		pm0->z.z = 1.0f;
	}
	else
	{
		IrisFMtx33 m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = 0.0f;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = 0.0f;
		m.z.x = 0.0f;
		m.z.y = 0.0f;
		m.z.z = 1.0f;
		FpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	ry	= Y]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33RotY(IrisFMtx33* pm0, const IrisFMtx33* pm1, f32 ry)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_Cos(ry);
	f32 s = F32_Sin(ry);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = 0.0f;
		pm0->x.z = +s;
		pm0->y.x = 0.0f;
		pm0->y.y = 1.0f;
		pm0->y.z = 0.0f;
		pm0->z.x = -s;
		pm0->z.y = 0.0f;
		pm0->z.z = +c;
	}
	else
	{
		IrisFMtx33 m;
		m.x.x = +c;
		m.x.y = 0.0f;
		m.x.z = +s;
		m.y.x = 0.0f;
		m.y.y = 1.0f;
		m.y.z = 0.0f;
		m.z.x = -s;
		m.z.y = 0.0f;
		m.z.z = +c;
		FpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idy	= Y]ʁiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33RotIdxY(IrisFMtx33* pm0, const IrisFMtx33* pm1, u16 idy)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_CosIdx(idy);
	f32 s = F32_SinIdx(idy);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = 0.0f;
		pm0->x.z = +s;
		pm0->y.x = 0.0f;
		pm0->y.y = 1.0f;
		pm0->y.z = 0.0f;
		pm0->z.x = -s;
		pm0->z.y = 0.0f;
		pm0->z.z = +c;
	}
	else
	{
		IrisFMtx33 m;
		m.x.x = +c;
		m.x.y = 0.0f;
		m.x.z = +s;
		m.y.x = 0.0f;
		m.y.y = 1.0f;
		m.y.z = 0.0f;
		m.z.x = -s;
		m.z.y = 0.0f;
		m.z.z = +c;
		FpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rx	= X]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33RotX(IrisFMtx33* pm0, const IrisFMtx33* pm1, f32 rx)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_Cos(rx);
	f32 s = F32_Sin(rx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = 1.0f;
		pm0->x.y = 0.0f;
		pm0->x.z = 0.0f;
		pm0->y.x = 0.0f;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->z.x = 0.0f;
		pm0->z.y = -s;
		pm0->z.z = +c;
	}
	else
	{
		IrisFMtx33 m;
		m.x.x = 1.0f;
		m.x.y = 0.0f;
		m.x.z = 0.0f;
		m.y.x = 0.0f;
		m.y.y = +c;
		m.y.z = +s;
		m.z.x = 0.0f;
		m.z.y = -s;
		m.z.z = +c;
		FpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idx	= X]ʁiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33RotIdxX(IrisFMtx33* pm0, const IrisFMtx33* pm1, u16 idx)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_CosIdx(idx);
	f32 s = F32_SinIdx(idx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = 1.0f;
		pm0->x.y = 0.0f;
		pm0->x.z = 0.0f;
		pm0->y.x = 0.0f;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->z.x = 0.0f;
		pm0->z.y = -s;
		pm0->z.z = +c;
	}
	else
	{
		IrisFMtx33 m;
		m.x.x = 1.0f;
		m.x.y = 0.0f;
		m.x.z = 0.0f;
		m.y.x = 0.0f;
		m.y.y = +c;
		m.y.z = +s;
		m.z.x = 0.0f;
		m.z.y = -s;
		m.z.z = +c;
		FpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiWAj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Rot(IrisFMtx33* pm0, const IrisFMtx33* pm1, const IrisFVec3* pv0)
{
	FpuMtx33RotZ(pm0, pm1, pv0->z);
	FpuMtx33RotY(pm0, pm0, pv0->y);
	FpuMtx33RotX(pm0, pm0, pv0->x);
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33RotIdx(IrisFMtx33* pm0, const IrisFMtx33* pm1, const IrisSVec3* pv0)
{
	FpuMtx33RotIdxZ(pm0, pm1, (u16)pv0->z);
	FpuMtx33RotIdxY(pm0, pm0, (u16)pv0->y);
	FpuMtx33RotIdxX(pm0, pm0, (u16)pv0->x);
	return pm0;
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
IRIS_FPU_INLINE IrisBool		FpuMtx33IsUnit(const IrisFMtx33* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	const IrisMtx33* m = (const IrisMtx33*)(pm0);
	if( m->im.x.x != 0x3F800000 || m->im.y.y != 0x3F800000 || m->im.z.z != 0x3F800000 )
		return IRIS_FALSE;
	if( (             m->im.x.y | m->im.x.z 
		| m->im.y.x             | m->im.y.z
		| m->im.z.x | m->im.z.y             ) != 0 )
		return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	}gbNX̃g[XԂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	}gbNX̃g[X
*/
IRIS_FPU_INLINE f32			FpuMtx33Trace(const IrisFMtx33* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	return (pm0->x.x + pm0->y.y + pm0->z.z);
}

/**
 * @brief	s񎮂Ԃ
 * @param [in]	pm0	= ̓}gbNX
 * @return	s
*/
IRIS_FPU_INLINE f32			FpuMtx33Determinant(const IrisFMtx33* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 a11 = pm0->x.x;
	f32 a21 = pm0->x.y;
	f32 a31 = pm0->x.z;
	f32 a12 = pm0->y.x;
	f32 a22 = pm0->y.y;
	f32 a32 = pm0->y.z;
	f32 a13 = pm0->z.x;
	f32 a23 = pm0->z.y;
	f32 a33 = pm0->z.z;
	return ( F32_Mul( F32_Mul(a11, a22), a33) + F32_Mul( F32_Mul(a12, a23), a31)
		+ F32_Mul( F32_Mul(a13, a21), a32) - F32_Mul( F32_Mul(a13, a22), a31)
		- F32_Mul( F32_Mul(a12, a21), a33) - F32_Mul( F32_Mul(a11, a23), a32) );
}

/**
 * @brief	]qsԂ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Adjoint(IrisFMtx33* pm0, const IrisFMtx33* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	f32 a11 = pm1->x.x;
	f32 a21 = pm1->x.y;
	f32 a31 = pm1->x.z;
	f32 a12 = pm1->y.x;
	f32 a22 = pm1->y.y;
	f32 a32 = pm1->y.z;
	f32 a13 = pm1->z.x;
	f32 a23 = pm1->z.y;
	f32 a33 = pm1->z.z;
	pm0->x.x =  ( F32_Mul(a22, a33) - F32_Mul(a23, a32) );
	pm0->x.y = -( F32_Mul(a21, a33) - F32_Mul(a23, a31) );
	pm0->x.z =  ( F32_Mul(a21, a32) - F32_Mul(a22, a31) );
	pm0->y.x = -( F32_Mul(a12, a33) - F32_Mul(a13, a32) );
	pm0->y.y =  ( F32_Mul(a11, a33) - F32_Mul(a13, a31) );
	pm0->y.z = -( F32_Mul(a11, a32) - F32_Mul(a12, a31) );
	pm0->z.x =  ( F32_Mul(a12, a23) - F32_Mul(a13, a22) );
	pm0->z.y = -( F32_Mul(a11, a23) - F32_Mul(a13, a21) );
	pm0->z.z =  ( F32_Mul(a11, a22) - F32_Mul(a12, a21) );
	return pm0;
}

/**
 * @brief	tsԂ
 * @param [out]	pm0				= o̓}gbNX
 * @param [out]	pDeterminant	= 
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Inverse(IrisFMtx33* pm0, f32* pDeterminant, const IrisFMtx33* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	f32 d = FpuMtx33Determinant(pm1);
	if( pDeterminant != nullptr ) *pDeterminant = d;
	if( d == 0.0f ) return nullptr;
	FpuMtx33Adjoint(pm0, pm1);
	FpuMtx33Scale(pm0, pm0, F32_Div(1.0f, d) );
	return pm0;
}

/**
 * @brief	}gNX̐K
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33Normalize(IrisFMtx33* pm0, const IrisFMtx33* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );

	// XY̊OςZ𓾂
	FpuVec3OuterProduct(&pm0->z, &pm1->x, &pm1->y);
	// Oς瓾ZY̊OςX𓾂
	FpuVec3OuterProduct(&pm0->x, &pm1->y, &pm0->z);
	FpuVec3Normalize(&pm0->x, &pm0->x);
	FpuVec3Normalize(&pm0->y, &pm1->y);
	FpuVec3Normalize(&pm0->z, &pm0->z);
	return pm0;
}

/**
 * @brief	8rbg̐x؂̂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx33*	FpuMtx33TruncatePrecision24(IrisFMtx33* pm0, const IrisFMtx33* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );

	FpuVec3TruncatePrecision24(&pm0->x, &pm1->x);
	FpuVec3TruncatePrecision24(&pm0->y, &pm1->y);
	FpuVec3TruncatePrecision24(&pm0->z, &pm1->z);
	return pm0;
}

}	// end of namespace math
}	// end of namespace iris

#endif
