//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathQuaternion.inl
 * @brief		NH[^jI֐t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathQuaternion_inl_
#define INCG_IRIS_MathQuaternion_inl_

namespace iris {
namespace math
{

//======================================================================
// function
/**
 * @brief	PʃNH[^jI𐶐
 * @param [out]	pq0	= o̓NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionUnit(IrisFQuaternion* pq0)
{
	MATH_FPU_NULLASSERT( pq0 );
	pq0->x = pq0->y = pq0->z = 0.0f;
	pq0->w = 1.0f;
	return pq0;
}

/**
 * @brief	NH[^jIRs[
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= Rs[NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionCopy(IrisFQuaternion* pq0, const IrisFQuaternion* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	memcpy(pq0, pq1, sizeof(IrisFQuaternion));
	return pq0;
}

/**
 * @brief	NH[^jI̘avZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionAdd(IrisFQuaternion* pq0, const IrisFQuaternion* pq1, const IrisFQuaternion* pq2)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	pq0->x = pq1->x + pq2->x;
	pq0->y = pq1->y + pq2->y;
	pq0->z = pq1->z + pq2->x;
	pq0->w = pq1->w + pq2->w;
	return pq0;
}

/**
 * @brief	NH[^jI̍vZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= 팸NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @return o̓xNg
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionSub(IrisFQuaternion* pq0, const IrisFQuaternion* pq1, const IrisFQuaternion* pq2)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	pq0->x = pq1->x - pq2->x;
	pq0->y = pq1->y - pq2->y;
	pq0->z = pq1->z - pq2->x;
	pq0->w = pq1->w - pq2->w;
	return pq0;
}

/**
 * @brief	NH[^jI̐ςvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= 搔NH[^jI
 * @param [in]	pq2	= 搔NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionMul(IrisFQuaternion* pq0, const IrisFQuaternion* pq1, const IrisFQuaternion* pq2)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	f32 x1 = pq1->x, y1 = pq1->y, z1 = pq1->z, w1 = pq1->w;
	f32 x2 = pq2->x, y2 = pq2->y, z2 = pq2->z, w2 = pq2->w;
	pq0->x = F32_Mul(w1, x2) + F32_Mul(w2, x1) + F32_Mul(y1, z2) - F32_Mul(z1, y2);
	pq0->y = F32_Mul(w1, y2) + F32_Mul(w2, y1) + F32_Mul(z1, x2) - F32_Mul(x1, z2);
	pq0->z = F32_Mul(w1, z2) + F32_Mul(w2, z1) + F32_Mul(x1, y2) - F32_Mul(y1, x2);
	pq0->w = F32_Mul(w1, w2) - F32_Mul(x1, x2) - F32_Mul(y1, y2) - F32_Mul(z1, z2);
	return pq0;
}

/**
 * @brief	NH[^jI̓ςvZ
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return 
*/
IRIS_FPU_INLINE	f32				FpuQuaternionInnerProduct(const IrisFQuaternion* pq0, const IrisFQuaternion* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	return ( F32_Mul(pq0->x, pq1->x) + F32_Mul(pq0->y, pq1->y) + F32_Mul(pq0->z, pq0->z) + F32_Mul(pq0->w, pq0->w) );
}

/**
 * @brief	NH[^jI̋vZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionConj(IrisFQuaternion* pq0, const IrisFQuaternion* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	pq0->x = -pq1->x;
	pq0->y = -pq1->y;
	pq0->z = -pq1->z;
	pq0->w =  pq1->w;
	return pq0;
}

/**
 * @brief	NH[^jIgăxNg]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pv1	= xNg
 * @return o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*			FpuQuaternionTransform(IrisFVec4* pv0, const IrisFQuaternion* pq0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv1 );
	IrisFQuaternion q1, q2;
	FpuQuaternionConj(&q2, pq0);
	FpuQuaternionMul(&q1, pq0, (const IrisFQuaternion*)pv1);
	FpuQuaternionMul((IrisFQuaternion*)pv0, &q1, &q2);
	return pv0;
}

/**
 * @brief	NH[^jI̋ʐ`⊮
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @param [in]	t	= 
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionSlerp(IrisFQuaternion* pq0, const IrisFQuaternion* pq1, const IrisFQuaternion* pq2, f32 t)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	f32 c = F32_Mul(pq1->x, pq2->x) + F32_Mul(pq1->y, pq2->y) + F32_Mul(pq1->z, pq2->z) + F32_Mul(pq1->w, pq2->w);
	f32 angle;
	f32 s;
	f32 r1, r2;

	int reverse = 0;
	if( c < 0 ) { c = - c; reverse = 1; }

	angle = F32_Acos( c < -1.0f ? -1.0f : ( c > 1.0f ? 1.0f : c ) );
	s = F32_Sin( angle );
	if( s < 0.00005f ) 
	{
		memcpy(pq0, pq2, sizeof(IrisFQuaternion));
		return (pq0);
	}
	r1 = F32_Div( F32_Sin( F32_Mul(( 1.0f - t ), angle) ), s);
	r2 = F32_Div( F32_Sin( F32_Mul(t, angle) ), s);
	if( reverse ) r2 = - r2;

	//@@
	//  : ٓ_ߕӂł̐xmۂɂ quat_normalizeg
	pq0->x = F32_Mul(pq1->x, r1) + F32_Mul(pq2->x, r2);
	pq0->y = F32_Mul(pq1->y, r1) + F32_Mul(pq2->y, r2);
	pq0->z = F32_Mul(pq1->z, r1) + F32_Mul(pq2->z, r2);
	pq0->w = F32_Mul(pq1->w, r1) + F32_Mul(pq2->w, r2);
	return pq0;
}

/**
 * @brief	NH[^jI̋3⊮
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @param [in]	pq3	= NH[^jI
 * @param [in]	pq4	= NH[^jI
 * @param [in]	t	= 
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionSquad(IrisFQuaternion* pq0, const IrisFQuaternion* pq1, const IrisFQuaternion* pq2
													   , const IrisFQuaternion* pq3, const IrisFQuaternion* pq4, f32 t)
{
	IrisFQuaternion qa, qb;
	FpuQuaternionSlerp(&qa, pq1, pq2, t);
	FpuQuaternionSlerp(&qb, pq3, pq4, t);
	FpuQuaternionSlerp(pq0, &qa, &qb, F32_Mul(2*t, (1.0f-t)));
	return pq0;
}

/**
 * @brief	NH[^jI̐K
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionNormalize(IrisFQuaternion* pq0, const IrisFQuaternion* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	f32 x = pq1->x, y = pq1->y, z = pq1->z, w = pq1->w;
	f32 q = F32_Div(1.0f, F32_Sqrt( F32_Mul(x, x) + F32_Mul(y, y) + F32_Mul(z, z) + F32_Mul(w, w) ));
	pq0->x = F32_Mul(x, q);
	pq0->y = F32_Mul(y, q);
	pq0->z = F32_Mul(z, q);
	pq0->w = F32_Mul(w, q);
	return pq0;
}

/**
 * @brief	tNH[^jǏvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionInverse(IrisFQuaternion* pq0, const IrisFQuaternion* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	f32 n = F32_Mul(pq1->x, pq1->x) + F32_Mul(pq1->y, pq1->y) + F32_Mul(pq1->z, pq1->z) + F32_Mul(pq1->w, pq1->w);
	pq0->x = F32_Div(-pq1->x, n);
	pq0->y = F32_Div(-pq1->y, n);
	pq0->z = F32_Div(-pq1->z, n);
	pq0->w = F32_Div( pq1->w, n);
	return pq0;
}

/**
 * @brief	NH[^jI̎RΐvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionLn(IrisFQuaternion* pq0, const IrisFQuaternion* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	f32 n, theta, m;
	n = F32_Sqrt( F32_Mul(pq1->x, pq1->x) + F32_Mul(pq1->y, pq1->y) + F32_Mul(pq1->z, pq1->z) );
	if(n > 0.0f)
	{
		theta = F32_Acos( F32_Div(pq1->w, n) );
		m     = F32_Div(theta, n);
		pq0->x = F32_Mul(pq1->x, m);
		pq0->y = F32_Mul(pq1->y, m);
		pq0->z = F32_Mul(pq1->z, m);
	} 
	else
	{
		pq0->x = pq1->x;
		pq0->y = pq1->y;
		pq0->z = pq1->z;
	}
	pq0->w = 0.0f;
	return pq0;
}

/**
 * @brief	NH[^jI̎wvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionExp(IrisFQuaternion* pq0, const IrisFQuaternion* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	f32 theta = F32_Sqrt( F32_Mul(pq1->x, pq1->x) + F32_Mul(pq1->y, pq1->y) + F32_Mul(pq1->z, pq1->z) );
	f32 s = F32_Div(F32_Sin(theta), theta);
	pq0->x = F32_Mul(pq1->x, s);
	pq0->y = F32_Mul(pq1->y, s);
	pq0->z = F32_Mul(pq1->z, s);
	pq0->w = F32_Cos(theta);
	return pq0;
}

/**
 * @brief	NH[^jI}gbNXɕϊ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pq0	= NH[^jI
 * @return o̓}gbNX
*/
IRIS_FPU_INLINE	IrisFMtx44*		FpuQuaternionToMatrix(IrisFMtx44* pm0, const IrisFQuaternion* pq0)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pq0 );
	f32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;
	pm0->x.x = 1.0f -	F32_Mul(F32_Mul(2.0f, y), y) - F32_Mul(F32_Mul(2.0f, z), z);
	pm0->x.y =			F32_Mul(F32_Mul(2.0f, x), y) + F32_Mul(F32_Mul(2.0f, w), z);
	pm0->x.z =			F32_Mul(F32_Mul(2.0f, z), x) - F32_Mul(F32_Mul(2.0f, w), y);
	pm0->x.w = 0.0f;
	pm0->y.x =			F32_Mul(F32_Mul(2.0f, x), y) - F32_Mul(F32_Mul(2.0f, w), z);
	pm0->y.y = 1.0f -	F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, z), z);
	pm0->y.z =			F32_Mul(F32_Mul(2.0f, y), z) + F32_Mul(F32_Mul(2.0f, w), x);
	pm0->y.w = 0.0f;
	pm0->z.x =			F32_Mul(F32_Mul(2.0f, z), x) + F32_Mul(F32_Mul(2.0f, w), y);
	pm0->z.y =			F32_Mul(F32_Mul(2.0f, y), z) - F32_Mul(F32_Mul(2.0f, w), x);
	pm0->z.z = 1.0f -	F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, y), y);
	pm0->z.w = 0.0f;
	pm0->w.x = 0.0f;
	pm0->w.y = 0.0f;
	pm0->w.z = 0.0f;
	pm0->w.w = 1.0f;
	return pm0;
}

/**
 * @brief	}gbNXNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pm0	= }gbNX
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionFromMatrix(IrisFQuaternion* pq0, const IrisFMtx44* pm0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pm0 );
	f32 x, y, z, w;
	f32 r, t;
	const IrisFVec4 *pv = pm0->v;

	t = pv[0].x + pv[1].y + pv[2].z + 1.0f;

	if( t > 0.01f ) 
	{
		w = F32_Mul(F32_Sqrt( t ), 0.5f);
		r = F32_Div(0.25f, w);
		x = F32_Mul(( pv[1].z - pv[2].y ), r);
		y = F32_Mul(( pv[2].x - pv[0].z ), r);
		z = F32_Mul(( pv[0].y - pv[1].x ), r);
	} 
	else if( pv[0].x > pv[1].y )
	{
		if( pv[0].x > pv[2].z ) 
		{
			x = F32_Mul(F32_Sqrt( 1.0f + pv[0].x - pv[1].y - pv[2].z ), 0.5f);
			r = F32_Div(0.25f, x);
			y = F32_Mul(( pv[1].x + pv[0].y ), r);
			z = F32_Mul(( pv[2].x + pv[0].z ), r);
			w = F32_Mul(( pv[1].z - pv[2].y ), r);
		}
		else
		{
			z = F32_Mul(F32_Sqrt( 1.0f + pv[2].z - pv[0].x - pv[1].y ), 0.5f);
			r = F32_Div(0.25f, z);
			x = F32_Mul(( pv[2].x + pv[0].z ), r);
			y = F32_Mul(( pv[2].y + pv[1].z ), r);
			w = F32_Mul(( pv[0].y - pv[1].x ), r);
		}
	} 
	else
	{
		if( pv[1].y > pv[2].z )
		{
			y = F32_Mul(F32_Sqrt( 1.0f + pv[1].y - pv[2].z - pv[0].x ), 0.5f);
			r = F32_Div(0.25f, y);
			x = F32_Mul(( pv[1].x + pv[0].y ), r);
			z = F32_Mul(( pv[2].y + pv[1].z ), r);
			w = F32_Mul(( pv[2].x - pv[0].z ), r);
		} 
		else 
		{
			z = F32_Mul(F32_Sqrt( 1.0f + pv[2].z - pv[0].x - pv[1].y ), 0.5f);
			r = F32_Div(0.25f, z);
			x = F32_Mul(( pv[2].x + pv[0].z ), r);
			y = F32_Mul(( pv[2].y + pv[1].z ), r);
			w = F32_Mul(( pv[0].y - pv[1].x ), r);
		}
	}
	pq0->x = x;
	pq0->y = y;
	pq0->z = z;
	pq0->w = w;
	return (pq0);
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionFromRotZYX(IrisFQuaternion* pq0, const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv0 );
	f32 x = F32_Mul(pv0->x, 0.5f);
	f32 y = F32_Mul(pv0->y, 0.5f);
	f32 z = F32_Mul(pv0->z, 0.5f);
	f32 cx = F32_Cos( x );	f32 sx = F32_Sin( x );
	f32 cy = F32_Cos( y );	f32 sy = F32_Sin( y );
	f32 cz = F32_Cos( z );	f32 sz = F32_Sin( z );
	pq0->x = F32_Mul(F32_Mul(cz, cy), sx) - F32_Mul(F32_Mul(cx, sz), sy);
	pq0->y = F32_Mul(F32_Mul(cx, cz), sy) + F32_Mul(F32_Mul(cy, sz), sx);
	pq0->z = F32_Mul(F32_Mul(cx, cy), sz) - F32_Mul(F32_Mul(cz, sy), sx);
	pq0->w = F32_Mul(F32_Mul(cz, cy), cx) + F32_Mul(F32_Mul(sz, sy), sx);
	return pq0;
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionFromRotXYZ(IrisFQuaternion* pq0, const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv0 );
	f32 x = F32_Mul(pv0->x, 0.5f);
	f32 y = F32_Mul(pv0->y, 0.5f);
	f32 z = F32_Mul(pv0->z, 0.5f);
	f32 cx = F32_Cos( x );	f32 sx = F32_Sin( x );
	f32 cy = F32_Cos( y );	f32 sy = F32_Sin( y );
	f32 cz = F32_Cos( z );	f32 sz = F32_Sin( z );
	pq0->x = F32_Mul(F32_Mul(sx, cy), cz) + F32_Mul(F32_Mul(cx, sy), sz);
	pq0->y = F32_Mul(F32_Mul(cx, sy), cz) - F32_Mul(F32_Mul(sx, cy), sz);
	pq0->z = F32_Mul(F32_Mul(cx, cy), sz) + F32_Mul(F32_Mul(sx, sy), cz);
	pq0->w = F32_Mul(F32_Mul(cx, cy), cz) - F32_Mul(F32_Mul(sx, sy), sz);
	return pq0;
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionFromRotYXZ(IrisFQuaternion* pq0, const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv0 );
	f32 x = F32_Mul(pv0->x, 0.5f);
	f32 y = F32_Mul(pv0->y, 0.5f);
	f32 z = F32_Mul(pv0->z, 0.5f);
	f32 cx = F32_Cos( x );	f32 sx = F32_Sin( x );
	f32 cy = F32_Cos( y );	f32 sy = F32_Sin( y );
	f32 cz = F32_Cos( z );	f32 sz = F32_Sin( z );
	pq0->x = F32_Mul(F32_Mul(cy, cz), sx) + F32_Mul(F32_Mul(sy, cx), sz);
	pq0->y = F32_Mul(F32_Mul(sy, cx), cz) - F32_Mul(F32_Mul(cy, sx), sz);
	pq0->z = F32_Mul(F32_Mul(cy, cx), sz) - F32_Mul(F32_Mul(sy, cz), sx);
	pq0->w = F32_Mul(F32_Mul(cy, cx), cz) + F32_Mul(F32_Mul(sy, sx), sz);
	return pq0;
}

/**
 * @brief	]xNgƉ]ʂANH[^jI𐶐
 * @param [out]	pq0		= o̓NH[^jI
 * @param [in]	angle	= ]
 * @param [in]	pvAxis	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_FPU_INLINE	IrisFQuaternion*	FpuQuaternionFromRotate(IrisFQuaternion* pq0, f32 angle, const IrisFVec4* pvAxis)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pvAxis );
	FpuVec4NormalizeXYZ((IrisFVec4*)pq0, pvAxis);

	angle = F32_Mul(angle, 0.5f);
	FpuVec4Scale((IrisFVec4*)pq0, (IrisFVec4*)pq0, F32_Sin(angle));
	pq0->w = F32_Cos( angle );
	return pq0;
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*			FpuQuaternionToRotZYX(IrisFVec4* pv0, const IrisFQuaternion* pq0)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	f32 cx, sx, sy, cz, sz;
	f32 rx, ry, rz;
	f32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sy = -( F32_Mul(F32_Mul(2.0f, z), x) - F32_Mul(F32_Mul(2.0f, w), y) );
	if( sy <= 0.99995f && sy >= -0.99995f ) 
	{
		sx =		F32_Mul(F32_Mul(2.0f, y), z) + F32_Mul(F32_Mul(2.0f, w), x);
		cx = 1.0f - F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, y), y);
		sz =		F32_Mul(F32_Mul(2.0f, x), y) + F32_Mul(F32_Mul(2.0f, w), z);
		cz = 1.0f - F32_Mul(F32_Mul(2.0f, y), y) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
	}
	else
	{
#if 1
		//J |cosY|0ɋ߂ƂAZ]0Ɖ肵YX]߂
		sx =		-(F32_Mul(F32_Mul(2.0f, y), z) - F32_Mul(F32_Mul(2.0f, w), x));
		cx = 1.0f	- F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = 0.0f;
#else
		//J |cosY|0ɋ߂ƂAX]0Ɖ肵ZY]߂
		sz =		-(F32_Mul(F32_Mul(2.0f, x), y) - F32_Mul(F32_Mul(2.0f, w), z));
		cz = 1.0f	- F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = 0.0f;
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = 0.0f;
	return (pv0);
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*			FpuQuaternionToRotXYZ(IrisFVec4* pv0, const IrisFQuaternion* pq0)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	f32 cx, sx, sy, cz, sz;
	f32 rx, ry, rz;
	f32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sy = F32_Mul(F32_Mul(2.0f, z), x) + F32_Mul(F32_Mul(2.0f, w), y);
	if( sy <= 0.99995f && sy >= -0.99995f ) 
	{
		sx =		-(F32_Mul(F32_Mul(2.0f, y), z) - F32_Mul(F32_Mul(2.0f, w), x));
		cx = 1.0f	- F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, y), y);
		sz =		-(F32_Mul(F32_Mul(2.0f, x), y) - F32_Mul(F32_Mul(2.0f, w), z));
		cz = 1.0f	- F32_Mul(F32_Mul(2.0f, y), y) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
	} 
	else 
	{
#if 1
		//J |cosY|0ɋ߂ƂAZ]0Ɖ肵XY]߂
		sx =		F32_Mul(F32_Mul(2.0f, y), z) + F32_Mul(F32_Mul(2.0f, w), x);
		cx = 1.0f - F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = 0.0f;
#else
		//J |cosY|0ɋ߂ƂAX]0Ɖ肵YZ]߂
		sz =		F32_Mul(F32_Mul(2.0f, x), y) + F32_Mul(F32_Mul(2.0f, w), z);
		cz = 1.0f - F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = 0.0f;
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = 0.0f;
	return (pv0);
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*			FpuQuaternionToRotYXZ(IrisFVec4* pv0, const IrisFQuaternion* pq0)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	f32 sx, sy, cy, sz, cz;
	f32 rx, ry, rz;
	f32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sx = -(F32_Mul(F32_Mul(2.0f, y), z) - F32_Mul(F32_Mul(2.0f, w), x));
	if( sx <= 0.99995f && sx >= -0.99995f )
	{
		sy =		F32_Mul(F32_Mul(2.0f, z), x) + F32_Mul(F32_Mul(2.0f, w), y);
		cy = 1.0f - F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, y), y);
		sz =		F32_Mul(F32_Mul(2.0f, x), y) + F32_Mul(F32_Mul(2.0f, w), z);
		cz = 1.0f - F32_Mul(F32_Mul(2.0f, x), x) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = F32_Asin( sx );
		ry = F32_Atan2( sy, cy );
		rz = F32_Atan2( sz, cz );
	} 
	else 
	{
#if 1
		//J |cosX|0ɋ߂ƂAZ]0Ɖ肵YX]߂
		sy =		-(F32_Mul(F32_Mul(2.0f, z), x) - F32_Mul(F32_Mul(2.0f, w), y));
		cy = 1.0f	- F32_Mul(F32_Mul(2.0f, y), y) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = F32_Asin( sx );
		ry = F32_Atan2( sy, cy );
		rz = 0.0f;
#else
		//J |cosX|0ɋ߂ƂAY]0Ɖ肵XZ]߂
		sz =		-(F32_Mul(F32_Mul(2.0f, x), y) - F32_Mul(F32_Mul(2.0f, w), z));
		cz = 1.0f	- F32_Mul(F32_Mul(2.0f, y), y) - F32_Mul(F32_Mul(2.0f, z), z);
		rx = F32_Asin( sx );
		ry = 0.0f;
		rz = F32_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = 0.0f;
	return (pv0);
}

}	// end of namespace math
}	// end of namespace iris

#endif
