//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathTMatrix33.h
 * @brief		3x3}gbNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathTMatrix33_H_
#define INCG_IRIS_MathTMatrix33_H_

//======================================================================
// include
#include "MathTVector3.h"

namespace iris {
namespace math
{

//======================================================================
// declare
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Unit(IrisTMtx33<_TN>* pm0);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Zero(IrisTMtx33<_TN>* pm0);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Copy(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1);
template<typename _TN>IrisTVec3<_TN>*		TFpuMtx33Transform(IrisTVec3<_TN>* pv0, const IrisTMtx33<_TN>* pm0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Mul(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, const IrisTMtx33<_TN>* pm2);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Scale(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN s);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Transpose(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33RotZ(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN rz);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33RotIdxZ(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, u16 idz);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33RotY(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN ry);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33RotIdxY(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, u16 idy);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33RotX(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN rx);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33RotIdxX(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, u16 idx);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Rot(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, const IrisTVec3<_TN>* pv0);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33RotIdx(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, const IrisSVec3* pv0);
template<typename _TN>IrisBool				TFpuMtx33IsUnit(const IrisTMtx33<_TN>* pm0);
template<typename _TN>_TN					TFpuMtx33Trace(const IrisTMtx33<_TN>* pm0);
template<typename _TN>_TN					TFpuMtx33Determinant(const IrisTMtx33<_TN>* pm0);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Adjoint(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Inverse(IrisTMtx33<_TN>* pm0, _TN* pDeterminant, const IrisTMtx33<_TN>* pm1);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Normalize(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33TruncatePrecision24(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Identity(IrisTMtx33<_TN>* pm0);
template<typename _TN>IrisTMtx33<_TN>*		TFpuMtx33Null(IrisTMtx33<_TN>* pm0);
template<typename _TN>IrisTVec3<_TN>*		TFpuMtx33Apply(IrisTVec3<_TN>* pv0, const IrisTMtx33<_TN>* pm0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisBool				TFpuMtx33IsIdentity(const IrisTMtx33<_TN>* pm0);

//======================================================================
// inline
/**
 * @brief	Pʃ}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>* TFpuMtx33Unit(IrisTMtx33<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	pm0->x.x = pm0->y.y = pm0->z.z = 1;
	pm0->x.y = pm0->x.z = pm0->y.x = pm0->y.z = pm0->z.x = pm0->z.y = 0;
	return pm0;
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Zero(IrisTMtx33<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	TFpuVec3Zero(&pm0->x);
	TFpuVec3Zero(&pm0->y);
	TFpuVec3Zero(&pm0->z);
	return pm0;
}

/**
 * @brief	}gbNX̃Rs[
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Copy(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
#if 1
	pm0->_00 = pm1->_00;
	pm0->_01 = pm1->_01;
	pm0->_02 = pm1->_02;
	pm0->_10 = pm1->_10;
	pm0->_11 = pm1->_11;
	pm0->_12 = pm1->_12;
	pm0->_20 = pm1->_20;
	pm0->_21 = pm1->_21;
	pm0->_22 = pm1->_22;
#else
	TFpuVec3Copy(&pm0->x, &pm1->x);
	TFpuVec3Copy(&pm0->y, &pm1->y);
	TFpuVec3Copy(&pm0->z, &pm1->z);
#endif
	return pm0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*		TFpuMtx33Transform(IrisTVec3<_TN>* pv0, const IrisTMtx33<_TN>* pm0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	IRIS_ASSERT( pv0 != pv1 );	// TODO : AhXɖΉ
	pv0->x = F32_Mul(pm0->x.x, pv1->x) + F32_Mul(pm0->y.x, pv1->y) + F32_Mul(pm0->z.x, pv1->z);
	pv0->y = F32_Mul(pm0->x.y, pv1->x) + F32_Mul(pm0->y.y, pv1->y) + F32_Mul(pm0->z.y, pv1->z);
	pv0->z = F32_Mul(pm0->x.z, pv1->x) + F32_Mul(pm0->y.z, pv1->y) + F32_Mul(pm0->z.z, pv1->z);
	return pv0;
}

/**
 * @brief	Q̃}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pm2	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Mul(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, const IrisTMtx33<_TN>* pm2)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	MATH_FPU_NULLASSERT( pm2 );
	IRIS_ASSERT( pm0 != pm1 );
	IRIS_ASSERT( pm0 != pm2 );
	pm0->x.x = F32_Mul(pm1->x.x, pm2->x.x) + F32_Mul(pm1->y.x, pm2->x.y) + F32_Mul(pm1->z.x, pm2->x.z);
	pm0->x.y = F32_Mul(pm1->x.y, pm2->x.x) + F32_Mul(pm1->y.y, pm2->x.y) + F32_Mul(pm1->z.y, pm2->x.z);
	pm0->x.z = F32_Mul(pm1->x.z, pm2->x.x) + F32_Mul(pm1->y.z, pm2->x.y) + F32_Mul(pm1->z.z, pm2->x.z);
	pm0->y.x = F32_Mul(pm1->x.x, pm2->y.x) + F32_Mul(pm1->y.x, pm2->y.y) + F32_Mul(pm1->z.x, pm2->y.z);
	pm0->y.y = F32_Mul(pm1->x.y, pm2->y.x) + F32_Mul(pm1->y.y, pm2->y.y) + F32_Mul(pm1->z.y, pm2->y.z);
	pm0->y.z = F32_Mul(pm1->x.z, pm2->y.x) + F32_Mul(pm1->y.z, pm2->y.y) + F32_Mul(pm1->z.z, pm2->y.z);
	pm0->z.x = F32_Mul(pm1->x.x, pm2->z.x) + F32_Mul(pm1->y.x, pm2->z.y) + F32_Mul(pm1->z.x, pm2->z.z);
	pm0->z.y = F32_Mul(pm1->x.y, pm2->z.x) + F32_Mul(pm1->y.y, pm2->z.y) + F32_Mul(pm1->z.y, pm2->z.z);
	pm0->z.z = F32_Mul(pm1->x.z, pm2->z.x) + F32_Mul(pm1->y.z, pm2->z.y) + F32_Mul(pm1->z.z, pm2->z.z);
	return pm0;
}

/**
 * @brief	}gbNX̃XP[O
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	s	= XJ[l
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Scale(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN s)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	TFpuVec3Scale(&pm0->x, &pm1->x, s);
	TFpuVec3Scale(&pm0->y, &pm1->y, s);
	TFpuVec3Scale(&pm0->z, &pm1->z, s);
	return pm0;
}

/**
 * @brief	}gbNX̓]us߂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Transpose(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	pm0->x.x = pm1->x.x;
	_TN f01 = pm1->x.y;
	_TN f02 = pm1->x.z;
	pm0->x.y = pm1->y.x;
	_TN f11 = pm1->y.y;
	_TN f12 = pm1->y.z;
	pm0->x.z = pm1->z.x;
	pm0->y.x = f01;
	pm0->y.y = f11;
	pm0->y.z = pm1->z.y;
	pm0->z.x = f02;
	pm0->z.y = f12;
	pm0->z.z = pm1->z.z;
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33RotZ(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN rz)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_Cos(rz);
	_TN s = F32_Sin(rz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = 0;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = 0;
		pm0->z.x = 0;
		pm0->z.y = 0;
		pm0->z.z = 1;
	}
	else
	{
		IrisTMtx33<_TN> m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = 0;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = 0;
		m.z.x = 0;
		m.z.y = 0;
		m.z.z = 1;
		TFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33RotIdxZ(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, u16 idz)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_CosIdx(idz);
	_TN s = F32_SinIdx(idz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = 0;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = 0;
		pm0->z.x = 0;
		pm0->z.y = 0;
		pm0->z.z = 1;
	}
	else
	{
		IrisTMtx33<_TN> m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = 0;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = 0;
		m.z.x = 0;
		m.z.y = 0;
		m.z.z = 1;
		TFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	ry	= Y]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33RotY(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN ry)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_Cos(ry);
	_TN s = F32_Sin(ry);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = 0;
		pm0->x.z = +s;
		pm0->y.x = 0;
		pm0->y.y = 1;
		pm0->y.z = 0;
		pm0->z.x = -s;
		pm0->z.y = 0;
		pm0->z.z = +c;
	}
	else
	{
		IrisTMtx33<_TN> m;
		m.x.x = +c;
		m.x.y = 0;
		m.x.z = +s;
		m.y.x = 0;
		m.y.y = 1;
		m.y.z = 0;
		m.z.x = -s;
		m.z.y = 0;
		m.z.z = +c;
		TFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idy	= Y]ʁiCfbNXj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33RotIdxY(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, u16 idy)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_CosIdx(idy);
	_TN s = F32_SinIdx(idy);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = 0;
		pm0->x.z = +s;
		pm0->y.x = 0;
		pm0->y.y = 1;
		pm0->y.z = 0;
		pm0->z.x = -s;
		pm0->z.y = 0;
		pm0->z.z = +c;
	}
	else
	{
		IrisTMtx33<_TN> m;
		m.x.x = +c;
		m.x.y = 0;
		m.x.z = +s;
		m.y.x = 0;
		m.y.y = 1;
		m.y.z = 0;
		m.z.x = -s;
		m.z.y = 0;
		m.z.z = +c;
		TFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rx	= X]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33RotX(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, _TN rx)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_Cos(rx);
	_TN s = F32_Sin(rx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = 1;
		pm0->x.y = 0;
		pm0->x.z = 0;
		pm0->y.x = 0;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->z.x = 0;
		pm0->z.y = -s;
		pm0->z.z = +c;
	}
	else
	{
		IrisTMtx33<_TN> m;
		m.x.x = 1;
		m.x.y = 0;
		m.x.z = 0;
		m.y.x = 0;
		m.y.y = +c;
		m.y.z = +s;
		m.z.x = 0;
		m.z.y = -s;
		m.z.z = +c;
		TFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idx	= X]ʁiCfbNXj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33RotIdxX(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, u16 idx)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_CosIdx(idx);
	_TN s = F32_SinIdx(idx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = 1;
		pm0->x.y = 0;
		pm0->x.z = 0;
		pm0->y.x = 0;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->z.x = 0;
		pm0->z.y = -s;
		pm0->z.z = +c;
	}
	else
	{
		IrisTMtx33<_TN> m;
		m.x.x = 1;
		m.x.y = 0;
		m.x.z = 0;
		m.y.x = 0;
		m.y.y = +c;
		m.y.z = +s;
		m.z.x = 0;
		m.z.y = -s;
		m.z.z = +c;
		TFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Rot(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, const IrisTVec3<_TN>* pv0)
{
	TFpuMtx33RotZ(pm0, pm1, pv0->z);
	TFpuMtx33RotY(pm0, pm0, pv0->y);
	TFpuMtx33RotX(pm0, pm0, pv0->x);
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiCfbNXj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33RotIdx(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1, const IrisSVec3* pv0)
{
	TFpuMtx33RotIdxZ(pm0, pm1, (u16)pv0->z);
	TFpuMtx33RotIdxY(pm0, pm0, (u16)pv0->y);
	TFpuMtx33RotIdxX(pm0, pm0, (u16)pv0->x);
	return pm0;
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
template<typename _TN>IrisBool		TFpuMtx33IsUnit(const IrisTMtx33<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	const IrisMtx33* m = (const IrisMtx33*)(pm0);
	if( m->im.x.x != 0x3F800000 || m->im.y.y != 0x3F800000 || m->im.z.z != 0x3F800000 )
		return IRIS_FALSE;
	if( (             m->im.x.y | m->im.x.z 
		| m->im.y.x             | m->im.y.z
		| m->im.z.x | m->im.z.y             ) != 0 )
		return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	}gbNX̃g[XԂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	}gbNX̃g[X
*/
template<typename _TN>_TN			TFpuMtx33Trace(const IrisTMtx33<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	return (pm0->x.x + pm0->y.y + pm0->z.z);
}

/**
 * @brief	s񎮂Ԃ
 * @param [in]	pm0	= ̓}gbNX
 * @return	s
*/
template<typename _TN>_TN			TFpuMtx33Determinant(const IrisTMtx33<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN a11 = pm0->x.x;
	_TN a21 = pm0->x.y;
	_TN a31 = pm0->x.z;
	_TN a12 = pm0->y.x;
	_TN a22 = pm0->y.y;
	_TN a32 = pm0->y.z;
	_TN a13 = pm0->z.x;
	_TN a23 = pm0->z.y;
	_TN a33 = pm0->z.z;
	return ( F32_Mul( F32_Mul(a11, a22), a33) + F32_Mul( F32_Mul(a12, a23), a31)
		+ F32_Mul( F32_Mul(a13, a21), a32) - F32_Mul( F32_Mul(a13, a22), a31)
		- F32_Mul( F32_Mul(a12, a21), a33) - F32_Mul( F32_Mul(a11, a23), a32) );
}

/**
 * @brief	]qsԂ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Adjoint(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	_TN a11 = pm1->x.x;
	_TN a21 = pm1->x.y;
	_TN a31 = pm1->x.z;
	_TN a12 = pm1->y.x;
	_TN a22 = pm1->y.y;
	_TN a32 = pm1->y.z;
	_TN a13 = pm1->z.x;
	_TN a23 = pm1->z.y;
	_TN a33 = pm1->z.z;
	pm0->x.x =  ( F32_Mul(a22, a33) - F32_Mul(a23, a32) );
	pm0->x.y = -( F32_Mul(a21, a33) - F32_Mul(a23, a31) );
	pm0->x.z =  ( F32_Mul(a21, a32) - F32_Mul(a22, a31) );
	pm0->y.x = -( F32_Mul(a12, a33) - F32_Mul(a13, a32) );
	pm0->y.y =  ( F32_Mul(a11, a33) - F32_Mul(a13, a31) );
	pm0->y.z = -( F32_Mul(a11, a32) - F32_Mul(a12, a31) );
	pm0->z.x =  ( F32_Mul(a12, a23) - F32_Mul(a13, a22) );
	pm0->z.y = -( F32_Mul(a11, a23) - F32_Mul(a13, a21) );
	pm0->z.z =  ( F32_Mul(a11, a22) - F32_Mul(a12, a21) );
	return pm0;
}

/**
 * @brief	tsԂ
 * @param [out]	pm0				= o̓}gbNX
 * @param [out]	pDeterminant	= 
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Inverse(IrisTMtx33<_TN>* pm0, _TN* pDeterminant, const IrisTMtx33<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	_TN d = TFpuMtx33Determinant(pm1);
	if( pDeterminant != nullptr ) *pDeterminant = d;
	if( d == 0 ) return nullptr;
	TFpuMtx33Adjoint(pm0, pm1);
	TFpuMtx33Scale(pm0, pm0, F32_Div(1, d) );
	return pm0;
}

/**
 * @brief	}gNX̐K
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Normalize(IrisTMtx33<_TN>* pm0, const IrisTMtx33<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );

	// XY̊OςZ𓾂
	TFpuVec3OuterProduct(&pm0->z, &pm1->x, &pm1->y);
	// Oς瓾ZY̊OςX𓾂
	TFpuVec3OuterProduct(&pm0->x, &pm1->y, &pm0->z);
	TFpuVec3Normalize(&pm0->x, &pm0->x);
	TFpuVec3Normalize(&pm0->y, &pm1->y);
	TFpuVec3Normalize(&pm0->z, &pm0->z);
	return pm0;
}

template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Identity(IrisTMtx33<_TN>* pm0)
{
	return TFpuMtx33Unit(pm0);
}

template<typename _TN>IrisTMtx33<_TN>*	TFpuMtx33Null(IrisTMtx33<_TN>* pm0)
{
	return TFpuMtx33Zero(pm0);
}
	
template<typename _TN>IrisTVec3<_TN>*	TFpuMtx33Apply(IrisTVec3<_TN>* pv0, const IrisTMtx33<_TN>* pm0, const IrisTVec3<_TN>* pv1)
{
	return TFpuMtx33Transform(pv0, pm0, pv1);
}

template<typename _TN>IrisBool			TFpuMtx33IsIdentity(const IrisTMtx33<_TN>* pm0)
{
	return TFpuMtx33IsUnit(pm0);
}

}	// end of namespace math
}	// end of namespace iris

#endif
