//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathTVector3.h
 * @brief		3DxNgt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathTVector3_H_
#define INCG_IRIS_MathTVector3_H_

//======================================================================
// include
#include "../fpu/MathFpuDef.h"

namespace iris {
namespace math
{

//======================================================================
// declare
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Set(IrisTVec3<_TN>* pv0, _TN x, _TN y, _TN z);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Copy(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3PositiveZero(IrisTVec3<_TN>* pv0);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3NegativeZero(IrisTVec3<_TN>* pv0);
template<typename _TN>IrisIVec3*		TFpuVec3Ceil(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisIVec3*		TFpuVec3Trunc(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisIVec3*		TFpuVec3Round(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisIVec3*		TFpuVec3Floor(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3FromIVec3(IrisTVec3<_TN>* pv0, const IrisIVec3* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Add(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Sub(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Mul(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Div(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Neg(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Abs(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Lerp(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN t);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Scale(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, _TN s);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3ScaleAdd(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN s);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Hermite(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pt1
					  , const IrisTVec3<_TN>* pv2, const IrisTVec3<_TN>* pt2, _TN t);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Bezier(IrisTVec3<_TN>* pv0, s32 n, const IrisTVec3<_TN>* pva, _TN t);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Clamp(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, _TN min, _TN max);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Max(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Min(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>_TN				TFpuVec3InnerProduct(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3OuterProduct(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>_TN				TFpuVec3Funnel(const IrisTVec3<_TN>* pv0);
template<typename _TN>_TN				TFpuVec3Average(const IrisTVec3<_TN>* pv0);
template<typename _TN>IrisBool			TFpuVec3IsEqual(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisBool			TFpuVec3IsZero(const IrisTVec3<_TN>* pv0);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3SignFloat(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisIVec3*		TFpuVec3SignInt(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Normalize(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>_TN				TFpuVec3Length(const IrisTVec3<_TN>* pv0);
template<typename _TN>_TN				TFpuVec3Distance(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3FaceForward(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, const IrisTVec3<_TN>* pv3);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Reflect(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Refract(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN eta);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3TruncatePrecision24(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Null(IrisTVec3<_TN>* pv0);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Zero(IrisTVec3<_TN>* pv0);
template<typename _TN>_TN				TFpuVec3Dot(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Cross(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Subtract(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Multiply(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Divide(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2);
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Inter(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN t);

//======================================================================
// inline function
/**
 * @brief	vfݒ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	x	= xl
 * @param [in]	y	= yl
 * @param [in]	z	= zl
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Set(IrisTVec3<_TN>* pv0, _TN x, _TN y, _TN z)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = x; pv0->y = y; pv0->z = z;
	return pv0;
}

/**
 * @brief	vfRs[
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Copy(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = pv1->x; pv0->y = pv1->y; pv0->z = pv1->z;
	return pv0;
}

/**
 * @brief	+0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3PositiveZero(IrisTVec3<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = pv0->y = pv0->z = 0;
	return pv0;
}

/**
 * @brief	-0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3NegativeZero(IrisTVec3<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = pv0->y = pv0->z = -0;
	return pv0;
}

/**
 * @brief	؂グ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec3*	TFpuVec3Ceil(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Ceil(pv1->x) );
	pv0->y = (s32)( F64_Ceil(pv1->y) );
	pv0->z = (s32)( F64_Ceil(pv1->z) );
	return pv0;
}

/**
 * @brief	0ۂ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec3*	TFpuVec3Trunc(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Trunc(pv1->x) );
	pv0->y = (s32)( F64_Trunc(pv1->y) );
	pv0->z = (s32)( F64_Trunc(pv1->z) );
	return pv0;
}

/**
 * @brief	ߖTۂ(ľܓ)
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec3*	TFpuVec3Round(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Round(pv1->x) );
	pv0->y = (s32)( F64_Round(pv1->y) );
	pv0->z = (s32)( F64_Round(pv1->z) );
	return pv0;
}

/**
 * @brief	؂̂
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec3*	TFpuVec3Floor(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Floor(pv1->x) );
	pv0->y = (s32)( F64_Floor(pv1->y) );
	pv0->z = (s32)( F64_Floor(pv1->z) );
	return pv0;
}

/**
 * @brief	xNgϊ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3FromIVec3(IrisTVec3<_TN>* pv0, const IrisIVec3* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (_TN)( pv1->x );
	pv0->y = (_TN)( pv1->y );
	pv0->z = (_TN)( pv1->z );
	return pv0;
}

/**
 * @brief	xNg̉Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Add(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + pv2->x;
	pv0->y = pv1->y + pv2->y;
	pv0->z = pv1->z + pv2->z;
	return pv0;
}

/**
 * @brief	xNǧZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Sub(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x - pv2->x;
	pv0->y = pv1->y - pv2->y;
	pv0->z = pv1->z - pv2->z;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Mul(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x * pv2->x;
	pv0->y = pv1->y * pv2->y;
	pv0->z = pv1->z * pv2->z;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Div(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x / pv2->x;
	pv0->y = pv1->y / pv2->y;
	pv0->z = pv1->z / pv2->z;
	return pv0;
}

/**
 * @brief	xNg̊evf̕]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Neg(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = -pv1->x;
	pv0->y = -pv1->y;
	pv0->z = -pv1->z;
	return pv0;
}

/**
 * @brief	xNg̊evf̐Βl擾
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Abs(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = F64_ABS(pv1->x);
	pv0->y = F64_ABS(pv1->y);
	pv0->z = F64_ABS(pv1->z);
	return pv0;
}

/**
 * @brief	xNg̊Ԃ̓}
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Lerp(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + t * (pv2->x - pv1->x);
	pv0->y = pv1->y + t * (pv2->y - pv1->y);
	pv0->z = pv1->z + t * (pv2->z - pv1->z);
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Scale(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, _TN s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = pv1->x * s;
	pv0->y = pv1->y * s;
	pv0->z = pv1->z * s;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 *				pv0 = pv1 * s + vp2
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ZxNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3ScaleAdd(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x * s + pv2->x;
	pv0->y = pv1->y * s + pv2->y;
	pv0->z = pv1->z * s + pv2->z;
	return pv0;
}

/**
 * @brief	xNg̃G~[gXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pt1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pt2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Hermite(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pt1
									  , const IrisTVec3<_TN>* pv2, const IrisTVec3<_TN>* pt2, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pt1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pt2 );
	_TN t2 = t * t;
	_TN t3 = t * t2;
	IrisTVec3<_TN> v[4];
	TFpuVec3Scale(&v[0], pv1, (( 2 * t3) - (3 * t2) + 1));
	TFpuVec3Scale(&v[1], pt1, ((     t3) - (2 * t2) + t       ));
	TFpuVec3Scale(&v[3], pv2, ((-2 * t3) + (3 * t2)           ));
	TFpuVec3Scale(&v[2], pt2, ((     t3) - (    t2)           ));
	TFpuVec3Add(&v[0], &v[0], &v[1]);		// v[0] = v[0] + v[1]
	TFpuVec3Add(&v[0], &v[0], &v[2]);		// v[0] = v[0] + v[2]
	TFpuVec3Add(pv0,   &v[0], &v[3]);		// *pv0 = v[0] + v[3]
	return pv0;
}

/**
 * @brief	xNg̃xWGXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	n	= xNgz
 * @param [in]	pva	= xNgz
 * @param [in]	t	= 
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Bezier(IrisTVec3<_TN>* pv0, s32 n, const IrisTVec3<_TN>* pva, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pva );
	IRIS_ASSERT( n >= 2 );
	if( t <= 0 )	{ TFpuVec3Copy(pv0, &pva[0]); return pv0; }
	if( t >= 1 )	{ TFpuVec3Copy(pv0, &pva[n-1]); return pv0; }
	TFpuVec3PositiveZero(pv0);
	_TN tt = 1;			// t^ip
	_TN it = 1 - t;		// (1-t)^ip
	s32 nn = n-1;
	const IrisTVec3<_TN>* pv = pva;
	for( s32 i=0; i < n; ++i, ++pv )
	{
		_TN J = Combination(nn, i) * tt * F64_Pow(it, (f64)(nn-i));
		pv0->x += pv->x * J;
		pv0->y += pv->y * J;
		pv0->z += pv->z * J;
		tt = tt * t;
	}
	return pv0;
}

/**
 * @brief	xNg̃Nv
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	min	= ŏl
 * @param [in]	max	= ől
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Clamp(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, _TN min, _TN max)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < min) ? min : ((pv1->x > max) ? max : pv1->x);
	pv0->y = (pv1->y < min) ? min : ((pv1->y > max) ? max : pv1->y);
	pv0->z = (pv1->z < min) ? min : ((pv1->z > max) ? max : pv1->z);
	return pv0;
}

/**
 * @brief	xNg̊evf̑傫Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Max(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x > pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y > pv2->y) ? pv1->y : pv2->y;
	pv0->z = (pv1->z > pv2->z) ? pv1->z : pv2->z;
	return pv0;
}

/**
 * @brief	xNg̊evf̏Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Min(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y < pv2->y) ? pv1->y : pv2->y;
	pv0->z = (pv1->z < pv2->z) ? pv1->z : pv2->z;
	return pv0;
}

/**
 * @brief	
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	ϒl
*/
template<typename _TN>_TN		TFpuVec3InnerProduct(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	return pv0->x * pv1->x + pv0->y * pv1->y + pv0->z * pv1->z;
}

/**
 * @brief	O
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3OuterProduct(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->y * pv2->z - pv2->y * pv1->z;
	pv0->y = pv1->z * pv2->x - pv2->z * pv1->x;
	pv0->z = pv1->x * pv2->y - pv2->x * pv1->y;
	return pv0;
}

/**
 * @brief	xNg̊evf̑a擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̑a
*/
template<typename _TN>_TN		TFpuVec3Funnel(const IrisTVec3<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y + pv0->z);
}

/**
 * @brief	xNg̊evf̕ς擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̕
*/
template<typename _TN>_TN		TFpuVec3Average(const IrisTVec3<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y + pv0->z) / 3;
}

/**
 * @brief	xNgǂԂ
 * @param [in]	pv0	= rxNg
 * @param [in]	pv1	= rxNg
 * @return	^Ul
*/
template<typename _TN>IrisBool	TFpuVec3IsEqual(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	if(    (pv0->x != pv1->x)
		|| (pv0->y != pv1->y) 
		|| (pv0->z != pv1->z) ) return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	[xNgǂԂ
 * @param [in]	pv0	= ̓xNg
 * @return	^Ul
*/
template<typename _TN>IrisBool	TFpuVec3IsZero(const IrisTVec3<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	if( (pv0->x == 0) && (pv0->y == 0) && (pv0->z == 0) )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	,̕IrisTVec3<_TN>ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3SignFloat(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0) ? -1 : (pv1->x > 0) ? 1 : 0;
	pv0->y = (pv1->y < 0) ? -1 : (pv1->y > 0) ? 1 : 0;
	pv0->z = (pv1->z < 0) ? -1 : (pv1->z > 0) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	,̕IrisIVec3ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec3*	TFpuVec3SignInt(IrisIVec3* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0) ? -1 : (pv1->x > 0) ? 1 : 0;
	pv0->y = (pv1->y < 0) ? -1 : (pv1->y > 0) ? 1 : 0;
	pv0->z = (pv1->z < 0) ? -1 : (pv1->z > 0) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	xNg̐K
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Normalize(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	_TN x = pv1->x, y = pv1->y, z = pv1->z;
	_TN q = F64_Sqrt( x * x + y * y + z * z );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0 );
	q = 1 / q;
#else
	if( q != 0 ) q = 1 / q;
#endif
	pv0->x = x * q;
	pv0->y = y * q;
	pv0->z = z * q;
	return pv0;
}

/**
 * @brief	xNg̒Ԃ
 * @param [in]	pv0	= ̓xNg
 * @return	xNg̒
*/
template<typename _TN>_TN	TFpuVec3Length(const IrisTVec3<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (F64_Sqrt( pv0->x * pv0->x + pv0->y * pv0->y + pv0->z * pv0->z ));
}

/**
 * @brief	xNg̊Ԃ̋
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	xNg̒
*/
template<typename _TN>_TN	TFpuVec3Distance(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	_TN x = pv0->x - pv1->x;
	_TN y = pv0->y - pv1->y;
	_TN z = pv0->z - pv1->z;
	return (F64_Sqrt( x * x + y * y + z * z ));
}

/**
 * @brief	IuWFNg\ʂJɌ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pv3	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3FaceForward(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, const IrisTVec3<_TN>* pv3)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pv3 );
	_TN d = pv2->x * pv3->x + pv2->y * pv3->y + pv2->z * pv3->z;
	if(d < 0)
	{
		pv0->x = pv1->x;
		pv0->y = pv1->y;
		pv0->z = pv1->z;
	} 
	else
	{
		pv0->x = -pv1->x;
		pv0->y = -pv1->y;
		pv0->z = -pv1->z;
	}
	return pv0;
}

/**
 * @brief	˃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Reflect(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	_TN d = pv1->x * pv2->x + pv1->y * pv2->y + pv1->z * pv2->z;
	pv0->x = pv1->x - 2 * d * pv2->x;
	pv0->y = pv1->y - 2 * d * pv2->y;
	pv0->z = pv1->z - 2 * d * pv2->z;
	return pv0;
}

/**
 * @brief	܃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	eta	= ܗ
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Refract(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN eta)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	_TN d, e, f;
	d = pv1->x * pv2->x + pv1->y * pv2->y + pv1->z * pv2->z;
	e = 1 - eta * eta * (1 - d * d);
	if(e < 0)
	{
		pv0->x = 0;
		pv0->y = 0;
		pv0->z = 0;
	} 
	else
	{
		f = eta * d - F64_Sqrt(e);
		pv0->x = eta * pv1->x - f * pv2->x;
		pv0->y = eta * pv1->y - f * pv2->y;
		pv0->z = eta * pv1->z - f * pv2->z;
	}
	return pv0;
}

/**
 * @brief	0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Null(IrisTVec3<_TN>* pv0)
{
	return TFpuVec3PositiveZero(pv0); 
}

/**
 * @brief	0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Zero(IrisTVec3<_TN>* pv0)
{
	return TFpuVec3PositiveZero(pv0);
}

/**
 * @brief	0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>_TN				TFpuVec3Dot(const IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1)
{ 
	return TFpuVec3InnerProduct(pv0, pv1);
}
	
/**
 * @brief	O
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Cross(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	return TFpuVec3OuterProduct(pv0, pv1, pv2);
}

/**
 * @brief	xNǧZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Subtract(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{
	return TFpuVec3Sub(pv0, pv1, pv2);
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Multiply(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{ 
	return TFpuVec3Mul(pv0, pv1, pv2);
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Divide(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2)
{ 
	return TFpuVec3Div(pv0, pv1, pv2);
}

/**
 * @brief	xNg̊Ԃ̓}
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec3<_TN>*	TFpuVec3Inter(IrisTVec3<_TN>* pv0, const IrisTVec3<_TN>* pv1, const IrisTVec3<_TN>* pv2, _TN t)
{
	return TFpuVec3Lerp(pv0, pv1, pv2, t);
}

}	// end of namespace math
}	// end of namespace iris

#endif
