//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		stm32_bootloader.c
 * @brief		stm32 bootloader t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_STMBootloader_C_

//======================================================================
// include
#include "STMBootloader.h"
#include "c++0x/cpp0x_nullptr.h"

//======================================================================
// define
#define CMD_CONNECT	0x7F
#define CMD_ACK		0x79
#define CMD_NACK	0x1F


//======================================================================
// function

/**********************************************************************//**
 *
 * 1byte 
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [in]	byte	= ݒl
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_write_byte(STM32PORT port, unsigned char value)
{
	return stm32_bootloader_write(port, &value, 1);
}

/**********************************************************************//**
 *
 * ԐM҂
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_wait_ack(STM32PORT port)
{
	stm32cmd_t cmd = 0;
	while(1)
	{
		if( stm32_bootloader_read(port, &cmd, 1) ) break;
	}
	if( cmd != CMD_ACK )
	{
		if( cmd == CMD_NACK )
		{
			printf("stm32 bootloader error : NACK\n");
		}
		else
		{
			printf("stm32 bootloader error : Wait ack got %.2x\n", cmd);
		}
		return FALSE;
	}
	return TRUE;
}

/**********************************************************************//**
 *
 * ڑ֐
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_connect(STM32PORT port)
{
	stm32cmd_t cmd = CMD_CONNECT;
	stm32_bootloader_write(port, &cmd, 1);
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * ڑ֐
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_command(STM32PORT port, stm32cmd_t cmd)
{
	stm32cmd_t c[2] = { cmd, ~cmd };
	stm32_bootloader_write(port, c, 2);
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * ID 擾
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_get(STM32PORT port)
{
	stm32cmd_t cmd = 0;
	char N = 0;
	char i = 0;
	if( !stm32_bootloader_command(port, STM32BL_CMD_GET) ) return FALSE;
	stm32_bootloader_read(port, &N, 1);
	printf("N = %d\n", N);
	for( ; i < N+1; ++i )
	{
		stm32_bootloader_read(port, &cmd, 1);
		printf("0x%.2x\n", cmd);
	}
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * ID 擾
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [out]	pid		= o̓obt@
 * @param [in]	size	= o̓obt@TCY
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_get_id(STM32PORT port, void* pid, unsigned int size)
{
	char N=0;
	char buf;
	int i=0;
	if( pid == nullptr ) return FALSE;
	if( !stm32_bootloader_command(port, STM32BL_CMD_GETID) ) return FALSE;
	stm32_bootloader_read(port, &N, 1);
	if( size > (unsigned int)(N+1) ) size = N+1;
	stm32_bootloader_read(port, (char*)pid, size);
	for( i=size; i <= N; ++i )
	{
		stm32_bootloader_read(port, &buf, 1);
	}
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * ID ̃obt@TCY 擾
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [out]	n		= o̓obt@TCY
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_get_id_size(STM32PORT port, int* n)
{
	char N=0;
	int i=0;
	char dumy;
	if( !stm32_bootloader_command(port, STM32BL_CMD_GETID) ) return FALSE;
	stm32_bootloader_read(port, &N, 1);
	if( n != nullptr ) *n = N+1;
	for( i=0; i <= N; ++i )
	{
		stm32_bootloader_read(port, &dumy, 1);
	}
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * get version & protection status
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [out]	pver	= o[W
 * @param [out]	opbyte	= IvVoCg
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_get_ver(STM32PORT port, int* pver, int* opbyte)
{
	char v;
	char ob[2];
	if( !stm32_bootloader_command(port, STM32BL_CMD_GETVER) ) return FALSE;
	stm32_bootloader_read(port, &v, 1);
	stm32_bootloader_read(port, ob, 2);
	if( pver != nullptr ) *pver = v;
	if( opbyte != nullptr ) *opbyte = (ob[0] << 8) | ob[1];
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * read memory
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [out]	dst		= o̓obt@
 * @param [in]	address	= JnAhX
 * @param [in]	size	= TCY
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_read_memory(STM32PORT port, void* dst, unsigned int address, unsigned int size)
{
	stm32cmd_t cksum;
	unsigned int addr = address;
	unsigned int read = 0;

	while( read < size )
	{
		unsigned int len = size - read;
		if( !stm32_bootloader_command(port, STM32BL_CMD_READ_MEMORY) ) return FALSE;
		if( len > 256 ) len = 256;
#ifdef __BIG_ENDIAN__
		stm32_bootloader_write(port, &addr, 4);
#else
		{
			int i;
			unsigned char* p;
			p = (unsigned char*)&addr;
			for( i=4; i > 0; --i )
			{
				stm32_bootloader_write(port, &p[i-1], 1);
			}
		}
#endif
		cksum = (addr & 0xFF) ^ ((addr >> 8) & 0xFF) ^ ((addr >> 16) & 0xFF) ^ ((addr >> 24) & 0xFF);
		stm32_bootloader_write(port, &cksum, 1);

		if( !stm32_bootloader_wait_ack(port) ) return FALSE;

		stm32_bootloader_write_byte(port, len-1);

		cksum = 0xFF ^ (len-1);
		stm32_bootloader_write(port, &cksum, 1);

		if( !stm32_bootloader_wait_ack(port) ) return FALSE;

		stm32_bootloader_read(port, (char*)dst, len);

		read += len;
		addr += len;
	}
	return TRUE;
}

/**********************************************************************//**
 *
 * go
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [out]	address	= JnAhX
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_go(STM32PORT port, unsigned int address)
{
	stm32cmd_t cksum;
	if( !stm32_bootloader_command(port, STM32BL_CMD_GO) ) return FALSE;

#if defined(__BIG_ENDIAN__)
	stm32_bootloader_write(port, &address, 4);
#else
	{
		int i;
		unsigned char* p = (unsigned char*)&address;
		for( i=4; i > 0; --i )
		{
			stm32_bootloader_write(port, &p[i-1], 1);
		}
	}
#endif

	cksum = (address & 0xFF) ^ ((address >> 8) & 0xFF) ^ ((address >> 16) & 0xFF) ^ ((address >> 24) & 0xFF);
	stm32_bootloader_write(port, &cksum, 1);

#if 0	// sI҂Ȃ
	if( !stm32_bootloader_wait_ack(port) ) return FALSE;
#endif
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * write memory
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [in]	src		= ̓obt@
 * @param [in]	address	= JnAhX
 * @param [in]	size	= TCY
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_write_memory(STM32PORT port, const void* src, unsigned int address, unsigned int size)
{
	stm32cmd_t cksum;
	unsigned int addr = address;
	unsigned int write = 0;
	unsigned int i=0;
	char* p = (char*)src;

	while( write < size )
	{
		unsigned int len = size - write;
		if( !stm32_bootloader_command(port, STM32BL_CMD_WRITE_MEMORY) ) return FALSE;

		if( len > 256 ) len = 256;
#ifdef __BIG_ENDIAN__
		stm32_bootloader_write(port, &addr, 4);
#else
		{
			int i;
			unsigned char* paddr;
			paddr = (unsigned char*)&addr;
			for( i=4; i > 0; --i )
			{
				stm32_bootloader_write(port, &paddr[i-1], 1);
			}
		}
#endif
		cksum = (addr & 0xFF) ^ ((addr >> 8) & 0xFF) ^ ((addr >> 16) & 0xFF) ^ ((addr >> 24) & 0xFF);
		stm32_bootloader_write(port, &cksum, 1);

		if( !stm32_bootloader_wait_ack(port) ) return FALSE;

		stm32_bootloader_write_byte(port, len-1);

		cksum = (len-1);
		for( i=0; i < len; ++i, ++p )
		{
			cksum ^= *p;
			stm32_bootloader_write(port, (char*)p, 1);
		}

		stm32_bootloader_write(port, &cksum, 1);

		if( !stm32_bootloader_wait_ack(port) ) return FALSE;

		write += len;
		addr += len;
	}

	return TRUE;
}

/**********************************************************************//**
 *
 * verify
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [in]	src		= robt@
 * @param [in]	address	= rJnAhX
 * @param [in]	size	= rTCY
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_verify(STM32PORT port, const void* src, unsigned int address, unsigned int size)
{
	BOOL ret = TRUE;
	stm32cmd_t cksum;
	unsigned int addr = address;
	unsigned int read = 0;
	char buf;
	const char* ps = (const char*)src;
	unsigned int i=0;

	while( read < size )
	{
		unsigned int len = size - read;
		if( !stm32_bootloader_command(port, STM32BL_CMD_READ_MEMORY) ) return FALSE;
		if( len > 256 ) len = 256;
#ifdef __BIG_ENDIAN__
		stm32_bootloader_write(port, &addr, 4);
#else
		{
			unsigned char* p;
			p = (unsigned char*)&addr;
			for( i=4; i > 0; --i )
			{
				stm32_bootloader_write(port, &p[i-1], 1);
			}
		}
#endif
		cksum = (addr & 0xFF) ^ ((addr >> 8) & 0xFF) ^ ((addr >> 16) & 0xFF) ^ ((addr >> 24) & 0xFF);
		stm32_bootloader_write(port, &cksum, 1);

		if( !stm32_bootloader_wait_ack(port) ) return FALSE;

		stm32_bootloader_write_byte(port, len-1);

		cksum = 0xFF ^ (len-1);
		stm32_bootloader_write(port, &cksum, 1);

		if( !stm32_bootloader_wait_ack(port) ) return FALSE;

		for( i=0; i < len; ++i, ++ps )
		{
			stm32_bootloader_read(port, &buf, 1);
			if( buf != *ps ) ret = FALSE;
		}
		read += len;
		addr += len;
	}
	return ret;
}

/**********************************************************************//**
 *
 * erase memory
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [in]	pagenum	= PAGE_NUM -1 or 0xFF(erase all)
 * @param [in]  pages	= PAGEԍz
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_erase_memory_pages(STM32PORT port, unsigned char pagenum, char* pages)
{
	if( !stm32_bootloader_command(port, STM32BL_CMD_ERASE_MEMORY) ) return FALSE;

	stm32_bootloader_write(port, &pagenum, 1);

	if( pagenum == 0xFF )
	{
		pagenum = 0;
		stm32_bootloader_write(port, &pagenum, 1);
	}
	else
	{
		int i=0;
		char* p = pages;
		stm32cmd_t cksum = pagenum;
		for( ; i <= pagenum; ++i, ++p )
		{
			cksum ^= *p;
			stm32_bootloader_write(port, p, 1);
		}
		stm32_bootloader_write(port, &cksum, 1);
	}

	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * erase memory
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [in]  pages	= PAGEԍ
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_erase_memory_page(STM32PORT port, char page)
{
	return stm32_bootloader_erase_memory_pages(port, 0, &page);
}

/**********************************************************************//**
 *
 * erase memory
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_erase_memory_all(STM32PORT port)
{
	return stm32_bootloader_erase_memory_pages(port, 0xFF, nullptr);
}

/**********************************************************************//**
 *
 * erase memory
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [in]	pagenum	= PAGE_NUM -1 or 0xFFFF(Mass) or 0xFFFE(Bank1) or 0xFFFD(Bank2)
 * @param [in]  pages	= PAGEԍz
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_extended_erase_memory(STM32PORT port, unsigned short pagenum, short* pages)
{
	stm32cmd_t cksum = (stm32cmd_t)(pagenum & 0xFF) ^ ((pagenum >> 8) & 0xFF);
	if( !stm32_bootloader_command(port, STM32BL_CMD_EXTENDED_ERASE) ) return FALSE;

#ifdef __BIG_ENDIAN__
	stm32_bootloader_write(port, &pagenum, 2);
#else
	{
		int i;
		unsigned char* p;
		p = (unsigned char*)&pagenum;
		for( i=2; i > 0; --i )
		{
			stm32_bootloader_write(port, &p[i-1], 1);
		}
	}
#endif

	if( pagenum == 0xFFFF || pagenum == 0xFFFE || pagenum == 0xFFFD )
	{
		cksum = 0xFF ^ (pagenum & 0xFF);
	}
	else
	{
		int i=0;
		short* p = pages;
		for( ; i <= pagenum; ++i, ++p )
		{
#ifdef __BIG_ENDIAN__
			cksum ^= (*p & 0xFF) ^ ((*p >> 8) & 0xFF);
			stm32_bootloader_write(port, p, 2);
#else
			int j;
			unsigned char* pp = (unsigned char*)p;
			for( j=2; j > 0; --j )
			{
				cksum ^= pp[i-1];
				stm32_bootloader_write(port, &pp[i-1], 1);
			}
#endif
		}
	}
	stm32_bootloader_write(port, &cksum, 1);

	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * write protect
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @param [in]	sectors	= 
 * @param [in]  size	= sectors TCY
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_write_protect(STM32PORT port, stm32cmd_t* sectors, unsigned char size)
{
	int i=0;
	stm32cmd_t* p = sectors;
	stm32cmd_t cksum = size;
	if( !stm32_bootloader_command(port, STM32BL_CMD_WRITE_PROTECT) ) return FALSE;

	stm32_bootloader_write(port, &size, 1);

	for( ; i <= size; ++i, ++p )
	{
		cksum ^= *p;
		stm32_bootloader_write(port, p, 1);
	}
	
	stm32_bootloader_write(port, &cksum, 1);
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * write unprotect
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_write_unprotect(STM32PORT port)
{
	if( !stm32_bootloader_command(port, STM32BL_CMD_WRITE_UNPROTECT) ) return FALSE;
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * readout protect
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_readout_protect(STM32PORT port)
{
	if( !stm32_bootloader_command(port, STM32BL_CMD_READOUT_PROTECT) ) return FALSE;
	return stm32_bootloader_wait_ack(port);
}

/**********************************************************************//**
 *
 * readout unprotect
 * 
 ----------------------------------------------------------------------
 * @param [in]	port	= |[gnh
 * @return	
*//***********************************************************************/
BOOL stm32_bootloader_readout_unprotect(STM32PORT port)
{
	if( !stm32_bootloader_command(port, STM32BL_CMD_READOUT_UNPROTECT) ) return FALSE;
	return stm32_bootloader_wait_ack(port);
}

