//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		XFBezierCurve.cpp
 * @brief		xf32 xWGȐt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_XFBezierCurve_CPP_

//======================================================================
// include
#include "XFBezierCurve.h"
#include "../../math/MathPractical.h"
#include "../XFPower.h"
#include "../../iris_debug.h"

namespace iris {
namespace xf
{

//======================================================================
// function
/**********************************************************************//**
 *
 * xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [in]	t	= (0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	...	= n̓_
 * @return o͒l
*//***********************************************************************/
xf32	XF_BezierCurve(xf32 t, s32 n, ...)
{
	IRIS_ASSERT( n > 0 );
	xf32 ret = XF32_ZERO;
	va_list va;
	va_start(va, n);
	xf32 tt = XF32_ONE;			// t^ip
	xf32 it = XF32_ONE - t;		// (1-t)^ip
	s32 nn = n-1;
	if( t <= XF32_ZERO ) { ret = (xf32)va_arg(va, xf32_va); goto func_end; }
	if( t >= XF32_ONE )
	{
		for( s32 i=0; i < n-1; ++i ) (void)va_arg(va, xf32_va);
		ret = (xf32)va_arg(va, xf32_va);
		goto func_end;
	}

	for( s32 i=0; i < n; ++i )
	{
		xf32 B = (xf32)va_arg(va, xf32_va);
		xf32 J = ::iris::math::Combination(nn, i) * XF32_Mul(tt, XF_Pow(it, nn-i));
		//xf32 J = XF_BernsteinPolynomial(nn, i, t);
		ret += XF32_Mul(B, J);
		tt = XF32_Mul(tt, t);
	}
func_end:
	va_end(va);
	return ret;
}

/**********************************************************************//**
 *
 * xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [in]	t	= ( 0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	pt	= _̔z(n)
 * @return o͒l
*//***********************************************************************/
xf32	XF_BezierCurveArray(xf32 t, s32 n, xf32 pt[])
{
	IRIS_ASSERT( pt != nullptr );
	IRIS_ASSERT( n >= 2 );
	xf32 ret = XF32_ZERO;
	if( t <= XF32_ZERO )	{ return *pt; }
	if( t >= XF32_ONE )		{ return pt[n-1]; }

	xf32 tt = XF32_ONE;			// t^ip
	xf32 it = XF32_ONE - t;		// (1-t)^ip
	s32 nn = n-1;
	xf32* pp = pt;
	for( s32 i=0; i < n; ++i, ++pp )
	{
		xf32 B = *pp;
		xf32 J = ::iris::math::Combination(nn, i) * XF32_Mul(tt, XF_Pow(it, nn-i));
		//xf32 J = XF_BernsteinPolynomial(nn, i, t);
		ret += XF32_Mul(B, J);
		tt = XF32_Mul(tt, t);
	}
	return ret;
}

/**********************************************************************//**
 *
 * NxNg̃xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [out]	pV	= xNg
 * @param [in]	N	= 
 * @param [in]	t	= (0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	...	= n̃xNg(xf32*)
 * @return o͒l
*//***********************************************************************/
xf32*	XF_BezierCurveN(xf32* pV, s32 N, xf32 t, s32 n, ...)
{
	IRIS_ASSERT( pV != nullptr );
	IRIS_ASSERT( N > 0 );
	IRIS_ASSERT( n > 0 );
	for( s32 i=0; i < N; ++i ) pV[i] = XF32_ZERO;
	va_list va;
	va_start(va, n);
	xf32 tt = XF32_ONE;			// t^ip
	xf32 it = XF32_ONE - t;		// (1-t)^ip
	s32 nn = n-1;
	if( t <= XF32_ZERO )
	{
		xf32* pv = va_arg(va, xf32*);
		for( s32 i=0; i < N; ++i, ++pv )
			pV[i] = *pv;
		goto func_end;
	}
	if( t >= XF32_ONE )
	{
		for( s32 i=0; i < n-1; ++i )
			(void)va_arg(va, xf32*);

		xf32* pv = va_arg(va, xf32*);
		for( s32 i=0; i < N; ++i, ++pv )
			pV[i] = *pv;
		goto func_end;
	}

	for( s32 i=0; i < n; ++i )
	{
		xf32* pv = va_arg(va, xf32*);
		//xf32 J = XF_BernsteinPolynomial(nn, i, t);
		xf32 J = ::iris::math::Combination(nn, i) * XF32_Mul(tt, XF_Pow(it, nn-i));
		for( s32 j=0; j < N; ++j, ++pv )
		{
			pV[j] += XF32_Mul(*pv, J);
		}
		tt = XF32_Mul(tt, t);
	}
func_end:
	va_end(va);
	return pV;
}

/**********************************************************************//**
 *
 * NxNg̃xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [out]	pV	= xNg
 * @param [in]	N	= 
 * @param [in]	t	= ( 0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	pt	= _̔z(n)
 * @return o͒l
*//***********************************************************************/
xf32*	XF_BezierCurveArrayN(xf32* pV, s32 N, xf32 t, s32 n, xf32 pt[])
{
	IRIS_ASSERT( pV != nullptr );
	IRIS_ASSERT( N > 0 );
	IRIS_ASSERT( pt != nullptr );
	IRIS_ASSERT( n >= 2 );
	if( t <= XF32_ZERO )	{ for(s32 i=0; i < N; ++i) pV[i] = pt[i];			return pV; }
	if( t >= XF32_ONE )		{ for(s32 i=0; i < N; ++i) pV[i] = pt[(n-1)*N+i];	return pV; }

	for( s32 j=0; j < N; ++j )	{ pV[j] = XF32_ZERO; }

	xf32 tt = XF32_ONE;			// t^ip
	xf32 it = XF32_ONE - t;		// (1-t)^ip
	s32 nn = n-1;
	xf32* pp = pt;
	for( s32 i=0; i < n; ++i)
	{
		xf32 J = ::iris::math::Combination(nn, i) * XF32_Mul(tt, XF_Pow(it, nn-i));
		//xf32 J = XF_BernsteinPolynomial(nn, i, t);
		for( s32 j=0; j < N; ++j, ++pp )
		{
			pV[j] += XF32_Mul(*pp, J);
		}
		tt = XF32_Mul(tt, t);
	}
	return pV;
}

}	// end of namespace xf
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "../../iris_using.h"
#include "../../iris_iostream.h"

//======================================================================
// test
IRIS_UNITTEST(CXFBezierCurveUnitTest, XFBezierCurveUnitTest)
{
	xf32 P[] = {
		XF32_CONST(0.0f),
		XF32_CONST(50.0f),
		XF32_CONST(100.0f),
		XF32_CONST(150.0f)
	};
	s32 N = sizeof(P)/sizeof(xf32);

	IrisXFVec2 V[] = 
	{
		{ XF32_CONST(0.0f), XF32_CONST(150.0f), },
		{ XF32_CONST(50.0f), XF32_CONST(100.0f), },
		{ XF32_CONST(100.0f), XF32_CONST(50.0f), },
		{ XF32_CONST(150.0f), XF32_CONST(0.0f), },
	};
	s32 NV = sizeof(V)/sizeof(IrisXFVec2);

	for( int i=0; i < 11; ++i )
	{
		std::cout << XF_BezierCurveArray(XF32_CONST(i/10.0f), N, P) << std::endl;
	}

	std::cout << "------------------------------" << std::endl;
	for( int i=0; i < 11; ++i )
	{
		std::cout << XF_BezierCurve(XF32_CONST(i/10.0f), N, P[0], P[1], P[2], P[3]) << std::endl;
	}

	std::cout << "------------------------------" << std::endl;
	for( int i=0; i < 11; ++i )
	{
		IrisXFVec2 tmp;
		xf32* p = (xf32*)&V;
		XF_BezierCurveArrayN((xf32*)&tmp
			, 2, XF32_CONST(i/10.0f), NV, p);

		std::cout << tmp.x << ", " << tmp.y  << std::endl;
	}

	std::cout << "------------------------------" << std::endl;
	for( int i=0; i < 11; ++i )
	{
		IrisXFVec2 tmp;
		XF_BezierCurveN((xf32*)&tmp
			, 2, XF32_CONST(i/10.0f), NV
			, (xf32*)&V[0], (xf32*)&V[1], (xf32*)&V[2], (xf32*)&V[3]);

		std::cout << tmp.x << ", " << tmp.y  << std::endl;
	}
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
