//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathXFColor.inl
 * @brief		J[Zt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathXFColor_inl_
#define INCG_IRIS_MathXFColor_inl_

namespace iris {
namespace xf
{

//======================================================================
// function
/**
 * @brief	J[^̏
 * @param [out]	pc	= o̓J[
 * @param [in]	r	= Rl
 * @param [in]	g	= Gl
 * @param [in]	b	= Bl
 * @param [in]	a	= Al
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorSet(IrisXFColor* pc, xf32 r, xf32 g, xf32 b, xf32 a)
{
	XFPU_NULLASSERT( pc );
	pc->r = r;
	pc->g = g;
	pc->b = b;
	pc->a = a;
	return pc;
}

/**
 * @brief	J[^̏
 * @param [out]	pc	= o̓J[
 * @param [in]	r	= Rl
 * @param [in]	g	= Gl
 * @param [in]	b	= Bl
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorSetRGB(IrisXFColor* pc, xf32 r, xf32 g, xf32 b)
{
	XFPU_NULLASSERT( pc );
	pc->r = r;
	pc->g = g;
	pc->b = b;
	return pc;
}

/**
 * @brief	J[^̃V[Rs[
 * @param [out]	pc0	= o̓J[
 * @param [in]	pc1	= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorCopy(IrisXFColor* pc0, const IrisXFColor* pc1)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->r = pc1->r;
	pc0->g = pc1->g;
	pc0->b = pc1->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̃[
 * @param [out]	pc0	= o̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorZero(IrisXFColor* pc0)
{
	XFPU_NULLASSERT( pc0 );
	pc0->r = XF32_ZERO;
	pc0->g = XF32_ZERO;
	pc0->b = XF32_ZERO;
	pc0->a = XF32_ZERO;
	return pc0;
}

/**
 * @brief	J[^RGBA8888ɕϊ
 * @param [in]	pc	= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisRGBA8888	XFpuColorToRGBA8888(const IrisXFColor* pc)
{
	XFPU_NULLASSERT( pc );
	IrisRGBA8888 rgba;
	rgba.r = (u8)(( pc->r < XF32_ZERO ) ? 0x00 : ( pc->r > XF32_255 ) ? 0xFF : XF_XF32_TO_U32(pc->r * 0xFF) ); 
	rgba.g = (u8)(( pc->g < XF32_ZERO ) ? 0x00 : ( pc->g > XF32_255 ) ? 0xFF : XF_XF32_TO_U32(pc->g * 0xFF) ); 
	rgba.b = (u8)(( pc->b < XF32_ZERO ) ? 0x00 : ( pc->b > XF32_255 ) ? 0xFF : XF_XF32_TO_U32(pc->b * 0xFF) ); 
	rgba.a = (u8)(( pc->a < XF32_ZERO ) ? 0x00 : ( pc->a > XF32_255 ) ? 0xFF : XF_XF32_TO_U32(pc->a * 0xFF) ); 
	return rgba;
}

/**
 * @brief	J[^RGBA4444ɕϊ
 * @param [in]	pc	= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisRGBA4444	XFpuColorToRGBA4444(const IrisXFColor* pc)
{
	XFPU_NULLASSERT( pc );
	IrisRGBA4444 rgba;
	rgba.r = ( pc->r < XF32_ZERO ) ? 0x0 : ( pc->r > XF32_ONE ) ? 0xF : (u8)(XF_XF32_TO_U32(pc->r * 0xF)); 
	rgba.g = ( pc->g < XF32_ZERO ) ? 0x0 : ( pc->g > XF32_ONE ) ? 0xF : (u8)(XF_XF32_TO_U32(pc->g * 0xF)); 
	rgba.b = ( pc->b < XF32_ZERO ) ? 0x0 : ( pc->b > XF32_ONE ) ? 0xF : (u8)(XF_XF32_TO_U32(pc->b * 0xF)); 
	rgba.a = ( pc->a < XF32_ZERO ) ? 0x0 : ( pc->a > XF32_ONE ) ? 0xF : (u8)(XF_XF32_TO_U32(pc->a * 0xF)); 
	return rgba;
}

/**
 * @brief	J[^RGBA5551ɕϊ
 * @param [in]	pc	= ̓J[
 * @param [in]	q	= 臒l
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisRGBA5551	XFpuColorToRGBA5551(const IrisXFColor* pc, xf32 q)
{
	XFPU_NULLASSERT( pc );
	IrisRGBA5551 rgba;
	rgba.r = ( pc->r < XF32_ZERO ) ? 0x0 : ( pc->r > XF32_ONE ) ? 0x1F : (u8)(XF_XF32_TO_U32(pc->r * 0x1F)); 
	rgba.g = ( pc->g < XF32_ZERO ) ? 0x0 : ( pc->g > XF32_ONE ) ? 0x1F : (u8)(XF_XF32_TO_U32(pc->g * 0x1F)); 
	rgba.b = ( pc->b < XF32_ZERO ) ? 0x0 : ( pc->b > XF32_ONE ) ? 0x1F : (u8)(XF_XF32_TO_U32(pc->b * 0x1F)); 
	rgba.a = ( pc->a <= q ) ? 0 : 1; 
	return rgba;
}

/**
 * @brief	J[^RGB565ɕϊ
 * @param [in]	pc	= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisRGB565		XFpuColorToRGB565(const IrisXFColor* pc)
{
	XFPU_NULLASSERT( pc );
	IrisRGB565 rgb;
	rgb.r = ( pc->r < XF32_ZERO ) ? 0x0 : ( pc->r > XF32_ONE ) ? 0xF : (u8)(XF_XF32_TO_U32(pc->r * 0x1F)); 
	rgb.g = ( pc->g < XF32_ZERO ) ? 0x0 : ( pc->g > XF32_ONE ) ? 0xF : (u8)(XF_XF32_TO_U32(pc->g * 0x3F)); 
	rgb.b = ( pc->b < XF32_ZERO ) ? 0x0 : ( pc->b > XF32_ONE ) ? 0xF : (u8)(XF_XF32_TO_U32(pc->b * 0x1F)); 
	return rgb;
}

/**
 * @brief	J[^CMYɕϊ
 * @param [out]	pcmy	= o̓J[
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFCMY*		XFpuColorToCMY(IrisXFCMY*  pcmy, const IrisXFColor* pc)
{
	XFPU_NULLASSERT( pcmy );
	XFPU_NULLASSERT( pc );
	pcmy->c = XF32_ONE - pc->r;
	pcmy->m = XF32_ONE - pc->g;
	pcmy->y = XF32_ONE - pc->b;
	return pcmy;
}

/**
 * @brief	J[^CMYKɕϊ
 * @param [out]	pcmyk	= o̓J[
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFCMYK*		XFpuColorToCMYK(IrisXFCMYK*  pcmyk, const IrisXFColor* pc)
{
	XFPU_NULLASSERT( pcmyk );
	XFPU_NULLASSERT( pc );
	xf32 c = XF32_ONE - pc->r;
	xf32 m = XF32_ONE - pc->g;
	xf32 y = XF32_ONE - pc->b;
	xf32 k = min(c, min(m, y));
	xf32 tk = XF32_ONE - k;
	pcmyk->c = XF32_Div((c - k) , tk);
	pcmyk->m = XF32_Div((m - k) , tk);
	pcmyk->y = XF32_Div((y - k) , tk);
	pcmyk->k = k;
	return pcmyk;
}

/**
 * @brief	J[^HSVɕϊ
 * @param [out]	phsv	= o̓J[
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFHSV*		XFpuColorToHSV(IrisXFHSV* phsv, const IrisXFColor* pc)
{
	XFPU_NULLASSERT( phsv );
	XFPU_NULLASSERT( pc );
	xf32 fmax = max(max(pc->r, pc->g), pc->b);
	xf32 fmin = min(min(pc->r, pc->g), pc->b);
	xf32 diff = fmax - fmin;
	xf32 h = XF32_ZERO;
	xf32 s = XF32_Div(diff, fmax);
	if( diff != XF32_ZERO )
	{
		if( fmax == pc->r )			h = XF32_Mul(XF32_60, XF32_Div((pc->g - pc->b), diff));
		else if( fmax == pc->g )	h = XF32_Mul(XF32_60, XF32_Div((pc->b - pc->r), diff)) + XF32_60*2;
		else if( fmax == pc->b )	h = XF32_Mul(XF32_60, XF32_Div((pc->r - pc->g), diff)) + XF32_60*4;
		if( h < XF32_ZERO ) h += XF32_360;
		h = XF32_Div(h, XF32_360);	// 0 ` 1 ɐK
	}
	phsv->h = h;
	phsv->s = s;
	phsv->v = fmax;
	return phsv;
}

/**
 * @brief	RGBA8888J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgba	= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorFromRGBA8888(IrisXFColor* pc, IrisRGBA8888 rgba)
{
	XFPU_NULLASSERT( pc );
	pc->r = XF_S32_TO_XF32(rgba.r) / 0xFF;
	pc->g = XF_S32_TO_XF32(rgba.g) / 0xFF;
	pc->b = XF_S32_TO_XF32(rgba.b) / 0xFF;
	pc->a = XF_S32_TO_XF32(rgba.a) / 0xFF;
	return pc;
}

/**
 * @brief	RGBA4444J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgba	= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorFromRGBA4444(IrisXFColor* pc, IrisRGBA4444 rgba)
{
	XFPU_NULLASSERT( pc );
	pc->r = XF_S32_TO_XF32(rgba.r) / 0xF;
	pc->g = XF_S32_TO_XF32(rgba.g) / 0xF;
	pc->b = XF_S32_TO_XF32(rgba.b) / 0xF;
	pc->a = XF_S32_TO_XF32(rgba.a) / 0xF;
	return pc;
}

/**
 * @brief	RGBA5551J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgba	= ̓J[
 * @param [in]	q		= 臒l
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorFromRGBA5551(IrisXFColor* pc, IrisRGBA5551 rgba, xf32 q)
{
	XFPU_NULLASSERT( pc );
	pc->r = XF_S32_TO_XF32(rgba.r) / 0x1F;
	pc->g = XF_S32_TO_XF32(rgba.g) / 0x1F;
	pc->b = XF_S32_TO_XF32(rgba.b) / 0x1F;
	pc->a = rgba.a ? q : XF32_ZERO;
	return pc;
}

/**
 * @brief	RGB555J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgb		= ̓J[
 * @param [in]	q		= At@
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorFromRGB565(IrisXFColor* pc, IrisRGB565 rgb, xf32 q)
{
	XFPU_NULLASSERT( pc );
	pc->r = XF_S32_TO_XF32(rgb.r) / 0x1F;
	pc->g = XF_S32_TO_XF32(rgb.g) / 0x3F;
	pc->b = XF_S32_TO_XF32(rgb.b) / 0x1F;
	pc->a = q;
	return pc;
}

/**
 * @brief	CMYJ[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	pcmy	= ̓J[
 * @param [in]	q		= At@
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorFromCMY(IrisXFColor* pc, const IrisXFCMY* pcmy, xf32 q)
{
	XFPU_NULLASSERT( pc );
	XFPU_NULLASSERT( pcmy );
	pc->r = XF32_ONE - pcmy->c;
	pc->g = XF32_ONE - pcmy->m;
	pc->b = XF32_ONE - pcmy->y;
	pc->a = q;
	return pc;
}

/**
 * @brief	CMYKJ[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	pcmy	= ̓J[
 * @param [in]	q		= At@
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorFromCMYK(IrisXFColor* pc, const IrisXFCMYK* pcmyk, xf32 q)
{
	XFPU_NULLASSERT( pc );
	XFPU_NULLASSERT( pcmyk );
	xf32 tk = XF32_ONE - pcmyk->k;
	pc->r = XF32_ONE - min(XF32_ONE, XF32_Mul(pcmyk->c, tk) + pcmyk->k);
	pc->g = XF32_ONE - min(XF32_ONE, XF32_Mul(pcmyk->m, tk) + pcmyk->k);
	pc->b = XF32_ONE - min(XF32_ONE, XF32_Mul(pcmyk->y, tk) + pcmyk->k);
	pc->a = q;
	return pc;
}

/**
 * @brief	HSVJ[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	pcmy	= ̓J[
 * @param [in]	q		= At@
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorFromHSV(IrisXFColor* pc, const IrisXFHSV* phsv, xf32 q)
{
	XFPU_NULLASSERT( pc );
	XFPU_NULLASSERT( phsv );
	if( phsv->s == XF32_ZERO )
	{
		pc->r = pc->g = pc->b = phsv->v;
	}
	else
	{
		xf32 V = phsv->v;
		xf32 th = phsv->h * 6;
		s32  iH = XF_XF32_TO_S32(th);
		xf32 f = th - XF_S32_TO_XF32(iH);
		xf32 p = XF32_Mul(V, XF32_ONE - phsv->s);
		xf32 q = XF32_Mul(V, XF32_ONE - XF32_Mul(f, phsv->s));
		xf32 t = XF32_Mul(V, XF32_ONE - XF32_Mul(XF32_ONE - f, phsv->s));
		xf32 r=0, g=0, b=0;
		switch(iH)
		{
		case 0:
			r = V; g = t; b = p;	break;
		case 1:
			r = q; g = V; b = p;	break;
		case 2:
			r = p; g = V; b = t;	break;
		case 3:
			r = p; g = q; b = V;	break;
		case 4:
			r = t; g = p; b = V;	break;
		case 5:
			r = V; g = p; b = q;	break;
		default:
			IRIS_ASSERT( iH >= 0 && iH < 6 );		break;
		}
		pc->r = r;
		pc->g = g;
		pc->b = b;
	}
	pc->a = q;
	return pc;
}

/**
 * @brief	J[^̉Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorAdd(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + pc2->r;
	pc0->g = pc1->g + pc2->g;
	pc0->b = pc1->b + pc2->b;
	pc0->a = pc1->a + pc2->a;
	return pc0;
}

/**
 * @brief	J[^RGBZ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorAddRGB(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + pc2->r;
	pc0->g = pc1->g + pc2->g;
	pc0->b = pc1->b + pc2->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̌Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorSub(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = pc1->r - pc2->r;
	pc0->g = pc1->g - pc2->g;
	pc0->b = pc1->b - pc2->b;
	pc0->a = pc1->a - pc2->a;
	return pc0;
}

/**
 * @brief	J[^RGBZ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorSubRGB(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = pc1->r - pc2->r;
	pc0->g = pc1->g - pc2->g;
	pc0->b = pc1->b - pc2->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̏Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorMul(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = XF32_Mul(pc1->r, pc2->r);
	pc0->g = XF32_Mul(pc1->g, pc2->g);
	pc0->b = XF32_Mul(pc1->b, pc2->b);
	pc0->a = XF32_Mul(pc1->a, pc2->a);
	return pc0;
}

/**
 * @brief	J[^RGBZ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorMulRGB(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = XF32_Mul(pc1->r, pc2->r);
	pc0->g = XF32_Mul(pc1->g, pc2->g);
	pc0->b = XF32_Mul(pc1->b, pc2->b);
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^1̕␔(RGBA)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorNeg(IrisXFColor* pc0, const IrisXFColor* pc1)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->r = XF32_ONE - pc1->r;
	pc0->g = XF32_ONE - pc1->g;
	pc0->b = XF32_ONE - pc1->b;
	pc0->a = XF32_ONE - pc1->a;
	return pc0;
}

/**
 * @brief	J[^1̕␔(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorNegRGB(IrisXFColor* pc0, const IrisXFColor* pc1)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->r = XF32_ONE - pc1->r;
	pc0->g = XF32_ONE - pc1->g;
	pc0->b = XF32_ONE - pc1->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	2̃J[Ԃ̓}
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @param [in]	t		= 
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorLerp(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2, xf32 t)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + XF32_Mul(t, (pc2->r - pc1->r));
	pc0->g = pc1->g + XF32_Mul(t, (pc2->g - pc1->g));
	pc0->b = pc1->b + XF32_Mul(t, (pc2->b - pc1->b));
	pc0->a = pc1->a + XF32_Mul(t, (pc2->a - pc1->a));
	return pc0;
}

/**
 * @brief	2̃J[Ԃ̓}(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @param [in]	t		= 
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorLerpRGB(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2, xf32 t)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + XF32_Mul(t, (pc2->r - pc1->r));
	pc0->g = pc1->g + XF32_Mul(t, (pc2->g - pc1->g));
	pc0->b = pc1->b + XF32_Mul(t, (pc2->b - pc1->b));
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̃XJ[l̏Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	s		= XJ[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorScale(IrisXFColor* pc0, const IrisXFColor* pc1, xf32 s)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->r = XF32_Mul(pc1->r, s);
	pc0->g = XF32_Mul(pc1->g, s);
	pc0->b = XF32_Mul(pc1->b, s);
	pc0->a = XF32_Mul(pc1->a, s);
	return pc0;
}

/**
 * @brief	J[^̃XJ[l̏Z(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	s		= XJ[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorScaleRGB(IrisXFColor* pc0, const IrisXFColor* pc1, xf32 s)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->r = XF32_Mul(pc1->r, s);
	pc0->g = XF32_Mul(pc1->g, s);
	pc0->b = XF32_Mul(pc1->b, s);
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̃Nv
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	min		= ŏ
 * @param [in]	max		= ő
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorClamp(IrisXFColor* pc0, const IrisXFColor* pc1, xf32 min, xf32 max)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->r = (pc1->r < min) ? min : ((pc1->r > max) ? max : pc1->r);
	pc0->g = (pc1->g < min) ? min : ((pc1->g > max) ? max : pc1->g);
	pc0->b = (pc1->b < min) ? min : ((pc1->b > max) ? max : pc1->b);
	pc0->a = (pc1->a < min) ? min : ((pc1->a > max) ? max : pc1->a);
	return pc0;
}

/**
 * @brief	J[^̃Nv(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	min		= ŏ
 * @param [in]	max		= ő
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorClampRGB(IrisXFColor* pc0, const IrisXFColor* pc1, xf32 min, xf32 max)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->r = (pc1->r < min) ? min : ((pc1->r > max) ? max : pc1->r);
	pc0->g = (pc1->g < min) ? min : ((pc1->g > max) ? max : pc1->g);
	pc0->b = (pc1->b < min) ? min : ((pc1->b > max) ? max : pc1->b);
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̊evf̑傫Ԃ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorMax(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = (pc1->r > pc2->r) ? pc1->r : pc2->r;
	pc0->g = (pc1->g > pc2->g) ? pc1->g : pc2->g;
	pc0->b = (pc1->b > pc2->b) ? pc1->b : pc2->b;
	pc0->a = (pc1->a > pc2->a) ? pc1->a : pc2->a;
	return pc0;
}

/**
 * @brief	J[^̊evf̏Ԃ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorMin(IrisXFColor* pc0, const IrisXFColor* pc1, const IrisXFColor* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->r = (pc1->r < pc2->r) ? pc1->r : pc2->r;
	pc0->g = (pc1->g < pc2->g) ? pc1->g : pc2->g;
	pc0->b = (pc1->b < pc2->b) ? pc1->b : pc2->b;
	pc0->a = (pc1->a < pc2->a) ? pc1->a : pc2->a;
	return pc0;
}

/**
 * @brief	J[^ǂԂ
 * @param [in]	pc0		= ̓J[
 * @param [in]	pc1		= ̓J[
 * @return	^Ul
*/
IRIS_XFPU_INLINE IrisBool		XFpuColorIsEqual(const IrisXFColor* pc0, const IrisXFColor* pc1)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	if( pc0->r == pc1->r && pc0->g == pc1->g && pc0->b == pc1->b && pc0->a == pc1->a )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	J[^ǂԂ
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisBool		XFpuColorIsZero(const IrisXFColor* pc)
{
	XFPU_NULLASSERT( pc );
	const IrisIVec4* pv = (const IrisIVec4*)pc;
	if( ((pv->x | pv->y | pv->z | pv->w) & 0x7FFFFFFF)==0 )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	J[^̐K
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorNormalize(IrisXFColor* pc0, const IrisXFColor* pc1)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	xf32 q = pc1->r;
	if(q < pc1->g) q = pc1->g;
	if(q < pc1->b) q = pc1->b;
	if(q < pc1->a) q = pc1->a;
	if(q != XF32_ZERO) q = XF32_Div(XF32_ONE, q);

	pc0->r = XF32_Mul(pc1->r, q);
	pc0->g = XF32_Mul(pc1->g, q);
	pc0->b = XF32_Mul(pc1->b, q);
	pc0->a = XF32_Mul(pc1->a, q);
	return pc0;
}

/**
 * @brief	J[^̐K(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @return	o̓J[
*/
IRIS_XFPU_INLINE IrisXFColor*	XFpuColorNormalizeRGB(IrisXFColor* pc0, const IrisXFColor* pc1)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	xf32 q = pc1->r;
	if(q < pc1->g) q = pc1->g;
	if(q < pc1->b) q = pc1->b;
	if(q != XF32_ZERO) q = XF32_Div(XF32_ONE, q);

	pc0->r = XF32_Mul(pc1->r, q);
	pc0->g = XF32_Mul(pc1->g, q);
	pc0->b = XF32_Mul(pc1->b, q);
	pc0->a = pc1->a;
	return pc0;
}


}	// end of namespace xf
}	// end of namespace iris

#endif
