//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathXFMatrix33.inl
 * @brief		3x3}gbNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathXFMatrix33_inl_
#define INCG_IRIS_MathXFMatrix33_inl_

namespace iris {
namespace xf
{

//======================================================================
// function
/**
 * @brief	Pʃ}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Unit(IrisXFMtx33* pm0)
{
	XFPU_NULLASSERT( pm0 );
	pm0->x.x = pm0->y.y = pm0->z.z = XF32_ONE;
	pm0->x.y = pm0->x.z = pm0->y.x = pm0->y.z = pm0->z.x = pm0->z.y = XF32_ZERO;
	return pm0;
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Zero(IrisXFMtx33* pm0)
{
	XFPU_NULLASSERT( pm0 );
	XFpuVec3Zero(&pm0->x);
	XFpuVec3Zero(&pm0->y);
	XFpuVec3Zero(&pm0->z);
	return pm0;
}

/**
 * @brief	}gbNX̃Rs[
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Copy(IrisXFMtx33* pm0, const IrisXFMtx33* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
#if 1
	pm0->_00 = pm1->_00;
	pm0->_01 = pm1->_01;
	pm0->_02 = pm1->_02;
	pm0->_10 = pm1->_10;
	pm0->_11 = pm1->_11;
	pm0->_12 = pm1->_12;
	pm0->_20 = pm1->_20;
	pm0->_21 = pm1->_21;
	pm0->_22 = pm1->_22;
#else
	XFpuVec3Copy(&pm0->x, &pm1->x);
	XFpuVec3Copy(&pm0->y, &pm1->y);
	XFpuVec3Copy(&pm0->z, &pm1->z);
#endif
	return pm0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec3*	XFpuMtx33Transform(IrisXFVec3* pv0, const IrisXFMtx33* pm0, const IrisXFVec3* pv1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	IRIS_ASSERT( pv0 != pv1 );	// TODO : AhXɖΉ
	pv0->x = XF32_Mul(pm0->x.x, pv1->x) + XF32_Mul(pm0->y.x, pv1->y) + XF32_Mul(pm0->z.x, pv1->z);
	pv0->y = XF32_Mul(pm0->x.y, pv1->x) + XF32_Mul(pm0->y.y, pv1->y) + XF32_Mul(pm0->z.y, pv1->z);
	pv0->z = XF32_Mul(pm0->x.z, pv1->x) + XF32_Mul(pm0->y.z, pv1->y) + XF32_Mul(pm0->z.z, pv1->z);
	return pv0;
}

/**
 * @brief	Q̃}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pm2	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Mul(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, const IrisXFMtx33* pm2)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	XFPU_NULLASSERT( pm2 );
	IRIS_ASSERT( pm0 != pm1 );
	IRIS_ASSERT( pm0 != pm2 );
	pm0->x.x = XF32_Mul(pm1->x.x, pm2->x.x) + XF32_Mul(pm1->y.x, pm2->x.y) + XF32_Mul(pm1->z.x, pm2->x.z);
	pm0->x.y = XF32_Mul(pm1->x.y, pm2->x.x) + XF32_Mul(pm1->y.y, pm2->x.y) + XF32_Mul(pm1->z.y, pm2->x.z);
	pm0->x.z = XF32_Mul(pm1->x.z, pm2->x.x) + XF32_Mul(pm1->y.z, pm2->x.y) + XF32_Mul(pm1->z.z, pm2->x.z);
	pm0->y.x = XF32_Mul(pm1->x.x, pm2->y.x) + XF32_Mul(pm1->y.x, pm2->y.y) + XF32_Mul(pm1->z.x, pm2->y.z);
	pm0->y.y = XF32_Mul(pm1->x.y, pm2->y.x) + XF32_Mul(pm1->y.y, pm2->y.y) + XF32_Mul(pm1->z.y, pm2->y.z);
	pm0->y.z = XF32_Mul(pm1->x.z, pm2->y.x) + XF32_Mul(pm1->y.z, pm2->y.y) + XF32_Mul(pm1->z.z, pm2->y.z);
	pm0->z.x = XF32_Mul(pm1->x.x, pm2->z.x) + XF32_Mul(pm1->y.x, pm2->z.y) + XF32_Mul(pm1->z.x, pm2->z.z);
	pm0->z.y = XF32_Mul(pm1->x.y, pm2->z.x) + XF32_Mul(pm1->y.y, pm2->z.y) + XF32_Mul(pm1->z.y, pm2->z.z);
	pm0->z.z = XF32_Mul(pm1->x.z, pm2->z.x) + XF32_Mul(pm1->y.z, pm2->z.y) + XF32_Mul(pm1->z.z, pm2->z.z);
	return pm0;
}

/**
 * @brief	}gbNX̃XP[O
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	s	= XJ[l
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Scale(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, xf32 s)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	XFpuVec3Scale(&pm0->x, &pm1->x, s);
	XFpuVec3Scale(&pm0->y, &pm1->y, s);
	XFpuVec3Scale(&pm0->z, &pm1->z, s);
	return pm0;
}

/**
 * @brief	}gbNX̓]us߂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Transpose(IrisXFMtx33* pm0, const IrisXFMtx33* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	pm0->x.x = pm1->x.x;
	xf32 f01 = pm1->x.y;
	xf32 f02 = pm1->x.z;
	pm0->x.y = pm1->y.x;
	xf32 f11 = pm1->y.y;
	xf32 f12 = pm1->y.z;
	pm0->x.z = pm1->z.x;
	pm0->y.x = f01;
	pm0->y.y = f11;
	pm0->y.z = pm1->z.y;
	pm0->z.x = f02;
	pm0->z.y = f12;
	pm0->z.z = pm1->z.z;
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33RotZ(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, xf32 rz)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_Cos(rz);
	xf32 s = XF32_Sin(rz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = XF32_ZERO;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = XF32_ZERO;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = XF32_ONE;
	}
	else
	{
		IrisXFMtx33 m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = XF32_ZERO;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = XF32_ZERO;
		m.z.x = XF32_ZERO;
		m.z.y = XF32_ZERO;
		m.z.z = XF32_ONE;
		XFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33RotIdxZ(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, u16 idz)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_CosIdx(idz);
	xf32 s = XF32_SinIdx(idz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = XF32_ZERO;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = XF32_ZERO;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = XF32_ONE;
	}
	else
	{
		IrisXFMtx33 m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = XF32_ZERO;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = XF32_ZERO;
		m.z.x = XF32_ZERO;
		m.z.y = XF32_ZERO;
		m.z.z = XF32_ONE;
		XFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	ry	= Y]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33RotY(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, xf32 ry)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_Cos(ry);
	xf32 s = XF32_Sin(ry);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = +s;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = XF32_ONE;
		pm0->y.z = XF32_ZERO;
		pm0->z.x = -s;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = +c;
	}
	else
	{
		IrisXFMtx33 m;
		m.x.x = +c;
		m.x.y = XF32_ZERO;
		m.x.z = +s;
		m.y.x = XF32_ZERO;
		m.y.y = XF32_ONE;
		m.y.z = XF32_ZERO;
		m.z.x = -s;
		m.z.y = XF32_ZERO;
		m.z.z = +c;
		XFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idy	= Y]ʁiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33RotIdxY(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, u16 idy)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_CosIdx(idy);
	xf32 s = XF32_SinIdx(idy);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = +s;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = XF32_ONE;
		pm0->y.z = XF32_ZERO;
		pm0->z.x = -s;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = +c;
	}
	else
	{
		IrisXFMtx33 m;
		m.x.x = +c;
		m.x.y = XF32_ZERO;
		m.x.z = +s;
		m.y.x = XF32_ZERO;
		m.y.y = XF32_ONE;
		m.y.z = XF32_ZERO;
		m.z.x = -s;
		m.z.y = XF32_ZERO;
		m.z.z = +c;
		XFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rx	= X]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33RotX(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, xf32 rx)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_Cos(rx);
	xf32 s = XF32_Sin(rx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = XF32_ONE;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = XF32_ZERO;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = -s;
		pm0->z.z = +c;
	}
	else
	{
		IrisXFMtx33 m;
		m.x.x = XF32_ONE;
		m.x.y = XF32_ZERO;
		m.x.z = XF32_ZERO;
		m.y.x = XF32_ZERO;
		m.y.y = +c;
		m.y.z = +s;
		m.z.x = XF32_ZERO;
		m.z.y = -s;
		m.z.z = +c;
		XFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idx	= X]ʁiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33RotIdxX(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, u16 idx)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_CosIdx(idx);
	xf32 s = XF32_SinIdx(idx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = XF32_ONE;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = XF32_ZERO;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = -s;
		pm0->z.z = +c;
	}
	else
	{
		IrisXFMtx33 m;
		m.x.x = XF32_ONE;
		m.x.y = XF32_ZERO;
		m.x.z = XF32_ZERO;
		m.y.x = XF32_ZERO;
		m.y.y = +c;
		m.y.z = +s;
		m.z.x = XF32_ZERO;
		m.z.y = -s;
		m.z.z = +c;
		XFpuMtx33Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Rot(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, const IrisXFVec3* pv0)
{
	XFpuMtx33RotZ(pm0, pm1, pv0->z);
	XFpuMtx33RotY(pm0, pm0, pv0->y);
	XFpuMtx33RotX(pm0, pm0, pv0->x);
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33RotIdx(IrisXFMtx33* pm0, const IrisXFMtx33* pm1, const IrisSVec3* pv0)
{
	XFpuMtx33RotIdxZ(pm0, pm1, (u16)pv0->z);
	XFpuMtx33RotIdxY(pm0, pm0, (u16)pv0->y);
	XFpuMtx33RotIdxX(pm0, pm0, (u16)pv0->x);
	return pm0;
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
IRIS_XFPU_INLINE IrisBool		XFpuMtx33IsUnit(const IrisXFMtx33* pm0)
{
	XFPU_NULLASSERT( pm0 );
	const IrisMtx33* m = (const IrisMtx33*)(pm0);
	if( m->im.x.x != 0x3F800000 || m->im.y.y != 0x3F800000 || m->im.z.z != 0x3F800000 )
		return IRIS_FALSE;
	if( (             m->im.x.y | m->im.x.z 
		| m->im.y.x             | m->im.y.z
		| m->im.z.x | m->im.z.y             ) != 0 )
		return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	}gbNX̃g[XԂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	}gbNX̃g[X
*/
IRIS_XFPU_INLINE xf32			XFpuMtx33Trace(const IrisXFMtx33* pm0)
{
	XFPU_NULLASSERT( pm0 );
	return (pm0->x.x + pm0->y.y + pm0->z.z);
}

/**
 * @brief	s񎮂Ԃ
 * @param [in]	pm0	= ̓}gbNX
 * @return	s
*/
IRIS_XFPU_INLINE xf32			XFpuMtx33Determinant(const IrisXFMtx33* pm0)
{
	XFPU_NULLASSERT( pm0 );
	xf32 a11 = pm0->x.x;
	xf32 a21 = pm0->x.y;
	xf32 a31 = pm0->x.z;
	xf32 a12 = pm0->y.x;
	xf32 a22 = pm0->y.y;
	xf32 a32 = pm0->y.z;
	xf32 a13 = pm0->z.x;
	xf32 a23 = pm0->z.y;
	xf32 a33 = pm0->z.z;
	return ( XF32_Mul( XF32_Mul(a11, a22), a33) + XF32_Mul( XF32_Mul(a12, a23), a31)
		+ XF32_Mul( XF32_Mul(a13, a21), a32) - XF32_Mul( XF32_Mul(a13, a22), a31)
		- XF32_Mul( XF32_Mul(a12, a21), a33) - XF32_Mul( XF32_Mul(a11, a23), a32) );
}

/**
 * @brief	]qsԂ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Adjoint(IrisXFMtx33* pm0, const IrisXFMtx33* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	xf32 a11 = pm1->x.x;
	xf32 a21 = pm1->x.y;
	xf32 a31 = pm1->x.z;
	xf32 a12 = pm1->y.x;
	xf32 a22 = pm1->y.y;
	xf32 a32 = pm1->y.z;
	xf32 a13 = pm1->z.x;
	xf32 a23 = pm1->z.y;
	xf32 a33 = pm1->z.z;
	pm0->x.x =  ( XF32_Mul(a22, a33) - XF32_Mul(a23, a32) );
	pm0->x.y = -( XF32_Mul(a21, a33) - XF32_Mul(a23, a31) );
	pm0->x.z =  ( XF32_Mul(a21, a32) - XF32_Mul(a22, a31) );
	pm0->y.x = -( XF32_Mul(a12, a33) - XF32_Mul(a13, a32) );
	pm0->y.y =  ( XF32_Mul(a11, a33) - XF32_Mul(a13, a31) );
	pm0->y.z = -( XF32_Mul(a11, a32) - XF32_Mul(a12, a31) );
	pm0->z.x =  ( XF32_Mul(a12, a23) - XF32_Mul(a13, a22) );
	pm0->z.y = -( XF32_Mul(a11, a23) - XF32_Mul(a13, a21) );
	pm0->z.z =  ( XF32_Mul(a11, a22) - XF32_Mul(a12, a21) );
	return pm0;
}

/**
 * @brief	tsԂ
 * @param [out]	pm0				= o̓}gbNX
 * @param [out]	pDeterminant	= 
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Inverse(IrisXFMtx33* pm0, xf32* pDeterminant, const IrisXFMtx33* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	xf32 d = XFpuMtx33Determinant(pm1);
	if( pDeterminant != nullptr ) *pDeterminant = d;
	if( d == XF32_ZERO ) return nullptr;
	XFpuMtx33Adjoint(pm0, pm1);
	XFpuMtx33Scale(pm0, pm0, XF32_Div(XF32_ONE, d) );
	return pm0;
}

/**
 * @brief	}gNX̐K
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33Normalize(IrisXFMtx33* pm0, const IrisXFMtx33* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );

	// XY̊OςZ𓾂
	XFpuVec3OuterProduct(&pm0->z, &pm1->x, &pm1->y);
	// Oς瓾ZY̊OςX𓾂
	XFpuVec3OuterProduct(&pm0->x, &pm1->y, &pm0->z);
	XFpuVec3Normalize(&pm0->x, &pm0->x);
	XFpuVec3Normalize(&pm0->y, &pm1->y);
	XFpuVec3Normalize(&pm0->z, &pm0->z);
	return pm0;
}

/**
 * @brief	8rbg̐x؂̂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx33*	XFpuMtx33TruncatePrecision24(IrisXFMtx33* pm0, const IrisXFMtx33* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );

	XFpuVec3TruncatePrecision24(&pm0->x, &pm1->x);
	XFpuVec3TruncatePrecision24(&pm0->y, &pm1->y);
	XFpuVec3TruncatePrecision24(&pm0->z, &pm1->z);
	return pm0;
}

}	// end of namespace xf
}	// end of namespace iris

#endif
