//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FBXMesh.cpp
 * @brief		FBX SDK meshNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FBXMesh_CPP_

//======================================================================
// include
#include "FBXMesh.h"

#ifdef _IRIS_SUPPORT_FBX
#include "../FBXNode.h"
#include "../../gl/GXGL.h"
#include "../../../iris_debug.h"

namespace iris {
namespace gx {
namespace fbx
{

//======================================================================
// class
/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CFBXMesh::CFBXMesh(void)
: m_pVertex(nullptr)
, m_Time(0)
{
}

/**********************************************************************//**
 *
 * RXgN^
 *
 ----------------------------------------------------------------------
 * @param [in] ptr	= FBXm[hNX|C^
*//***********************************************************************/
CFBXMesh::CFBXMesh(kfbx_ptr ptr)
: m_pVertex(nullptr)
, m_Time(0)
{
	Attach(ptr);
}

/**********************************************************************//**
 *
 * RXgN^
 *
 ----------------------------------------------------------------------
 * @param [in] obj	= bVNX
*//***********************************************************************/
CFBXMesh::CFBXMesh(CFBXMesh& obj)
: m_pVertex(nullptr)
, m_Time(0)
{
	Attach(obj.m_pKfbx);
}

/**********************************************************************//**
 *
 * fXgN^
 *
*//***********************************************************************/
CFBXMesh::~CFBXMesh(void)
{
	Release();
}

/**********************************************************************//**
 *
 * ֘At
 *
 ----------------------------------------------------------------------
 * @param [in]	pMesh	= KFbxMesh*
 * @return	
*//***********************************************************************/
bool CFBXMesh::Attach(KFbxMesh* pMesh)
{
	if( pMesh == nullptr ) return false;
	Release();

	m_pKfbx = pMesh;
	return true;
}

/**********************************************************************//**
 *
 * Ǎ
 *
 ----------------------------------------------------------------------
 * @param [in]	pMesh	= KFbxMesh*
 * @return	
*//***********************************************************************/
bool CFBXMesh::Load(KFbxMesh* pMesh)
{
	if( pMesh == nullptr ) return false;
	Release();

	m_pKfbx = pMesh;
	int VtxNum = m_pKfbx->GetControlPointsCount();
	if( VtxNum == 0 ) return false;
	m_pVertex = new KFbxVector4 [VtxNum];
	memcpy(m_pVertex, m_pKfbx->GetControlPoints(), sizeof(KFbxVector4)*VtxNum);
	return true;
}

/**********************************************************************//**
 *
 * ēǍ
 *
 ----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CFBXMesh::Reload(void)
{
	KFbxMesh* pMesh = m_pKfbx;
	Release();
	if( pMesh == nullptr ) return false;

	m_pKfbx = pMesh;
	int VtxNum = m_pKfbx->GetControlPointsCount();
	if( VtxNum == 0 ) return false;
	m_pVertex = new KFbxVector4 [VtxNum];
	memcpy(m_pVertex, m_pKfbx->GetControlPoints(), sizeof(KFbxVector4)*VtxNum);
	return true;
}

/**********************************************************************//**
 *
 * 
 *
*//***********************************************************************/
void CFBXMesh::Release(void)
{
	m_pKfbx = nullptr;
	IRIS_SAFE_DELETE_ARRAY(m_pVertex);
}

/**********************************************************************//**
 *
 * XV
 *
 ----------------------------------------------------------------------
 * @param [in]	time	= XVb
*//***********************************************************************/
void CFBXMesh::Update(xf32 time)
{
	KTime ktime;
	ktime.SetSecondDouble(static_cast<double>(XF_XF32_TO_F32(time)));
	m_Time += ktime;

	//int DeformerNum = m_pKfbx->GetDeformerCount(KFbxDeformer::eVERTEX_CACHE);
	KFbxVertexCacheDeformer* pVtxDeformer = static_cast<KFbxVertexCacheDeformer*>(m_pKfbx->GetDeformer(0, KFbxDeformer::eVERTEX_CACHE));
	if( pVtxDeformer->IsActive() )
	{
		ReadVertexCacheData(ktime);
	}
	else
	{
		if( m_pKfbx->GetShapeCount() )
		{
			ShapeDeformation(ktime);
		}

		int SkinNum = m_pKfbx->GetDeformerCount(KFbxDeformer::eSKIN);
		int ClusterNum = 0;
		for( int i=0; i < SkinNum; ++i )
			ClusterNum += (static_cast<KFbxSkin*>(m_pKfbx->GetDeformer(i, KFbxDeformer::eSKIN)))->GetClusterCount();
		if( ClusterNum > 0 )
		{
			ClusterDeformation(ktime);
		}
	}
}

/**********************************************************************//**
 *
 * `
 *
*//***********************************************************************/
void CFBXMesh::Draw(void)
{
	// TODO : Ƃ肠OpenGLŕ`
	KFbxLayerElementArrayTemplate<KFbxVector2>* pUVArray = nullptr;
	m_pKfbx->GetTextureUV(&pUVArray, KFbxLayerElement::eDIFFUSE_TEXTURES);

    KFbxLayerElement::EMappingMode MappingMode = KFbxLayerElement::eNONE;

	KFbxLayer* pLayer = m_pKfbx->GetLayer(0);
	if( (pLayer != nullptr) && (pLayer->GetUVs() != nullptr) )
		MappingMode = pLayer->GetUVs()->GetMappingMode();

	GLenum Primitive = GL_TRIANGLES;
	gx::gl::gxglLoad3DMatrix();

	const KFbxVector4* pVertex = m_pVertex != nullptr ? m_pVertex : m_pKfbx->GetControlPoints();
	int PolygonNum = m_pKfbx->GetPolygonCount();
	glEnableClientState(GL_VERTEX_ARRAY);

    glBegin(Primitive);
	glVertexPointer(3, GL_FLOAT, sizeof(float), pVertex);
	glDrawElements(Primitive, PolygonNum*3, GL_INT, m_pKfbx->GetPolygonVertices());
    glEnd();

	glDisableClientState(GL_VERTEX_ARRAY);
    gx::gl::gxglUnload3DMatrix();
}

/**********************************************************************//**
 *
 * }eA̎擾
 *
 ----------------------------------------------------------------------
 * @return }eA
*//***********************************************************************/
int CFBXMesh::GetMaterialNum(void) const
{
	if( m_pKfbx == nullptr ) return 0;
	const KFbxNode* node = m_pKfbx->GetNode();
	return node->GetMaterialCount();
}

/**********************************************************************//**
 *
 * }eA̎擾
 *
 ----------------------------------------------------------------------
 * @param [in]	idx	= }eACfbNX
 * @return }eA
*//***********************************************************************/
KFbxSurfaceMaterial* CFBXMesh::GetMaterial(int idx) const
{
	if( m_pKfbx == nullptr ) return nullptr;
	const KFbxNode* node = m_pKfbx->GetNode();
	return node->GetMaterial(idx);
}

/**********************************************************************//**
 *
 * LbVf[^璸_擾
 *
 ----------------------------------------------------------------------
 * @param [in]	rTime	= XV
 * @return 
*//***********************************************************************/
bool CFBXMesh::ReadVertexCacheData(KTime& rTime)
{
	KFbxVertexCacheDeformer* pVtxDeformer = static_cast<KFbxVertexCacheDeformer*>(m_pKfbx->GetDeformer(0, KFbxDeformer::eVERTEX_CACHE));
    KFbxCache* pCache = pVtxDeformer->GetCache();
	unsigned int VtxNum = static_cast<unsigned int>(m_pKfbx->GetControlPointsCount());
	unsigned int PointsNum = VtxNum * 3;
	double* pReadBuf = new double [PointsNum];

	if( pCache->GetCacheFileFormat() == KFbxCache::eMC )
	{
		int index = pCache->GetChannelIndex(pVtxDeformer->GetCacheChannel());
		if( index < 0 ) return false;
		if( !pCache->Read(index, rTime, pReadBuf, VtxNum) ) return false;
	}
	else
	{
		if( !pCache->Read((int)rTime.GetFrame(true), rTime, pReadBuf, VtxNum) ) return false;
	}

	{
		double* buf = pReadBuf;
		KFbxVector4* pVertex = m_pVertex != nullptr ? m_pVertex : m_pKfbx->GetControlPoints();
		for( unsigned int index = 0; index < VtxNum; ++index )
		{
			pVertex[index].SetAt(0, *(buf++));
			pVertex[index].SetAt(1, *(buf++));
			pVertex[index].SetAt(2, *(buf++));
		}
	}

	delete [] pReadBuf;
	return true;
}

/**********************************************************************//**
 *
 * `ftH[[V
 *
 ----------------------------------------------------------------------
 * @param [in]	rTime	= XV
 * @return 
*//***********************************************************************/
bool CFBXMesh::ShapeDeformation(KTime& rTime)
{
	int ShapeNum = m_pKfbx->GetShapeCount();
	int PointsNum = m_pKfbx->GetControlPointsCount();
	KFbxVector4* pVertex = m_pVertex != nullptr ? m_pVertex : m_pKfbx->GetControlPoints();

	for( int i=0; i < ShapeNum; ++i )
	{
		KFCurve* pFCurve = m_pKfbx->GetShapeChannel(i);
		if( pFCurve == nullptr ) continue;
		KFbxShape* pShape = m_pKfbx->GetShape(i);
		double weight = pFCurve->Evaluate(rTime) / 100.0;

		for( int j=0; j < PointsNum; ++j )
		{
			KFbxVector4 Influence = (pShape->GetControlPoints()[j] - pVertex[j]) * weight;
			pVertex[j] += Influence;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * NX^[ftH[[V
 *
 ----------------------------------------------------------------------
 * @param [in]	rTime	= XV
 * @param [in]	pPose	= |[Y
 * @return 
*//***********************************************************************/
bool CFBXMesh::ClusterDeformation(KTime& rTime, KFbxPose* pPose)
{
	KFbxSkin* pSkin = static_cast<KFbxSkin*>(m_pKfbx->GetDeformer(0, KFbxDeformer::eSKIN));
	KFbxCluster::ELinkMode mode = pSkin->GetCluster(0)->GetLinkMode();

	int VtxNum	= m_pKfbx->GetControlPointsCount();
	int SkinNum	= m_pKfbx->GetDeformerCount(KFbxDeformer::eSKIN);
	KFbxXMatrix* pDeformation = new KFbxXMatrix [VtxNum];
	KFbxVector4* pVertex = m_pVertex != nullptr ? m_pVertex : m_pKfbx->GetControlPoints();
	double* pWeight = new double [VtxNum];
	int i, j, k;

	if( mode == KFbxCluster::eADDITIVE )
	{
		for( i=0; i < VtxNum; ++i ) pDeformation[i].SetIdentity();
	}
	for( i=0; i < SkinNum; ++i )
	{
		pSkin = static_cast<KFbxSkin*>(m_pKfbx->GetDeformer(i, KFbxDeformer::eSKIN));
		int ClusterNum = pSkin->GetClusterCount();
		for( j=0; j < ClusterNum; ++j )
		{
			KFbxCluster* pCluster = pSkin->GetCluster(j);
			if( pCluster->GetLink() == nullptr ) continue;
			KFbxXMatrix global_initpos;
			KFbxXMatrix global_currpos;
			KFbxXMatrix cluster_initpos;
			KFbxXMatrix cluster_currpos;
			KFbxXMatrix relative_initpos;
			KFbxXMatrix relative_currpos;
			KFbxXMatrix geometory_mtx;
			KFbxXMatrix vertex_mtx;
			if( mode == KFbxCluster::eADDITIVE && pCluster->GetAssociateModel() != nullptr )
			{
				pCluster->GetTransformAssociateModelMatrix(global_initpos);
				global_currpos = fbx::GetGlobalPosition(pCluster->GetAssociateModel(), rTime, pPose);
				geometory_mtx = fbx::GetGeometricMatrix(pCluster->GetAssociateModel());
				global_currpos *= geometory_mtx;
			}
			else
			{
				pCluster->GetTransformMatrix(global_initpos);
				//global_currpos = ;
				geometory_mtx = fbx::GetGeometricMatrix(pCluster->GetAssociateModel());
				global_initpos *= geometory_mtx;
			}
			pCluster->GetTransformLinkMatrix(cluster_initpos);
			cluster_currpos = fbx::GetGlobalPosition(pCluster->GetLink(), rTime, pPose);

			relative_initpos = cluster_initpos.Inverse() * global_initpos;
			relative_currpos = global_currpos.Inverse() * cluster_currpos;

			vertex_mtx = relative_currpos * relative_initpos;

			int IdxNum = pCluster->GetControlPointIndicesCount();
			for( k=0; k < IdxNum; ++k )
			{
				int index = pCluster->GetControlPointIndices()[k];
				double weight = pCluster->GetControlPointWeights()[k];
				if( weight == 0.0 ) continue;
				KFbxXMatrix Influence = vertex_mtx;
				Influence *= weight;

				if( mode == KFbxCluster::eADDITIVE )
				{
					double w = 1.0 - weight;
					Influence[0][0] += w;
					Influence[1][1] += w;
					Influence[2][2] += w;
					Influence[3][3] += w;
					pDeformation[index] = Influence * pDeformation[index];
					pWeight[index] = 1.0;
				}
				else
				{
					for( int x=0; x < 4; ++x )
						for( int y=0; y < 4; ++y )
							(pDeformation[index])[x][y] += Influence[x][y];
					pWeight[index] += weight;
				}
			}
		}
	}

	for( i=0; i < VtxNum; ++i )
	{
		double weight = pWeight[i];
		if( weight != 0.0 )
		{
			KFbxVector4  src = pVertex[i];
			KFbxVector4& dst = pVertex[i];
			dst = pDeformation[i].MultT(src);
			if( mode == KFbxCluster::eNORMALIZE )
			{
				dst /= weight;
			}
            else if (mode == KFbxCluster::eTOTAL1)
			{
				src *= 1.0 - weight;
				dst += src;
			}
		}
	}
	delete [] pDeformation;
	delete [] pWeight;
	return true;
}

}	// end of namespace fbx
}	// end of namespace gx
}	// end of namespace iris

#endif
