package jp.kirikiri.tjs2;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.CharBuffer;
import java.nio.DoubleBuffer;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.ShortBuffer;
import java.util.ArrayList;

/**
 * TJS2 バイトコードを読み込んで、ScriptBlock を返す
 *
 */
public class ByteCodeLoader {

	private static final boolean LOAD_SRC_POS = false;

	private static final int FILE_TAG_LE = ('T') | ('J'<<8) | ('S'<<16) | ('2'<<24);
	private static final int OBJ_TAG_LE = ('O') | ('B'<<8) | ('J'<<16) | ('S'<<24);
	private static final int DATA_TAG_LE = ('D') | ('A'<<8) | ('T'<<16) | ('A'<<24);

	private static final int TYPE_VOID = 0;
	private static final int TYPE_OBJECT = 1;
	private static final int TYPE_INTER_OBJECT = 2;
	private static final int TYPE_STRING = 3;
	private static final int TYPE_OCTET = 4;
	private static final int TYPE_REAL = 5;
	private static final int TYPE_BYTE = 6;
	private static final int TYPE_SHORT = 7;
	private static final int TYPE_INTEGER = 8;
	private static final int TYPE_LONG = 9;
	private static final int TYPE_INTER_GENERATOR = 10; // temporary
	private static final int TYPE_UNKNOWN = -1;

	//private static final String UTF16_LE_STR = "UTF-16LE";
	//private static Charset UTF16LE;
	private byte[] mByteArray;
	private short[] mShortArray;
	private int[] mIntArray;
	private long[] mLongArray;
	private double[] mDoubleArray;
	private String[] mStringArray;
	private ByteBuffer[] mByteBufferArray;

	private ByteBuffer mByteBuffer;
	private ShortBuffer mShortBuffer;
	private IntBuffer mIntBuffer;
	private LongBuffer mLongBuffer;
	private DoubleBuffer mDoubleBuffer;

	public ByteCodeLoader() {
		//if( UTF16LE != null ) UTF16LE = Charset.forName(UTF16_LE_STR);
	}

	public ScriptBlock readByteCode( TJS owner, String name, BinaryStream input ) throws TJSException {
		int size = (int) input.getSize();
		byte[] databuff = new byte[size];
		input.read(databuff);
		input.close();
		input  = null;
		ByteBuffer bbuff = ByteBuffer.wrap(databuff);
		bbuff.order(ByteOrder.LITTLE_ENDIAN);
		IntBuffer ibuff = bbuff.asIntBuffer();
		int tag = ibuff.get(); // TJS2
		if( tag != FILE_TAG_LE ) return null;

		int filesize = ibuff.get();
		if( filesize != size ) return null;

		tag = ibuff.get(); // DATA
		if( tag != DATA_TAG_LE ) return null;
		size = ibuff.get();
		readDataArea( databuff, ibuff.position()*4, size );
		ibuff.position( ibuff.position() + size/4 - 1 - 1 );

		tag = ibuff.get(); // OBJS
		if( tag != OBJ_TAG_LE ) return null;
		int objsize = ibuff.get();
		ScriptBlock block = new ScriptBlock(owner, name, 0, null, null );
		readObjects( block, ibuff );
		return block;
	}

	/**
	 * InterCodeObject へ置換するために一時的に覚えておくクラス
	 */
	static class VariantRepalace {
		public Variant Work;
		public int Index;
		public VariantRepalace( Variant w, int i ) {
			Work = w;
			Index = i;
		}
	}
	/*
	static class Property {
		public int Name;
		public int Object;
		public Property( int n, int o ) {
			Name = n;
			Object = o;
		}
	}
	*/

	private static final int
		MEMBERENSURE		= 0x00000200, // create a member if not exists
		IGNOREPROP			= 0x00000800; // ignore property invoking
	private void readObjects( ScriptBlock block, IntBuffer ibuff ) throws TJSException {
		int toplevel = ibuff.get();
		int objcount = ibuff.get();
		ArrayList<InterCodeObject> objs = new ArrayList<InterCodeObject>(objcount);
		ArrayList<VariantRepalace> work = new ArrayList<VariantRepalace>();

		int[] parent = new int[objcount];
		int[] propSetter = new int[objcount];
		int[] propGetter = new int[objcount];
		int[] superClassGetter = new int[objcount];
		int[][] properties = new int[objcount][];
		for( int o = 0; o < objcount; o++ ) {
			int tag = ibuff.get();
			if( tag != FILE_TAG_LE ) {
				throw new TJSException(Error.ByteCodeBroken);
			}
			int objsize = ibuff.get();
			parent[o] = ibuff.get();
			int name = ibuff.get();
			int contextType = ibuff.get();
			int maxVariableCount = ibuff.get();
			int variableReserveCount = ibuff.get();
			int maxFrameCount = ibuff.get();
			int funcDeclArgCount = ibuff.get();
			int funcDeclUnnamedArgArrayBase = ibuff.get();
			int funcDeclCollapseBase = ibuff.get();
			propSetter[o] = ibuff.get();
			propGetter[o] = ibuff.get();
			superClassGetter[o] = ibuff.get();

			int count = ibuff.get();

			LongBuffer srcpos;
			if( LOAD_SRC_POS ) {
				int[] codePos = new int[count];
				int[] srcPos = new int[count];
				ibuff.get( codePos );
				ibuff.get( srcPos );
				// codePos/srcPos は今のところ使ってない TODO ソート済みなので、longにする必要はないが……
				ByteBuffer code2srcpos = ByteBuffer.allocate(count*8);
				code2srcpos.order(ByteOrder.LITTLE_ENDIAN);
				srcpos = code2srcpos.asLongBuffer();
				for( int i = 0; i < count; i++ ) {
					srcpos.put( ((long)(codePos[i]) << 32) | (long)(srcPos[i]) );
				}
				srcpos.flip();
			} else {
				ibuff.position(ibuff.position()+count*2);
				srcpos = null;
			}

			count = ibuff.get();
			int[] code = new int[count];
			ibuff.get( code );
			count = ibuff.get();
			int[] data = new int[count*2];
			ibuff.get( data );

			Variant[] vdata = new Variant[count];
			int datacount = count;
			Variant tmp;
			for( int i = 0; i < datacount; i++ ) {
				int type = data[i*2];
				int index = data[i*2+1];
				switch( type ) {
				case TYPE_VOID:
					vdata[i] = new Variant(); // null
					break;
				case TYPE_OBJECT:
					vdata[i] = new Variant(null,null); // null Array Dictionary はまだサポートしていない TODO
					break;
				case TYPE_INTER_OBJECT:
					tmp = new Variant();
					work.add( new VariantRepalace( tmp, index ) );
					vdata[i] = tmp;
					break;
				case TYPE_INTER_GENERATOR:
					tmp = new Variant();
					work.add( new VariantRepalace( tmp, index ) );
					vdata[i] = tmp;
					break;
				case TYPE_STRING:
					vdata[i] = new Variant( mStringArray[index] );
					break;
				case TYPE_OCTET:
					vdata[i] = new Variant( mByteBufferArray[index] );
					break;
				case TYPE_REAL:
					vdata[i] = new Variant( mDoubleArray[index] );
					break;
				case TYPE_BYTE:
					vdata[i] = new Variant( mByteArray[index] );
					break;
				case TYPE_SHORT:
					vdata[i] = new Variant( mShortArray[index] );
					break;
				case TYPE_INTEGER:
					vdata[i] = new Variant( mIntArray[index] );
					break;
				case TYPE_LONG:
					vdata[i] = new Variant( mLongArray[index] );
					break;
				case TYPE_UNKNOWN:
				default:
					vdata[i] = new Variant(); // null;
					break;
				}
			}
			count = ibuff.get();
			int[] scgetterps = new int[count];
			ibuff.get( scgetterps );
			// properties
			count = ibuff.get();
			if( count > 0 ) {
				properties[o] = new int[count*2];
				ibuff.get( properties[o] );
			}

			IntVector superpointer = IntVector.wrap( scgetterps );
			InterCodeObject obj = new InterCodeObject( block, mStringArray[name], contextType, code, vdata, maxVariableCount, variableReserveCount,
					maxFrameCount, funcDeclArgCount, funcDeclUnnamedArgArrayBase, funcDeclCollapseBase, true, srcpos, superpointer );
			objs.add(obj);
		}
		Variant val = new Variant();
		for( int o = 0; o < objcount; o++ ) {
			InterCodeObject parentObj = null;
			InterCodeObject propSetterObj = null;
			InterCodeObject propGetterObj = null;
			InterCodeObject superClassGetterObj = null;

			if( parent[o] >= 0 ) {
				parentObj = objs.get(parent[o]);
			}
			if( propSetter[o] >= 0 ) {
				propSetterObj = objs.get(propSetter[o]);
			}
			if( propGetter[o] >= 0 ) {
				propGetterObj = objs.get(propGetter[o]);
			}
			if( superClassGetter[o] >= 0 ) {
				superClassGetterObj = objs.get(superClassGetter[o]);
			}
			objs.get(o).setCodeObject(parentObj, propSetterObj, propGetterObj, superClassGetterObj );
			if( properties[o] != null ) {
				InterCodeObject obj = parentObj; // objs.get(o).mParent;
				int[] prop = properties[o];
				int length = prop.length / 2;
				for( int i = 0; i < length; i++ ) {
					int pname = prop[i*2];
					int pobj = prop[i*2+1];
					val.set( objs.get(pobj) );
					obj.propSet( MEMBERENSURE|IGNOREPROP, mStringArray[pname], val, obj );
				}
			}
		}
		int count = work.size();
		for( int i = 0; i < count; i++ ) {
			VariantRepalace w = work.get(i);
			w.Work.set( objs.get(w.Index) );
		}
		InterCodeObject top = null;
		if( toplevel >= 0 ) {
			top = objs.get(toplevel);
		}
		block.setObjects( top, objs );
	}

	private void readDataArea( byte[] buff, int offset, int size ) {
		ByteBuffer bbuff = ByteBuffer.wrap(buff);
		bbuff.order(ByteOrder.LITTLE_ENDIAN);
		IntBuffer ibuff = bbuff.asIntBuffer();
		ibuff.position(offset/4);
		int count = ibuff.get();
		mByteArray = new byte[count];
		if( count > 0 ) {
			bbuff.position(offset+4);
			bbuff.get(mByteArray);
			int stride = ( count + 3 ) / 4;
			ibuff.position(ibuff.position()+stride);
		}
		count = ibuff.get();
		mShortArray = new short[count];
		if( count > 0 ) {	// load short
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()*4, count*2 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			ShortBuffer sb = sbuff.asShortBuffer();
			sb.get(mShortArray);
			int stride = ( count + 1 ) / 2;
			ibuff.position(ibuff.position()+stride);
		}
		count = ibuff.get();
		mIntArray = new int[count];
		if( count > 0 ) {
			ibuff.get(mIntArray);
		}
		count = ibuff.get();
		mLongArray = new long[count];
		if( count > 0 ) {	// load long
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()*4, count*8 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			LongBuffer lb = sbuff.asLongBuffer();
			lb.get(mLongArray);
			ibuff.position(ibuff.position()+count*2);
		}
		count = ibuff.get();
		mDoubleArray = new double[count];
		if( count > 0 ) {	// load double
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()*4, count*8 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			LongBuffer lb = sbuff.asLongBuffer();
			long[] tmp = new long[count];
			lb.get(tmp);
			for( int i = 0; i < count; i++ ) {
				mDoubleArray[i] = Double.longBitsToDouble(tmp[i]);
			}
			ibuff.position(ibuff.position()+count*2);
		}
		count = ibuff.get();
		mStringArray = new String[count];

		/*
		int stroff = ibuff.position()*4 + 4;
		for( int i = 0; i < count; i++ ) {
			int len = ibuff.get() * 2;
			mStringArray[i] = TJS.mapGlobalStringMap( new String( buff, stroff, len, UTF16LE ) );
			len = ((len+3)/4)*4;
			stroff += len + 4;
			ibuff.position( stroff/4 - 1 );
		}
		*/

		//bbuff.order(ByteOrder.BIG_ENDIAN);
		bbuff.position(0);
		CharBuffer cbuff = bbuff.asCharBuffer();
		int stroff = ibuff.position()*2+2;
		for( int i = 0; i < count; i++ ) {
			int len = ibuff.get();
			cbuff.position(stroff);
			mStringArray[i] = TJS.mapGlobalStringMap(cbuff.subSequence(0, len).toString());
			len = ((len+1)/2)*2;
			stroff += len + 2;
			ibuff.position( stroff/2 - 1 );
		}

		count = ibuff.get();
		mByteBufferArray = new ByteBuffer[count];
		int octetoff = ibuff.position()*4 + 4;
		for( int i = 0; i < count; i++ ) {
			int len = ibuff.get();
			mByteBufferArray[i] = ByteBuffer.allocate(len);
			mByteBufferArray[i].put( buff, octetoff, len );
			len = ((len+3)/4)*4;
			octetoff += len + 4;
			ibuff.position( octetoff/4-4 );
		}
	}
	private void readObjects2( ScriptBlock block, IntBuffer ibuff ) throws TJSException {
		int toplevel = ibuff.get();
		int objcount = ibuff.get();
		ArrayList<InterCodeObject> objs = new ArrayList<InterCodeObject>(objcount);
		ArrayList<VariantRepalace> work = new ArrayList<VariantRepalace>();

		int[] parent = new int[objcount];
		int[] propSetter = new int[objcount];
		int[] propGetter = new int[objcount];
		int[] superClassGetter = new int[objcount];
		int[][] properties = new int[objcount][];
		for( int o = 0; o < objcount; o++ ) {
			int tag = ibuff.get();
			if( tag != FILE_TAG_LE ) {
				throw new TJSException(Error.ByteCodeBroken);
			}
			int objsize = ibuff.get();
			parent[o] = ibuff.get();
			int name = ibuff.get();
			int contextType = ibuff.get();
			int maxVariableCount = ibuff.get();
			int variableReserveCount = ibuff.get();
			int maxFrameCount = ibuff.get();
			int funcDeclArgCount = ibuff.get();
			int funcDeclUnnamedArgArrayBase = ibuff.get();
			int funcDeclCollapseBase = ibuff.get();
			propSetter[o] = ibuff.get();
			propGetter[o] = ibuff.get();
			superClassGetter[o] = ibuff.get();

			int count = ibuff.get();

			int[] codePos = new int[count];
			int[] srcPos = new int[count];
			ibuff.get( codePos );
			ibuff.get( srcPos );
			// codePos/srcPos は今のところ使ってない TODO ソート済みなので、longにする必要はないが……
			ByteBuffer code2srcpos = ByteBuffer.allocate(count*8);
			code2srcpos.order(ByteOrder.LITTLE_ENDIAN);
			LongBuffer srcpos = code2srcpos.asLongBuffer();
			for( int i = 0; i < count; i++ ) {
				srcpos.put( ((long)(codePos[i]) << 32) | (long)(srcPos[i]) );
			}
			srcpos.flip();

			/*
			ibuff.position(ibuff.position()+count*2);
			LongBuffer srcpos = null;
			*/

			count = ibuff.get();
			int[] code = new int[count];
			ibuff.get( code );
			count = ibuff.get();
			int[] data = new int[count*2];
			ibuff.get( data );

			Variant[] vdata = new Variant[count];
			int datacount = count;
			Variant tmp;
			for( int i = 0; i < datacount; i++ ) {
				int type = data[i*2];
				int index = data[i*2+1];
				switch( type ) {
				case TYPE_VOID:
					vdata[i] = new Variant(); // null
					break;
				case TYPE_OBJECT:
					vdata[i] = new Variant(null,null); // null Array Dictionary はまだサポートしていない TODO
					break;
				case TYPE_INTER_OBJECT:
					tmp = new Variant();
					work.add( new VariantRepalace( tmp, index ) );
					vdata[i] = tmp;
					break;
				case TYPE_INTER_GENERATOR:
					tmp = new Variant();
					work.add( new VariantRepalace( tmp, index ) );
					vdata[i] = tmp;
					break;
				case TYPE_STRING:
					vdata[i] = new Variant( mStringArray[index] );
					break;
				case TYPE_OCTET:
					vdata[i] = new Variant( mByteBufferArray[index] );
					break;
				case TYPE_REAL:
					vdata[i] = new Variant( mDoubleBuffer.get(index) );
					break;
				case TYPE_BYTE:
					//vdata[i] = new Variant( mByteBuffer.get(index) );
					vdata[i] = new Variant( mByteArray[index] );
					break;
				case TYPE_SHORT:
					vdata[i] = new Variant( mShortBuffer.get(index) );
					break;
				case TYPE_INTEGER:
					vdata[i] = new Variant( mIntBuffer.get(index) );
					break;
				case TYPE_LONG:
					vdata[i] = new Variant( mLongBuffer.get(index) );
					break;
				case TYPE_UNKNOWN:
				default:
					vdata[i] = new Variant(); // null;
					break;
				}
			}
			count = ibuff.get();
			int[] scgetterps = new int[count];
			ibuff.get( scgetterps );
			// properties
			count = ibuff.get();
			if( count > 0 ) {
				properties[o] = new int[count*2];
				ibuff.get( properties[o] );
			}

			IntVector superpointer = IntVector.wrap( scgetterps );
			InterCodeObject obj = new InterCodeObject( block, mStringArray[name], contextType, code, vdata, maxVariableCount, variableReserveCount,
					maxFrameCount, funcDeclArgCount, funcDeclUnnamedArgArrayBase, funcDeclCollapseBase, true, srcpos, superpointer );
			objs.add(obj);
		}
		Variant val = new Variant();
		for( int o = 0; o < objcount; o++ ) {
			InterCodeObject parentObj = null;
			InterCodeObject propSetterObj = null;
			InterCodeObject propGetterObj = null;
			InterCodeObject superClassGetterObj = null;

			if( parent[o] >= 0 ) {
				parentObj = objs.get(parent[o]);
			}
			if( propSetter[o] >= 0 ) {
				propSetterObj = objs.get(propSetter[o]);
			}
			if( propGetter[o] >= 0 ) {
				propGetterObj = objs.get(propGetter[o]);
			}
			if( superClassGetter[o] >= 0 ) {
				superClassGetterObj = objs.get(superClassGetter[o]);
			}
			objs.get(o).setCodeObject(parentObj, propSetterObj, propGetterObj, superClassGetterObj );
			if( properties[o] != null ) {
				InterCodeObject obj = parentObj; // objs.get(o).mParent;
				int[] prop = properties[o];
				int length = prop.length / 2;
				for( int i = 0; i < length; i++ ) {
					int pname = prop[i*2];
					int pobj = prop[i*2+1];
					val.set( objs.get(pobj) );
					obj.propSet( MEMBERENSURE|IGNOREPROP, mStringArray[pname], val, obj );
				}
			}
		}
		int count = work.size();
		for( int i = 0; i < count; i++ ) {
			VariantRepalace w = work.get(i);
			w.Work.set( objs.get(w.Index) );
		}
		InterCodeObject top = null;
		if( toplevel >= 0 ) {
			top = objs.get(toplevel);
		}
		block.setObjects( top, objs );
	}
	private void readDataArea2( byte[] buff, int offset, int size ) {
		ByteBuffer bbuff = ByteBuffer.wrap(buff);
		bbuff.order(ByteOrder.LITTLE_ENDIAN);
		IntBuffer ibuff = bbuff.asIntBuffer();
		ibuff.position(offset/4);
		int count = ibuff.get();
		/*
		mByteBuffer = ByteBuffer.wrap( buff, offset+4, count);
		int stride = ( count + 3 ) / 4;
		ibuff.position(ibuff.position()+stride);
		*/

		mByteArray = new byte[count];
		if( count > 0 ) {
			bbuff.position(offset+4);
			bbuff.get(mByteArray);
			int stride = ( count + 3 ) / 4;
			ibuff.position(ibuff.position()+stride);
		}

		count = ibuff.get();
		ByteBuffer tmp = ByteBuffer.wrap( buff, ibuff.position()*4, count*2);
		tmp.order(ByteOrder.LITTLE_ENDIAN);
		mShortBuffer = tmp.asShortBuffer();
		int stride = ( count + 1 ) / 2;
		ibuff.position(ibuff.position()+stride);
		/*
		mShortArray = new short[count];
		if( count > 0 ) {	// load short
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()*4, count*2 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			ShortBuffer sb = sbuff.asShortBuffer();
			sb.get(mShortArray);
			int stride = ( count + 1 ) / 2;
			ibuff.position(ibuff.position()+stride);
		}
		*/
		count = ibuff.get();
		tmp = ByteBuffer.wrap( buff, ibuff.position()*4, count*4);
		tmp.order(ByteOrder.LITTLE_ENDIAN);
		mIntBuffer = tmp.asIntBuffer();
		ibuff.position(ibuff.position()+count);
		/*
		mIntArray = new int[count];
		if( count > 0 ) {
			ibuff.get(mIntArray);
		}
		*/
		count = ibuff.get();
		tmp = ByteBuffer.wrap( buff, ibuff.position()*4, count*8);
		tmp.order(ByteOrder.LITTLE_ENDIAN);
		mLongBuffer = tmp.asLongBuffer();
		ibuff.position(ibuff.position()+count*2);

		/*
		mLongArray = new long[count];
		if( count > 0 ) {	// load long
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()*4, count*8 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			LongBuffer lb = sbuff.asLongBuffer();
			lb.get(mLongArray);
			ibuff.position(ibuff.position()+count*2);
		}
		*/
		count = ibuff.get();
		tmp = ByteBuffer.wrap( buff, ibuff.position()*4, count*8);
		tmp.order(ByteOrder.LITTLE_ENDIAN);
		mDoubleBuffer = tmp.asDoubleBuffer();
		ibuff.position(ibuff.position()+count*2);
		/*
		mDoubleArray = new double[count];
		if( count > 0 ) {	// load double
			ByteBuffer sbuff = ByteBuffer.wrap( buff, ibuff.position()*4, count*8 );
			sbuff.order(ByteOrder.LITTLE_ENDIAN);
			LongBuffer lb = sbuff.asLongBuffer();
			long[] tmp = new long[count];
			lb.get(tmp);
			for( int i = 0; i < count; i++ ) {
				mDoubleArray[i] = Double.longBitsToDouble(tmp[i]);
			}
			ibuff.position(ibuff.position()+count*2);
		}
		*/
		count = ibuff.get();
		mStringArray = new String[count];
		bbuff.position(0);
		CharBuffer cbuff = bbuff.asCharBuffer();
		int stroff = ibuff.position()*2+2;
		for( int i = 0; i < count; i++ ) {
			int len = ibuff.get();
			cbuff.position(stroff);
			mStringArray[i] = TJS.mapGlobalStringMap(cbuff.subSequence(0, len).toString());
			len = ((len+1)/2)*2;
			stroff += len + 2;
			ibuff.position( stroff/2 - 1 );
		}

		count = ibuff.get();
		mByteBufferArray = new ByteBuffer[count];
		int octetoff = ibuff.position()*4 + 4;
		for( int i = 0; i < count; i++ ) {
			int len = ibuff.get();
			mByteBufferArray[i] = ByteBuffer.allocate(len);
			mByteBufferArray[i].put( buff, octetoff, len );
			len = ((len+3)/4)*4;
			octetoff += len + 4;
			ibuff.position( octetoff/4-4 );
		}
	}
}
