package jp.kirikiri.tvp2.env;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.UnsupportedAudioFileException;

import jp.kirikiri.tvp2.TVP;
import jp.kirikiri.tvp2.sound.WaveSoundBufferNI;

public class SoundStream implements Runnable, LineListener {

	private WaveSoundBufferNI mOwner;

	private String mFileName;
	private Thread mThread;
	private SourceDataLine mLine;

	private FloatControl mPanControl;
	private FloatControl mGainControl;
	//private FloatControl mVolumeControl;
	//private FloatControl mSampleRateControl;
	private float mMinPanValue;
	private float mMaxPanValue;
	private float mMinGainValue;
	private float mMaxGainValue;
	//private float mMinVolumeValue;
	//private float mMaxVolumeValue;
	//private float mMinSampleRateValue;
	//private float mMaxSampleRateValue;

	private int mRequenstBufferSize;
	private AudioInputStream mAudioStream;
	private AudioFormat mFormat;
	private InputStream mInputStream;
	private boolean mLoop;
	private long mNumberOfRead;
	private long mLastRestFramePosition;

	public SoundStream( WaveSoundBufferNI o ) {
		mOwner = o;
	}
	@Override
	public void run() {
		if( mAudioStream == null || mLine == null ) return;
		int retryCount = 0;

		int numBytesRead = 0;
		mLine.start();
		byte[] buffer = new byte[mRequenstBufferSize];
		while( retryCount < 5 ) {
			try {
				do {
					numBytesRead = mAudioStream.read(buffer, 0, buffer.length);
					if( numBytesRead != -1 ) {
						mNumberOfRead += numBytesRead;
						mLine.write(buffer, 0, numBytesRead);
					} else if( mLoop ) {
						mAudioStream.reset();
						mNumberOfRead = 0;
						mLastRestFramePosition = mLine.getLongFramePosition();
						numBytesRead = 0;
					}
				} while( mLine.isActive() && numBytesRead != -1);
				mLine.drain();
				mLine.close();
				break;
			} catch (IOException e) {
				//System.out.println(e.getMessage());
				//e.printStackTrace();
				retryCount++;
			}
		}
		mOwner.setStatus(WaveSoundBufferNI.ssStop);
	}

	/*
	public void openFromFile( String filename ) throws UnsupportedAudioFileException, IOException, LineUnavailableException {
		mFileName = filename;
		mAudioStream = AudioSystem.getAudioInputStream(new File(mFileName) );
		mAudioStream.mark(mAudioStream.available());
		mFormat = mAudioStream.getFormat();
		open( mFormat );
	}
	*/
	public void openFromStream( InputStream stream, String filename ) throws IOException {
		mFileName = filename;
		mInputStream = stream;
		//mInputStream.mark(mInputStream.available());
		try {
			mAudioStream = AudioSystem.getAudioInputStream(mInputStream);
			mAudioStream.mark(mAudioStream.available());
			mFormat = mAudioStream.getFormat();
			open( mFormat );
		} catch (UnsupportedAudioFileException e) {
			throw new IOException();
		} catch (LineUnavailableException e) {
			throw new IOException();
		}
	}
	public void open( AudioFormat format ) throws LineUnavailableException {
		mRequenstBufferSize = (int) (format.getSampleRate() * format.getChannels() * format.getSampleSizeInBits()/8);
		DataLine.Info info = new DataLine.Info(SourceDataLine.class, format);
		//mLine = (SourceDataLine) AudioSystem.getLine(info);
		mLine = TVP.SoundMixer.getSourceDataLine(info);
		mLine.addLineListener(this);
		mLine.open(format, mRequenstBufferSize);
		mRequenstBufferSize = mLine.getBufferSize();
		try {
			mPanControl = (FloatControl) mLine.getControl(FloatControl.Type.PAN);
			mMinPanValue = mPanControl.getMinimum();
			mMaxPanValue = mPanControl.getMaximum();
		} catch( IllegalArgumentException e ) {
			mPanControl = null; // may be mono
		}
		mGainControl = (FloatControl) mLine.getControl(FloatControl.Type.MASTER_GAIN);
		mMinGainValue = mGainControl.getMinimum();
		mMaxGainValue = mGainControl.getMaximum();

		//mVolumeControl = (FloatControl) mLine.getControl(FloatControl.Type.VOLUME);
		//mSampleRateControl = (FloatControl) mLine.getControl(FloatControl.Type.SAMPLE_RATE);
		//mMinVolumeValue = mVolumeControl.getMinimum();
		//mMaxVolumeValue = mVolumeControl.getMaximum();
		//mMinSampleRateValue = mSampleRateControl.getMinimum();
		//mMaxSampleRateValue = mSampleRateControl.getMaximum();
	}
	public void play() {
		if( mLine != null ) {
			mThread = new Thread(this);
			mThread.setName("Sound Stream : " + mFileName );
			mThread.start();
		}
	}
	public void stop() {
		if( mLine != null ) {
			mLine.stop();
		}
	}

	/**
	 * パンを設定する
	 * @param pan 有効な値の範囲は -1.0 (左チャネルのみ) ～ 1.0 (右チャネルのみ) です。デフォルトは 0.0 (中央)
	 */
	public void setPanInternal( float pan ) {
		if( mPanControl == null ) return;
		if( pan < mMinPanValue ) pan = mMinPanValue;
		else if( pan > mMaxPanValue ) pan = mMaxPanValue;
		mPanControl.setValue( pan );
	}
	public float getPanInternal() {
		if( mPanControl == null ) return 0;
		else return mPanControl.getValue();
	}
	public void setGain( float gain ) {
		if( gain < mMinGainValue ) gain = mMinGainValue;
		else if( gain > mMaxGainValue ) gain = mMaxGainValue;
		mGainControl.setValue(gain);
	}
	public float getGain() { return mGainControl.getValue(); }
	public void setFade( float from, float to, int msec ) {
		mGainControl.shift( from, to, msec );
	}
	/*
	public void setVolumeInternal( float volume ) {
		if( volume < mMinVolumeValue ) volume = mMinVolumeValue;
		else if( volume > mMaxVolumeValue ) volume = mMaxVolumeValue;
		mVolumeControl.setValue(volume);
	}
	public float getVolumeInternal() { return mVolumeControl.getValue(); }
	*/

	@Override
	public void update(LineEvent event) {
		LineEvent.Type type = event.getType();
		if( LineEvent.Type.CLOSE.equals(type) ) {
			mOwner.setStatus(WaveSoundBufferNI.ssUnload);
		} else if( LineEvent.Type.OPEN.equals(type) ) {
		} else if( LineEvent.Type.START.equals(type) ) {
			mOwner.setStatus(WaveSoundBufferNI.ssPlay);
		} else if( LineEvent.Type.STOP.equals(type) ) {
			mOwner.setStatus(WaveSoundBufferNI.ssStop);
		}
	}

	public void setVolume( int vol ) {
		if( vol < 0 ) vol = 0;
		else if( vol > 100000  ) vol = 100000;

		vol = vol - 100000;
		float v = vol / Math.abs(mMinGainValue);
		setGain( v );
	}
	public int getVolume() {
		float v = getGain();
		if( v > 0 ) {
			return 100000;
		} else {
			float vol = v * 100000 / Math.abs(mMinGainValue);
			vol = 100000 - vol;
			return (int)vol;
		}
	}

	/**
	 * 音量をフェードする
	 * @param to : 目標値
	 * @param time : フェード期間
	 * @param delay : 現状無視 TODO 対応するように
	 */
	public void fade(int to, int time, int delay) {
		if( to < 0 ) to = 0;
		else if( to > 100000  ) to = 100000;
		to = to - 100000;
		float v = to / Math.abs(mMinGainValue);
		float from = mGainControl.getValue();
		mGainControl.shift( from, v, time );
	}

	// -100000 ～ 0 ～ 100000
	public int getPan() {
		if( mPanControl == null ) return 0;
		float p = mPanControl.getValue();
		if( p < 0.0f ) {
			return (int) (p * 100000 / Math.abs(mMinPanValue));
		} else if( p > 0.0 ) {
			return (int) (p * 100000 / Math.abs(mMaxPanValue));
		} else {
			return 0;
		}
	}

	public void setPan(int v) {
		if( mPanControl == null ) return;
		if( v < 0 ) {
			mPanControl.setValue( v * Math.abs(mMinPanValue) / 100000 );
		} else if( v > 0 ) {
			mPanControl.setValue( v * Math.abs(mMaxPanValue) / 100000 );
		} else {
			mPanControl.setValue( 0.0f );
		}
	}
	public boolean getLooping() {
		return mLoop;
	}
	public void setLooping(boolean b) {
		mLoop = b;
	}
	public boolean getPaused() {
		// TODO 自動生成されたメソッド・スタブ
		return false;
	}
	public void setPaused(boolean b) {
		// TODO 自動生成されたメソッド・スタブ
	}

	public int getPosition() {
		long sample = mLine.getLongFramePosition() - mLastRestFramePosition;
		long time = (long) (sample * 1000 / mFormat.getSampleRate());
		return (int) time;
	}
	public void setPosition(int pos) {
		// TODO 自動生成されたメソッド・スタブ
	}
	public int getSamplePosition() {
		long sample = mLine.getLongFramePosition() - mLastRestFramePosition;
		return (int) sample;
	}
	public void setSamplePosition(int sample) {
		// TODO 自動生成されたメソッド・スタブ
	}

	public int getTotalTime() {
		long length = mAudioStream.getFrameLength();
		long time = (long) (length * 1000 / mFormat.getSampleRate());
		return (int) time;
	}
	public int getFrequency() {
		/*
		float rate = mSampleRateControl.getValue();
		//rate = rate * getSampleRate();
		return (int) rate;
		*/
		return (int) mFormat.getSampleRate();
	}
	public void setFrequency(int freq) {
		/*
		float f = freq;
		if( f < mMinSampleRateValue ) f = mMinSampleRateValue;
		else if( f > mMaxSampleRateValue ) f = mMaxSampleRateValue;
		mSampleRateControl.setValue(f);
		*/
	}
	public int getGitsPerSample() {
		return mFormat.getSampleSizeInBits();
	}
	public int getChannels() {
		return mFormat.getChannels();
	}
}
