# coding: UTF-8

if $0 == __FILE__ then
	begin
		require 'rubygems'
	rescue LoadError
	end

	ROOT_PATH = File.dirname(File.expand_path(__FILE__))

	$:.unshift(ROOT_PATH + "/../")
	$:.unshift(ROOT_PATH + "/../../main/")
	$:.unshift(ROOT_PATH + "/../../main/lib/")

	require 'minitest/autorun'
end

require 'base64'
require 'net/pop'
require 'net/smtp'

require 'conf/01_system/ruby_version_absorber'
require 'mailutils/mail_processor'
require 'testutils/mail_testutils'

require "#{File.dirname(File.expand_path(__FILE__))}/conf/setting"

#=MailProcessor のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_mailutils.rb 675 2012-05-12 15:12:21Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestMailProcessor < MiniTest::Unit::TestCase

	def setup
		@expected_html_content = "<div>内容</div>"
	end

	def test_load_attachment
		attachment = MailProcessor.load_attachment("#{File.dirname(File.expand_path(__FILE__))}/data/eyes0367.jpg")
		assert_equal 'eyes0367.jpg', attachment.filename
		assert_equal open("#{File.dirname(File.expand_path(__FILE__))}/data/eyes0367.jpg", 'rb') {|file| file.read}, attachment.content
	end

	def test_load_attachments
		attachments = MailProcessor.load_attachments("#{File.dirname(File.expand_path(__FILE__))}/data/eyes0367.jpg")
		assert_equal 1, attachments.size
		assert_equal 'eyes0367.jpg', attachments[0].filename
		assert_equal open("#{File.dirname(File.expand_path(__FILE__))}/data/eyes0367.jpg", 'rb') {|file| file.read}, attachments[0].content
	end

	def test_escaped_prefix_subject
		mail = Mail.new("Re:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Fw:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Fwd:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Re>title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Re > title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject
	end

	def test_encode_header1
		assert_equal '=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', MailProcessor.encode_header('テスト')
	end

	def test_decode_header2
		assert_equal 'テスト', MailProcessor.decode_header('=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', 'UTF-8')
	end

	def test_decode_header3
		assert_equal 'm2w:post:1', MailProcessor.decode_header('=?iso-8859-1?q?m2w:post:1?=', 'UTF-8')
	end

	def test_parse_address_content1
		address = MailProcessor.parse_mail_address('name <abcdef@ghjik.lmnop>')
		assert_equal('name', address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_content2
		address = MailProcessor.parse_mail_address('"name" <abcdef@ghjik.lmnop>')
		assert_equal('name', address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_content3
		address = MailProcessor.parse_mail_address("'name' <abcdef@ghjik.lmnop>")
		assert_equal("'name'", address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_conten4
		address = MailProcessor.parse_mail_address("<w-w.w-w.0310@docomo.ne.jp>")
		assert_equal("", address.name)
		assert_equal('w-w.w-w.0310@docomo.ne.jp', address.address)
	end

	def test_make1
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = <<"END_CONTENT"
Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=
From: test_sender@ownway.info
To: test_receiver1@ownway.info
Date: Sun, 02 Jan 2011 03:04:05 +0900
Content-Type: text/plain; charset="UTF-8";
Content-Transfer-Encoding: base64

44OG44K544OI

END_CONTENT

		assert_equal expected, result
	end

	def test_make2
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info', 'テスト送信者'),
			[MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = <<"END_CONTENT"
Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=
From: "=?ISO-2022-JP?B?GyRCJUYlOSVIQXc/LjxUGyhC?=" <test_sender@ownway.info>
To: "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzEbKEI=?=" <test_receiver1@ownway.info>, "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzIbKEI=?=" <test_receiver2@ownway.info>
Date: Sun, 02 Jan 2011 03:04:05 +0900
Content-Type: text/plain; charset="UTF-8";
Content-Transfer-Encoding: base64

44OG44K544OI

END_CONTENT

		assert_equal expected, result
	end

	def test_parse
		from_address = MailAddress.new('test_sender@ownway.info', 'テスト送信者')
		to_addresses = [MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')]
		attachments = [MailProcessor.load_attachment("#{File.dirname(File.expand_path(__FILE__))}/data/eyes0367.jpg")]
		result = MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments), Time.local(2011, 1, 2, 3, 4, 5, 6))

		mail = MailProcessor.parse(result, 'UTF-8')
		assert_equal 'テスト', mail.subject
		assert_equal from_address.address, mail.from_address.address
		assert_equal from_address.name, mail.from_address.name
		assert_equal to_addresses.size, mail.to_addresses.size
		assert_equal to_addresses[0].address, mail.to_addresses[0].address
		assert_equal to_addresses[0].name, mail.to_addresses[0].name
		assert_equal to_addresses[1].address, mail.to_addresses[1].address
		assert_equal to_addresses[1].name, mail.to_addresses[1].name
		assert_equal 'テスト', mail.content
		assert_equal attachments.size, mail.attachments.size
		assert_equal attachments[0].filename, mail.attachments[0].filename
		assert_equal attachments[0].content, mail.attachments[0].content
	end

	def test_mail_case1_text_plain
		mail_content = MailProcessor.make(Mail.new(
			'タイトル',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'内容')
		)

		mail = MailProcessor.parse(mail_content, 'UTF-8')

		assert_equal 'test_sender@ownway.info', mail.from_address.address
		assert_equal 1, mail.to_addresses.size
		assert_equal 'test_receiver1@ownway.info', mail.to_addresses[0].address
		assert_equal 'タイトル', mail.subject
		assert mail.has_content
		assert_equal '内容', mail.content
		assert_equal 0, mail.attachments.size
		assert !mail.has_html_content
		assert_equal 0, mail.related_attachments.size
	end

	def test_mail_case2_multipart_mixed
		mail_content = MailProcessor.make(Mail.new(
			'タイトル',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'内容', [MailAttachment.new('sample1.txt', 'sample1'), MailAttachment.new('sample2.txt', 'sample2')])
		)

		mail = MailProcessor.parse(mail_content, 'UTF-8')

		assert_equal 'test_sender@ownway.info', mail.from_address.address
		assert_equal 1, mail.to_addresses.size
		assert_equal 'test_receiver1@ownway.info', mail.to_addresses[0].address
		assert_equal 'タイトル', mail.subject
		assert mail.has_content
		assert_equal '内容', mail.content
		assert_equal 2, mail.attachments.size
		assert_equal MailAttachment.new('sample1.txt', 'sample1'), mail.attachments[0]
		assert_equal MailAttachment.new('sample2.txt', 'sample2'), mail.attachments[1]
		assert !mail.has_html_content
		assert_equal 0, mail.related_attachments.size
	end

	def test_mail_case3_multipart_alternative
		mail_content = MailProcessor.make(Mail.new(
			'タイトル',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'内容', [], '<div>内容</div>', [])
		)

		mail = MailProcessor.parse(mail_content, 'UTF-8')

		assert_equal 'test_sender@ownway.info', mail.from_address.address
		assert_equal 1, mail.to_addresses.size
		assert_equal 'test_receiver1@ownway.info', mail.to_addresses[0].address
		assert_equal 'タイトル', mail.subject
		assert mail.has_content
		assert_equal '内容', mail.content
		assert_equal 0, mail.attachments.size
		assert mail.has_html_content
		assert_equal @expected_html_content, mail.html_content
		assert_equal 0, mail.related_attachments.size
	end

	def test_mail_case4_multipart_mixed_alternative
		mail_content = MailProcessor.make(Mail.new(
			'タイトル',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'内容', [MailAttachment.new('sample1.txt', 'sample1'), MailAttachment.new('sample2.txt', 'sample2')],
			'<div>内容</div>', [])
		)

		mail = MailProcessor.parse(mail_content, 'UTF-8')

		assert_equal 'test_sender@ownway.info', mail.from_address.address
		assert_equal 1, mail.to_addresses.size
		assert_equal 'test_receiver1@ownway.info', mail.to_addresses[0].address
		assert_equal 'タイトル', mail.subject
		assert mail.has_content
		assert_equal '内容', mail.content
		assert_equal 2, mail.attachments.size
		assert_equal MailAttachment.new('sample1.txt', 'sample1'), mail.attachments[0]
		assert_equal MailAttachment.new('sample2.txt', 'sample2'), mail.attachments[1]
		assert mail.has_html_content
		assert_equal @expected_html_content, mail.html_content
		assert_equal 0, mail.related_attachments.size
	end

	def test_mail_case5_multipart_related_alternative
		mail_content = MailProcessor.make(Mail.new(
			'タイトル',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'内容', [],
			'<div>内容</div>', [MailAttachment.new('sample3.txt', 'sample3', 'sample3-id'), MailAttachment.new('sample4.txt', 'sample4', 'sample4-id')])
		)

		mail = MailProcessor.parse(mail_content, 'UTF-8')

		assert_equal 'test_sender@ownway.info', mail.from_address.address
		assert_equal 1, mail.to_addresses.size
		assert_equal 'test_receiver1@ownway.info', mail.to_addresses[0].address
		assert_equal 'タイトル', mail.subject
		assert mail.has_content
		assert_equal '内容', mail.content
		assert_equal 0, mail.attachments.size
		assert mail.has_html_content
		assert_equal @expected_html_content, mail.html_content
		assert_equal 2, mail.related_attachments.size
		assert_equal MailAttachment.new('sample3.txt', 'sample3', 'sample3-id'), mail.related_attachments[0]
		assert_equal MailAttachment.new('sample4.txt', 'sample4', 'sample4-id'), mail.related_attachments[1]
	end

	def test_mail_case6_multipart_mixed_related_alternative
		mail_content = MailProcessor.make(Mail.new(
			'タイトル',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'内容', [MailAttachment.new('sample1.txt', 'sample1', 'sample1-id'), MailAttachment.new('sample2.txt', 'sample2', 'sample2-id')],
			'<div>内容</div>', [MailAttachment.new('sample3.txt', 'sample3', 'sample3-id'), MailAttachment.new('sample4.txt', 'sample4', 'sample4-id')])
		)

		mail = MailProcessor.parse(mail_content, 'UTF-8')

		assert_equal 'test_sender@ownway.info', mail.from_address.address
		assert_equal 1, mail.to_addresses.size
		assert_equal 'test_receiver1@ownway.info', mail.to_addresses[0].address
		assert_equal 'タイトル', mail.subject
		assert mail.has_content
		assert_equal '内容', mail.content
		assert_equal 2, mail.attachments.size
		assert_equal MailAttachment.new('sample1.txt', 'sample1'), mail.attachments[0]
		assert_equal MailAttachment.new('sample2.txt', 'sample2'), mail.attachments[1]
		assert mail.has_html_content
		assert_equal @expected_html_content, mail.html_content
		assert_equal 2, mail.related_attachments.size
		assert_equal MailAttachment.new('sample3.txt', 'sample3', 'sample3-id'), mail.related_attachments[0]
		assert_equal MailAttachment.new('sample4.txt', 'sample4', 'sample4-id'), mail.related_attachments[1]
	end

	if REAL_MAIL_TEST then
		def test_pseudo_send_receive
			# 送信用ユーティリティを準備する
			mail_testutils = MailTestUtils.new(
				SMTP_SERVER_ADDRESS,
				SMTP_SERVER_PORT,
				SMTP_SECURITY_TYPE,
				SMTP_SERVER_USER,
				SMTP_SERVER_PASSWORD,
				nil,
				nil,
				nil,
				nil,
				'UTF-8')

			# 受信用ユーティリティー１を準備する
			mail_testutils1 = MailTestUtils.new(
				nil,
				nil,
				nil,
				nil,
				nil,
				POP3_SERVER1_ADDRESS,
				POP3_SERVER1_PORT,
				POP3_SERVER1_USER,
				POP3_SERVER1_PASSWORD,
				'UTF-8')

			# 受信用ユーティリティー２を準備する
			mail_testutils2 = MailTestUtils.new(
				nil,
				nil,
				nil,
				nil,
				nil,
				POP3_SERVER2_ADDRESS,
				POP3_SERVER2_PORT,
				POP3_SERVER2_USER,
				POP3_SERVER2_PASSWORD,
				'UTF-8')

			# メールを送信する
			from_address = MailAddress.new(SMTP_SERVER_MAILADDRESS, 'テスト送信者')
			to_addresses = [MailAddress.new(POP3_SERVER1_MAILADDRESS, 'テスト受信者１'), MailAddress.new(POP3_SERVER2_MAILADDRESS, 'テスト受信者２')]
			attachments = [MailProcessor.load_attachment("#{File.dirname(File.expand_path(__FILE__))}/data/eyes0367.jpg")]
			mail_testutils.sendmail(
				MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments))
			)

			# サーバ１でメールを受信する
			mail = mail_testutils1.get_mail()

			assert mail != nil
			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content

			# サーバ２でメールを受信する
			mail = mail_testutils2.get_mail()

			assert mail != nil
			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content
		end
	end

	def test_parse_real_mail
		default_expected_default_title = 'タイトル'
		default_expected_from_address = 'test_sender@ownway.info'
		default_expected_to_addresses = ['test_receiver1@ownway.info']

default_expected_content = <<"END_CONTENT"
メール本文１行目
メール本文２行目

空行入れてメール本文４行目
メール本文５行目

空行入れてメール本文７行目
メール本文８行目
END_CONTENT

		default_expected_attachments = [MailProcessor.load_attachment("#{File.dirname(File.expand_path(__FILE__))}/data/eyes0367.jpg")]
		default_expected_html_content = nil

		Dir::glob("#{File.dirname(File.expand_path(__FILE__))}/data/*.txt") do |filename|
			open(filename, 'rb') do |file|
				mail = MailProcessor.parse(file.read, 'UTF-8')

				expected_name = File.basename(filename, '.*')
				expected_title = EXPECTED_PARSE_REAL_MAIL[expected_name] == nil || EXPECTED_PARSE_REAL_MAIL[expected_name]['title'] == nil ?
					default_expected_default_title : EXPECTED_PARSE_REAL_MAIL[expected_name]['title']
				expected_from_address = EXPECTED_PARSE_REAL_MAIL[expected_name] == nil || EXPECTED_PARSE_REAL_MAIL[expected_name]['from_address'] == nil ?
					default_expected_from_address : EXPECTED_PARSE_REAL_MAIL[expected_name]['from_address']
				expected_to_addresses = EXPECTED_PARSE_REAL_MAIL[expected_name] == nil || EXPECTED_PARSE_REAL_MAIL[expected_name]['to_address'] == nil ?
					default_expected_to_addresses : EXPECTED_PARSE_REAL_MAIL[expected_name]['to_address']
				expected_content = EXPECTED_PARSE_REAL_MAIL[expected_name] == nil || EXPECTED_PARSE_REAL_MAIL[expected_name]['content'] == nil ?
					default_expected_content : EXPECTED_PARSE_REAL_MAIL[expected_name]['content']
				expected_attachments = EXPECTED_PARSE_REAL_MAIL[expected_name] == nil || EXPECTED_PARSE_REAL_MAIL[expected_name]['attachments'] == nil ?
					default_expected_attachments : EXPECTED_PARSE_REAL_MAIL[expected_name]['attachments']
				expected_html_content = EXPECTED_PARSE_REAL_MAIL[expected_name] == nil || EXPECTED_PARSE_REAL_MAIL[expected_name]['html_content'] == nil ?
					default_expected_html_content : EXPECTED_PARSE_REAL_MAIL[expected_name]['html_content']

				assert_equal expected_title, mail.subject, "test_target: #{File.basename(filename)}"
				assert_equal expected_from_address, mail.from_address.address, "test_target: #{File.basename(filename)}"
				assert_equal expected_to_addresses.size, mail.to_addresses.size, "test_target: #{File.basename(filename)}"
				mail.to_addresses do |to_address|
					assert expected_to_addresses.index(to_address) != nil, "test_target: #{File.basename(filename)}"
				end
				assert_equal expected_content.encode('UTF-8', 'UTF-8').chomp(''), mail.content.encode('UTF-8', 'UTF-8').chomp(''), "test_target: #{File.basename(filename)}"
				assert_equal expected_attachments.size, mail.attachments.size, "test_target: #{File.basename(filename)}"
				if mail.html_content != nil then
					assert_equal expected_html_content.encode('UTF-8', 'UTF-8').chomp(''), mail.html_content.encode('UTF-8', 'UTF-8').chomp(''), "test_target: #{File.basename(filename)}"
				else
					assert default_expected_html_content == nil, "test_target: #{File.basename(filename)}"
				end
				i = 0
				expected_attachments.each do |attachment|
					assert_equal attachment.filename, mail.attachments[i].filename, "test_target: #{File.basename(filename)}"
					assert_match attachment.content, mail.attachments[i].content, "test_target: #{File.basename(filename)}"
					i = i + 1
				end
			end
		end
	end

end
