# coding: UTF-8
# 

ROOT_PATH = File.dirname(File.expand_path(__FILE__))

$:.unshift(ROOT_PATH + "/../")
$:.unshift(ROOT_PATH + "/../../main/")
$:.unshift(ROOT_PATH + "/../../main/lib")

# mail2weblog のルートパスを設定する
M2W_ROOT_PATH = ROOT_PATH + "/../../main/"

require 'm2w_configurator'
require 'm2w_plugin_manager'
require 'style/formatter'
require 'mailutils/mail_processor'
require 'plugin/format/lib/format_plugin_utils'

require 'minitest/unit'

MiniTest::Unit.autorun

class Example_FormatPlugin
	include FormatPluginUtils

	def plugin_contents(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning", "warning")
			when "error"
				raise FormatPluginError.new("error", "error")
			end
		else
			if content == nil then
				return "example"
			else
				return "example[#{content}]"
			end
		end
	end

	def plugin_contents_inline(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning_inline", "warning_inline")
			when "error"
				raise FormatPluginError.new("error_inline", "error_inline")
			end
		else
			if content == nil then
				return "example_inline"
			else
				return "example_inline[#{content}]"
			end
		end
	end

end

#=Mail2WeblogMailParser のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_formatter.rb 324 2011-10-08 00:42:51Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestFormatter < MiniTest::Unit::TestCase

	def setup
		@formatter = Formatter.new(Mail2WeblogPluginManager.new)
	end

	def test_preformat
content = <<"END_CONTENT"
１行目

２行目
END_CONTENT

content_with_header = <<"END_CONTENT"
tit.タイトル <&>

１行目

２行目
END_CONTENT

expected_content = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] tit.タイトル &lt;&amp;&gt;

１行目

２行目
 --><p>１行目</p>
<p>２行目</p>
END_CONTENT

		assert_equal expected_content.chomp, @formatter.preformat(content, content_with_header, {}, false)[0]
	end

	def test_format_with_plugin
expected_precontent = <<"END_CONTENT"
<div class="plugin">{{example}}</div>
END_CONTENT

		assert_equal expected_precontent.chomp, @formatter.preformat('{{example}}', '{{example}}', {}, false)[0]

expected_postcontent = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] {{example}} -->example
END_CONTENT

		assert_equal expected_postcontent.chomp, @formatter.postformat(expected_precontent, {})[0]
	end

	def test_format_with_plugin_warning
		precontent = @formatter.preformat('{{example(type@warning)}}', '{{example(type@warning)}}', {}, false)[0]
		assert_match %r!\{\{警告: warning\}\}$!, @formatter.postformat(precontent, {})[0]
	end

	def test_format_with_plugin_error
		precontent = @formatter.preformat('{{example(type@error)}}', '{{example(type@error)}}', {}, false)[0]
		assert_match %r!\{\{エラー: error\}\}$!, @formatter.postformat(precontent, {})[0]
	end

	def test_format_with_plugin_inline
expected_precontent = <<"END_CONTENT"
<p>テスト<span class="plugin">{{example}}</span></p>
END_CONTENT

		assert_equal expected_precontent.chomp, @formatter.preformat('テスト{{example}}', 'テスト{{example}}', {}, false)[0]

expected_postcontent = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト{{example}} --><p>テストexample_inline</p>
END_CONTENT

		assert_equal expected_postcontent.chomp, @formatter.postformat(expected_precontent, {})[0]
	end

	def test_format_with_plugin_inline4multiline
expected_precontent = <<"END_CONTENT"
<p>テスト<span class="plugin">{{example(
line1
line2
)}}</span></p>
END_CONTENT

		assert_equal expected_precontent.chomp, @formatter.preformat("テスト{{example(\nline1\nline2\n)}}", "テスト{{example(\nline1\nline2\n)}}", {}, false)[0]

expected_postcontent = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト{{example(\nline1\nline2\n)}} --><p>テストexample_inline[line1\nline2\n]</p>
END_CONTENT

		assert_equal expected_postcontent.chomp, @formatter.postformat(expected_precontent, {})[0]
	end

	def test_format_with_plugin_warning_inline
		precontent = @formatter.preformat('テスト{{example(type@warning)}}', 'テスト{{example(type@warning)}}', {}, false)[0]
		assert_match %r!<p>テスト\{\{警告: warning_inline\}\}</p>$!, @formatter.postformat(precontent, {})[0]
	end

	def test_format_with_plugin_error_inline
		precontent = @formatter.preformat('テスト{{example(type@error)}}', 'テスト{{example(type@error)}}', {}, false)[0]
		assert_match %r!<p>テスト\{\{エラー: error_inline\}\}</p>$!, @formatter.postformat(precontent, {})[0]
	end

	def test_format_with_attachment
expected_precontent = <<"END_CONTENT"
<p>テスト</p>
END_CONTENT

		assert_equal expected_precontent.chomp, @formatter.preformat('テスト', 'テスト', {}, true)[0]

expected_postcontent = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト --><!--[attachment_urls] 1=http://abc.def.ghi/jkl.txt --><p>テスト</p>
END_CONTENT

		vars = {
			:attachment_urls => {1 => 'http://abc.def.ghi/jkl.txt'}
		}
		assert_equal expected_postcontent.chomp, @formatter.postformat(expected_precontent, vars)[0]
	end

	def test_format_with_attachments
expected_precontent = <<"END_CONTENT"
<p>テスト</p>
END_CONTENT

		assert_equal expected_precontent.chomp, @formatter.preformat('テスト', 'テスト', {}, true)[0]

expected_postcontent = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト --><!--[attachment_urls] 1=http://abc.def.ghi/jkl.txt,2=http://mno.pqr.stu/vwx.txt --><p>テスト</p>
END_CONTENT

		vars = {
			:attachment_urls => {
				1 => 'http://abc.def.ghi/jkl.txt',
				2 => 'http://mno.pqr.stu/vwx.txt',
			}
		}
		assert_equal expected_postcontent.chomp, @formatter.postformat(expected_precontent, vars)[0]
	end

	def test_get_uncommentout_content
content = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] tit.タイトル &lt;&amp;&gt;

１行目

２行目 --><p>１行目</p>
<p>２行目</p>
END_CONTENT

expected_content = <<"END_CONTENT"
tit.タイトル <&>

１行目

２行目
END_CONTENT

		(version, content) = @formatter.get_uncommentout_content(content)
		assert_equal M2W_VERSION, version
		assert_equal expected_content.chomp, content
	end

	def test_get_uncommentout_content_old
content = <<"END_CONTENT"
<!-- １行目

２行目 --><p>１行目</p>
<p>２行目</p>
END_CONTENT

expected_content = <<"END_CONTENT"
１行目

２行目
END_CONTENT

		(version, content) = @formatter.get_uncommentout_content(content)
		assert version == nil
		assert_equal expected_content.chomp, content
	end

	def test_get_uncommentout_attachment_urls_content
		content = "<!--[attachment_urls] 1=abc.txt,2=def.txt,3=ghi.txt -->"
		urls = @formatter.get_uncommentout_attachment_urls_content(content)

		assert_equal 3, urls.length
		assert_equal 'abc.txt', urls['1']
		assert_equal 'def.txt', urls['2']
		assert_equal 'ghi.txt', urls['3']
	end

end
