# coding: UTF-8

ROOT_PATH = File.dirname(File.expand_path(__FILE__))

$:.unshift(ROOT_PATH + "/../")
$:.unshift(ROOT_PATH + "/../../main/")
$:.unshift(ROOT_PATH + "/../../main/lib")

# mail2weblog のルートパスを設定する
M2W_ROOT_PATH = ROOT_PATH + "/../../main/"

require 'm2w_configurator'
require 'm2w_plugin_manager'

require 'time'
require 'minitest/unit'

MiniTest::Unit.autorun

class Example_FormatPlugin
	include FormatPluginUtils

	def plugin_preset(param, vars)
		return {"preset1" => "value1"}
	end

	def plugin_preset_inline(param, vars)
		return {"presetinline1" => "value1"}
	end

	def plugin_contents(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning", "warning")
			when "error"
				raise FormatPluginError.new("error", "error")
			end
		else
			if content == nil then
				return "example"
			else
				return "example[#{content}]"
			end
		end
	end

	def plugin_contents_inline(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning_inline", "warning_inline")
			when "error"
				raise FormatPluginError.new("error_inline", "error_inline")
			end
		else
			if content == nil then
				return "example_inline"
			else
				return "example_inline[#{content}]"
			end
		end
	end

end

class Empty_FormatPlugin
end

#=Mail2WeblogPluginManager のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_m2w_plugin_manager.rb 625 2012-03-21 02:43:40Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestMail2WeblogPluginManager < MiniTest::Unit::TestCase
	include FormatPluginUtils

	def test_command_plugin
		plugin_manager = Mail2WeblogPluginManager.new

		command = plugin_manager.get_command_plugin('post')
		assert_instance_of Post_CommandPlugin, command

		command = plugin_manager.get_command_plugin('no_command')
		assert_nil command
	end

	def test_format_plugin
		plugin_manager = Mail2WeblogPluginManager.new

		format = plugin_manager.get_format_plugin('img')
		assert_instance_of Img_FormatPlugin, format

		format = plugin_manager.get_format_plugin('no_format')
		assert_nil format
	end

	def test_weblog_plugin
		plugin_manager = Mail2WeblogPluginManager.new

		weblog = plugin_manager.get_weblog_plugin('word_press')
		assert_instance_of WordPress_WeblogPlugin, weblog

		weblog = plugin_manager.get_weblog_plugin('movable_type')
		assert_instance_of MovableType_WeblogPlugin, weblog

		weblog = plugin_manager.get_weblog_plugin('fc2')
		assert_instance_of Fc2_WeblogPlugin, weblog

		weblog = plugin_manager.get_weblog_plugin('hatena_diary')
		assert_instance_of HatenaDiary_WeblogPlugin, weblog

		weblog = plugin_manager.get_weblog_plugin('no_weblog')
		assert_nil weblog
	end

	def test_get_format_plugin_vars
		plugin_manager = Mail2WeblogPluginManager.new
		result = plugin_manager.get_format_plugin_vars('example', {})
		assert_equal 1, result.size
		assert_equal 'example', result[M2W_FORMAT_PLUGIN_CONF_NAME][M2W_FORMAT_PLUGIN_NAME]

		plugin_manager = Mail2WeblogPluginManager.new({}, 
			{
				'COMMON' => {
					'common1' => 'value1',
				},
				'example' => {
					'attr1' => 'value1',
				},
			},
		{})
		result = plugin_manager.get_format_plugin_vars('example', {})
		assert_equal 2, result.size
		assert_equal 'value1', result[M2W_FORMAT_PLUGIN_COMMON_CONF_NAME]['common1']
		assert_equal 'value1', result[M2W_FORMAT_PLUGIN_CONF_NAME]['attr1']
		assert_equal 'example', result[M2W_FORMAT_PLUGIN_CONF_NAME][M2W_FORMAT_PLUGIN_NAME]
	end

	def test_call_format_plugin_preset
		plugin_manager = Mail2WeblogPluginManager.new

		result = plugin_manager.call_format_plugin_preset('example', '', {})
		assert_equal 1, result.size
		assert 'value1', result['preset1']

		assert_raises(FormatPluginWarning) do
			plugin_manager.call_format_plugin_preset('noexists', '', {})
		end
	end

	def test_call_format_plugin_preset_inline
		plugin_manager = Mail2WeblogPluginManager.new

		result = plugin_manager.call_format_plugin_preset_inline('example', '', {})
		assert_equal 1, result.size
		assert 'value1', result['presetinline1']

		assert_raises(FormatPluginWarning) do
			plugin_manager.call_format_plugin_preset_inline('noexists', '', {})
		end
	end

	def test_call_format_plugin_contents
		plugin_manager = Mail2WeblogPluginManager.new

		assert_equal 'example', plugin_manager.call_format_plugin_contents('example', '', {})

		assert_raises(FormatPluginWarning) do
			plugin_manager.call_format_plugin_contents('noexists', '', {})
		end

		assert_raises(FormatPluginWarning) do
			plugin_manager.call_format_plugin_contents('empty', '', {})
		end
	end

	def test_call_format_plugin_contents_inline
		plugin_manager = Mail2WeblogPluginManager.new

		assert_equal 'example_inline', plugin_manager.call_format_plugin_contents_inline('example', '', {})

		assert_raises(FormatPluginWarning) do
			plugin_manager.call_format_plugin_contents_inline('noexists', '', {})
		end

		assert_raises(FormatPluginWarning) do
			plugin_manager.call_format_plugin_contents_inline('empty', '', {})
		end
	end

	def test_get_weblog_filepath
		postid = 9999
		date = Time.parse('2000-01-02T03:04:05')
		index = 6
		filename = 'file.txt'
		default = 'default.txt'

		plugin_manager = Mail2WeblogPluginManager.new()
		assert_equal 'default.txt', plugin_manager.get_weblog_filepath('attachment_url', postid, date, index, filename, default)

		plugin_manager = Mail2WeblogPluginManager.new({}, {},
			{
				'COMMON' => {
					'attachment_url' => '${postid}_${year}_${month}_${day}_${hour}_${min}_${sec}_${index}_${filename}_${filename_basename}_${filename_extname}',
				}
			})
		assert_equal '9999_2000_01_02_03_04_05_6_file.txt_file_.txt',
			plugin_manager.get_weblog_filepath('attachment_url', postid, date, index, filename, default)
	end

	# TimeZone が +09:00（日本）であることを想定しています。
	def test_parse_weblog_date
		plugin_manager = Mail2WeblogPluginManager.new()
		date = plugin_manager.parse_weblog_date('2011-11-10T19:45:15')
		date = date - date.gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 10, date.hour
		assert_equal 45, date.min
		assert_equal 15, date.sec
		assert (date.gmt_offset == 0 && date.gmt?) || (date.gmt_offset != 0 && !date.gmt?)

		plugin_manager = Mail2WeblogPluginManager.new(
			{},
			{},
			{
				'COMMON' => {
					'gmt_offset_hour' => '+01',
					'gmt_offset_min'  => '+15',
				},
			}
		)
		date = plugin_manager.parse_weblog_date('2011-11-10T19:45:15')
		gmt_offset = date.gmt_offset
		date = date - gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 18, date.hour
		assert_equal 30, date.min
		assert_equal 15, date.sec
		assert date.gmt?

		plugin_manager = Mail2WeblogPluginManager.new(
			{},
			{},
			{
				'COMMON' => {
					'gmt_offset_hour' => '-01',
					'gmt_offset_min'  => '-15',
				},
			}
		)
		date = plugin_manager.parse_weblog_date('2011-11-10T19:45:15')
		gmt_offset = date.gmt_offset
		date = date - gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 21, date.hour
		assert_equal 0, date.min
		assert_equal 15, date.sec
		assert date.gmt?
	end

	# TimeZone が +09:00（日本）であることを想定しています。
	def test_parse_weblog_date2
		plugin_manager = Mail2WeblogPluginManager.new()
		date = plugin_manager.parse_weblog_date(nil, Time.parse('2011-11-10T19:45:15'))
		date = date - date.gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 10, date.hour
		assert_equal 45, date.min
		assert_equal 15, date.sec
		assert (date.gmt_offset == 0 && date.gmt?) || (date.gmt_offset != 0 && !date.gmt?)

		plugin_manager = Mail2WeblogPluginManager.new(
			{},
			{},
			{
				'COMMON' => {
					'gmt_offset_hour' => '+01',
					'gmt_offset_min'  => '+15',
				},
			}
		)
		date = plugin_manager.parse_weblog_date(nil, Time.parse('2011-11-10T19:45:15+09:00'))
		gmt_offset = date.gmt_offset
		date = date - gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 18, date.hour
		assert_equal 30, date.min
		assert_equal 15, date.sec
		assert date.gmt?

		plugin_manager = Mail2WeblogPluginManager.new(
			{},
			{},
			{
				'COMMON' => {
					'gmt_offset_hour' => '-01',
					'gmt_offset_min'  => '-15',
				},
			}
		)
		date = plugin_manager.parse_weblog_date(nil, Time.parse('2011-11-10T19:45:15'))
		gmt_offset = date.gmt_offset
		date = date - gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 21, date.hour
		assert_equal 0, date.min
		assert_equal 15, date.sec
		assert date.gmt?
	end

	# TimeZone が +09:00（日本）であることを想定しています。
	def test_parse_weblog_date3
		plugin_manager = Mail2WeblogPluginManager.new()
		date = plugin_manager.parse_weblog_date('2011-11-10T19:45:15', Time.now)
		date = date - date.gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 10, date.hour
		assert_equal 45, date.min
		assert_equal 15, date.sec
		assert (date.gmt_offset == 0 && date.gmt?) || (date.gmt_offset != 0 && !date.gmt?)

		plugin_manager = Mail2WeblogPluginManager.new(
			{},
			{},
			{
				'COMMON' => {
					'gmt_offset_hour' => '+01',
					'gmt_offset_min'  => '+15',
				},
			}
		)
		date = plugin_manager.parse_weblog_date('2011-11-10T19:45:15', Time.now)
		gmt_offset = date.gmt_offset
		date = date - gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 18, date.hour
		assert_equal 30, date.min
		assert_equal 15, date.sec
		assert date.gmt?

		plugin_manager = Mail2WeblogPluginManager.new(
			{},
			{},
			{
				'COMMON' => {
					'gmt_offset_hour' => '-01',
					'gmt_offset_min'  => '-15',
				},
			}
		)
		date = plugin_manager.parse_weblog_date('2011-11-10T19:45:15', Time.now)
		gmt_offset = date.gmt_offset
		date = date - gmt_offset
		assert_equal 2011, date.year
		assert_equal 11, date.month
		assert_equal 10, date.day
		assert_equal 21, date.hour
		assert_equal 0, date.min
		assert_equal 15, date.sec
		assert date.gmt?
	end

	def test_default_format_mode
		plugin_manager = Mail2WeblogPluginManager.new()

		assert_equal 'html', plugin_manager.default_format_mode

		plugin_manager = Mail2WeblogPluginManager.new(
			{},
			{},
			{
				'COMMON' => {
					'default_format_mode' => 'wiki',
				},
			}
		)

		assert_equal 'wiki', plugin_manager.default_format_mode
	end

end
