#
# Cookbook Name:: ganglia
# Recipe Spec:: web_spec
#
# Copyright 2015, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

require_relative '../spec_helper'

describe 'ganglia::web' do
  let(:chef_run_on_debian) {
    ChefSpec::SoloRunner.new(platform: 'debian', version: '7.6') {|node|
      node.set['ganglia']['web']['AuthType'] = 'Kerberos'
    }.converge(described_recipe)
  }

  it 'installs ganglia webfrontend packages on debian family.' do
    expected = expect(chef_run_on_debian)
    expected.to install_package('apache2')
    expected.to install_package('ganglia-webfrontend')
    expected.to install_package('php5-gd')
    expected.to install_package('libapache2-mod-auth-kerb')
    expected.to enable_service('apache2')
  end

  it 'links /etc/apache2/conf.d/ganglia-webfrontend.conf to /etc/ganglia-webfrontend/apache.conf' do
    expect(chef_run_on_debian).to \
      create_link('/etc/apache2/conf.d/ganglia-webfrontend.conf').
        with(to: '/etc/ganglia-webfrontend/apache.conf')
  end

  it 'renders /etc/ganglia-webfrontend/apache.conf on debian family.' do
    expect(chef_run_on_debian).to \
      render_file('/etc/ganglia-webfrontend/apache.conf').
        with_content(%r|^\s+AuthType\s+Kerberos$|)
  end

  it 'notifies apache2 service\'s restart after rendering on debian family.' do
    expect(chef_run_on_debian.template('/etc/ganglia-webfrontend/apache.conf')).to \
      notify('service[apache2]').to(:restart).delayed
  end

  let(:chef_run_on_rhel) {
    ChefSpec::SoloRunner.new(platform: 'centos', version: '7.0') {|node|
      node.set['ganglia']['web']['AuthType'] = 'Kerberos'
    }.converge(described_recipe)
  }

  it 'installs ganglia-web package on rhel family.' do
    expected = expect(chef_run_on_rhel)
    expected.to install_package('ganglia-web')
    expected.to install_package('mod_auth_kerb')
    expected.to enable_service('httpd')
  end

  it 'renders /etc/httpd/conf.d/ganglia.conf on rhel family.' do
    expect(chef_run_on_rhel).to \
      render_file('/etc/httpd/conf.d/ganglia.conf').
        with_content(%r|^\s+AuthType\s+Kerberos$|)
  end

  it 'notifies httpd service\'s restart after rendering on rhel family.' do
    expect(chef_run_on_rhel.template('/etc/httpd/conf.d/ganglia.conf')).to \
      notify('service[httpd]').to(:restart).delayed
  end
end

