#
# Cookbook Name:: dcos-grid
# Recipe:: node-commons
#
# Copyright 2016, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

# https://dcos.io/docs/1.8/administration/installing/custom/system-requirements/

genconf_dir = node['dcos-grid']['bootstrap']['genconf_dir']

directory genconf_dir do
  owner 'root'
  group 'root'
  mode '0755'
  recursive true
  action :create
end

template "#{File.dirname(genconf_dir)}/uninstall.sh" do
  source  'opt/dcos-grid/uninstall.sh'
  owner 'root'
  group 'root'
  mode '0755'
end

group 'nogroup' do
  action :create
end

case node['platform_family']
when 'rhel'
  service 'firewalld' do
    action [:stop, :disable]
  end

  [
    'tar',
    'xz',
    'unzip',
    'curl',
    'ipset',
  ].each {|pkg|
    package pkg do
      action :install
    end
  }

  bash 'disable_selinux' do
    code <<-EOH
      setenforce Permissive
    EOH
    not_if 'getenforce | grep -i \'\(disabled\|permissive\)\''
  end
  template '/etc/selinux/config' do
    source  'etc/selinux/config'
    owner 'root'
    group 'root'
    mode '0644'
  end

  # https://dcos.io/docs/1.8/administration/installing/custom/system-requirements/install-docker-centos/
  if node['dcos-grid']['docker-engine']['setup']
    bash 'load_overlayfs_kernel_module' do
      code <<-EOH
        modprobe overlay
      EOH
      not_if 'lsmod | grep overlay'
    end
    template '/etc/modules-load.d/overlay.conf' do
      source  'etc/modules-load.d/overlay.conf'
      owner 'root'
      group 'root'
      mode '0644'
    end

    template '/etc/yum.repos.d/docker.repo' do
      source  'etc/yum.repos.d/docker.repo'
      owner 'root'
      group 'root'
      mode '0644'
    end

    directory '/etc/systemd/system/docker.service.d' do
      owner 'root'
      group 'root'
      mode '0755'
      action :create
    end

    template '/etc/systemd/system/docker.service.d/override.conf' do
      source  'etc/systemd/system/docker.service.d/override.conf'
      owner 'root'
      group 'root'
      mode '0644'
    end

    [
      'docker-engine-selinux',
      'docker-engine',
    ].each {|pkg|
      package pkg do
        action :install
        version node['dcos-grid']['docker-engine']['version']
        # dockerrepo is disabled by default to prevent automatic update.
        options '--enablerepo=dockerrepo'
      end
    }

    service 'docker' do
      action [:start, :enable]
    end
  end
when 'debian'
  [
    'tar',
    'xz-utils',
    'unzip',
    'curl',
    'ipset',
  ].each {|pkg|
    package pkg do
      action :install
    end
  }

  # https://docs.docker.com/engine/installation/linux/ubuntulinux/
  if node['dcos-grid']['docker-engine']['setup']
    [
      'apt-transport-https',
      'ca-certificates',
      # for aufs
      "linux-image-extra-#{node['os_version']}",
      'linux-image-extra-virtual',
    ].each {|pkg|
      package pkg do
        action :install
      end
    }

    template '/etc/apt/sources.list.d/docker.list' do
      source  'etc/apt/sources.list.d/docker.list'
      owner 'root'
      group 'root'
      mode '0644'
    end

    apt_repo_config = node['dcos-grid']['docker']['apt_repo']
    bash 'apt-key_adv_docker_tools_key' do
      code <<-"EOH"
        apt-key adv --keyserver #{apt_repo_config['keyserver']} --recv-keys #{apt_repo_config['recv-keys']}
        apt-get update
      EOH
      not_if 'apt-key list | grep -i docker'
    end

    # Pinning Docker version
    template '/etc/apt/preferences.d/docker.pref' do
      source  'etc/apt/preferences.d/docker.pref'
      owner 'root'
      group 'root'
      mode '0644'
    end

    # Default Storage Driver: aufs
    [
      'docker-engine',
    ].each {|pkg|
      package pkg do
        action :install
        version node['dcos-grid']['docker-engine']['version']
      end
    }

    service 'docker' do
      action [:start, :enable]
    end
  end

  # work around for DC/OS service units
  link '/usr/bin/mkdir' do
    to '/bin/mkdir'
  end
  link '/usr/bin/ln' do
    to '/bin/ln'
  end
  link '/usr/bin/tar' do
    to '/bin/tar'
  end
  link '/sbin/useradd' do
    to '/usr/sbin/useradd'
  end
end
