/* arguments.cpp
   Copyright (C) 2005 Free Software Foundation, Inc.

This file is part of Mysaifu JVM

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.
*/

#include "StdAfx.h"
#include "arguments.h"
#include "common_funcs.h"
#include "jni_funcs.h"

/**
 * WXgL[
 */
#define REG_ROOT						_T("Software\\Mysaifu\\Mysaifu JVM")
#define REG_DEFAULT_BOOT_CLASSPATH		_T("DefaultBootClassPath")
#define REG_DEFAULT_CLASSPATH			_T("DefaultClassPath")
#define REG_DEFAULT_MAX_HEAP_SIZE		_T("DefaultMaxHeapSize")
#define REG_DEFAULT_CURRENT_DIRECTORY	_T("DefaultCurrentDirectory")

/**
 * ftHg̃u[gNXpX
 * %s ̕ɂ̓CXg[pX
 */
#define DEFAULT_BOOT_CLASSPATH_FORMAT	_T("%s\\lib\\rt.jar")
static _TCHAR g_default_boot_classpath[MAX_PATH + 1];

/**
 * ftHg̍őq[vTCY
 */
#define DEFAULT_MAX_HEAP_SIZE_VALUE	(2 * 1024 * 1024)
static unsigned int g_default_max_heap_size = DEFAULT_MAX_HEAP_SIZE_VALUE;

/**
 * ftHg̃VXeNXpX
 */
#define DEFAULT_CLASSPATH_VALUE	_T("\\;\\My Documents")
static _TCHAR g_default_classpath[MAX_PATH + 1] = DEFAULT_CLASSPATH_VALUE;

/**
 * ftHg̃JgfBNg
 */
#define DEFAULT_CURRENT_DIRECTORY_VALUE	_T("\\");
static _TCHAR g_default_current_directory[MAX_PATH + 1] = DEFAULT_CURRENT_DIRECTORY_VALUE;

/**
 * ftHgJavaX^bNTCY
 */
#define DEFAULT_JAVA_STACK_SIZE	(32*1024)

/**
 * ftHg̃lCeBuX^bNTCY
 */
#define DEFAULT_NATIVE_STACK_SIZE	163840

/**
 * w肳ꂽp^obt@ɓǂݍ
 */
static int read_parameter(const _TCHAR* str, _TCHAR* buff, int bufflen);

/**
 * ftHglWXgǂݍ
 */
static void load_default_values_from_registory();

/**
 * arguments
 */
static void init_arguments(arguments* config);

/**
 * ʂǂݍ
 */
static int parse_memory_size(const _TCHAR* opt);

/**
 * ɊւԂ
 */
void init_arguments_settings() {
	// WXg珉lǂݍł
	load_default_values_from_registory();
}

/**
 * 쐬
 */
arguments* arguments_create() {
	arguments* args = (arguments*) calloc(1, sizeof(arguments));
	if (! args) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	init_arguments(args);
	return args;
}

/**
 * Ɋ蓖Ăꂽ폜
 */
void arguments_delete(arguments* args) {
	// classpathJ
	free(args->classpath);
	args->classpath = NULL;

	// vpeBJ
	free(args->properties);
	args->properties = NULL;
	args->properties_count = 0;

	// J
	free(args->argv);
	args->argv = NULL;

	free(args);
}

/**
 * ʂǂݍ
 */
static int parse_memory_size(_TCHAR* opt) {
	int result = 0;

	// 񖖔ɂ͒PʁiK܂M)w\
	int len = _tcslen(opt);
	_TCHAR unit = opt[len - 1];
	if (unit == _T('0') || unit == _T('1') || unit == _T('2') || unit == _T('3') || unit == _T('4')
			|| unit == _T('5') || unit == _T('6') || unit == _T('7') || unit == _T('8')
			|| unit == _T('9')) {
		// l̏ꍇ͉HȂ
	} else if (unit == _T('K') || unit == _T('M')) {
		//  'K' ܂ 'M' ̏ꍇ͐l݂̂ɂ
		opt[len - 1] = _T('\0');
	} else {
		// w肪
		result = -1;
	}

	if (! result) {
		// lǂݎ
		result = _ttoi(opt);
		if (unit == _T('k') || unit == _T('K')) {
			result *= 1024;
		} else if (unit == _T('m') || unit == _T('M')) {
			result *= (1024 * 1024);
		}
	}
	return result;
}

/**
 * w肳ꂽ̓eɊÂA𐶐
 */
arguments* arguments_load(const _TCHAR* params) {
	arguments* args = arguments_create();

	bool error = false;
	bool option_end = false;
	while (*params && ! error) {
		// 󔒂͓ǂݔ΂
		while (*params && *params == _T(' ')) {
			params++;
		}
		if (! *params) {
			break;
		}
		
		if (*params == '-' && (! option_end)) {
			// IvV̏
			if (_tcsncmp(params, _T("-cp "), 4) == 0
				|| _tcsncmp(params, _T("-classpath "), 11) == 0) {
				// -cp ܂ -classpath IvV

				// ̋󔒂T
				while (*params != _T(' ')) {
					params++;
				}
				if (! *params) {
					break;
				}

				// p^ǂݍ
				// classpath͉ϒƂ(Bug #12863)
				int param_length = read_parameter(params,
												  NULL,
												  0);
				args->classpath = (_TCHAR*) malloc(sizeof(_TCHAR) * (param_length + 1));
				if (! args->classpath) {
					fatal_error(FATAL_ERROR_NO_MEMORY);
				}

				params += read_parameter(params,
										 args->classpath,
										 param_length + 1);

			} else if (_tcsncmp(params, _T("-jar "), 5) == 0) {
				// jarIvV
				args->jar = true;
				params += 5;
				params += read_parameter(params,
										 args->jar_file_name,
										 sizeof(args->jar_file_name) / sizeof(*args->jar_file_name));
				
				// ŃIvV͏I
				option_end = true;

			} else if (_tcsncmp(params, _T("-Xbootclasspath:"), 16) == 0) {
				// -Xbootclasspath: IvV
				params += 16;
				params += read_parameter(params,
										 args->boot_classpath,
										 sizeof(args->boot_classpath) / sizeof(*args->boot_classpath));

			} else if (_tcsncmp(params, _T("-Xmx"), 4) == 0) {
				// -XmxIvV
				params += 4;
				_TCHAR opt[256];
				params += read_parameter(params, opt, sizeof(opt) / sizeof(*opt));

				int value = parse_memory_size(opt);
				// w肪
				if (value < 0) {
					error = true;
					break;
				}
				args->max_heap_size = value;

			} else if (_tcsncmp(params, _T("-Xoss"), 5) == 0) {
				// -XossIvV(JavaX^bNTCYj
				params += 5;
				_TCHAR opt[256];
				params += read_parameter(params, opt, sizeof(opt) / sizeof(*opt));

				int value = parse_memory_size(opt);
				// w肪
				if (value < 0) {
					error = true;
					break;
				}
				args->java_stack_size = value;

			} else if (_tcsncmp(params, _T("-Xss"), 4) == 0) {
				// -XssIvV(lCeBuX^bNTCYj
				params += 4;
				_TCHAR opt[256];
				params += read_parameter(params, opt, sizeof(opt) / sizeof(*opt));

				int value = parse_memory_size(opt);
				// w肪
				if (value < 0) {
					error = true;
					break;
				}
				args->native_stack_size = value;

			} else if (_tcsncmp(params, _T("-Xloggc:"), 8) == 0) {
				// -Xloggc: IvV
				params += 8;
				params += read_parameter(params, args->loggc, sizeof(args->loggc) / sizeof(*args->loggc));

			} else if (_tcsncmp(params, _T("-ea "), 4) == 0) {
				// -ea
				params += 4;
				args->enableAssertions = true;
			
			} else if (_tcsncmp(params, _T("-enableassertions "), 18) == 0) {
				// -enableassertions
				params += 18;
				args->enableAssertions = true;

			} else if (_tcsncmp(params, _T("-da "), 4) == 0) {
				// -da
				params += 4;
				args->enableAssertions = false;

			} else if (_tcsncmp(params, _T("-disableassertions "), 19) == 0) {
				// -disableassertions
				params += 19;
				args->enableAssertions = false;

			} else if (_tcsncmp(params, _T("-D"), 2) == 0) {
				// -Dkey=value
				params += 2;
				property prop;
				int index = 0;
				// ŏ = ʒu܂ł key
				while (*params && *params != _T('=')) {
					prop.key[index++] = *params++;
				}
				prop.key[index] = _T('\0');
				if (*params == _T('=')) {
					params++;
				}
				params += read_parameter(params, prop.value, sizeof(prop.value) / sizeof(*prop.value));
				arguments_put_property(args, prop.key, prop.value);

			} else if (_tcsncmp(params, _T("-verbose:gc "), 12) == 0) {
				// -verbose:gc IvV
				params += 12;
				args->verbose_gc = true;

			} else if (_tcsncmp(params, _T("-verify "), 8) == 0) {
				// -verify IvV
				params += 8;
				args->verify_mode = VERIFY_MODE_ALL;

			} else if (_tcsncmp(params, _T("-noverify "), 10) == 0) {
				// -noverify IvV
				params += 10;
				args->verify_mode = VERIFY_MODE_NONE;

			} else if (_tcsncmp(params, _T("-setcwd "), 8) == 0) {
				// -setcwdIvV
				params += 8;
				params += read_parameter(params,
										 args->current_directory,
										 sizeof(args->current_directory) / sizeof(args->current_directory[0]));

			} else if (_tcsncmp(params, _T("-Xhidevmwindow "), 15) == 0) {
				// -Xhidevmwindow IvV
				params += 15;
				args->hide_vm_window = true;

			} else if (_tcsncmp(params, _T("-Xconsole "), 10) == 0) {
				// -Xconsole IvV
				params += 10;
				args->show_console = true;

			} else if (_tcsncmp(params, _T("-Xlogfile:"), 10) == 0) {
				// -Xlogfile: IvV
				params += 10;
				params += read_parameter(params, args->logfile, sizeof(args->logfile) / sizeof(*args->logfile));

			} else if (_tcsncmp(params, _T("-Xusestdio "), 11) == 0) {
				// -Xusestdio IvV
				params += 11;
				args->use_stdio = true;

			} else if (_tcsncmp(params, _T("-Xmemusage "), 11) == 0) {
				// Option -Xmemusage
				params += 11;
				args->show_current_heap_usage = true;

			} else {
				// sȃIvVw
				error = true;
				break;
			}
		} else {
			if (! args->jar) {
				// jarIvVw肳ĂȂꍇ́A
				// sNXǂݍ
				const _TCHAR* start = params;
				while (*params) {
					if (*params == _T(' ')) {
						break;
					}
					params++;
				}
				int len = (params - start) + 1;
				int buffsize = sizeof(args->class_name) / sizeof(*args->class_name); 
				len = (len < buffsize) ? len : buffsize;
				_tcsncpy(args->class_name, start, len - 1);
				args->class_name[len - 1] = _T('\0');
			}

			// c̕ JavavOɓnƂȂ
			while (*params && *params == _T(' ')) {
				params++;
			}
			if (*params) {
				int len = _tcslen(params);
				args->argv = (_TCHAR*) malloc(sizeof(_TCHAR) * (len + 1));
				if (! args->argv) {
					fatal_error(FATAL_ERROR_NO_MEMORY);
				}
				_tcscpy(args->argv, params);
			}
			goto END;
		}
	}

END:
	if (error) {
		arguments_delete(args);
		args = NULL;
	}
	return args;
}

/**
 * w肳ꂽt@CɁAۑ
 */
int arguments_store(arguments* args, FILE* file) {
	// u[gNXpXo͂
	if (_tcscmp(args->boot_classpath, g_default_boot_classpath)) {
		// ftHglƈقȂꍇɂ̂ݏo
		_fputts(_T(" -Xbootclasspath:\""), file);
		_fputts(args->boot_classpath, file);
		_fputts(_T("\""), file);
	}
	
	// őq[vTCYo͂
	if (args->max_heap_size != g_default_max_heap_size) {
		// ftHglƈقȂꍇɂ̂ݏo
		_ftprintf(file, _T(" -Xmx%dK"), args->max_heap_size / 1024);
	}

	// JavaX^bNTCYo͂
	if (args->java_stack_size != DEFAULT_JAVA_STACK_SIZE) {
		// ftHglƈقȂꍇɂ̂ݏo
		_ftprintf(file, _T(" -Xoss%d"), args->java_stack_size);
	}

	// lCeBuX^bNTCYo͂
	if (args->native_stack_size != DEFAULT_NATIVE_STACK_SIZE) {
		// ftHglƈقȂꍇɂ̂ݏo
		_ftprintf(file, _T(" -Xss%d"), args->native_stack_size);
	}


	// AT[V̗L^o͂
	if (args->enableAssertions) {
		_fputts(_T(" -ea"), file);
	} else {
		// ftHg͖
		// _fputts(_T(" -da"), file);
	}

	// -verbose:gc
	if (args->verbose_gc) {
		_fputts(_T(" -verbose:gc "), file);
	}
	
	// xt@C[h
	switch (args->verify_mode) {
	case VERIFY_MODE_ALL:
		_fputts(_T(" -verify "), file);
		break;

	case VERIFY_MODE_NONE:
		// ftHgl
		break;

	}

	// JgfBNgݒ肷
	if (_tcscmp(args->current_directory, g_default_current_directory) != 0) {
		// ftHglƈقȂꍇɂ̂ݏo
		_ftprintf(file, _T(" -setcwd \"%s\" "), args->current_directory);
	}

	// VMEChE\ɂ
	if (args->hide_vm_window) {
		_fputts(_T(" -Xhidevmwindow "), file);
	}

	// R\[\^\
	if (args->show_console) {
		_fputts(_T(" -Xconsole "), file);
	}

	if (_tcslen(args->logfile)) {
		_ftprintf(file, _T(" -Xlogfile:\"%s\""), args->logfile);
	}

	// Wo͂̎gp
	if (args->use_stdio) {
		_fputts(_T(" -Xusestdio "), file);
	}

	// Show current memory usage
	if (args->show_current_heap_usage) {
		_fputts(_T(" -Xmemusage "), file);
	}

	// VXevpeBo͂
	for (int i = 0; i < args->properties_count; ++i) {
		_fputts(_T(" -D"), file);
		_fputts(args->properties[i].key, file);
		_fputts(_T("=\""), file);
		_fputts(args->properties[i].value, file);
		_fputts(_T("\" "), file);
	}
	
	if (args->jar) {
		// jarIvVw肳Ăꍇ
		_ftprintf(file, _T(" -jar \"%s\""), args->jar_file_name);
	} else {
		// jarIvVw肳ĂȂꍇ
		// NXpXo͂
		if (_tcscmp(args->classpath, g_default_classpath)) {
			_fputts(_T(" -cp \""), file);
			_fputts(args->classpath, file);
			_fputts(_T("\""), file);
		}
		// NXo͂
		_fputts(_T(" "), file);
		_fputts(args->class_name, file);
	}

	// JavavOւ̈o͂
	if (args->argv) {
		_ftprintf(file, _T(" %s"), args->argv);
	}
	return 1;
}

/**
 * vpeBݒ肷
 */
void arguments_put_property(arguments* args, const _TCHAR* key, const _TCHAR* value) {
	for (int i = 0; i < args->properties_count; ++i) {
		if (! _tcscmp(args->properties[i].key, key)) {
			// vL[eu
			_tcscpy(args->properties[i].value, value);
			return;
		}
	}

	// VKɃobt@쐬Ēǉ
	args->properties_count++;
	args->properties = (property*) realloc(args->properties,
											 sizeof(property) * args->properties_count);
	if (! args->properties) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	_tcscpy(args->properties[args->properties_count - 1].key, key);
	_tcscpy(args->properties[args->properties_count - 1].value, value);
}

/**
 * ׂẴvpeBNA
 */
void arguments_clear_all_properties(arguments* args) {
	free(args->properties);
	args->properties_count = 0;
	args->properties = NULL;
}

/**
 * vpeB̃L[Ԃ
 */
const _TCHAR* arguments_get_property_key(arguments* args, int index) {
	assert(index >= 0 && index < args->properties_count);
	return args->properties[index].key;
}

/**
 * w肳ꂽL[ɑΉvpeBlԂ
 */
const _TCHAR* arguments_get_property(arguments* args, const _TCHAR* key) {
	for (int i = 0; i < args->properties_count; ++i) {
		if (! _tcscmp(args->properties[i].key, key)) {
			return args->properties[i].value;
		}
	}
	return NULL;
}

/**
 * w肳ꂽp^obt@ɓǂݍ݁Ap^̕ԂB
 * buffNULLw肳ꂽꍇɂ́Aǂݍݏ͍s킸p^̕ԂB
 *
 * @param	str	p^܂ޕB
 * @param	buff Ǎobt@BNULLw肵ꍇɂ̓obt@ւ̓ǂݍ݂͍s킸Ap^̕ԂB
 * @param	bufflen	Ǎobt@̃TCYBbuffNULL̏ꍇɂ͖B
 * @return	ǂݍ񂾃p[^B
 */
static int read_parameter(const _TCHAR* str, _TCHAR* buff, int bufflen) {
	const _TCHAR* start = str;

	// 󔒂ǂݔ΂
	while (*str && *str == _T(' ')) {
		str++;
	}

	// NI[g𒲂ׂ
	bool quote = (*str == _T('"')) ? true : false;
	if (quote) {
		str++;
	}
	int copylen = 0;
	if (!buff) {
		// obt@w肳ĂȂꍇ
		bufflen = INT_MAX;
	}
	while (*str && copylen < bufflen) {
		if (quote) {
			if (*str == _T('"')) {
				str++;
				break;
			}
		} else if (*str == _T(' ')) {
			str++;
			break;
		}
		if (buff) {
			*buff++ = *str;
		}
		str++;
		copylen++;
	}
	if (buff) {
		*buff = _T('\0');
	}
	return str - start;
}

static void load_default_values_from_registory() {
	// ftHg̃u[gNXpX
	// iJAVA_HOMEĂ邽߁AŒlɂłȂj
	_stprintf(g_default_boot_classpath,
		      DEFAULT_BOOT_CLASSPATH_FORMAT,
			  get_java_home()); 

	// ftHglWXg擾
	// WXgftHgl[h
	HKEY hKey;
	long error = RegOpenKeyEx(HKEY_CURRENT_USER,
							  REG_ROOT,
							  0,
							  0,
							  &hKey);
    if (error == ERROR_SUCCESS) {
		// L[݂Ăꍇ́AL[̒lǂݍ
		DWORD dwType;
		DWORD cbData = MAX_PATH + 1;
		_TCHAR buff[MAX_PATH + 1];
		
		// u[gNXpX
		error = RegQueryValueEx(hKey,
						REG_DEFAULT_BOOT_CLASSPATH,
						NULL,
						&dwType,
						(LPBYTE) buff,
						&cbData);
		if (error == ERROR_SUCCESS && dwType == REG_SZ) {
			_tcscpy(g_default_boot_classpath, buff);
		}

		// NXpX
		error = RegQueryValueEx(hKey,
							   REG_DEFAULT_CLASSPATH,
							   NULL,
							   &dwType,
							   (LPBYTE) buff,
							   &cbData);
		if (error == ERROR_SUCCESS && dwType == REG_SZ) {
			_tcscpy(g_default_classpath, buff);
		}

		// JgfBNg
		error = RegQueryValueEx(hKey,
							   REG_DEFAULT_CURRENT_DIRECTORY,
							   NULL,
							   &dwType,
							   (LPBYTE) buff,
							   &cbData);
		if (error == ERROR_SUCCESS && dwType == REG_SZ) {
			_tcscpy(g_default_current_directory, buff);
		}

		// őq[vTCY
		DWORD dwHeapSize;
		cbData = sizeof(dwHeapSize);
		error = RegQueryValueEx(hKey,
						REG_DEFAULT_MAX_HEAP_SIZE,
						NULL,
						&dwType,
						(LPBYTE) &dwHeapSize,
						&cbData);
		if (error == ERROR_SUCCESS && dwType == REG_DWORD) {
			// PʂKBŎw肳Ă
			g_default_max_heap_size = dwHeapSize * 1024;
		}
		RegCloseKey(hKey);
	}
}

/**
 * arguments
 */
static void init_arguments(arguments* config) {
	memset(config, 0, sizeof(arguments));

	// NXpX
	config->classpath = (_TCHAR*) malloc(sizeof(g_default_classpath) + 1);
	if (! config->classpath) {
		fatal_error(FATAL_ERROR_NO_MEMORY);
	}
	_tcscpy(config->classpath, g_default_classpath);

	// őq[vTCY
	config->max_heap_size = g_default_max_heap_size;

	// JavaX^bNTCY
	config->java_stack_size = DEFAULT_JAVA_STACK_SIZE;

	// lCeBuX^bNTCY
	config->native_stack_size = DEFAULT_NATIVE_STACK_SIZE;

	// u[gNXpX
	_tcscpy(config->boot_classpath, g_default_boot_classpath);
    
	// xt@C[h
	config->verify_mode = VERIFY_MODE_NONE;	// -noverify

	// JgfBNg
	_tcscpy(config->current_directory, g_default_current_directory);
}