package away3d.animators.data
{
	import away3d.arcane;
	import away3d.core.managers.Stage3DProxy;
	import away3d.materials.passes.MaterialPassBase;

	use namespace arcane;

	/**
	 * VertexAnimation defines an animation type that blends different poses (geometries) together to create a final pose.
	 */
	public class VertexAnimation extends AnimationBase
	{
		arcane var _streamIndex : uint;
		arcane var _useNormals : Boolean;
		arcane var _useTangents : Boolean;
		arcane var _numPoses : uint;
		private var _blendMode : String;

		/**
		 * Creates a new VertexAnimation object
		 * @param numPoses The amount of poses to be blended together.
		 * @param blendMode The type of blending to be performed by the animation. The following values are supported:
		 * <ul>
		 * <li>VertexAnimationMode.ADDITIVE: The pose is generated from a base mesh and a number of additive "difference" poses.</li>
		 * <li>VertexAnimationMode.ABSOLUTE: The pose is generated by a weighted average of a number of poses.</li>
		 * </ul>
		 */
		public function VertexAnimation(numPoses : uint, blendMode : String)
		{
			super();
			_numPoses = numPoses;
			_blendMode = blendMode;
		}

		/**
		 * The type of blending to be performed by the animation.
		 */
		public function get blendMode() : String
		{
			return _blendMode;
		}

		/**
		 * @inheritDoc
		 */
		override arcane function createAnimationState() : AnimationStateBase
		{
			return new VertexAnimationState(this);
		}

		/**
		 * @inheritDoc
		 */
		override arcane function deactivate(stage3DProxy : Stage3DProxy, pass : MaterialPassBase) : void
		{
			// TODO: not used
			pass = pass;
			stage3DProxy.setSimpleVertexBuffer(_streamIndex, null, null, 0);
			if (_useNormals)
				stage3DProxy.setSimpleVertexBuffer(_streamIndex + 1, null, null, 0);
			if (_useTangents)
				stage3DProxy.setSimpleVertexBuffer(_streamIndex + 2, null, null, 0);
		}

		/**
		 * @inheritDoc
		 */
		override arcane function getAGALVertexCode(pass : MaterialPassBase, sourceRegisters : Array, targetRegisters : Array) : String
		{
			if (_blendMode == VertexAnimationMode.ABSOLUTE)
				return getAbsoluteAGALCode(pass, sourceRegisters, targetRegisters);
			else
				return getAdditiveAGALCode(pass, sourceRegisters, targetRegisters);
		}

		/**
		 * Generates the vertex AGAL code for absolute blending.
		 */
		private function getAbsoluteAGALCode(pass : MaterialPassBase, sourceRegisters : Array, targetRegisters : Array) : String
		{
			var code : String = "";
			var temp1 : String = findTempReg(targetRegisters);
			var temp2 : String = findTempReg(targetRegisters, temp1);
			var regs : Array = ["x", "y", "z", "w"];
			var len : uint = sourceRegisters.length;
			_useNormals = len > 1;
			_useTangents = len > 2;
			if (len > 2) len = 2;
			_streamIndex = pass.numUsedStreams;

			var k : uint;
			for (var i : uint = 0; i < len; ++i) {
				for (var j : uint = 0; j < _numPoses; ++j) {
					if (j == 0) {
						code += "mul " + temp1 + ", " + sourceRegisters[i] + ", vc" + pass.numUsedVertexConstants + "." + regs[j] + "\n";
					}
					else {
						code += "mul " + temp2 + ", va" + (_streamIndex + k) + ", vc" + pass.numUsedVertexConstants + "." + regs[j] + "\n";
						if (j < _numPoses - 1) code += "add " + temp1 + ", " + temp1 + ", " + temp2 + "\n";
						else code += "add " + targetRegisters[i] + ", " + temp1 + ", " + temp2 + "\n";
						++k;
					}
				}
			}

			if (_useTangents) {
				code += "dp3 " + temp1 + ".x, " + sourceRegisters[uint(2)] + ", " + targetRegisters[uint(1)] + "\n" +
						"mul " + temp1 + ", " + targetRegisters[uint(1)] + ", " + temp1 + ".x			 \n" +
						"sub " + targetRegisters[uint(2)] + ", " + sourceRegisters[uint(2)] + ", " + temp1 + "\n";
			}
			return code;
		}

		private function getAdditiveAGALCode(pass : MaterialPassBase, sourceRegisters : Array, targetRegisters : Array) : String
		{
			var code : String = "";
			var len : uint = sourceRegisters.length;
			var regs : Array = ["x", "y", "z", "w"];
			var temp1 : String = findTempReg(targetRegisters);
			var k : uint;

			_useNormals = len > 1;
			_useTangents = len > 2;

			if (len > 2) len = 2;

			code += "mov  " + targetRegisters[0] + ", " + sourceRegisters[0] + "\n";
			if (_useNormals) code += "mov " + targetRegisters[1] + ", " + sourceRegisters[1] + "\n";

			for (var i : uint = 0; i < len; ++i) {
				for (var j : uint = 0; j < _numPoses; ++j) {
					code += "mul " + temp1 + ", va" + (_streamIndex + k) + ", vc" + pass.numUsedVertexConstants + "." + regs[j] + "\n" +
							"add " + targetRegisters[i] + ", " + targetRegisters[i] + ", " + temp1 + "\n";
					k++;
				}
			}

			if (_useTangents) {
				code += "dp3 " + temp1 + ".x, " + sourceRegisters[uint(2)] + ", " + targetRegisters[uint(1)] + "\n" +
						"mul " + temp1 + ", " + targetRegisters[uint(1)] + ", " + temp1 + ".x			 \n" +
						"sub " + targetRegisters[uint(2)] + ", " + sourceRegisters[uint(2)] + ", " + temp1 + "\n";
			}

			return code;
		}

		/**
		 * Retrieves a temporary register that's still free.
		 * @param exclude An array of non-free temporary registers
		 * @param excludeAnother An additional register that's not free
		 * @return A temporary register that can be used
		 */
		private function findTempReg(exclude : Array, excludeAnother : String = null) : String
		{
			var i : uint;
			var reg : String;

			while (true) {
				reg = "vt" + i;
				if (exclude.indexOf(reg) == -1 && excludeAnother != reg) return reg;
				++i;
			}

			// can't be reached
			return null;
		}
	}
}