/*
* Copyright 2009 Funambol, Inc.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/* $Id$ */

#pragma once

#include "UserInteractionDefs.h"
#include "common/Buffer.h"
#include <sys/types.h>
#include "CritSection.h"

#ifndef DM_IPCENGINE_API

#ifdef PLATFORM_LINUX
#define DM_IPCENGINE_API
#elif defined(PLATFORM_WINDOWS)
#ifdef DM_IPCENGINE_EXPORTS
#define DM_IPCENGINE_API __declspec(dllexport)
#else
#define DM_IPCENGINE_API __declspec(dllimport)
#endif // DM_IPCENGINE_EXPORTS
#else
#define DM_IPCENGINE_API
#endif

#endif

namespace NS_DM_Client
{

//------------------------------------------------------------------------------------------------------
class IFIFOWrapper;

class DM_IPCENGINE_API Messenger
{
public:
    StatusCode Init();
    bool Release();
    	//------------------------------------------------------------------------------------------------------
    // e_Ok - display message
    // e_OperationCancelled - no message listener
    StatusCode Display(const String& message, const UIOptionalParameters& parameters);
    //------------------------------------------------------------------------------------------------------
    // e_Ok - user choise "Yes"
    // e_OperationCancelled:
    //  - user cancel
    //  - user choise "No"
    //  - MAXDT is spent and no DR or DR set as "No"
    //  - no message listener
    StatusCode Confirmation(const String& message, const UIOptionalParameters& parameters);
    //------------------------------------------------------------------------------------------------------
    // e_Ok - user type input
    // e_OperationCancelled:
    //  - user cancel
    //  - MAXDT is spent and no DR
    //  - no message listener
    StatusCode UserInput(const String& message, const UIOptionalParameters& parameters, String& input);
    //------------------------------------------------------------------------------------------------------
    // e_Ok - user set chiose
    // e_OperationCancelled:
    //  - user cancel
    //  - MAXDT is spent and no DR
    //  - no message listener
    StatusCode UserChoice(const String& message, const UIOptionalParameters& parameters,
        const UserAvailableChoiseSet& available_choise_set, UserChoiseSet& choise_set, bool allowMultipleChoise = true);
    //------------------------------------------------------------------------------------------------------
    // e_Ok - progress is displayed
    // e_OperationCancelled:
    //  - user cancel
    //  - no message listener

    // size_to_download:
    //  > 0 - size in bytes download needed
    //  = 0 - download is finished
    //  < 0 - download is terminated from the side of server
    StatusCode ProgressNotification(const String& message, const UIOptionalParameters& parameters,
        long size_to_download);
    //------------------------------------------------------------------------------------------------------

    ~Messenger();
protected:

    StatusCode ReadDisplay(const UIOptionalParameters& parameters);
    StatusCode ReadConfirmation(const UIOptionalParameters& parameters);
    StatusCode ReadUserInput(const UIOptionalParameters& parameters, String& input);
    StatusCode ReadUserChoice(const UIOptionalParameters& parameters, String& choise);
    StatusCode ReadProgressNotification(const UIOptionalParameters& parameters);

private:
    IFIFOWrapper* m_fifo_request;
    IFIFOWrapper* m_fifo_response;
    NS_Common::CritSection m_critical_section;

};

//------------------------------------------------------------------------------------------------------

/// abstract method for Messenger instance creation
DM_IPCENGINE_API Messenger* CreateMessenger();

}
