require "osdn/cli/version"
require "osdn/cli/runner"
require "osdn/cli/overrides"
require 'getoptlong'
require 'logger'
require 'pathname'
require 'yaml'
require 'fileutils'
require 'osdn-client'
require 'hashie'
require 'json'
require 'digest'

module OSDN
  module CLI
    @@client_id = "osdn-cli"
    @@client_secret = "not-secret"
    def client_id
      @@client_id
    end
    def client_secret
      @@client_secret
    end

    @@_show_progress = false
    def _show_progress
      @@_show_progress
    end
    def _show_progress=(v)
      @@_show_progress = v
    end

    @@_rate_limit = nil
    def _rate_limit
      @@_rate_limit
    end
    def _rate_limit=(v)
      @@_rate_limit = v
    end

    module_function :client_id, :client_secret,
                    :_show_progress, :_show_progress=,
                    :_rate_limit, :_rate_limit=
    
    module Command
      autoload :Login,     'osdn/cli/command/login'
      autoload :Package,   'osdn/cli/command/package'
      autoload :Release,   'osdn/cli/command/release'
      autoload :Relfile,   'osdn/cli/command/relfile'
      autoload :FrsMkdirs, 'osdn/cli/command/frs_mkdirs'
      autoload :FrsUpload, 'osdn/cli/command/frs_upload'
      
      class Base
        def initialize(logger)
          @logger = logger
          @credential = Hashie::Mash.new
          @format = 'pretty'
        end
        attr_reader :logger
        attr_accessor :credential, :format

        def credential_path
          Pathname.new(ENV['HOME']) + '.config/osdn/credential.yml'
        end

        def load_credential
          begin
            stat = credential_path.stat()
            unless credential_path.owned?
              logger.error "Invalid ownership of credential file #{credential_path}, skip loading."
              return
            end
            if RUBY_PLATFORM !~ /mswin|msys|mingw|cygwin|bccwin|wince|emc/ && (stat.mode & 0777).to_s(8) != "600"
              logger.error "Invalid permission #{(stat.mode & 0777).to_s(8)} of credential file #{credential_path}, skip loading."
              return
            end
          rescue Errno::ENOENT
            return
          end
          logger.debug "Loading credentials from #{credential_path}"
          @credential = Hashie::Mash.new(YAML.load_file(credential_path))
          set_client_token
        end

        def write_credential
          FileUtils.mkdir_p credential_path.dirname, verbose: false
          cio = credential_path.open('w', 0600)
          YAML.dump(credential.to_hash, cio)
          cio.close
        end

        def update_token
          logger.debug "Checking token expires..."
          load_credential
          unless credential.access_token
            logger.fatal "You have no access token. Please login with '#{$0} login'."
            return
          end
          if credential.expires_at > Time.now + 30
            logger.debug "You have valid access token, skip to refresh."
            return
          end

          logger.debug "Access token has been expired. Trying to refresh token..."
          api = OSDNClient::DefaultApi.new
          begin
            set_credential api.token(CLI.client_id, CLI.client_secret, grant_type: 'refresh_token', refresh_token: credential.refresh_token)
          rescue OSDNClient::ApiError => e
            begin
              err = JSON.parse(e.response_body)
              logger.fatal err["error_description"]
            rescue
              logger.fatal "Failed to refresh access token."
            end
            logger.fatal "Please login again."
            return
          end
          logger.debug "Access token is refreshed successfully."
        end

        def set_credential(token, update_expires = true)
          token = Hashie::Mash.new(token.to_hash)
          if update_expires
            token.expires_at = Time.now + token.expires_in.to_i
          end
          token.scope = [*token.scope].join(' ').split(' ')
          
          credential.update token
          write_credential
          set_client_token
        end

        def set_client_token
          if credential.access_token && !credential.access_token.empty?
            OSDNClient.configure do |config|
              config.access_token = credential.access_token
            end
          end
        end

        def load_variables(path = '.', recursive_merge = true)
          vars = {}
          path = Pathname.new(Dir.getwd) + path
          cur_dir = Pathname.new('/')
          if recursive_merge
            path.each_filename do |d|
              cur_dir = cur_dir + d
              vf = cur_dir + '.osdn.vars'
              vf.exist? or next
              begin
                logger.debug "Load and merge variables from #{vf}"
                vars.update YAML.load_file(vf.to_s)
              rescue => e
                logger.warn "Failed to load variables from #{vf}; #{e.message}"
              end
            end
          else
            begin
              path = path+'.osdn.vars'
              if path.exist?
                logger.debug "Load and merge variables from #{path}"
                vars.update YAML.load_file(path)
              end
            rescue => e
              logger.warn "Failed to load variables from #{path}; #{e.message}"
            end
          end
          logger.debug "Variables: #{vars.inspect}"
          Hashie::Mash.new(vars)
        end

        def write_variables(vars, dir = nil)
          path = Pathname.new(dir || '.') + '.osdn.vars'
          logger.info "Save variables to #{path}"
          vio = path.open('w')
          YAML.dump(vars.to_hash, vio)
          vio.close
        end

        def update_variables(dir, vars)
          write_variables(load_variables(dir, false).deep_merge(vars), dir)
        end

        private
        def hexdigest(klass, file)
          fio = file.open
          dig = klass.new
          while buf = fio.read(1024*1024) and buf.length > 0
            dig << buf
          end
          fio.close
          dig.hexdigest
        end
        
      end

      class Ping < Base
        def run
          update_token
          api = OSDNClient::DefaultApi.new
          pp api.ping
        end

        def help
          puts "#{$0} ping"
        end

        def self.description
          "Test API request."
        end
      end
        
      class Vars < Base
        def run
          subcommand = ARGV.shift ||'show'
          self.send subcommand
        end

        def show
          name = ARGV.shift
          if name
            puts load_variables[name]
          else
            pp load_variables
          end
        end

        def set
          name, value = ARGV.shift, ARGV.shift
          if !name || name.empty?
            logger.fatal "Missing variable name"
            help
            exit
          end
          if !value || value.empty?
            logger.fatal "Missing variable value"
            help
            exit
          end
          vars = load_variables('.', false)
          vars[name] = value
          write_variables vars
        end

        def help
          puts "#{$0} vars show [name]            -- Show current variable"
          puts "#{$0} vars set <name> <value>     -- Save variable to .osdn.vars"
        end

        def self.description
          "Get/set request environment variable."
        end
      end
    end
  end
end
