<?php

/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */

/**
 * It is a tool to test easily according to RDB. 
 * 
 * PHP versions 4 and 5
 * 
 * LICENSE: This source file is subject to version 3.0 of the PHP license
 * that is available through the world-wide-web at the following URI:
 * http://www.php.net/license/3_0.txt.  If you did not receive a copy of
 * the PHP License and are unable to obtain it through the web, please
 * send a note to license@php.net so we can mail you a copy immediately.
 * 
 * @category    Tools and Utilities
 * @package     PHPUnit_DB
 * @author      uguu <uguu@mail.uguu.mydns.jp>
 * @copyright   Copyright (C) 2005, uguu All rights reserved.
 * @license     http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version     0.1.0
 */

require_once 'PEAR.php';
require_once 'DB.php';
require_once 'PHPUnit.php';

/**
 * PHPUnit_DB_Tool offers the method of the easy examination according to RDB.
 * 
 * Operation by following DBMS was confirmed. 
 *   - MySQL 3.23.58
 *   - MySQL 4.0.26
 *   - MySQL 4.1.15
 *   - PostgreSQL 8.0.4-ja
 *   - PostgreSQL 8.1-beta3
 * 
 * @category    Tools and Utilities
 * @package     PHPUnit_DB
 * @author      uguu <uguu@mail.uguu.mydns.jp>
 * @copyright   Copyright (C) 2005, uguu All rights reserved.
 * @license     http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version     0.1.0
 */
class PHPUnit_DB_Tool extends PEAR {
    /**
     * DSN.
     * 
     * @var     string
     * @access  private
     */
    var $dsn;
    /**
     * The PHPUnit_DB_Tool instance accessed RDB by using specified DSN is initialized.
     * 
     * Notes: For PHP4
     * 
     * @param   string  $dsn    DSN.
     * @access  public
     */
    function PHPUnit_DB_Tool($dsn = '')
    {
        PHPUnit_DB_Tool::__construct($dsn);
    }
    /**
     * The PHPUnit_DB_Tool instance accessed RDB by using specified DSN is initialized.
     * 
     * Notes: For PHP5
     * 
     * @param   string  $dsn    DSN.
     * @access  public
     */
    function __construct($dsn = '')
    {
        $this->PEAR('PHPUnit_DB');
        
        $this->dsn = $dsn;
    }
    /**
     * The data stored in the table is returned by the array.
     * The acquired array can be used for the test data by describing it as PHP code by using the var_export() function for instance.
     * 
     * @param   array   $tableNames Array of exporting table name.
     * @param   string  $encoding   Character encoding of field value.
     * @return  mixed   Array where data of table is stored.
     *                  DB_Error object when failing in method call.
     * @access  public
     */
    function &exportTables($tableNames = array(), $encoding = null)
    {
        $result = array();
        
        $conn = &$this->_getConnection();
        if (PEAR::isError($conn)) {
            return $conn;
        }
        
        foreach ($tableNames as $tableName) {
            $tableInfo = $conn->tableInfo($tableName);
            if (PEAR::isError($tableInfo)) {
                $result = $tableInfo;
                break;
            }
            $primaryKeys = '';
            foreach ($tableInfo as $fieldInfo) {
                if (strstr($fieldInfo['flags'], 'primary_key') !== false) {
                    if (strlen($primaryKeys) === 0) {
                        $primaryKeys = $fieldInfo['name'];
                    } else {
                        $primaryKeys .= ',' . $fieldInfo['name'];
                    }
                }
            }
            
            if (strlen($primaryKeys) === 0) {
                $sql = "select * from ${tableName}";
            } else {
                $sql = "select * from ${tableName} order by ${primaryKeys}";
            }
            
            $cur = &$conn->query($sql);
            if (PEAR::isError($cur)) {
                $result = $cur;
            }
            $data = array();
            while (($row = &$cur->fetchRow()) !== null) {
                $dataRow = array();
                foreach ($row as $fieldName => $fieldValue) {
                    if ($encoding === null) {
                        $dataRow[$fieldName] = $fieldValue;
                    } else {
                        $dataRow[$fieldName] = mb_convert_encoding($fieldValue, $encoding);
                    }
                }
                $data[] = $dataRow;
            }
            $result[$tableName] = $data;
        }
        
        return $result;
    }
    /**
     * The data of the table is deleted.
     * 
     * Notes: When this method fails, the rollback is not done in mysql 3.23.x.
     * 
     * @param   array   $tableNames Array of deleted table name.
     * @return  mixed   true.
     *                  DB_Error object when failing in method call.
     * @access  public
     */
    function &clearTables($tableNames = array())
    {
        $conn = &$this->_getConnection();
        if (PEAR::isError($conn)) {
            return $conn;
        }
        
        $conn->autoCommit(false);
        $result = &$this->_clearTables($conn, $tableNames);
        if ($result === true) {
            $conn->commit();
        } else {
            $conn->rollback();
        }
        $conn->disconnect();
        
        return $result;
    }
    /**
     * The data of the table is deleted.
     * The data of the specified table is deleted by continuously executing the DELETE sql.
     * The method call fails when the execution of the DELETE sql fails even once.
     * 
     * @param   resource    $conn       RDB connection object.
     * @param   array       $tableNames Array of deleted table name.
     * @return  mixed       true.
     *                      DB_Error object when failing in method call.
     * @access  private
     */
    function &_clearTables($conn, $tableNames)
    {
        foreach ($tableNames as $tableName) {
            $sql = "delete from ${tableName}";
            $res = &$conn->query($sql);
            if (PEAR::isError($res)) {
                return $res;
            }
        }
        return true;
    }
    /**
     * The data shown by the array is stored in the table. 
     * Please specify the form of the exportTables() function for the array. 
     * The data that exists in the table before data is stored in the table is deleted. 
     * 
     * @param   array   $data       Array of data stored in table.
     * @param   string  $encoding   Character encoding of field value.
     * @return  mixed   true.
     *                  DB_Error object when failing in method call.
     * @access  public
     */
    function &importTables($data = array(), $encoding = null)
    {
        $conn = &$this->_getConnection();
        if (PEAR::isError($conn)) {
            return $conn;
        }
        
        $conn->autoCommit(false);
        $result = $this->_importTables($conn, $data, $encoding);
        if ($result === true) {
            $conn->commit();
        } else {
            $conn->rollback();
        }
        $conn->disconnect();
        
        return $result;
    }
    /**
     * The data shown by the array is stored in the table. 
     * Please specify the form of the exportTables() function for the array. 
     * The data that exists in the table before data is stored in the table is deleted. 
     * 
     * @param   resource    $conn       RDB connection object.
     * @param   array   $data   Array of data stored in table.
     * @param   string  $encoding   Character encoding of field value.
     * @return  mixed   true.
     *                  DB_Error object when failing in method call.
     * @access  private
     */
    function &_importTables($conn, $data, $encoding) {
        $tableNames = array();
        foreach ($data as $tableName => $tableData) {
            $tableNames[] = $tableName;
        }
        $result = &$this->_clearTables($conn, $tableNames);
        if (PEAR::isError($result)) {
            return $result;
        }
        
        foreach ($data as $tableName => $tableData) {
            foreach ($tableData as $rowData) {
                $sqlFields = '';
                $sqlValues = '';
                $params = array();
                foreach ($rowData as $columnName => $columnValue) {
                    if (strlen($sqlFields) === 0) {
                        $sqlFields = $columnName;
                    } else {
                        $sqlFields .= ',' . $columnName;
                    }
                    if (strlen($sqlValues) === 0) {
                        $sqlValues = '?';
                    } else {
                        $sqlValues .= ',?';
                    }
                    if ($encoding === null) {
                        $params[] = $columnValue;
                    } else {
                        $params[] = mb_convert_encoding($columnValue, $encoding);
                    }
                }
                $sql = "insert into ${tableName} (${sqlFields}) values (${sqlValues})";
                $result = &$conn->query($sql, $params);
                if (PEAR::isError($result)) {
                    return $result;
                }
            }
        }
        return true;
    }
    /**
     * The PEAR::DB connection is returned.
     * 
     * @return  mixed   PEAR::DB connection.
     *                  DB_Error object when failing in method call.
     * @access  public
     */
    function &getConnection()
    {
        $conn = &DB::connect($this->dsn);
        return $conn;
    }
    /**
     * The PEAR::DB connection is returned.
     * setOption('portability', DB_PORTABILITY_LOWERCASE)
     * setFetchMode(DB_FETCHMODE_ASSOC)
     * 
     * @return  mixed   PEAR::DB connection.
     *                  DB_Error object when failing in method call.
     * @access  private
     */
    function &_getConnection()
    {
        $conn = &$this->getConnection();
        $conn->setOption('portability', DB_PORTABILITY_LOWERCASE);
        $conn->setFetchMode(DB_FETCHMODE_ASSOC);
        return $conn;
    }
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 */

?>
