/**
 * 
 */
package jp.mydns.masahase.abaqus;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.io.StringWriter;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * バイナリ形式の結果ファイルを読み取る
 * 
 * @author MASA.H
 * @version 1.0
 * 
 */
public class ReadBinFil implements IReadFil { // NOPMD by MASA.H on 07/09/10
	// 17:39
	private transient RandomAccessFile r_file;
	private transient int recordKey;
	private transient Object[] rcontens;
	private transient boolean btype;

	private int readCount;

	public ReadBinFil() {
		btype = false;
	}

	public int getNextRecordKey() throws EOFException {
		if (!btype)
			return -1;
		try {
			readNextRecord();
		} catch (IOException e) {
			if(recordKey!=2001)
				throw new EOFException(e.toString());
			return -1;
		}
		return recordKey;
	}

	public Object[] getRecordContents() {
		if (!btype)
			return null;
		Object[] ret = new Object[rcontens.length];
		try {
			final ByteArrayOutputStream bos = new ByteArrayOutputStream();
			final ObjectOutputStream oos = new ObjectOutputStream(bos);
			for (int i = 0; i < rcontens.length; i++) {
				oos.writeObject(rcontens[i]);
			}
			final ObjectInputStream ois = new ObjectInputStream(
					new ByteArrayInputStream(bos.toByteArray()));
			for (int i = 0; i < ret.length; i++) {
				ret[i] = ois.readObject();
			}
		} catch (IOException e) {
			e.printStackTrace();
			ret = new Object[0];
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
			ret = new Object[0];
		}
		return ret;
	}

	public boolean isType() {
		return btype;
	}

	/**
	 * 浮動小数点データの読み取り
	 * 
	 * @return
	 * @throws IOException
	 */
	private Double readDouble() throws IOException {
		final byte[] buf = new byte[8];
		skipCheck();
		if (r_file.read(buf) != 8)
			throw new IOException("readDouble()");
		final ByteBuffer buffer = ByteBuffer.wrap(buf);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		readCount++;
		return new Double(buffer.getDouble());
	}

	/**
	 * 整数データの読み取り
	 * 
	 * @return
	 * @throws IOException
	 */
	private Long readInt() throws IOException {
		final byte[] buf = new byte[8];
		skipCheck();
		if (r_file.read(buf) != 8)
			throw new IOException("readInt() at "+recordKey);
		final ByteBuffer buffer = ByteBuffer.wrap(buf);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		Long ret = Long.valueOf(Long.valueOf(buffer.getLong()).intValue());
		assert ret >= 0;
		readCount++;
		return ret;
	}

	/**
	 * 次のレコードのデータを読み取る
	 * 
	 * @throws IOException
	 */
	private void readNextRecord() throws IOException { // NOPMD by MASA.H on
		// 07/09/10 17:44
		final int NumLen = readInt().intValue();
		if (NumLen - 2 > 0) {
			rcontens = new Object[NumLen - 2];
			recordKey = (int) readInt().longValue();
			switch (recordKey) {
			// ?�??�?�???�??�ータ
			case 1900:
				rcontens[0] = readInt();
				rcontens[1] = readString();
				for (int i = 2; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1990:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			// ??�???�??�??�??タ
			case 1901:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// 有効自由度
			case 1902:
				// 部??�???�??�??�?のパス
			case 1903:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			// ABAQUSバ�?ジョン??�?
			case 1921:
				assert NumLen == (7 + 2);
				for (int i = 0; i < 4; i++) {
					rcontens[i] = readString();
				}
				rcontens[4] = readInt();
				rcontens[5] = readInt();
				rcontens[6] = readDouble();
				break;
			// 表??�?
			case 1922:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readString();
				}
				break;
			// ??�???�??�??�???�??�
			case 1931:
				rcontens[0] = readString();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1932:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			// ?�??�?�???�??�??�?
			case 1933:
				rcontens[0] = readString();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1934:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			// ラベル相互参照
			case 1940:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readString();
				}
				break;
			// モー??�???�??�
			case 1980:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// インクリメント開??�?
			case 2000:
				assert NumLen >= (12 + 2);
				for (int i = 0; i < 4; i++) {
					rcontens[i] = readDouble();
				}
				for (int i = 4; i < 8; i++) {
					rcontens[i] = readInt();
				}
				for (int i = 8; i < 11; i++) {
					rcontens[i] = readDouble();
				}
				rcontens[11] = readString();
				for (int i = 12; i < rcontens.length; i++) {
					rcontens[i] = readString();
				}
				break;
			// インクリメント終�?
			case 2001:
				rcontens = new Object[0];
				for (int i = 0; i < NumLen - 2; i++) {
					readDouble();
				}
				break;
			// ?�??�?�???�??�数の先�?
			case 1:
				assert NumLen == (9 + 2);
				for (int i = 0; i < 4; i++) {
					rcontens[i] = readInt();
				}
				rcontens[4] = readString();
				for (int i = 5; i < 9; i++) {
					rcontens[i] = readInt();
				}
				break;
			// 出力変数
			// TEMP
			case 2:
				assert NumLen == (1 + 2);
				rcontens[0] = readDouble();
				break;
			// LOADS
			case 3:
				// FLUXS
			case 4:
				assert NumLen == 4;
				rcontens[0] = readInt();
				rcontens[1] = readDouble();
				break;
			// SDV
			case 5:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// VOIDR
			case 6:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// FOUND
			case 7:
				assert NumLen == 4;
				rcontens[0] = readInt();
				rcontens[1] = readDouble();
				break;
			// COORD
			case 8:
				// FV
			case 9:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// NFLUX
			case 10:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// S
			case 11:
				// CS11
			case 475:
				// SINV
			case 12:
				// SF
			case 13:
				// ESFI
			case 449:
				// ENER
			case 14:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// NFORC
			case 15:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 16:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// JK
			case 17:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// POR
			case 18:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// ELEN
			case 19:
				// E
			case 21:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// PE
			case 22:
				for (int i = 0; i < NumLen - 2 - 2; i++) {
					rcontens[i] = readDouble();
				}
				rcontens[NumLen - 2 - 2] = readString();
				rcontens[NumLen - 2 - 1] = readDouble();
				break;
			// CE
			case 23:
				// IE
			case 24:
				// EE
			case 25:
				// CRACK
			case 26:
				// STH
			case 27:
				// HFL
			case 28:
				// SE
			case 29:
				// DG
			case 30:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// CONF
			case 31:
				assert NumLen == 3;
				rcontens[0] = readInt();
				break;
			// SJP
			case 32:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// FILM
			case 33:
				// RAD
			case 34:
				assert NumLen == 5;
				rcontens[0] = readInt();
				rcontens[1] = readDouble();
				rcontens[2] = readDouble();
				break;
			// SAT
			case 35:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// SS
			case 36:
				assert NumLen == 4;
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// CONC
			case 38:
				// ISOL
			case 446:
				// ESOL
			case 447:
				// SOL
			case 448:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// MFL
			case 39:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// GELVR
			case 40:
				// FLUVR
			case 43:
				// STATUS
			case 61:
				// PEEQ
			case 73:
				// PRESS
			case 74:
				// MISES
			case 75:
				// RATIO
				// ERV
			case 79:
				// AMPCU
			case 80:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// SSAVG
			case 83:
			case 85:
				// ALPHA
			case 86:
				// UVARM
			case 87:
				// THE
			case 88:
				// LE
			case 89:
				// NE
			case 90:
				// ER
			case 91:
				// MFLT
			case 96:
				// FLVEL
			case 97:
				// EMDF
			case 476:
				// EDT
			case 477:
				// SP
			case 401:
				// ALPHAP
			case 402:
				// EP
			case 403:
				// NEP
			case 404:
				// LEP
			case 405:
				// ERP
			case 406:
				// DGP
			case 407:
				// EEP
			case 408:
				// IEP
			case 409:
				// THEP
			case 410:
				// PEP
			case 411:
				// CEP
			case 412:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// VVF
			case 413:
				// VVFG
			case 414:
				// VVFN
			case 415:
				// RD
			case 416:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// CKE
			case 421:
				// CKLE
			case 422:
				// CKLS
			case 423:
				// CKSTAT
			case 424:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// CKEMAG
			case 441:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// SPE
			case 42:
				// SEPE
			case 47:
				// SEE
			case 462:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// SEP
			case 463:
				assert NumLen == 8;
				for (int i = 0; i < 4; i++) {
					rcontens[i] = readDouble();
				}
				rcontens[4] = readString();
				rcontens[5] = readString();
				break;
			// SALPHA
			case 464:
				// CTF
			case 495:
				// CEF
			case 496:
				// CVF
			case 497:
				// CSF
			case 498:
				// CSLST
			case 499:
				// CRF
			case 500:
				// CCF
			case 501:
				// CP
			case 502:
				// CU
			case 503:
				// CCU
			case 504:
				// CV
			case 505:
				// CA
			case 506:
				// CFAILST
			case 507:
				// CNF
			case 542:
				// CIVC
			case 546:
				// CASU
			case 548:
				// CUE
			case 556:
				// CUP
			case 557:
				// CUPEQ
			case 558:
				// CDMG
			case 559:
				// CDIF
			case 560:
				// CDIM
			case 561:
				// CDIP
			case 562:
				// CALPHAF
			case 563:
				// CFFAILURE
			case 44:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// PEQC
			case 45:
				assert NumLen == 10;
				rcontens[0] = readDouble();
				rcontens[1] = readString();
				rcontens[2] = readDouble();
				rcontens[3] = readString();
				rcontens[4] = readDouble();
				rcontens[5] = readString();
				rcontens[6] = readDouble();
				rcontens[7] = readString();
				break;
			// PEEQT
			case 473:
				assert NumLen == 4;
				rcontens[0] = readDouble();
				rcontens[1] = readString();
				break;
			// VS
			case 524:
				// PS
			case 525:
				// VE
			case 526:
				// EPG
			case 50:
				// RBFOR
			case 442:
				// RBANG
			case 443:
				// RBROT
			case 444:
				// MFR
			case 445:
				// PHEPG
			case 46:
				// PHEFL
			case 49:
				// EFLX
			case 51:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// CHRGS
			case 60:
				assert NumLen == 4;
				rcontens[0] = readInt();
				rcontens[1] = readDouble();
				break;
			// ECD
			case 425:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// ECURS
			case 426:
				// NCURS
			case 427:
				assert NumLen == 4;
				rcontens[0] = readInt();
				rcontens[1] = readDouble();
				break;
			// XC
			case 52:
				// UC
			case 53:
				// VC
			case 54:
				// HC
			case 55:
				// HO
			case 56:
				// RI
			case 57:
				// MASS
			case 58:
				// VOL
			case 59:
				// TSHR
			case 48:
				// PHS
			case 62:
				// RS
			case 63:
				// PHE
			case 65:
				// RE
			case 66:
				// PHCTF
			case 508:
				// PHCEF
			case 509:
				// PHCVF
			case 510:
				// PHCRF
			case 511:
				// PHCSF
			case 520:
				// PHCU
			case 512:
				// PHCCU
			case 513:
				// PHCV
			case 522:
				// PHCA
			case 523:
				// PHCNF
			case 543:
				// PHCIVSL
			case 547:
				// RCTF
			case 514:
				// RCEF
			case 515:
				// RCVF
			case 516:
				// RCRF
			case 517:
				// RCSF
			case 521:
				// RCU
			case 518:
				// RCCU
			case 519:
				// RCNF
			case 544:
				// PHMFL
			case 94:
				// PHMFT
			case 95:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// IVOL
			case 76:
				// SVOL
			case 77:
				// EVOL
			case 78:
				// VOLC
			case 264:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// U
			case 101:
				// V
			case 102:
				// A
			case 103:
				// RF
			case 104:
				// EPOT
			case 105:
				// CF
			case 106:
				// COORD
			case 107:
				// POR
			case 108:
				// RVF
			case 109:
				// RVT
			case 110:
				// RCHG
			case 119:
				// CECHG
			case 120:
				// PCAV
			case 136:
				// CVOL
			case 137:
				// RECUR
			case 138:
				// CECUR
			case 139:
				// VF
			case 145:
				// PABS
			case 151:
				// NT
			case 201:
				// RFL
			case 204:
				// CFL
			case 206:
				// RFLE
			case 214:
				// NNC
			case 237:
				// CFF
			case 320:
				// PU
			case 111:
				// PTU
			case 112:
				// TU
			case 113:
				// TV
			case 114:
				// TA
			case 115:
				// PPOR
			case 116:
				// PHPOT
			case 117:
				// PHCHG
			case 118:
				// RU
			case 123:
				// RTU
			case 124:
				// RV
			case 127:
				// RTV
			case 128:
				// RA
			case 131:
				// RTA
			case 132:
				// RRF
			case 134:
				// PRF
			case 135:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// GU
			case 301:
				// GV
			case 302:
				// GA
			case 303:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// BM
			case 304:
				assert NumLen == 10;
				rcontens[0] = readInt();
				rcontens[1] = readDouble();
				rcontens[2] = readDouble();
				rcontens[3] = readDouble();
				rcontens[4] = readDouble();
				rcontens[5] = readDouble();
				rcontens[6] = readDouble();
				rcontens[7] = readString();
				break;
			// GPU
			case 305:
				// GPV
			case 306:
				// GPA
			case 307:
				// SNE
			case 308:
				// KE
			case 309:
				// T
			case 310:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1001:
				rcontens[0] = readInt();
				rcontens[1] = readString();
				for (int i = 2; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1002:
			case 1003:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1004:
				assert NumLen == 3;
				rcontens[0] = readInt();
				break;
			case 1005:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1011:
			case 1012:
			case 1013:
			case 1022:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1031:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1032:
				rcontens[0] = readString();
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1041:
				rcontens[0] = readInt();
				rcontens[1] = readString();
				for (int i = 2; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1042:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1043:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1999:
				assert NumLen == 20;
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1991:
			case 1992:
			case 1995:
			case 1996:
				rcontens[0] = readInt();
				rcontens[1] = readString();
				rcontens[2] = readInt();
				for (int i = 2; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1993:
				assert NumLen == 11;
				rcontens[0] = readInt();
				rcontens[1] = readString();
				rcontens[2] = readString();
				rcontens[3] = readInt();
				rcontens[4] = readInt();
				rcontens[5] = readInt();
				rcontens[6] = readDouble();
				rcontens[7] = readDouble();
				rcontens[8] = readDouble();
				break;
			case 1501:
				assert NumLen == 7;
				rcontens[0] = readString();
				rcontens[1] = readInt();
				rcontens[2] = readInt();
				rcontens[3] = readInt();
				rcontens[4] = readInt();
				break;
			case 1502:
				//Surface facet
				rcontens[0]=readInt();
				rcontens[1]=readInt();
				rcontens[2]=readInt();
				for(int i=3; i<NumLen-2;i++){
					rcontens[i]=readInt();
				}
				break;
			case 1052:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1503:
				assert NumLen == 6;
				rcontens[0] = readInt();
				rcontens[1] = readString();
				rcontens[2] = readString();
				rcontens[3] = readString();
				break;
			case 1504:
				assert NumLen == 4;
				rcontens[0] = readInt();
				rcontens[1] = readInt();
				break;
			// CSTRESS
			case 1511:
				// CDSTRESS
			case 1512:
				// CDISP
			case 1521:
				// CFN
			case 1522:
				// CFS
			case 1523:
				// CFT
			case 1575:
				// CAREA
			case 1524:
				// CMN
			case 1526:
				// CMS
			case 1527:
				// CMT
			case 1576:
				// CTRQ
			case 1578:
				// XN
			case 1573:
				// XS
			case 1574:
				// XT
			case 1577:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// HFL
			case 1528:
				// HFLA
			case 1529:
				// HTL
			case 1530:
				// HTLA
			case 1531:
				// SFDR
			case 1532:
				// SFDRA
			case 1533:
				// SFDRT
			case 1534:
				// SFDRTA
			case 1535:
				// WEIGHT
			case 1536:
				// SJD
			case 1537:
				// SJDA
			case 1538:
				// SJDT
			case 1539:
				// SJDTA
			case 1540:
				// ECD
			case 1541:
				// ECDA
			case 1542:
				// ECDT
			case 1543:
				// ECDTA
			case 1544:
				// PFL
			case 1545:
				// PFLA
			case 1546:
				// PTL
			case 1547:
				// PTLA
			case 1548:
				// TPFL
			case 1549:
				// TPTL
			case 1550:
				// DBT
			case 1570:
				// DBSF
			case 1571:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// DBS
			case 1572:
				assert NumLen == 4;
				rcontens[0] = readDouble();
				rcontens[1] = readDouble();
				break;
			// PRESS
			case 1592:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			case 1601:
				rcontens[0] = readInt();
				for (int i = 1; i < NumLen - 2; i++) {
					rcontens[i] = readString();
				}
				break;
			case 1610:
				assert NumLen == 3;
				rcontens[0] = readInt();
				break;
			case 1602:
				rcontens[0] = readInt();
				rcontens[1] = readString();
				for (int i = 2; i < NumLen - 2; i++) {
					rcontens[i] = readInt();
				}
				break;
			case 1608:
			case 1605:
				assert NumLen == 4;
				rcontens[0] = readInt();
				rcontens[1] = readString();
				break;
			case 1609:
				assert NumLen == 3;
				rcontens[0] = readInt();
				break;
			case 1606:
			case 1607:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			case 1603:
				assert NumLen == 6;
				rcontens[0] = readInt();
				rcontens[1] = readString();
				rcontens[2] = readString();
				rcontens[3] = readString();
				break;
			case 1604:
				assert NumLen == 5;
				rcontens[0] = readInt();
				rcontens[1] = readInt();
				rcontens[2] = readDouble();
				break;
			case 231:
			case 232:
			case 233:
			case 234:
			case 235:
			case 236:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			case 1580:
				assert NumLen == 4;
				rcontens[0] = readInt();
				rcontens[1] = readString();
				break;
			case 1581:
				assert NumLen == 5;
				rcontens[0] = readString();
				rcontens[1] = readInt();
				rcontens[2] = readInt();
				break;
			case 1582:
			case 1583:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// SOAREA
			case 1584:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			// SOF
			case 1585:
				// SOM
			case 1586:
				// SOCF
			case 1587:
				for (int i = 0; i < NumLen - 2; i++) {
					rcontens[i] = readDouble();
				}
				break;
			// SOH
			case 1588:
				// SOE
			case 1589:
				// SOD
			case 1590:
				// SOP
			case 1591:
				assert NumLen == 3;
				rcontens[0] = readDouble();
				break;
			case 1911:
				assert NumLen >= 4;
				rcontens[0] = readInt();
				for (int i = 1; i < rcontens.length; i++) {
					rcontens[i] = readString();
				}
				break;					
			default:
				throw new IOException("Can't read valid record key!("
						+ recordKey + ")");
			}
		} else {
			throw new IOException("Record size(" + NumLen
					+ ") is too small at " + r_file.getFilePointer() + ".");
		}
	}

	/**
	 * 文字列データの読み取り
	 * 
	 * @return
	 * @throws IOException
	 */
	private String readString() throws IOException {
		final StringWriter swbuf = new StringWriter();
		skipCheck();
		for (int i = 0; i < 8; i++) {
			swbuf.write(r_file.read());
		}
		readCount++;
		return swbuf.toString();
	}

	public void rewind() {
		try {
			r_file.seek(4);
			readCount = 0;
		} catch (IOException e) {
			e.printStackTrace();
			System.err.println("This file can't rewind."); // NOPMD by MASA.H
			// on 07/09/11 16:09
		}
	}

	public void setFileName(final String fname) throws FileNotFoundException {
		r_file = new RandomAccessFile(fname, "r");
		// ヘッ??�???�??�み取り
		try {
			final byte[] buf = new byte[4];
			if (r_file.read(buf) != 4)
				throw new IOException();
			if (buf[0] == 0x00 && buf[1] == 0x10 && buf[2] == 0x00
					&& buf[3] == 0x00) {
				btype = true;
			}
			readCount = 0;
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			btype = false;
		}
	}

	/**
	 * ファイル中に含まれる余分なデータの回避
	 * 
	 * @throws IOException
	 */
	private void skipCheck() throws IOException {
		if (readCount == (0x1000 / 8)) {
			r_file.skipBytes(8);
			readCount = 0;
		}
	}

}
