using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;

//using SystemNeo;
using SystemNeo.IO;
using SystemNeo.Text;

namespace SystemNeo.Drawing.Imaging.Icc
{
	/// <summary>
	/// 
	/// </summary>
	public sealed class IccTagElement
	{
		#region private static fields
		private const int tagLength = 4;
		private const int typeLength = 4;
		private static readonly Formatter valueFormatter = new Formatter();
		private static readonly IDictionary<string, string> tagDic;
		#endregion

		#region private fields
		private readonly string tagCode;
		private readonly int size;
		#endregion

		// public vpeB //

		/// <summary>
		/// 
		/// </summary>
		public int Offset { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public string Tag { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public string Type { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public object Value { get; private set; }

		// static RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		static IccTagElement()
		{
			valueFormatter.ByteArrayFormatter.MaxCount = 20;
			tagDic = CreateTagDictionary();
		}

		// internal RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		internal IccTagElement(BinaryReaderNeo reader, long basePosition)
		{
			this.tagCode = reader.ReadString(tagLength);
			if (! tagDic.ContainsKey(this.tagCode)) {
				string msg = "^OsłB" + new QuotedStringFormatter().ToString(this.tagCode);
				throw new BadImageFormatException(msg);
			}
			this.Tag = (string)tagDic[this.tagCode];
			this.Offset = reader.ReadInt32();
			this.size = reader.ReadInt32();
			using (reader.KeepPosition()) {
				reader.Position = basePosition + this.Offset;
				this.Type = reader.ReadString(typeLength).TrimEnd();
				this.Value = this.ReadValue(reader);
			}
		}

		// public \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			var sw = new StringWriter();
			sw.Write("{Tag=");
			sw.Write(this.tagCode);
			sw.Write("=");
			sw.Write(this.Tag);
			sw.Write(", Offset=");
			sw.Write(this.Offset.ToString("N0"));
			sw.Write(", Size=");
			sw.Write(this.size.ToString("N0"));
			sw.Write(", Type=");
			sw.Write(this.Type);
			sw.Write(", Value=");
			if (this.Value is string) {
				new QuotedStringFormatter().WriteTo(sw, this.Value);
			} else if (this.Value is Fraction[]) {
				IccUtil.fractionArrayFormatter.WriteTo(sw, this.Value);
			} else {
				valueFormatter.WriteTo(sw, this.Value);
			}
			sw.Write("}");
			return sw.ToString();
		}

		// private static \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		private static IDictionary<string, string> CreateTagDictionary()
		{
			var dic = new Dictionary<string, string>();
			dic["A2B0"] = "AToB0";
			dic["A2B1"] = "AToB1";
			dic["A2B2"] = "AToB2";
			dic["B2A0"] = "BToA0";
			dic["B2A1"] = "BToA1";
			dic["B2A2"] = "BToA2";
			dic["bkpt"] = "MediaBlackPoint";
			dic["bfd "] = "UCRBG";
			dic["bTRC"] = "BlueTRC";
			dic["bXYZ"] = "BlueMatrixColumn";
			dic["calt"] = "CalibrationDateTime";
			dic["chad"] = "ChromaticAdaptation";
			dic["chrm"] = "Chromaticity";
			dic["clro"] = "ColorantOrder";
			dic["clrt"] = "ColorantTable";
			dic["cprt"] = "ProfileCopyright";
			dic["crdi"] = "CRDInfo";
			dic["desc"] = "ProfileDescription";
			dic["devs"] = "DeviceSettings";
			dic["dmdd"] = "DeviceModelDesc";
			dic["dmnd"] = "DeviceMfgDesc";
			dic["dscm"] = "ProfileDescriptionML";
			dic["gamt"] = "Gamut";
			dic["gTRC"] = "GreenTRC";
			dic["gXYZ"] = "GreenMatrixColumn";
			dic["K007"] = "'K007'";
			dic["kTRC"] = "GrayTRC";
			dic["lumi"] = "Luminance";
			dic["meas"] = "Measurement";
			dic["mmod"] = "MakeAndModel";
			dic["ncl2"] = "NamedColor2";
			dic["ncol"] = "NamedColor";
			dic["ndin"] = "NativeDisplayInfo";
			dic["ps2i"] = "PS2RenteringIntent";
			dic["ps2s"] = "PostScript2CSA";
			dic["psd0"] = "PostScript2CRD0";
			dic["psd1"] = "PostScript2CRD1";
			dic["psd2"] = "PostScript2CRD2";
			dic["psec"] = "ProfileSequenceDesc";
			dic["psvm"] = "PS2CRDVMSize";
			dic["pre0"] = "Preview0";
			dic["pre1"] = "Preview1";
			dic["pre2"] = "Preview2";
			dic["resp"] = "OutputResponse";
			dic["rTRC"] = "RedTRC";
			dic["rXYZ"] = "RedMatrixColumn";
			dic["scrd"] = "ScreeningDesc";
			dic["scrn"] = "Screening";
			dic["targ"] = "CharTarget";
			dic["tech"] = "Technology";
			dic["ucmI"] = "'ucmI'";
			dic["vcgt"] = "VideoCardGamma";
			dic["view"] = "ViewingConditions";
			dic["vued"] = "ViewingCondDesc";
			dic["wtpt"] = "MediaWhitePoint";
			return dic;
		}

		// private \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <returns></returns>
		private object ReadValue(BinaryReaderNeo reader)
		{
			byte[] reserved = reader.ReadBytes(4);
			DataType t = 0;
			try {
				t = (DataType)Enum.Parse(typeof(DataType), this.Type);
			} catch (ArgumentException) {
			}
			switch (t) {
			case DataType.curv:
				return new Curve(reader);
			case DataType.desc:
				return new TextDescription(reader);
			case DataType.meas:
				return new Measurement(reader);
			case DataType.mluc:
				return new LocalizedStringCollection(reader);
			case DataType.sf32:
				int count = (this.size - 8) / sizeof(int);
				return IccUtil.ReadFractions(reader, count);
			case DataType.sig:
				return reader.ReadString(4).TrimEnd(' ');
			case DataType.text:
				return reader.ReadString(this.size - 8).TrimEnd('\0');
			case DataType.view:
				return new ViewingConditions(reader);
			case DataType.XYZ:
				return IccUtil.ReadFractions(reader, 3);
			}
			return reader.ReadBytes(this.size - 4);
		}

		// ^ //

		/// <summary>
		/// 
		/// </summary>
		private enum DataType
		{
			clro = 1,
			curv,
			data,
			desc,
			dtim,
			meas,
			mluc,
			sf32,
			sig,
			text,
			uf32,
			ui32,
			ui64,
			ui08,
			view,
			XYZ
		}
	}
}
